//
// DOSDoom Main Init + Program Loop Code
//
// Based on the sources released by Id Software (c) 1994-1996
//
// DESCRIPTION:
//      DOOM main program (D_DoomMain) and
//      game loop (D_DoomLoop), plus functions
//      to determine game mode (shareware, registered),
//      parse command line parameters,
//      configure game parameters (turbo),
//      and call the startup functions.
//
// -MH- 1998/07/02 "shootupdown" --> "true3dgameplay"
// -MH- 1998/08/19 added up/down movement variables
//
#ifdef DEVELOPERS
// -KM- 1998/09/27 I don't like this; conio is non-portable.
#ifdef DJGPP
#include <conio.h>
#else
int getch(void) { return 0;}
#endif
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifdef LINUX
#include <termcap.h>
#endif

#include "d_debug.h"

#include "dm_defs.h"
#include "dm_state.h"

#include "dstrings.h"
#include "lu_sound.h"

#include "z_zone.h"
#include "w_wad.h"
#include "s_sound.h"

#include "f_finale.h"
#include "f_wipe.h"

#include "m_argv.h"
#include "m_misc.h"
#include "m_menu.h"

#include "i_system.h"
#include "i_sound.h"
#include "i_video.h"
#include "i_allegv.h"

#include "g_game.h"

#include "hu_stuff.h"
#include "wi_stuff.h"
#include "st_stuff.h"
#include "am_map.h"

#include "p_setup.h"
#include "r_local.h"
#include "v_res.h"

#include "rad_trig.h"

#include "d_main.h"

//
// D_DoomLoop
//
// Not a globally visible function, just included for source reference,
// called by D_DoomMain, never exits.
//
// Manages timing and IO, calls all ?_Responder, ?_Ticker, and ?_Drawer,
// calls I_GetTime, I_StartFrame, and I_StartTic.
//
void D_DoomLoop (void) __attribute__((noreturn));

// Might as well alloc space for both wads now...
int maxwadfiles = 2;
int addwadnum = 0;
char** wadfiles = NULL;

boolean devparm;    // started game with -devparm
boolean singletics = false; // debug flag to cancel adaptiveness

// -KM- 1998/12/16 These flags hold everything needed about a level
// -KM- 1999/01/29 Added autoaim flag.
gameflags_t gameflags = {
 false, // nomonsters
 false, // fastparm
 RS_TELEPORT, // respawn setting
 false, // respawn
 false, // item respawn
 false, // true 3d gameplay
 8, // gravity
 false, // extra blood
 true, // jump
 true, // freelook
 AA_ON, // Autoaim
 true, // trans
 true, // cheats
};
// -KM- 1998/12/16 These flags are the users prefs and are copied to
//   gameflags when a new level is started.
gameflags_t settingflags = {
 false, // nomonsters
 false, // fastparm
 RS_TELEPORT, // respawn setting
 false, // respawn
 false, // item respawn
 false, // true 3d gameplay
 8, // gravity
 false, // extra blood
 true, // jump
 true, // freelook
 AA_ON, // autoaim
 true, // trans
 true, // cheats
};

boolean drone = false;

extern boolean inhelpscreens;

skill_t startskill;
char *startmap;

boolean autostart;

FILE* debugfile = NULL;

boolean advancedemo;

char basedefault[1024];      // default file

int newnmrespawn=0;    
int lessaccuratemon=0;
int lessaccuratezom=1; //default is inaccurate
int rotatemap=0;
int showstats=0;
int novert=0;

// causes an update of the available weapons by key
boolean weaponupdate = true;

boolean swapstereo=0;
boolean mus_pause_stop = false;

int infight=0;
int crosshair=0;
//int stretchsky=0;

int missileteleport=0; 
int teleportdelay=0;   
int oldsetup=0;

int retrace=0;

void D_CheckNetGame (void);
void D_ProcessEvents (void);
void G_BuildTiccmd (ticcmd_t* cmd);
void D_DoAdvanceDemo (void);


//
// EVENT HANDLING
//
// Events are asynchronous inputs generally generated by the game user.
// Events can be discarded if no responder claims them
//
event_t         events[MAXEVENTS];
int             eventhead;
int             eventtail;


//
// D_PostEvent
// Called by the I/O functions when input is detected
//
void D_PostEvent (event_t* ev)
{
    events[eventhead] = *ev;
    eventhead = (++eventhead)&(MAXEVENTS-1);
}


//
// D_ProcessEvents
// Send all the events of the given timestamp down the responder chain
//
void D_ProcessEvents (void)
{
    event_t*    ev;

    for ( ; eventtail != eventhead ; eventtail = (++eventtail)&(MAXEVENTS-1) )
    {
        ev = &events[eventtail];
        if (M_Responder (ev))
            continue;               // menu ate the event
        G_Responder (ev);
    }
}

//
// D_Display
//  draw current display, possibly wiping it from the previous
//
// -ACB- 1998/07/27 Removed doublebufferflag check (unneeded).  
//

// wipegamestate can be set to -1 to force a wipe on the next draw
gamestate_t wipegamestate = GS_DEMOSCREEN;
int wipe_method = wipe_Melt;
extern int showMessages;
boolean redrawsbar;
void R_ExecuteChangeResolution (void); // -ES- 1998/08/20
void R_ExecuteSetViewSize (void);

void D_Display (void)
{
    static boolean viewactivestate = false;
    static boolean menuactivestate = false;
    static boolean inhelpscreensstate = false;
    static boolean fullscreen = false;
    static int  borderdrawcount;
    static gamestate_t oldgamestate = -1;
    int nowtime;
    int tics;
    int wipestart;
    int y;
    boolean done;
    boolean wipe;

    if (nodrawers)
         return;                    // for comparative timing / profiling

    // -ES- 1998/08/20 Resolution Change Check
    if (changeresneeded)
      R_ExecuteChangeResolution();
                    
    // change the view size if needed
    if (setsizeneeded)
    {
         R_ExecuteSetViewSize ();
         oldgamestate = -1;                      // force background redraw
         borderdrawcount = 3;
    }

    // save the current screen if about to wipe
    if (gamestate != wipegamestate)
    {
        wipe = true;
        wipe_StartScreen(0, 0, SCREENWIDTH, SCREENHEIGHT);
    }
    else
        wipe = false;

    if (gamestate == GS_LEVEL && gametic)
        HU_Erase();
    
    // do buffered drawing
    switch (gamestate)
    {
      case GS_LEVEL:
        if (!gametic)
            break;
        if (automapactive==2)
            AM_Drawer ();
        if (wipe || (viewheight != SCREENHEIGHT && fullscreen) )
            redrawsbar = true;
        if (inhelpscreensstate && !inhelpscreens)
            redrawsbar = true;              // just put away the help screen
        ST_Drawer (viewheight == SCREENHEIGHT, redrawsbar );
        redrawsbar = false;
        fullscreen = viewheight == SCREENHEIGHT;
        break;

      case GS_INTERMISSION:
        WI_Drawer ();
        break;

      case GS_FINALE:
        F_Drawer ();
        break;

      case GS_DEMOSCREEN:
        D_PageDrawer ();
        break;
    }
    
    // draw buffered stuff to screen
    I_UpdateNoBlit ();
    
    // draw the view directly
    if (gamestate == GS_LEVEL && gametic && automapactive != 2 )
    {
      R_Render ();

      if (automapactive)
        AM_Drawer ();
    }

    if (gamestate == GS_LEVEL && gametic)
        HU_Drawer ();
    
    // clean up border stuff
    if (gamestate != oldgamestate && gamestate != GS_LEVEL)
        I_SetPalette (W_CacheLumpName ("PLAYPAL",PU_CACHE),0);

    // see if the border needs to be initially drawn
    if (gamestate == GS_LEVEL && oldgamestate != GS_LEVEL)
    {
        viewactivestate = false;        // view was not active
        R_FillBackScreen ();    // draw the pattern into the back screen
    }

    // see if the border needs to be updated to the screen
   if (gamestate == GS_LEVEL && automapactive != 2 )
    {
        if (menuactive || menuactivestate || !viewactivestate)
            borderdrawcount = 3;
        if (borderdrawcount)
        {
            R_DrawViewBorder ();    // erase old menu stuff
            borderdrawcount--;
        }

    }

    menuactivestate = menuactive;
    viewactivestate = viewactive;
    inhelpscreensstate = inhelpscreens;
    oldgamestate = wipegamestate = gamestate;
    
    // draw pause pic
    if (paused)
    {
        if (automapactive)
            y = 4;
        else
            y = viewwindowy+4;
        V_DrawPatchDirect(viewwindowx+(scaledviewwidth-68)/2,
                          y,0,W_CacheLumpName ("M_PAUSE", PU_CACHE));
    }

    // menus go directly to the screen
    M_Drawer ();          // menu is drawn even on top of everything
    NetUpdate ();         // send out any new accumulation

    // 98-7-10 KM Removed songsync cause it stuffs up midi music.

    // normal update
    if (!wipe)
    {
        I_FinishUpdate ();              // page flip or blit buffer
        return;
    }
    
    // wipe update
    wipe_EndScreen(0, 0, SCREENWIDTH, SCREENHEIGHT);

    // -ES- 1998/11/28 Added wipe_method setting
    wipe_ScreenWipe(wipe_method, 0, 0, SCREENWIDTH, SCREENHEIGHT, 1);
    wipestart = I_GetTime () - 1;

    do
    {
        do
        {
            nowtime = I_GetTime ();
            tics = nowtime - wipestart;
        } while (!tics);
        wipestart = nowtime;
        done = wipe_ScreenWipe(wipe_method
                               , 0, 0, SCREENWIDTH, SCREENHEIGHT, tics);
        I_UpdateNoBlit ();
        M_Drawer ();                            // menu is drawn even on top of wipes
        I_FinishUpdate ();                      // page flip or blit buffer
    } while (!done);
}

//
//  D_DoomLoop
//
extern boolean demorecording;
void D_DoomLoop (void)
{
    extern int mselapsed;
    if (demorecording)
      G_BeginRecording ();
                
    I_InitGraphics ();
    // -ES- 1998/09/11 Use R_ChangeResolution to enter gfx mode
    R_ChangeResolution(SCREENWIDTH,SCREENHEIGHT,BPP);

    // -KM- 1998/09/27 Change res now, so music doesn't start before
    // screen.  Reset clock too.
    R_ExecuteChangeResolution();
    mselapsed = 0;

    while (1)
    {
        // -ES- 1998/09/11 It's a good idea to frequently check the heap
        #ifdef DEVELOPERS
        Z_CheckHeap ();
        #endif

        // frame syncronous IO operations
        I_StartFrame ();                
        
        // process one or more tics
        if (singletics)
        {
            I_StartTic ();
            D_ProcessEvents ();
            G_BuildTiccmd (&netcmds[consoleplayer][maketic%BACKUPTICS]);
            if (advancedemo)
              D_DoAdvanceDemo ();
            M_Ticker ();
            G_Ticker ();
            gametic++;
            maketic++;
        }
        else
        {
            TryRunTics (); // will run at least one tic
        }
                
            S_UpdateSounds (players[consoleplayer].mo);// move positional sounds

            // Update display, next frame, with current state.
            D_Display ();
        }
}

//
//  DEMO LOOP
//
int             demosequence;
int             pagetic;
char                    *pagename;

//
// D_PageTicker
// Handles timing for warped projection
//
void D_PageTicker (void)
{
    if (--pagetic < 0)
        D_AdvanceDemo ();
}

//
// D_PageDrawer
//
void D_PageDrawer (void)
{
    // 98-7-10 KM Added screen clear.
    V_ClearPageBackground(0);
    V_DrawPatchInDirect (0,0, 0, W_CacheLumpName(pagename, PU_CACHE));
}

//
// D_AdvanceDemo
// Called after each demo or intro demosequence finishes
//
void D_AdvanceDemo (void)
{
    advancedemo = true;
}

//
// This cycles through the demo sequences.
// -KM- 1998/12/16 Fixed for DDF.
//
void D_DoAdvanceDemo (void)
{
    static int mapon = 0;
    static int picnumber = 0;
    static int demonumber = 1;
    static char buffer[9]; // - Kester
    
    players[consoleplayer].playerstate = PST_LIVE;  // not reborn
    advancedemo = false;
    usergame = false;               // no save / end game here
    paused = false;
    gameaction = ga_nothing;

    demosequence = (demosequence + 1) % 2; // - Kester

    switch (demosequence) // - Kester
    {
      case 0: // Picture
        if (picnumber == 0)
        {
              while ((W_CheckNumForName(wi_maps[mapon].titlemusic) == -1)
                 || (picnumber >= wi_maps[mapon].numtitlepics))
              {
                mapon = (mapon + 1) % wi_nummaps;
                picnumber = 0;
              }
              pagetic = wi_maps[mapon].titletics;
              S_ChangeMusic(wi_maps[mapon].titlemusic, false); // DOOM II TITLE (game->intromusic);
        }
        else
           pagetic = 320;

        gamestate = GS_DEMOSCREEN;
        while (W_CheckNumForName(wi_maps[mapon].titlepics[picnumber]) < 0)
        {
            picnumber ++;
            while (picnumber >= wi_maps[mapon].numtitlepics)
            {
              mapon = (mapon + 1) % wi_nummaps;
              picnumber = 0;
            }
        }
        pagename = wi_maps[mapon].titlepics[picnumber];
        picnumber++;
        while (picnumber >= wi_maps[mapon].numtitlepics)
        {
          mapon = (mapon + 1) % wi_nummaps;
          picnumber = 0;
        }
        break;

      default: // Demo
        sprintf(buffer, "DEMO%x", demonumber++);
        if (W_CheckNumForName(buffer) < 0) {
          demonumber = 1;
          sprintf(buffer, "DEMO1");
        }
        G_DeferedPlayDemo (buffer);
        break;
    }
}

//
// D_StartTitle
//
void D_StartTitle (void)
{
    gameaction = ga_nothing;
    demosequence = -1;
    D_AdvanceDemo ();
}

//
// D_AddFile
//
void D_AddFile (char *file)
{
    char    *newfile;
        
    if (addwadnum == maxwadfiles) {
      wadfiles = realloc(wadfiles, (++maxwadfiles + 1) * sizeof(char *));
    }

    newfile = malloc (strlen(file)+1);
    strcpy (newfile, file);
        
    wadfiles[addwadnum++] = newfile;
    wadfiles[addwadnum] = NULL;
}

//
// D_IdentifyVersion
// Checks availability of IWAD files by name,
// to determine whether registered/commercial features
// should be executed (notably loading PWAD's).
//
// -KM-  1998/07/10 Modified, de-hacked and cleaned up
//
// -ACB- 1998/07/11 Reformatted (again)
//
// -KM-  1998/07/21 Reformatted again.  Andy's version
//                    didn't load Doom2 & UDoom wadfiles;
//                    they are in a different directory.
//                    Added some comments :-)
//
// -ACB- 1998/07/30 DOH! loading plutonia.wad in dosdoom mode, stopped that :);
//                  Resurrected my old version to do this (sorry, Kester).
//
// -ACB- 1998/08/09 Nuked the lot for DDF......
//
// -KM- 1998/12/16 OK, fixed this for DDF.  All iwads in the doomwaddir directory
//      are found and loaded.  The -iwad param is used to load either only a
//      single wad file, or specify a new directory to search for iwads in.
//      dosdoom.wad will also be found in the current directory.
// -KM- 1999/01/29 Fixed this for portability.  The opendir method works in DJGPP,
//      CYGWIN, and probably Linux.
//
void D_IdentifyVersion (void)
{
    int p;
    char *dosdoomwad;
    char *home;
    char *doomwaddir;
    char *iwad = NULL;
    DIR  *d = NULL;
    struct dirent* de;
    boolean done = false;
    doomwaddir = getenv("DOOMWADDIR");
    if (!doomwaddir)
        doomwaddir = ".";

    p=M_CheckParm("-gwaddir");
    if (p && p<myargc-1)
    {
      doomwaddir=(char *)alloca(strlen(myargv[p+1])+1);
      strcpy(doomwaddir,myargv[p+1]);
    }

#ifndef LINUX
    home = getenv("DOSDOOM");
    if (!home)
      sprintf(basedefault,"default.cfg");
    else
      sprintf(basedefault, "%s\\default.cfg", home);
    I_Printf("  Config File: %s\n", basedefault);
#else
    home=getenv("HOME");
    if (!home)
      I_Error("Please set $HOME to your home directory");
    sprintf(basedefault,"%s/.doomrc",home);
#endif

    p = M_CheckParm("-iwad");
    if (p && p < myargc-1)
    {
      iwad = myargv[p+1];
      d = opendir(iwad);
      if (!d)
      {
        D_AddFile(iwad);
        done = true;
      }
    }

    if (!iwad)
      iwad = doomwaddir;

    if (!done)
    {
      if (!d)
      {
        d = opendir(doomwaddir);
      }

      if (d)
      {
        while ((de = readdir(d)))
        {
           if (strlen(de->d_name) >= 4)
           if (!strcasecmp(de->d_name + strlen(de->d_name) - 4, ".wad"))
           {
             int fd;
             char id[4];
             char temp[strlen(iwad) + strlen(de->d_name) + 2];
             sprintf(temp, "%s/%s", iwad, de->d_name);
             fd = open(temp, O_RDONLY|O_BINARY);
             read(fd, id, 4);
             close(fd);
             if (!memcmp(id, "IWAD", 4))
               D_AddFile(temp);
           }
        }
        closedir(d);
      }
    }
    if (!addwadnum)
      I_Error("No IWADS found!\n");

    // dosdoom wad is a special case.  If it exists, add it.
    // Find the path in the config file (this changes in ddf)
    dosdoomwad = get_config_string("wads", "dosdoom", NULL);
    if (!dosdoomwad) {
      // Alloca enough space for waddir + "/" + "dosdoom" + ".wad" + NULL terminator
      dosdoomwad = alloca(strlen(doomwaddir) + 1 + 7 + 4 + 1);
      sprintf(dosdoomwad, "%s/dosdoom.wad", doomwaddir);
    }
    if (!access(dosdoomwad, R_OK))
        D_AddFile(dosdoomwad);
    else
    {
      sprintf(dosdoomwad, "dosdoom.wad");
      if (!access(dosdoomwad, R_OK))
          D_AddFile(dosdoomwad);
    }
}


void D_ApplyResponseFile (char *filename, int i)
{
#define MAXARGVS        200

  FILE *          handle;
  int             size;
  int             k;
  int             index;
  int             indexinfile;
  char    *infile;
  char    *file;
  char    *moreargs[20];
  char    *firstargv;
                        
  // READ THE RESPONSE FILE INTO MEMORY
  handle = fopen (filename,"rb");
  if (!handle)
    {
    I_Printf ("\nNo such response file!");
    exit(1);
    }
  I_Printf("Found response file %s!\n",filename);
  fseek (handle,0,SEEK_END);
  size = ftell(handle);
  fseek (handle,0,SEEK_SET);
  file = malloc (size);
  fread (file,size,1,handle);
  fclose (handle);

  // KEEP ALL CMDLINE ARGS FOLLOWING @RESPONSEFILE ARG
  for (index = 0,k = i+1; k < myargc; k++)
    moreargs[index++] = myargv[k];
                        
  firstargv = myargv[0];
  myargv = malloc(sizeof(char *)*MAXARGVS);
  memset(myargv,0,sizeof(char *)*MAXARGVS);
  myargv[0] = firstargv;
                        
  infile = file;
  indexinfile = k = 0;
  indexinfile++;  // SKIP PAST ARGV[0] (KEEP IT)
  do{
    myargv[indexinfile++] = infile+k;
    while(k < size &&
     ((*(infile+k)>= ' '+1) && (*(infile+k)<='z')))
       k++;
    *(infile+k) = 0;
    while(k < size &&
     ((*(infile+k)<= ' ') || (*(infile+k)>'z')))
       k++;
    } while(k < size);
                        
    for (k = 0;k < index;k++)
        myargv[indexinfile++] = moreargs[k];
    myargc = indexinfile;
        
    // DISPLAY ARGS
    //
    // -ACB- 1998/06/19 Removed the display of the dosdoom.cmd
    //                  (was it really necessary).
    //
    Debug_Printf("%d command-line args:\n",myargc);
    for (k=1;k<myargc;k++)
        Debug_Printf("%s\n",myargv[k]);
}
//
// Find a Response File
//
void D_FindResponseFile (void)
{
    int             i;

    if ( !access ("dosdoom.cmd", R_OK ) )
      D_ApplyResponseFile("dosdoom.cmd",0);

    for (i = 1;i < myargc;i++)
    {
        if (myargv[i][0] == '@')
        {
        D_ApplyResponseFile(&(myargv[i][1]),i);
        break;
        }
    }
}

//
// D_DoomMain
//
// -ACB- 1998/08/10 Removed all reference to a gamemap, episode and mission
//                  Used LanguageLookup() for lang specifics.
//
// -ACB- 1998/09/06 Removed all the unused code that no longer has
//                  relevance.    
//
void D_DoomMain (void)
{
  int p;
  char file[256];
    
  //      print title for every printed line
  char title[] = "DOSDoom";

  wadfiles = malloc(sizeof(char *) * (maxwadfiles + 1));

#ifdef DJGPP
  set_config_file("dosdoom.cfg");
#endif

  I_PutTitle(title);

  // 23-6-98 KM Changed to hex to allow versions such as 0.65a etc
  I_Printf("DOSDoom v%x.%02x compiled on "__DATE__" at "__TIME__"\n", DOSDOOMVER / DOSDOOMVERFIX, DOSDOOMVER % DOSDOOMVERFIX);
  I_Printf("DOSDoom homepage is at http://www.frag.com/dosdoom/\n");
  I_Printf("DOOM is by id Software http://www.idsoftware.com/\n");

#ifdef DEVELOPERS
  // -ACB- 1998/09/06 Only used for debugging.
  //                  Moved here to setup debug file for DDF Parsing...
  p=M_CheckParm ("-debugfile");
  if (p)
  {
    char filename[100];
    int i = 1;

    // -ES- 1999/03/29 allow -debugfile <file>
    if (p+1 < myargc && myargv[p+1][0] != '-')
    {
      strcpy(filename,myargv[p+1]);
    } else
    {

      // -KM- 1999/01/29 Consoleplayer is always 0 at this stage.
      sprintf (filename,"debug0.txt");
      while (!access(filename, R_OK))
        sprintf(filename, "debug%d.txt", i++);
    }
    I_Printf ("debug output to: %s\n",filename);
    debugfile = fopen (filename,"w");
    Debug_Printf(title);
  }
#endif

  DDF_MainInit();

  D_FindResponseFile();
        
  D_IdentifyVersion();

  modifiedgame = false;

  oldsetup = M_CheckParm("-oldset"); // Original Doom Packet Send

  if (M_CheckParm ("-nomonsters"))
    settingflags.nomonsters = true;
  if (M_CheckParm ("-fast"))
    settingflags.fastparm = true;
  devparm = M_CheckParm ("-devparm");
    
  if (M_CheckParm ("-altdeath"))
    deathmatch = 2;
  else if ((p = M_CheckParm ("-deathmatch")))
  {
    deathmatch = 1;
    if (p && p < myargc-1)
      deathmatch = atoi(myargv[p+1]);

    if (!deathmatch)
      deathmatch = 1;
  }

  if (devparm)
    I_Printf(DDF_LanguageLookup("DevelopmentMode"));
    
  // turbo option
  if ((p=M_CheckParm ("-turbo")))
  {
    int scale = 200;
    extern int upwardmove[2]; // -MH- 1998/08/19 up/down movement factors
    extern int forwardmove[2];
    extern int sidemove[2];
        
    if (p<myargc-1)
      scale = atoi (myargv[p+1]);

    if (scale < 10)
      scale = 10;

    if (scale > 400)
      scale = 400;

    I_Printf (DDF_LanguageLookup("TurboScale"),scale);
    // -MH- 1998/08/19 initialise up/down movement factors
    upwardmove[0] = upwardmove[0]*scale/100;
    upwardmove[1] = upwardmove[1]*scale/100;
    forwardmove[0] = forwardmove[0]*scale/100;
    forwardmove[1] = forwardmove[1]*scale/100;
    sidemove[0] = sidemove[0]*scale/100;
    sidemove[1] = sidemove[1]*scale/100;
  }

  I_CheckCPU();
  
  p = M_CheckParm ("-file");

  if (p)
  {
    // the parms after p are wadfile/lump names,
    // until end of parms or another - preceded parm
    modifiedgame = true;

    while (++p != myargc && myargv[p][0] != '-')
      D_AddFile (myargv[p]);
  }

  p = M_CheckParm ("-playdemo");

  if (!p)
    p = M_CheckParm ("-timedemo");

  if (p && p < myargc-1)
  {
    sprintf (file,"%s.lmp", myargv[p+1]);
    D_AddFile (file);
    I_Printf("Playing demo %s.lmp.\n",myargv[p+1]);
  }

  // 98-7-10 KM Use this to test the power of your machine.
  if (M_CheckParm("-singletics"))
    singletics = true;
    
  // get skill / episode / map from parms
  startskill = sk_medium;
  autostart = false;
                
  // -KM- 1999/01/29 Use correct skill: 1 is easiest, not 0
  p = M_CheckParm ("-skill");
  if (p && p < myargc-1)
  {
    startskill = atoi(myargv[p+1])-1;
    autostart = true;
  }
        
  p = M_CheckParm ("-timer");
  if (p && p < myargc-1 && deathmatch)
  {
    int time;
    time = atoi(myargv[p+1]);
    I_Printf("Levels will end after %d minute",time);

    if (time>1)
      I_Printf("s");

    I_Printf(".\n");
  }

  //-----------------------------------------------------------------
  // -ACB- 1998/09/06 Use mapping system for warping...
  p = M_CheckParm ("-warp");

  if (p && p < myargc-1)
  {
    startmap = strdup(myargv[p+1]);

    if (!startmap)
      I_Error("D_DoomMain: Malloc failure\n");

    autostart = true;
  }
  else
  {
    startmap = strdup("MAP01");

    if (!startmap)
      I_Error("D_DoomMain: Malloc failure\n");
  }
  //-----------------------------------------------------------------

  I_Printf (DDF_LanguageLookup("DefaultLoad"));
  M_LoadDefaults ();              // load before initing other systems

  // -ACB- 1998/07/30 Respawnsetting replaces respawnparm & newnmrespawn.
  if (M_CheckParm ("-newnmrespawn"))
  {
    settingflags.respawnsetting = RS_RESURRECT;
    settingflags.respawn = gameflags.respawn = true;
  } else if (M_CheckParm ("-respawn"))
    settingflags.respawn = gameflags.respawn = true;

  if (M_CheckParm ("-rotatemap"))
    rotatemap = true;
  if (M_CheckParm ("-itemrespawn"))
    settingflags.itemrespawn = true;
  if (M_CheckParm ("-stretchsky"))
    settingflags.stretchsky = true;

  //mlook stuff
  if ( M_CheckParm ("-mlook"))
    settingflags.freelook = true;
  if (M_CheckParm("invertmouse"))
      invertmouse = true;
    
  // speed for mouse look
  p = M_CheckParm("-vspeed");

  if (p && p<myargc-1)
    keylookspeed=atoi(myargv[p+1])/64;

  // init subsystems
  // -ES- 1998/08/20 Dynamic Resolution Changes: V_Init->V_InitResolution
  I_Printf (DDF_LanguageLookup("AllocScreens"));
  // -ES- 1998/09/11 Removed V_InitResolution call
  V_MultiResInit();

  I_Printf (DDF_LanguageLookup("ZoneMemoryAlloc"));
  Z_Init ();

  I_Printf (DDF_LanguageLookup("WADFileInit"));
  W_InitMultipleFiles (wadfiles);
  DDF_MainCleanUpLooseEnds();

  // -KM- 1998/11/25 Check for command line script loading
  p = M_CheckParm("-script");
  if (p && p < myargc-1)
    RAD_LoadScript(myargv[p+1], -1);

  for (p = 4; p < MAXPLAYERS; p++)
  {
    char name[9];
    sprintf(name, "STFB%d", p);

    if (W_CheckNumForName(name) < 0)
      I_Error("Player Background %d (%s) not found!  Get DOSDOOM.WAD\n", p, name);

    sprintf(name, "STPB%d", p);

    if (W_CheckNumForName(name) < 0)
      I_Error("Player Background %d (%s) not found!  Get DOSDOOM.WAD\n", p, name);
  }

  I_Printf (DDF_LanguageLookup("Notice"));

#ifdef DEVELOPERS
  getch();
#endif

  I_Printf (DDF_LanguageLookup("MiscInfo"));
  M_Init ();

  I_Printf (DDF_LanguageLookup("RefreshDaemon"));
  R_Init ();

  I_Printf (DDF_LanguageLookup("PlayState"));
  P_Init ();

  I_Printf (DDF_LanguageLookup("InitMachine"));
  I_Init ();

  I_Printf (DDF_LanguageLookup("CheckNetGame"));
  D_CheckNetGame ();

  // 98-7-10 KM Support for more save game slots.
  if (netgame)
    mkdir(NETSAVEDIR, 0);
  else
    mkdir(SAVEGAMEDIR, 0);

  I_Printf (DDF_LanguageLookup("SoundInit"));
  S_Init (snd_SfxVolume, snd_MusicVolume);

  I_Printf (DDF_LanguageLookup("HeadsUpInit"));
  HU_Init ();

  I_Printf (DDF_LanguageLookup("STBarInit"));
  ST_Init ();

  // check for a driver that wants intermission stats
  p = M_CheckParm ("-statcopy");

  if (p && p<myargc-1)
  {
    // for statistics driver Kester thinks this is corrupt
    extern void* statcopy;

    statcopy = (void*)atoi(myargv[p+1]);

    I_Printf ("External statistics registered.\n");
  }
    
  // start the apropriate game based on parms
  p = M_CheckParm ("-record");

  if (p && p < myargc-1)
  {
    G_RecordDemo (myargv[p+1]);
    autostart = true;
  }
        
  p = M_CheckParm ("-playdemo");

  if (p && p < myargc-1)
  {
    singledemo = true;              // quit after one demo
    G_DeferedPlayDemo (myargv[p+1]);
    D_DoomLoop ();  // never returns
  }
        
  p = M_CheckParm ("-timedemo");
  if (p && p < myargc-1)
  {
    G_TimeDemo (myargv[p+1]);
    D_DoomLoop ();  // never returns
  }
        
  p = M_CheckParm ("-loadgame");
  if (p && p < myargc-1)
  {
    // 98-7-10 KM Support for more savegame slots
    if (netgame)
      sprintf(file, NETSAVEDIR"/"SAVEGAMENAME"%04x.dsg",myargv[p+1][0]);
    else
      sprintf(file, SAVEGAMEDIR"/"SAVEGAMENAME"%04x.dsg",myargv[p+1][0]);

    G_LoadGame (file);
  }

  // -ACB- 1998/09/06 use new mapping system
  if (gameaction != ga_loadgame)
  {
    if (autostart || netgame)
    {
      mapstuff_t* map = DDF_LevelGetNewMap(startmap);
      free(startmap);

      // if startmap is failed, do normal start.
      // -KM- 1998/12/21 G_DeferedInitNew will set netgame to false!
      if (map)
          G_InitNew (startskill, map);
      else
          D_StartTitle();
    }
    else
    {
      D_StartTitle();                // start up intro loop
    }
  }

  D_DoomLoop ();  // never returns
}
