/*
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation. NO WARRANTY.
*/

#include "doomstat.h"
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>

#include "w_wad.h"

lumpinfo_t *lumpinfo;
int numlumps;
void **lumpcache;

void ExtractFileBase(const char *path, char *dest)
{
  const char *src = path + strlen(path) - 1;
  int length;

  while (src != path && src[-1] != ':'
         && *(src-1) != '\\' && *(src-1) != '/')
    src--;

  memset(dest,0,8);
  length = 0;

  while (*src && *src != '.')
    if (++length == 9)
      I_Error ("Base de nom de archive %s >8 chars",path);
    else
      *dest++ = toupper(*src++);
}

char *AddDefaultExtension(char *path, const char *ext)
{
  char *p = path;
  while (*p++);
  while (p-->path && *p!='/' && *p!='\\')
    if (*p=='.')
      return path;
  if (*ext!='.')
    strcat(path,".");
  return strcat(path,ext);
}

void NormalizeSlashes(char *str)
{
  char *p;
  for (p = str; *p; p++)
    if (*p == '\\')
      *p = '/';
  while (p > str && *--p == '/')
    *p = 0;
  for (p = str; (*str++ = *p);)
    if (*p++ == '/')
      while (*p == '/')
	p++;
}

static void W_AddFile(const char *name)
{
  wadinfo_t   header;
  lumpinfo_t* lump_p;
  unsigned    i;
  int         handle,length,startlump;
  filelump_t  *fileinfo,*fileinfo2free=NULL,singleinfo;
  char        *filename = strcpy(malloc(strlen(name)+5), name);

  NormalizeSlashes(AddDefaultExtension(filename, ".wad"));
  if ((handle = open(filename,O_RDONLY | O_BINARY)) == -1)
    {
      if (strlen(name) > 4 && !strcasecmp(name+strlen(name)-4 , ".lmp" ))
	{
	  free(filename);
	  return;
	}
      NormalizeSlashes(AddDefaultExtension(strcpy(filename, name), ".lmp"));
      if ((handle = open(filename,O_RDONLY | O_BINARY)) == -1)
	I_Error("Erreur: ne pouvez ouvrir %s",name);
    }
  printf("  ajoutant %s\n",filename);
  startlump = numlumps;

  if (strlen(filename)<=4 || strcasecmp(filename+strlen(filename)-4, ".wad" ))
    {
      fileinfo = &singleinfo;
      singleinfo.filepos = 0;
      singleinfo.size =  LONG(__filelength(handle));
      ExtractFileBase(filename, singleinfo.name);
      numlumps++;
    }
  else
    {
      read(handle, &header, sizeof(header));
      if (strncmp(header.identification,"IWAD",4) &&
          strncmp(header.identification,"PWAD",4))
        I_Error("Wad archive %s n'ai pas IWAD or PWAD id", filename);
      header.numlumps = LONG(header.numlumps);
      header.infotableofs = LONG(header.infotableofs);
      length = header.numlumps*sizeof(filelump_t);
      fileinfo2free = fileinfo = malloc(length);
      lseek(handle, header.infotableofs, SEEK_SET);
      read(handle, fileinfo, length);
      numlumps += header.numlumps;
    }
    free(filename);
    lumpinfo = realloc(lumpinfo, numlumps*sizeof(lumpinfo_t));
    lump_p = &lumpinfo[startlump];

    for (i=startlump ; i<numlumps ; i++,lump_p++, fileinfo++)
      {
        lump_p->handle = handle;
        lump_p->position = LONG(fileinfo->filepos);
        lump_p->size = LONG(fileinfo->size);
        lump_p->data = NULL;
        lump_p->namespace = ns_global;
        strncpy (lump_p->name, fileinfo->name, 8);
      }

    free(fileinfo2free);
}

static int IsMarker(const char *marker, const char *name)
{
  return !strncasecmp(name, marker, 8) ||
    (*name == *marker && !strncasecmp(name+1, marker, 7));
}

static void W_CoalesceMarkedResource(const char *start_marker,
                                     const char *end_marker, int namespace)
{
  lumpinfo_t *marked = malloc(sizeof(*marked) * numlumps);
  size_t i, num_marked = 0, num_unmarked = 0;
  int is_marked = 0, mark_end = 0;
  lumpinfo_t *lump = lumpinfo;

  for (i=numlumps; i--; lump++)
    if (IsMarker(start_marker, lump->name))
      { if (!num_marked)
          {
            strncpy(marked->name, start_marker, 8);
            marked->size = 0;
            marked->namespace = ns_global;
            num_marked = 1;
          }
        is_marked = 1;
      }
    else
      if (IsMarker(end_marker, lump->name))
        {
          mark_end = 1;
          is_marked = 0;
        }
      else
        if (is_marked)
          {
            marked[num_marked] = *lump;
            marked[num_marked++].namespace = namespace;
          }
        else
          lumpinfo[num_unmarked++] = *lump;

  memcpy(lumpinfo + num_unmarked, marked, num_marked * sizeof(*marked));
  free(marked);
  numlumps = num_unmarked + num_marked;
  if (mark_end)
    {
      lumpinfo[numlumps].size = 0;
      lumpinfo[numlumps].namespace = ns_global;
      strncpy(lumpinfo[numlumps++].name, end_marker, 8);
    }
}

unsigned W_LumpNameHash(const char *s)
{
  unsigned hash;
  (void) ((hash =        toupper(s[0]), s[1]) &&
          (hash = hash*3+toupper(s[1]), s[2]) &&
          (hash = hash*2+toupper(s[2]), s[3]) &&
          (hash = hash*2+toupper(s[3]), s[4]) &&
          (hash = hash*2+toupper(s[4]), s[5]) &&
          (hash = hash*2+toupper(s[5]), s[6]) &&
          (hash = hash*2+toupper(s[6]),
           hash = hash*2+toupper(s[7]))
         );
  return hash;
}

int (W_CheckNumForName)(register const char *name, register int namespace)
{
  register int i = lumpinfo[W_LumpNameHash(name) % (unsigned) numlumps].index;

  while (i >= 0 && (strncasecmp(lumpinfo[i].name, name, 8) ||
                    lumpinfo[i].namespace != namespace))
    i = lumpinfo[i].next;

  return i;
}

static void W_InitLumpHash(void)
{
  int i;

  for (i=0; i<numlumps; i++)
    lumpinfo[i].index = -1;

  for (i=0; i<numlumps; i++)
    {
      int j = W_LumpNameHash(lumpinfo[i].name) % (unsigned) numlumps;
      lumpinfo[i].next = lumpinfo[j].index;
      lumpinfo[j].index = i;
    }
}

int W_GetNumForName (const char* name)
{
  int i = W_CheckNumForName (name);
  if (i == -1)
    I_Error ("W_GetNumForName: %.8s ne finde", name);
  return i;
}

void W_InitMultipleFiles(char *const *filenames)
{
  numlumps = num_predefined_lumps;
  lumpinfo = malloc(numlumps*sizeof(*lumpinfo));
  memcpy(lumpinfo, predefined_lumps, numlumps*sizeof(*lumpinfo));
  while (*filenames)
    W_AddFile(*filenames++);
  if (!numlumps)
    I_Error ("W_InitFiles: ne archives finde");
  W_CoalesceMarkedResource("S_START", "S_END", ns_sprites);
  W_CoalesceMarkedResource("F_START", "F_END", ns_flats);
  W_CoalesceMarkedResource("C_START", "C_END", ns_colormaps);
  lumpcache = calloc(sizeof *lumpcache, numlumps);
  if (!lumpcache)
    I_Error ("Ne pouvez commence lumpcache");
  W_InitLumpHash();
}

int W_LumpLength (int lump)
{
  if (lump >= numlumps)
    I_Error ("W_LumpLength: %i >= numlumps",lump);
  return lumpinfo[lump].size;
}

void W_ReadLump(int lump, void *dest)
{
  lumpinfo_t *l = lumpinfo + lump;

  if (l->data)
    memcpy(dest, l->data, l->size);
  else
    {
      int c;

      lseek(l->handle, l->position, SEEK_SET);
      c = read(l->handle, dest, l->size);
      if (c < l->size)
        I_Error("W_ReadLump: seul lire %i de %i on lump %i", c, l->size, lump);
    }
  I_BeginRead();
}

void *W_CacheLumpNum(int lump, int tag)
{
  if (!lumpcache[lump])
    W_ReadLump(lump, Z_Malloc(W_LumpLength(lump), tag, &lumpcache[lump]));
  else
    Z_ChangeTag(lumpcache[lump],tag);

  return lumpcache[lump];
}