/*
 *	Name:		OPL2/OPL3 Music driver
 *	Project:	MUS File Player Library
 *	Version:	1.67
 *	Author:		Vladimir Arnost (QA-Software)
 *	Last revision:	May-2-1996
 *	Compiler:	Borland C++ 3.1, Watcom C/C++ 10.0
 *
 */

#include <mem.h>
#include <io.h>
#include "muslib.h"

struct OPLdata {
 uint channelInstr[CHANNELS];
 uchar channelVolume[CHANNELS];
 uchar channelLastVolume[CHANNELS];
 schar channelPan[CHANNELS];
 schar channelPitch[CHANNELS];
 uchar channelSustain[CHANNELS];
 uchar channelModulation[CHANNELS];
};
static char OPL2name[] = "OPL2 FM";
struct driverBlock OPL2driver = {
 NULL,
 DRV_OPL2,
 OPL2name,
 sizeof(struct OPLdata),
 OPLinitDriver,
 OPLdeinitDriver,
 OPLdriverParam,
 OPLloadBank,
 OPL2detectHardware,
 OPL2initHardware,
 OPL2deinitHardware,
 OPLplayNote,
 OPLreleaseNote,
 OPLpitchWheel,
 OPLchangeControl,
 OPLplayMusic,
 OPLstopMusic,
 OPLchangeVolume,
 OPLpauseMusic,
 OPLunpauseMusic,
 OPLsendMIDI};

static char OPL3name[] = "OPL3 FM";
struct driverBlock OPL3driver = {
 NULL,
 DRV_OPL3,
 OPL3name,
 sizeof(struct OPLdata),
 OPLinitDriver,
 OPLdeinitDriver,
 OPLdriverParam,
 OPLloadBank,
 OPL3detectHardware,
 OPL3initHardware,
 OPL3deinitHardware,
 OPLplayNote,
 OPLreleaseNote,
 OPLpitchWheel,
 OPLchangeControl,
 OPLplayMusic,
 OPLstopMusic,
 OPLchangeVolume,
 OPLpauseMusic,
 OPLunpauseMusic};

static uint	OPLsinglevoice = 0;
static struct OP2instrEntry *OPLinstruments = NULL;

static struct channelEntry {
 uchar channel;
 uchar musnumber;
 uchar note;
 uchar flags;
 uchar realnote;
 schar finetune;
 sint pitch;
 uint volume;
 uint realvolume;
 struct OPL2instrument *instr;
 ulong time;
} channels[MAXCHANNELS];

#define CH_SECONDARY 0x01
#define CH_SUSTAIN   0x02
#define CH_VIBRATO   0x04
#define CH_FREE      0x80
#define MOD_MIN      40
#define CHANNEL_ID(ch) (*(ushort *)&(ch))

WORD MAKE_ID(uchar ch, uchar mus)
{
 return ch | (mus << 8);
}

static WORD freqtable[] = {
 345, 365, 387, 410, 435, 460, 488, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517, 547, 580, 615, 651,
 690, 731, 774, 820, 869, 921, 975, 517};

static BYTE octavetable[] = {
 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1,
 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2,
 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3,
 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4,
 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5,
 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6,
 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7,
 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8,
 8, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9,
 9, 9, 9, 9, 9, 9, 9,10};

#define HIGHEST_NOTE 127

static WORD pitchtable[] = {
 29193U,29219U,29246U,29272U,29299U,29325U,29351U,29378U,
 29405U,29431U,29458U,29484U,29511U,29538U,29564U,29591U,
 29618U,29644U,29671U,29698U,29725U,29752U,29778U,29805U,
 29832U,29859U,29886U,29913U,29940U,29967U,29994U,30021U,
 30048U,30076U,30103U,30130U,30157U,30184U,30212U,30239U,
 30266U,30293U,30321U,30348U,30376U,30403U,30430U,30458U,
 30485U,30513U,30541U,30568U,30596U,30623U,30651U,30679U,
 30706U,30734U,30762U,30790U,30817U,30845U,30873U,30901U,
 30929U,30957U,30985U,31013U,31041U,31069U,31097U,31125U,
 31153U,31181U,31209U,31237U,31266U,31294U,31322U,31350U,
 31379U,31407U,31435U,31464U,31492U,31521U,31549U,31578U,
 31606U,31635U,31663U,31692U,31720U,31749U,31778U,31806U,
 31835U,31864U,31893U,31921U,31950U,31979U,32008U,32037U,
 32066U,32095U,32124U,32153U,32182U,32211U,32240U,32269U,
 32298U,32327U,32357U,32386U,32415U,32444U,32474U,32503U,
 32532U,32562U,32591U,32620U,32650U,32679U,32709U,32738U,
 32768U,32798U,32827U,32857U,32887U,32916U,32946U,32976U,
 33005U,33035U,33065U,33095U,33125U,33155U,33185U,33215U,
 33245U,33275U,33305U,33335U,33365U,33395U,33425U,33455U,
 33486U,33516U,33546U,33576U,33607U,33637U,33667U,33698U,
 33728U,33759U,33789U,33820U,33850U,33881U,33911U,33942U,
 33973U,34003U,34034U,34065U,34095U,34126U,34157U,34188U,
 34219U,34250U,34281U,34312U,34343U,34374U,34405U,34436U,
 34467U,34498U,34529U,34560U,34591U,34623U,34654U,34685U,
 34716U,34748U,34779U,34811U,34842U,34874U,34905U,34937U,
 34968U,35000U,35031U,35063U,35095U,35126U,35158U,35190U,
 35221U,35253U,35285U,35317U,35349U,35381U,35413U,35445U,
 35477U,35509U,35541U,35573U,35605U,35637U,35669U,35702U,
 35734U,35766U,35798U,35831U,35863U,35895U,35928U,35960U,
 35993U,36025U,36058U,36090U,36123U,36155U,36188U,36221U,
 36254U,36286U,36319U,36352U,36385U,36417U,36450U,36483U,
 36516U,36549U,36582U,36615U,36648U,36681U,36715U,36748U};

static void writeFrequency(uint slot, uint note, int pitch, uint keyOn)
{
 uint freq = freqtable[note];
 uint octave = octavetable[note];

 if (pitch)
 {
	if (pitch > 127) pitch = 127;
	else if (pitch < -128) pitch = -128;
	freq = ((ulong)freq * pitchtable[pitch + 128]) >> 15;
	if (freq >= 1024)
	{
 freq >>= 1;
 octave++;
	}
 }
 if (octave > 7)
	octave = 7;
 OPLwriteFreq(slot, freq, octave, keyOn);
}
static void writeModulation(uint slot, struct OPL2instrument *instr, int state)
{
 if (state)
	state = 0x40;
 OPLwriteChannel(0x20, slot,
	(instr->feedback & 1) ? (instr->trem_vibr_1 | state) : instr->trem_vibr_1,
	instr->trem_vibr_2 | state);
}
static uint calcVolume(uint channelVolume, uint MUSvolume, uint noteVolume)
{
 noteVolume = ((ulong)channelVolume * MUSvolume * noteVolume) / (256*127);
 if (noteVolume > 127)
	return 127;
 else
	return noteVolume;
}
static int occupyChannel(struct musicBlock *mus, uint slot, uint channel,
	int note, int volume, struct OP2instrEntry *instrument, uchar secondary)
{
 struct OPL2instrument *instr;
 struct OPLdata *data = (struct OPLdata *)mus->driverdata;
 struct channelEntry *ch = &channels[slot];

 playingChannels++;

 ch->channel = channel;
 ch->musnumber = mus->number;
 ch->note = note;
 ch->flags = secondary ? CH_SECONDARY : 0;
 if (data->channelModulation[channel] >= MOD_MIN)
	ch->flags |= CH_VIBRATO;
 ch->time = MLtime;
 if (volume == -1)
	volume = data->channelLastVolume[channel];
 else
	data->channelLastVolume[channel] = volume;
 ch->realvolume = calcVolume(data->channelVolume[channel], mus->volume, ch->volume = volume);
 if (instrument->flags & FL_FIXED_PITCH)
	note = instrument->note;
 else if (channel == PERCUSSION)
	note = 60;
 if (secondary && (instrument->flags & FL_DOUBLE_VOICE))
	ch->finetune = instrument->finetune - 0x80;
 else
	ch->finetune = 0;
 ch->pitch = ch->finetune + data->channelPitch[channel];
 if (secondary)
	instr = &instrument->instr[1];
 else
	instr = &instrument->instr[0];
 ch->instr = instr;
 if ( (note += instr->basenote) < 0)
	while ((note += 12) < 0);
 else if (note > HIGHEST_NOTE)
	while ((note -= 12) > HIGHEST_NOTE);
 ch->realnote = note;

 OPLwriteInstrument(slot, instr);
 if (ch->flags & CH_VIBRATO)
	writeModulation(slot, instr, 1);
 OPLwritePan(slot, instr, data->channelPan[channel]);
 OPLwriteVolume(slot, instr, ch->realvolume);
 writeFrequency(slot, note, ch->pitch, 1);
 return slot;
}
static int releaseChannel(uint slot, uint killed)
{
 struct channelEntry *ch = &channels[slot];
 playingChannels--;
 writeFrequency(slot, ch->realnote, ch->pitch, 0);
 ch->channel |= CH_FREE;
 ch->flags = CH_FREE;
 if (killed)
 {
 OPLwriteChannel(0x80, slot, 0x0F, 0x0F);
 OPLwriteChannel(0x40, slot, 0x3F, 0x3F);
 }
 return slot;
}
static int releaseSustain(struct musicBlock *mus, uint channel)
{
 uint i;
 uint id = MAKE_ID(channel, mus->number);

 for(i = 0; i < OPLchannels; i++)
 {
	if (CHANNEL_ID(channels[i]) == id && channels[i].flags & CH_SUSTAIN)
 releaseChannel(i, 0);
 }
 return 0;
}
static int findFreeChannel(uint flag)
{
 static uint last = -1U;
 uint i;
 uint oldest = -1U;
 ulong oldesttime = MLtime;

 for(i = 0; i < OPLchannels; i++)
 {
	if (++last == OPLchannels)
 last = 0;
	if (channels[last].flags & CH_FREE)
 return last;
 }

 if (flag & 1) return -1;
 for(i = 0; i < OPLchannels; i++)
 {
	if (channels[i].flags & CH_SECONDARY)
	{
 releaseChannel(i, -1);
 return i;
	} else
 if (channels[i].time < oldesttime)
 {
		oldesttime = channels[i].time;
		oldest = i;
 }
 }

 if ( !(flag & 2) && oldest != -1U)
 {
	releaseChannel(oldest, -1);
	return oldest;
 }

 return -1;
}
static struct OP2instrEntry *getInstrument(struct musicBlock *mus, uint channel, uchar note)
{
 uint instrnumber;

 if (mus->percussMask & (1 << channel))
 {
	if (note < 35 || note > 81)
 return NULL;
	instrnumber = note + (128-35);
 } else
	instrnumber = ((struct OPLdata *)mus->driverdata)->channelInstr[channel];

 if (OPLinstruments)
	return &OPLinstruments[instrnumber];
 else
	return NULL;
}
void OPLplayNote(struct musicBlock *mus, uint channel, uchar note, int volume)
{
 int i;
 struct OP2instrEntry *instr;

 if ( (instr = getInstrument(mus, channel, note)) == NULL )
	return;

 if ( (i = findFreeChannel((channel == PERCUSSION) ? 2 : 0)) != -1)
 {
	occupyChannel(mus, i, channel, note, volume, instr, 0);
	if (!OPLsinglevoice && instr->flags == FL_DOUBLE_VOICE)
	{
 if ( (i = findFreeChannel((channel == PERCUSSION) ? 3 : 1)) != -1)
		occupyChannel(mus, i, channel, note, volume, instr, 1);
	}
 }
}
void OPLreleaseNote(struct musicBlock *mus, uint channel, uchar note)
{
 uint i;
 uint id = MAKE_ID(channel, mus->number);
 struct OPLdata *data = (struct OPLdata *)mus->driverdata;
 uint sustain = data->channelSustain[channel];

 for(i = 0; i < OPLchannels; i++)
	if (CHANNEL_ID(channels[i]) == id && channels[i].note == note)
	{
 if (sustain < 0x40)
		releaseChannel(i, 0);
 else
		channels[i].flags |= CH_SUSTAIN;
	}
}
void OPLpitchWheel(struct musicBlock *mus, uint channel, int pitch)
{
 uint i;
 uint id = MAKE_ID(channel, mus->number);
 struct OPLdata *data = (struct OPLdata *)mus->driverdata;

 data->channelPitch[channel] = pitch;
 for(i = 0; i < OPLchannels; i++)
 {
	struct channelEntry *ch = &channels[i];
	if (CHANNEL_ID(*ch) == id)
	{
 ch->time = MLtime;
 ch->pitch = ch->finetune + pitch;
 writeFrequency(i, ch->realnote, ch->pitch, 1);
	}
 }
}
void OPLchangeControl(struct musicBlock *mus, uint channel, uchar controller, int value)
{
 uint i;
 uint id = MAKE_ID(channel, mus->number);
 struct OPLdata *data = (struct OPLdata *)mus->driverdata;

 switch (controller) {
	case ctrlPatch:
 data->channelInstr[channel] = value;
 break;
	case ctrlModulation:
 data->channelModulation[channel] = value;
 for(i = 0; i < OPLchannels; i++)
 {
		struct channelEntry *ch = &channels[i];
		if (CHANNEL_ID(*ch) == id)
		{
 uchar flags = ch->flags;
 ch->time = MLtime;
 if (value >= MOD_MIN)
 {
			ch->flags |= CH_VIBRATO;
			if (ch->flags != flags)
 writeModulation(i, ch->instr, 1);
 } else {
			ch->flags &= ~CH_VIBRATO;
			if (ch->flags != flags)
 writeModulation(i, ch->instr, 0);
 }
		}
 }
 break;
	case ctrlVolume:
 data->channelVolume[channel] = value;
 for(i = 0; i < OPLchannels; i++)
 {
		struct channelEntry *ch = &channels[i];
		if (CHANNEL_ID(*ch) == id)
		{
 ch->time = MLtime;
 ch->realvolume = calcVolume(value, mus->volume, ch->volume);
 OPLwriteVolume(i, ch->instr, ch->realvolume);
		}
 }
 break;
	case ctrlPan:
 data->channelPan[channel] = value -= 64;
 for(i = 0; i < OPLchannels; i++)
 {
		struct channelEntry *ch = &channels[i];
		if (CHANNEL_ID(*ch) == id)
		{
 ch->time = MLtime;
 OPLwritePan(i, ch->instr, value);
		}
 }
 break;
	case ctrlSustainPedal:
 data->channelSustain[channel] = value;
 if (value < 0x40)
		releaseSustain(mus, channel);
 break;
 }
}
void OPLplayMusic(struct musicBlock *mus)
{
 uint i;
 struct OPLdata *data = (struct OPLdata *)mus->driverdata;

 for (i = 0; i < CHANNELS; i++)
 {
	data->channelVolume[i] = 127;
	data->channelSustain[i] = data->channelLastVolume[i] = 0;
 }
}
void OPLstopMusic(struct musicBlock *mus)
{
 uint i;
 for(i = 0; i < OPLchannels; i++)
	if (channels[i].musnumber == mus->number && !(channels[i].flags & CH_FREE))
 releaseChannel(i, -1);
}
void OPLchangeVolume(struct musicBlock *mus, uint volume)
{
 uchar *channelVolume = ((struct OPLdata *)mus->driverdata)->channelVolume;
 uint i;
 for(i = 0; i < OPLchannels; i++)
 {
	struct channelEntry *ch = &channels[i];
	if (ch->musnumber == mus->number)
	{
 ch->realvolume = calcVolume(channelVolume[ch->channel & 0xF], volume, ch->volume);
 if (mus->state == ST_PLAYING)
		OPLwriteVolume(i, ch->instr, ch->realvolume);
	}
 }
}
void OPLpauseMusic(struct musicBlock *mus)
{
 uint i;
 for(i = 0; i < OPLchannels; i++)
 {
	struct channelEntry *ch = &channels[i];
	if (ch->musnumber == mus->number)
	{
 struct OPL2instrument *instr = ch->instr;
 if (OPL3mode)
		OPLwriteValue(0xC0, i, instr->feedback);
 OPLwriteVolume(i, instr, 0);
 OPLwriteChannel(0x60, i, 0, 0);
 OPLwriteChannel(0x80, i, instr->sust_rel_1 & 0xF0,
		instr->sust_rel_2 & 0xF0);
	}
 }
}
void OPLunpauseMusic(struct musicBlock *mus)
{
 struct OPLdata *data = (struct OPLdata *)mus->driverdata;
 uint i;
 for(i = 0; i < OPLchannels; i++)
 {
	struct channelEntry *ch = &channels[i];
	if (ch->musnumber == mus->number)
	{
 struct OPL2instrument *instr = ch->instr;

 OPLwriteChannel(0x60, i, instr->att_dec_1, instr->att_dec_2);
 OPLwriteChannel(0x80, i, instr->sust_rel_1, instr->sust_rel_2);
 OPLwriteVolume(i, instr, ch->realvolume);
 if (OPL3mode)
		OPLwritePan(i, instr, data->channelPan[ch->channel & 0xF]);
	}
 }
}
int OPLinitDriver(void)
{
 memset(channels, 0xFF, sizeof channels);
 OPLinstruments = NULL;
 return 0;
}
int OPLdeinitDriver(void)
{
 free(OPLinstruments);
 OPLinstruments = NULL;
 return 0;
}
int OPLdriverParam(uint message, uint param1,void *param2)
{
 return 0;
}
int OPLloadBank(int fd)
{
 static uchar masterhdr[8] = "#OPL_II#";
 uchar hdr[8];
 struct OP2instrEntry *instruments;
 if (read(fd, &hdr, sizeof hdr) != sizeof hdr)
 return -1;
 if (memcmp(hdr, masterhdr, sizeof hdr))
 return -2;
 if ( (instruments = (struct OP2instrEntry *)calloc(OP2INSTRCOUNT, OP2INSTRSIZE)) == NULL)
 return -3;
 if (read(fd, instruments, OP2INSTRSIZE * OP2INSTRCOUNT) != OP2INSTRSIZE * OP2INSTRCOUNT)
 {
	free(instruments);
 return -1;
 }
 free(OPLinstruments);
 OPLinstruments = instruments;
 return 0;
}
int OPL2detectHardware(uint port) { return OPL2detect(port); }
int OPL3detectHardware(uint port) { return OPL3detect(port); }
int OPL2initHardware(uint port) { OPLinit(port, 0); return 0; }
int OPL3initHardware(uint port) { OPLinit(port, 1); return 0; }
int OPL2deinitHardware() { OPLdeinit(); return 0; }
int OPL3deinitHardware() { OPLdeinit(); return 0; }
int OPLsendMIDI() { return 0; }
