#include "system.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <math.h>
#include <limits.h>
#include <assert.h>
#include <dos.h>
#include "blockmap.h"
#include "level.h"
#include "node.h"
#include "seg.h"
#include "structs.h"
#include "util.h"
#include "wad.h"
#include "glbsp.h"

char message_buf[1024];
void FatalError(const char *str, ...)
{
  va_list args;
  va_start(args, str);
  vsprintf(message_buf, str, args);
  va_end(args);
  (* cur_funcs->fatal_error)("\nError: %s", message_buf);
}
void PrintMsg(const char *str, ...)
{
  va_list args;
  va_start(args, str);
  vsprintf(message_buf, str, args);
  va_end(args);
  (* cur_funcs->print_msg)("%s", message_buf);
}
void TextPrintMsg(const char *str, ...)
{
  va_list args;
  va_start(args, str);
  vprintf(str, args);
  va_end(args);
  fflush(stdout);
}
void TextFatalError(const char *str, ...)
{
  va_list args;
  va_start(args, str);
  vfprintf(stderr, str, args);
  va_end(args);
  exit(5);
}

const nodebuildfuncs_t cmdline_funcs =
{
  TextFatalError,
  TextPrintMsg
};
void TextStartup(void)
{
  setbuf(stdout, NULL);
}

const nodebuildinfo_t *cur_info = NULL;
const nodebuildfuncs_t *cur_funcs = NULL;
volatile nodebuildcomms_t *cur_comms = NULL;

static char glbsp_message_buf[1024];

const nodebuildinfo_t default_buildinfo =
{
  NULL,NULL,
  DEFAULT_FACTOR,
  0,0,0,0
};

const nodebuildcomms_t default_buildcomms =
{
  NULL,0
};

void SetMessage(const char *str)
{
  GlbspFree(cur_comms->message);

  cur_comms->message = GlbspStrDup(str);
}

#define HANDLE_BOOLEAN(name, field)  \
    if (StrCaseCmp(opt_str, name) == 0)  \
    {  \
      info->field = TRUE;  \
      argv++; argc--;  \
      continue;  \
    }

glbsp_ret_e GlbspParseArgs(nodebuildinfo_t *info, 
    volatile nodebuildcomms_t *comms,
    const char ** argv, int argc)
{
  const char *opt_str;
  int num_files = 0;

  cur_comms = comms;
  SetMessage(NULL);

  while (argc > 0)
  {
    if (argv[0][0] != '-')
    {
      if (num_files >= 1)
      {
        SetMessage("Too many filenames.  Use the -o option");
        cur_comms = NULL;
        return GLBSP_E_BadArgs;
      }

      GlbspFree(info->input_file);
      info->input_file = GlbspStrDup(argv[0]);
      num_files++;

      argv++; argc--;
      continue;
    }

    opt_str = &argv[0][1];
    if (opt_str[0] == '-')
      opt_str++;

    if (StrCaseCmp(opt_str, "o") == 0)
    {
      if (argc < 2)
      {
        SetMessage("Missing filename for the -o option");
        cur_comms = NULL;
        return GLBSP_E_BadArgs;
      }

      GlbspFree(info->output_file);
      info->output_file = GlbspStrDup(argv[1]);

      argv += 2; argc -= 2;
      continue;
    }

    if (StrCaseCmp(opt_str, "factor") == 0)
    {
      if (argc < 2)
      {
        SetMessage("Missing factor value");
        cur_comms = NULL;
        return GLBSP_E_BadArgs;
      }

      info->factor = (int) strtol(argv[1], NULL, 10);

      argv += 2; argc -= 2;
      continue;
    }

    HANDLE_BOOLEAN("loadall", load_all)
    HANDLE_BOOLEAN("build", force_normal)

    sprintf(glbsp_message_buf, "Unknown option: %s", argv[0]);
    SetMessage(glbsp_message_buf);

    cur_comms = NULL;
    return GLBSP_E_BadArgs;
  }

  cur_comms = NULL;
  return GLBSP_E_OK;
}

glbsp_ret_e GlbspCheckInfo(nodebuildinfo_t *info,
    volatile nodebuildcomms_t *comms)
{
  cur_comms = comms;
  SetMessage(NULL);

  info->same_filenames = FALSE;
  info->missing_output = FALSE;

  if (!info->input_file || info->input_file[0] == 0)
  {
    SetMessage("Missing input filename !");
    return GLBSP_E_BadArgs;
  }

  if (!info->output_file || info->output_file[0] == 0)
  {
    GlbspFree(info->output_file);
    info->output_file = "Temp.wad";
  }

  if (StrCaseCmp(info->input_file, info->output_file) == 0)
  {
    info->load_all = 1;
    info->same_filenames = TRUE;
  }

  if (info->factor <= 0)
  {
    info->factor = DEFAULT_FACTOR;
    SetMessage("Bad factor value");
    return GLBSP_E_BadInfoFixed;
  }
  return GLBSP_E_OK;
}

const char *GlbspStrDup(const char *str)
{
  if (! str)
    return NULL;

  return UtilStrDup(str);
}

void GlbspFree(const char *str)
{
  if (! str)
    return;

  UtilFree((char *) str);
}
extern boolean_g doing_normal;
glbsp_ret_e HandleLevel(void)
{
  superblock_t *seg_list;
  node_t *root_node;
  subsec_t *root_sub;
  glbsp_ret_e ret=0;

  LoadLevel();
  if (doing_normal)
 {  InitBlockmap(), seg_list = CreateSegs();
    ret = BuildNodes(seg_list, &root_node, &root_sub, 0); FreeSuper(seg_list);
    if (ret == GLBSP_E_OK)
    {
    ClockwiseBspTree(root_node);

    PrintMsg("Built %d NODES, %d SSECTORS, %d SEGS, %d VERTEXES\n",
        num_nodes, num_subsecs, num_segs, num_normal_vert);

    if (root_node)
      PrintMsg("Heights of left and right subtrees = (%d,%d)\n",
          ComputeHeight(root_node->r.node), ComputeHeight(root_node->l.node));

    SaveLevel(root_node);
    }
 } else  PutSidedefs(), PutLinedefs();
  FreeLevel();
  FreeQuickAllocCuts();
  FreeQuickAllocSupers();

  return ret;
}

glbsp_ret_e GlbspBuildNodes(const nodebuildinfo_t *info,
    const nodebuildfuncs_t *funcs, volatile nodebuildcomms_t *comms)
{
  char strbuf[256];

  glbsp_ret_e ret = GLBSP_E_OK;

  cur_info  = info;
  cur_funcs = funcs;
  cur_comms = comms;

  comms->cancelled = FALSE;

  if (!cur_info->input_file  || cur_info->input_file[0] == 0 ||
      !cur_info->output_file || cur_info->output_file[0] == 0)
  {
    SetMessage("ERROR: Missing in/out filename");
    return GLBSP_E_BadArgs;
  }
  
  if (info->missing_output)
    PrintMsg("No output file specified. Using: %s\n\n", info->output_file);
  
  if (info->same_filenames)
    PrintMsg("Output file is same as input file. Using -loadall\n\n");

  ret = ReadWadFile(cur_info->input_file);

  if (ret != GLBSP_E_OK)
  {
    return ret;
  }

  if (CountLevels() <= 0)
  {
    CloseWads();
    SetMessage("No levels found in wad");
    return GLBSP_E_Unknown;
  }

  PrintMsg("\nCreating nodes using factor %d\n", info->factor);

  sprintf(strbuf, "File: %s", cur_info->input_file);

  while (FindNextLevel())
  {
    ret = HandleLevel();

    if (ret != GLBSP_E_OK)
      break;
  }

  if (ret == GLBSP_E_OK)
    ret = WriteWadFile(cur_info->output_file);

  CloseWads();
  cur_info  = NULL;
  cur_comms = NULL;
  cur_funcs = NULL;
  return ret;
}

nodebuildinfo_t info;
volatile nodebuildcomms_t comms;
int main(int argc, char **argv)
{
  TextPrintMsg(
    "Use: CDoombsp [options] input.wad -o output.wad\n"
    "Options:\n"
    "  -factor <n> Change cost assigned to SEG splits\n"
    "  -build      Build nodes to replace the present ones\n"
    "  otherwise only pack sidedefs\n"
    "\n"
  );
  argv++, argc--;
  if (argc <= 0)
    exit(1);
  info  = default_buildinfo;
  comms = default_buildcomms;
  if (GLBSP_E_OK != GlbspParseArgs(&info, &comms, 
      (const char **)argv, argc))
  {
    TextFatalError("Error: %s\n", comms.message ? comms.message : 
        "(Unknown error when parsing args)");
  }
  if (GLBSP_E_OK != GlbspCheckInfo(&info, &comms)) 
  {
    TextFatalError("Error: %s\n", comms.message ? comms.message : 
        "(Unknown error when checking args)");
  }
  if (GLBSP_E_OK != GlbspBuildNodes(&info, &cmdline_funcs, &comms))
  {
    TextFatalError("Error: %s\n", comms.message ? comms.message : 
        "(Unknown error during build)");
  }
  return 0;
}
