/*
   console.c - graphical console support functions
   by Bruce Lewis
*/

#include <windows.h>
#include <stdio.h>
#include <stdarg.h>
#include "wcedoom.h"

#ifndef _WIN32_WCE
#define wcscpy strcpy
#define wcslen strlen
#endif

#define MAX_MESSAGE  256              // size of message loop
#define MSG_MASK     0xFF             // mask for recycling the write line

typedef struct
   {
    int            cxChar;
    int            cxCaps;
    int            cyChar;
    int            cyLines;
    int            cxChars;
    unsigned char *console;
    int            width;           // console window width
    int            height;          // console window height
    HWND           hWnd;            // Window handle of main window
    HDC            hDC;             // screen DC
    HBITMAP        hBitmap;         // console bitmap for GDI
    HDC            hMemDC;          // console memory DC
   }condata_t;

condata_t            ConData;         // console font information

BOOL                 con_surface = FALSE;  // is console initialized yet?
HPEN                 ltgray_pen;           // for drawing same color border with rectangle

TCHAR                msgloop[MAX_MESSAGE][256];  // message loop buffer
unsigned int         ml_head = 0, ml_draw = 0;   // current message loop draw line
int                  ccp = 0;                    // current message loop write character position

void con_draw()
   {
    int          i, lines;
    unsigned int ml_tail;
    RECT         drect;

    if (!con_surface)
       {
        return;
       }

    if (ml_head < ConData.cyLines)
       {
        lines = ml_head;
       }
    else
       {
        lines = ConData.cyLines;
       }
    SelectObject(ConData.hMemDC, ConData.hBitmap);
    SelectObject(ConData.hMemDC, GetStockObject(LTGRAY_BRUSH));
    SelectObject(ConData.hMemDC, ltgray_pen);
#ifndef _WIN32_WCE
    SelectObject(ConData.hMemDC, GetStockObject(SYSTEM_FIXED_FONT));
#else
    SelectObject(ConData.hMemDC, GetStockObject(SYSTEM_FONT));
#endif
    Rectangle(ConData.hMemDC, 0, 0, ConData.width, ConData.height);
    SetBkColor(ConData.hMemDC, RGB(191, 191, 191));
    SetTextColor(ConData.hMemDC, RGB(0, 0, 0));
    drect.right = ConData.width;
    drect.left = 0;
    for (ml_tail = ml_head-lines, i = 0; ml_tail < ml_head; ml_tail++, i++)
       {
        drect.top = ConData.cyChar*i;
        drect.bottom = drect.top + ConData.cyChar;
        DrawText(ConData.hMemDC, msgloop[ml_tail], wcslen(msgloop[ml_tail]), &drect, DT_LEFT | DT_NOCLIP);
       }
    BitBlt(ConData.hDC, 0, 0, ConData.width, ConData.height, ConData.hMemDC, 0, 0, SRCCOPY);
   }

void con_printf( TCHAR *message, ... )
   {
    va_list	argptr;
    int     i, bc;
    TCHAR   tstr[1024];

    if (!con_surface)
       {
        return;
       }

    if (ml_head < 0)
       {
        ml_head = 0;
       }

    va_start(argptr,message);
#ifdef _WIN32_WCE
    wvsprintf (tstr,message,argptr);
#else
    vsprintf (tstr,message,argptr);
#endif
    va_end(argptr);

    if (wcslen(tstr) > ConData.cxChars)
       {
        bc = ConData.cxChars;
        while ((tstr[bc--] != TEXT(' ')) && (bc > 0));
        if (bc == 0)
           {
            bc = ConData.cxChars;
           }
       }
    else
       {
        bc = 0xFFFF;
       }

    for (i = 0; i < wcslen(tstr); i++)
       {
        switch(tstr[i])
           {
            case TEXT('\n'): // handle "newline" chars
                 //msgloop[ml_head&MSG_MASK][ccp] = TEXT('\0');
                 ml_head++;
                 ccp = 0;
                 if (wcslen(&tstr[i]) > ConData.cxChars)
                    {
                     bc = i+ConData.cxChars;
                     while ((tstr[bc--] != TEXT(' ')) && (bc > i));
                     if (bc == i)
                        {
                         bc = i+ConData.cxChars;
                        }
                    }
                 else
                    {
                     bc = 0xFFFF;
                    }
                 memset(msgloop[ml_head&MSG_MASK], 0, 256);
                 break;
            case TEXT('\b'): // handle "back-space" chars
                 if (ccp > 0)
                    {
                     ccp--;
                    }
                 break;
            default:
                 if (bc <= i)
                    {
                     ml_head++;
                     ccp = 0;
                     if (wcslen(&tstr[i]) > ConData.cxChars)
                        {
                         bc = i+ConData.cxChars;
                         while ((tstr[bc--] != TEXT(' ')) && (bc > i));
                         if (bc == i)
                            {
                             bc = i+ConData.cxChars;
                            }
                        }
                     else
                        {
                         bc = 0xFFFF;
                        }
                    }
                 msgloop[ml_head&MSG_MASK][ccp] = tstr[i];
                 ccp++;
                 break;
           }
       }
    if (con_surface)
       {
        con_draw();
       }
    WriteDebug(StrFromUnicode(tstr));
   }

void con_clear()
   {
    SelectObject(ConData.hMemDC, ConData.hBitmap);
    SelectObject(ConData.hMemDC, GetStockObject(LTGRAY_BRUSH));
    SelectObject(ConData.hMemDC, ltgray_pen);
    Rectangle(ConData.hMemDC, 0, 0, ConData.width, ConData.height);
    BitBlt(ConData.hDC, 0, 0, ConData.width, ConData.height, ConData.hMemDC, 0, 0, SRCCOPY);
   }

void con_setup(HWND hwnd, int width, int height)
   {
    HDC             hdc;
    TEXTMETRIC      tm;
    int             y;

    ConData.hWnd = hwnd;
    ConData.width = width;
    ConData.height = height;
    hdc = GetDC (hwnd);
    ConData.hDC = hdc;
    GetTextMetrics(hdc, &tm);
#ifndef _WIN32_WCE
    GetStockObject(SYSTEM_FIXED_FONT);
#else
    GetStockObject(SYSTEM_FONT);
#endif
    ConData.cxChar = tm.tmAveCharWidth;
    ConData.cxCaps = (tm.tmPitchAndFamily & 1 ? 3 : 2) * ConData.cxChar / 2;
    ConData.cyChar = tm.tmHeight + tm.tmExternalLeading;
    ConData.cyLines = ConData.height/ConData.cyChar;
#ifndef _WIN32_WCE
    ConData.cxChars = ConData.width/ConData.cxChar;
#else
    ConData.cxChars = ConData.width/ConData.cxCaps;
#endif
    ConData.hBitmap = CreateCompatibleBitmap(hdc, ConData.width, ConData.height);
    ConData.hMemDC = CreateCompatibleDC(hdc);
    ltgray_pen = CreatePen(PS_SOLID, 1, RGB(191,191,191));
    con_surface = TRUE;
    for (y = 0; y < MAX_MESSAGE; y++)
       {
        memset(msgloop[y], 0, 256);
       }
   }

void con_shutdown()
   {
    if (ltgray_pen != NULL)
       {
        SelectObject(ConData.hMemDC, GetStockObject(BLACK_PEN));
        DeleteObject(ltgray_pen);
        ltgray_pen = NULL;
       }
    ReleaseDC (ConData.hWnd, ConData.hDC);
   }
