//
// Copyright(C) 1999,2007 Simon Howard
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307, USA.
//

/***************************** FraggleScript ******************************/
                // Copyright(C) 1999 Simon Howard 'Fraggle' //
//
// Parsing.
//
// Takes lines of code, or groups of lines and runs them.
//

/* includes ************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include "main.h"
#include "parse.h"
#include "prepro.h"
#include "spec.h"
#include "operator.h"
#include "variable.h"
#include "func.h"

void run_script();
svalue_t evaluate_expression(int start, int stop);

char *tokens[MAXTOKENS];
tokentype_t tokentype[MAXTOKENS];
int num_tokens = 0;
int script_debug = false;

script_t *current_script;       // the current script
svalue_t nullvar = { vt_int,  {0} };      // null var for empty return
int killscript;         // when set to true, stop the script quickly
section_t *prev_section;       // the section from the previous statement

/************ Divide into tokens **************/

char *linestart;        // start of line
char *rover;            // current point reached in script

        // inline for speed
#define isnum(c) ( (c)>='0' && (c)<='9' )
        // isop: is an 'operator' character, eg '=', '%'
#define isop(c)   !( ( (c)<='Z' && (c)>='A') || ( (c)<='z' && (c)>='a') || \
                     ( (c)<='9' && (c)>='0') || ( (c)=='_') )

        // for simplicity:
#define tt (tokentype[num_tokens-1])
#define tok (tokens[num_tokens-1])

section_t *current_section; // the section (if any) found in parsing the line
int bracetype;              // bracket_open or bracket_close
void add_char(char c);

// next_token: end this token, go onto the next

void next_token()
{
        if(tok[0] || tt==string)
        {
                num_tokens++;
                tokens[num_tokens-1] = tokens[num_tokens-2]
                                 + strlen(tokens[num_tokens-2]) + 1;
                tok[0] = 0;
        }

                // get to the next token, ignoring spaces, newlines,
                // useless chars, comments etc
        while(1)
        {
                        // empty whitespace
                if(*rover && (*rover==' ' || *rover<32))
                {
                    while((*rover==' ' || *rover<32) && *rover) rover++;
                }
                        // end-of-script?
                if(!*rover)
                {
                        if(tokens[0][0])
                        {
                                // line contains text, but no semicolon
                                script_error("missing ';'");
                        }
                                // empty line, end of command-list
                        return;
                }
                        // 11/8 comments moved to new preprocessor

                break;  // otherwise
        }

        if(*rover == '{' || *rover == '}')
        {
                if(*rover == '{')
                {
                    bracetype = bracket_open;
                    current_section = find_section_start(rover);
                }
                else            // closing brace
                {
                    bracetype = bracket_close;
                    current_section = find_section_end(rover);
                }
                if(!current_section)
                {
                        script_error("section not found!");
                        return;
                }
        }
        else if(*rover == '\"')
        {
                tt = string;
                if(tokentype[num_tokens-2] == string)
                        num_tokens--;   // join strings
                rover++;
        }
        else
        {
                tt = isop(*rover) ? operator :
                     isnum(*rover) ? number : name;
        }
}

// return an escape sequence (prefixed by a '\')
// do not use all C escape sequences

char escape_sequence(char c)
{
        if(c == 'n') return '\n';
        if(c == '\\') return '\\';
        if(c == '"') return '"';
        if(c == '?') return '?';
        if(c == 'a') return '\a'; // alert beep
        if(c == 't') return '\t'; //tab

        return c;
}

// add_char: add one character to the current token

void add_char(char c)
{
        char *out = tok + strlen(tok);

//        printf("%c",c); fflush(stdout);

        *out++ = c;
        *out = 0;
}

        // second stage parsing.
        // go through the tokens and mark brackets as brackets
        // also mark function names, which are name tokens
        // followed by open brackets

void find_brackets()
{
        int i;

        for(i=0; i<num_tokens; i++)
        {
            if(tokentype[i] == operator)
            {
               if(tokens[i][0] == '(')
               {
                  tokentype[i] = bracket_open;
                  if(i && tokentype[i-1] == name)
                    tokentype[i-1] = function;
               }
               else
                  if(tokens[i][0] == ')')
                    tokentype[i] = bracket_close;
            }
        }
}

// get_tokens.
// Take a string, break it into tokens.

        // individual tokens are stored inside the tokens[] array
        // which is created by run_line. tokentype is also used
        // to hold the type for each token:

        //   name: a piece of text which starts with an alphabet letter.
        //         probably a variable name. Some are converted into
        //         function types later on in find_brackets
        //   number: a number. like '12' or '1337'
        //   operator: an operator such as '&&' or '+'. All FraggleScript
        //             operators are either one character, or two character
        //             (if 2 character, 2 of the same char or ending in '=')
        //   string: a text string that was enclosed in quote "" marks in
        //           the original text
        //   unset: shouldn't ever end up being set really.
        //   bracket_open, bracket_close : open and closing () brackets
        //   function: a function name (found in second stage parsing)


void get_tokens(char *s)
{
        rover = s;
        num_tokens = 1;
        tokens[0][0] = 0; tt = name;

        current_section = NULL;   // default to no section found

        next_token();
        linestart = rover;      // save the start

        if(*rover)
        while(1)
        {
             if(killscript) return;
             if(current_section)
             {
                        // a { or } section brace has been found
                  break;        // stop parsing now
             }
             else if(tt != string)
             {
                if(*rover == ';') break;     // check for end of command ';'
             }

             switch(tt)
             {
                  case unset:
                  case string:
                    while(*rover != '\"')     // dedicated loop for speed
                    {
                       if(*rover == '\\')       // escape sequences
                       {
                            rover++;
                            add_char(escape_sequence(*rover));
                       }
                       else
                            add_char(*rover);
                       rover++;
                    }
                    rover++;
                    next_token();       // end of this token
                    continue;

                  case operator:
                        // all 2-character operators either end in '=' or
                        // are 2 of the same character
                        // do not allow 2-characters for brackets '(' ')'
                        // which are still being considered as operators

                        // operators are only 2-char max, do not need
                        // a seperate loop

                    if((*tok && *rover != '=' && *rover!=*tok) ||
                        *tok == '(' || *tok == ')')
                    {
                          // end of operator
                        next_token();
                        continue;
                    }
                    add_char(*rover);
                    break;

                  case number:  // same for number or name
                  case name:
                                // add the chars
                    while(!isop(*rover))        // dedicated loop
                            add_char(*rover++);
                    next_token();
                    continue;
                  default:
                    break;
             }
             rover++;
        }
        
        //        // check for empty last token
        if(!tok[0])
        {
                num_tokens = num_tokens - 1;
        }

                        // now do second stage parsing
        find_brackets();

        rover++;
}

        // time_script: same as parse_script
        // but timed
void time_script()
{
        int timeseconds = 4;
        int timecount = 0;
        clock_t timeclock;

        timeclock = clock() + CLOCKS_PER_SEC*timeseconds;

        while(timeclock > clock())
        {
//                printf("%i\n", timecount);
                run_script();
                timecount++;
        }

        printf("%i per second\n", timecount/timeseconds);
}

void print_tokens()	// DEBUG
        {
            int i;
            for(i=0; i<num_tokens; i++)
            {
                printf("\n'%s' \t\t --", tokens[i]);
                switch(tokentype[i])
                {
                    case string: printf("string");        break;
                    case operator: printf("operator");    break;
                    case name: printf("name");            break;
                    case number: printf("number");        break;
                    case unset : printf("duh");           break;
                    case bracket_open: printf("open bracket"); break;
                    case bracket_close: printf("close bracket"); break;
                    case function: printf("function name"); break;
                }
            }
            printf("\n");
            if(current_section)
                printf("current section: offset %i\n",
                       (int)(current_section->start-current_script->data) );
        }


        // parse_script
        //
        // the function called in main.c

void parse_script(script_t *script)
{
        current_script = script;

        if(0)   // for timing scripts
                time_script();
        else
                run_script();
}

void run_script()
{
        char *token_alloc;      // allocated memory for tokens

        killscript = false;     // dont kill the script straight away
        rover = current_script->data;   // start at the beginning of the script

                // allocate space for the tokens
        token_alloc = malloc(current_script->len + MAXTOKENS);

        prev_section = NULL;  // clear it

        while(*rover)   // go through the script executing each statement
        {
                        // reset the tokens before getting the next line
                tokens[0] = token_alloc;

                prev_section = current_section; // store from prev. statement

                        // get the line and tokens
                get_tokens(rover);
  
                if(killscript) break;

                if(!num_tokens)
                {
                        if(current_section)       // no tokens but a brace
                        {
                                   // possible } at end of loop:
                                   // refer to spec.c
                                spec_brace();
                        }

                        continue;  // continue to next statement
                }

                if(script_debug) print_tokens();   // debug
                run_statement();         // run the statement
        }
        free(token_alloc);

}

void run_statement()
{
        // decide what to do with it

        // NB this stuff is a bit hardcoded:
        //    it could be nicer really but i'm
        //    aiming for speed

                // if() and while() will be mistaken for functions
                // during token processing
        if(tokentype[0] == function)
        {
                if(!strcmp(tokens[0], "if"))
                {
                    spec_if();
                    return;
                }
                else if(!strcmp(tokens[0], "while"))
                {
                    spec_while();
                    return;
                }
                else if(!strcmp(tokens[0], "for"))
                {
                    spec_for();
                    return;
                }
        }
        else if(tokentype[0] == name)
        {
                if(!strcmp(tokens[0], "int"))
                {
                    spec_int();
                    return;
                }
                else if(!strcmp(tokens[0], "string"))
                {
                    spec_string();
                    return;
                }
                // goto is handled as a function

                // NB "float" or other types could be added here
        }

                // just a plain expression
        evaluate_expression(0, num_tokens-1);
}

/***************** Evaluating Expressions ************************/

        // find a token, ignoring things in brackets        
int find_token(int start, int stop, char *value)
{
        int i;
        int bracketlevel = 0;

        for(i=start; i<=stop; i++)
        {
                        // use bracketlevel to check the number of brackets
                        // which we are inside
                bracketlevel += tokentype[i]==bracket_open ? 1 :
                                tokentype[i]==bracket_close ? -1 : 0;

                        // only check when we are not in brackets
                if(!bracketlevel && !strcmp(value, tokens[i]))
                        return i;
        }

        return -1;
}

        // go through tokens the same as find_token, but backwards
int find_token_backwards(int start, int stop, char *value)
{
        int i;
        int bracketlevel = 0;

        for(i=stop; i>=start; i--)      // check backwards
        {
                        // use bracketlevel to check the number of brackets
                        // which we are inside
                bracketlevel += tokentype[i]==bracket_open ? 1 :
                                tokentype[i]==bracket_close ? -1 : 0;

                        // only check when we are not in brackets
                if(!bracketlevel && !strcmp(value, tokens[i]))
                        return i;
        }

        return -1;
}

// simple_evaluate is used once evalute_expression gets to the level
// where it is evaluating just one token

// converts number tokens into svalue_ts and returns
// the same with string tokens
// name tokens are considered to be variables and
// attempts are made to find the value of that variable
// command tokens are executed (does not return a svalue_t)

extern svalue_t nullvar;

static svalue_t simple_evaluate(int n)
{
        svalue_t returnvar;
        svariable_t *var;

        switch(tokentype[n])
        {
            case string: returnvar.type = vt_string;
                         returnvar.value.s = tokens[n];
                         return returnvar;
            case number: returnvar.type = vt_int;
                         returnvar.value.i = atoi(tokens[n]);
                         return returnvar;
            case name:   var = find_variable(tokens[n]);
                         if(!var)
                         {
                            script_error("unknown variable '%s'", tokens[n]);
                            return nullvar;
                         }
                         else
                            return getvariablevalue(var);
            default: return nullvar;
        }
}

// pointless_brackets checks to see if there are brackets surrounding
// an expression. eg. "(2+4)" is the same as just "2+4"
//
// because of the recursive nature of evaluate_expression, this function is
// neccesary as evaluating expressions such as "2*(2+4)" will inevitably
// lead to evaluating "(2+4)"

static void pointless_brackets(int *start, int *stop)
{
        int bracket_level, i;

                // check that the start and end are brackets

        while(tokentype[*start] == bracket_open &&
               tokentype[*stop] == bracket_close)
        {

            bracket_level = 0;

                  // confirm there are pointless brackets..
                  // if they are, bracket_level will only get to 0
                  // at the last token
                  // check up to <*stop rather than <=*stop to ignore
                  // the last token
            for(i = *start; i<*stop; i++)
            {
                bracket_level += (tokentype[i] == bracket_open);
                bracket_level -= (tokentype[i] == bracket_close);
                if(bracket_level == 0) return;
            }

                // move both brackets in

            *start = *start + 1;
            *stop = *stop - 1;
        }
}

// evaluate_expresion is the basic function used to evaluate
// a FraggleScript expression.
// start and stop denote the tokens which are to be evaluated.
//
// works by recursion: it finds operators in the expression
// (checking for each in turn), then splits the expression into
// 2 parts, left and right of the operator found.
// The handler function for that particular operator is then
// called, which in turn calls evaluate_expression again to
// evaluate each side. When it reaches the level of being asked
// to evaluate just 1 token, it calls simple_evaluate

svalue_t evaluate_expression(int start, int stop)
{
        int n, count;

        if(killscript) return nullvar;  // killing the script

        // debug
//        printf("evaluate_expression(%i, %i)\n", start, stop);

                // possible pointless brackets
        if(tokentype[start] == bracket_open
        && tokentype[stop] == bracket_close)
                pointless_brackets(&start, &stop);

        if(start == stop)       // only 1 thing to evaluate
        {
                return simple_evaluate(start);
        }

        // go through each operator in order of precedence

        for(count=0; count<num_operators; count++)
        {
                // check backwards for the token. it has to be
                // done backwards for left-to-right reading: eg so
                // 5-3-2 is (5-3)-2 not 5-(3-2)
           if( -1 != (n =
                (operators[count].direction==forward ?
                        find_token_backwards : find_token)
                (start, stop, operators[count].string)) )
           {
                // useful for debug:
//                printf("operator %s: %i,%i,%i\n",
//                        operators[count].string, start, n, stop);

                        // call the operator function
                        // and evaluate this chunk of code
                return operators[count].handler(start, n, stop);
           }
        }

        if(tokentype[start] == function)
                return evaluate_function(start, stop);

        // error ?
        script_error("couldnt evaluate expression (tokens %i-%i)", start, stop);
        return nullvar;
}

void script_error(char *s, ...)
{
        va_list args;
        va_start(args, s);

        if(killscript) return;  //already killing script

        printf("\n\n");

                // find the line number
        {
                int linenum = 1;
                char *temp;
                for(temp = current_script->data; temp<linestart; temp++)
                        if(*temp == '\n') linenum++;    // count EOLs
                printf("%i: ", linenum); fflush(stdout);
        }

                // print the error
        vfprintf(stderr, s, args);

        killscript = 1;
}
