// Internal Node builder for DoomEd 4.0
// 
// Copyright  1995 by Geoff Allan
// All Rights Reserved. Unauthorised distribution of this source
// is a violation of Canadian and International Copyright laws.
//
// Loosely based on the Verda Node Builder by Robert Fenske, Jr.
//
// Note: only BspBuild is declared in prototyp.h

#include "DoomEd40.hpp"
#include <math.h>
#undef abs
#include "nodebldr.h"

HDC     OurDC;

#define BspBranchTree(c)  ((c) = blockmem(BSPNodeTree, 1),  \
                           (c)->left = NULL,                \
                           (c)->right = NULL, (c))
#define two_sided(i)      ((LineDef[i].sidedef1 != Nothing) && \
                           (LineDef[i].sidedef2 != Nothing))
#define vdist2(v1, v2)    ((long)((v1).x - (v2).x) *        \
                                 ((v1).x - (v2).x) +        \
                           (long)((v1).y - (v2).y) *        \
                                 ((v1).y - (v2).y))

typedef struct BSPSegsTag {
    WadSegs seg;                        // a SEGment
    struct  BSPSegsTag *prev, *next;    // to previous, next SEGment
} BSPSegs;

typedef struct BSPNodeTreeTag {
    int    ymin, ymax, xmin, xmax;      // node rectangle limits
    BSPSegs *pseg;                      // partition line SEG
    BSPSegs *segs;                      // node's SEGS
    int    SegsCount;                   // # initial SEGS in node
    struct BSPNodeTreeTag *left, *right;// left, right children
} BSPNodeTree;

typedef struct BSPNodeInfoTag {
    int          NodeCount;             // total # NODES
    int          SSecCount;             // total # SSectors
    int          SegsCount;             // total # Segs
    BSPSegs     *SegInfo;               // all SEGS lists
} BSPNodeInfo;

BSPNodeInfo ninfo;                      // node information

static int progress;

void ReportProgress(void)
{
  strcpy(lpszMessage, "----------------------------------------");  // 40 -'s
  progress++;
  if(progress == 40)
    progress = 0;
  lpszMessage[progress] = 'o';
  BottomMessage(lpszMessage);
}

int BspInitialize(void)
{
  POINT         *vf, *vt;
  BSPSegs       *SegInfo;
  int            SegsCount, i;

  progress = 39;
  ninfo.NodeCount = ninfo.SSecCount = 0;
  SegsCount = 0;
  ninfo.SegInfo = SegInfo = blockmem(BSPSegs, 1);
  SegInfo->prev = NULL;
  for(i = 0; i < LineDefsNum; i++) {        // scan all lines
    SegInfo->seg.from = LineDef[i].from;
    SegInfo->seg.to   = LineDef[i].to;
    vf = &Vertex[SegInfo->seg.from];
    vt = &Vertex[SegInfo->seg.to];
    SegInfo->seg.angle = (bams)((vt->y == vf->y) && (vt->x < vf->x) ? BAMS180 :
                                 atan2((double)(vt->y - vf->y),
                                       (double)(vt->x - vf->x)) * rad_to_bams +
                                       0.5 * sgn(vt->y - vf->y));
    SegInfo->seg.normal  = 0;               // right side
    SegInfo->seg.linedef = i;
    SegInfo->seg.distance = 0;
    SegsCount++;
    SegInfo->next = blockmem(BSPSegs, 1);   // set up for next one
    SegInfo->next->prev = SegInfo;
    SegInfo = SegInfo->next;
    if(two_sided(i)) {                      // a left side also
      SegInfo->seg.from    = LineDef[i].to; // switch vertices
      SegInfo->seg.to      = LineDef[i].from;
      SegInfo->seg.angle   = SegInfo->prev->seg.angle + BAMS180;
      SegInfo->seg.normal  = 1;              // left side
      SegInfo->seg.linedef = i;
      SegInfo->seg.distance = 0;
      SegsCount++;
      SegInfo->next        = blockmem(BSPSegs, 1);
      SegInfo->next->prev  = SegInfo;
      SegInfo              = SegInfo->next;
      }
    }
  SegInfo->prev->next = NULL;
  blockfree(SegInfo);                       // don't need this one
  return SegsCount;                         // return # created SEGS
}

void BspSplitSeg(BSPSegs *pseg,
                 BSPSegs *seg,
                 BSPSegs **right,
                 BSPSegs **left)
{
  // must get new vertex BEFORE getting pointers to existing vertices,
  // in case the new vertex forces memory to move:
  int    v = VertexNew();

  POINT FAR *pf = &Vertex[pseg->seg.from];
  POINT FAR *pt = &Vertex[pseg->seg.to];
  POINT FAR *vf = &Vertex[seg->seg.from];
  POINT FAR *vt = &Vertex[seg->seg.to];

  long   Ap = -((long)pt->y - (long)pf->y);       // partition line is
  long   Bp = (long)pt->x - (long)pf->x;          // Ax + By + C = 0  
  long   Cp = ((long)pt->y * (long)pf->x) - ((long)pt->x * (long)pf->y);
  long   As = -((long)vt->y - (long)vf->y);       // SEG line is    
  long   Bs = (long)vt->x - (long)vf->x;          // Ax + By + C = 0
  long   Cs = ((long)vt->y * (long)vf->x) - ((long)vt->x * (long)vf->y);
  double x, y;                              // intersection coords
  POINT  iv;                                // intersection vertex

  *right = blockmem(BSPSegs, 1);            // new right side SEG
  (*right)->seg = seg->seg;
  (*right)->next = NULL;

  *left = blockmem(BSPSegs, 1);             // new left side SEG
  (*left)->seg = seg->seg;
  (*left)->next = NULL;

  x =  ((double)Bs * Cp - (double)Bp * Cs) /
       ((double)Bp * As - (double)Bs * Ap);
  y = -((double)As * Cp - (double)Ap * Cs) /
       ((double)Bp * As - (double)Bs * Ap);

  iv.x = (int)(x + sgn(x) * 0.5);
  iv.y = (int)(y + sgn(y) * 0.5);
  Vertex[v] = iv;                           // assign value to new vertex

  if(seg->seg.normal == 0) {                // this is a right side SEG
    if(Ap * vf->x + Bp * vf->y + Cp < 0) {
      (*right)->seg.to = v;
      (*left)->seg.from = v;
      (*left)->seg.distance = seg->seg.distance + (int)sqrt((double)vdist2(*vf,iv));
      SelectPen(OurDC, hPenMarked);
      MoveTo(OurDC, vf->x, vf->y);
      LineTo(OurDC, iv.x, iv.y);
      }
    else {
      (*right)->seg.from = v;
      (*right)->seg.distance = seg->seg.distance + (int)sqrt((double)vdist2(*vt,iv));
      (*left)->seg.to = v;
      SelectPen(OurDC, hPenMarked);
      MoveTo(OurDC, vt->x, vt->y);
      LineTo(OurDC, iv.x, iv.y);
      }
    }
  else {  // seg->seg.normal != 0           // this is a left side SEG
    if(Ap * vt->x + Bp * vt->y + Cp < 0) {
      (*right)->seg.from = v;
      (*right)->seg.distance = seg->seg.distance + (int)sqrt((double)vdist2(*vt,iv));
      (*left)->seg.to = v;
      SelectPen(OurDC, hPenMarked);
      MoveTo(OurDC, vt->x, vt->y);
      LineTo(OurDC, iv.x, iv.y);
      }
    else {
      (*right)->seg.to = v;
      (*left)->seg.from = v;
      (*left)->seg.distance = seg->seg.distance + (int)sqrt((double)vdist2(*vf,iv));
      SelectPen(OurDC, hPenMarked);
      MoveTo(OurDC, vf->x, vf->y);
      LineTo(OurDC, iv.x, iv.y);
      }
    }
}

void BspInsertSeg(BSPSegs *seglist, 
                  BSPSegs *seg,
                  int preinsert)
{
  if(preinsert) {                       // insert before
    seg->prev = seglist->prev;
    seg->next = seglist;
    }
  else {                                // insert after
    seg->prev = seglist;
    seg->next = seglist->next;
    }
  if(seg->prev != NULL)
    seg->prev->next = seg;
  if(seg->next != NULL)
    seg->next->prev = seg;
}

void BspCalculateSegBounds(BSPNodeTree *node)
{
  BSPSegs *SegInfo;
  POINT   *vf, *vt;
  int      s;

  node->xmin = node->ymin = MAXINT;
  node->xmax = node->ymax = MININT;
  for(SegInfo = node->segs, s = 0; s < node->SegsCount; s++) {
    vf = &Vertex[SegInfo->seg.from];
    vt = &Vertex[SegInfo->seg.to];
    
    node->xmin = min(vf->x, node->xmin);
    node->ymin = min(vf->y, node->ymin);
    node->xmax = max(vf->x, node->xmax);
    node->ymax = max(vf->y, node->ymax);
    node->xmin = min(vt->x, node->xmin);
    node->ymin = min(vt->y, node->ymin);
    node->xmax = max(vt->x, node->xmax);
    node->ymax = max(vt->y, node->ymax);
    
    SegInfo = SegInfo->next;
    }
}

int BspSelectSide(WadSegs *pseg,
                  WadSegs *seg)
{
  static int rightleft[3][3] = { {  1,  1, -1 },
                                 {  1, -2,  0 },
                                 { -1,  0,  0 } };
  static POINT FAR *lpf = NULL;
  static POINT FAR *lpt = NULL;             // last partition line verts
  static long A, B, C;                      // describes partition line
  static long pd;
  POINT FAR *pf = &Vertex[pseg->from];      // partition line vertices
  POINT FAR *pt = &Vertex[pseg->to];
  POINT FAR *vf = &Vertex[seg->from];       // segment vertices
  POINT FAR *vt = &Vertex[seg->to];
  long pfd, ptd;
  int  sideflag;
  int  fside, tside;                        // "from"/"to" vertex side

  if((lpf != pf) || (lpt != pt)) {          // update A,B,C if have to
    A = -((long)pt->y - pf->y);             // partition line is
    B = (long)pt->x - pf->x;                // Ax + By + C = 0  
    C = (long)pt->y * pf->x - (long)pt->x * pf->y;
    pd = (long)sqrt((double)A * A + (double)B * B);
    lpf = pf;                               // save new vertices
    lpt = pt;
  }
  pfd = A * vf->x + B * vf->y + C;
  fside = (pfd >= pd) - (pfd <= -pd);       // "from" vertex side
  ptd = A * vt->x + B * vt->y + C;
  tside = (ptd >= pd) - (ptd <= -pd);       // "to" vertex side
  sideflag = rightleft[1 - fside][1 - tside];
  if(sideflag == -2)                        // need to determine based
    sideflag = pseg->angle != seg->angle;   // on direction           
  return sideflag;
}

void BspDivideNode(BSPNodeTree *node,
                   BSPNodeTree *right,
                   BSPNodeTree *left)
{
  int      sideflag;
  int      i;
  BSPSegs *next, *end;
  BSPSegs *lseg, *rseg;                     // for splitting seg in two
  BSPSegs *SegInfo;

  SegInfo = node->segs;
  right->SegsCount = left->SegsCount = 0;
  for(end = SegInfo, i = 0; i < node->SegsCount - 1; i++)
    end = end->next;
  for(i = 0; i < node->SegsCount; i++) {    // scan all node's SEGS
    sideflag = BspSelectSide(&node->pseg->seg, &SegInfo->seg);
    next = SegInfo->next;
    switch(sideflag) {
      case 0:
        right->SegsCount++;                 // just add to right's total
        break;
      case 1:
        if(SegInfo->prev != NULL)           // move to end of list
          SegInfo->prev->next = SegInfo->next;
        if(SegInfo->next != NULL)
          SegInfo->next->prev = SegInfo->prev;
        if(end == SegInfo)
          end = SegInfo->prev;
        if(node->segs == SegInfo)
          node->segs = SegInfo->next;
        BspInsertSeg(end, SegInfo, FALSE);
        end = SegInfo;
        left->SegsCount++;
        break;
      case -1:
        BspSplitSeg(node->pseg, SegInfo, &rseg, &lseg);
        SegInfo->seg = rseg->seg;           // make this the right SEG
        right->SegsCount++;
        blockfree(rseg);                    // don't need this now
        if(end == SegInfo)
          node->segs = SegInfo->prev;
        BspInsertSeg(end, lseg, FALSE);     // add left SEG to end
        end = lseg;
        left->SegsCount++;
        ninfo.SegsCount++;                  // one more for total
      // no default
      } // end switch
    SegInfo = next;
    }
  for(SegInfo = node->segs, i = 0; i < right->SegsCount; i++)
    SegInfo = SegInfo->next;
  right->segs = node->segs;                 // SEGS on right side of  
  BspCalculateSegBounds(right);             // partition line are here
  left->segs = SegInfo;                     // SEGS on left side of    
  BspCalculateSegBounds(left);              // partition line are here
}

void BspFindPartitionSeg(BSPNodeTree *node)
{
  long agm = -1, igm = -1;                  // max geometric mean counts
  long gm;                                  // geometric mean count
  int apndx, ipndx;                         // partition line indices
  int rcnt, lcnt, splits;
  int sideflag;
  BSPSegs *SegInfo, *iseg;
  int s, i;

  SegInfo = node->segs;
  for(s = 0; s < node->SegsCount; s++) {    // scan SEGS in node
    node->pseg = SegInfo;                   // try as partition line
    rcnt = lcnt = splits = 0;
    iseg = node->segs;
    for(i = 0; i < node->SegsCount; i++) {
      sideflag = BspSelectSide(&node->pseg->seg, &iseg->seg);
      switch(sideflag) {
        case 0:
          rcnt++;
          break;                            // count SEGS on both sides
        case 1:
          lcnt++;
          break;                            // of the partition line   
        case -1:
          splits++;
        }   // end switch
      iseg = iseg->next;
      }     // next i
    // 200, 16/3 are empirical
    gm = 100 * rcnt * lcnt / (16 * splits / 3 + 1);
    if(agm < gm) {
       agm = gm;
       apndx = s;
       }
    if((rcnt != node->SegsCount) && (lcnt != node->SegsCount))
      if(igm < gm) {
        igm = gm;
        ipndx = s;
        }
    SegInfo = SegInfo->next;
    }
  if(igm >= 0)
    s = ipndx;
  else
    s = apndx;
  // assign partition line; note that this SEG may be split  
  // later and so won't represent the SEG as it currently is,
  // but it shouldn't really matter since it will still 
  // represent the partition line                            
  node->pseg = node->segs;
  for(i = 0; i < s; i++)
    node->pseg = node->pseg->next;
}

int BspSSectorTest(BSPNodeTree *node)
{
  BOOL     subsector = TRUE;
  BSPSegs *SegInfo, *iseg;
  int      s, i;

  SegInfo = node->segs;
  for(s = 0; subsector && s < node->SegsCount; s++) { // scan all SEGS
    for(iseg = node->segs, i = 0; i < node->SegsCount; i++) {
      if(i != s) {
        if(BspSelectSide(&SegInfo->seg, &iseg->seg) != 0) {
          subsector = FALSE;                // interior angle > 180 degs
          break;                            // so not an SSECTOR        
          }
        }
      iseg = iseg->next;
      }
    SegInfo = SegInfo->next;
    }
  return subsector;
}

int BspPartitionNode(BSPNodeTree *node)
{
  int nl, nr;                               // # left, right nodes
  BSPNodeTree *left, *right;                // left, right children

  BspFindPartitionSeg(node);                // obtain partition line
  node->right = BspBranchTree(right);
  node->left = BspBranchTree(left);
  BspDivideNode(node,right,left);
  if((right->SegsCount == 0) || BspSSectorTest(left)) {
    ReportProgress();
    // found an SSECTOR
    if(right->SegsCount == 0) {
//      BottomMessage("Potential Problem!");
      SelectPen(OurDC, hPenSelected);
      MoveTo(OurDC, Vertex[node->pseg->seg.from].x, Vertex[node->pseg->seg.from].y);
      LineTo(OurDC, Vertex[node->pseg->seg.to].x,   Vertex[node->pseg->seg.to].y);
      SelectPen(OurDC, hPenMarked);
      }
    nl = 1;
    }
  else {                                    // need further subdivision
//    BottomMessage("Left Side Division");
    nl = BspPartitionNode(left);
    }
  if((left->SegsCount == 0) || BspSSectorTest(right)) {
    ReportProgress();
    // found an SSECTOR
    if(left->SegsCount == 0) {
//      BottomMessage("Potential Problem!");
      SelectPen(OurDC, hPenSelected);
      MoveTo(OurDC, Vertex[node->pseg->seg.from].x, Vertex[node->pseg->seg.from].y);
      LineTo(OurDC, Vertex[node->pseg->seg.to].x,   Vertex[node->pseg->seg.to].y);
      SelectPen(OurDC, hPenMarked);
      }
    nr = 1;
    }
  else {                                    // need further subdivision
//    BottomMessage("Right Side Division");
    nr = BspPartitionNode(right);
    }
  return nl + nr + 1;                       // # left + # right + this 1
}

int BspPlaceNode(WadNodes *nodes,
                 int *nndx,
                 int *sndx,
                 BSPNodeTree *nodetree)
{
  int       nnum = 0;
  int       lnum, rnum;
  BSPSegs  *SegInfo, *next;
  WadNodes *node;
  int       s;

  if(nodetree->left != NULL) {              // traverse node subtree
    lnum = BspPlaceNode(nodes, nndx, sndx, nodetree->left);
    rnum = BspPlaceNode(nodes, nndx, sndx, nodetree->right);
    node = &nodes[nnum = (*nndx)++];
    node->x =  Vertex[nodetree->pseg->seg.from].x;          // <=
    node->y =  Vertex[nodetree->pseg->seg.from].y;          // these 4 describe
    node->dx = Vertex[nodetree->pseg->seg.to].x - node->x;  // the partition line
    node->dy = Vertex[nodetree->pseg->seg.to].y - node->y;  // <=
    node->maxy2 = nodetree->left->ymax;
    node->miny2 = nodetree->left->ymin;
    node->minx2 = nodetree->left->xmin;
    node->maxx2 = nodetree->left->xmax;
    if(lnum < 0)
      node->ssector2 = 0x8000 | (-lnum-1);  // mark as SSECTOR
    else
      node->ssector2 = lnum;                // mark as NODE
    blockfree(nodetree->left);              // done with left subtree
    node->maxy1 = nodetree->right->ymax;
    node->miny1 = nodetree->right->ymin;
    node->minx1 = nodetree->right->xmin;
    node->maxx1 = nodetree->right->xmax;
    if(rnum < 0)
      node->ssector1 = 0x8000 | (-rnum-1);  // mark as SSECTOR
    else
      node->ssector1 = rnum;                // mark as NODE
    blockfree(nodetree->right);             // done with right subtree
    }
  else {                                    // SSECTOR (tree leaf)
    SSector[*sndx].num = nodetree->SegsCount;
    if(*sndx == 0)
      SSector[*sndx].segs = 0;
    else
      SSector[*sndx].segs = SSector[*sndx - 1].segs +
                            SSector[*sndx - 1].num;
    SegInfo = nodetree->segs;
    for(s = 0; s < nodetree->SegsCount; s++) { // copy node's SEGS to full
      Seg[ninfo.SegsCount++] = SegInfo->seg;   // SEGS array              
      next = SegInfo->next;
      blockfree(SegInfo);
      SegInfo = next;
      }
    nnum = -++(*sndx);                      // mark as leaf
    }
  return nnum;                              // ret node # or <0 if leaf
}

void BspBuild(void)
{
  if(NodeBuilderAction == NODE_NONE) {
    BottomMessage("Nodes not rebuilt...");
    return;
    }
  if(NodeBuilderAction == NODE_IDBSP) {
    ExportDWD();
    return;
    }

  BSPNodeTree *nodetree;                    // BSP node tree
  DWORD       pTime;
  
  pTime = GetTickCount();
  ReallyBusy = TRUE;
  FindVertexesInUse();                      // clear unused vertexes
  
  OurDC = GetDC(hwnd);
  SelectPen(OurDC, hPenMarked);

  CURSOR_BUSY
  ninfo.SegsCount = BspInitialize();        // initialize SEGS list
  nodetree = BspBranchTree(nodetree);       // node tree root
  nodetree->SegsCount = ninfo.SegsCount;
  nodetree->segs = ninfo.SegInfo;
  BspCalculateSegBounds(nodetree);
  wsprintf(lpszMessage, "Building BSP tree for %d sides: PLEASE BE PATIENT!",
           ninfo.SegsCount);
  BottomMessage(lpszMessage);
  ninfo.NodeCount = BspPartitionNode(nodetree) / 2; // BSP it
  
  // get space for DoomEd data structures:
  NodesNum = ninfo.NodeCount;
  SSectorsNum = ninfo.NodeCount + 1;
  SegsNum = ninfo.SegsCount;
  AllocateNodes(NodesNum);
  AllocateSSectors(SSectorsNum);
  AllocateSegs(SegsNum);
  
  ninfo.SegsCount = ninfo.NodeCount = ninfo.SSecCount = 0;

  (void)BspPlaceNode(Node, &ninfo.NodeCount, &ninfo.SSecCount, nodetree);

  blockfree(nodetree);                      // done with the node tree
  CURSOR_NOTBUSY

  ReallyBusy = FALSE;

  pTime = GetTickCount() - pTime;
  DWORD   min = pTime / 60000;
  DWORD   sec = (pTime - (min * 60000)) / 1000;
  wsprintf(lpszMessage, "BSP: %d Nodes, %d SSectors, %d Segs in %i min %i seconds",
           NodesNum, SSectorsNum, SegsNum, (int)min, (int)sec);
  BottomMessage(lpszMessage);

  ReleaseDC(hwnd, OurDC);
  MapNodes = FALSE;
}

