/*
 *	Name:		MUS File Dumper
 *	Version:	1.63
 *	Author:		Vladimir Arnost (QA-Software)
 *	Last revision:	Aug-15-1995
 *	Compiler:	Borland C++ 3.1, Watcom C/C++ 10.0
 *
 */

/*
 * Revision History:
 *
 *	Aug-1-1994	V1.00	V.Arnost
 *		Written from scratch
 *	Aug-9-1994	V1.20	V.Arnost
 *		Fixed some minor bugs
 *	Sep-16-1994	V1.30	V.Arnost
 *		Added command line parser
 *		Added single channel and single event type dump
 *	Sep-29-1994	V1.40	V.Arnost
 *		Fixed improper instrument number handling
 *	Apr-13-1995	V1.50	V.Arnost
 *		Added `Secondary Channels' header field
 *		Added instrument usage statistics
 *	Jul-4-1995	V1.60	V.Arnost
 *		Added new controller values and event type 3
 *		Added /N option
 *	Jul-11-1995	V1.61	V.Arnost
 *		Fixed some minor bugs
 *	Aug-10-1995	V1.62	V.Arnost
 *		Added include file "deftypes.h"
 *		Added time information output
 *	Aug-15-1995	V1.63	V.Arnost
 *		Fixed instrument list report for non-zero bank values
 *		Added drum bank/program names
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "deftypes.h"

#define VERSION	   "1.63"
#define COPYRIGHT  "MUS File Dump  Version "VERSION"  (c) 1994,95 QA-Software\n"


/* MUS file header structure */
struct MUSheader {
	char	ID[4];		// identifier "MUS" 0x1A
	WORD	scoreLen;	// score length
	WORD	scoreStart;	// score start
	WORD	channels;	// primary channels
	WORD	sec_channels;	// secondary channels (??)
	WORD    instrCnt;	// used instrument count
	WORD	dummy;
//	instruments ...
} header;

uchar channelInstr[16];		/* the arrays are zeroed upon start */
uchar instrPlaying[256];
uint  instrCntUsed[256];
ulong instrTimeUsed[256];


/* Help message */
static char help_msg[] =
	"Syntax:\n"
	"  DUMP_MUS filename.MUS  [options]\n"
	"Options:\n"
	"    /Cn   Dump events in channel number n only.\n"
	"    /En   Dump events of type n only. n is 0-7.\n"
	"    /S    Print instrument usage statistics.\n"
	"    /T[n] Print event time. n=ticks per second (default: 140)\n"
	"    /N    Do NOT print file header.";

/* General MIDI instrument names */
static char *instrname[] =
       {"Acoustic Grand Piano",
	"Bright Acoustic Piano",
	"Electric Grand Piano",
	"Honky-tonk Piano",
	"Rhodes Piano",
	"Chorused Piano",
	"Harpsichord",
	"Clavinet",
	"Celesta",
	"Glockenspiel",
	"Music Box",
	"Vibraphone",
	"Marimba",
	"Xylophone",
	"Tubular-bell",
	"Dulcimer",
	"Hammond Organ",
	"Percussive Organ",
	"Rock Organ",
	"Church Organ",
	"Reed Organ",
	"Accordion",
	"Harmonica",
	"Tango Accordion",
	"Acoustic Guitar (nylon)",
	"Acoustic Guitar (steel)",
	"Electric Guitar (jazz)",
	"Electric Guitar (clean)",
	"Electric Guitar (muted)",
	"Overdriven Guitar",
	"Distortion Guitar",
	"Guitar Harmonics",
	"Acoustic Bass",
	"Electric Bass (finger)",
	"Electric Bass (pick)",
	"Fretless Bass",
	"Slap Bass 1",
	"Slap Bass 2",
	"Synth Bass 1",
	"Synth Bass 2",
	"Violin",
	"Viola",
	"Cello",
	"Contrabass",
	"Tremolo Strings",
	"Pizzicato Strings",
	"Orchestral Harp",
	"Timpani",
	"String Ensemble 1",
	"String Ensemble 2",
	"Synth Strings 1",
	"Synth Strings 2",
	"Choir Aahs",
	"Voice Oohs",
	"Synth Voice",
	"Orchestra Hit",
	"Trumpet",
	"Trombone",
	"Tuba",
	"Muted Trumpet",
	"French Horn",
	"Brass Section",
	"Synth Brass 1",
	"Synth Bass 2",
	"Soprano Sax",
	"Alto Sax",
	"Tenor Sax",
	"Baritone Sax",
	"Oboe",
	"English Horn",
	"Bassoon",
	"Clarinet",
	"Piccolo",
	"Flute",
	"Recorder",
	"Pan Flute",
	"Bottle Blow",
	"Shakuhachi",
	"Whistle",
	"Ocarina",
	"Lead 1 (square)",
	"Lead 2 (sawtooth)",
	"Lead 3 (calliope)",
	"Lead 4 (chiffer)",
	"Lead 5 (charang)",
	"Lead 6 (voice)",
	"Lead 7 (5th sawtooth)",
	"Lead 8 (bass & lead)",
	"Pad 1 (new age)",
	"Pad 2 (warm)",
	"Pad 3 (polysynth)",
	"Pad 4 (choir)",
	"Pad 5 (bowed glass)",
	"Pad 6 (metal)",
	"Pad 7 (halo)",
	"Pad 8 (sweep)",
	"FX 1 (rain)",
	"FX 2 (soundtrack)",
	"FX 3 (crystal)",
	"FX 4 (atmosphere)",
	"FX 5 (brightness)",
	"FX 6 (goblin)",
	"FX 7 (echo drops)",
	"FX 8 (star-theme)",
	"Sitar",
	"Banjo",
	"Shamisen",
	"Koto",
	"Kalimba",
	"Bag Pipe",
	"Fiddle",
	"Shanai",
	"Tinkle Bell",
	"Agogo",
	"Steel Drums",
	"Woodblock",
	"Taiko Drum",
	"Melodic Tom",
	"Synth Drum",
	"Reverse Cymbal",
	"Guitar Fret Noise",
	"Breath Noise",
	"Seashore",
	"Bird Tweet",
	"Telephone Ring",
	"Helicopter",
	"Applause",
	"Gun Shot"};

/* General MIDI percussive instrument names */
static char *percussion[] =
       {"Acoustic Bass Drum",
	"Acoustic Bass Drum",
	"Slide Stick",
	"Acoustic Snare",
	"Hand Clap",
	"Electric Snare",
	"Low Floor Tom",
	"Closed High-Hat",
	"High Floor Tom",
	"Pedal High Hat",
	"Low Tom",
	"Open High Hat",
	"Low-Mid Tom",
	"High-Mid Tom",
	"Crash Cymbal 1",
	"High Tom",
	"Ride Cymbal 1",
	"Chinses Cymbal",
	"Ride Bell",
	"Tambourine",
	"Splash Cymbal",
	"Cowbell",
	"Crash Cymbal 2",
	"Vibraslap",
	"Ride Cymbal 2",
	"High Bongo",
	"Low Bango",
	"Mute High Conga",
	"Open High Conga",
	"Low Conga",
	"High Timbale",
	"Low Timbale",
	"High Agogo",
	"Low Agogo",
	"Cabasa",
	"Maracas",
	"Short Whistle",
	"Long Whistle",
	"Short Guiro",
	"Long Guiro",
	"Claves",
	"High Wood Block",
	"Low Wood Block",
	"Mute Cuica",
	"Open Cuica",
	"Mute Triangle",
	"Open Triangle"};


/* Command-line parameters */
char   *musname = NULL;
int	help = 0;
int	statistics = 0;
int	printtime = 0;
int	speed = 140;
int	onlychannel = -1;
int	onlyevent = -1;
int	noheader = 0;


/* Function prototypes */
int	dispatchOption(char *option);
int	dispatchCmdLine(int argc, char *argv[]);
char *	note2name(uint note);
char *	instr2name(uint instr);
int	dumpHeader(FILE *f);
char *	time2text(ulong ticks);
int	dumpScore(FILE *f);
void	printStatistics(void);
int	main(int argc, char *argv[]);


/* Dispatch one command-line option */
int dispatchOption(char *option)
{
    int n;
    switch (*option) {
	case '?':
	case 'H':		// help
	    help = 1;
	    break;
	case 'C':		// channel
	    n = atoi(option+1);
	    if (onlychannel == -1) onlychannel = 0;
	    onlychannel |= 1 << n;
	    break;
	case 'E':		// event
	    n = atoi(option+1);
	    if (onlyevent == -1) onlyevent = 0;
	    onlyevent |= 1 << n;
	    break;
	case 'S':		// statistics
	    statistics = 1;
	    break;
	case 'T':		// time
	    printtime = 1;
	    if (option[1])
	    {
		speed = strtol(option+1, NULL, 10);
		if (speed <= 0)
		    return -1;
	    }
	    break;
	case 'N':		// no header
	    noheader = 1;
	    break;
	default:
	    return -1;
    }
    return 0;
}

/* Dispatch the whole command-line */
int dispatchCmdLine(int argc, char *argv[])
{
    while (argv++, --argc)
    {
	strupr(*argv);
	if (**argv == '/' || **argv == '-')
	{
	    if (dispatchOption(*argv+1))
	    {
		printf("Invalid option %s\n", *argv);
		return 2;
	    }
	} else {
	    if (!musname)
		musname = *argv;
	    else {
		printf("Too many filenames.\n");
		return 3;
	    }
	}
    }

    if (!musname || help)
    {
	puts(help_msg);
	return 1;
    }

    return 0;
}

/* Convert note number to name */
char *note2name(uint note)
{
    static char name[8] = "";
    static char *notes[] = {"C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"};
    static char *octaves[] = {"-0", "-1", "-2", "-3", "-4", "-5", "-6", "-7", "-8", "-9", "-10"};

    strcpy(name, notes[note % 12]);
    strcat(name, octaves[note / 12]);
    return name;
}

/* Convert instrument number to name */
char *instr2name(uint instr)
{
    if (instr < 128)
	return instrname[instr];
    else if (instr >= 135 && instr < (135+47))	// there are 47 percussive instruments
	return percussion[instr-135];
    else
	return "*Unknown*";
}

/* Convert drum bank number to name */
char *bank2name(uint bank)
{
    switch (bank) {
	case 0:
	    return "Standard";
	case 8:
	    return "Room";
	case 16:
	    return "Power";
	case 24:
	    return "Electronic";
	case 25:
	    return "TR-808";
	case 32:
	    return "Jazz";
	case 40:
	    return "Brush";
	case 48:
	    return "Orchestra";
	case 56:
	    return "SFX";
	case 127:
	    return "CM-64/32L";
    }
    return NULL;
}

/* Dump MUS file header */
int dumpHeader(FILE *f)
{
    uint i;

    if (fread(&header, 1, sizeof header, f) != sizeof header ||
	header.ID[0] != 'M' ||
	header.ID[1] != 'U' ||
	header.ID[2] != 'S' ||
	header.ID[3] != 0x1A)
    {
	puts("Bad file.");
	return -1;
    }

    if (!noheader)
    {
	printf("File\t\t\t%s\n", musname);
	printf("Score Length\t\t%04Xh  (%u bytes)\n", header.scoreLen, header.scoreLen);
	printf("Score Start\t\t%04Xh\n", header.scoreStart);
	printf("Primary Channels\t%d\n", header.channels);
	printf("Secondary Channels\t%d\n", header.sec_channels);
	printf("Instruments Used\t%d\n", header.instrCnt);

	for (i = 0; i < header.instrCnt; i++)
	{
	    uint ins = fgetc(f);
	    uint banks = fgetc(f);
	    printf("  %3d  %-30s", ins, instr2name(ins));
	    if (banks)
	    {
		printf("  bank");
		do {
		    uint bank;
		    char *bankName;
		    printf(" %d", bank = fgetc(f));
		    bankName = bank2name(bank);
		    if (bankName)
			printf(" (%s)", bankName);
		} while (--banks != 0);
	    }
	    printf("\n");
	}
	printf("\n");
    }

    return 0;
}

/* Print elapsed time */
char *time2text(ulong ticks)
{
    static char str[16];
    ldiv_t lx;
    uint min, sec, msec;
    lx = ldiv((ticks * 1000) / speed, 1000);
    msec = lx.rem;
    lx = ldiv(lx.quot, 60);
    sec = lx.rem;
    min = lx.quot;

    sprintf(str, "%d:%02d.%03d", min, sec, msec);
    return str;
}

/* Dump MUS file body */
int dumpScore(FILE *f)
{
    int cmd, channel, timemark, note, volume, ctrl, value, show, fulldump, first;
    ulong delay = 0, totaltime = 0;

    fulldump = onlychannel == -1 && onlyevent == -1;
    first = 0;

    fseek(f, header.scoreStart, SEEK_SET);

    while (!feof(f))
    {
	DWORD offs = ftell(f);

	/* read and decode command */
	if ( (value = fgetc(f)) == -1)
	    break;

	timemark = value & 0x80;
	channel = value & 0x0F;
	cmd = (value >> 4) & 0x07;
	show = fulldump || cmd == 6 || ((onlychannel & (1 << channel)) &&
	       (onlyevent & (1 << cmd)));

	/* print file offset and delay value */
	if (show)
	{
	    if (delay)
	    {
		printf("\t\t\tdelay: %ld", delay);
		if (printtime)
		    printf("  (%s)\n", time2text(delay));
		else
		    printf("\n");
		delay = 0;
	    }
	    if (printtime && first)
		printf("\t\t\ttime: %s\n", time2text(totaltime));
	    first = 0;
	    printf("%06lX:  %02X ", offs, value);
	}

	/* print command and its parameters */
	switch (cmd) {
	    case 0:	// release note
		note = fgetc(f);

		if (statistics)		// update statistics
		{
		    register WORD instr = channel == 15 ?
			(note & 0x7F) + 100 : channelInstr[channel];
		    instrPlaying[instr]--;
		}

		if (show)
		    printf("%02X\t\t#%d: release note %d (%s)\n",
			note, channel, note, channel == 15 ?
			instr2name(note + 100) : note2name(note));
		break;
	    case 1:	// play note
		note = fgetc(f);

		if (statistics)		// update statistics
		{
		    register WORD instr = channel == 15 ?
			(note & 0x7F) + 100 : channelInstr[channel];
		    instrPlaying[instr]++;
		    instrCntUsed[instr]++;
		}

		if (note & 0x80)
		{
		    note &= 0x7F;
		    volume = fgetc(f);
		    if (show)
			printf("%02X %02X\t#%d: play note %d (%s), volume %d\n",
			    note | 0x80, volume, channel, note, channel == 15 ?
			    instr2name(note + 100) : note2name(note), volume);
		} else
		    if (show)
			printf("%02X\t\t#%d: play note %d (%s)\n",
			    note, channel, note, channel == 15 ?
			    instr2name(note + 100) : note2name(note));
		break;
	    case 2:	// pitch wheel
		value = fgetc(f);
		if (show)
		    printf("%02X\t\t#%d: pitch wheel %+d\n", value, channel, value - 0x80);
		break;
	    case 5:	// ???
	    case 7:	// ???
		value = fgetc(f);
		if (show)
		    printf("%02X\t\t#%d: unknown command, parameter %d\n", value, channel, value);
		break;
	    case 3:	// system event
		value = fgetc(f);
		if (show)
		{
		    static char *sysev[] = {
			"all sounds off\n",		/* 10 */
			"all notes off\n",		/* 11 */
			"mono\n",			/* 12 */
			"poly\n",			/* 13 */
			"reset all controllers\n"	/* 14 */
		    };
		    printf("%02X\t\t#%d: ", value, channel);
		    if (value >= 10 && value <= 14)
			printf(sysev[value-10]);
		    else
			printf("system event %d\n", value);
		}
		break;
	    case 4:	// change control
		ctrl = fgetc(f);
		value = fgetc(f);
		if (ctrl == 0)
		    channelInstr[channel] = value;
		if (show)
		{
		    static char *ctrls[] = {
			"set instrument %d (%s)\n",	/* 0 */
			"select bank %d\n",		/* 1 */
			"set modulation %d\n",		/* 2 */
			"set volume %d\n",		/* 3 */
			"set pan %+d\n",		/* 4 */
			"set expression %d\n",		/* 5 */
			"set reverb depth %d\n",	/* 6 */
			"set chorus depth %d\n",	/* 7 */
			"set sustain pedal %d\n",	/* 8 */
			"set soft pedal %d\n"		/* 9 */
		    };
		    printf("%02X %02X\t#%d: ", ctrl, value, channel);
		    switch (ctrl) {
			case 0: // change instrument/program
			    if (channel == 15)
			    {
				char *bankName = bank2name(value);
				if (bankName)
				    printf("set program %d (%s)\n", value, bankName);
				else
				    printf("set program %d\n", value);
			    } else
				printf(ctrls[0], value, value < 128 ?
				    instrname[value] : "*Invalid*");
			    break;
			case 4: // change pan
			    printf(ctrls[4], value - 0x40);
			    break;
			default:
			    if (ctrl < 10)
				printf(ctrls[ctrl], value);
			    else
				printf("set control %d, %d\n", ctrl, value);
		    }
		}
		break;
	    case 6:	// end
		printf("\t\t#%d: end of score\n", channel);
		break;
	}

	/* read and print time information */
	if (timemark)
	{
	    ulong   time = 0;
	    uchar   b;

	    if (fulldump)
		printf("%06lX:  ", ftell(f));
	    do {
		b = fgetc(f);
		if (fulldump)
		    printf("%02X ", b);
		time <<= 7;
		time += b & 0x7F;
	    } while (b & 0x80);

	    if (fulldump)
	    {
		printf("\t\tdelay: %ld", time);
		if (printtime)
		    printf("  (%s)\n", time2text(time));
		else
		    printf("\n");
	    } else
		delay += time;
	    totaltime += time;

	    if (statistics)		// update statistics
	    {
		uint i;
		for (i = 0; i < 256; i++)
		    if (instrPlaying[i])
			instrTimeUsed[i] += time * instrPlaying[i];
	    }
	    first = 1;
	}
    } /* while */

    return 0;
}

/* Initialize the counters */
void initStatistics(void)
{
    memset(channelInstr, sizeof channelInstr, 0);
    memset(instrPlaying, sizeof instrPlaying, 0);
    memset(instrCntUsed, sizeof instrCntUsed, 0);
    memset(instrTimeUsed, sizeof instrTimeUsed, 0);
}

/* Print statistics results */
void printStatistics(void)
{
    uint i;

    puts("\nInstrument Usage Statistics:\n"
    	 "\n"
	 "  #  Name                      Used cnt   Total time\n"
	 "--------------------------------------------------------------");

    for (i = 0; i < 256; i++)
	if (instrCntUsed[i])
	{
	    printf("%3d  %-25s %5d      %8ld  %9s\n", i, instr2name(i),
		instrCntUsed[i], instrTimeUsed[i],
		printtime ? time2text(instrTimeUsed[i]) : "");
	}
}

/* Main program */
main(int argc, char *argv[])
{
    int retcode;
    FILE *f;

    /* initialization */
    fprintf(stderr, COPYRIGHT);
    if ( (retcode = dispatchCmdLine(argc, argv)) != 0)
	return retcode;

    if (statistics)
	initStatistics();

    /* open MUS file */
    if ( (f = fopen(musname, "rb")) == NULL)
    {
	printf("Can't open file %s\n", musname);
	return 4;
    }

    /* dump file header */
    if (dumpHeader(f))
    {
	fclose(f);
	return 5;
    }

    /* dump file body */
    dumpScore(f);

    if (statistics)
	printStatistics();

    /* close file */
    fclose(f);

    return 0;
}

/* End of file */
