/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

WAD cleaner and optimizer by Serge Smirnov (sxs111@po.cwru.edu). Written at Olivier Montanuy's request as a stand-alone utility to
be used with WinTex and extended by Martin Howe to cope with OpenGL and HeXen resources, etc.

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include "services.h"
#include "strmanip.h"
#include "waddef.h"
#include "wadlst.h"
#include "waddir.h"
#include "wadhdr.h"
#include "wadlru.h"
#include "wadopt.h"
#include "requests.h"

// Name and version
static const char name[]="CleanWAD";							// program name
static const char version[]="1.54";								// program version

// Environment variables
#define ENV_MASTER_DIRECTORY	"CLEANWADMAINDIR"				// master configuration file directory
#define ENV_USER_DIRECTORY		"CLEANWADUSERDIR"				// user configuration file directory
#define ENV_DEFAULT_OPTIONS		"CLEANWADDFLTOPS"				// default options string

/**********************************************************************************************************************************/
/******************************************************* Request Processing *******************************************************/
/**********************************************************************************************************************************/

void WADCleanGetArgs (int argc, char *argv[], WADREQ_t *WADREQ)
{
	// VARIABLES
	int last_argp;												// index of last (option) argument parsed from command line
	int dummy_int=0;											// dummy integer for default and environment options
	char *env_copy;												// copy of environment options string
	int env_argc;												// environment options argc
	char **env_argv;											// environment options argv
	char *envopts;												// envrironment string containing options
	char dummy_envstring=NUL;									// dummy envrironment string for when no environment options
	options_t default_options;									// options to take defaults from
	options_t environment_options;								// options taken from the environment
	options_t commandline_options;								// options taken from the command line

	// PARSE DEFAULT OPTIONS
	WADParseOptions (
		&default_options,										// options parsed from arguments
		OPTION_IS_DEFAULT,										// ultimate source of these options
		WADGetDefaultArgC(),									// program argument count
		WADGetDefaultArgV(),									// program argument strings
		&dummy_int);											// index of last argument parsed

	// VALIDATE DEFAULT OPTIONS
	WADValidateOptions (
		&default_options,										// options to be validated
		OPTION_IS_DEFAULT);										// ultimate source of these options

	// GET ENVIRONMENT OPTIONS IF ANY ELSE AN EMPTY STRING
	envopts=getenv(ENV_DEFAULT_OPTIONS);
	if (envopts==NULL)
	{
		envopts=&dummy_envstring;								// just a single NUL character
	}

	// PRE-PARSE ENVIRONMENT STRING
	ParseArgvAndArgcFromString (
		envopts,												// command line to create argv and argc from
		&env_copy,												// modified copy of string (needed for subsequent parsing)
		&env_argc,												// number of arguments (0 is a dummy)
		&env_argv);												// pointer to array of string pointers

	// PARSE OPTIONS FROM THE STRING
	WADParseOptions (
		&environment_options,									// options parsed from arguments
		OPTION_IS_ENVIRONMENT,									// ultimate source of these options
		env_argc,												// program argument count
		env_argv,												// program argument strings
		&dummy_int);											// index of last argument parsed

	// VALIDATE OPTIONS FROM THE STRING
	WADValidateOptions (
		&environment_options,									// options to be validated
		OPTION_IS_ENVIRONMENT);									// ultimate source of these options

	// DEALLOCATE RETURNED STRING MEMORY
	free(env_copy);
	free(env_argv);

	// PARSE OPTIONS FROM THE COMMAND LINE
	WADParseOptions (
		&commandline_options,									// options parsed from arguments
		OPTION_IS_COMMANDLINE,									// ultimate source of these options
		argc,													// program argument count
		argv,													// program argument strings
		&last_argp);											// index of last argument parsed

	// VALIDATE OPTIONS FROM THE COMMAND LINE
	WADValidateOptions (
		&commandline_options,									// options to be validated
		OPTION_IS_COMMANDLINE);									// ultimate source of these options

	// PARSE ARGUMENTS FROM THE COMMAND LINE
	WADParseCommandLineArguments (
		argc,													// program argument count
		argv,													// program argument strings
		last_argp,												// index of last option argument parsed
		WADREQ);												// WAD file processing request

	// MERGE THE THREE SETS OF OPTIONS
	WADMergeOptions (
		WADREQ->options,										// options to be merged into
		&default_options);										// options to merge into them
	WADMergeOptions (
		WADREQ->options,										// options to be merged into
		&environment_options);									// options to merge into them
	WADMergeOptions (
		WADREQ->options,										// options to be merged into
		&commandline_options);									// options to merge into them

	// VALIDATE THE RESULTANT OPTIONS
#ifdef PARANOID
	WADValidateOptions (
		WADREQ->options,										// options to be validated
		OPTION_IS_RESULTANT);									// ultimate source of these options
#endif
}

void WADCleanProcess (WADREQ_t *WADREQ)
{
	// VARIABLES
	WADHDR_t *WADHDR=NULL;										// file header
	WADDIR_t *WADDIR=NULL;										// file directory
	LST_t *LST=NULL;											// symbol table build during file parse
	FILE *infile=NULL;											// file being processed
	FILE *outfile=NULL;											// file after processing
	fofs_t fpos;												// file position after writing output file directory

	// STATEMENT
	//
	// This is where the program announcement statement should
	// logically occur; however, in order to ensure that it
	// occurs before ANY other messages, it is handled at the
	// start of the program and is treated as a special case.
	//
//	EventState(VERBOSITY_STATEMENTS,"%s %s [%s] Created by Serge Smirnov and updated by Martin Howe",name,version,__DATE__);

	// INITIALISE
	WADInitHeader(&WADHDR);
	WADInitDirectory(&WADDIR);
	LSTInit(&LST);

	// SHORTHAND TO MAKE PROGRAM READABLE
	#define OPTIONS (*(WADREQ->options))

	// SET VERBOSITY LEVEL
	EventSetVerbosity(OPTIONS[OPTION_VERBOSITY].value.verbosity);

	// OPEN INPUT FILE
	infile=fopen(WADREQ->infilename,"rb");
	if (infile==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not open input file %s",WADREQ->infilename);
	}
	EventState(VERBOSITY_PROGRESS,"Opened input file %s",WADREQ->infilename);

	// READ WAD FILE HEADER FROM INPUT FILE
	EventState(VERBOSITY_PROGRESS,"Reading header");
	WADReadHeader(WADHDR,infile);

	// VALIDATE INPUT FILE
	if ( (strcmp(WADHDR->magic,"IWAD")!=0) && (strcmp(WADHDR->magic,"PWAD")!=0) )
	{
		if (fclose(infile)!=0)
		{
			EventState(VERBOSITY_ERRORS,"Could not properly close input file %s",WADREQ->infilename);
		}
		ErrorAbort(1,__FILE__,__LINE__,"Input file %s is not a WAD file",WADREQ->infilename);
	}

	// OPEN OUTPUT FILE
	outfile=fopen(WADREQ->outfilename,(OPTIONS[OPTION_PACK_LUMPS].value.flag?"wb+":"wb"));
	if (outfile==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not open output file %s",WADREQ->outfilename);
	}
	EventState(VERBOSITY_PROGRESS,"Opened output file %s",WADREQ->outfilename);

	// SEEK TO INPUT FILE DIRECTORY
	if (fseek(infile,WADHDR->dir_start,SEEK_SET)!=0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not seek to directory (position %ld) in input file %s",WADHDR->dir_start,WADREQ->infilename);
	}

	// READ INPUT FILE DIRECTORY
	EventState(VERBOSITY_PROGRESS,"Reading directory (%lu entries)",WADHDR->num_entries);
	WADReadDirectory(WADDIR,infile,WADHDR->num_entries);
	if (WADHDR->num_entries!=WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Header and directory entry counts differ: HDR=%lu DIR=%lu",WADHDR->num_entries,WADDIR->count);
	}

	// SCAN DIRECTORY AND IDENTIFY CONTENTS
	EventState(VERBOSITY_PROGRESS,"Scanning directory");
	WADScanDirectory (
		WADDIR,													// directory to scan
		LST,													// symbol table resulting from the scan
		OPTIONS[OPTION_IDENTIFY_NAMES].value.flag,				// identify screen images by name (READ MANUAL BEFORE USING!)
		OPTIONS[OPTION_IDENTIFY_PAGES].value.flag,				// identify screen images as pages (READ MANUAL BEFORE USING!)
		OPTIONS[OPTION_IDENTIFY_GRAPHICS].value.flag,			// identify screen images as graphics (READ MANUAL BEFORE USING!)
		OPTIONS[OPTION_IDENTIFY_VOICES].value.flag,				// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
		OPTIONS[OPTION_DETECT_SOUNDS].value.flag,				// detect sounds by content (READ MANUAL BEFORE USING!)
		OPTIONS[OPTION_DETECT_MUSICS].value.flag,				// detect musics by content (READ MANUAL BEFORE USING!)
		OPTIONS[OPTION_DETECT_GRAPHICS].value.flag,				// detect graphics by content (READ MANUAL BEFORE USING!)
		OPTIONS[OPTION_RECOGNISED_NAMES].value.flag,			// identify sounds/musics/dialogs/conversations by recognised names
		OPTIONS[OPTION_LOOSE_MARKERS].value.flag,				// allow nonstandard list marker characters
		OPTIONS[OPTION_NAMED_MARKERS].value.flag,				// loose markers use names only (READ MANUAL BEFORE USING!)
		OPTIONS[OPTION_TOLERATE_MULTIPLES].value.flag,			// do not treat multiple instances of structured lists as an error
		OPTIONS[OPTION_QUIET_MULTIPLES].value.flag,				// do not treat multiple instances of structured lists as a problem
		OPTIONS[OPTION_DECLASSIFY_PNAMES].value.flag,			// treat PNAMES as an unclassified lump (separate from TEXTURES)
		OPTIONS[OPTION_LOOSE_HEADERS].value.flag,				// allow nonstandard map name headers (not just E\?M\? and MAP\?\?)
		OPTIONS[OPTION_QUIET_HEADERS].value.flag,				// do not warn about non-empty map name headers
		OPTIONS[OPTION_GAME].value.game,						// game for which (WAD whose lump names these are) was designed
		infile);												// file that directory was read from

	// DETERMINE IF ANY LUMPS REUSED BETWEEN DIRECTORY ENTRIES
	WADCheckForAnyReuse(
		WADDIR,													// WAD file directory that was scanned
		&WADDIR->reuse);										// flag for some lump reuse was found

	// ISSUE APPROPRIATE DIAGNOSTICS IF ANY ARE BEING REUSED
	if (WADDIR->reuse)
	{
		// VARIABLES
		bool_t abort_this;
		bool_t dummy_bool_t;

		// ISSUE REUSE WARNINGS FIRST (IF ANY)
		WADHandleExistingReuseOnInput (
			WADDIR,													// WAD file directory that was scanned
			LST,													// symbol table resulting from the scan of that directory
			OPTIONS[OPTION_REUSE_LUMPS].value.flag,					// reuse lumps where directory entries specify same size and offset
			OPTIONS[OPTION_PACK_LUMPS].value.flag,					// share lumps with identical contents between directory entries
			OPTIONS[OPTION_UNPACK_LUMPS].value.flag,				// output each lump explicitly even if it could be shared or reused
			OPTIONS[OPTION_TOLERATE_LINKS].value.flag,				// do not treat lump reuse as an error
			OPTIONS[OPTION_QUIET_LINKS].value.flag,					// do not treat lump reuse as a problem
			OPTIONS[OPTION_TOLERATE_CONFLICTS].value.flag,			// do not treat ineligible lump reuse as an error
			OPTIONS[OPTION_QUIET_CONFLICTS].value.flag,				// do not treat ineligible lump reuse as a problem
			VERBOSITY_WARNINGS,										// priority selection (must be WARNINGS or ERRORS)
			&dummy_bool_t);											// warnings alone can never stop the program

		// THEN REUSE ERRORS (IF ANY)
		WADHandleExistingReuseOnInput (
			WADDIR,													// WAD file directory that was scanned
			LST,													// symbol table resulting from the scan of that directory
			OPTIONS[OPTION_REUSE_LUMPS].value.flag,					// reuse lumps where directory entries specify same size and offset
			OPTIONS[OPTION_PACK_LUMPS].value.flag,					// share lumps with identical contents between directory entries
			OPTIONS[OPTION_UNPACK_LUMPS].value.flag,				// output each lump explicitly even if it could be shared or reused
			OPTIONS[OPTION_TOLERATE_LINKS].value.flag,				// do not treat lump reuse as an error
			OPTIONS[OPTION_QUIET_LINKS].value.flag,					// do not treat lump reuse as a problem
			OPTIONS[OPTION_TOLERATE_CONFLICTS].value.flag,			// do not treat ineligible lump reuse as an error
			OPTIONS[OPTION_QUIET_CONFLICTS].value.flag,				// do not treat ineligible lump reuse as a problem
			VERBOSITY_ERRORS,										// priority selection (must be WARNINGS or ERRORS)
			&abort_this);											// we need to know if any errors were found

		// ABORT IF ANY REUSE ERRORS WERE FOUND
		if (abort_this)
		{
			// At least one error message will have been written
			ErrorAbort(1,__FILE__,__LINE__,NULL);
		}
	}

	// NORMALISE DIRECTORY ACCORDING TO OPTIONS
	EventState(VERBOSITY_PROGRESS,"Normalising directory");
	WADNormaliseDirectory (
		WADDIR,													// WAD file directory to normalise
		LST,													// symbol table resulting from the scan of that directory
		OPTIONS[OPTION_REMOVE_DUPLICATES].value.flag,			// remove all but the first of entries having the same name
		OPTIONS[OPTION_IDENTIFY_VOICES].value.flag,				// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
		OPTIONS[OPTION_KEEP_WINTEX].value.flag,					// do not remove _DEUTEX_ lump
		OPTIONS[OPTION_KEEP_PLATFORM].value.flag,				// do not remove PLATFORM lump
		OPTIONS[OPTION_KEEP_HISTORY].value.flag,				// do not remove HISTORY lump
		OPTIONS[OPTION_KEEP_TAGDESC].value.flag,				// do not remove TAGDESC lump
		OPTIONS[OPTION_KEEP_PCSFX].value.flag,					// do not remove PC speaker sound effects
		OPTIONS[OPTION_KEEP_DOUBLES].value.flag,				// do not normalise double-letter list markers (e.g., PP_START)
		OPTIONS[OPTION_KEEP_BORDERS].value.flag,				// do not remove sub-list border markers (e.g., F1_START)
		OPTIONS[OPTION_KEEP_EMPTIES].value.flag,				// do not remove empty structured lists
		OPTIONS[OPTION_REMOVE_SCRIPTS].value.flag,				// remove all SCRIPTS and SCRIPTnn entries
		OPTIONS[OPTION_FORCE_REMOVAL].value.flag);				// remove duplicate entries even if in different lists

	// SORT DIRECTORY ACCORDING TO OPTIONS
	EventState(VERBOSITY_PROGRESS,"Sorting directory");
	SortWadDirectory (
		WADDIR,													// directory to sort
		LST,													// symbol table resulting from initial scan
		OPTIONS[OPTION_SORT].value.sort,						// sort order to apply to entries in the output file
		OPTIONS[OPTION_GAME].value.game);						// game for which (WAD whose lump names these are) was designed

	// PAD OUTPUT FILE WITH DUMMY HEADER FOR POSITIONING
	WADWriteHeader(WADHDR,outfile);

	// COPY DIRECTORY CONTENTS FROM INPUT FILE TO OUTPUT FILE
	EventState(VERBOSITY_PROGRESS,"Writing output file data");
	CopyDirectoryEntriesOptimized (
		infile,													// file to copy entries from
		outfile,												// file to copy entries to
		WADDIR,													// directory to copy entries of
		LST,													// symbol table resulting from initial scan
		OPTIONS[OPTION_TRUNCATE_WAVES].value.flag,				// remove wasted space at the end of sound resources
		OPTIONS[OPTION_REBUILD_BLOCKMAPS].value.flag,			// rebuild (no pack/unpack) blockmaps to remove wasted space
		OPTIONS[OPTION_PACK_BLOCKMAPS].value.flag,				// pack blockmaps to remove current and potential wasted space
		OPTIONS[OPTION_UNPACK_BLOCKMAPS].value.flag,			// unpack blockmaps into normalised (but inefficient) format
		OPTIONS[OPTION_FLUSH_BLOCKMAPS].value.flag,				// write out modified blockmaps even if the size has not changed
		OPTIONS[OPTION_REBUILD_PICTURES].value.flag,			// rebuild (no pack/unpack) pictures to remove wasted space
		OPTIONS[OPTION_PACK_PICTURES].value.flag,				// pack pictures to remove current and potential wasted space
		OPTIONS[OPTION_UNPACK_PICTURES].value.flag,				// unpack pictures into normalised (but inefficient) format
		OPTIONS[OPTION_FLUSH_PICTURES].value.flag,				// write out modified pictures even if the size has not changed
		OPTIONS[OPTION_REUSE_LUMPS].value.flag,					// reuse lumps where directory entries specify same size and offset
		OPTIONS[OPTION_PACK_LUMPS].value.flag,					// share lumps with identical contents between directory entries
		OPTIONS[OPTION_UNPACK_LUMPS].value.flag,				// output each lump explicitly even if it could be shared or reused
		OPTIONS[OPTION_UNPACK_CONFLICTS].value.flag,			// do not preserve ineligible lump reuse (if tolerated)
		OPTIONS[OPTION_ALIGN_RESOURCES].value.flag,				// align resources on 32-bit boundaries for speed at cost of space
		OPTIONS[OPTION_ALIGN_DIRECTORY].value.flag,				// align directory on 32-bit boundaries for speed at cost of space
		OPTIONS[OPTION_MAINTAIN_BLOCKMAPS].value.flag,			// maintain nonstandard blockmaps (preserve non-optimal blocks)
		OPTIONS[OPTION_MAINTAIN_PICTURES].value.flag);			// maintain nonstandard pictures (preserve non-optimal columns)

	// GET CURRENT FILE POSITION
	fpos=ftell(outfile);
	if (fpos<0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not get current position in output file %s",WADREQ->outfilename);
	}

	// PAD FILE DATA IF APPLICABLE READY FOR WAD DIRECTORY
	if (OPTIONS[OPTION_ALIGN_DIRECTORY].value.flag && ((fpos%4)!=0) )
	{
		static const char padding[3]={NUL,NUL,NUL};
		/*****************************************/
		if (fwrite(padding,(size_t)(4-(fpos%4)),1,outfile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not write data alignment padding to file %s",WADREQ->outfilename);
		}
		fpos+=(4-(fpos%4));
	}

	// UPDATE WAD FILE HEADER WITH NEW INFORMATION
	WADHDR->num_entries=WADDIR->count;
	WADHDR->dir_start=fpos;

	// WRITE OUTPUT FILE DIRECTORY
	EventState(VERBOSITY_PROGRESS,"Writing output directory (%lu entries)",WADDIR->count);
	WADWriteDirectory(WADDIR,outfile);

	// SEEK TO OUTPUT FILE HEADER
	if (fseek(outfile,0,SEEK_SET)!=0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not seek to header (position 0) in output file %s",WADREQ->outfilename);
	}

	// WRITE OUTPUT FILE HEADER
	EventState(VERBOSITY_PROGRESS,"Writing header");
	WADWriteHeader(WADHDR,outfile);

	// CLOSE INPUT FILE
	if (fclose(infile)!=0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not properly close input file %s",WADREQ->infilename);
	}
	EventState(VERBOSITY_PROGRESS,"Closed input file %s",WADREQ->infilename);

	// CLOSE OUTPUT FILE
	if (fclose(outfile)!=0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not properly close output file %s",WADREQ->outfilename);
	}
	EventState(VERBOSITY_PROGRESS,"Closed output file %s",WADREQ->outfilename);

	// DEINITIALISE
	WADDoneRequest(&WADREQ);
	WADDoneHeader(&WADHDR);
	WADDoneDirectory(&WADDIR);
	LSTDone(&LST);

	// ALL DONE
	EventState(VERBOSITY_PROGRESS,"\nCompleted");
}

/**********************************************************************************************************************************/
/************************************************************** Main **************************************************************/
/**********************************************************************************************************************************/

// Announcer for the "clean" mode
void WADCleanAnnounce (void)
{
	EventState(VERBOSITY_STATEMENTS,"%s %s [%s] Created by Serge Smirnov and updated by Martin Howe",name,version,__DATE__);
}

// Main routine
int main (int argc, char *argv[])
{
#if 0
	if (argc>1)
	{
		if (strcmp(argv[1],"-clean")==0)
		{
			WADCleanAnnounce();
			clean(argc-1,&argv[1]);
			return 0;
		}
	}
	(void) fprintf(stderr,"\n");
	(void) fprintf(stderr,"Error:\n");
	(void) fprintf(stderr,"   Invalid request %s\n",((argc>1) && (strlen(argv[1])>0))?argv[1]:"<none>");
	(void) fprintf(stderr,"Usage:\n");
	(void) fprintf(stderr,"   %s -<request> [options] [arguments]\n",argv[0]);
	(void) fprintf(stderr,"Requests:\n");
	(void) fprintf(stderr,"   -clean ... clean and optimize a WAD file\n");
	(void) fprintf(stderr,"Notes:\n");
	(void) fprintf(stderr,"   To get help on a particular request, run the\n");
	(void) fprintf(stderr,"   program with the option -<request> but no other\n");
	(void) fprintf(stderr,"   arguments; for example \"%s -clean\".\n",argv[0]);
	return 1;
#else
	WADREQ_t *WADREQ=NULL;										// file processing request
	WADInitRequest(&WADREQ);									// initialise request
	WADCleanAnnounce();											// program (and mode) announcement
	WADCleanGetArgs(argc,argv,WADREQ);							// impose defaults on and parse command-line into request
	WADCleanProcess(WADREQ);									// process request
	return 0;
#endif
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
