/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Option and argument handling
Part of CleanWAD

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes

#include <stdlib.h>
#include <malloc.h>
#include <stdarg.h>
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include "services.h"
#include "filesys.h"
#include "waddef.h"
#include "requests.h"

// Configuration
#define WUSAGE 1												// use 132 column usage message (0=no,1=yes)

// CygWin has poor support for wide terminals
#if defined(__CYGWIN__)
#undef WUSAGE
#define WUSAGE 0
#endif

/**********************************************************************************************************************************/
/******************************************************** Default Options *********************************************************/
/**********************************************************************************************************************************/

static char *def_argv[OPTION_MAXPLUSONE+1]=
{
	""      ,													// dummy for command name
	"-rd"   ,													// remove all but the first of entries having the same name
	"-tw"   ,													// remove wasted space at the end of sound resources
	"-norb" ,													// rebuild (no pack/unpack) blockmaps to remove wasted space
	"-pb"   ,													// pack blockmaps to remove current and potential wasted space
	"-noub" ,													// unpack blockmaps into normalised (but inefficient) format
	"-fb"   ,													// write out modified blockmaps even if the size has not changed
	"-norp" ,													// rebuild (no pack/unpack) pictures to remove wasted space
	"-pp"   ,													// pack pictures to remove current and potential wasted space
	"-noup" ,													// unpack pictures into normalised (but inefficient) format
	"-fp"   ,													// write out modified pictures even if the size has not changed
	"-norl" ,													// reuse lumps where directory entries specify same size and offset
	"-nopl" ,													// share lumps with identical contents between directory entries
	"-noul" ,													// output each lump explicitly even if it could be shared or reused
	"-ar"   ,													// align resources on 32-bit boundaries for speed at cost of space
	"-ad"   ,													// align directory on 32-bit boundaries for speed at cost of space
	"-in"   ,													// identify screen images by name (READ MANUAL BEFORE USING!)
	"-ip"   ,													// identify screen images as pages (READ MANUAL BEFORE USING!)
	"-ig"   ,													// identify screen images as graphics (READ MANUAL BEFORE USING!)
	"-noiv" ,													// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
	"-ds"   ,													// detect sounds by content (READ MANUAL BEFORE USING!)
	"-dm"	,													// detect musics by content (READ MANUAL BEFORE USING!)
	"-dg"	,													// detect graphics by content (READ MANUAL BEFORE USING!)
	"-norn" ,													// identify sounds/musics/dialogs/conversations by recognised names
	"-nolm" ,													// allow nonstandard list marker characters
	"-nonm" ,													// loose markers use names only (READ MANUAL BEFORE USING!)
	"-nomb" ,													// maintain nonstandard blockmaps (preserve non-optimal blocks)
	"-nomp" ,													// maintain nonstandard pictures (preserve non-optimal columns)
	"-nokw" ,													// do not remove _DEUTEX_ lump
	"-nokf" ,													// do not remove PLATFORM lump
	"-nokh" ,													// do not remove HISTORY lump
	"-nokt" ,													// do not remove TAGDESC lump
	"-nokp" ,													// do not remove PC speaker sound effects
	"-nokd" ,													// do not normalise double-letter list markers (e.g., PP_START)
	"-nokb" ,													// do not remove sub-list border markers (e.g., F1_START)
	"-noke" ,													// do not remove empty structured lists
	"-notm" ,													// do not treat multiple instances of structured lists as an error
	"-noqm" ,													// do not treat multiple instances of structured lists as a problem
	"-notl" ,													// do not treat lump reuse as an error
	"-noql" ,													// do not treat lump reuse as a problem
	"-notc" ,													// do not treat lump ineligible lump reuse as an error
	"-noqc" ,													// do not treat lump ineligible lump reuse as a problem
	"-nouc" ,													// do not preserve ineligible lump reuse (if tolerated)
	"-nors" ,													// remove all SCRIPTS and SCRIPTnn entries
	"-nodp" ,													// treat PNAMES as an unclassified lump (separate from TEXTURES)
	"-nolh" ,													// allow nonstandard map name headers (not just E\?M\? and MAP\?\?)
	"-noqh" ,													// do not warn about non-empty map name headers
	"-nofr" ,													// remove duplicate entries even if in different lists
	"-sl"   ,													// sort order to apply to entries in the output file
	"-gn"   ,													// game for which (WAD whose lump names these are) was designed
	"-va"    													// verbosity level for reporting of events and errors
};

static int def_argc=OPTION_MAXPLUSONE+1;						// number of options in default argument argv[] (includes dummy)

/**********************************************************************************************************************************/
/************************************************ Options Structure Initialisation ************************************************/
/**********************************************************************************************************************************/

// Initialise options metrics
static void InitialiseOptionsMetrics (
	options_t *options)											// options whose metrics are to be initialised
{
	// DOCUMENTATION
	/*
		This MUST be called on an options variable BEFORE
		attempting to use InitialiseOptionsValues() on it.
		That function needs the option code letters to be
		set so that it can match options during parsing.
	*/

	// VARIABLES
	int i;

	// PRIMARY ERROR CHECK
	if (options==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise options metrics with NULL argument");
	}

	// INITIALISE THE OPTIONS
	for (i=0;i<OPTION_MAXPLUSONE;i++)
	{
		option_t *entry;
		/**************/
		entry=&((*options)[i]);
		switch (i)
		{
			case OPTION_REMOVE_DUPLICATES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"rd");
				(void) strcpy(entry->name,"remove duplicates");
				(void) strcpy(entry->desc,"remove all but the first of entries having the same name");
				break;
			case OPTION_TRUNCATE_WAVES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"tw");
				(void) strcpy(entry->name,"truncate waves");
				(void) strcpy(entry->desc,"remove wasted space at the end of sound resources");
				break;
			case OPTION_REBUILD_BLOCKMAPS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"rb");
				(void) strcpy(entry->name,"rebuild blockmaps");
				(void) strcpy(entry->desc,"rebuild (no pack/unpack) blockmaps to remove wasted space");
				break;
			case OPTION_PACK_BLOCKMAPS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"pb");
				(void) strcpy(entry->name,"pack blockmaps");
				(void) strcpy(entry->desc,"pack blockmaps to remove current and potential wasted space");
				break;
			case OPTION_UNPACK_BLOCKMAPS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"ub");
				(void) strcpy(entry->name,"unpack blockmaps");
				(void) strcpy(entry->desc,"unpack blockmaps into normalised (but inefficient) format");
				break;
			case OPTION_FLUSH_BLOCKMAPS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"fb");
				(void) strcpy(entry->name,"flush blockmaps");
				(void) strcpy(entry->desc,"write out modified blockmaps even if the size has not changed");
				break;
			case OPTION_REBUILD_PICTURES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"rp");
				(void) strcpy(entry->name,"rebuild pictures");
				(void) strcpy(entry->desc,"rebuild (no pack/unpack) pictures to remove wasted space");
				break;
			case OPTION_PACK_PICTURES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"pp");
				(void) strcpy(entry->name,"pack pictures");
				(void) strcpy(entry->desc,"pack pictures to remove current and potential wasted space");
				break;
			case OPTION_UNPACK_PICTURES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"up");
				(void) strcpy(entry->name,"unpack pictures");
				(void) strcpy(entry->desc,"unpack pictures into normalised (but inefficient) format");
				break;
			case OPTION_FLUSH_PICTURES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"fp");
				(void) strcpy(entry->name,"flush pictures");
				(void) strcpy(entry->desc,"write out modified pictures even if the size has not changed");
				break;
			case OPTION_REUSE_LUMPS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"rl");
				(void) strcpy(entry->name,"reuse lumps");
				(void) strcpy(entry->desc,"reuse lumps where directory entries specify same size and offset");
				break;
			case OPTION_PACK_LUMPS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"pl");
				(void) strcpy(entry->name,"pack lumps");
				(void) strcpy(entry->desc,"share lumps with identical contents between directory entries");
				break;
			case OPTION_UNPACK_LUMPS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"ul");
				(void) strcpy(entry->name,"unpack lumps");
				(void) strcpy(entry->desc,"output each lump explicitly even if it could be shared or reused");
				break;
			case OPTION_ALIGN_RESOURCES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"ar");
				(void) strcpy(entry->name,"align resources");
				(void) strcpy(entry->desc,"align resources on 32-bit boundaries for speed at cost of space");
				break;
			case OPTION_ALIGN_DIRECTORY:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"ad");
				(void) strcpy(entry->name,"align directory");
				(void) strcpy(entry->desc,"align directory on 32-bit boundaries for speed at cost of space");
				break;
			case OPTION_IDENTIFY_NAMES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"in");
				(void) strcpy(entry->name,"identify names");
				(void) strcpy(entry->desc,"identify screen images by name (READ MANUAL BEFORE USING!)");
				break;
			case OPTION_IDENTIFY_PAGES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"ip");
				(void) strcpy(entry->name,"identify pages");
				(void) strcpy(entry->desc,"identify screen images as pages (READ MANUAL BEFORE USING!)");
				break;
			case OPTION_IDENTIFY_GRAPHICS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"ig");
				(void) strcpy(entry->name,"identify graphics");
				(void) strcpy(entry->desc,"identify screen images as graphics (READ MANUAL BEFORE USING!)");
				break;
			case OPTION_IDENTIFY_VOICES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"iv");
				(void) strcpy(entry->name,"identify voices");
				(void) strcpy(entry->desc,"identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)");
				break;
			case OPTION_DETECT_SOUNDS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"ds");
				(void) strcpy(entry->name,"detect sounds");
				(void) strcpy(entry->desc,"detect sounds by content (READ MANUAL BEFORE USING!)");
				break;
			case OPTION_DETECT_MUSICS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"dm");
				(void) strcpy(entry->name,"detect musics");
				(void) strcpy(entry->desc,"detect musics by content (READ MANUAL BEFORE USING!)");
				break;
			case OPTION_DETECT_GRAPHICS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"dg");
				(void) strcpy(entry->name,"detect graphics");
				(void) strcpy(entry->desc,"detect graphics by content (READ MANUAL BEFORE USING!)");
				break;
			case OPTION_RECOGNISED_NAMES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"rn");
				(void) strcpy(entry->name,"recognised names");
				(void) strcpy(entry->desc,"identify sounds/musics/dialogs/conversations by recognised names");
				break;
			case OPTION_LOOSE_MARKERS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"lm");
				(void) strcpy(entry->name,"loose markers");
				(void) strcpy(entry->desc,"allow nonstandard list marker characters");
				break;
			case OPTION_NAMED_MARKERS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"nm");
				(void) strcpy(entry->name,"named markers");
				(void) strcpy(entry->desc,"loose markers use names only (READ MANUAL BEFORE USING!)");
				break;
			case OPTION_MAINTAIN_BLOCKMAPS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"mb");
				(void) strcpy(entry->name,"maintain blockmaps");
				(void) strcpy(entry->desc,"maintain nonstandard blockmaps (preserve non-optimal blocks)");
				break;
			case OPTION_MAINTAIN_PICTURES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"mp");
				(void) strcpy(entry->name,"maintain pictures");
				(void) strcpy(entry->desc,"maintain nonstandard pictures (preserve non-optimal columns)");
				break;
			case OPTION_KEEP_WINTEX:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"kw");
				(void) strcpy(entry->name,"keep wintex");
				(void) strcpy(entry->desc,"do not remove _DEUTEX_ lump");
				break;
			case OPTION_KEEP_PLATFORM:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"kf");
				(void) strcpy(entry->name,"keep platform");
				(void) strcpy(entry->desc,"do not remove PLATFORM lump");
				break;
			case OPTION_KEEP_HISTORY:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"kh");
				(void) strcpy(entry->name,"keep history");
				(void) strcpy(entry->desc,"do not remove HISTORY lump");
				break;
			case OPTION_KEEP_TAGDESC:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"kt");
				(void) strcpy(entry->name,"keep tagdesc");
				(void) strcpy(entry->desc,"do not remove TAGDESC lump");
				break;
			case OPTION_KEEP_PCSFX:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"kp");
				(void) strcpy(entry->name,"keep pcsfx");
				(void) strcpy(entry->desc,"do not remove PC speaker sound effects");
				break;
			case OPTION_KEEP_DOUBLES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"kd");
				(void) strcpy(entry->name,"keep doubles");
				(void) strcpy(entry->desc,"do not normalise double-letter list markers (e.g., PP_START)");
				break;
			case OPTION_KEEP_BORDERS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"kb");
				(void) strcpy(entry->name,"keep borders");
				(void) strcpy(entry->desc,"do not remove sub-list border markers (e.g., F1_START)");
				break;
			case OPTION_KEEP_EMPTIES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"ke");
				(void) strcpy(entry->name,"keep empties");
				(void) strcpy(entry->desc,"do not remove empty structured lists");
				break;
			case OPTION_TOLERATE_MULTIPLES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"tm");
				(void) strcpy(entry->name,"tolerate multiples");
				(void) strcpy(entry->desc,"do not treat multiple instances of structured lists as an error");
				break;
			case OPTION_QUIET_MULTIPLES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"qm");
				(void) strcpy(entry->name,"quiet multiples");
				(void) strcpy(entry->desc,"do not treat multiple instances of structured lists as a problem");
				break;
			case OPTION_TOLERATE_LINKS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"tl");
				(void) strcpy(entry->name,"tolerate links");
				(void) strcpy(entry->desc,"do not treat lump reuse as an error");
				break;
			case OPTION_QUIET_LINKS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"ql");
				(void) strcpy(entry->name,"quiet links");
				(void) strcpy(entry->desc,"do not treat lump reuse as a problem");
				break;
			case OPTION_TOLERATE_CONFLICTS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"tc");
				(void) strcpy(entry->name,"tolerate conflicts");
				(void) strcpy(entry->desc,"do not treat ineligible lump reuse as an error");
				break;
			case OPTION_QUIET_CONFLICTS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"qc");
				(void) strcpy(entry->name,"quiet conflicts");
				(void) strcpy(entry->desc,"do not treat ineligible lump reuse as a problem");
				break;
			case OPTION_UNPACK_CONFLICTS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"uc");
				(void) strcpy(entry->name,"unpack conflicts");
				(void) strcpy(entry->desc,"do not preserve ineligible lump reuse (if tolerated)");
				break;
			case OPTION_REMOVE_SCRIPTS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"rs");
				(void) strcpy(entry->name,"remove scripts");
				(void) strcpy(entry->desc,"remove all SCRIPTS and SCRIPTnn entries");
				break;
			case OPTION_DECLASSIFY_PNAMES:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"dp");
				(void) strcpy(entry->name,"declassify pnames");
				(void) strcpy(entry->desc,"treat PNAMES as an unclassified lump (separate from TEXTURES)");
				break;
			case OPTION_LOOSE_HEADERS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"lh");
				(void) strcpy(entry->name,"loose headers");
				(void) strcpy(entry->desc,"allow nonstandard map name headers (not just E\?M\? and MAP\?\?)");
				break;
			case OPTION_QUIET_HEADERS:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"qh");
				(void) strcpy(entry->name,"quiet headers");
				(void) strcpy(entry->desc,"do not warn about non-empty map name headers");
				break;
			case OPTION_FORCE_REMOVAL:
				entry->type=OPTION_IS_BOOLEAN;
				entry->value.flag=FALSE;
				(void) strcpy(entry->code,"fr");
				(void) strcpy(entry->name,"force removal");
				(void) strcpy(entry->desc,"remove duplicate entries even if in different lists");
				break;
			case OPTION_SORT:
				entry->type=OPTION_IS_SORT;
				entry->value.sort=SORT_NONE;
				(void) strcpy(entry->code,"sS");
				(void) strcpy(entry->name,"sort order");
				(void) strcpy(entry->desc,"sort order to apply to entries in the output file");
				break;
			case OPTION_GAME:
				entry->type=OPTION_IS_GAME;
				entry->value.game=GAME_UNKNOWN;
				(void) strcpy(entry->code,"gG");
				(void) strcpy(entry->name,"game selection");
				(void) strcpy(entry->desc,"game for which (WAD whose lump names these are) was designed");
				break;
			case OPTION_VERBOSITY:
				entry->type=OPTION_IS_VERBOSITY;
				entry->value.verbosity=VERBOSITY_STATEMENTS;
				(void) strcpy(entry->code,"vV");
				(void) strcpy(entry->name,"verbosity level");
				(void) strcpy(entry->desc,"verbosity level for reporting of events and errors");
				break;
			default:
				FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"i\"",i);
				break;
			////
		}
		entry->given=FALSE;
#ifdef PARANOID
		if ( (entry->code[0]=='n') && (entry->code[1]=='o') )
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid master option code \"%s\" (begins with \"no\")",entry->code);
		}
#endif
	}
}

// Initialise options values
static void InitialiseOptionsValues (
	options_t *options,											// options whose values are to be initialised
	option_mode_t option_mode,									// ultimate source of these options
	void (report) (												// function to report error messages with
		unsigned int level,										// error level
		char *message,											// error message format
		...),													// error message arguments
	int argc,													// argc to parse options from
	char *argv[],												// argv to parse options from
	int *last_argp)												// index of last argument parsed
{
	// DOCUMENTATION
	/*
		InitialiseOptionsMetrics() MUST have be called on an
		options variable BEFORE	attempting to run this on it.

		The function pointer REPORT should be be NULL when
		parsing the default options and non-NULL otherwise;
		any error in the default options is a programmer error,
		not a user error and is thus handled with FatalAbort().
	*/

	// METRICS
	#define DESCRIPTION_MAXSIZE 16

	// VARIABLES
	size_t i;
	char description[DESCRIPTION_MAXSIZE+1];

	// PRIMARY ERROR CHECK
	if (options==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise options values with NULL argument");
	}
	if ( (report==NULL) && (option_mode!=OPTION_IS_DEFAULT) )
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise options values with NULL REPORT argument when options_mode is not OPTION_IS_DEFAULT");
	}
	if ( (report!=NULL) && (option_mode==OPTION_IS_DEFAULT) )
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise options values with non-NULL REPORT argument when options_mode is OPTION_IS_DEFAULT");
	}

	// INITIALISE
	*last_argp=0;
	// INITIALISE
	switch (option_mode)
	{
		case OPTION_IS_DEFAULT:
			(void) strcpy(description,"default");
			break;
		case OPTION_IS_ENVIRONMENT:
			(void) strcpy(description,"environment");
			break;
		case OPTION_IS_COMMANDLINE:
			(void) strcpy(description,"command-line");
			break;
		case OPTION_IS_RESULTANT:
			(void) strcpy(description,"resultant");
			break;
		default:
			FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"option_mode\"",option_mode);
			break;
		////
	}

	// PARSE OPTIONS
	for (i=1 ; i<(size_t)(argc) && (argv[i][0]=='-') ; i++)
	{
		size_t j;
		bool_t found,negative;
		option_type_t option_type;
		/************************/
		if (argv[i][1]=='s')
		{
			negative=FALSE;
			option_type=OPTION_IS_SORT;
		}
		else if (argv[i][1]=='g')
		{
			negative=FALSE;
			option_type=OPTION_IS_GAME;
		}
		else if (argv[i][1]=='v')
		{
			negative=FALSE;
			option_type=OPTION_IS_VERBOSITY;
		}
		else
		{
			negative=((argv[i][1]=='n') && (argv[i][2]=='o'))?TRUE:FALSE;
			option_type=OPTION_IS_BOOLEAN;
		}
		switch (option_type)
		{
			case OPTION_IS_BOOLEAN:
				found=FALSE;
				for (j=0 ; j<OPTION_MAXPLUSONE && !found ; j++)
				{
					if (strcmp((*options)[j].code,&argv[i][negative?3:1])==0)
					{
						found=TRUE;
						(*options)[j].given=TRUE;
						(*options)[j].value.flag=(negative?FALSE:TRUE);
					}
				}
				if (!found)
				{
					if (report==NULL)
					{
						FatalAbort(2,__FILE__,__LINE__,"Invalid %s boolean option: %s",description,&argv[i][1]);
					}
					else
					{
						report(2,"Invalid %s boolean option: %s",description,&argv[i][1]);
					}
				}
				break;
			case OPTION_IS_SORT:
				(*options)[OPTION_SORT].given=TRUE;
				switch (argv[i][2])
				{
					case 'n':
						(*options)[OPTION_SORT].value.sort=SORT_NONE;
						break;
					case 'l':
						(*options)[OPTION_SORT].value.sort=SORT_LIST;
						break;
					case 'a':
						(*options)[OPTION_SORT].value.sort=SORT_ALPHABETIC;
						break;
					case 'i':
						(*options)[OPTION_SORT].value.sort=SORT_INTRINSIC;
						break;
					default:
						if (report==NULL)
						{
							FatalAbort(2,__FILE__,__LINE__,"Invalid %s sort option: %s",description,&argv[i][1]);
						}
						else
						{
							report(2,"Invalid %s sort option: %s",description,&argv[i][1]);
						}
						break;
					////
				}
				break;
			case OPTION_IS_GAME:
				(*options)[OPTION_GAME].given=TRUE;
				switch (argv[i][2])
				{
					case 'n':
						(*options)[OPTION_GAME].value.game=GAME_UNKNOWN;
						break;
					case '1':
						(*options)[OPTION_GAME].value.game=GAME_DOOM;
						break;
					case '2':
						(*options)[OPTION_GAME].value.game=GAME_DOOM2;
						break;
					case 't':
						(*options)[OPTION_GAME].value.game=GAME_TNT;
						break;
					case 'p':
						(*options)[OPTION_GAME].value.game=GAME_PLUTONIA;
						break;
					case 'h':
						(*options)[OPTION_GAME].value.game=GAME_HERETIC;
						break;
					case 'x':
						(*options)[OPTION_GAME].value.game=GAME_HEXEN;
						break;
					case 'd':
						(*options)[OPTION_GAME].value.game=GAME_DEATHKINGS;
						break;
					case 's':
						(*options)[OPTION_GAME].value.game=GAME_STRIFE;
						break;
					default:
						if (report==NULL)
						{
							FatalAbort(2,__FILE__,__LINE__,"Invalid %s game option: %s",description,&argv[i][1]);
						}
						else
						{
							report(2,"Invalid %s game option: %s",description,&argv[i][1]);
						}
						break;
					////
				}
				break;
			case OPTION_IS_VERBOSITY:
				(*options)[OPTION_VERBOSITY].given=TRUE;
				switch (argv[i][2])
				{
					case 's':
						(*options)[OPTION_VERBOSITY].value.verbosity=VERBOSITY_STATEMENTS;
						break;
					case 'e':
						(*options)[OPTION_VERBOSITY].value.verbosity=VERBOSITY_ERRORS;
						break;
					case 'w':
						(*options)[OPTION_VERBOSITY].value.verbosity=VERBOSITY_WARNINGS;
						break;
					case 'c':
						(*options)[OPTION_VERBOSITY].value.verbosity=VERBOSITY_CORRECTIONS;
						break;
					case 'p':
						(*options)[OPTION_VERBOSITY].value.verbosity=VERBOSITY_PROGRESS;
						break;
					case 'a':
						(*options)[OPTION_VERBOSITY].value.verbosity=VERBOSITY_ACTIONS;
						break;
					case 'd':
						(*options)[OPTION_VERBOSITY].value.verbosity=VERBOSITY_DETAILS;
						break;
					case 'i':
						(*options)[OPTION_VERBOSITY].value.verbosity=VERBOSITY_INTERNALS;
						break;
					default:
						if (report==NULL)
						{
							FatalAbort(2,__FILE__,__LINE__,"Invalid %s verbosity option: %s",description,&argv[i][1]);
						}
						else
						{
							report(2,"Invalid %s verbosity option: %s",description,&argv[i][1]);
						}
						break;
					////
				}
				break;
			default:
				FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"option_type\"",option_type);
				break;
			////
		}
		*last_argp=(int)(i);
	}
}

/**********************************************************************************************************************************/
/********************************************************* Error Handling *********************************************************/
/**********************************************************************************************************************************/

// List options
static void ListOptions(
	options_t *options,											// options to list
	FILE *console)												// file to list options to
{
	// VARIABLES
	size_t i;

#if WUSAGE==1
	(void) fprintf (console,"  OPTION  MEANING                EXPLANATION                                                       DEFAULT\n");
#else
	(void) fprintf (console,"  OPTION  MEANING                DEFAULT\n");
#endif
	for (i=0;i<OPTION_MAXPLUSONE;i++)
	{
#if WUSAGE==1
		(void) fprintf (console,"    %-2s    %-21s  %-64s  ",(*options)[i].code,(*options)[i].name,(*options)[i].desc);
#else
		(void) fprintf (console,"    %-2s    %-21s  ",(*options)[i].code,(*options)[i].name);
#endif
		switch ((*options)[i].type)
		{
			case OPTION_IS_BOOLEAN:
				if ((*options)[i].value.flag)
				{
					(void) fprintf (console,"ON\n");
				}
				else
				{
					(void) fprintf (console,"OFF\n");
				}
				break;
			case OPTION_IS_SORT:
				switch ((*options)[i].value.sort)
				{
					case SORT_NONE:
						(void) fprintf (console,"None\n");
						break;
					case SORT_LIST:
						(void) fprintf (console,"List\n");
						break;
					case SORT_ALPHABETIC:
						(void) fprintf (console,"Alphabetic\n");
						break;
					case SORT_INTRINSIC:
						(void) fprintf (console,"Intrinsic\n");
						break;
					default:
						FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"options[%d].value.sort\"",(*options)[i].value.sort,i);
						break;
					////
				}
				break;
			case OPTION_IS_GAME:
				switch ((*options)[i].value.game)
				{
					case GAME_UNKNOWN:
						(void) fprintf (console,"Neutral\n");
						break;
					case GAME_DOOM:
						(void) fprintf (console,"DOOM\n");
						break;
					case GAME_DOOM2:
						(void) fprintf (console,"DOOM2\n");
						break;
					case GAME_TNT:
						(void) fprintf (console,"TNT\n");
						break;
					case GAME_PLUTONIA:
						(void) fprintf (console,"Plutonia\n");
						break;
					case GAME_HERETIC:
						(void) fprintf (console,"Heretic\n");
						break;
					case GAME_HEXEN:
						(void) fprintf (console,"HeXen\n");
						break;
					case GAME_DEATHKINGS:
						(void) fprintf (console,"Deathkings\n");
						break;
					case GAME_STRIFE:
						(void) fprintf (console,"Strife\n");
						break;
					default:
						FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"options[%d].value.game\"",(*options)[i].value.game,i);
						break;
					////
				}
				break;
			case OPTION_IS_VERBOSITY:
				switch ((*options)[i].value.verbosity)
				{
					case VERBOSITY_STATEMENTS:
						(void) fprintf (console,"Statements\n");
						break;
					case VERBOSITY_ERRORS:
						(void) fprintf (console,"Errors\n");
						break;
					case VERBOSITY_WARNINGS:
						(void) fprintf (console,"Warnings\n");
						break;
					case VERBOSITY_CORRECTIONS:
						(void) fprintf (console,"Corrections\n");
						break;
					case VERBOSITY_PROGRESS:
						(void) fprintf (console,"Progress\n");
						break;
					case VERBOSITY_ACTIONS:
						(void) fprintf (console,"Actions\n");
						break;
					case VERBOSITY_DETAILS:
						(void) fprintf (console,"Details\n");
						break;
					case VERBOSITY_INTERNALS:
						(void) fprintf (console,"Internals\n");
						break;
					default:
						FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"options[%d].value.verbosity\"",(*options)[i].value.verbosity,i);
						break;
					////
				}
				break;
			default:
				FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"options[%d].type\"",(*options)[i].type,i);
				break;
			////
		}
	}
}

// Print usage message
static void Usage(
	FILE *console)												// file to write message to
{
	// VARIABLES
	int dummy_int=0;
	options_t default_options;

	// SET DEFAULT OPTIONS FOR USAGE MESSAGE
	InitialiseOptionsMetrics (
		&default_options);										// options whose metrics are to be initialised
	InitialiseOptionsValues (
		&default_options,										// options whose values are to be initialised
		OPTION_IS_DEFAULT,										// ultimate source of these options
		NULL,													// function to report error messages with
		def_argc,												// argc to parse options from
		def_argv,												// argv to parse options from
		&dummy_int);											// index of last argument parsed

	// PRINT USAGE MESSAGE
	(void) fprintf (console,"\n");
	(void) fprintf (console,"Usage: cleanwad [-[no]option...] infilename outfilename\n");
	(void) fprintf (console,"\n");
	ListOptions(
		&default_options,										// options to list
		console);												// file to list options to
	(void) fprintf (console,"\n");
	(void) fprintf (console,"  SORT:\n");
	(void) fprintf (console,"    N:None L:List A:Alphabetic I:Intrinsic\n");
	(void) fprintf (console,"\n");
	(void) fprintf (console,"  GAME:\n");
#if WUSAGE==1
	(void) fprintf (console,"    N:Neutral 1:Doom 2:Doom2 T:TNT P:Plutonia H:Heretic X:HeXen D:Deathkings S:Strife\n");
#else
	(void) fprintf (console,"    N:Neutral\n");
	(void) fprintf (console,"    1:Doom     2:Doom2  T:TNT         P:Plutonia\n");
	(void) fprintf (console,"    H:Heretic  X:HeXen  D:Deathkings  S:Strife\n");
#endif
	(void) fprintf (console,"\n");
	(void) fprintf (console,"  VERBOSITY:\n");
#if WUSAGE==1
	(void) fprintf (console,"    S:Statements E:Errors W:Warnings C:Corrections P:Progress A:Actions D:Details I:Internals\n");
#else
	(void) fprintf (console,"    S:Statements E:Errors  W:Warnings C:Corrections\n");
	(void) fprintf (console,"    P:Progress   A:Actions D:Details  I:Internals\n");
#endif
	(void) fprintf (console,"\n");
	(void) fprintf (console,"Refer to MANUAL.TXT for detailed information\n");
}

/**********************************************************************************************************************************/
/********************************************************* Usage Message **********************************************************/
/**********************************************************************************************************************************/

// Bad argument logger
static void UsageAbort(
	unsigned int level,											// error level
	char *message,												// error message format
	...)														// error message arguments
{
	// VARIABLES
	FILE *console;

	// PRIMARY ERROR CHECK
	if (level==0)
	{
		FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"level\"",level);
	}

	// SET CONSOLE FILE
	console=((level==1)?stdout:stderr);

	// PRINT ERROR HEADING if applicable
	if (level>1)
	{
		(void) fprintf (console,"\n");
		(void) fprintf (console, "\7");
		if ( (message!=NULL) && (strlen(message)!=0) )
		{
#ifndef S_SPLINT_S
			va_list ap;
			(void) fprintf (console, "ERROR: ");
			va_start(ap, message);
			(void) vfprintf (console,message,ap);
			va_end(ap);
#endif
		}
		(void) fprintf (console,"\n");
	}

	//PRINT THE USAGE MESSAGE
	Usage(console);

	// ABORT THE PROGRAM
	exit ((int)(level));
}

/**********************************************************************************************************************************/
/******************************************** WAD File Processing Request Maintanance *********************************************/
/**********************************************************************************************************************************/

// Initialise a WAD file processing request
void WADInitRequest (
	WADREQ_t **WADREQp)											// WAD file processing request to be initialised
{
	// VARIABLES
	size_t sizeb;
	WADREQ_t *objptr;
	options_t *optptr;

	// PRIMARY ERROR CHECK
	if (WADREQp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise WAD file processing request with NULL argument pointer");
	}
	if (*WADREQp!=NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise WAD file processing request with non-NULL argument");
	}

	// ALLOCATE MEMORY FOR THE WADREQ
	sizeb=sizeof(WADREQ_t);
	objptr=malloc(sizeb);
	if (objptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for WAD file processing request",sizeb);
	}
	(void) memset(objptr,0,sizeb);

	// ALLOCATE MEMORY FOR THE OPTIONS MEMBER
	sizeb=sizeof(options_t);
	optptr=malloc(sizeb);
	if (optptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for options member of WAD file processing request",sizeb);
	}
	(void) memset(optptr,0,sizeb);

	// INITIALISE THE WADREQ STATIC FIELDS
	objptr->request_type=WADREQ_UNKNOWN;						// type of request
	objptr->infilename=NULL;									// input file name
	objptr->outfilename=NULL;									// output file name
	objptr->options=optptr;										// list of options

	// INITIALISE THE OPTIONS
	InitialiseOptionsMetrics (
		optptr);												// options whose metrics are to be initialised

	// RETURN RESULT
	*WADREQp=objptr;
}

// Deinitialise a WAD file processing request
void WADDoneRequest (
	WADREQ_t **WADREQp)											// WAD file processing request to be deinitialised
{
	// VARIABLES
	WADREQ_t *objptr;

	// PRIMARY ERROR CHECK
	if (WADREQp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise WAD file processing request with NULL argument pointer");
	}
	if (*WADREQp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise WAD file processing request with NULL argument");
	}

	// GET POINTER TO THE REQUEST
	objptr=*WADREQp;

	// DEALLOCATE MEMORY FOR THE WADREQ INPUT FILE NAME
	if (objptr->infilename)
	{
		free(objptr->infilename);
	}

	// DEALLOCATE MEMORY FOR THE WADREQ OUTPUT FILE NAME
	if (objptr->outfilename)
	{
		free(objptr->outfilename);
	}

	// DEALLOCATE MEMORY FOR THE WADREQ OPTIONS
	if (objptr->options)
	{
		free(objptr->options);
	}

	// DEALLOCATE MEMORY FOR THE WADREQ
	free(*WADREQp);

	// RETURN RESULT
	*WADREQp=NULL;
}

/**********************************************************************************************************************************/
/***************************************************** Options and Arguments ******************************************************/
/**********************************************************************************************************************************/

// Return default options argc
int WADGetDefaultArgC (
	void)														// no arguments
{
	return def_argc;
}

// Return default options argv
char **WADGetDefaultArgV (
	void)														// no arguments
{
	return def_argv;
}

// WADValidate options
void WADValidateOptions (
	options_t *options,											// options to be validated
	option_mode_t option_mode)									// ultimate source of these options
{
	// METRICS
	#define DESCRIPTION_MAXSIZE 16

	// VARIABLES
	char description[DESCRIPTION_MAXSIZE+1];

	// INITIALISE
	switch (option_mode)
	{
		case OPTION_IS_DEFAULT:
			(void) strcpy(description,"default");
			break;
		case OPTION_IS_ENVIRONMENT:
			(void) strcpy(description,"environment");
			break;
		case OPTION_IS_COMMANDLINE:
			(void) strcpy(description,"command-line");
			break;
		case OPTION_IS_RESULTANT:
			(void) strcpy(description,"resultant");
			break;
		default:
			FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"option_mode\"",option_mode);
			break;
		////
	}

	// VALIDATE BLOCKMAP OPTIONS
	if (
	    (
	     ((*options)[OPTION_REBUILD_BLOCKMAPS].value.flag) &&
	     ((*options)[OPTION_REBUILD_BLOCKMAPS].given     ) &&
	     ((*options)[OPTION_PACK_BLOCKMAPS].value.flag   ) &&
	     ((*options)[OPTION_PACK_BLOCKMAPS].given        )
	    )
	    ||
	    (
	     ((*options)[OPTION_REBUILD_BLOCKMAPS].value.flag) &&
	     ((*options)[OPTION_REBUILD_BLOCKMAPS].given     ) &&
	     ((*options)[OPTION_UNPACK_BLOCKMAPS].value.flag ) &&
	     ((*options)[OPTION_UNPACK_BLOCKMAPS].given      )
	    )
	    ||
	    (
	     ((*options)[OPTION_PACK_BLOCKMAPS].value.flag   ) &&
	     ((*options)[OPTION_PACK_BLOCKMAPS].given        ) &&
	     ((*options)[OPTION_UNPACK_BLOCKMAPS].value.flag ) &&
	     ((*options)[OPTION_UNPACK_BLOCKMAPS].given      )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting for BLOCKMAP options (rb, pb, ub): more than one is ON",description);
		}
		else
		{
			UsageAbort(2,"Invalid %s setting for BLOCKMAP options (rb, pb, ub): more than one is ON",description);
		}
	}

	// VALIDATE PICTURE OPTIONS
	if (
	    (
	     ((*options)[OPTION_REBUILD_PICTURES].value.flag) &&
	     ((*options)[OPTION_REBUILD_PICTURES].given     ) &&
	     ((*options)[OPTION_PACK_PICTURES].value.flag   ) &&
	     ((*options)[OPTION_PACK_PICTURES].given        )
	    )
	    ||
	    (
	     ((*options)[OPTION_REBUILD_PICTURES].value.flag) &&
	     ((*options)[OPTION_REBUILD_PICTURES].given     ) &&
	     ((*options)[OPTION_UNPACK_PICTURES].value.flag ) &&
	     ((*options)[OPTION_UNPACK_PICTURES].given      )
	    )
	    ||
	    (
	     ((*options)[OPTION_PACK_PICTURES].value.flag   ) &&
	     ((*options)[OPTION_PACK_PICTURES].given        ) &&
	     ((*options)[OPTION_UNPACK_PICTURES].value.flag ) &&
	     ((*options)[OPTION_UNPACK_PICTURES].given      )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting for PICTURE options (rp, pp, up): more than one is ON",description);
		}
		else
		{
			UsageAbort(2,"Invalid %s setting for PICTURE options (rp, pp, up): more than one is ON",description);
		}
	}

	// VALIDATE LUMP REUSE HANDLING OPTIONS
	if (
	    (
	     ((*options)[OPTION_REUSE_LUMPS].value.flag ) &&
	     ((*options)[OPTION_REUSE_LUMPS].given      ) &&
	     ((*options)[OPTION_PACK_LUMPS].value.flag  ) &&
	     ((*options)[OPTION_PACK_LUMPS].given       )
	    )
	    ||
	    (
	     ((*options)[OPTION_REUSE_LUMPS].value.flag ) &&
	     ((*options)[OPTION_REUSE_LUMPS].given      ) &&
	     ((*options)[OPTION_UNPACK_LUMPS].value.flag) &&
	     ((*options)[OPTION_UNPACK_LUMPS].given     )
	    )
	    ||
	    (
	     ((*options)[OPTION_PACK_LUMPS].value.flag  ) &&
	     ((*options)[OPTION_PACK_LUMPS].given       ) &&
	     ((*options)[OPTION_UNPACK_LUMPS].value.flag) &&
	     ((*options)[OPTION_UNPACK_LUMPS].given     )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting for LUMP REUSE HANDLING options (rl, pl, ul): more than one is ON",description);
		}
		else
		{
			UsageAbort(2,"Invalid %s setting for LUMP REUSE HANDLING options (rl, pl, ul): more than one is ON",description);
		}
	}

	// VALIDATE IDENTIFY OPTIONS
	if (
	    (
	     (!(*options)[OPTION_IDENTIFY_NAMES].value.flag   ) &&
	     ( (*options)[OPTION_IDENTIFY_NAMES].given        )
	    )
	    &&
	    (
	     (!(*options)[OPTION_IDENTIFY_PAGES].value.flag   ) &&
	     ( (*options)[OPTION_IDENTIFY_PAGES].given        )
	    )
	    &&
	    (
	     (!(*options)[OPTION_IDENTIFY_GRAPHICS].value.flag) &&
	     ( (*options)[OPTION_IDENTIFY_GRAPHICS].given     )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting for IDENTIFY options (in, ip, ig): all are OFF",description);
		}
		else
		{
			UsageAbort(2,"Invalid %s setting for IDENTIFY options (in, ip, ig): all are OFF",description);
		}
	}

	// CROSS-VALIDATE OPTION_IDENTIFY_VOICES WITH GAME
	if (
	    (
	     ((*options)[OPTION_IDENTIFY_VOICES].value.flag   ) &&
	     ((*options)[OPTION_IDENTIFY_VOICES].given        )
	    )
	    &&
	    (
	     ((*options)[OPTION_GAME].given                   ) &&
	     ((*options)[OPTION_GAME].value.game!=GAME_UNKNOWN) &&
	     ((*options)[OPTION_GAME].value.game!=GAME_STRIFE )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting: -iv is unnecessary if GAME is neither NEUTRAL nor STRIFE",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Redundant %s setting: -iv is unnecessary if GAME is neither NEUTRAL nor STRIFE",description);
		}
	}

	// VALIDATE STRUCTURED LIST OPTIONS
	if (
	    ((*options)[OPTION_TOLERATE_MULTIPLES].value.flag  ) &&
	    ((*options)[OPTION_TOLERATE_MULTIPLES].given       ) &&
	    ((*options)[OPTION_QUIET_MULTIPLES].value.flag     ) &&
	    ((*options)[OPTION_QUIET_MULTIPLES].given          )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting for STRUCTURED LIST options (tm, qm): more than one is ON",description);
		}
		else
		{
			UsageAbort(2,"Invalid %s setting for STRUCTURED LIST options (tm, qm): more than one is ON",description);
		}
	}

	// VALIDATE LUMP REUSE TREATMENT OPTIONS
	if (
	    (
	     ((*options)[OPTION_TOLERATE_LINKS].value.flag) &&
	     ((*options)[OPTION_TOLERATE_LINKS].given     ) &&
	     ((*options)[OPTION_QUIET_LINKS].value.flag   ) &&
	     ((*options)[OPTION_QUIET_LINKS].given        )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting for LUMP REUSE TREATMENT options (tl, ql): more than one is ON",description);
		}
		else
		{
			UsageAbort(2,"Invalid %s setting for LUMP REUSE TREATMENT options (tl, ql): more than one is ON",description);
		}
	}
	if (
	    (
	     ((*options)[OPTION_TOLERATE_CONFLICTS].value.flag) &&
	     ((*options)[OPTION_TOLERATE_CONFLICTS].given     ) &&
	     ((*options)[OPTION_QUIET_CONFLICTS].value.flag   ) &&
	     ((*options)[OPTION_QUIET_CONFLICTS].given        )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting for LUMP REUSE TREATMENT options (tc, qc): more than one is ON",description);
		}
		else
		{
			UsageAbort(2,"Invalid %s setting for LUMP REUSE TREATMENT options (tc, qc): more than one is ON",description);
		}
	}

	// CROSS-VALIDATE LUMP REUSE ENTRY AND TREATMENT OPTIONS
	if (
	    (!(
	       ((*options)[OPTION_REUSE_LUMPS].value.flag ) ||
	       ((*options)[OPTION_PACK_LUMPS].value.flag  ) ||
	       ((*options)[OPTION_UNPACK_LUMPS].value.flag)
	    ) )
	    &&
	    (!(
	       ((*options)[OPTION_REUSE_LUMPS].given ) ||
	       ((*options)[OPTION_PACK_LUMPS].given  ) ||
	       ((*options)[OPTION_UNPACK_LUMPS].given)
	    ) )
	    &&
	    (!(
	       ((*options)[OPTION_TOLERATE_LINKS].value.flag ) ||
	       ((*options)[OPTION_QUIET_LINKS].value.flag    )
	    ) )
	    &&
	    (!(
	       ((*options)[OPTION_TOLERATE_LINKS].given ) ||
	       ((*options)[OPTION_QUIET_LINKS].given    )
	    ) )
	    &&
	    (
	     ((*options)[OPTION_TOLERATE_CONFLICTS].value.flag) &&
	     ((*options)[OPTION_TOLERATE_CONFLICTS].given     )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting for LUMP REUSE options: (-tc) without (one of: -rl, -pl, -ul, -tl or -ql)",description);
		}
		else
		{
			UsageAbort(2,"Invalid %s setting for LUMP REUSE options: (-tc) without (one of: -rl, -pl, -ul, -tl or -ql)",description);
		}
	}
	if (
	    (!(
	       ((*options)[OPTION_REUSE_LUMPS].value.flag ) ||
	       ((*options)[OPTION_PACK_LUMPS].value.flag  ) ||
	       ((*options)[OPTION_UNPACK_LUMPS].value.flag)
	    ) )
	    &&
	    (!(
	       ((*options)[OPTION_REUSE_LUMPS].given ) ||
	       ((*options)[OPTION_PACK_LUMPS].given  ) ||
	       ((*options)[OPTION_UNPACK_LUMPS].given)
	    ) )
	    &&
	    (!(
	       ((*options)[OPTION_TOLERATE_LINKS].value.flag ) ||
	       ((*options)[OPTION_QUIET_LINKS].value.flag    )
	    ) )
	    &&
	    (!(
	       ((*options)[OPTION_TOLERATE_LINKS].given ) ||
	       ((*options)[OPTION_QUIET_LINKS].given    )
	    ) )
	    &&
	    (
	     ((*options)[OPTION_QUIET_CONFLICTS].value.flag) &&
	     ((*options)[OPTION_QUIET_CONFLICTS].given     )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting for LUMP REUSE options: (-qc) without (one of: -rl, -pl, -ul, -tl or -ql)",description);
		}
		else
		{
			UsageAbort(2,"Invalid %s setting for LUMP REUSE options: (-qc) without (one of: -rl, -pl, -ul, -tl or -ql)",description);
		}
	}

	// CHECK FOR OPTION_REMOVE_SCRIPTS BEING USED IN STRIFE
	//
	// The -rs option is only used for HeXen ACS source code,
	// but conversation scripts in Strife have matching names.
	// We can't even disallow it, because somebody might make
	// a "Strife in HeXen format" map someday. Warn the user!
	//
	if (
	    ((*options)[OPTION_REMOVE_SCRIPTS].value.flag) &&
	    ((*options)[OPTION_GAME].value.game==GAME_STRIFE)
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Ambiguous %s meaning for REMOVE SCRIPTS (-rs)  option when GAME is STRIFE",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Ambiguous %s meaning for REMOVE SCRIPTS (-rs) option when GAME is STRIFE",description);
		}
	}

	// CROSS-VALIDATE ENTRY PROCESSING AND TREATMENT OPTIONS
	// AND A FEW OTHER USAGES THAT ARE POTENTIALLY REDUNDANT.
//#if 0
	//
	// This is nitpicking and probably will not be checked in
	// the final version. The end user would have to really not
	// understand what they are doing in order to trigger these
	// warnings. However, they are not really a problem.
	//
	if (
	    (
	     (
	      (!(*options)[OPTION_REBUILD_BLOCKMAPS].value.flag) &&
	      ( (*options)[OPTION_REBUILD_BLOCKMAPS].given     )
	     )
	     &&
	     (
	      (!(*options)[OPTION_PACK_BLOCKMAPS].value.flag   ) &&
	      ( (*options)[OPTION_PACK_BLOCKMAPS].given        )
	     )
	     &&
	     (
	      (!(*options)[OPTION_UNPACK_BLOCKMAPS].value.flag ) &&
	      ( (*options)[OPTION_UNPACK_BLOCKMAPS].given      )
	     )
	    )
	    &&
	    (
	     ( (*options)[OPTION_FLUSH_BLOCKMAPS].value.flag   ) &&
	     ( (*options)[OPTION_FLUSH_BLOCKMAPS].given        )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting: -fb is unnecessary with none of -rb, -pb or -ub",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Redundant %s setting: -fb is unnecessary with none of -rb, -pb or -ub",description);
		}
	}
	if (
	    (
	     (
	      (!(*options)[OPTION_REBUILD_BLOCKMAPS].value.flag) &&
	      ( (*options)[OPTION_REBUILD_BLOCKMAPS].given     )
	     )
	     &&
	     (
	      (!(*options)[OPTION_PACK_BLOCKMAPS].value.flag   ) &&
	      ( (*options)[OPTION_PACK_BLOCKMAPS].given        )
	     )
	     &&
	     (
	      (!(*options)[OPTION_UNPACK_BLOCKMAPS].value.flag ) &&
	      ( (*options)[OPTION_UNPACK_BLOCKMAPS].given      )
	     )
	    )
	    &&
	    (
	     ( (*options)[OPTION_MAINTAIN_BLOCKMAPS].value.flag) &&
	     ( (*options)[OPTION_MAINTAIN_BLOCKMAPS].given     )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting: -mb is unnecessary with none of -rb, -pb or -ub",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Redundant %s setting: -mb is unnecessary with none of -rb, -pb or -ub",description);
		}
	}
	if (
	    (
	     (
	      (!(*options)[OPTION_REBUILD_PICTURES].value.flag) &&
	      ( (*options)[OPTION_REBUILD_PICTURES].given     )
	     )
	     &&
	     (
	      (!(*options)[OPTION_PACK_PICTURES].value.flag   ) &&
	      ( (*options)[OPTION_PACK_PICTURES].given        )
	     )
	     &&
	     (
	      (!(*options)[OPTION_UNPACK_PICTURES].value.flag ) &&
	      ( (*options)[OPTION_UNPACK_PICTURES].given      )
	     )
	    )
	    &&
	    (
	     ( (*options)[OPTION_FLUSH_PICTURES].value.flag   ) &&
	     ( (*options)[OPTION_FLUSH_PICTURES].given        )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting: -fp is unnecessary with none of -rp, -pp or -up",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Redundant %s setting: -fp is unnecessary with none of -rp, -pp or -up",description);
		}
	}
	if (
	    (
	     (
	      (!(*options)[OPTION_REBUILD_PICTURES].value.flag) &&
	      ( (*options)[OPTION_REBUILD_PICTURES].given     )
	     )
	     &&
	     (
	      (!(*options)[OPTION_PACK_PICTURES].value.flag   ) &&
	      ( (*options)[OPTION_PACK_PICTURES].given        )
	     )
	     &&
	     (
	      (!(*options)[OPTION_UNPACK_PICTURES].value.flag ) &&
	      ( (*options)[OPTION_UNPACK_PICTURES].given      )
	     )
	    )
	    &&
	    (
	     ( (*options)[OPTION_MAINTAIN_PICTURES].value.flag) &&
	     ( (*options)[OPTION_MAINTAIN_PICTURES].given     )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting: -mp is unnecessary with none of -rp, -pp or -up",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Redundant %s setting: -mp is unnecessary with none of -rp, -pp or -up",description);
		}
	}
	if (
	    (
	     ( (*options)[OPTION_NAMED_MARKERS].value.flag) &&
	     ( (*options)[OPTION_NAMED_MARKERS].given     )
	    )
	    &&
	    (
	     (!(*options)[OPTION_LOOSE_MARKERS].value.flag) &&
	     ( (*options)[OPTION_LOOSE_MARKERS].given     )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting: -nm is unnecessary without -lm",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Redundant %s setting: -nm is unnecessary without -lm",description);
		}
	}
	if (
	    (
	     (
	      ((*options)[OPTION_REUSE_LUMPS].value.flag  ) &&
	      ((*options)[OPTION_REUSE_LUMPS].given       )
	     )
	     ||
	     (
	      ((*options)[OPTION_PACK_LUMPS].value.flag   ) &&
	      ((*options)[OPTION_PACK_LUMPS].given        )
	     )
	     ||
	     (
	      ((*options)[OPTION_UNPACK_LUMPS].value.flag ) &&
	      ((*options)[OPTION_UNPACK_LUMPS].given      )
	     )
	    )
	    &&
	    (
	     ((*options)[OPTION_TOLERATE_LINKS].value.flag) &&
	     ((*options)[OPTION_TOLERATE_LINKS].given     )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting: -tl is unnecessary with any of -rl, -pl or -ul",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Redundant %s setting: -tl is unnecessary with any of -rl, -pl or -ul",description);
		}
	}
	if (
	    (
	     (
	      ((*options)[OPTION_REUSE_LUMPS].value.flag  ) &&
	      ((*options)[OPTION_REUSE_LUMPS].given       )
	     )
	     ||
	     (
	      ((*options)[OPTION_PACK_LUMPS].value.flag   ) &&
	      ((*options)[OPTION_PACK_LUMPS].given        )
	     )
	     ||
	     (
	      ((*options)[OPTION_UNPACK_LUMPS].value.flag ) &&
	      ((*options)[OPTION_UNPACK_LUMPS].given      )
	     )
	    )
	    &&
	    (
	     ((*options)[OPTION_QUIET_LINKS].value.flag   ) &&
	     ((*options)[OPTION_QUIET_LINKS].given        )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting: -ql is unnecessary with any of -rl, -pl or -ul",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Redundant %s setting: -ql is unnecessary with any of -rl, -pl or -ul",description);
		}
	}
	if (
	    (
	     ( (*options)[OPTION_FORCE_REMOVAL].value.flag) &&
	     ( (*options)[OPTION_FORCE_REMOVAL].given     )
	    )
	    &&
	    (
	     (!(*options)[OPTION_REMOVE_DUPLICATES].value.flag) &&
	     ( (*options)[OPTION_REMOVE_DUPLICATES].given     )
	    )
	   )
	{
		if (option_mode==OPTION_IS_DEFAULT)
		{
			FatalAbort(1,__FILE__,__LINE__,"Invalid %s setting: -fr is unnecessary without -rd",description);
		}
		else
		{
			EventState(VERBOSITY_WARNINGS,"Redundant %s setting: -fr is unnecessary without -rd",description);
		}
	}
//#endif
}

// Merge options
void WADMergeOptions (
	options_t *old_options,										// options to be merged into
	options_t *new_options)										// options to merge into them
{
	// DOCUMENTATION
	/*
		Merge new_options into old_options. This function is
		used to add, in a consistent way, some given options to
		any existing or default options that may have been set.

		Both sets of options must have been validated in their
		own rights with WADValidateOptions() before using this.
	*/

	// VARIABLES
	option_enum_t i;

	// PRIMARY ERROR CHECK
	if (*old_options==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to merge options with NULL \"old_options\" argument");
	}
	if (*new_options==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to parse options with NULL \"new_options\" argument");
	}

	// MERGE THE OPTIONS
	for (i=0;i<OPTION_MAXPLUSONE;i++)
	{
		switch (i)
		{
			case OPTION_REMOVE_DUPLICATES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_TRUNCATE_WAVES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_REBUILD_BLOCKMAPS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_REBUILD_BLOCKMAPS].value.flag=TRUE;
						(*old_options)[OPTION_PACK_BLOCKMAPS   ].value.flag=FALSE;
						(*old_options)[OPTION_UNPACK_BLOCKMAPS ].value.flag=FALSE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_PACK_BLOCKMAPS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_REBUILD_BLOCKMAPS].value.flag=FALSE;
						(*old_options)[OPTION_PACK_BLOCKMAPS   ].value.flag=TRUE;
						(*old_options)[OPTION_UNPACK_BLOCKMAPS ].value.flag=FALSE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_UNPACK_BLOCKMAPS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_REBUILD_BLOCKMAPS].value.flag=FALSE;
						(*old_options)[OPTION_PACK_BLOCKMAPS   ].value.flag=FALSE;
						(*old_options)[OPTION_UNPACK_BLOCKMAPS ].value.flag=TRUE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_FLUSH_BLOCKMAPS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_REBUILD_PICTURES:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_REBUILD_PICTURES].value.flag=TRUE;
						(*old_options)[OPTION_PACK_PICTURES   ].value.flag=FALSE;
						(*old_options)[OPTION_UNPACK_PICTURES ].value.flag=FALSE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_PACK_PICTURES:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_REBUILD_PICTURES].value.flag=FALSE;
						(*old_options)[OPTION_PACK_PICTURES   ].value.flag=TRUE;
						(*old_options)[OPTION_UNPACK_PICTURES ].value.flag=FALSE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_UNPACK_PICTURES:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_REBUILD_PICTURES].value.flag=FALSE;
						(*old_options)[OPTION_PACK_PICTURES   ].value.flag=FALSE;
						(*old_options)[OPTION_UNPACK_PICTURES ].value.flag=TRUE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_FLUSH_PICTURES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_REUSE_LUMPS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_REUSE_LUMPS ].value.flag=TRUE;
						(*old_options)[OPTION_PACK_LUMPS  ].value.flag=FALSE;
						(*old_options)[OPTION_UNPACK_LUMPS].value.flag=FALSE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_PACK_LUMPS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_REUSE_LUMPS ].value.flag=FALSE;
						(*old_options)[OPTION_PACK_LUMPS  ].value.flag=TRUE;
						(*old_options)[OPTION_UNPACK_LUMPS].value.flag=FALSE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_UNPACK_LUMPS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_REUSE_LUMPS ].value.flag=FALSE;
						(*old_options)[OPTION_PACK_LUMPS  ].value.flag=FALSE;
						(*old_options)[OPTION_UNPACK_LUMPS].value.flag=TRUE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_ALIGN_RESOURCES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_ALIGN_DIRECTORY:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_IDENTIFY_NAMES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_IDENTIFY_PAGES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_IDENTIFY_GRAPHICS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_IDENTIFY_VOICES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_DETECT_SOUNDS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_DETECT_MUSICS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_DETECT_GRAPHICS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_RECOGNISED_NAMES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_LOOSE_MARKERS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_NAMED_MARKERS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_MAINTAIN_BLOCKMAPS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_MAINTAIN_PICTURES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_KEEP_WINTEX:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_KEEP_PLATFORM:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_KEEP_HISTORY:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_KEEP_TAGDESC:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_KEEP_PCSFX:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_KEEP_DOUBLES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_KEEP_BORDERS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_KEEP_EMPTIES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_TOLERATE_MULTIPLES:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_TOLERATE_MULTIPLES].value.flag=TRUE;
						(*old_options)[OPTION_QUIET_MULTIPLES   ].value.flag=FALSE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_QUIET_MULTIPLES:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_TOLERATE_MULTIPLES].value.flag=FALSE;
						(*old_options)[OPTION_QUIET_MULTIPLES   ].value.flag=TRUE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_TOLERATE_LINKS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_TOLERATE_LINKS].value.flag=TRUE;
						(*old_options)[OPTION_QUIET_LINKS   ].value.flag=FALSE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_QUIET_LINKS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_TOLERATE_LINKS].value.flag=FALSE;
						(*old_options)[OPTION_QUIET_LINKS   ].value.flag=TRUE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_TOLERATE_CONFLICTS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_TOLERATE_CONFLICTS].value.flag=TRUE;
						(*old_options)[OPTION_QUIET_CONFLICTS   ].value.flag=FALSE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_QUIET_CONFLICTS:
				if ((*new_options)[i].given)
				{
					if ((*new_options)[i].value.flag)
					{
						(*old_options)[OPTION_TOLERATE_CONFLICTS].value.flag=FALSE;
						(*old_options)[OPTION_QUIET_CONFLICTS   ].value.flag=TRUE;
					}
					else
					{
						(*old_options)[i].value.flag=FALSE;
					}
				}
				break;
			case OPTION_UNPACK_CONFLICTS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_REMOVE_SCRIPTS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_DECLASSIFY_PNAMES:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_LOOSE_HEADERS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_QUIET_HEADERS:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_FORCE_REMOVAL:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.flag=(*new_options)[i].value.flag;
				}
				break;
			case OPTION_SORT:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.sort=(*new_options)[i].value.sort;
				}
				break;
			case OPTION_GAME:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.game=(*new_options)[i].value.game;
				}
				break;
			case OPTION_VERBOSITY:
				if ((*new_options)[i].given)
				{
					(*old_options)[i].value.verbosity=(*new_options)[i].value.verbosity;
				}
				break;
			default:
				FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"i\"",i);
				break;
			////
		}
	}
}

// Parse options
void WADParseOptions (
	options_t *options,											// options parsed from arguments
	option_mode_t option_mode,									// ultimate source of these options
	int argc,													// program argument count
	char *argv[],												// program argument strings
	int *last_argp)												// index of last argument parsed
{
	// PRIMARY ERROR CHECK
	if (options==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to parse options with NULL argument");
	}

	// INITIALISE
	*last_argp=0;

	// IMPOSE METRICS
	InitialiseOptionsMetrics (
		options);												// options whose metrics are to be initialised

	// IMPOSE VALUES
	InitialiseOptionsValues (
		options,												// options whose values are to be initialised
		option_mode,											// ultimate source of these options
		((option_mode==OPTION_IS_DEFAULT)?NULL:UsageAbort),		// function to report error messages with
		argc,													// argc to parse options from
		argv,													// argv to parse options from
		last_argp);												// index of last argument parsed
}

// Parse command line arguments
void WADParseCommandLineArguments (
	int argc,													// program argument count
	char *argv[],												// program argument strings
	int last_argp,												// index of last option argument parsed
	WADREQ_t *WADREQ)											// WAD file processing request
{
	// DOCUMENTATION
	/*
		This function should be called when all options have
		been parsed, after which all remaining arguments should
		be filenames and suchlike. The LAST_ARGP input variable
		tells the function where to start parsing (after that).
	*/

	// VARIABLES
	size_t last_arg;

	// PRIMARY ERROR CHECK
	if (WADREQ==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to parse command-line arguments with NULL WAD file processing request argument");
	}

	// INITIALISE
	last_arg=(size_t)(last_argp);

	// GET NAME OF INPUT FILE
	if (last_arg==(size_t)(argc-1))
	{
		UsageAbort(1,"Too few arguments");
	}
	last_arg++;
	FILEGetFullSpec(argv[last_arg],&WADREQ->infilename);
#if __FILENAMESCC__==1											// if can PRESENT filenames case insensitive...
#if FNCASE==1													// lowercase them
	lowercase(WADREQ->infilename);
#endif
#if FNCASE==2													// uppercase them
	uppercase(WADREQ->infilename);
#endif
#endif

	// GET NAME OF OUTPUT FILE
	if (last_arg==(size_t)argc-1)
	{
		UsageAbort(1,"Too few arguments");
	}
	last_arg++;
	FILEGetFullSpec(argv[last_arg],&WADREQ->outfilename);
#if __FILENAMESCC__==1											// if can PRESENT filenames case insensitive...
#if FNCASE==1													// lowercase them
	lowercase(WADREQ->outfilename);
#endif
#if FNCASE==2													// uppercase them
	uppercase(WADREQ->outfilename);
#endif
#endif

	// PENULTIMATE ERROR CHECK
	/*
		// Same file name? Not perfect, as it ignores links,
		// but might as well check this because it is easy.
	*/
#if __FILENAMESCI__==1											// if filenames actually ARE case insensitive...
	if (cimatch(WADREQ->infilename,WADREQ->outfilename))		// case insensitive file names
#else
	if (csmatch(WADREQ->infilename,WADREQ->outfilename))		// case sensitive file names
#endif
	{
		UsageAbort(2,"Input file same as output file");
	}

	// FINAL ERROR CHECK
	if (last_arg<((size_t)argc-1))
	{
		last_arg++;
		if (argv[last_arg][0]!='-')
		{
			UsageAbort(2,"Too many arguments");
		}
		else
		{
			UsageAbort(2,"Syntax error in arguments");
		}
	}
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
