#ifndef __REQUESTS_H__
#define __REQUESTS_H__

/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Option and argument handling
Part of CleanWAD

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include "services.h"
#include "waddef.h"

/**********************************************************************************************************************************/
/****************************************************** Command-Line Options ******************************************************/
/**********************************************************************************************************************************/

// Boolean options
typedef enum
{
	OPTION_REMOVE_DUPLICATES,									// remove all but the first of entries having the same name
	OPTION_TRUNCATE_WAVES,										// remove wasted space at the end of sound resources
	OPTION_REBUILD_BLOCKMAPS,									// rebuild (no pack/unpack) blockmaps to remove wasted space
	OPTION_PACK_BLOCKMAPS,										// pack blockmaps to remove current and potential wasted space
	OPTION_UNPACK_BLOCKMAPS,									// unpack blockmaps into normalised (but inefficient) format
	OPTION_FLUSH_BLOCKMAPS,										// write out modified blockmaps even if the size has not changed
	OPTION_REBUILD_PICTURES,									// rebuild (no pack/unpack) pictures to remove wasted space
	OPTION_PACK_PICTURES,										// pack pictures to remove current and potential wasted space
	OPTION_UNPACK_PICTURES,										// unpack pictures into normalised (but inefficient) format
	OPTION_FLUSH_PICTURES,										// write out modified pictures even if the size has not changed
	OPTION_REUSE_LUMPS,											// reuse lumps where directory entries specify same size and offset
	OPTION_PACK_LUMPS,											// share lumps with identical contents between directory entries
	OPTION_UNPACK_LUMPS,										// output each lump explicitly even if it could be shared or reused
	OPTION_ALIGN_RESOURCES,										// align resources on 32-bit boundaries for speed at cost of space
	OPTION_ALIGN_DIRECTORY,										// align directory on 32-bit boundaries for speed at cost of space
	OPTION_IDENTIFY_NAMES,										// identify screen images by name (READ MANUAL BEFORE USING!)
	OPTION_IDENTIFY_PAGES,										// identify screen images as pages (READ MANUAL BEFORE USING!)
	OPTION_IDENTIFY_GRAPHICS,									// identify screen images as graphics (READ MANUAL BEFORE USING!)
	OPTION_IDENTIFY_VOICES,										// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
	OPTION_DETECT_SOUNDS,										// detect sounds by content (READ MANUAL BEFORE USING!)
	OPTION_DETECT_MUSICS,										// detect musics by content (READ MANUAL BEFORE USING!)
	OPTION_DETECT_GRAPHICS,										// detect graphics by content (READ MANUAL BEFORE USING!)
	OPTION_RECOGNISED_NAMES,									// identify sounds/musics/dialogs/conversations by recognised names
	OPTION_LOOSE_MARKERS,										// allow nonstandard list marker characters
	OPTION_NAMED_MARKERS,										// loose markers use names only (READ MANUAL BEFORE USING!)
	OPTION_MAINTAIN_BLOCKMAPS,									// maintain nonstandard blockmaps (preserve non-optimal blocks)
	OPTION_MAINTAIN_PICTURES,									// maintain nonstandard pictures (preserve non-optimal columns)
	OPTION_KEEP_WINTEX,											// do not remove _DEUTEX_ lump
	OPTION_KEEP_PLATFORM,										// do not remove PLATFORM lump
	OPTION_KEEP_HISTORY,										// do not remove HISTORY lump
	OPTION_KEEP_TAGDESC,										// do not remove TAGDESC lump
	OPTION_KEEP_PCSFX,											// do not remove PC speaker sound effects
	OPTION_KEEP_DOUBLES,										// do not normalise double-letter list markers (e.g., PP_START)
	OPTION_KEEP_BORDERS,										// do not remove sub-list border markers (e.g., F1_START)
	OPTION_KEEP_EMPTIES,										// do not remove empty structured lists
	OPTION_TOLERATE_MULTIPLES,									// do not treat multiple instances of structured lists as an error
	OPTION_QUIET_MULTIPLES,										// do not treat multiple instances of structured lists as a problem
	OPTION_TOLERATE_LINKS,										// do not treat lump reuse as an error
	OPTION_QUIET_LINKS,											// do not treat lump reuse as a problem
	OPTION_TOLERATE_CONFLICTS,									// do not treat lump ineligible lump reuse as an error
	OPTION_QUIET_CONFLICTS,										// do not treat lump ineligible lump reuse as a problem
	OPTION_UNPACK_CONFLICTS,									// do not preserve ineligible lump reuse (if tolerated)
	OPTION_REMOVE_SCRIPTS,										// remove all SCRIPTS and SCRIPTnn entries
	OPTION_DECLASSIFY_PNAMES,									// treat PNAMES as an unclassified lump (separate from TEXTURES)
	OPTION_LOOSE_HEADERS,										// allow nonstandard map name headers (not just E\?M\? and MAP\?\?)
	OPTION_QUIET_HEADERS,										// do not warn about non-empty map name headers
	OPTION_FORCE_REMOVAL,										// remove duplicate entries even if in different lists
	OPTION_SORT,												// sort order to apply to entries in the output file
	OPTION_GAME,												// game for which (WAD whose lump names these are) was designed
	OPTION_VERBOSITY,											// verbosity level for reporting of events and errors
	OPTION_MAXPLUSONE											// highest type enumeration value plus one
} option_enum_t;

#define OPTION_MAXCODELENGTH 2
#define OPTION_MAXNAMELENGTH 22
#define OPTION_MAXDESCLENGTH 64

// Option mode type
typedef enum
{
	OPTION_IS_DEFAULT,											// these options came from the default options
	OPTION_IS_ENVIRONMENT,										// these options came from the environment
	OPTION_IS_COMMANDLINE,										// these options came from the command line
	OPTION_IS_RESULTANT											// these options came from merging the other three
} option_mode_t;

// Option details type
typedef enum
{
	OPTION_IS_BOOLEAN,											// this option is a boolean
	OPTION_IS_SORT,												// this option is a sort order
	OPTION_IS_GAME,												// this option is a game selection
	OPTION_IS_VERBOSITY											// this option is a verbosity level
} option_type_t;

// Option details entry
typedef union
{
	bool_t flag;												// flag that states if set
	sort_t sort;												// sort order
	game_t game;												// game selection
	verbosity_t verbosity;										// verbosity level
} option_value_t;

// Option details
typedef struct
{
	option_type_t type;											// what type of option it is
	option_value_t value;										// flag, game or verbosity
	bool_t given;												// if value was given explicitly (not simply defaulted to)
	char code[OPTION_MAXCODELENGTH+1];							// code (command line switch characters)
	char name[OPTION_MAXNAMELENGTH+1];							// name (brief summary)
	char desc[OPTION_MAXDESCLENGTH+1];							// description (meaning)
}
option_t;

typedef option_t options_t[OPTION_MAXPLUSONE];					// list of options

// Function request type
typedef enum
{
	WADREQ_UNKNOWN,												// unknown
	WADREQ_CLEAN,												// clean a WAD file
	WADREQ_MAXPLUSONE											// highest type enumeration value plus one
} WADREQ_type_t;

// Function request
typedef struct
{
	WADREQ_type_t request_type;									// type of request
	char *infilename;											// input file name
	char *outfilename;											// output file name
	options_t *options;											// list of options
} WADREQ_t;

/**********************************************************************************************************************************/
/******************************************** WAD File Processing Request Maintanance *********************************************/
/**********************************************************************************************************************************/

// Initialise a WAD file processing request
void WADInitRequest (
	WADREQ_t **WADREQp);										// WAD file processing request to be initialised

// Deinitialise a WAD file processing request
void WADDoneRequest (
	WADREQ_t **WADREQp);										// WAD file processing request to be deinitialised

/**********************************************************************************************************************************/
/***************************************************** Options and Arguments ******************************************************/
/**********************************************************************************************************************************/

// Return default options argc
int WADGetDefaultArgC (
	void);														// no arguments

// Return default options argv
char **WADGetDefaultArgV (
	void);														// no arguments

// WADValidate options
void WADValidateOptions (
	options_t *options,											// options to be validated
	option_mode_t option_mode);									// ultimate source of these options

// Merge options
void WADMergeOptions (
	options_t *old_options,										// options to be merged into
	options_t *new_options);									// options to merge into them

// Parse options
void WADParseOptions (
	options_t *options,											// options parsed from arguments
	option_mode_t option_mode,									// ultimate source of these options
	int argc,													// program argument count
	char *argv[],												// program argument strings
	int *last_argp);											// index of last argument parsed

// Parse command line arguments
void WADParseCommandLineArguments (
	int argc,													// program argument count
	char *argv[],												// program argument strings
	int last_argp,												// index of last option argument parsed
	WADREQ_t *WADREQ);											// WAD file processing request

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/

#endif // __REQUESTS_H__
