/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Base services

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <stdlib.h>
#include <malloc.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include "services.h"

/**********************************************************************************************************************************/
/********************************************************* Error Handling *********************************************************/
/**********************************************************************************************************************************/

#if 0
static UINT32 event_count=0;									// current number of events
static event_item_t event_items[MAX_EVENT_LIST];				// array of events
#endif

/**********************************************************************************************************************************/
/**************************************************** Command-Line Pre-Parsing ****************************************************/
/**********************************************************************************************************************************/

// Parse argv[] and argc style parameters from a string
void ParseArgvAndArgcFromString (
	char *str,													// command line to create argv and argc from
	char **out_copy,											// modified copy of string (needed for subsequent parsing)
	int *out_argc,												// number of arguments (0 is a dummy)
	char **out_argv[])											// pointer to array of string pointers
{
	// DOCUMENTATION
	/*
		The input string is not modified by this routine, but a
		copy of it is created so that the copy can be: NULL
		characters are placed wherever an argument ends, except
		the last one in the string (which has one anyway).

		There is no program name argument for command line
		obtained from an arbitrary string, so that pointer is
		set to point to a NULL character, specifically the
		terminating character of the string copy.

		It is the responsibility of the caller to free the
		string copy and the constructed string pointer array
		when they are no longer needed. This restriction and
		the others are not unreasonable; the input string copy
		and the resultant variables usually only need to exist
		long enough to parse the extracted arguments into some
		more formal structure, after which they can be freed.
	*/

	// VARIABLES
	char *this_copy;
	int this_argc;
	char **this_argv;
	size_t in_length;
	char **temp_argv;
	char *this_char;

	// CREATE EDITABLE COPY OF THE INPUT STRING
	in_length=strlen(str);
	this_copy=malloc(in_length+1);
	if (this_copy==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for input string copy",in_length+1);
	}
	(void) strcpy(this_copy,str);

	// INITIALISE THE SCAN POINTER
	this_char=this_copy;

	// INITIALISE THE OUTPUT ARGUMENT METRICS
	this_argc=1;
	temp_argv=malloc(this_argc*sizeof(char*));
	if (temp_argv==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for environment argv buffer",this_argc*sizeof(char*));
	}
	this_argv=temp_argv;
	this_argv[this_argc-1]=&this_copy[in_length];

	// SKIP ANY INITIAL WHITESPACE
	while ( (*this_char==SPACE) || (*this_char==TAB) )
	{
		this_char++;
	}

	// SCAN THE STRING
	while (*this_char!=NUL)
	{
		this_argc++;
		temp_argv=realloc(this_argv,(this_argc*sizeof(char*)));
		if (temp_argv==NULL)
		{
			FatalAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for environment argv buffer",(this_argc*sizeof(char*)));
		}
		this_argv=temp_argv;
		this_argv[this_argc-1]=this_char;
		while ( (*this_char!=SPACE) && (*this_char!=TAB) && (*this_char!=NUL) )
		{
			this_char++;
		}
		if (*this_char!=NUL)
		{
			*this_char++=NUL;
		}
		while ( (*this_char==SPACE) || (*this_char==TAB) )
		{
			this_char++;
		}
	}

	// TERMINATE THE LIST THE SAME WAY AS THE REAL ARGV
	this_argc++;
	temp_argv=realloc(this_argv,this_argc*sizeof(char*));
	if (temp_argv==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for environment argv buffer",this_argc*sizeof(char*));
	}
	this_argv=temp_argv;
	this_argv[this_argc-1]=&this_copy[in_length];

	// RETURN RESULTS
	*out_copy=this_copy;
	*out_argc=this_argc-1;
	*out_argv=this_argv;
}

/**********************************************************************************************************************************/
/****************************************************** Endian-Ness Handling ******************************************************/
/**********************************************************************************************************************************/

// Read a little-endian SINT16 by reference
void RDLend16SR (SINT16 *out, const SINT16 in)
{
#if __ISBIGENDIAN__
	*out=((SINT16)
				    ((in >>  8) & 0x00FF) |
				    ((in <<  8) & 0xFF00)
				  );
#else
	*out=in;
#endif
}

// Read a little-endian SINT32 by reference
void RDLend32SR (SINT32 *out, const SINT32 in)
{
#if __ISBIGENDIAN__
	*out=((SINT32)
				    ((in >> 24) & 0x000000FF) |
				    ((in >>  8) & 0x0000FF00) |
				    ((in <<  8) & 0x00FF0000) |
				    ((in << 24) & 0xFF000000)
				  );
#else
	*out=in;
#endif
}

// Read a little-endian SINT64 by reference
void RDLend64SR (SINT64 *out, const SINT64 in)
{
#if __ISBIGENDIAN__
	*out=((SINT64)
				    ((in >> 56) & 0x00000000000000FF) |
				    ((in >> 40) & 0x000000000000FF00) |
				    ((in << 24) & 0x0000000000FF0000) |
				    ((in <<  8) & 0x00000000FF000000) |
				    ((in >>  8) & 0x000000FF00000000) |
				    ((in >> 24) & 0x0000FF0000000000) |
				    ((in << 40) & 0x00FF000000000000) |
				    ((in << 56) & 0xFF00000000000000)
				  );
#else
	*out=in;
#endif
}

// Read a little-endian UINT16 by reference
void RDLend16UR (UINT16 *out, const UINT16 in)
{
#if __ISBIGENDIAN__
	*out=((UINT16)
				    ((in >>  8) & 0x00FF) |
				    ((in <<  8) & 0xFF00)
				  );
#else
	*out=in;
#endif
}

// Read a little-endian UINT32 by reference
void RDLend32UR (UINT32 *out, const UINT32 in)
{
#if __ISBIGENDIAN__
	*out=((UINT32)
				    ((in >> 24) & 0x000000FF) |
				    ((in >>  8) & 0x0000FF00) |
				    ((in <<  8) & 0x00FF0000) |
				    ((in << 24) & 0xFF000000)
				  );
#else
	*out=in;
#endif
}

// Read a little-endian UINT64 by reference
void RDLend64UR (UINT64 *out, const UINT64 in)
{
#if __ISBIGENDIAN__
	*out=((UINT64)
				    ((in >> 56) & 0x00000000000000FF) |
				    ((in >> 40) & 0x000000000000FF00) |
				    ((in << 24) & 0x0000000000FF0000) |
				    ((in <<  8) & 0x00000000FF000000) |
				    ((in >>  8) & 0x000000FF00000000) |
				    ((in >> 24) & 0x0000FF0000000000) |
				    ((in << 40) & 0x00FF000000000000) |
				    ((in << 56) & 0xFF00000000000000)
				  );
#else
	*out=in;
#endif
}

// Read a little-endian SINT16 by value
SINT16 RDLend16SV (const SINT16 in)
{
#if __ISBIGENDIAN__
	return ((SINT16)
				    ((in >>  8) & 0x00FF) |
				    ((in <<  8) & 0xFF00)
				  );
#else
	return in;
#endif
}

// Read a little-endian SINT32 by value
SINT32 RDLend32SV (const SINT32 in)
{
#if __ISBIGENDIAN__
	return ((SINT32)
				    ((in >> 24) & 0x000000FF) |
				    ((in >>  8) & 0x0000FF00) |
				    ((in <<  8) & 0x00FF0000) |
				    ((in << 24) & 0xFF000000)
				  );
#else
	return in;
#endif
}

// Read a little-endian SINT64 by value
SINT64 RDLend64SV (const SINT64 in)
{
#if __ISBIGENDIAN__
	return ((SINT64)
				    ((in >> 56) & 0x00000000000000FF) |
				    ((in >> 40) & 0x000000000000FF00) |
				    ((in << 24) & 0x0000000000FF0000) |
				    ((in <<  8) & 0x00000000FF000000) |
				    ((in >>  8) & 0x000000FF00000000) |
				    ((in >> 24) & 0x0000FF0000000000) |
				    ((in << 40) & 0x00FF000000000000) |
				    ((in << 56) & 0xFF00000000000000)
				  );
#else
	return in;
#endif
}

// Read a little-endian UINT16 by value
UINT16 RDLend16UV (const UINT16 in)
{
#if __ISBIGENDIAN__
	return ((UINT16)
				    ((in >>  8) & 0x00FF) |
				    ((in <<  8) & 0xFF00)
				  );
#else
	return in;
#endif
}

// Read a little-endian UINT32 by value
UINT32 RDLend32UV (const UINT32 in)
{
#if __ISBIGENDIAN__
	return ((UINT32)
				    ((in >> 24) & 0x000000FF) |
				    ((in >>  8) & 0x0000FF00) |
				    ((in <<  8) & 0x00FF0000) |
				    ((in << 24) & 0xFF000000)
				  );
#else
	return in;
#endif
}

// Read a little-endian UINT64 by value
UINT64 RDLend64UV (const UINT64 in)
{
#if __ISBIGENDIAN__
	return ((UINT64)
				    ((in >> 56) & 0x00000000000000FF) |
				    ((in >> 40) & 0x000000000000FF00) |
				    ((in << 24) & 0x0000000000FF0000) |
				    ((in <<  8) & 0x00000000FF000000) |
				    ((in >>  8) & 0x000000FF00000000) |
				    ((in >> 24) & 0x0000FF0000000000) |
				    ((in << 40) & 0x00FF000000000000) |
				    ((in << 56) & 0xFF00000000000000)
				  );
#else
	return in;
#endif
}

// Read a big-endian SINT16 by reference
void RDBend16SR (SINT16 *out, const SINT16 in)
{
#if __ISLITTLEENDIAN__
	*out=((SINT16)
				    ((in >>  8) & 0x00FF) |
				    ((in <<  8) & 0xFF00)
				  );
#else
	*out=in;
#endif
}

// Read a big-endian SINT32 by reference
void RDBend32SR (SINT32 *out, const SINT32 in)
{
#if __ISLITTLEENDIAN__
	*out=((SINT32)
				    ((in >> 24) & 0x000000FF) |
				    ((in >>  8) & 0x0000FF00) |
				    ((in <<  8) & 0x00FF0000) |
				    ((in << 24) & 0xFF000000)
				  );
#else
	*out=in;
#endif
}

// Read a big-endian SINT64 by reference
void RDBend64SR (SINT64 *out, const SINT64 in)
{
#if __ISLITTLEENDIAN__
	*out=((SINT64)
				    ((in >> 56) & 0x00000000000000FF) |
				    ((in >> 40) & 0x000000000000FF00) |
				    ((in << 24) & 0x0000000000FF0000) |
				    ((in <<  8) & 0x00000000FF000000) |
				    ((in >>  8) & 0x000000FF00000000) |
				    ((in >> 24) & 0x0000FF0000000000) |
				    ((in << 40) & 0x00FF000000000000) |
				    ((in << 56) & 0xFF00000000000000)
				  );
#else
	*out=in;
#endif
}

// Read a big-endian UINT16 by reference
void RDBend16UR (UINT16 *out, const UINT16 in)
{
#if __ISLITTLEENDIAN__
	*out=((UINT16)
				    ((in >>  8) & 0x00FF) |
				    ((in <<  8) & 0xFF00)
				  );
#else
	*out=in;
#endif
}

// Read a big-endian UINT32 by reference
void RDBend32UR (UINT32 *out, const UINT32 in)
{
#if __ISLITTLEENDIAN__
	*out=((UINT32)
				    ((in >> 24) & 0x000000FF) |
				    ((in >>  8) & 0x0000FF00) |
				    ((in <<  8) & 0x00FF0000) |
				    ((in << 24) & 0xFF000000)
				  );
#else
	*out=in;
#endif
}

// Read a big-endian UINT64 by reference
void RDBend64UR (UINT64 *out, const UINT64 in)
{
#if __ISLITTLEENDIAN__
	*out=((UINT64)
				    ((in >> 56) & 0x00000000000000FF) |
				    ((in >> 40) & 0x000000000000FF00) |
				    ((in << 24) & 0x0000000000FF0000) |
				    ((in <<  8) & 0x00000000FF000000) |
				    ((in >>  8) & 0x000000FF00000000) |
				    ((in >> 24) & 0x0000FF0000000000) |
				    ((in << 40) & 0x00FF000000000000) |
				    ((in << 56) & 0xFF00000000000000)
				  );
#else
	*out=in;
#endif
}

// Read a big-endian SINT16 by value
SINT16 RDBend16SV (const SINT16 in)
{
#if __ISLITTLEENDIAN__
	return ((SINT16)
				    ((in >>  8) & 0x00FF) |
				    ((in <<  8) & 0xFF00)
				  );
#else
	return in;
#endif
}

// Read a big-endian SINT32 by value
SINT32 RDBend32SV (const SINT32 in)
{
#if __ISLITTLEENDIAN__
	return ((SINT32)
				    ((in >> 24) & 0x000000FF) |
				    ((in >>  8) & 0x0000FF00) |
				    ((in <<  8) & 0x00FF0000) |
				    ((in << 24) & 0xFF000000)
				  );
#else
	return in;
#endif
}

// Read a big-endian SINT64 by value
SINT64 RDBend64SV (const SINT64 in)
{
#if __ISLITTLEENDIAN__
	return ((SINT64)
				    ((in >> 56) & 0x00000000000000FF) |
				    ((in >> 40) & 0x000000000000FF00) |
				    ((in << 24) & 0x0000000000FF0000) |
				    ((in <<  8) & 0x00000000FF000000) |
				    ((in >>  8) & 0x000000FF00000000) |
				    ((in >> 24) & 0x0000FF0000000000) |
				    ((in << 40) & 0x00FF000000000000) |
				    ((in << 56) & 0xFF00000000000000)
				  );
#else
	return in;
#endif
}

// Read a big-endian UINT16 by value
UINT16 RDBend16UV (const UINT16 in)
{
#if __ISLITTLEENDIAN__
	return ((UINT16)
				    ((in >>  8) & 0x00FF) |
				    ((in <<  8) & 0xFF00)
				  );
#else
	return in;
#endif
}

// Read a big-endian UINT32 by value
UINT32 RDBend32UV (const UINT32 in)
{
#if __ISLITTLEENDIAN__
	return ((UINT32)
				    ((in >> 24) & 0x000000FF) |
				    ((in >>  8) & 0x0000FF00) |
				    ((in <<  8) & 0x00FF0000) |
				    ((in << 24) & 0xFF000000)
				  );
#else
	return in;
#endif
}

// Read a big-endian UINT64 by value
UINT64 RDBend64UV (const UINT64 in)
{
#if __ISLITTLEENDIAN__
	return ((UINT64)
				    ((in >> 56) & 0x00000000000000FF) |
				    ((in >> 40) & 0x000000000000FF00) |
				    ((in << 24) & 0x0000000000FF0000) |
				    ((in <<  8) & 0x00000000FF000000) |
				    ((in >>  8) & 0x000000FF00000000) |
				    ((in >> 24) & 0x0000FF0000000000) |
				    ((in << 40) & 0x00FF000000000000) |
				    ((in << 56) & 0xFF00000000000000)
				  );
#else
	return in;
#endif
}

/**********************************************************************************************************************************/
/******************************************************** Case Conversion *********************************************************/
/**********************************************************************************************************************************/

// Convert a string to lower case
void lowercase (char *s)
{
	// VARIABLES
	char *p;

	// PRIMARY ERROR CHECK
	if (s==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to lowercase string with NULL argument");
	}

	// CONVERT
	p=s;
	while (*p!=NUL)
	{
		*p=TOLOWER(*p);
		p++;
	}
}

// Convert a string to upper case
void uppercase (char *s)
{
	// VARIABLES
	char *p;

	// PRIMARY ERROR CHECK
	if (s==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to uppercase string with NULL argument");
	}

	// CONVERT
	p=s;
	while (*p!=NUL)
	{
		*p=TOUPPER(*p);
		p++;
	}
}

/**********************************************************************************************************************************/
/***************************************************** String Equality Tests ******************************************************/
/**********************************************************************************************************************************/

// Case Insensitive String Equality Test
bool_t cimatch (const char *s1, const char *s2)
{
	// DOCUMENTATION
	/*
		This function is designed to operate on arbitrary
		length C strings and only checks for equality (or not).
	*/

	// VARIABLES
	const char *p1;
	const char *p2;
	bool_t match;

	// PRIMARY ERROR CHECK
	if (s1==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to compare strings (case insensitive) for equality with NULL first argument");
	}
	if (s2==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to compare strings (case insensitive) for equality with NULL second argument");
	}

	// COMPARE
	p1=s1;
	p2=s2;
	match=TRUE;
	while (TRUE)
	{
		char v1,v2;
		/*********/
		v1=TOUPPER(*p1);
		v2=TOUPPER(*p2);
		if ( (v1==NUL) && (v2==NUL) )
		{
			match=TRUE;
			break;
		}
		if (v1!=v2)
		{
			match=FALSE;
			break;
		}
#ifdef PARANOID
		if ( (v1==NUL) || (v2==NUL) )
		{
			// Should never get here with
			// null-terminated strings.
			FatalAbort(1,__FILE__,__LINE__,"Attempt to compare strings (case insensitive) for equality failed\n\"%s\"\n\"%s\"",s1,s2);
		}
#endif
		p1++;
		p2++;
	}

	// RETURN RESULT
	return match;
}

// Case Sensitive String Equality Test
bool_t csmatch (const char *s1, const char *s2)
{
	// DOCUMENTATION
	/*
		This function is designed to operate on arbitrary
		length C strings and only checks for equality (or not).
	*/

	// VARIABLES
	const char *p1;
	const char *p2;
	bool_t match;

	// PRIMARY ERROR CHECK
	if (s1==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to compare strings (case insensitive) for equality with NULL first argument");
	}
	if (s2==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to compare strings (case insensitive) for equality with NULL second argument");
	}

	// COMPARE
	p1=s1;
	p2=s2;
	match=TRUE;
	while (TRUE)
	{
		char v1,v2;
		/*********/
		v1=*p1;
		v2=*p2;
		if ( (v1==NUL) && (v2==NUL) )
		{
			match=TRUE;
			break;
		}
		if (v1!=v2)
		{
			match=FALSE;
			break;
		}
#ifdef PARANOID
		if ( (v1==NUL) || (v2==NUL) )
		{
			// Should never get here with
			// null-terminated strings.
			FatalAbort(1,__FILE__,__LINE__,"Attempt to compare strings (case sensitive) for equality failed\n\"%s\"\n\"%s\"",s1,s2);
		}
#endif
		p1++;
		p2++;
	}

	// RETURN RESULT
	return match;
}

/**********************************************************************************************************************************/
/*************************************************** Memory Array Equality Test ***************************************************/
/**********************************************************************************************************************************/

// Memory Array Equality Test
bool_t mamatch (const void *x, size_t xsize, const void *y, size_t ysize)
{
	// DOCUMENTATION
	/*
		This function is designed to operate on arbitrary
		length memory arrays and only checks for equality (or
		not). It returns FALSE at the first found non-match.
	*/

	// VARIABLES
	const char *p;
	const char *q;
	size_t count;

	// PRIMARY ERROR CHECK
	if (x==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to compare memory arrays for equality with NULL first argument");
	}
	if (y==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to compare memory arrays for equality with NULL second argument");
	}

	// UNEQUAL IF DIFFERENT SIZES
	if (xsize!=ysize)
	{
		return FALSE;
	}

	// COMPARE
	for (p=x,q=y,count=0;count<xsize;p++,q++,count++)
	{
		if (*p!=*q)
		{
			return FALSE;
		}
	}

	// MATCH
	return TRUE;
}

/**********************************************************************************************************************************/
/********************************************************* File Handling **********************************************************/
/**********************************************************************************************************************************/

// Read a string from a file
void FILEStringReadFromFile (
	FILE *infile,
	char **str)
{
	// DOCUMENTATION
	/*
		Read a string from a file and return a pointer to
		allocated storage containing that string. It is the
		caller's responsibility to free the string. If the file
		is at eof then return NULL and if the string is empty,
		return a 1 byte buffer containing the character NUL;
	*/

	// METRICS
	#define STRING_APPORTION 32
	#define STRING_INCREMENT 32

	// VARIABLES
	int c;
	char *buffer;
	size_t count,limit,sizeb;

	// ALLOCATE INITIAL MEMORY
	count=0;
	sizeb=STRING_APPORTION+1;
	buffer=malloc(sizeb);
	if (buffer==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for string buffer",sizeb);
	}
	limit=STRING_APPORTION;

	// READ THE STRING
	while (TRUE)
	{
		c=fgetc(infile);
	    if (ferror(infile)!=0)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
		}
		if (feof(infile)!=0)
		{
			if (count==0)
			{
				free(buffer);
				*str=NULL;
				return;
			}
			ErrorAbort(1,__FILE__,__LINE__,"Unexpected end of file");
		}
		if (c==(int)CR)
		{
			continue;
		}
		if (c==(int)LF)
		{
			buffer[count]=NUL;
			*str=buffer;
			return;
		}
		if (count==limit)
		{
			char *newptr;
			/***********/
			sizeb=limit+STRING_INCREMENT+1;
			newptr=realloc(buffer,sizeb);
			if (newptr==NULL)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not reallocate %ld bytes for string buffer",sizeb);
			}
			buffer=newptr;
			limit+=STRING_INCREMENT;
		}
		buffer[count++]=(char)((unsigned int)(c) & 255);
	}

	// TERMINATE THE STRING
	buffer[count]=NUL;
	*str=buffer;
}

// Copy data from a file at current position into memory
void FILESegmentReadIntoMemory (
	FILE *infile,												// file to read from
	void *buffer,												// segment
	size_t count)												// size of segment
{
	// FILE BUFFERING METRICS
	// #define MAX_FREAD_SIZE ((size_t)32768)

	// VARIABLES
#if defined(MAX_FREAD_SIZE)
	size_t i,sizeb,quotient,remainder,completed;
#endif

	// PRIMARY ERROR CHECK
	if (infile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to read from file with NULL argument");
	}
	if (buffer==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to write into memory with NULL argument");
	}

	// HANDLE TRIVIAL CASE
	if (count==0)
	{
		return;
	}

#if !defined(MAX_FREAD_SIZE)
	if (fread(buffer,count,1,infile)<1)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
	}
#else
	// INITIALIZE
	sizeb=(count>MAX_FREAD_SIZE)?MAX_FREAD_SIZE:count;
	quotient=sizeb/MAX_FREAD_SIZE;
	remainder=sizeb%MAX_FREAD_SIZE;
	completed=0;

	// READ WHOLE MULTIPLES OF MAX_FREAD_SIZE
	for (i=0;i<quotient;i++)
	{
		if (fread(&((unsigned char*)buffer)[completed],sizeb,1,infile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
		}
		completed+=sizeb;
	};

	// READ ANY REMAINING PART OF MAX_FREAD_SIZE
	if (remainder!=0)
	{
		if (fread(&((unsigned char*)buffer)[completed],remainder,1,infile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
		}
//		completed+=remainder;
	}
#endif
}

// Copy data from memory into a file at current position
void FILESegmentWriteFromMemory (
	FILE *outfile,												// file to write to
	void *buffer,												// segment
	size_t count)												// size of segment
{
	// FILE BUFFERING METRICS
	// #define MAX_FWRITE_SIZE ((size_t)32768)

	// VARIABLES
#if defined(MAX_FWRITE_SIZE)
	size_t i,sizeb,quotient,remainder,completed;
#endif

	// PRIMARY ERROR CHECK
	if (buffer==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to read from memory with NULL argument");
	}
	if (outfile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to write to file with NULL argument");
	}

	// HANDLE TRIVIAL CASE
	if (count==0)
	{
		return;
	}

#if !defined(MAX_FWRITE_SIZE)
	if (fwrite(buffer,count,1,outfile)<1)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not write to file");
	}
#else
	// INITIALIZE
	sizeb=(count>MAX_FWRITE_SIZE)?MAX_FWRITE_SIZE:count;
	quotient=sizeb/MAX_FWRITE_SIZE;
	remainder=sizeb%MAX_FWRITE_SIZE;
	completed=0;

	// WRITE WHOLE MULTIPLES OF MAX_FWRITE_SIZE
	for (i=0;i<quotient;i++)
	{
		if (fwrite(&((unsigned char*)buffer)[completed],sizeb,1,outfile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not write to file");
		}
		completed+=sizeb;
	};

	// WRITE ANY REMAINING PART OF MAX_FWRITE_SIZE
	if (remainder!=0)
	{
		if (fwrite(&((unsigned char*)buffer)[completed],remainder,1,outfile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not write to file");
		}
//		completed+=remainder;
	}
#endif
}

// Copy segment of one file into another at current positions
void FILESegmentCopyBetweenFiles (
	FILE *infile,												// file to read from
	FILE *outfile,												// file to write to
	size_t count)												// size of segment
{
	// FILE COPY BUFFERING METRICS
	#define MAX_BUFFER_SIZE ((size_t)32768)

	// VARIABLES
	void *buffer;
	size_t i,sizeb,quotient,remainder;

	// PRIMARY ERROR CHECK
	if (infile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to read from file with NULL argument");
	}
	if (outfile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to write to file with NULL argument");
	}

	// HANDLE TRIVIAL CASE
	if (count==0)
	{
		return;
	}

	// INITIALIZE
	sizeb=(count>MAX_BUFFER_SIZE)?MAX_BUFFER_SIZE:count;
	quotient=sizeb/MAX_BUFFER_SIZE;
	remainder=sizeb%MAX_BUFFER_SIZE;

	// ALLOCATE THE REQUIRED MEMORY
	buffer=malloc(sizeb);
	if (buffer==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for file segment copy buffer",sizeb);
	}

	// COPY WHOLE MULTIPLES OF MAX_BUFFER_SIZE
	for (i=0;i<quotient;i++)
	{
		if (fread(buffer,sizeb,1,infile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
		}
		if (fwrite(buffer,sizeb,1,outfile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not write to file");
		}
	};

	// COPY ANY REMAINING PART OF MAX_BUFFER_SIZE
	if (remainder!=0)
	{
		if (fread(buffer,remainder,1,infile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
		}
		if (fwrite(buffer,remainder,1,outfile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not write to file");
		}
	}

	// DEALLOCATE THE MEMORY
	free(buffer);
}

/**********************************************************************************************************************************/
/**************************************************** Non-Error Event Logging *****************************************************/
/**********************************************************************************************************************************/

// Flag for if this is the first message printed
static bool_t first_message=TRUE;
static verbosity_t last_priority=VERBOSITY_STATEMENTS;

// Verbosity level during program execution
static verbosity_t verbosity_level=VERBOSITY_PROGRESS;			// Suitable for most situations (programs can change it if desired)

// Set verbosity level
void EventSetVerbosity(verbosity_t verbosity)
{
	verbosity_level=verbosity;
};

// Get overall verbosity level
void EventGetVerbosity(verbosity_t *verbosity)
{
	*verbosity=verbosity_level;
};

// Generic event logger
void EventState(verbosity_t priority, char *message,...)
{
	if (priority>verbosity_level)
	{
		return;
	}
	if (first_message)
	{
		(void) fprintf (stdout,"\n");
		first_message=FALSE;
	}
	else if ( (priority==VERBOSITY_ERRORS) && (last_priority!=VERBOSITY_ERRORS) )
	{
		(void) fprintf (stdout,"\7\n");
	}
	else if ( (last_priority==VERBOSITY_STATEMENTS) && (priority!=VERBOSITY_STATEMENTS) )
	{
		(void) fprintf (stdout,"\n");
	}
	else
	{
		// not a special case
		// no action required
	}
	if (priority==VERBOSITY_ERRORS)
	{
		(void) fprintf (stdout, "ERROR: ");
	}
	if (priority==VERBOSITY_WARNINGS)
	{
		(void) fprintf (stdout, "WARNING: ");
	}
	if (priority==VERBOSITY_CORRECTIONS)
	{
		(void) fprintf (stdout, "CORRECTION: ");
	}
	if ( (priority==VERBOSITY_PROGRESS) && (verbosity_level==VERBOSITY_DETAILS) )
	{
		(void) fprintf (stdout, "******** ");
	}
	if ( (message==NULL) && (strlen(message)==0) )
	{
		(void) fprintf (stdout,"[no message specified]");
	}
	else
	{
		va_list ap;
		va_start(ap, message);
		(void) vfprintf (stdout,message,ap);
		va_end(ap);
	}
	(void) fprintf (stdout,"\n");
	last_priority=priority;
}

/**********************************************************************************************************************************/
/********************************************************* Error Handling *********************************************************/
/**********************************************************************************************************************************/

// Generic error logger
void ErrorAbort(
	unsigned int status,										// status code for this error
	char *filenm,												// source file name (from __FILE__)
	long int lineno,											// source line number (from __LINE__)
	char *message,												// message format
	...)														// message arguments
{
	// Variables
	char text[MAX_EVENT_TEXT+1]="";
	int check;
	size_t used=0;
	size_t left=MAX_EVENT_TEXT;

	// Initialisation and (if applicable) leading newline
	if (message==NULL)
	{
		// Messages already done (abort only)
		last_priority=VERBOSITY_ERRORS;
		exit ((int)(status));
	}
	if (first_message)
	{
		(void) sprintf(text,"\n");
		used=strlen(text);
		left=MAX_EVENT_TEXT-used;
		first_message=FALSE;
	}
	else if (last_priority!=VERBOSITY_ERRORS)
	{
		(void) sprintf(text,"\n");
		used=strlen(text);
		left=MAX_EVENT_TEXT-used;
	}
	else if (last_priority==VERBOSITY_STATEMENTS)
	{
		(void) sprintf(text,"\n");
		used=strlen(text);
		left=MAX_EVENT_TEXT-used;
	}
	else
	{
		// not a special case
		// no action required
	}

	// Heading
	if (left>0)
	{
#ifdef STATEALL
		check=snprintf(&text[used],left,"\7ERROR: [%s] [%ld] ",filenm,lineno);
#else
		check=snprintf(&text[used],left,"\7ERROR: [%ld] ",lineno);
#endif
		if ( (check<0) || (check==MAX_EVENT_TEXT) )
		{
			used=MAX_EVENT_TEXT;
			left=0;
			text[MAX_EVENT_TEXT]=NUL;
		}
		else
		{
			used=strlen(text);
			left=MAX_EVENT_TEXT-used;
		}
	}

	// Message
	if ( (message==NULL) || (strlen(message)==0) )
	{
		if (left>0)
		{
			check=snprintf(&text[used],left,"[no message specified]");
			if ( (check<0) || (check==MAX_EVENT_TEXT) )
			{
				used=MAX_EVENT_TEXT;
				left=0;
				text[MAX_EVENT_TEXT]=NUL;
			}
			else
			{
				used=strlen(text);
				left=MAX_EVENT_TEXT-used;
			}
		}
	}
	else
	{
		va_list ap;
		va_start(ap,message);
		if (left>0)
		{
			check=vsnprintf(&text[used],left,message,ap);
			if ( (check<0) || (check==MAX_EVENT_TEXT) )
			{
				used=MAX_EVENT_TEXT;
				left=0;
				text[MAX_EVENT_TEXT]=NUL;
			}
			else
			{
				used=strlen(text);
				left=MAX_EVENT_TEXT-used;
			}
		}
		va_end(ap);
	}

	// Trailing newline
	if (left>0)
	{
		check=snprintf(&text[used],left,"\n");
		if ( (check<0) || (check==MAX_EVENT_TEXT) )
		{
			used=MAX_EVENT_TEXT;
			left=0;
			text[MAX_EVENT_TEXT]=NUL;
		}
		else
		{
			used=strlen(text);
			left=MAX_EVENT_TEXT-used;
		}
	}

	// Set previous priority flag
	last_priority=VERBOSITY_ERRORS;

	// Print text
	fprintf(stderr,text);

	// Abort program
	exit ((int)(status));
}

// Severe (internal) error logger
void FatalAbort(
	unsigned int status,										// status code for this error
	char *filenm,												// source file name (from __FILE__)
	long int lineno,											// source line number (from __LINE__)
	char *message,												// message format
	...)														// message arguments
{
	// Variables
	char text[MAX_EVENT_TEXT+1]="";
	int check;
	size_t used=0;
	size_t left=MAX_EVENT_TEXT;

	// Newline
	(void) sprintf(text,"\n");
	used=strlen(text);
	left=MAX_EVENT_TEXT-used;

	// Heading
	if (left>0)
	{
		check=snprintf(&text[used],left,"\7FATAL: Error in program file %s line %ld\n",filenm,lineno);
		if ( (check<0) || (check==MAX_EVENT_TEXT) )
		{
			used=MAX_EVENT_TEXT;
			left=0;
			text[MAX_EVENT_TEXT]=NUL;
		}
		else
		{
			used=strlen(text);
			left=MAX_EVENT_TEXT-used;
		}
	}

	// Message
	if ( (message==NULL) || (strlen(message)==0) )
	{
		if (left>0)
		{
			check=snprintf(&text[used],left,"[no message specified]");
			if ( (check<0) || (check==MAX_EVENT_TEXT) )
			{
				used=MAX_EVENT_TEXT;
				left=0;
				text[MAX_EVENT_TEXT]=NUL;
			}
			else
			{
				used=strlen(text);
				left=MAX_EVENT_TEXT-used;
			}
		}
	}
	else
	{
#ifndef S_SPLINT_S
		va_list as;
		if (left>0)
		{
			check=snprintf(&text[used],left,"\7CAUSE: ");
			if ( (check<0) || (check==MAX_EVENT_TEXT) )
			{
				used=MAX_EVENT_TEXT;
				left=0;
				text[MAX_EVENT_TEXT]=NUL;
			}
			else
			{
				used=strlen(text);
				left=MAX_EVENT_TEXT-used;
			}
		}
		va_start(as, message);
		if (left>0)
		{
			check=vsnprintf(&text[used],left,message,as);
			if ( (check<0) || (check==MAX_EVENT_TEXT) )
			{
				used=MAX_EVENT_TEXT;
				left=0;
				text[MAX_EVENT_TEXT]=NUL;
			}
			else
			{
				used=strlen(text);
				left=MAX_EVENT_TEXT-used;
			}
		}
		va_end(as);
		if (left>0)
		{
			check=snprintf(&text[used],left,"\n");
			if ( (check<0) || (check==MAX_EVENT_TEXT) )
			{
				used=MAX_EVENT_TEXT;
				left=0;
				text[MAX_EVENT_TEXT]=NUL;
			}
			else
			{
				used=strlen(text);
				left=MAX_EVENT_TEXT-used;
			}
		}
#endif
	}

	// Print text
	fprintf(stderr,text);

	// Abort program
	exit ((int)(status));
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
