/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Definitions

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <malloc.h>
#include "services.h"
#include "waddef.h"

/**********************************************************************************************************************************/
/************************************************************ Utility *************************************************************/
/**********************************************************************************************************************************/

// WildCard string match for WAD lump names
bool_t wcmatch(const char *str, const char *pat)
{
	// DOCUMENTATION
	/*
		? match single character at that point
		# match single digit at that point
		* match anything from this point onwards

		This function is designed to work irrespective of
		whether the strings are C strings (8+NUL) or WAD
		lump names (8 padded with NUL where less than 8).
		THE INPUT IS ASSUMED TO BE 8 OR LESS IN ALL CASES.
	*/

	// VARIABLES
	size_t i,str_len,pat_len,max_len;
	/*******************************/

	// GET STRING LENGTH
	str_len=0;
	while ( (str_len<8) && (str[str_len]!=NUL) )
	{
		str_len++;
	}

	// GET PATTERN LENGTH
	pat_len=0;
	while ( (pat_len<8) && (pat[pat_len]!=NUL) )
	{
		pat_len++;
	}

	// GET MAXIMUM LENGTH
	max_len=(str_len>=pat_len)?str_len:pat_len;

	//PERFORM FUNCTION
	for (i=0;i<max_len;i++)
	{
		// If past end of pattern then no match
		if (i>=pat_len)
		{
			return FALSE;
		}
		// If past end of string with no '*' then no match
		if ( (i>=str_len) && (pat[i]!='*') )
		{
			return FALSE;
		}
		// If hit '*' in pattern than match
		if (pat[i]=='*')
		{
			return TRUE;
		}
		// If not equal and not ('?' or '#') then no match
		if (pat[i]!=str[i])
		{
			if (pat[i]=='?')
			{
				continue;
			}
			if ( (pat[i]=='#') && (ISDIGIT(str[i])) )
			{
				continue;
			}
			return FALSE;
		}
	}
	return TRUE;
}

// String match for WAD lump names
bool_t wnmatch (const char *p, const char *q)
{
	/*
		This function is designed to work irrespective of
		whether the strings are C strings (8+NUL) or WAD
		lump names (8 padded with NUL where less than 8).
		THE INPUT IS ASSUMED TO BE 8 OR LESS IN ALL CASES.
	*/
	if ( (p[0]==NUL) && (q[0]==NUL) )
		return TRUE;
	if (p[0]!=q[0])
		return FALSE;
	if ( (p[1]==NUL) && (q[1]==NUL) )
		return TRUE;
	if (p[1]!=q[1])
		return FALSE;
	if ( (p[2]==NUL) && (q[2]==NUL) )
		return TRUE;
	if (p[2]!=q[2])
		return FALSE;
	if ( (p[3]==NUL) && (q[3]==NUL) )
		return TRUE;
	if (p[3]!=q[3])
		return FALSE;
	if ( (p[4]==NUL) && (q[4]==NUL) )
		return TRUE;
	if (p[4]!=q[4])
		return FALSE;
	if ( (p[5]==NUL) && (q[5]==NUL) )
		return TRUE;
	if (p[5]!=q[5])
		return FALSE;
	if ( (p[6]==NUL) && (q[6]==NUL) )
		return TRUE;
	if (p[6]!=q[6])
		return FALSE;
	if ( (p[7]==NUL) && (q[7]==NUL) )
		return TRUE;
	if (p[7]!=q[7])
		return FALSE;
	return TRUE;
}

// Page (screen image) auto-detection
bool_t ispage (const void *lump, size_t length)
{
	//  DOCUMENTATION
	/*
		THE INPUT MUST BE EITHER A DOOM GRAPHIC OR PAGE GRAPHIC

		Heretic and HeXen just have to be different, don't they.
		Some of the graphics they use, such as the TITLE and
		FINALE graphics, are not DOOM format graphics but raw
		page images 320x200 (64000). Worse still, some of them
		have the same names as DOOM format images. Even worse,
		you CAN replace them with DOOM format images as some
		source ports analyse the image and by themselves decide
		what it is: so we have to do the same.
	*/

	// VARIABLES
	size_t i;
	SINT16 height;
	SINT16 width;
	SINT16 left_offset;
	SINT16 top_offset;

	// PRIMARY CHECK
	if (length!=GRAPHIC_PAGE_SIZE)
	{
		// Doesn't fit
		return FALSE;
	}

	// GET METRICS ASSUMING THAT IT IS A DOOM FORMAT GRAPHIC
	width=RDLend16SV(*((SINT16*)((picture_header_t*)lump)));
	height=RDLend16SV(*((SINT16*)((picture_header_t*)lump+1)));
	left_offset=RDLend16SV(*((SINT16*)((picture_header_t*)lump+2)));
	top_offset=RDLend16SV(*((SINT16*)((picture_header_t*)lump+3)));

	// IF 64000 IN SIZE THERE IS A HARD LIMIT ON WIDTH OF 16996
	// FOR A DOOM-FORMAT PICTURE. SUCH PICTURES ARE AS FOLLOWS:
	//
	// 8 ......... HEADER
	// 16996*4 ... POINTER ARRAY
	// 5..7  ..... A VERY SHORT PIXEL RUN (SHARED BY ALL COLUMNS)
	// 3..0 ...... UNUSED SPACE APART FROM PIXEL RUN
	// 1 ......... END OF COLUMN FLAG
	//
	if (width>15996)
	{
		// Not valid for a DOOM format graphic, so assume page
		return TRUE;
	}

	// OTHERWISE USE THE SAME CHECKS AS IN THE MAIN PROGRAM
	if (
	     (width       <=  0                      ) ||
	     (width       >   GRAPHIC_MAX_WIDTH      ) ||
	     (height      <=  0                      ) ||
	     (height      >   GRAPHIC_MAX_HEIGHT     ) ||
	     (left_offset >   GRAPHIC_MAX_LEFTOFFSET ) ||
	     (left_offset <  -GRAPHIC_MAX_LEFTOFFSET ) ||
	     (top_offset  >   GRAPHIC_MAX_TOPOFFSET  ) ||
	     (top_offset  <  -GRAPHIC_MAX_TOPOFFSET  )
	   )
	{
		// Not valid for a DOOM format graphic, so assume page
		return TRUE;
	}

	// CHECK COLUMN POINTERS AND DATA
	/*
		None of the column pointer array entries may point past
		the end of the lump and none of the column data entries
		may extend past the end of the lump. We might assume
		that at least one must point to the first byte past the
		column pointer array; but we do not, because a patch,
		might have holes in it if it has not been rebuilt.
	*/
	for (i=0;i<(size_t)(width);i++)
	{
		// VARIABLES
		UINT08 *curr_column_ptr;
		UINT32 curr_column_ofs;
		UINT32 *inpointers=(UINT32*)lump+(sizeof(picture_header_t)/sizeof(UINT32));

		// INITIALISE
		curr_column_ofs=RDLend32UV(inpointers[i]);

		// CHECK COLUMN POINTER
		if (curr_column_ofs>GRAPHIC_PAGE_LAST)
		{
			// Not valid for a DOOM format graphic, so assume page
			return TRUE;
		}

		// CHECK COLUMN DATA
		/*
			COLPTR-->[row_number][pixel_count][dummy_pixel]PIXELS(pixel_count)[dummy_pixel]
			IF row_number==255 THEN END-OF-COLUMN
		*/
		curr_column_ptr=(UINT08*)lump+curr_column_ofs;
		while (*curr_column_ptr!=(UINT08)255)
		{
			// VARIABLES
			size_t pixel_count,next_run_offset;

			// GET PIXEL COUNT
			pixel_count=(size_t)(*(curr_column_ptr+1));

			// CHECK THAT (END OF RUN + 1) IS INSIDE LUMP
			next_run_offset=GRAPHIC_PIXEL_RUN_HEADER_SIZE+pixel_count+GRAPHIC_DUMMY_PIXEL_COUNT;
			if (curr_column_ofs+next_run_offset>GRAPHIC_PAGE_LAST)
			{
				// Not valid for a DOOM format graphic, so assume page
				return TRUE;
			}

			// GO TO NEXT RUN
			curr_column_ptr+=next_run_offset;
#ifdef _WIN64
			if (next_run_offset>UINT32_MAX)
			{
				FatalAbort(1,__FILE__,__LINE__,"Value in \"next_run_offset\" is > 32 bits");
			}
#endif
			#ifdef __VISUALC__
			#pragma warning( push )
			#pragma warning( disable : 4267 )
			#endif
			curr_column_ofs+=next_run_offset;
			#ifdef __VISUALC__
			#pragma warning( pop )
			#endif
		}
	}
	// LOOKS LIKE A VALID DOOM FORMAT GRAPHIC
	return FALSE;
}

// Sound auto-detection
bool_t maybesound (const void *header, size_t lump_size)
{
	//
	// Variables
	//
	sound_header_t *sound_header=(sound_header_t*)header;
	wave_header_t *wave_header=(wave_header_t*)header;

	//
	// Check for MS-WAVE format
	//
	if (
	    (wave_header->riff_magic[0]=='R') &&
	    (wave_header->riff_magic[1]=='I') &&
	    (wave_header->riff_magic[2]=='F') &&
	    (wave_header->riff_magic[3]=='F')   
	   )
	{
		// VARIABLES
		char *char_header=(char*)header;

		// MAY BE WAVE FILE
		if (sizeof(wave_header_t)+(RDLend32UV(wave_header->data_size)*sizeof(UINT08))!=lump_size)
		{
			goto not_wave;
		}
		if (
		    ((char_header[WAVE_MAGIC_OFFSET+0])!='W') ||
		    ((char_header[WAVE_MAGIC_OFFSET+1])!='A') ||
		    ((char_header[WAVE_MAGIC_OFFSET+2])!='V') ||
		    ((char_header[WAVE_MAGIC_OFFSET+3])!='E')   
		   )
		{
		   goto not_wave;
		}
		return TRUE;
	}
	not_wave:

	//
	// Check for DOOM-DMX format
	//
	if (RDLend16UV(sound_header->magic)==3)
	{
		// MAY BE SOUND FILE
		if (sizeof(sound_header_t)+(RDLend32UV(sound_header->num_samples)*sizeof(UINT08))>lump_size)
		{
			goto not_sound;
		}
		if ( (RDLend32UV(sound_header->sample_rate)!=11025) && (RDLend32UV(sound_header->sample_rate)!=22050) )
		{
			goto not_sound;
		}
		return TRUE;
	}
	not_sound:

	//
	// Not recognized
	//
	return FALSE;
};

// Music auto-detection
bool_t maybemusic (const void *header, size_t lump_size)
{
	//
	// Variables
	//
	music_header_t *music_header=(music_header_t*)header;
	midi_header_t *midi_header=(midi_header_t*)header;

	//
	// Check for DOOM-MUS format
	//
	if (
	    (music_header->magic[0]=='M' ) &&
	    (music_header->magic[1]=='U' ) &&
	    (music_header->magic[2]=='S' ) &&
	    (music_header->magic[3]==0x1A)  
	   )
	{
		// MAY BE MUSIC FILE
#if 0
		// Randy does this in ZDOOM.
		// Wonder how valid this is?
		if (RDLend16UV(music_header->num_primary)>15)
		{
			goto not_music;
		}
#endif
		if (RDLend16UV(music_header->mustbezero)!=0)
		{
			goto not_music;
		}
		if ((RDLend16UV(music_header->header_size)+(RDLend16UV(music_header->num_samples)*sizeof(UINT08)))>lump_size)
		{
			goto not_music;
		}
#if 0
		// Some real-world musics violate this
		if ((16/*fixed fields*/+(RDLend16UV(music_header->num_patches)*sizeof(UINT16)))!=RDLend16UV(music_header->header_size))
#else
		if ((16/*fixed fields*/+(RDLend16UV(music_header->num_patches)*sizeof(UINT16)))>RDLend16UV(music_header->header_size))
#endif
		{
			goto not_music;
		}
		return TRUE;
	}
	not_music:

	//
	// Check for MIDI format
	//
	if (
	    (midi_header->magic[0]=='M') &&
	    (midi_header->magic[1]=='T') &&
	    (midi_header->magic[2]=='h') &&
	    (midi_header->magic[3]=='d')  
	   )
	{
		// VARIABLES
		UINT16 format=RDBend16UV(midi_header->format);
		char *char_header=(char*)header;

		// MAY BE MIDI FILE
		if (RDBend32UV(midi_header->header_rest)!=6)
		{
			goto not_midi;
		}
		if ( (format!=0) && (format!=1) && (format!=2) )
		{
			goto not_midi;
		}
		if ((format==0) && (RDBend16UV(midi_header->num_tracks)!=1))
		{
			goto not_midi;
		}
		if (
		    ((char_header[MTRK_MAGIC_OFFSET+0])!='M') ||
		    ((char_header[MTRK_MAGIC_OFFSET+1])!='T') ||
		    ((char_header[MTRK_MAGIC_OFFSET+2])!='r') ||
		    ((char_header[MTRK_MAGIC_OFFSET+3])!='k')   
		   )
		{
		   goto not_midi;
		}
		return TRUE;
	}
	not_midi:

	//
	// Not recognized
	//
	return FALSE;
}

// Graphic auto-detection 
bool_t maybegraphic (const void *header, size_t lump_size, FILE *infile, fofs_t lump_pos, fofs_t curr_pos)
{
	//
	// Variables
	//
	picture_header_t *picture_header=(picture_header_t*)header;

	if (1)
	{
		// VARIABLES
		size_t i;
		SINT16 height;
		SINT16 width;
		SINT16 left_offset;
		SINT16 top_offset;
		void *lump;

		//
		// MAY BE A DOOM FORMAT PICTURE
		//

		// GET METRICS ASSUMING THAT IT IS A DOOM FORMAT GRAPHIC
		width=RDLend16SV(picture_header->width);
		height=RDLend16SV(picture_header->height);
		left_offset=RDLend16SV(picture_header->left_offset);
		top_offset=RDLend16SV(picture_header->top_offset);

		// USE THE SAME CHECKS AS IN THE MAIN PROGRAM
		if (
		     (width       <=  0                      ) ||
		     (width       >   GRAPHIC_MAX_WIDTH      ) ||
		     (height      <=  0                      ) ||
		     (height      >   GRAPHIC_MAX_HEIGHT     ) ||
		     (left_offset >   GRAPHIC_MAX_LEFTOFFSET ) ||
		     (left_offset <  -GRAPHIC_MAX_LEFTOFFSET ) ||
		     (top_offset  >   GRAPHIC_MAX_TOPOFFSET  ) ||
		     (top_offset  <  -GRAPHIC_MAX_TOPOFFSET  )
		   )
		{
			goto not_picture;
		}

		//
		// HAVE TO LOAD THE ENTIRE LUMP NOW
		//

		// SEEK TO THE ENTRY
		if (fseek(infile,lump_pos,SEEK_SET)!=0)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not seek in file");
		}

		// ALLOCATE THE NECESSARY MEMORY
		lump=malloc(lump_size);
		if (lump==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for lump",lump_size);
		}

		// READ IN THE ENTRY
		if (fread(lump,lump_size,1,infile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
		}

		// RESTORE PREVIOUS FILE POSITION
		if (fseek(infile,curr_pos,SEEK_SET)!=0)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not seek in file");
		}

		// CHECK COLUMN POINTERS AND DATA
		/*
			None of the column pointer array entries may point past
			the end of the lump and none of the column data entries
			may extend past the end of the lump. We might assume
			that at least one must point to the first byte past the
			column pointer array; but we do not, because a patch,
			might have holes in it if it has not been rebuilt.
		*/
		for (i=0;i<(size_t)(width);i++)
		{
			// VARIABLES
			UINT08 *curr_column_ptr;
			UINT32 curr_column_ofs;
			UINT32 *inpointers=(UINT32*)lump+(sizeof(picture_header_t)/sizeof(UINT32));

			// INITIALISE
			curr_column_ofs=RDLend32UV(inpointers[i]);

			// CHECK COLUMN POINTER
			if (curr_column_ofs>lump_size)
			{
				free(lump);
				goto not_picture;
			}

			// CHECK COLUMN DATA
			/*
				COLPTR-->[row_number][pixel_count][dummy_pixel]PIXELS(pixel_count)[dummy_pixel]
				IF row_number==255 THEN END-OF-COLUMN
			*/
			curr_column_ptr=(UINT08*)lump+curr_column_ofs;
			while (*curr_column_ptr!=(UINT08)255)
			{
				// VARIABLES
				size_t pixel_count,next_run_offset;
	
				// GET PIXEL COUNT
				pixel_count=(size_t)(*(curr_column_ptr+1));
	
				// CHECK THAT (END OF RUN + 1) IS INSIDE LUMP
				next_run_offset=GRAPHIC_PIXEL_RUN_HEADER_SIZE+pixel_count+GRAPHIC_DUMMY_PIXEL_COUNT;
				if (curr_column_ofs+next_run_offset>lump_size)
				{
					free(lump);
					goto not_picture;
				}
	
				// GO TO NEXT RUN
				curr_column_ptr+=next_run_offset;
				#ifdef _WIN64
				if (next_run_offset>UINT32_MAX)
				{
					FatalAbort(1,__FILE__,__LINE__,"Value in \"next_run_offset\" is > 32 bits");
				}
				#endif
				#ifdef __VISUALC__
				#pragma warning( push )
				#pragma warning( disable : 4267 )
				#endif
				curr_column_ofs+=next_run_offset;
				#ifdef __VISUALC__
				#pragma warning( pop )
				#endif
			}
		}
		free(lump);
		return TRUE;
	}
	not_picture:

	//
	// Not recognized
	//
	return FALSE;
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
