#ifndef __WADDEF_H__
#define __WADDEF_H__

/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Definitions

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include "services.h"

/**********************************************************************************************************************************/
/********************************************************* Game Selection *********************************************************/
/**********************************************************************************************************************************/

// Games
typedef enum
{
	GAME_UNKNOWN,												// game is unknown
	GAME_DOOM,													// game is DOOM
	GAME_DOOM2,													// game is DOOM II
	GAME_TNT,													// game is TNT: Evilution
	GAME_PLUTONIA,												// game is The Plutionia Experiment
	GAME_HERETIC,												// game is Heretic
	GAME_HEXEN,													// game is HeXen
	GAME_DEATHKINGS,											// game is Deathkings of the Dark Citadel
	GAME_STRIFE,												// game is Strife
	GAME_MAXPLUSONE												// number of games
} game_t;

/**********************************************************************************************************************************/
/*********************************************************** Entry Type ***********************************************************/
/**********************************************************************************************************************************/

// Types
typedef enum
{
	TYPE_UNKNOWN,												// Unknown
	TYPE_MAPNAME,												// MAP01, E3M4, ...
	TYPE_MAPDATA,												// THINGS, LINEDEFS, ...
	TYPE_GLMAPNAME,												// GL_MAP01, GL_E3M4, ...
	TYPE_GLMAPDATA,												// GL_VERT, GL_SEGS, ...
	TYPE_GRAPHIC,												// Wall or generic graphic
	TYPE_PAGE,													// Raw page graphic (Heretic and HeXen: credit, help, title, etc.)
	TYPE_AUTO,													// Either a graphic or a page but must auto-detect which
	TYPE_FLAT,													// Floor or ceiling tile graphic
	TYPE_PCSFX,													// PC speaker sound
	TYPE_SOUND,													// Sound
	TYPE_VOICE,													// Voice (loose STRIFE VOICE; special case of Sound)
	TYPE_MUSIC,													// Music
	TYPE_TEXTURE,												// TEXTURES or TEXTUREn
	TYPE_PNAMES,												// PNAMES
	TYPE_DIALOG,												// Dialog message (from STRIFE)
	TYPE_CONVERSATION,											// Conversation script (from STRIFE)
	TYPE_BEHAVIOR,												// Named BEHAVIOUR lump in an ACS library
	TYPE_SCRIPTS,												// SCRIPTS lump in an ACS library
	TYPE_DETECTED_SOUND,										// Sound (auto-detected, temporary use, replaced by Sound)
	TYPE_DETECTED_MUSIC,										// Music (auto-detected, temporary use, replaced by Music)
	TYPE_DETECTED_GRAPHIC,										// Graphic (auto-detected, temporary use, replaced by Graphic)
	TYPE_GOODSTART,												// S_START,P_START,F_START,C_START,A_START,TX_START,V_START
	TYPE_POORSTART,												// SS_START,PP_START,FF_START
	TYPE_SUBSTART,												// S[1-3]_START,P[1-3]_START,F[1-3]_START
	TYPE_SOMESTART,												// A START, Jim,but not as we know it ...
	TYPE_HERIANFONTSTART,										// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
	TYPE_HERETICFONTSTART,										// hexen font start FONTAY_S
	TYPE_HEXENFONTSTART,										// heretic/hexen font start FONTA_S,FONTAY_S,FONTB_S,
	TYPE_GOODEND,												// S_END,P_END,F_END,C_END,A_END,TX_END,V_END
	TYPE_POOREND,												// SS_END,PP_END,FF_END
	TYPE_SUBEND,												// S[1-3]_END,P[1-3]_END,F[1-3]_END
	TYPE_SOMEEND,												// An END, Jim,but not as we know it ...
	TYPE_HERIANFONTEND,											// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
	TYPE_HERETICFONTEND,										// hexen font end FONTAY_E
	TYPE_HEXENFONTEND,											// heretic/hexen font end FONTA_E,FONTAY_E,FONTB_E,
	TYPE_MAXPLUSONE												// highest type enumeration value plus one
} type_t;

/**********************************************************************************************************************************/
/***************************************************** Entry List Membership ******************************************************/
/**********************************************************************************************************************************/

// Lists
typedef enum
{
	LIST_UNKNOWN,												// unknown list type
	LIST_MAPS,													// levels
	LIST_TEXTURES,												// textures (composed of wall patches)
	LIST_PNAMES,												// list of wall patch names
	LIST_PCSFXS,												// PC speaker sounds
	LIST_SOUNDS,												// sounds
	LIST_MUSICS,												// music track
	LIST_DIALOGS,												// dialog messages (from STRIFE)
	LIST_CONVERSATIONS,											// conversation scripts (from STRIFE)
	LIST_LUMPS,													// generic lumps
	LIST_LOOSEWALLPATCHES,										// wall patches (as listed in PNAMES but not found between markers)
	LIST_GRAPHICS,												// miscellaneous graphics
	LIST_HERIANFONTA,											// heretic/hexen font "A"
	LIST_HERIANFONTAY,											// hexen font "AY"
	LIST_HERIANFONTB,											// heretic/hexen font "B"
	LIST_HERETICFONTA,											// heretic font "A"
	LIST_HERETICFONTB,											// heretic font "B"
	LIST_HEXENFONTA,											// hexen font "A"
	LIST_HEXENFONTAY,											// hexen font "AY"
	LIST_HEXENFONTB,											// hexen font "B"
	LIST_SPRITES,												// actor sprites
	LIST_PATCHES,												// wall patches (as found between markers)
	LIST_FLATS,													// foors and ceilings
	LIST_COLORMAPS,												// color maps (from BOOM)
	LIST_ACSLIBRARIES,											// ACS libraries (from ZDOOM)
	LIST_AUTOTEXTURES,											// single-patch textures (independent of PNAMES) (from ZDOOM)
	LIST_VOICES,												// actor voices (from STRIFE)
	LIST_SPRITESSUB1,											// sprites sub-list #1
	LIST_SPRITESSUB2,											// sprites sub-list #2
	LIST_SPRITESSUB3,											// sprites sub-list #3
	LIST_PATCHESSUB1,											// patches sub-list #1
	LIST_PATCHESSUB2,											// patches sub-list #2
	LIST_PATCHESSUB3,											// patches sub-list #3
	LIST_FLATSSUB1,												// flats sub-list #1
	LIST_FLATSSUB2,												// flats sub-list #2
	LIST_FLATSSUB3,												// flats sub-list #3
	LIST_MAXPLUSONE												// highest (standard) list enumeration value plus one
} list_enum_t;

typedef size_t list_t;											// must allow for custom list types

#define LIST_MAXREALVALUE ((list_t)(SINT32_MAX-1))				// should be big enough ((2^31)-1) for any "real" WAD file
#define LIST_UNRECOGNISED ((list_t)(SINT32_MAX))				// used to indicate "unrecognised (not unknown) list type symbol"

//
// The kind_t (see below) is used by the "intrinsic" sortation
// to sort lumps that have a defined order by that order where
// they would otherwise be sorted strictly alphabetically.
//

//
// HANDLING OF GAME_UNKNOWN WHEN SORTING
//
// Referring to file "wadlin.c": where GAME_UNKNOWN is set TRUE
// in the lookup table (variable "kind_maps"), this is allowed
// because it does not matter which game "owns" the kind that
// is actually assigned to a lump if game is GAME_UNKNOWN, as
// long as that kind is DEFINED in the right order in kind_t.
//
// Understand that kind_t is used PURELY as a numeric index for
// ordering lumps within a list and thus the actual values are
// immaterial and have no meaning. When game is GAME_UNKNOWN,
// it is quite all right to assign the first matching kind
// found, as long as the lump name is unique to one game.
//
// It is also all right if the lump name is NOT unique to one
// game, provided the location of the assigned kind within the
// definition of kind_t is set so as to yield the correct sort
// order. The reader will notice that the generic lump kinds
// are all interleaved, as are the DOOM and STRIFE music names.
// This is precisely the reason: there are some names in common
// but they must be defined in the correct order relative to
// each other in their host game and this is the only way to do
// this without requiring a complex multi-pass logic solution.
//
// Fortunately there are no instances in the standard IWADs in
// which intrinsic order dependencies conflict for any sequence
// of generic lump or DOOM/STRIFE music names. If there were,
// some definitions would need to occur in two places at once,
// and that would render this simplistic solution unusable.
//

// Given the foregoing, the location in the lookup table of
// each entry is immaterial as regards the kind values: it is
// THE ORDER OF THESE VALUES IN THE DEFINITION OF "kind_t" that
// is important. Where order IS important in the lookup table,
// is in the use of wildcards that overlap. For example to get
// WILV## followed by ALL other "WI" lumps, you need the match
// strings "WILV##" and "WI*" IN THAT ORDER. In that way, the
// lumps wanted first (WILV##) are matched and then the more
// aggressive wildcard (WI*) matches the remaining "WI" lumps.
//

// If a PWAD contains custom music or generic lump names (i.e.,
// lumps whose names do not exist in the applicable IWAD), then
// it is best to avoid the "intrinsic" sortation, as CleanWAD
// will not know how to order the new names. In this context,
// "custom" includes the use of existing names as if they were
// custom names; for example, using D_E1M1 in a DOOM2 game.
//

// Kind (sub-list type for intra-list sortation)
typedef enum
{
	KIND_BEGINLIST,												// entry is the START marker of a list
	//
	// DOOM/STRIFE MUSIC NAMES
	//
	KIND_1________D_EXMY,										// entry name is "D_E#M#"
	KIND________S_D_LOGO,										// entry name is "D_LOGO"
	KIND________S_D_ACTION,										// entry name is "D_ACTION"
	KIND________S_D_TAVERN,										// entry name is "D_TAVERN"
	KIND________S_D_DANGER,										// entry name is "D_DANGER"
	KIND________S_D_FAST,										// entry name is "D_FAST"
	KIND_1______S_D_INTRO,										// entry name is "D_INTRO"
	KIND________S_D_DARKER,										// entry name is "D_SARKER"
	KIND________S_D_STRIKE,										// entry name is "D_STRIKE"
	KIND________S_D_SLIDE,										// entry name is "D_SLIDE"
	KIND________S_D_TRIBAL,										// entry name is "D_TRIBAL"
	KIND________S_D_MARCH,										// entry name is "D_MARCH"
	KIND________S_D_MOOD,										// entry name is "D_MOOD"
	KIND________S_D_CASTLE,										// entry name is "D_CASTLE"
	KIND________S_D_FIGHT,										// entry name is "D_FIGHT"
	KIND________S_D_SPENSE,										// entry name is "D_SPENSE"
	KIND________S_D_DARK,										// entry name is "D_DARK"
	KIND________S_D_TECH,										// entry name is "D_TECH"
	KIND________S_D_DRONE,										// entry name is "D_DRONE"
	KIND________S_D_PANTHR,										// entry name is "D_PANTHR"
	KIND________S_D_SAD,										// entry name is "D_SAD"
	KIND________S_D_INSTRY,										// entry name is "D_INSTRY"
	KIND________S_D_HAPPY,										// entry name is "D_HAPPY"
	KIND________S_D_END,										// entry name is "D_END"
	KIND_1________D_INTROA,										// entry name is "D_INTROA"
	KIND_1________D_INTER,										// entry name is "D_INTER"
	KIND_1________D_VICTOR,										// entry name is "D_VICTOR"
	KIND_1________D_BUNNY,										// entry name is "D_BUNNY"
	//
	// DOOM2/TNT/PLUTONIA MUSIC NAMES
	//
	KIND__2TP_____D_RUNNIN,										// entry name is "D_RUNNIN"
	KIND__2TP_____D_STALKS,										// entry name is "D_STALKS"
	KIND__2TP_____D_COUNTD,										// entry name is "D_COUNTD"
	KIND__2TP_____D_BETWEE,										// entry name is "D_BETWEE"
	KIND__2TP_____D_DOOM,										// entry name is "D_DOOM"
	KIND__2TP_____D_THE_DA,										// entry name is "D_THE_DA"
	KIND__2TP_____D_SHAWN,										// entry name is "D_SHAWN"
	KIND__2TP_____D_DDTBLU,										// entry name is "D_DDTBLU"
	KIND__2TP_____D_IN_CIT,										// entry name is "D_IN_CIT"
	KIND__2TP_____D_DEAD,										// entry name is "D_DEAD"
	KIND__2TP_____D_STLKS2,										// entry name is "D_STLKS2"
	KIND__2TP_____D_THEDA2,										// entry name is "D_THEDA2"
	KIND__2TP_____D_DOOM2,										// entry name is "D_DOOM2"
	KIND__2TP_____D_DDTBL2,										// entry name is "D_DDTBL2"
	KIND__2TP_____D_RUNNI2,										// entry name is "D_RUNNI2"
	KIND__2TP_____D_DEAD2,										// entry name is "D_DEAD2"
	KIND__2TP_____D_STLKS3,										// entry name is "D_STLKS3"
	KIND__2TP_____D_ROMERO,										// entry name is "D_ROMERO"
	KIND__2TP_____D_SHAWN2,										// entry name is "D_SHAWN2"
	KIND__2TP_____D_MESSAG,										// entry name is "D_MESSAG"
	KIND__2TP_____D_COUNT2,										// entry name is "D_COUNT2"
	KIND__2TP_____D_DDTBL3,										// entry name is "D_DDTBL3"
	KIND__2TP_____D_AMPIE,										// entry name is "D_AMPIE"
	KIND__2TP_____D_THEDA3,										// entry name is "D_THEDA3"
	KIND__2TP_____D_ADRIAN,										// entry name is "D_ADRIAN"
	KIND__2TP_____D_MESSG2,										// entry name is "D_MESSG2"
	KIND__2TP_____D_ROMER2,										// entry name is "D_ROMER2"
	KIND__2TP_____D_TENSE,										// entry name is "D_TENSE"
	KIND__2TP_____D_SHAWN3,										// entry name is "D_SHAWN3"
	KIND__2TP_____D_OPENIN,										// entry name is "D_OPENIN"
	KIND__2TP_____D_EVIL,										// entry name is "D_EVIL"
	KIND__2TP_____D_ULTIMA,										// entry name is "D_ULTIMA"
	KIND__2TP_____D_READ_M,										// entry name is "D_READ_M"
	KIND__2TP_____D_DM2TTL,										// entry name is "D_DM2TTL"
	KIND__2TP_____D_DM2INT,										// entry name is "D_DM2INT"
	//
	// HERETIC MUSIC NAMES
	//
	KIND_____H____D_EXMY,										// entry name is "MUS_E#M#"
	KIND_____H____D_TITL,										// entry name is "MUS_TITL"
	KIND_____H____D_INTR,										// entry name is "MUS_INTR"
	KIND_____H____D_CPTD,										// entry name is "MUS_CPTD"
	//
	// HEXEN/DEATHKINGS MUSIC NAMES
	//
	KIND______XD__D_WINNOWR,									// entry name is "WINNOWR"
	KIND______XD__D_JACHR,										// entry name is "JACHR"
	KIND______XD__D_SIMONR,										// entry name is "SIMONR"
	KIND______XD__D_WUTZITR,									// entry name is "WUTZITR"
	KIND______XD__D_FALCONR,									// entry name is "FALCONR"
	KIND______XD__D_LEVELR,										// entry name is "LEVELR"
	KIND______XD__D_CHARTR,										// entry name is "CHARTR"
	KIND______XD__D_SWAMPR,										// entry name is "SWAMPR"
	KIND______XD__D_DEEPR,										// entry name is "DEEPR"
	KIND______XD__D_FUBASR,										// entry name is "FUBASR"
	KIND______XD__D_GROVER,										// entry name is "GROVER"
	KIND______XD__D_FORTR,										// entry name is "FORTR"
//	KIND______XD__D_CRYPTR,										// entry name is "CRYPTR"
//	KIND______XD__D_SWAMPR,										// entry name is "SWAMPR"
	KIND______XD__D_FOOJAR,										// entry name is "FOOJAR"
	KIND______XD__D_SIXATER,									// entry name is "SIXATER"
	KIND______XD__D_WOBABYR,									// entry name is "WOBABYR"
	KIND______XD__D_CRYPTR,										// entry name is "CRYPTR"
	KIND______XD__D_FANTAR,										// entry name is "FANTAR"
	KIND______XD__D_BLECHR,										// entry name is "BLECHR"
	KIND______XD__D_VOIDR,										// entry name is "VOIDR"
	KIND______XD__D_CHAP_1R,									// entry name is "CHAP_1R"
	KIND______XD__D_CHAP_2R,									// entry name is "CHAP_2R"
	KIND______XD__D_CHAP_3R,									// entry name is "CHAP_3R"
	KIND______XD__D_CHAP_4R,									// entry name is "CHAP_4R"
	KIND______XD__D_CHIPPYR,									// entry name is "CHIPPYR"
	KIND______XD__D_PERCR,										// entry name is "PERCR"
	KIND______XD__D_SECRETR,									// entry name is "SECRETR"
	KIND______XD__D_BONESR,										// entry name is "BONESR"
	KIND______XD__D_OCTOR,										// entry name is "OCTOR"
	KIND______XD__D_RITHMR,										// entry name is "RITHMR"
	KIND______XD__D_STALKR,										// entry name is "STALKR"
	KIND______XD__D_BORKR,										// entry name is "BORKR"
	KIND______XD__D_CRUCIBR,									// entry name is "CRUCIBR"
	KIND______XD__D_HEXEN,										// entry name is "HEXEN"
	KIND______XD__D_HUB,										// entry name is "HUB"
	KIND______XD__D_HALL,										// entry name is "HALL"
	KIND______XD__D_ORB,										// entry name is "ORB"
	KIND______XD__D_CHESS,										// entry name is "CHESS"
	//
	// LUMP NAMES
	//
	KIND_12TPHXDS_L_DEMO,										// entry name is "DEMO#"
	KIND_12TPHXDS_L_ANIMATED,									// entry name is "ANIMATED"
	KIND_12TPHXDS_L_ANIMDEFS,									// entry name is "ANIMDEFS"
	KIND_12TPHXDS_L_DECALDEF,									// entry name is "DECALDEF"
	KIND_12TPHXDS_L_DECORATE,									// entry name is "DECORATE"
	KIND_12TP_____L_DEHACKED,									// entry name is "DEHACKED"
	KIND_12TP_____L_DEHSUPP,									// entry name is "DEHSUPP"
	KIND_12TPHXDS_L_KEYCONF,									// entry name is "KEYCONF"
	KIND_12TPHXDS_L_MAPINFO,									// entry name is "MAPINFO"
	KIND_12TPHXDS_L_SNDINFO,									// entry name is "SNDINFO"
	KIND_12TPHXDS_L_SNDSEQ,										// entry name is "SNDSEQ"
	KIND_12TPHXDS_L_TERRAIN,									// entry name is "TERRAIN"
	KIND_12TP_____L_DDFLANG,									// entry name is "DDFLANG"
	KIND_12TP_____L_DDFSFX,										// entry name is "DDFSFX"
	KIND_12TP_____L_DDFCOLM,									// entry name is "DDFCOLM"
	KIND_12TP_____L_DDFATK,										// entry name is "DDFATK"
	KIND_12TP_____L_DDFWEAP,									// entry name is "DDFWEAP"
	KIND_12TP_____L_DDFTHING,									// entry name is "DDFTHING"
	KIND_12TP_____L_DDFPLAY,									// entry name is "DDFPLAY"
	KIND_12TP_____L_DDFLINE,									// entry name is "DDFLINE"
	KIND_12TP_____L_DDFSECT,									// entry name is "DDFSECT"
	KIND_12TP_____L_DDFSWTH,									// entry name is "DDFSWTH"
	KIND_12TP_____L_DDFANIM,									// entry name is "DDFANIM"
	KIND_12TP_____L_DDFGAME,									// entry name is "DDFGAME"
	KIND_12TP_____L_DDFLEVL,									// entry name is "DDFLEVL"
	KIND_12TP_____L_RSCRIPT,									// entry name is "RSCRIPT"
	KIND_____H____L_LOADING,									// entry name is "LOADING"
	KIND_12TP_____L_ENDOOM,										// entry name is "ENDOOM"
	KIND_____H____L_ENDTEXT,									// entry name is "ENDTEXT"
	KIND______XD__L_CLUS1MSG,									// entry name is "CLUS1MSG"
	KIND______XD__L_CLUS2MSG,									// entry name is "CLUS2MSG"
	KIND______XD__L_CLUS3MSG,									// entry name is "CLUS3MSG"
	KIND______XD__L_CLUS4MSG,									// entry name is "CLUS4MSG"
	KIND______XD__L_WIN1MSG,									// entry name is "WIN1MSG"
	KIND______XD__L_WIN2MSG,									// entry name is "WIN2MSG"
	KIND______XD__L_WIN3MSG,									// entry name is "WIN3MSG"
	KIND________S_L_T1TEXT,										// entry name is "T1TEXT"
	KIND________S_L_C1TEXT,										// entry name is "C1TEXT"
	KIND________S_L_ENDSTRF,									// entry name is "ENDSTRF"
	KIND______XD__L_TRANTBL0,									// entry name is "TRANTBL0"
	KIND______XD__L_TRANTBL1,									// entry name is "TRANTBL1"
	KIND______XD__L_TRANTBL2,									// entry name is "TRANTBL2"
	KIND______XD__L_TRANTBL3,									// entry name is "TRANTBL3"
	KIND______XD__L_TRANTBL4,									// entry name is "TRANTBL4"
	KIND______XD__L_TRANTBL5,									// entry name is "TRANTBL5"
	KIND______XD__L_TRANTBL6,									// entry name is "TRANTBL6"
	KIND______XD__L_TRANTBL7,									// entry name is "TRANTBL7"
	KIND______XD__L_TRANTBL8,									// entry name is "TRANTBL8"
	KIND______XD__L_TRANTBL9,									// entry name is "TRANTBL9"
	KIND______XD__L_TRANTBLA,									// entry name is "TRANTBLA"
	KIND______XD__L_TRANTBLB,									// entry name is "TRANTBLB"
	KIND______XD__L_TRANTBLC,									// entry name is "TRANTBLC"
	KIND______XD__L_TRANTBLD,									// entry name is "TRANTBLD"
	KIND______XD__L_TRANTBLE,									// entry name is "TRANTBLE"
	KIND______XD__L_TRANTBLF,									// entry name is "TRANTBLF"
	KIND______XD__L_TRANTBLG,									// entry name is "TRANTBLG"
	KIND______XD__L_TRANTBLH,									// entry name is "TRANTBLH"
	KIND______XD__L_TRANTBLI,									// entry name is "TRANTBLI"
	KIND______XD__L_TRANTBLJ,									// entry name is "TRANTBLJ"
	KIND______XD__L_TRANTBLK,									// entry name is "TRANTBLK"
	KIND_12TPHXDS_L_PLAYPAL,									// entry name is "PLAYPAL"
	KIND_____H____L_E2PAL,										// entry name is "E2PAL"
	KIND_12TPHXDS_L_COLORMAP,									// entry name is "COLORMAP"
	KIND______XD__L_FOGMAP,										// entry name is "FOGMAP"
	KIND_____HXD__L_TINTTAB,									// entry name is "TINTTAB"
	KIND________S_L_XLATAB,										// entry name is "XLATAB"
	KIND_12TPHXDS_L_GENMIDI,									// entry name is "GENMIDI"
	KIND_1___HXDS_L_DMXGUS,										// entry name is "DMXGUS"
	KIND__2TP_____L_DMXGUSC,									// entry name is "DMXGUSC"
	KIND_____HXD__L_AUTOPAGE,									// entry name is "AUTOPAGE"
	KIND______XD__L_STARTUP,									// entry name is "STARTUP"
	KIND________S_L_STARTUP0,									// entry name is "STARTUP0"
	KIND________S_L_STRTLZ1,									// entry name is "STRTLZ1"
	KIND________S_L_STRTLZ2,									// entry name is "STRTLZ2"
	KIND________S_L_STRTBOT,									// entry name is "STRTBOT"
	KIND________S_L_STRTPA1,									// entry name is "STRTPA1"
	KIND________S_L_STRTPB1,									// entry name is "STRTPB1"
	KIND________S_L_STRTPC1,									// entry name is "STRTPC1"
	KIND________S_L_STRTPD1,									// entry name is "STRTPD1"
	KIND_____HXD__L_SNDCURVE,									// entry name is "SNDCURVE"
	KIND_____H____L_EXTENDED,									// entry name is "EXTENDED"
	KIND______XD__L_XXTIC,										// entry name is "XXTIC"
	KIND________S_L_SERIAL,										// entry name is "SERIAL"
	KIND________S_L_COPYRITE,									// entry name is "COPYRITE"
//	KIND_____HXD__L_CREDIT,										// entry name is "CREDIT"
//	KIND______XD__L_FINALE1,									// entry name is "FINALE1"
//	KIND______XD__L_FINALE2,									// entry name is "FINALE2"
//	KIND______XD__L_FINALE3,									// entry name is "FINALE3"
//	KIND_____H____L_F_SKY1,										// entry name is "F_SKY1"
//	KIND_____H____L_GFRAG,										// entry name is "GFRAG"
//	KIND_____HXD__L_HELP1,										// entry name is "HELP1"
//	KIND_____HXD__L_HELP2,										// entry name is "HELP2"
//	KIND______XD__L_INTERPIC,									// entry name is "INTERPIC"
//	KIND_____HXD__L_TITLE,										// entry name is "TITLE"
#if 0
	/* BEGIN ************************************************
    * Graphic ordering not yet implemented (compiler limit) *
	********************************************************/
	//
	// DOOM/DOOM2/TNT/PLUTONIA GRAPHIC NAMES
	//
	KIND_12TP_____G_AMMNUM,										// entry name is "AMMNUM#"
	KIND_12TP_____G_BRDR_L,										// entry name is "BRDR_L"
	KIND_12TP_____G_BRDR_R,										// entry name is "BRDR_R"
	KIND_12TP_____G_BRDR_B,										// entry name is "BRDR_B"
	KIND_12TP_____G_BRDR_T,										// entry name is "BRDR_T"
	KIND_12TP_____G_BRDR_BL,									// entry name is "BRDR_BL"
	KIND_12TP_____G_BRDR_BR,									// entry name is "BRDR_BR"
	KIND_12TP_____G_BRDR_TL,									// entry name is "BRDR_TL"
	KIND_12TP_____G_BRDR_TR,									// entry name is "BRDR_TR"
	KIND_12TP_____G_TITLEPIC,									// entry name is "TITLEPIC"
	KIND_12TP_____G_INTERPIC,									// entry name is "INTERPIC"
	KIND_1________G_VICTORY2,									// entry name is "VICTORY2"
	KIND__2TP_____G_BOSSBACK,									// entry name is "BOSSBACK"
	KIND_12TP_____G_CREDIT,										// entry name is "CREDIT"
	KIND_1________G_HELP1,										// entry name is "HELP1"
	KIND__2TP_____G_HELP,										// entry name is "HELP"
	KIND_1________G_PFUB,										// entry name is "PFUB#"
	KIND_1________G_ENDPIC,										// entry name is "ENDPIC"
	KIND_1________G_END,										// entry name is "END#"
	KIND_12TP_____G_M_MENU,										// entry name is "M_*"
	KIND_12TP_____G_STBAR,										// entry name is "STBAR"
	KIND_12TP_____G_STARMS,										// entry name is "STARMS"
	KIND_12TP_____G_STKEYS,										// entry name is "STKEYS#"
	KIND_12TP_____G_STDISK,										// entry name is "STDISK"
	KIND_12TP_____G_STCDROM,									// entry name is "STCDROM"
	KIND_12TP_____G_STGNUM,										// entry name is "STGNUM#"
	KIND_12TP_____G_STTMINUS,									// entry name is "STTMINUS"
	KIND_12TP_____G_STTNUM,										// entry name is "STTNUM#"
	KIND_12TP_____G_STTPRCNT,									// entry name is "STTPRCNT"
	KIND_12TP_____G_STYSNUM,									// entry name is "STYSNUM#"
	KIND_12TP_____G_STCFN0,										// entry name is "STCFN0##"
	KIND_12TP_____G_STCFN1,										// entry name is "STCFN1##"
	KIND_12TP_____G_STFB,										// entry name is "STFB#"
	KIND_12TP_____G_STPB,										// entry name is "STPB#"
	KIND_12TP_____G_STFST01,									// entry name is "STFST01"
	KIND_12TP_____G_STFST00,									// entry name is "STFST00"
	KIND_12TP_____G_STFST02,									// entry name is "STFST02"
	KIND_12TP_____G_STFTL00,									// entry name is "STFTL00"
	KIND_12TP_____G_STFTR00,									// entry name is "STFTR00"
	KIND_12TP_____G_STFOUCH0,									// entry name is "STFOUCH0"
	KIND_12TP_____G_STFEVL0,									// entry name is "STFEVL0"
	KIND_12TP_____G_STFKILL0,									// entry name is "STFKILL0"
	KIND_12TP_____G_STFST11,									// entry name is "STFST11"
	KIND_12TP_____G_STFST10,									// entry name is "STFST10"
	KIND_12TP_____G_STFST12,									// entry name is "STFST12"
	KIND_12TP_____G_STFTL10,									// entry name is "STFTL10"
	KIND_12TP_____G_STFTR10,									// entry name is "STFTR10"
	KIND_12TP_____G_STFOUCH1,									// entry name is "STFOUCH1"
	KIND_12TP_____G_STFEVL1,									// entry name is "STFEVL1"
	KIND_12TP_____G_STFKILL1,									// entry name is "STFKILL1"
	KIND_12TP_____G_STFST21,									// entry name is "STFST21"
	KIND_12TP_____G_STFST20,									// entry name is "STFST20"
	KIND_12TP_____G_STFST22,									// entry name is "STFST22"
	KIND_12TP_____G_STFTL20,									// entry name is "STFTL20"
	KIND_12TP_____G_STFTR20,									// entry name is "STFTR20"
	KIND_12TP_____G_STFOUCH2,									// entry name is "STFOUCH2"
	KIND_12TP_____G_STFEVL2,									// entry name is "STFEVL2"
	KIND_12TP_____G_STFKILL2,									// entry name is "STFKILL2"
	KIND_12TP_____G_STFST31,									// entry name is "STFST31"
	KIND_12TP_____G_STFST30,									// entry name is "STFST30"
	KIND_12TP_____G_STFST32,									// entry name is "STFST32"
	KIND_12TP_____G_STFTL30,									// entry name is "STFTL30"
	KIND_12TP_____G_STFTR30,									// entry name is "STFTR30"
	KIND_12TP_____G_STFOUCH3,									// entry name is "STFOUCH3"
	KIND_12TP_____G_STFEVL3,									// entry name is "STFEVL3"
	KIND_12TP_____G_STFKILL3,									// entry name is "STFKILL3"
	KIND_12TP_____G_STFST41,									// entry name is "STFST41"
	KIND_12TP_____G_STFST40,									// entry name is "STFST40"
	KIND_12TP_____G_STFST42,									// entry name is "STFST42"
	KIND_12TP_____G_STFTL40,									// entry name is "STFTL40"
	KIND_12TP_____G_STFTR40,									// entry name is "STFTR40"
	KIND_12TP_____G_STFOUCH4,									// entry name is "STFOUCH4"
	KIND_12TP_____G_STFEVL4,									// entry name is "STFEVL4"
	KIND_12TP_____G_STFKILL4,									// entry name is "STFKILL4"
	KIND_12TP_____G_STFGOD0,									// entry name is "STFGOD0"
	KIND_12TP_____G_STFDEAD0,									// entry name is "STFDEAD0"
	KIND_1________G_WIMAP,										// entry name is "WIMAP#"
	KIND_1________G_WIA,										// entry name is "WIA#####"
	KIND_1________G_WILV,										// entry name is "WILV##"
	KIND__2TP_____G_CWILV,										// entry name is "CWILV##"
	KIND_12TP_____G_WI,											// entry name is "WI*" (if not already matched)
	//
	// HERETIC GRAPHIC NAMES
	//
	KIND_____H____G_BORDL,										// entry name is "BORDL"
	KIND_____H____G_BORDR,										// entry name is "BORDR"
	KIND_____H____G_BORDT,										// entry name is "BORDT"
	KIND_____H____G_BORDB,										// entry name is "BORDB"
	KIND_____H____G_BORDTL,										// entry name is "BORDTL"
	KIND_____H____G_BORDTR,										// entry name is "BORDTR"
	KIND_____H____G_BORDBL,										// entry name is "BORDBL"
	KIND_____H____G_BORDBR,										// entry name is "BORDBR"
	KIND_____H____G_STATBAR,									// entry name is "STATBAR"
	KIND_____H____G_BLACKSQ,									// entry name is "BLACKSQ"
	KIND_____H____G_ARMCLEAR,									// entry name is "ARMCLEAR"
	KIND_____H____G_INVBAR,										// entry name is "INVBAR"
	KIND_____H____G_ARTIBOX,									// entry name is "ARTIBOX"
	KIND_____H____G_SELECTBO,									// entry name is "SELECTBO"
	KIND_____H____G_USEARTIA,									// entry name is "USEARTIA"
	KIND_____H____G_USEARTIB,									// entry name is "USEARTIB"
	KIND_____H____G_USEARTIC,									// entry name is "USEARTIC"
	KIND_____H____G_USEARTID,									// entry name is "USEARTID"
	KIND_____H____G_USEARTIE,									// entry name is "USEARTIE"
	KIND_____H____G_INVGEML1,									// entry name is "INVGEML1"
	KIND_____H____G_INVGEMR1,									// entry name is "INVGEMR1"
	KIND_____H____G_INVGEML2,									// entry name is "INVGEML2"
	KIND_____H____G_INVGEMR2,									// entry name is "INVGEMR2"
	KIND_____H____G_BKEYICON,									// entry name is "BKEYICON"
	KIND_____H____G_GKEYICON,									// entry name is "GKEYICON"
	KIND_____H____G_YKEYICON,									// entry name is "YKEYICON"
	KIND_____H____G_IN,											// entry name is "IN#"
	KIND_____H____G_NEGNUM,										// entry name is "NEGNUM"
	KIND_____H____G_LAME,										// entry name is "LAME"
	KIND_____H____G_SMALLIN,									// entry name is "SMALLIN#"
	KIND_____H____G_INAMGLD,									// entry name is "INAMGLD"
	KIND_____H____G_INAMBOW,									// entry name is "INAMBOW"
	KIND_____H____G_INAMBST,									// entry name is "INAMBST"
	KIND_____H____G_INAMRAM,									// entry name is "INAMRAM"
	KIND_____H____G_INAMPNX,									// entry name is "INAMPNX"
	KIND_____H____G_INAMLOB,									// entry name is "INAMLOB"
	KIND_____H____G_LIFEBAR,									// entry name is "LIFEBAR"
	KIND_____H____G_BARBACK,									// entry name is "BARBACK"
	KIND_____H____G_CHAINBAC,									// entry name is "CHAINBAC"
	KIND_____H____G_CHAIN,										// entry name is "CHAIN"
	KIND_____H____G_LIFEGEM0,									// entry name is "LIFEGEM0"
	KIND_____H____G_LIFEGEM1,									// entry name is "LIFEGEM1"
	KIND_____H____G_LIFEGEM2,									// entry name is "LIFEGEM2"
	KIND_____H____G_LIFEGEM3,									// entry name is "LIFEGEM3"
	KIND_____H____G_GOD1,										// entry name is "GOD1"
	KIND_____H____G_GOD2,										// entry name is "GOD2"
	KIND_____H____G_LTFACE,										// entry name is "LTFACE"
	KIND_____H____G_RTFACE,										// entry name is "RTFACE"
	KIND_____H____G_LTFCTOP,									// entry name is "LTFCTOP"
	KIND_____H____G_RTFCTOP,									// entry name is "RTFCTOP"
	KIND_____H____G_FACEA0,										// entry name is "FACEA0"
	KIND_____H____G_FACEA1,										// entry name is "FACEA1"
	KIND_____H____G_FACEA2,										// entry name is "FACEA2"
	KIND_____H____G_FACEA3,										// entry name is "FACEA3"
	KIND_____H____G_FACEB0,										// entry name is "FACEB0"
	KIND_____H____G_FACEB1,										// entry name is "FACEB1"
	KIND_____H____G_FACEB2,										// entry name is "FACEB2"
	KIND_____H____G_FACEB3,										// entry name is "FACEB3"
	KIND_____H____G_M_MENU,										// entry name is "M_*"
	KIND_____H____G_SPINBK0,									// entry name is "SPINBK0"
	KIND_____H____G_SPINBK1,									// entry name is "SPINBK1"
	KIND_____H____G_SPINBK2,									// entry name is "SPINBK2"
	KIND_____H____G_SPINBK3,									// entry name is "SPINBK3"
	KIND_____H____G_SPINBK4,									// entry name is "SPINBK4"
	KIND_____H____G_SPINBK5,									// entry name is "SPINBK5"
	KIND_____H____G_SPINBK6,									// entry name is "SPINBK6"
	KIND_____H____G_SPINBK7,									// entry name is "SPINBK7"
	KIND_____H____G_SPINBK8,									// entry name is "SPINBK8"
	KIND_____H____G_SPINBK9,									// entry name is "SPINBK9"
	KIND_____H____G_SPINBK10,									// entry name is "SPINBK10"
	KIND_____H____G_SPINBK11,									// entry name is "SPINBK11"
	KIND_____H____G_SPINBK12,									// entry name is "SPINBK12"
	KIND_____H____G_SPINBK13,									// entry name is "SPINBK13"
	KIND_____H____G_SPINBK14,									// entry name is "SPINBK14"
	KIND_____H____G_SPINBK15,									// entry name is "SPINBK15"
	KIND_____H____G_SPFLY0,										// entry name is "SPFLY0"
	KIND_____H____G_SPFLY1,										// entry name is "SPFLY1"
	KIND_____H____G_SPFLY2,										// entry name is "SPFLY2"
	KIND_____H____G_SPFLY3,										// entry name is "SPFLY3"
	KIND_____H____G_SPFLY4,										// entry name is "SPFLY4"
	KIND_____H____G_SPFLY5,										// entry name is "SPFLY5"
	KIND_____H____G_SPFLY6,										// entry name is "SPFLY6"
	KIND_____H____G_SPFLY7,										// entry name is "SPFLY7"
	KIND_____H____G_SPFLY8,										// entry name is "SPFLY8"
	KIND_____H____G_SPFLY9,										// entry name is "SPFLY9"
	KIND_____H____G_SPFLY10,									// entry name is "SPFLY10"
	KIND_____H____G_SPFLY11,									// entry name is "SPFLY11"
	KIND_____H____G_SPFLY12,									// entry name is "SPFLY12"
	KIND_____H____G_SPFLY13,									// entry name is "SPFLY13"
	KIND_____H____G_SPFLY14,									// entry name is "SPFLY14"
	KIND_____H____G_SPFLY15,									// entry name is "SPFLY15"
	KIND_____H____G_MAPE1,										// entry name is "MAPE1"
	KIND_____H____G_MAPE2,										// entry name is "MAPE2"
	KIND_____H____G_MAPE3,										// entry name is "MAPE3"
	KIND_____H____G_IN_X,										// entry name is "IN_X"
	KIND_____H____G_IN_YAH,										// entry name is "IN_YAH"
	KIND_____H____G_E2END,										// entry name is "E2END"
	KIND_____H____G_PAUSED,										// entry name is "PAUSED"
	KIND_____H____G_ADVISOR,									// entry name is "ADVISOR"
	//
	// HEXEN/DEATHKINGS GRAPHIC NAMES
	//
	KIND______XD__G_BORDL,										// entry name is "BORDL"
	KIND______XD__G_BORDR,										// entry name is "BORDR"
	KIND______XD__G_BORDT,										// entry name is "BORDT"
	KIND______XD__G_BORDB,										// entry name is "BORDB"
	KIND______XD__G_BORDTL,										// entry name is "BORDTL"
	KIND______XD__G_BORDTR,										// entry name is "BORDTR"
	KIND______XD__G_BORDBL,										// entry name is "BORDBL"
	KIND______XD__G_BORDBR,										// entry name is "BORDBR"
	KIND______XD__G_STATBAR,									// entry name is "STATBAR"
	KIND______XD__G_H2BAR,										// entry name is "H2BAR"
	KIND______XD__G_H2TOP,										// entry name is "H2TOP"
	KIND______XD__G_LFEDGE,										// entry name is "LFEDGE"
	KIND______XD__G_RTEDGE,										// entry name is "RTEDGE"
	KIND______XD__G_ARMCLS,										// entry name is "ARMCLS"
	KIND______XD__G_ARTICLS,									// entry name is "ARTICLS"
	KIND______XD__G_INVBAR,										// entry name is "INVBAR"
	KIND______XD__G_ARTIBOX,									// entry name is "ARTIBOX"
	KIND______XD__G_SELECTBO,									// entry name is "SELECTBO"
	KIND______XD__G_USEARTIA,									// entry name is "USEARTIA"
	KIND______XD__G_USEARTIB,									// entry name is "USEARTIB"
	KIND______XD__G_USEARTIC,									// entry name is "USEARTIC"
	KIND______XD__G_USEARTID,									// entry name is "USEARTID"
	KIND______XD__G_USEARTIE,									// entry name is "USEARTIE"
	KIND______XD__G_INVGEML1,									// entry name is "INVGEML1"
	KIND______XD__G_INVGEMR1,									// entry name is "INVGEMR1"
	KIND______XD__G_INVGEML2,									// entry name is "INVGEML2"
	KIND______XD__G_INVGEMR2,									// entry name is "INVGEMR2"
	KIND______XD__G_IN,											// entry name is "IN#"
	KIND______XD__G_NEGNUM,										// entry name is "NEGNUM"
	KIND______XD__G_SMALLIN,									// entry name is "SMALLIN#"
	KIND______XD__G_INRED,										// entry name is "INRED#"
	KIND______XD__G_MANAVL1,									// entry name is "MANAVL1"
	KIND______XD__G_MANAVL2,									// entry name is "MANAVL2"
	KIND______XD__G_MANAVL1D,									// entry name is "MANAVL1D"
	KIND______XD__G_MANAVL2D,									// entry name is "MANAVL2D"
	KIND______XD__G_MANACLS,									// entry name is "MANACLS"
	KIND______XD__G_MANADIM1,									// entry name is "MANADIM1"
	KIND______XD__G_MANADIM2,									// entry name is "MANADIM2"
	KIND______XD__G_MANABRT1,									// entry name is "MANABRT1"
	KIND______XD__G_MANABRT2,									// entry name is "MANABRT2"
	KIND______XD__G_TALLYTOP,									// entry name is "TALLYTOP"
	KIND______XD__G_TALLYLFT,									// entry name is "TALLYLFT"
	KIND______XD__G_CHESSC,										// entry name is "CHESSC"
	KIND______XD__G_CHESSM,										// entry name is "CHESSM"
	KIND______XD__G_CHESSALL,									// entry name is "CHESSALL"
	KIND______XD__G_KILLS,										// entry name is "KILLS"
	KIND______XD__G_KEYBAR,										// entry name is "KEYBAR"
	KIND______XD__G_KEYSLOT1,									// entry name is "KEYSLOT1"
	KIND______XD__G_KEYSLOT2,									// entry name is "KEYSLOT2"
	KIND______XD__G_KEYSLOT3,									// entry name is "KEYSLOT3"
	KIND______XD__G_KEYSLOT4,									// entry name is "KEYSLOT4"
	KIND______XD__G_KEYSLOT5,									// entry name is "KEYSLOT5"
	KIND______XD__G_KEYSLOT6,									// entry name is "KEYSLOT6"
	KIND______XD__G_KEYSLOT7,									// entry name is "KEYSLOT7"
	KIND______XD__G_KEYSLOT8,									// entry name is "KEYSLOT8"
	KIND______XD__G_KEYSLOT9,									// entry name is "KEYSLOT9"
	KIND______XD__G_KEYSLOTA,									// entry name is "KEYSLOTA"
	KIND______XD__G_KEYSLOTB,									// entry name is "KEYSLOTB"
	KIND______XD__G_ARMSLOT1,									// entry name is "ARMSLOT1"
	KIND______XD__G_ARMSLOT2,									// entry name is "ARMSLOT2"
	KIND______XD__G_ARMSLOT3,									// entry name is "ARMSLOT3"
	KIND______XD__G_ARMSLOT4,									// entry name is "ARMSLOT4"
	KIND______XD__G_WPSLOT0,									// entry name is "WPSLOT0"
	KIND______XD__G_WPSLOT1,									// entry name is "WPSLOT1"
	KIND______XD__G_WPSLOT2,									// entry name is "WPSLOT2"
	KIND______XD__G_WPFULL0,									// entry name is "WPFULL0"
	KIND______XD__G_WPFULL1,									// entry name is "WPFULL1"
	KIND______XD__G_WPFULL2,									// entry name is "WPFULL2"
	KIND______XD__G_WPIECEF1,									// entry name is "WPIECEF1"
	KIND______XD__G_WPIECEF2,									// entry name is "WPIECEF2"
	KIND______XD__G_WPIECEF3,									// entry name is "WPIECEF3"
	KIND______XD__G_WPIECEC1,									// entry name is "WPIECEC1"
	KIND______XD__G_WPIECEC2,									// entry name is "WPIECEC2"
	KIND______XD__G_WPIECEC3,									// entry name is "WPIECEC3"
	KIND______XD__G_WPIECEM1,									// entry name is "WPIECEM1"
	KIND______XD__G_WPIECEM2,									// entry name is "WPIECEM2"
	KIND______XD__G_WPIECEM3,									// entry name is "WPIECEM3"
	KIND______XD__G_CHAIN,										// entry name is "CHAIN"
	KIND______XD__G_CHAIN2,										// entry name is "CHAIN2"
	KIND______XD__G_CHAIN3,										// entry name is "CHAIN3"
	KIND______XD__G_LIFEGEM,									// entry name is "LIFEGEM"
	KIND______XD__G_LIFEGMF2,									// entry name is "LIFEGMF2"
	KIND______XD__G_LIFEGMF3,									// entry name is "LIFEGMF3"
	KIND______XD__G_LIFEGMF4,									// entry name is "LIFEGMF4"
	KIND______XD__G_LIFEGMF5,									// entry name is "LIFEGMF5"
	KIND______XD__G_LIFEGMF6,									// entry name is "LIFEGMF6"
	KIND______XD__G_LIFEGMF7,									// entry name is "LIFEGMF7"
	KIND______XD__G_LIFEGMF8,									// entry name is "LIFEGMF8"
	KIND______XD__G_LIFEGMC1,									// entry name is "LIFEGMC1"
	KIND______XD__G_LIFEGMC2,									// entry name is "LIFEGMC2"
	KIND______XD__G_LIFEGMC3,									// entry name is "LIFEGMC3"
	KIND______XD__G_LIFEGMC4,									// entry name is "LIFEGMC4"
	KIND______XD__G_LIFEGMC5,									// entry name is "LIFEGMC5"
	KIND______XD__G_LIFEGMC6,									// entry name is "LIFEGMC6"
	KIND______XD__G_LIFEGMC7,									// entry name is "LIFEGMC7"
	KIND______XD__G_LIFEGMC8,									// entry name is "LIFEGMC8"
	KIND______XD__G_LIFEGMM1,									// entry name is "LIFEGMM1"
	KIND______XD__G_LIFEGMM2,									// entry name is "LIFEGMM2"
	KIND______XD__G_LIFEGMM3,									// entry name is "LIFEGMM3"
	KIND______XD__G_LIFEGMM4,									// entry name is "LIFEGMM4"
	KIND______XD__G_LIFEGMM5,									// entry name is "LIFEGMM5"
	KIND______XD__G_LIFEGMM6,									// entry name is "LIFEGMM6"
	KIND______XD__G_LIFEGMM7,									// entry name is "LIFEGMM7"
	KIND______XD__G_LIFEGMM8,									// entry name is "LIFEGMM8"
	KIND______XD__G_M_MENU,										// entry name is "M_*"
	KIND______XD__G_SPFLY0,										// entry name is "SPFLY0"
	KIND______XD__G_SPFLY1,										// entry name is "SPFLY1"
	KIND______XD__G_SPFLY2,										// entry name is "SPFLY2"
	KIND______XD__G_SPFLY3,										// entry name is "SPFLY3"
	KIND______XD__G_SPFLY4,										// entry name is "SPFLY4"
	KIND______XD__G_SPFLY5,										// entry name is "SPFLY5"
	KIND______XD__G_SPFLY6,										// entry name is "SPFLY6"
	KIND______XD__G_SPFLY7,										// entry name is "SPFLY7"
	KIND______XD__G_SPFLY8,										// entry name is "SPFLY8"
	KIND______XD__G_SPFLY9,										// entry name is "SPFLY9"
	KIND______XD__G_SPFLY10,									// entry name is "SPFLY10"
	KIND______XD__G_SPFLY11,									// entry name is "SPFLY11"
	KIND______XD__G_SPFLY12,									// entry name is "SPFLY12"
	KIND______XD__G_SPFLY13,									// entry name is "SPFLY13"
	KIND______XD__G_SPFLY14,									// entry name is "SPFLY14"
	KIND______XD__G_SPFLY15,									// entry name is "SPFLY15"
	KIND______XD__G_SPSHLD0,									// entry name is "SPSHLD0"
	KIND______XD__G_SPSHLD1,									// entry name is "SPSHLD1"
	KIND______XD__G_SPSHLD2,									// entry name is "SPSHLD2"
	KIND______XD__G_SPSHLD3,									// entry name is "SPSHLD3"
	KIND______XD__G_SPSHLD4,									// entry name is "SPSHLD4"
	KIND______XD__G_SPSHLD5,									// entry name is "SPSHLD5"
	KIND______XD__G_SPSHLD6,									// entry name is "SPSHLD6"
	KIND______XD__G_SPSHLD7,									// entry name is "SPSHLD7"
	KIND______XD__G_SPSHLD8,									// entry name is "SPSHLD8"
	KIND______XD__G_SPSHLD9,									// entry name is "SPSHLD9"
	KIND______XD__G_SPSHLD10,									// entry name is "SPSHLD10"
	KIND______XD__G_SPSHLD11,									// entry name is "SPSHLD11"
	KIND______XD__G_SPSHLD12,									// entry name is "SPSHLD12"
	KIND______XD__G_SPSHLD13,									// entry name is "SPSHLD13"
	KIND______XD__G_SPSHLD14,									// entry name is "SPSHLD14"
	KIND______XD__G_SPSHLD15,									// entry name is "SPSHLD15"
	KIND______XD__G_SPBOOT0,									// entry name is "SPBOOT0"
	KIND______XD__G_SPBOOT1,									// entry name is "SPBOOT1"
	KIND______XD__G_SPBOOT2,									// entry name is "SPBOOT2"
	KIND______XD__G_SPBOOT3,									// entry name is "SPBOOT3"
	KIND______XD__G_SPBOOT4,									// entry name is "SPBOOT4"
	KIND______XD__G_SPBOOT5,									// entry name is "SPBOOT5"
	KIND______XD__G_SPBOOT6,									// entry name is "SPBOOT6"
	KIND______XD__G_SPBOOT7,									// entry name is "SPBOOT7"
	KIND______XD__G_SPBOOT8,									// entry name is "SPBOOT8"
	KIND______XD__G_SPBOOT9,									// entry name is "SPBOOT9"
	KIND______XD__G_SPBOOT10,									// entry name is "SPBOOT10"
	KIND______XD__G_SPBOOT11,									// entry name is "SPBOOT11"
	KIND______XD__G_SPBOOT12,									// entry name is "SPBOOT12"
	KIND______XD__G_SPBOOT13,									// entry name is "SPBOOT13"
	KIND______XD__G_SPBOOT14,									// entry name is "SPBOOT14"
	KIND______XD__G_SPBOOT15,									// entry name is "SPBOOT15"
	KIND______XD__G_SPMINO0,									// entry name is "SPMINO0"
	KIND______XD__G_SPMINO1,									// entry name is "SPMINO1"
	KIND______XD__G_SPMINO2,									// entry name is "SPMINO2"
	KIND______XD__G_SPMINO3,									// entry name is "SPMINO3"
	KIND______XD__G_SPMINO4,									// entry name is "SPMINO4"
	KIND______XD__G_SPMINO5,									// entry name is "SPMINO5"
	KIND______XD__G_SPMINO6,									// entry name is "SPMINO6"
	KIND______XD__G_SPMINO7,									// entry name is "SPMINO7"
	KIND______XD__G_SPMINO8,									// entry name is "SPMINO8"
	KIND______XD__G_SPMINO9,									// entry name is "SPMINO9"
	KIND______XD__G_SPMINO10,									// entry name is "SPMINO10"
	KIND______XD__G_SPMINO11,									// entry name is "SPMINO11"
	KIND______XD__G_SPMINO12,									// entry name is "SPMINO12"
	KIND______XD__G_SPMINO13,									// entry name is "SPMINO13"
	KIND______XD__G_SPMINO14,									// entry name is "SPMINO14"
	KIND______XD__G_SPMINO15,									// entry name is "SPMINO15"
	KIND______XD__G_LOADICON,									// entry name is "LOADICON"
	KIND______XD__G_SAVEICON,									// entry name is "SAVEICON"
	KIND______XD__G_TELEICON,									// entry name is "TELEICON"
	KIND______XD__G_PAUSED,										// entry name is "PAUSED"
	KIND______XD__G_ADVISOR,									// entry name is "ADVISOR"
	//
	// STRIFE GRAPHIC NAMES
	//
	KIND________S_G_AMMNUM,										// entry name is "AMMNUM#"
	KIND________S_G_PLMNUM,										// entry name is "PLMNUM#"
	KIND________S_G_BRDR_L,										// entry name is "BRDR_L"
	KIND________S_G_BRDR_R,										// entry name is "BRDR_R"
	KIND________S_G_BRDR_B,										// entry name is "BRDR_B"
	KIND________S_G_BRDR_T,										// entry name is "BRDR_T"
	KIND________S_G_BRDR_BL,									// entry name is "BRDR_BL"
	KIND________S_G_BRDR_BR,									// entry name is "BRDR_BR"
	KIND________S_G_BRDR_TL,									// entry name is "BRDR_TL"
	KIND________S_G_BRDR_TR,									// entry name is "BRDR_TR"
	KIND________S_G_TITLEPIC,									// entry name is "TITLEPIC"
	KIND________S_G_VELLOGO,									// entry name is "VELLOGO"
	KIND________S_G_RGELOGO,									// entry name is "RGELOGO"
	KIND________S_G_CREDIT,										// entry name is "CREDIT"
	KIND________S_G_HELP0,										// entry name is "HELP0"
	KIND________S_G_HELP1,										// entry name is "HELP1"
	KIND________S_G_HELP2,										// entry name is "HELP2"
	KIND________S_G_HELP3,										// entry name is "HELP3"
	KIND________S_G_PANEL0,										// entry name is "PANEL0"
	KIND________S_G_PANEL1,										// entry name is "PANEL1"
	KIND________S_G_PANEL2,										// entry name is "PANEL2"
	KIND________S_G_PANEL3,										// entry name is "PANEL3"
	KIND________S_G_PANEL4,										// entry name is "PANEL4"
	KIND________S_G_PANEL5,										// entry name is "PANEL5"
	KIND________S_G_PANEL6,										// entry name is "PANEL6"
	KIND________S_G_PANEL7,										// entry name is "PANEL7"
	KIND________S_G_SS2F1,										// entry name is "SS2F1"
	KIND________S_G_SS2F2,										// entry name is "SS2F2"
	KIND________S_G_SS2F3,										// entry name is "SS2F3"
	KIND________S_G_SS2F4,										// entry name is "SS2F4"
	KIND________S_G_SS3F1,										// entry name is "SS3F1"
	KIND________S_G_SS3F2,										// entry name is "SS3F2"
	KIND________S_G_SS3F3,										// entry name is "SS3F3"
	KIND________S_G_SS3F4,										// entry name is "SS3F4"
	KIND________S_G_SS4F1,										// entry name is "SS4F1"
	KIND________S_G_SS4F2,										// entry name is "SS4F2"
	KIND________S_G_SS4F3,										// entry name is "SS4F3"
	KIND________S_G_SS4F4,										// entry name is "SS4F4"
	KIND________S_G_SS5F1,										// entry name is "SS5F1"
	KIND________S_G_SS5F2,										// entry name is "SS5F2"
	KIND________S_G_SS5F3,										// entry name is "SS5F3"
	KIND________S_G_SS6F1,										// entry name is "SS6F1"
	KIND________S_G_SS6F2,										// entry name is "SS6F2"
	KIND________S_G_SS6F3,										// entry name is "SS6F3"
	KIND________S_G_STBACK,										// entry name is "STBACK##"
	KIND________S_G_STDISK,										// entry name is "STDISK"
	KIND________S_G_STCOLOR1,									// entry name is "STCOLOR1"
	KIND________S_G_STCOLOR2,									// entry name is "STCOLOR2"
	KIND________S_G_STCOLOR3,									// entry name is "STCOLOR3"
	KIND________S_G_STCOLOR4,									// entry name is "STCOLOR4"
	KIND________S_G_STCOLOR5,									// entry name is "STCOLOR5"
	KIND________S_G_STCOLOR6,									// entry name is "STCOLOR6"
	KIND________S_G_STCOLOR7,									// entry name is "STCOLOR7"
	KIND________S_G_STCOLOR8,									// entry name is "STCOLOR8"
	KIND________S_G_STBFN0,										// entry name is "STBFN0##"
	KIND________S_G_STCFN0,										// entry name is "STCFN0##"
	KIND________S_G_INVBACK,									// entry name is "INVBACK"
	KIND________S_G_INVPBAK2,									// entry name is "INVPBAK2"
	KIND________S_G_INVPOP,										// entry name is "INVPOP"
	KIND________S_G_INVPOP2,									// entry name is "INVPOP2"
	KIND________S_G_INVTOP,										// entry name is "INVTOP"
	KIND________S_G_INVPBAK,									// entry name is "INVPBAK"
	KIND________S_G_INVCURS,									// entry name is "INVCURS"
	KIND________S_G_INVFONG0,									// entry name is "INVFONG0"
	KIND________S_G_INVFONG1,									// entry name is "INVFONG1"
	KIND________S_G_INVFONG2,									// entry name is "INVFONG2"
	KIND________S_G_INVFONG3,									// entry name is "INVFONG3"
	KIND________S_G_INVFONG4,									// entry name is "INVFONG4"
	KIND________S_G_INVFONG5,									// entry name is "INVFONG5"
	KIND________S_G_INVFONG6,									// entry name is "INVFONG6"
	KIND________S_G_INVFONG7,									// entry name is "INVFONG7"
	KIND________S_G_INVFONG8,									// entry name is "INVFONG8"
	KIND________S_G_INVFONG9,									// entry name is "INVFONG9"
	KIND________S_G_INVFONGP,									// entry name is "INVFONG%"
	KIND________S_G_INVFONY0,									// entry name is "INVFONY0"
	KIND________S_G_INVFONY1,									// entry name is "INVFONY1"
	KIND________S_G_INVFONY2,									// entry name is "INVFONY2"
	KIND________S_G_INVFONY3,									// entry name is "INVFONY3"
	KIND________S_G_INVFONY4,									// entry name is "INVFONY4"
	KIND________S_G_INVFONY5,									// entry name is "INVFONY5"
	KIND________S_G_INVFONY6,									// entry name is "INVFONY6"
	KIND________S_G_INVFONY7,									// entry name is "INVFONY7"
	KIND________S_G_INVFONY8,									// entry name is "INVFONY8"
	KIND________S_G_INVFONY9,									// entry name is "INVFONY9"
	KIND________S_G_INVFONYP,									// entry name is "INVFONY%"
	KIND________S_G_I_BKPK,										// entry name is "I_BKPK"
	KIND________S_G_I_ARM1,										// entry name is "I_ARM1"
	KIND________S_G_I_ARM2,										// entry name is "I_ARM2"
	KIND________S_G_I_BLIT,										// entry name is "I_BLIT"
	KIND________S_G_I_SHD1,										// entry name is "I_SHD1"
	KIND________S_G_I_STMP,										// entry name is "I_STMP"
	KIND________S_G_I_KY1G,										// entry name is "I_KY1G"
	KIND________S_G_I_KY2S,										// entry name is "I_KY2S"
	KIND________S_G_I_ROKT,										// entry name is "I_ROKT"
	KIND________S_G_I_KY3B,										// entry name is "I_KY3B"
	KIND________S_G_I_MDKT,										// entry name is "I_MDKT"
	KIND________S_G_I_BRY1,										// entry name is "I_BRY1"
	KIND________S_G_I_GRN1,										// entry name is "I_GRN1"
	KIND________S_G_I_GRN2,										// entry name is "I_GRN2"
	KIND________S_G_I_HAND,										// entry name is "I_HAND"
	KIND________S_G_I_ORAC,										// entry name is "I_ORAC"
	KIND________S_G_I_XQRL,										// entry name is "I_XQRL"
	KIND________S_G_I_PQRL,										// entry name is "I_PQRL"
	KIND________S_G_I_COIN,										// entry name is "I_COIN"
	KIND________S_G_I_TPAS,										// entry name is "I_TPAS"
	KIND________S_G_I_CRD1,										// entry name is "I_CRD1"
	KIND________S_G_I_CRD2,										// entry name is "I_CRD2"
	KIND________S_G_I_TARG,										// entry name is "I_TARG"
	KIND________S_G_I_CHST,										// entry name is "I_CHST"
	KIND________S_G_I_MASK,										// entry name is "I_MASK"
	KIND________S_G_I_BEAC,										// entry name is "I_BEAC"
	KIND________S_G_I_PRIS,										// entry name is "I_PRIS"
	KIND________S_G_I_PWR3,										// entry name is "I_PWR3"
	KIND________S_G_I_PWR1,										// entry name is "I_PWR1"
	KIND________S_G_I_PWR2,										// entry name is "I_PWR2"
	KIND________S_G_I_GYID,										// entry name is "I_GYID"
	KIND________S_G_I_FUSL,										// entry name is "I_FUSL"
	KIND________S_G_I_FUBR,										// entry name is "I_FUBR"
	KIND________S_G_I_GOID,										// entry name is "I_GOID"
	KIND________S_G_I_GUNT,										// entry name is "I_GUNT"
	KIND________S_G_I_HELT,										// entry name is "I_HELT"
	KIND________S_G_I_REPT,										// entry name is "I_REPT"
	KIND________S_G_I_FULL,										// entry name is "I_FULL"
	KIND________S_G_I_REBL,										// entry name is "I_REBL"
	KIND________S_G_I_WARE,										// entry name is "I_WARE"
	KIND________S_G_I_RELC,										// entry name is "I_RELC"
	KIND________S_G_I_COMM,										// entry name is "I_COMM"
	KIND________S_G_I_PMUP,										// entry name is "I_PMUP"
	KIND________S_G_I_UNIF,										// entry name is "I_UNIF"
	KIND________S_G_I_XPRK,										// entry name is "I_XPRK"
	KIND________S_G_I_GASG,										// entry name is "I_GASG"
	KIND________S_G_I_TOKN,										// entry name is "I_TOKN"
	KIND________S_G_I_BOAT,										// entry name is "I_BOAT"
	KIND________S_G_I_AQID,										// entry name is "I_AQID"
	KIND________S_G_I_PROC,										// entry name is "I_PROC"
	KIND________S_G_I_ANKH,										// entry name is "I_ANKH"
	KIND________S_G_I_RCRY,										// entry name is "I_RCRY"
	KIND________S_G_I_BCRY,										// entry name is "I_BCRY"
	KIND________S_G_I_CHAP,										// entry name is "I_CHAP"
	KIND________S_G_I_TUNL,										// entry name is "I_TUNL"
	KIND________S_G_I_BLTK,										// entry name is "I_BLTK"
	KIND________S_G_I_SGL1,										// entry name is "I_SGL1"
	KIND________S_G_I_SGL2,										// entry name is "I_SGL2"
	KIND________S_G_I_SGL3,										// entry name is "I_SGL3"
	KIND________S_G_I_SGL4,										// entry name is "I_SGL4"
	KIND________S_G_I_SGL5,										// entry name is "I_SGL5"
	KIND________S_G_I_BFLM,										// entry name is "I_BFLM"
	KIND________S_G_I_RING,										// entry name is "I_RING"
	KIND________S_G_I_EARS,										// entry name is "I_EARS"
	KIND________S_G_I_CRYS,										// entry name is "I_CRYS"
	KIND________S_G_I_OFIC,										// entry name is "I_OFIC"
	KIND________S_G_I_EPI1,										// entry name is "I_EPI1"
	KIND________S_G_I_EPI2,										// entry name is "I_EPI2"
	KIND________S_G_I_OTOK,										// entry name is "I_OTOK"
	KIND________S_G_I_RATT,										// entry name is "I_RATT"
	KIND________S_G_I_COUP,										// entry name is "I_COUP"
	KIND________S_G_I_SECK,										// entry name is "I_SECK"
	KIND________S_G_I_MINE,										// entry name is "I_MINE"
	KIND________S_G_M_PEAS1,									// entry name is "M_PEAS1"
	KIND________S_G_M_PEAS7,									// entry name is "M_PEAS7"
	KIND________S_G_M_LEDR0,									// entry name is "M_LEDR0"
	KIND________S_G_M_DERWIN,									// entry name is "M_DERWIN"
	KIND________S_G_M_DKWRKR,									// entry name is "M_DKWRKR"
	KIND________S_G_M_FALSP1,									// entry name is "M_FALSP1"
	KIND________S_G_M_FALSP2,									// entry name is "M_FALSP2"
	KIND________S_G_M_FALSP3,									// entry name is "M_FALSP3"
	KIND________S_G_M_GEOFF,									// entry name is "M_GEOFF"
	KIND________S_G_M_GUARD,									// entry name is "M_GUARD"
	KIND________S_G_M_GUARD2,									// entry name is "M_GUARD2"
	KIND________S_G_M_GUARD3,									// entry name is "M_GUARD3"
	KIND________S_G_M_GUARD4,									// entry name is "M_GUARD4"
	KIND________S_G_M_GUARD5,									// entry name is "M_GUARD5"
	KIND________S_G_M_GUARD6,									// entry name is "M_GUARD6"
	KIND________S_G_M_JUDGE,									// entry name is "M_JUDGE"
	KIND________S_G_M_KETRIC,									// entry name is "M_KETRIC"
	KIND________S_G_M_KEVIN,									// entry name is "M_KEVIN"
	KIND________S_G_M_MCGUFF,									// entry name is "M_MCGUFF"
	KIND________S_G_M_ORACLE,									// entry name is "M_ORACLE"
	KIND________S_G_M_PRISNR,									// entry name is "M_PRISNR"
	KIND________S_G_M_QUINCY,									// entry name is "M_QUINCY"
	KIND________S_G_M_RETRIC,									// entry name is "M_RETRIC"
	KIND________S_G_M_RICHTR,									// entry name is "M_RICHTR"
	KIND________S_G_M_SAMMIS,									// entry name is "M_SAMMIS"
	KIND________S_G_M_TEKBRN,									// entry name is "M_TEKBRN"
	KIND________S_G_M_TEKGRY,									// entry name is "M_TEKGRY"
	KIND________S_G_M_TEKTAN,									// entry name is "M_TEKTAN"
	KIND________S_G_M_WARDEN,									// entry name is "M_WARDEN"
	KIND________S_G_M_WERAN,									// entry name is "M_WERAN"
	KIND________S_G_M_WORNER,									// entry name is "M_WORNER"
	KIND________S_G_M_MENU,										// entry name is "M_*"
	/* ENDOF ************************************************
    * Graphic ordering not yet implemented (compiler limit) *
	********************************************************/
#endif
	//
	// END OF KINDS
	//
	KIND_PUTINLIST,												// entry is an ENTRY in a list
	KIND_ENDOFLIST,												// entry is the END marker of a list
	KIND_MAXPLUSONE												// highest kind value plus one
} kind_t;

/**********************************************************************************************************************************/
/****************************************************** Sort Order Selection ******************************************************/
/**********************************************************************************************************************************/

// Sorts
typedef enum
{
	SORT_NONE,													// sort order is: none
	SORT_LIST,													// sort order is: list
	SORT_ALPHABETIC,											// sort order is: list then alphabetic
	SORT_INTRINSIC,												// sort order is: list then intrinsic (use alphabetic if none)
	SORT_MAXPLUSONE												// number of sorts
} sort_t;

/**********************************************************************************************************************************/
/*********************************************************** Structures ***********************************************************/
/**********************************************************************************************************************************/

// Packed name
typedef char packed_name_t[8];

// Blockmap header
typedef struct
{
	UINT16 xorigin;
	UINT16 yorigin;
	UINT16 columns;
	UINT16 rows;
}
blockmap_header_t;

// Picture header
typedef struct
{
	SINT16 width;
	SINT16 height;
	SINT16 left_offset;
	SINT16 top_offset;
}
picture_header_t;

// Sound header (DMX)
typedef struct
{
	UINT16 magic;
	UINT16 sample_rate;
	UINT32 num_samples;
}
sound_header_t;

// Sound header (WAVE)
typedef struct
{
	char riff_magic[4];											// "RIFF"
	UINT32 data_size;											// lump size-8
}
wave_header_t;

#define WAVE_MAGIC_SIZE 4										// sizeof("WAVE" magic field in first WAVE data chunk)
#define WAVE_MAGIC_OFFSET 8										// offsetof("WAVE" magic field in first WAVE data chunk)

// Music header (MUS)
typedef struct
{
	char magic[4];												// 'M','U','S',0x1A
	SINT16 num_samples;											// number of bytes of music data
	SINT16 header_size;											// offset to start of music
	SINT16 num_primary;											// number of primary channels
	SINT16 num_secondary;										// number of secondary channels
	SINT16 num_patches;											// number of instrument patches
	SINT16 mustbezero;											// 0
}
music_header_t;

// Music header (MIDI)
typedef struct
{
	char magic[4];												// "MThd"
	UINT32 header_rest;											// 6 (size of header excluding this and the magic)
	UINT16 format;												// 0, 1, or 2 (MIDI format)
	UINT16 num_tracks;											// number of tracks
	UINT16 time_division;										// time division
}
midi_header_t;

#define MTRK_MAGIC_SIZE 4										// sizeof("MTrk" magic field in first MIDI track chunk)
#define MTRK_MAGIC_OFFSET 14									// offsetof("MTrk" magic field in first MIDI track chunk)

typedef struct
{
	SINT16 start_vertex;
	SINT16 end_vertex;
	SINT16 flags;
	SINT16 function;
	SINT16 tag;
	SINT16 left_sidedef;
	SINT16 right_sidedef;
} linedef_t;

/**********************************************************************************************************************************/
/******************************************************** Picture Metrics *********************************************************/
/**********************************************************************************************************************************/

#define GRAPHIC_PIXEL_RUN_HEADER_SIZE         2					// start row (255 means "end of column"), number of pixels
#define GRAPHIC_DUMMY_PIXEL_COUNT             2					// two zeros, one either size of actual pixel run pixels
#define GRAPHIC_PAGE_SIZE                 64000					// 320x200
#define GRAPHIC_PAGE_LAST (GRAPHIC_PAGE_SIZE-1)					// for logic simplification elsewhere
#define GRAPHIC_MAX_WIDTH            SINT16_MAX					// maximum short integer; DOOM theoretically has no limit on width
#define GRAPHIC_MAX_FIRST_ROW               254					// hard limit due to 255 used as "end-of-column" flag
#define GRAPHIC_MAX_PIXEL_RUN_LENGTH        255					// hard limit due to use of an unsigned byte
#define GRAPHIC_MAX_HEIGHT                  509					// hard limit due to 254 rows + one more run of 255 pixels
#define GRAPHIC_MAX_LEFTOFFSET              400					// picked these out of a hat...
#define GRAPHIC_MAX_TOPOFFSET               640					// picked these out of a hat...

/**********************************************************************************************************************************/
/************************************************************ Utility *************************************************************/
/**********************************************************************************************************************************/

// WildCard string match for WAD lump names
bool_t wcmatch(const char *str, const char *pat);

// String match for WAD lump names
bool_t wnmatch (const char *p, const char *q);

// Page (screen image) auto-detection
bool_t ispage (const void *lump, size_t length);

// Sound auto-detection
bool_t maybesound (const void *header, size_t lump_size);

// Music auto-detection
bool_t maybemusic (const void *header, size_t lump_size);

// Graphic auto-detection
bool_t maybegraphic (const void *header, size_t lump_size, FILE *infile, fofs_t lump_pos, fofs_t curr_pos);

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/

#endif // __WADDEF_H__
