#ifndef __WADDIR_H__
#define __WADDIR_H__

/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Wad directory

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include "services.h"
#include "waddef.h"
#include "wadlst.h"

/**********************************************************************************************************************************/
/******************************************************* WAD File Directory *******************************************************/
/**********************************************************************************************************************************/

// WAD file directory entry
typedef struct
{
	char name[9];												// name of entry
	fofs_t start;												// start of entry in file
	size_t length;												// length of entry
	type_t type;												// type of entry
	list_t list;												// list membership of entry
}
WADDIRentry_t;

// WAD file directory
typedef struct
{
	size_t limit;												// maximum number of entries
	size_t count;												// current number of entries
	bool_t reuse;												// flag for if lump reuse might be present
	WADDIRentry_t *entries;										// entries
}
WADDIR_t;

/**********************************************************************************************************************************/
/************************************************* WAD File Directory Maintanance *************************************************/
/**********************************************************************************************************************************/

// Initialise a WAD file directory
void WADInitDirectory (
	WADDIR_t **WADDIRp);										// WAD file directory to be initialised

// Deinitialise a WAD file directory
void WADDoneDirectory (
	WADDIR_t **WADDIRp);										// WAD file directory to be deinitialised

/**********************************************************************************************************************************/
/**************************************************** WAD File Directory Usage ****************************************************/
/**********************************************************************************************************************************/

// Metrics for WADCheckDirectoryEntryReuseCompatiblity
#define REUSE_DESCRIPTION_SIZE 255								// static buffer size that is GUARANTEED to be big enough for this

// Check reuse compatibility of two entries that share a lump
void WADCheckDirectoryEntryReuseCompatiblity (
	const WADDIR_t *WADDIR,										// WAD file directory to check in
	const LST_t *LST,											// symbol table resulting from the scan of that directory
	size_t index1,												// index of first entry to check
	size_t index2,												// index of second entry to check
	bool_t *compatible,											// flag for if they are compatible
	char *descript1,											// describes lump type of index 1 if incompatible (else "")
	char *descript2);											// describes lump type of index 2 if incompatible (else "")

// Read a WAD file directory into memory from a file
void WADReadDirectory (
	WADDIR_t *WADDIR,											// WAD file directory to read from file
	FILE *infile,												// file to read WAD file directory from
	size_t count);												// number of entries in WAD file directory

// Scan a WAD file directory and identify its entries
void WADScanDirectory (
	WADDIR_t *WADDIR,											// WAD file directory to scan
	LST_t *LST,													// symbol table resulting from the scan of that directory
	bool_t identify_names,										// identify screen images by name (READ MANUAL BEFORE USING!)
	bool_t identify_pages,										// identify screen images as pages (READ MANUAL BEFORE USING!)
	bool_t identify_graphics,									// identify screen images as graphics (READ MANUAL BEFORE USING!)
	bool_t identify_voices,										// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
	bool_t detect_sounds,										// detect sounds by content (READ MANUAL BEFORE USING!)
	bool_t detect_musics,										// detect musics by content (READ MANUAL BEFORE USING!)
	bool_t detect_graphics,										// detect graphics by content (READ MANUAL BEFORE USING!)
	bool_t recognised_names,									// identify sounds/musics/dialogs/conversations by recognised names
	bool_t loose_markers,										// allow nonstandard list marker characters
	bool_t named_markers,										// loose markers use names only (READ MANUAL BEFORE USING!)
	bool_t tolerate_multiples,									// do not treat multiple instances of structured lists as an error
	bool_t quiet_multiples,										// do not treat multiple instances of structured lists as a problem
	bool_t declassify_pnames,									// treat PNAMES as an unclassified lump (separate from TEXTURES)
	bool_t loose_headers,										// allow nonstandard map name headers (not just E\?M\? and MAP\?\?)
	bool_t quiet_headers,										// do not warn about non-empty map name headers
	game_t game,												// game for which (WAD whose lump names these are) was designed
	FILE *infile);												// file that directory was read from

// Copy a WAD file directory
void WADCopyDirectory (
	WADDIR_t *outWADDIR,										// WAD file directory that is the copy
	const WADDIR_t *inWADDIR);									// WAD file directory to copy

// Normalise a WAD file directory
void WADNormaliseDirectory (
	WADDIR_t *WADDIR,											// WAD file directory to normalise
	LST_t *LST,													// symbol table resulting from the scan of that directory
	bool_t remove_duplicates,									// remove all but the first of entries having the same name
	bool_t identify_voices,										// identify loose STRIFE VOICES by name (READ MANUAL BEFORE USING!)
	bool_t keep_platform,										// do not remove PLATFORM lump
	bool_t keep_wintex,											// do not remove _DEUTEX_ lump
	bool_t keep_history,										// do not remove HISTORY lump
	bool_t keep_tagdesc,										// do not remove TAGDESC lump
	bool_t keep_pcsfx,											// do not remove PC speaker sound effects
	bool_t keep_doubles,										// do not normalise double-letter list markers (e.g., PP_START)
	bool_t keep_borders,										// do not remove sub-list border markers (e.g., F1_START)
	bool_t keep_empties,										// do not remove empty structured lists
	bool_t remove_scripts,										// remove all SCRIPTS and SCRIPTnn entries
	bool_t force_removal);										// remove duplicate entries even if in different lists

// Sort a WAD file directory
void SortWadDirectory (
	WADDIR_t *WADDIR,											// WAD file directory to sort
	const LST_t *LST,											// symbol table resulting from the scan of that directory
	sort_t sort,												// sort order to apply to entries in the output file
	game_t game);												// game for which (WAD whose lump names these are) was designed

// Write a WAD file directory from memory out to a file
void WADWriteDirectory (
	const WADDIR_t *WADDIR,										// WAD file directory to write to file
	FILE *outfile);												// file to write directory to

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/

#endif // __WADDIR_H__
