/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Wad list parse stack

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <malloc.h>
#include <string.h>
#include "services.h"
#include "waddef.h"
#include "wadlps.h"

/**********************************************************************************************************************************/
/************************************************* List Parse Stack Maintainance **************************************************/
/**********************************************************************************************************************************/

#define LPS_APPORTION 2
#define LPS_INCREMENT 2

// Initialise a list parse stack
void LPSInit (
	LPS_t **LPSp)												// list parse stack to be initialised
{
	// VARIABLES
	size_t sizeb;
	LPS_t *objptr;
	LPSentry_t *entptr;

	// PRIMARY ERROR CHECK
	if (LPSp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise list parse stack with NULL argument pointer");
	}
	if (*LPSp!=NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise list parse stack with non-NULL argument");
	}

	// ALLOCATE MEMORY FOR THE LPS
	sizeb=sizeof(LPS_t);
	objptr=malloc(sizeb);
	if (objptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for list parse stack",sizeb);
	}
	(void) memset(objptr,0,sizeb);

	// ALLOCATE MEMORY FOR THE LPS ENTRIES
	sizeb=LPS_APPORTION*sizeof(LPSentry_t);
	entptr=malloc(sizeb);
	if (entptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for list parse stack entries",sizeb);
	}
	(void) memset(entptr,0,sizeb);

	// INITIALISE THE LPS STATIC FIELDS
	objptr->limit=LPS_APPORTION;
	objptr->count=0;
	objptr->entries=entptr;

	// RETURN RESULT
	*LPSp=objptr;
}

// Deinitialise a list parse stack
void LPSDone (
	LPS_t **LPSp)												// list parse stack to be deinitialised
{
	// PRIMARY ERROR CHECK
	if (LPSp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise list parse stack with NULL argument pointer");
	}
	if (*LPSp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise list parse stack with NULL argument");
	}
	if ((*LPSp)->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise list parse stack with NULL entries");
	}
#ifdef PARANOID
	if ((*LPSp)->count>(*LPSp)->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List parse stack is corrupt");
	}
#endif

	// DEALLOCATE MEMORY FOR THE LPS ENTRIES
	free((*LPSp)->entries);

	// DEALLOCATE MEMORY FOR THE LPS
	free(*LPSp);

	// RETURN RESULT
	*LPSp=NULL;
}

/**********************************************************************************************************************************/
/***************************************************** List Parse Stack Usage *****************************************************/
/**********************************************************************************************************************************/

// Push a list marker on a list parse stack
void LPSPush (
	LPS_t *LPS,													// list parse stack on which to put the marker
	list_t list,												// list identifier associated with marker
	bool_t start,												// start of list flag (FALSE means end of list) for marker
	bool_t corrected)											// corrected (e.g., PP_END -> P_END) flag
{
	// PRIMARY ERROR CHECK
	if (LPS==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to push on list parse stack with NULL argument");
	}
	if (LPS->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to push on list parse stack with NULL entries");
	}
#ifdef PARANOID
	if (LPS->count>LPS->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List parse stack is corrupt");
	}
#endif

	// REALLOCATE [IF NECESSARY] MEMORY FOR THE LPS ENTRIES
	if (LPS->count==LPS->limit)
	{
		LPSentry_t *newptr;
		size_t size,sizeb,incrb;
		/**********************/
		size=(LPS->limit)+LPS_INCREMENT;
		sizeb=size*sizeof(LPSentry_t);
		incrb=LPS_INCREMENT*sizeof(LPSentry_t);
		newptr=realloc(LPS->entries,sizeb);
		if (newptr==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not grow list parse stack entries by %ld bytes (from %ld bytes)",incrb,sizeb);
		}
		LPS->entries=newptr;
		(void) memset(&LPS->entries[LPS->limit],0,incrb);
		LPS->limit=size;
	}

	// PERFORM THE OPERATION
	LPS->entries[LPS->count].list=list;
	LPS->entries[LPS->count].start=start;
	LPS->entries[LPS->count].corrected=corrected;
	LPS->count++;
}

// Pop a list marker off a list parse stack
void LPSPop (
	LPS_t *LPS,													// list parse stack from which to pop the marker
	list_t *list,												// list identifier associated with marker
	bool_t *start,												// start of list flag (FALSE means end of list) for marker
	bool_t *corrected)											// corrected (e.g., PP_END -> P_END) flag
{
	// PRIMARY ERROR CHECK
	if (LPS==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to pop off list parse stack with NULL argument");
	}
	if (LPS->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to pop off list parse stack with NULL entries");
	}
#ifdef PARANOID
	if (LPS->count>LPS->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List parse stack is corrupt");
	}
#endif

	// SECONDARY ERROR CHECK
	if (LPS->count==0)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to pop off list parse stack that is empty");
	}

	// PERFORM THE OPERATION
	*list=LPS->entries[LPS->count-1].list;
	*start=LPS->entries[LPS->count-1].start;
	*corrected=LPS->entries[LPS->count-1].corrected;
	(void) memset(&LPS->entries[LPS->count-1],0,sizeof(LPSentry_t));
	LPS->count--;
}

// Get the depth of a list parse stack
size_t LPSGetStackDepth (
	const LPS_t *LPS)											// list parse stack to be queried
{
	// PRIMARY ERROR CHECK
	if (LPS==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get depth of list parse stack with NULL argument");
	}
	if (LPS->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get depth of list parse stack with NULL entries");
	}
#ifdef PARANOID
	if (LPS->count>LPS->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List parse stack is corrupt");
	}
#endif

	// PERFORM THE OPERATION
	return LPS->count;
}

// Get the list type on top of a list parse stack
void LPSGetStackTopList (
	const LPS_t *LPS,											// list parse stack from which to pop the marker
	list_t *list)												// list identifier associated with marker
{
	// PRIMARY ERROR CHECK
	if (LPS==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get top of a list parse stack with NULL argument");
	}
	if (LPS->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get top of a list parse stack with NULL entries");
	}
#ifdef PARANOID
	if (LPS->count>LPS->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List parse stack is corrupt");
	}
#endif

	// SECONDARY ERROR CHECK
	if (LPS->count==0)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get top of a list parse stack that is empty");
	}

	// PERFORM THE OPERATION
	*list=LPS->entries[LPS->count-1].list;
}

// Get details of top two on a list parse stack
void LPSGetStackTopTwoLists (
	const LPS_t *LPS,											// list parse stack to get details from
	list_t *curr_list,											// list on top
	bool_t *curr_corrected,										// corrected flag for lump name associated with list on top
	list_t *prev_list,											// list below top
	bool_t *prev_corrected)										// corrected flag for lump name associated with list below top
{
	// PRIMARY ERROR CHECK
	if (LPS==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get top two entries of a list parse stack with NULL argument");
	}
	if (LPS->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get top two entries of a list parse stack with NULL entries");
	}
#ifdef PARANOID
	if (LPS->count>LPS->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List parse stack is corrupt");
	}
#endif

	// SECONDARY ERROR CHECK
	if (LPS->count==0)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get top two entries of a list parse stack that is empty");
	}
	if (LPS->count==1)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get top two entries of a list parse stack that has only one entry");
	}

	// PERFORM THE OPERATION
	*curr_list=LPS->entries[LPS->count-1].list;
	*curr_corrected=LPS->entries[LPS->count-1].corrected;
	*prev_list=LPS->entries[LPS->count-2].list;
	*prev_corrected=LPS->entries[LPS->count-2].corrected;
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
