/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Wad lump reuse

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include "services.h"
#include "waddef.h"
#include "wadlst.h"
#include "waddir.h"
#include "wadlru.h"

/**********************************************************************************************************************************/
/********************************************************* WAD lump reuse *********************************************************/
/**********************************************************************************************************************************/

// Find a lump that is being reused for a given lump
static void WADFindSpecificReuse (
	const WADDIR_t *WADDIR,										// WAD file directory to search
	size_t wanted,												// entry to search for previous use of
	bool_t *found,												// flag for found such a lump
	size_t *index)												// index in directory of found lump
{
	// VARIABLES
	fofs_t estart;
	size_t i,elength;
	WADDIRentry_t *entries;

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to find specific lump reuse in WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to find specific lump reuse in WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (wanted>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check for identical lump with \"wanted\" (%ld) greater directory entry count (%ld)",wanted,WADDIR->count);
	}

	// INITIALISE
	*found=FALSE;
	*index=0;

	// GET POINTER TO WAD FILE DIRECTORY ENTRIES
	entries=WADDIR->entries;

	// GET START AND LENGTH OF LUMP BEING SEARCHED FOR
	estart=entries[wanted].start;
	elength=entries[wanted].length;

	// PERFORM FUNCTION
	for (i=0;i<wanted;i++)
	{
		if (
		     (estart==entries[i].start) &&
		     (elength==entries[i].length)
		   )
		{
			*found=TRUE;
			*index=i;
			return;
		}
	}
}

// Check reuse compatibility of two entries
static void WADValidateSpecificReuse (
	const WADDIR_t *WADDIR,										// WAD file directory to check in
	const LST_t *LST,											// symbol table resulting from scan of directory (for descriptions)
	size_t index1,												// index of first entry to check
	size_t index2,												// index of second entry to check
	bool_t *compatible,											// flag for if they are compatible
	char *descript1,											// describes lump type of index 1 if incompatible (else "")
	char *descript2)											// describes lump type of index 2 if incompatible (else "")
{
	// DOCUMENTATION
	/*
		If you only want the result, then supply LST
		and both of string return arguments as NULL.
	*/

	// VARIABLES
	list_t p,q;

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check directory entry reuse compatiblity in WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check directory entry reuse compatiblity in WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (index1>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Index value \"index1\" (%ld) is greater than directory entry count (%ld)",index1,WADDIR->count);
	}
	if (index2>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Index value \"index2\" (%ld) is greater than directory entry count (%ld)",index2,WADDIR->count);
	}
	if (!(
	      ( (LST==NULL) && (descript1==NULL) && (descript2==NULL) ) ||
	      ( (LST!=NULL) && (descript1!=NULL) && (descript2!=NULL) )
	   ) )
	{
		FatalAbort(1,__FILE__,__LINE__,"Parameters \"LST\", \"descript1\" and \"descript2\" should all be NULL or all be non-NULL");
	}

	// INITIALISE
	*compatible=FALSE;
	if (LST!=NULL)
	{
		descript1[0]=NUL;
		descript2[0]=NUL;
	}
	p=WADDIR->entries[index1].list;
	q=WADDIR->entries[index2].list;

	// PROHIBIT THE FOLLOWING EVEN IF IN THE SAME LIST
	if (
	     (p==LIST_MAPS)					||
	     (q==LIST_MAPS)					||
	     (p==LIST_TEXTURES)				||
	     (q==LIST_TEXTURES)				||
	     (p==LIST_PNAMES)				||
	     (q==LIST_PNAMES)				||
	     (p==LIST_LUMPS)				||
	     (q==LIST_LUMPS)
	   )
	{
		*compatible=FALSE;
		if (LST!=NULL)
		{
			(void) strcpy(descript1,LST->entries[p].name);
			(void) strcpy(descript2,LST->entries[q].name);
		}
		return;
	}

	// ALLOW ALL OTHER MATCHES IF IN SAME LIST
	if (p==q)
	{
		*compatible=TRUE;
		return;
	}

	//
	// SPECIAL CASES THAT CAN MATCH (LISTS ARE SIMILAR)
	//

	// GRAPHIC FONTS
	if (
	    (
	     (p==LIST_HERIANFONTA)  ||								// heretic/hexen font "A"
	     (p==LIST_HERIANFONTAY) ||								// hexen font "AY"
	     (p==LIST_HERIANFONTB)  ||								// heretic/hexen font "B"
	     (p==LIST_HERETICFONTA) ||								// heretic font "A"
	     (p==LIST_HERETICFONTB) ||								// heretic font "B"
	     (p==LIST_HEXENFONTA)   ||								// hexen font "A"
	     (p==LIST_HEXENFONTAY)  ||								// hexen font "AY"
	     (p==LIST_HEXENFONTB)									// hexen font "B"
	    )
	    &&
	    (
	     (q==LIST_HERIANFONTA)  ||								// heretic/hexen font "A"
	     (q==LIST_HERIANFONTAY) ||								// hexen font "AY"
	     (q==LIST_HERIANFONTB)  ||								// heretic/hexen font "B"
	     (q==LIST_HERETICFONTA) ||								// heretic font "A"
	     (q==LIST_HERETICFONTB) ||								// heretic font "B"
	     (q==LIST_HEXENFONTA)   ||								// hexen font "A"
	     (q==LIST_HEXENFONTAY)  ||								// hexen font "AY"
	     (q==LIST_HEXENFONTB)									// hexen font "B"
	    )
	   )
	{
		*compatible=TRUE;
		return;
	}

	// SPRITES
	if (
	    (
	     (p==LIST_SPRITES)      ||								// actor sprites
	     (p==LIST_SPRITESSUB1)  ||								// sprites sub-list #1
	     (p==LIST_SPRITESSUB2)  ||								// sprites sub-list #2
	     (p==LIST_SPRITESSUB3)									// sprites sub-list #3
	    )
	    &&
	    (
	     (q==LIST_SPRITES)      ||								// actor sprites
	     (q==LIST_SPRITESSUB1)  ||								// sprites sub-list #1
	     (q==LIST_SPRITESSUB2)  ||								// sprites sub-list #2
	     (q==LIST_SPRITESSUB3)									// sprites sub-list #3
	    )
	   )
	{
		*compatible=TRUE;
		return;
	}

	// PATCHES
	if (
	    (
	     (p==LIST_PATCHES)      ||								// wall patches (as listed in PNAMES)
	     (p==LIST_PATCHESSUB1)  ||								// patches sub-list #1
	     (p==LIST_PATCHESSUB2)  ||								// patches sub-list #2
	     (p==LIST_PATCHESSUB3)									// patches sub-list #3
	    )
	    &&
	    (
	     (q==LIST_PATCHES)      ||								// wall patches (as listed in PNAMES)
	     (q==LIST_PATCHESSUB1)  ||								// patches sub-list #1
	     (q==LIST_PATCHESSUB2)  ||								// patches sub-list #2
	     (q==LIST_PATCHESSUB3)									// patches sub-list #3
	    )
	   )
	{
		*compatible=TRUE;
		return;
	}

	// FLATS
	if (
	    (
	     (p==LIST_FLATS)      ||								// foors and ceilings
	     (p==LIST_FLATSSUB1)  ||								// flats sub-list #1
	     (p==LIST_FLATSSUB2)  ||								// flats sub-list #2
	     (p==LIST_FLATSSUB3)									// flats sub-list #3
	    )
	    &&
	    (
	     (q==LIST_FLATS)      ||								// foors and ceilings
	     (q==LIST_FLATSSUB1)  ||								// flats sub-list #1
	     (q==LIST_FLATSSUB2)  ||								// flats sub-list #2
	     (q==LIST_FLATSSUB3)									// flats sub-list #3
	    )
	   )
	{
		*compatible=TRUE;
		return;
	}

	// SOUNDS AND VOICES
	if (
	     (
	      (WADDIR->entries[index1].list==LIST_VOICES) &&
	      (WADDIR->entries[index1].type==TYPE_VOICE)  &&
	      (WADDIR->entries[index2].list==LIST_SOUNDS) &&
	      (WADDIR->entries[index2].type==TYPE_VOICE)
	     )
	    ||
	     (
	      (WADDIR->entries[index1].list==LIST_SOUNDS) &&
	      (WADDIR->entries[index1].type==TYPE_VOICE)  &&
	      (WADDIR->entries[index2].list==LIST_VOICES) &&
	      (WADDIR->entries[index2].type==TYPE_VOICE)
	     )
	   )
	{
		*compatible=TRUE;
		return;
	}

	// TOO DIFFERENT
	*compatible=FALSE;
	if (LST!=NULL)
	{
		(void) strcpy(descript1,LST->entries[p].name);
		(void) strcpy(descript2,LST->entries[q].name);
	}
	return;
}

// Check if there is a lump identical to this one
static void WADFindCopyToReuse (
	FILE *outfile,												// WAD file to search in
	const WADDIR_t *WADDIR,										// WAD file directory to search
	const void *lump_wanted,									// lump to check for identical copy of
	size_t wanted,												// index in WAD file directory of that lump
	void (canonicalize) (										// lump canonicalization function
		const void *lump_in,									// lump to process
		const WADDIR_t *WADDIR,									// WAD file directory to take entry metrics from
		size_t index,											// index of lump in directory entries
		bool_t truncate_waves,									// remove wasted space at the end of sound resources
		bool_t rebuild_blockmaps,								// rebuild (no pack/unpack) blockmaps to remove wasted space
		bool_t pack_blockmaps,									// pack blockmaps to remove current and potential wasted space
		bool_t unpack_blockmaps,								// unpack blockmaps into normalised (but inefficient) format
		bool_t flush_blockmaps,									// write out modified blockmaps even if the size has not changed
		bool_t rebuild_pictures,								// rebuild (no pack/unpack) pictures to remove wasted space
		bool_t pack_pictures,									// pack pictures to remove current and potential wasted space
		bool_t unpack_pictures,									// unpack pictures into normalised (but inefficient) format
		bool_t flush_pictures,									// write out modified pictures even if the size has not changed
		bool_t maintain_blockmaps,								// maintain nonstandard blockmaps (preserve non-optimal blocks)
		bool_t maintain_pictures,								// maintain nonstandard pictures (preserve non-optimal columns)
		bool_t process_only,									// issue no messages except for fatal errors
		void **lump_out,										// processed lump
		size_t *size_out),										// size of processed lump
	bool_t truncate_waves,										// remove wasted space at the end of sound resources
	bool_t rebuild_blockmaps,									// rebuild (no pack/unpack) blockmaps to remove wasted space
	bool_t pack_blockmaps,										// pack blockmaps to remove current and potential wasted space
	bool_t unpack_blockmaps,									// unpack blockmaps into normalised (but inefficient) format
	bool_t flush_blockmaps,										// write out modified blockmaps even if the size has not changed
	bool_t rebuild_pictures,									// rebuild (no pack/unpack) pictures to remove wasted space
	bool_t pack_pictures,										// pack pictures to remove current and potential wasted space
	bool_t unpack_pictures,										// unpack pictures into normalised (but inefficient) format
	bool_t flush_pictures,										// write out modified pictures even if the size has not changed
	bool_t maintain_blockmaps,									// maintain nonstandard blockmaps (preserve non-optimal blocks)
	bool_t maintain_pictures,									// maintain nonstandard pictures (preserve non-optimal columns)
	bool_t process_only,										// issue no messages except for fatal errors
	bool_t *found,												// flag for found such a lump
	size_t *index)												// index in directory of found lump
{
	// VARIABLES
	fofs_t fpos;
	size_t i,temp_size_wanted;
	void *temp_lump_wanted;
	WADDIRentry_t *entries;
	bool_t already_canonical;

	// INITIALISE
	*found=FALSE;
	*index=0;
	temp_lump_wanted=NULL;
	already_canonical=FALSE;

	// PRIMARY ERROR CHECK
	if (outfile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to find specific lump reuse in WAD file directory with NULL output file argument");
	}
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to find specific lump reuse in WAD file directory with NULL directory argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to find specific lump reuse in WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (lump_wanted==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify new lump reuse with NULL lump argument");
	}
	if (wanted>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to find specific lump reuse with \"wanted\" (%ld) greater than directory entry count (%ld)",wanted,WADDIR->count);
	}

	// THERE CANNOT BE A PREVIOUS ENTRY TO ENTRY ZERO
	if (wanted==0)
	{
		return;
	}

	//	REUSE MAKES NO SENSE IF LUMP HAS ZERO SIZE
	if (WADDIR->entries[wanted].length==0)
	{
		return;
	}

	// GET POINTER TO WAD FILE DIRECTORY ENTRIES
	entries=WADDIR->entries;

	// SAVE INITIAL FILE POSITION
	fpos=ftell(outfile);
	if (fpos<0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not tell position in file");
	}

	// DETERMINE IF ALREADY CANONICAL
	if ( (rebuild_blockmaps || pack_blockmaps || unpack_blockmaps ) && (entries[wanted].type==TYPE_MAPDATA) && (strcmp(entries[wanted].name,"BLOCKMAP")==0) )
	{
		already_canonical=((pack_blockmaps || unpack_blockmaps) && flush_blockmaps)?TRUE:FALSE;
	}
	else if ( (rebuild_pictures || pack_pictures || unpack_pictures ) && (entries[wanted].type==TYPE_GRAPHIC) )
	{
		already_canonical=((pack_pictures || unpack_pictures) && flush_pictures)?TRUE:FALSE;
	}
	else if ( truncate_waves && ((entries[wanted].type==TYPE_SOUND) || (entries[wanted].type==TYPE_VOICE)) )
	{
		already_canonical=(truncate_waves)?TRUE:FALSE;
	}
	else
	{
		already_canonical=TRUE;
	}

	// CREATE CANONICALIZED COPY OF WANTED LUMP
	if (!already_canonical)
	{
		canonicalize (
			lump_wanted,										// lump to process
			WADDIR,												// WAD file directory to take entry metrics from
			wanted,												// index of lump in directory entries
			/*truncate_waves=*/TRUE,							// remove wasted space at the end of sound resources
			/*rebuild_blockmaps=*/FALSE,						// rebuild (no pack/unpack) blockmaps to remove wasted space
			/*pack_blockmaps=*/FALSE,							// pack blockmaps to remove current and potential wasted space
			/*unpack_blockmaps=*/TRUE,							// unpack blockmaps into normalised (but inefficient) format
			/*flush_blockmaps=*/TRUE,							// write out modified blockmaps even if the size has not changed
			/*rebuild_pictures=*/FALSE,							// rebuild (no pack/unpack) pictures to remove wasted space
			/*pack_pictures=*/FALSE,							// pack pictures to remove current and potential wasted space
			/*unpack_pictures=*/TRUE,							// unpack pictures into normalised (but inefficient) format
			/*flush_pictures=*/TRUE,							// write out modified pictures even if the size has not changed
			maintain_blockmaps,									// maintain nonstandard blockmaps (preserve non-optimal blocks)
			maintain_pictures,									// maintain nonstandard pictures (preserve non-optimal columns)
			/*process_only=*/TRUE,								// issue no messages except for fatal errors
			&temp_lump_wanted,									// processed lump
			&temp_size_wanted);									// size of processed lump
	}

	// SEARCH FOR A COPY OF THE WANTED LUMP
	for (i=0;i<wanted;i++)
	{
		// VARIABLES
		bool_t same,compatible_entries;
		void *lump_found,*temp_lump_found;
		size_t temp_size_found;

		// INITIALISE
		lump_found=NULL;
		temp_lump_found=NULL;

		// IF ZERO SIZE THEN NOT ELIGIBLE ANYWAY
		if (entries[i].length==0)
		{
			continue;
		}

		// NO MATCH IF ALREADY CANONICAL BUT NOT SAME SIZE
		if (already_canonical && (entries[i].length!=entries[wanted].length))
		{
			continue;
		}

		// CHECK THE POTENTIAL MATCH FOR COMPATIBILITY
		WADValidateSpecificReuse (
			WADDIR,												// WAD file directory to check in
			NULL,												// symbol table resulting from scan of directory (for descriptions)
			wanted,												// index of current entry to check
			i,													// index of previous entry to check
			&compatible_entries,								// flag for if they are compatible
			NULL,												// describes lump type of current entry if incompatible (else "")
			NULL);												// describes lump type of previous entry if incompatible (else "")

		// IF NOT COMPATIBLE THEN NO MATCH
		if (!compatible_entries)
		{
			continue;
		}

		// SEEK TO POSITION OF POTENTIALLY MATCHING LUMP
		if (fseek(outfile,entries[i].start,SEEK_SET)!=0)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not seek in file");
		}

		// ALLOCATE MEMORY FOR POTENTIALLY MATCHING LUMP
		lump_found=malloc(entries[i].length);
		if (lump_found==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %d bytes for lump buffer",entries[i].length);
		}

		// READ IN FOUND LUMP
		if (fread(lump_found,entries[i].length,1,outfile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
		}

		// CREATE CANONICALIZED COPY OF FOUND LUMP
		if (!already_canonical)
		{
			canonicalize (
				lump_found,										// lump to process
				WADDIR,											// WAD file directory to take entry metrics from
				i,												// index of lump in directory entries
				/*truncate_waves=*/TRUE,						// remove wasted space at the end of sound resources
				/*rebuild_blockmaps=*/FALSE,					// rebuild (no pack/unpack) blockmaps to remove wasted space
				/*pack_blockmaps=*/FALSE,						// pack blockmaps to remove current and potential wasted space
				/*unpack_blockmaps=*/TRUE,						// unpack blockmaps into normalised (but inefficient) format
				/*flush_blockmaps=*/TRUE,						// write out modified blockmaps even if the size has not changed
				/*rebuild_pictures=*/FALSE,						// rebuild (no pack/unpack) pictures to remove wasted space
				/*pack_pictures=*/FALSE,						// pack pictures to remove current and potential wasted space
				/*unpack_pictures=*/TRUE,						// unpack pictures into normalised (but inefficient) format
				/*flush_pictures=*/TRUE,						// write out modified pictures even if the size has not changed
				maintain_blockmaps,								// maintain nonstandard blockmaps (preserve non-optimal blocks)
				maintain_pictures,								// maintain nonstandard pictures (preserve non-optimal columns)
				/*process_only=*/TRUE,							// issue no messages except for fatal errors
				&temp_lump_found,								// processed lump
				&temp_size_found);								// size of processed lump
		}

		// COMPARE THE LUMPS
		same=(mamatch(
			(temp_lump_wanted!=NULL)?temp_lump_wanted:lump_wanted,
			(temp_lump_wanted!=NULL)?temp_size_wanted:entries[wanted].length,
			(temp_lump_found!=NULL)?temp_lump_found:lump_found,
			(temp_lump_found!=NULL)?temp_size_found:entries[i].length
			))?TRUE:FALSE;

		// DEALLOCATE MEMORY FOR THE FOUND LUMP
		free(lump_found);

		// DEALLOCATE MEMORY FOR THE CANONICALIZED FOUND LUMP
		if (temp_lump_found!=NULL)
		{
			free(temp_lump_found);
		}

		// IF LUMPS ARE IDENTICAL THEN FOUND
		if (same)
		{
			*found=TRUE;
			*index=i;
			break;
		}
	}

	// FREE CANONICALIZED COPY OF WANTED LUMP
	if (temp_lump_wanted!=NULL)
	{
		free(temp_lump_wanted);
	}

	// RESTORE INITIAL FILE POSITION
	if (fseek(outfile,fpos,SEEK_SET)!=0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not seek in file");
	}
}

// Check for lumps being reused between entries
void WADCheckForAnyReuse(
	const WADDIR_t *WADDIR,										// WAD file directory that was scanned
	bool_t *found_reuse)										// flag for some lump reuse was found
{
	// VARIABLES
	size_t i,count;

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check for lump reuse in WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check for lump reuse in WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif

	// INITIALISE
	*found_reuse=FALSE;
	count=WADDIR->count;

	// PERFORM FUNCTION
	for (i=0;i<count;i++)
	{
		if (WADDIR->entries[i].length!=0)
		{
			size_t j;
			/*******/
			for (j=0;j<i;j++)
			{
				if (
				     (WADDIR->entries[i].start==WADDIR->entries[j].start) &&
				     (WADDIR->entries[i].length==WADDIR->entries[j].length)
				   )
				{
					*found_reuse=TRUE;
					return;
				}
			}
		}
	}
}

// Handle (on input) existing reuse of a list of lumps
void WADHandleExistingReuseOnInput (
	const WADDIR_t *WADDIR,										// WAD file directory that was scanned
	const LST_t *LST,											// symbol table resulting from the scan of that directory
	bool_t reuse_lumps,											// reuse lumps where directory entries specify same size and offset
	bool_t pack_lumps,											// share lumps with identical contents between directory entries
	bool_t unpack_lumps,										// output each lump explicitly even if it could be shared or reused
	bool_t tolerate_links,										// do not treat lump reuse as an error
	bool_t quiet_links,											// do not treat lump reuse as a problem
	bool_t tolerate_conflicts,									// do not treat ineligible lump reuse as an error
	bool_t quiet_conflicts,										// do not treat ineligible lump reuse as a problem
	verbosity_t priority,										// priority selection (must be WARNINGS or ERRORS)
	bool_t *found_errors)										// report if any errors were found (ignored if processing WARNINGS)
{
	// DOCUMENTATION
	/*
		If any situation should produce a warning AND an error
		then report them together during the ERRORS pass.
	*/

	// VARIABLES
	size_t count,current;

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to handle existing lump reuse in WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to handle existing lump reuse in WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif

	// SECONDARY ERROR CHECK
	switch (priority)
	{
		case VERBOSITY_STATEMENTS:
			FatalAbort(1,__FILE__,__LINE__,"Priority selection STATEMENTS invalid for reuse diagnostics");
			break;
		case VERBOSITY_ERRORS:
			break;
		case VERBOSITY_WARNINGS:
			break;
		case VERBOSITY_CORRECTIONS:
			FatalAbort(1,__FILE__,__LINE__,"Priority selection CORRECTIONS invalid for reuse diagnostics");
			break;
		case VERBOSITY_PROGRESS:
			FatalAbort(1,__FILE__,__LINE__,"Priority selection PROGRESS invalid for reuse diagnostics");
			break;
		case VERBOSITY_ACTIONS:
			FatalAbort(1,__FILE__,__LINE__,"Priority selection ACTIONS invalid for reuse diagnostics");
			break;
		case VERBOSITY_DETAILS:
			FatalAbort(1,__FILE__,__LINE__,"Priority selection DETAILS invalid for reuse diagnostics");
			break;
		case VERBOSITY_INTERNALS:
			FatalAbort(1,__FILE__,__LINE__,"Priority selection INTERNALS invalid for reuse diagnostics");
			break;
		default:
			FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %d in variable \"priority\"",priority);
			break;
		////
	}

	// INITIALISE
	count=WADDIR->count;
	*found_errors=FALSE;

	// ISSUE DIAGNOSTICS FOR ANY LUMPS THAT ARE BEING REUSED
	for (current=0;current<count;current++)
	{
		// VARIABLES
		size_t previous;
		char *enamec,*enamep;

		// IGNORE IF ZERO SIZE
		if (WADDIR->entries[current].length==0)
		{
			// continue OUTER for (current)
			continue;
		}

		// GET NAME METRICS FOR CURRENT LUMP
		enamec=WADDIR->entries[current].name;

		// FIND THE FIRST MATCH IF ANY
		for (previous=0;previous<current;previous++)
		{
			// VARIABLES
			bool_t expected_reuse,compatible_entries;
			char descriptc[REUSE_DESCRIPTION_SIZE+1];
			char descriptp[REUSE_DESCRIPTION_SIZE+1];

			// IGNORE IF NO MATCH
			if (
			     (WADDIR->entries[current].start!=WADDIR->entries[previous].start) ||
			     (WADDIR->entries[current].length!=WADDIR->entries[previous].length)
			   )
			{
				// continue INNER for (previous)
				continue;
			}

			// GET NAME METRICS FOR MATCHING (PREVIOUS) LUMP
			enamep=WADDIR->entries[previous].name;

			// GET UNEXPECTED REUSE FLAG
			expected_reuse=(reuse_lumps || pack_lumps || unpack_lumps)?TRUE:FALSE;

			// GET FLAG FOR IF THEIR DIRECTORY ENTRES ARE
			// COMPATIBLE AS REGARDS ALLOWING LUMP REUSE.
			WADValidateSpecificReuse (
				WADDIR,											// WAD file directory to check in
				LST,											// symbol table resulting from scan of directory (for descriptions)
				current,										// index of first entry to check
				previous,										// index of second entry to check
				&compatible_entries,							// flag for if they are compatible
				descriptc,										// describes lump type of index 1 if incompatible (else "")
				descriptp);										// describes lump type of index 2 if incompatible (else "")
			// HANDLE REUSED ENTRIES
			if (compatible_entries)
			{//compatible entries
				if (expected_reuse)
				{//compatible entries && expected reuse
					// NOW:QUIET
					// WRT:MESSG(ASPEROPTIONS)
				}
				else
				{//!expected_reuse
					if (tolerate_links || quiet_links)
					{//compatible entries && !expected reuse && (tolerate_links || quiet_links)
						// NOW:MESSG(WARN_UNEXPECTED)
						// WRT:MESSG(WARN_BEINGUNPACKED)
						if (priority==VERBOSITY_WARNINGS)
						{
							if (quiet_links)
							{
								// Suppress the warning
							}
							else
							{
								EventState(VERBOSITY_WARNINGS,"Unexpected lump reuse of %s (entry %ld) for %s (entry %ld)",enamep,previous,enamec,current);
							}
						}
					}
					else
					{//compatible entries && !expected reuse && !(tolerate_links || quiet_links)
						// NOW:MESSG(ERROR_UNEXPECTED)
						// WRT:ABORT
						if (priority==VERBOSITY_ERRORS)
						{
							*found_errors=TRUE;
							if (quiet_links)
							{
								FatalAbort(1,__FILE__,__LINE__,"Contradictory lump reuse handling options should have been trapped by now");
							}
							else
							{
								EventState(VERBOSITY_ERRORS,"Unexpected lump reuse of %s (entry %ld) for %s (entry %ld)",enamep,previous,enamec,current);
							}
						}
					}
				}
			}
			else
			{//!compatible entries
				if (expected_reuse)
				{//!compatible entries && expected reuse
					if (tolerate_conflicts || quiet_conflicts)
					{//!compatible entries && expected reuse && (tolerate_conflicts || quiet_conflicts)
						// NOW:MESSG(WARN: INELIGIBLE)
						// WRT:MESSG(FUNC: AS PER OPTIONS)
						if (priority==VERBOSITY_WARNINGS)
						{
							if (quiet_conflicts)
							{
								// Suppress the warning
							}
							else
							{
								EventState(VERBOSITY_WARNINGS,"Ineligible lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
							}
						}
					}
					else
					{//!compatible entries && expected reuse && !(tolerate_conflicts || quiet_conflicts)
						// NOW:MESSG(ERROR: INELIGIBLE)
						// WRT:ABORT
						if (priority==VERBOSITY_ERRORS)
						{
							*found_errors=TRUE;
							if (quiet_conflicts)
							{
								FatalAbort(1,__FILE__,__LINE__,"Contradictory lump reuse handling options should have been trapped by now");
							}
							else
							{
								EventState(VERBOSITY_ERRORS,"Ineligible lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
							}
						}
					}
				}
				else
				{//!expected_reuse
					if (tolerate_links || quiet_links)
					{//!compatible entries && !expected reuse && (tolerate_links || quiet_links)
						if (tolerate_conflicts || quiet_conflicts)
						{//!compatible entries && !expected reuse && (tolerate_links || quiet_links) && (tolerate_conflicts || quiet_conflicts)
							// NOW:MESSG(WARN: UNEXPECTED, WARN: INELIGIBLE)
							// WRT:MESSG(WARN: WILL BE UNPACKED)
							if (priority==VERBOSITY_WARNINGS)
							{
								if      (!quiet_links && !quiet_conflicts)
								{
									EventState(VERBOSITY_WARNINGS,"Unexpected and ineligible lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
								}
								else if (!quiet_links &&  quiet_conflicts)
								{
									EventState(VERBOSITY_WARNINGS,"Unexpected lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
								}
								else if ( quiet_links && !quiet_conflicts)
								{
									EventState(VERBOSITY_WARNINGS,"Ineligible lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
								}
								else if ( quiet_links &&  quiet_conflicts)
								{
									// Suppress the warning
								}
								else
								{
									// Should never get here
									FatalAbort(1,__FILE__,__LINE__,"Logic failure when handling lump reuse (was the preceding code modified?)");
								}
							}
						}
						else
						{//!compatible entries && !expected reuse && (tolerate_links || quiet_links) && !(tolerate_conflicts || quiet_conflicts)
							// NOW:MESSG(WARN: UNEXPECTED, ERROR: INELIGIBLE)
							// WRT:ABORT;
							if (priority==VERBOSITY_ERRORS)
							{
								*found_errors=TRUE;
								if      (!quiet_links && !quiet_conflicts)
								{
									EventState(VERBOSITY_ERRORS,"[Unexpected and] ineligible lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
								}
								else if (!quiet_links &&  quiet_conflicts)
								{
									FatalAbort(1,__FILE__,__LINE__,"Contradictory lump reuse handling options should have been trapped by now");
								}
								else if ( quiet_links && !quiet_conflicts)
								{
									EventState(VERBOSITY_ERRORS,"Ineligible lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
								}
								else if ( quiet_links &&  quiet_conflicts)
								{
									FatalAbort(1,__FILE__,__LINE__,"Contradictory lump reuse handling options should have been trapped by now");
								}
								else
								{
									// Should never get here
									FatalAbort(1,__FILE__,__LINE__,"Logic failure when handling lump reuse (was the preceding code modified?)");
								}
							}
						}
					}
					else
					{//!compatible entries && !expected reuse && !(tolerate_links || quiet_links)
						if (tolerate_conflicts || quiet_conflicts)
						{//!compatible entries && !expected reuse && !(tolerate_links || quiet_links) && (tolerate_conflicts || quiet_conflicts)
							// NOW:MESSG(ERROR: UNEXPECTED, WARN: INELIGIBLE)
							// WRT:ABORT;
							if (priority==VERBOSITY_ERRORS)
							{
								*found_errors=TRUE;
								if      (!quiet_links && !quiet_conflicts)
								{
									EventState(VERBOSITY_ERRORS,"Unexpected [and ineligible] lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
								}
								else if (!quiet_links &&  quiet_conflicts)
								{
									EventState(VERBOSITY_ERRORS,"Unexpected lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
								}
								else if ( quiet_links && !quiet_conflicts)
								{
									FatalAbort(1,__FILE__,__LINE__,"Contradictory lump reuse handling options should have been trapped by now");
								}
								else if ( quiet_links &&  quiet_conflicts)
								{
									FatalAbort(1,__FILE__,__LINE__,"Contradictory lump reuse handling options should have been trapped by now");
								}
								else
								{
									// Should never get here
									FatalAbort(1,__FILE__,__LINE__,"Logic failure when handling lump reuse (was the preceding code modified?)");
								}
							}
						}
						else
						{//!compatible entries && !expected reuse && !(tolerate_links || quiet_links) && !(tolerate_conflicts || quiet_conflicts)
							// NOW:MESSG(ERROR: UNEXPECTED, ERROR: INELIGIBLE)
							// WRT:ABORT;
							if (priority==VERBOSITY_ERRORS)
							{
								*found_errors=TRUE;
								if      (!quiet_links && !quiet_conflicts)
								{
									EventState(VERBOSITY_ERRORS,"Unexpected and ineligible lump reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,previous,descriptc,enamec,current);
								}
								else if (!quiet_links &&  quiet_conflicts)
								{
									FatalAbort(1,__FILE__,__LINE__,"Contradictory lump reuse handling options should have been trapped by now");
								}
								else if ( quiet_links && !quiet_conflicts)
								{
									FatalAbort(1,__FILE__,__LINE__,"Contradictory lump reuse handling options should have been trapped by now");
								}
								else if ( quiet_links &&  quiet_conflicts)
								{
									FatalAbort(1,__FILE__,__LINE__,"Contradictory lump reuse handling options should have been trapped by now");
								}
								else
								{
									// Should never get here
									FatalAbort(1,__FILE__,__LINE__,"Logic failure when handling lump reuse (was the preceding code modified?)");
								}
							}
						}
					}
				}
			}

			// FIRST MATCH IS MASTER COPY:
			// BREAK INNER FOR (PREVIOUS).
			break;
		}
	}
}

// Handle (on output) existing reuse of a given lump
void WADHandleExistingReuseOnOutput (
	const WADDIR_t *WADDIR,										// WAD file directory to check for reuse in
	const LST_t *LST,											// symbol table resulting from the scan of that directory
	bool_t reuse_lumps,											// reuse lumps where directory entries specify same size and offset
	bool_t pack_lumps,											// share lumps with identical contents between directory entries
	bool_t unpack_lumps,										// output each lump explicitly even if it could be shared or reused
	bool_t unpack_conflicts,									// do not preserve ineligible lump reuse (if tolerated)
	size_t wanted,												// index of entry to check for reuse of
	bool_t *reused)												// flag for this lump reuses a previous directory entry
{
	// DOCUMENTATION
	/*
		Check for a previously-written lump with the same size
		and file offset as this one and if there is one then
		flag that it can be reused. However, do not bother if
		this has zero size, as it would not be written out.
		The supplied WAD file directory should be copy of the
		ORIGINAL directory because the first "occurence" of
		the wanted lump may have been modified. This is safe
		because the modifications that CleanWAD performs, such
		as BLOCKMAP packing, are deterministic; that is to say
		that given the same input, the output is identical.
		Also issue any appropriate diagnostics for the write.
	*/

	// VARIABLES
	bool_t found;												// flag for found previous usage
	size_t index;												// index in directory of found usage
	char *enamec;												// pointer to name of current (wanted) entry
	char *enamep;												// pointer to name of previous (found) entry
	verbosity_t priority;										// priority of messages to be issued about this
	bool_t compatible_entries;									// flag for entries are compatible for reuse
	char descriptc[REUSE_DESCRIPTION_SIZE+1];					// description of current (wanted) entry
	char descriptp[REUSE_DESCRIPTION_SIZE+1];					// description of previous (found) entry

	// PRIMARY ERROR CHECK
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to handle existing reuse of lump in WAD file directory with NULL argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to handle existing reuse of lump in WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (wanted>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to handle existing reuse of lump with \"wanted\" (%ld) greater than directory entry count (%ld)",wanted,WADDIR->count);
	}

	// THERE CANNOT BE A PREVIOUS ENTRY TO ENTRY ZERO
	if (wanted==0)
	{
		*reused=FALSE;
		return;
	}

	//	REUSE MAKES NO SENSE IF LUMP HAS ZERO SIZE
	if (WADDIR->entries[wanted].length==0)
	{
		*reused=FALSE;
		return;
	}

	// FIND A LUMP THAT IS REUSABLE AS THIS ONE
	WADFindSpecificReuse (
		WADDIR,													// WAD file directory to search
		wanted,													// entry to search for previous use of
		&found,													// flag for found such a lump
		&index);												// index in directory of found lump

	//	IF NONE FOUND THEN RETURN IMMEDIATELY
	if (!found)
	{
		*reused=FALSE;
		return;
	}

	// IF DEFINITELY REUSING OR PACKING THEN LEAVE THIS ALONE
	if (!unpack_conflicts)
	{
		if (reuse_lumps || pack_lumps)
		{
			*reused=TRUE;
			return;
		}
	}

	// DETERMINE IF THIS REUSE IS VALID
	WADValidateSpecificReuse (
		WADDIR,													// WAD file directory to check in
		LST,													// symbol table resulting from scan of directory (for descriptions)
		wanted,													// index of current entry to check
		index,													// index of previous entry to check
		&compatible_entries,									// flag for entries are compatible for reuse
		descriptc,												// describes lump type of current entry if incompatible (else "")
		descriptp);												// describes lump type of previous entry if incompatible (else "")

	// IF ACTUALLY REUSING OR PACKING THEN LEAVE THIS ALONE
	if (!unpack_conflicts || compatible_entries)
	{
		if (reuse_lumps || pack_lumps)
		{
			*reused=TRUE;
			return;
		}
	}

	// WE MUST LATER UNPACK THIS REUSE
	//
	// This means we were either
	//
	// (a) asked to unpack any existing reuse,
	// (b) asked nothing special (which is PHYSICALLY
	//     equivalent to asking to unpack existing reuse),
	// (c) asked to unpack ineligible lump reuse (which
	//     overrides the generic reuse and unpack options if
	//     the reuse concerned is ineligible),
	//
	// so warn that we will destroy an existing reuse.
	//
	enamec=WADDIR->entries[wanted].name;
	enamep=WADDIR->entries[index].name;
	priority=((unpack_lumps || (unpack_conflicts && !compatible_entries))?VERBOSITY_DETAILS:VERBOSITY_WARNINGS);
	if (!compatible_entries)
	{
		EventState(priority,"Removing reuse of [%s] %s (entry %ld) for [%s] %s (entry %ld)",descriptp,enamep,index,descriptc,enamec,wanted);
	}
	else
	{
		EventState(priority,"Removing reuse of lump %s (entry %ld) for lump %s (entry %ld)",enamep,index,enamec,wanted);
	}

	// RETURN RESULTS
	*reused=FALSE;
//	return;
}

// Identify (on output) new reuse of a given lump
void WADIdentifyNewReuseOnOutput (
	FILE *outfile,												// WAD file to search in
	const WADDIR_t *WADDIR,										// WAD file directory to look for potential reuse in
	const LST_t *LST,											// symbol table resulting from the scan of that directory
	const void *lump_wanted,									// lump to check for potential reuse of
	size_t wanted,												// index in WAD file directory of the lump
	void (canonicalize) (										// lump canonicalization function
		const void *lump_in,									// lump to process
		const WADDIR_t *WADDIR,									// WAD file directory to take entry metrics from
		size_t index,											// index of lump in directory entries
		bool_t truncate_waves,									// remove wasted space at the end of sound resources
		bool_t rebuild_blockmaps,								// rebuild (no pack/unpack) blockmaps to remove wasted space
		bool_t pack_blockmaps,									// pack blockmaps to remove current and potential wasted space
		bool_t unpack_blockmaps,								// unpack blockmaps into normalised (but inefficient) format
		bool_t flush_blockmaps,									// write out modified blockmaps even if the size has not changed
		bool_t rebuild_pictures,								// rebuild (no pack/unpack) pictures to remove wasted space
		bool_t pack_pictures,									// pack pictures to remove current and potential wasted space
		bool_t unpack_pictures,									// unpack pictures into normalised (but inefficient) format
		bool_t flush_pictures,									// write out modified pictures even if the size has not changed
		bool_t maintain_blockmaps,								// maintain nonstandard blockmaps (preserve non-optimal blocks)
		bool_t maintain_pictures,								// maintain nonstandard pictures (preserve non-optimal columns)
		bool_t process_only,									// issue no messages except for fatal errors
		void **lump_out,										// processed lump
		size_t *size_out),										// size of processed lump
	bool_t truncate_waves,										// remove wasted space at the end of sound resources
	bool_t rebuild_blockmaps,									// rebuild (no pack/unpack) blockmaps to remove wasted space
	bool_t pack_blockmaps,										// pack blockmaps to remove current and potential wasted space
	bool_t unpack_blockmaps,									// unpack blockmaps into normalised (but inefficient) format
	bool_t flush_blockmaps,										// write out modified blockmaps even if the size has not changed
	bool_t rebuild_pictures,									// rebuild (no pack/unpack) pictures to remove wasted space
	bool_t pack_pictures,										// pack pictures to remove current and potential wasted space
	bool_t unpack_pictures,										// unpack pictures into normalised (but inefficient) format
	bool_t flush_pictures,										// write out modified pictures even if the size has not changed
	bool_t maintain_blockmaps,									// maintain nonstandard blockmaps (preserve non-optimal blocks)
	bool_t maintain_pictures,									// maintain nonstandard pictures (preserve non-optimal columns)
	bool_t process_only,										// issue no messages except for fatal errors
	bool_t *reused,												// flag for this lump reuses a previous directory entry
	size_t *reused_index)										// index of entry that this lump can now reuse
{
	// PRIMARY ERROR CHECK
	if (outfile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to find specific lump reuse in WAD file directory with NULL output file argument");
	}
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify new lump reuse in WAD file directory with NULL directory argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify new lump reuse in WAD file directory with NULL entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (lump_wanted==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify new lump reuse with NULL lump argument");
	}
	if (wanted>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to identify new lump reuse with \"wanted\" (%ld) greater than directory entry count (%ld)",wanted,WADDIR->count);
	}

	// INITIALIZE
	*reused=FALSE;
	*reused_index=0;

	// THERE CANNOT BE A PREVIOUS ENTRY TO ENTRY ZERO
	if (wanted==0)
	{
		return;
	}

	//	REUSE MAKES NO SENSE IF LUMP HAS ZERO SIZE
	if (WADDIR->entries[wanted].length==0)
	{
		return;
	}

	// FIND A COMPATIBLE COPY OF THIS IF ANY EXISTS
	WADFindCopyToReuse (
		outfile,											// WAD file to search in
		WADDIR,												// WAD file directory to search
		lump_wanted,										// lump to check for identical copy of
		wanted,												// index in WAD file directory of that lump
		canonicalize,										// lump canonicalization function
		truncate_waves,										// remove wasted space at the end of sound resources
		rebuild_blockmaps,									// rebuild (no pack/unpack) blockmaps to remove wasted space
		pack_blockmaps,										// pack blockmaps to remove current and potential wasted space
		unpack_blockmaps,									// unpack blockmaps into normalised (but inefficient) format
		flush_blockmaps,									// write out modified blockmaps even if the size has not changed
		rebuild_pictures,									// rebuild (no pack/unpack) pictures to remove wasted space
		pack_pictures,										// pack pictures to remove current and potential wasted space
		unpack_pictures,									// unpack pictures into normalised (but inefficient) format
		flush_pictures,										// write out modified pictures even if the size has not changed
		maintain_blockmaps,									// maintain nonstandard blockmaps (preserve non-optimal blocks)
		maintain_pictures,									// maintain nonstandard pictures (preserve non-optimal columns)
		process_only,										// issue no messages except for fatal errors
		reused,												// flag for found such a lump
		reused_index);										// index in directory of found lump

	// IF FOUND A MATCH THEN REPORT IT
	if (*reused)
	{
		EventState(VERBOSITY_ACTIONS,
				   "Lump %s (entry %ld) reused for lump %s (entry %ld)",
				   WADDIR->entries[*reused_index].name,
				   *reused_index,
				   WADDIR->entries[wanted].name,
				   wanted);
	}

	// RETURN RESULTS
	return;
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
