/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

Wad list symbol table

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include "services.h"
#include "waddef.h"
#include "wadlst.h"

/**********************************************************************************************************************************/
/************************************************* List Symbol Table Maintainance *************************************************/
/**********************************************************************************************************************************/

#define LST_APPORTION LIST_MAXPLUSONE
#define LST_INCREMENT 4

// Initialise a list symbol table
void LSTInit (
	LST_t **LSTp)												// list symbol table to be initialised
{
	// VARIABLES
	size_t sizeb;
	list_t list;
	LST_t *objptr;
	LSTentry_t *entptr;

	// PRIMARY ERROR CHECK
	if (LSTp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise list symbol table with NULL argument pointer");
	}
	if (*LSTp!=NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to initialise list symbol table with non-NULL argument");
	}

	// ALLOCATE MEMORY FOR THE LST
	sizeb=sizeof(LST_t);
	objptr=malloc(sizeb);
	if (objptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for list symbol table",sizeb);
	}
	(void) memset(objptr,0,sizeb);

	// ALLOCATE MEMORY FOR THE LST ENTRIES
	sizeb=LST_APPORTION*sizeof(LSTentry_t);
	entptr=malloc(sizeb);
	if (entptr==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for list symbol table entries",sizeb);
	}
	(void) memset(entptr,0,sizeb);

	// INITIALISE THE LST STATIC FIELDS
	objptr->limit=LST_APPORTION;
	objptr->count=0;
	objptr->entries=entptr;

	// INITIALISE THE STANDARD ENTRIES
	for (list=0;list<LIST_MAXPLUSONE;list++)
	{
		LSTentry_t *entry;
		/****************/
		entry=&entptr[list];
		switch (list)
		{
			case LIST_UNKNOWN:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"UNKNOWN");
				(void) strcpy(entry->name,"unknown");
				(void) strcpy(entry->desc,"Unknown list");
				break;
			case LIST_MAPS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"MAPS");
				(void) strcpy(entry->name,"maps");
				(void) strcpy(entry->desc,"Levels)");
				break;
			case LIST_TEXTURES:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"TEXTURES");
				(void) strcpy(entry->name,"textures");
				(void) strcpy(entry->desc,"Textures (composed of wall patches)");
				break;
			case LIST_PNAMES:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"PNAMES");
				(void) strcpy(entry->name,"pnames");
				(void) strcpy(entry->desc,"List of wall patch names");
				break;
			case LIST_PCSFXS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"PCSFX");
				(void) strcpy(entry->name,"pcsfx");
				(void) strcpy(entry->desc,"PC speaker sound effects");
				break;
			case LIST_SOUNDS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"SOUNDS");
				(void) strcpy(entry->name,"sounds");
				(void) strcpy(entry->desc,"Sound effects");
				break;
			case LIST_MUSICS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"MUSICS");
				(void) strcpy(entry->name,"musics");
				(void) strcpy(entry->desc,"Music tracks");
				break;
			case LIST_DIALOGS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"DIALOGS");
				(void) strcpy(entry->name,"dialogs");
				(void) strcpy(entry->desc,"Dialog messages (from STRIFE)");
				break;
			case LIST_CONVERSATIONS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"CONVERSATIONS");
				(void) strcpy(entry->name,"conversations)");
				(void) strcpy(entry->desc,"Conversation scripts (from STRIFE)");
				break;
			case LIST_LUMPS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"LUMPS");
				(void) strcpy(entry->name,"lumps");
				(void) strcpy(entry->desc,"Generic lumps");
				break;
			case LIST_LOOSEWALLPATCHES:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"LOOSEWALLPATCHES");
				(void) strcpy(entry->name,"loose wall patches");
				(void) strcpy(entry->desc,"Wall patches (as listed in PNAMES but not found between markers)");
				break;
			case LIST_GRAPHICS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"");
				(void) strcpy(entry->code,"GRAPHICS");
				(void) strcpy(entry->name,"graphics");
				(void) strcpy(entry->desc,"Miscellaneous graphics");
				break;
			case LIST_HERIANFONTA:
				entry->seen=0;
				entry->font=TRUE;
				(void) strcpy(entry->flag,"A");
				(void) strcpy(entry->code,"HERIANFONTA");
				(void) strcpy(entry->name,"heretic/hexen font [a]");
				(void) strcpy(entry->desc,"Heretic/HeXen font \"A\"");
				break;
			case LIST_HERIANFONTAY:
				entry->seen=0;
				entry->font=TRUE;
				(void) strcpy(entry->flag,"AY");
				(void) strcpy(entry->code,"HERIANFONTAY");
				(void) strcpy(entry->name,"hexen font [ay]");
				(void) strcpy(entry->desc,"HeXen font \"AY\"");
				break;
			case LIST_HERIANFONTB:
				entry->seen=0;
				entry->font=TRUE;
				(void) strcpy(entry->flag,"B");
				(void) strcpy(entry->code,"HERIANFONTB");
				(void) strcpy(entry->name,"heretic/hexen font [b]");
				(void) strcpy(entry->desc,"Heretic/HeXen font \"B\"");
				break;
			case LIST_HERETICFONTA:
				entry->seen=0;
				entry->font=TRUE;
				(void) strcpy(entry->flag,"A");
				(void) strcpy(entry->code,"HERETICFONTA");
				(void) strcpy(entry->name,"heretic font [a]");
				(void) strcpy(entry->desc,"Heretic font \"A\"");
				break;
			case LIST_HERETICFONTB:
				entry->seen=0;
				entry->font=TRUE;
				(void) strcpy(entry->flag,"B");
				(void) strcpy(entry->code,"HERETICFONTB");
				(void) strcpy(entry->name,"heretic font [b]");
				(void) strcpy(entry->desc,"Heretic font \"B\"");
				break;
			case LIST_HEXENFONTA:
				entry->seen=0;
				entry->font=TRUE;
				(void) strcpy(entry->flag,"A");
				(void) strcpy(entry->code,"HEXENFONTA");
				(void) strcpy(entry->name,"hexen font [a]");
				(void) strcpy(entry->desc,"HeXen font \"A\"");
				break;
			case LIST_HEXENFONTAY:
				entry->seen=0;
				entry->font=TRUE;
				(void) strcpy(entry->flag,"AY");
				(void) strcpy(entry->code,"HEXENFONTAY");
				(void) strcpy(entry->name,"hexen font [ay]");
				(void) strcpy(entry->desc,"Hexen font \"AY\"");
				break;
			case LIST_HEXENFONTB:
				entry->seen=0;
				entry->font=TRUE;
				(void) strcpy(entry->flag,"B");
				(void) strcpy(entry->code,"HEXENFONTB");
				(void) strcpy(entry->name,"hexen font [b]");
				(void) strcpy(entry->desc,"HeXen font \"B\"");
				break;
			case LIST_SPRITES:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"S");
				(void) strcpy(entry->code,"SPRITES");
				(void) strcpy(entry->name,"sprites");
				(void) strcpy(entry->desc,"Actor sprites");
				break;
			case LIST_PATCHES:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"P");
				(void) strcpy(entry->code,"PATCHES");
				(void) strcpy(entry->name,"patches");
				(void) strcpy(entry->desc,"Wall patches (as found between markers)");
				break;
			case LIST_FLATS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"F");
				(void) strcpy(entry->code,"FlATS");
				(void) strcpy(entry->name,"flats");
				(void) strcpy(entry->desc,"Foors and ceilings");
				break;
			case LIST_COLORMAPS:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"C");
				(void) strcpy(entry->code,"COLORMAPS");
				(void) strcpy(entry->name,"colormaps");
				(void) strcpy(entry->desc,"Color maps (from BOOM)");
				break;
			case LIST_ACSLIBRARIES:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"A");
				(void) strcpy(entry->code,"ACSLIBRARIES");
				(void) strcpy(entry->name,"acs libraries");
				(void) strcpy(entry->desc,"ACS libraries (from ZDOOM)");
				break;
			case LIST_AUTOTEXTURES:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"TX");
				(void) strcpy(entry->code,"AUTOTEXTURES");
				(void) strcpy(entry->name,"auto-textures");
				(void) strcpy(entry->desc,"Single-patch textures (independent of PNAMES) (from ZDOOM)");
				break;
			case LIST_VOICES:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"V");
				(void) strcpy(entry->code,"VOICES");
				(void) strcpy(entry->name,"voices");
				(void) strcpy(entry->desc,"Actor voices (from STRIFE)");
				break;
			case LIST_SPRITESSUB1:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"S1");
				(void) strcpy(entry->code,"SPRITES1");
				(void) strcpy(entry->name,"sprites (sub-list #1)");
				(void) strcpy(entry->desc,"Sprites (sub-list #1)");
				break;
			case LIST_SPRITESSUB2:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"S2");
				(void) strcpy(entry->code,"SPRITES2");
				(void) strcpy(entry->name,"sprites (sub-list #2)");
				(void) strcpy(entry->desc,"Sprites (sub-list #2)");
				break;
			case LIST_SPRITESSUB3:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"S3");
				(void) strcpy(entry->code,"SPRITES3");
				(void) strcpy(entry->name,"sprites (sub-list #3)");
				(void) strcpy(entry->desc,"Sprites (sub-list #3)");
				break;
			case LIST_PATCHESSUB1:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"P1");
				(void) strcpy(entry->code,"PATCHES1");
				(void) strcpy(entry->name,"patches (sub-list #1)");
				(void) strcpy(entry->desc,"Patches (sub-list #1)");
				break;
			case LIST_PATCHESSUB2:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"P2");
				(void) strcpy(entry->code,"PATCHES2");
				(void) strcpy(entry->name,"patches (sub-list #2)");
				(void) strcpy(entry->desc,"Patches (sub-list #2)");
				break;
			case LIST_PATCHESSUB3:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"P3");
				(void) strcpy(entry->code,"PATCHES3");
				(void) strcpy(entry->name,"patches (sub-list #3)");
				(void) strcpy(entry->desc,"Patches (sub-list #3)");
				break;
			case LIST_FLATSSUB1:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"F1");
				(void) strcpy(entry->code,"FLATS1");
				(void) strcpy(entry->name,"flats (sub-list #1)");
				(void) strcpy(entry->desc,"Flats (sub-list #1)");
				break;
			case LIST_FLATSSUB2:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"F2");
				(void) strcpy(entry->code,"FLATS2");
				(void) strcpy(entry->name,"flats (sub-list #2)");
				(void) strcpy(entry->desc,"Flats (sub-list #2)");
				break;
			case LIST_FLATSSUB3:
				entry->seen=0;
				entry->font=FALSE;
				(void) strcpy(entry->flag,"F3");
				(void) strcpy(entry->code,"FLATS3");
				(void) strcpy(entry->name,"flats (sub-list #3)");
				(void) strcpy(entry->desc,"Flats (sub-list #3)");
				break;
			default:
				FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %ld in variable \"list\"",list);
				break;
			////
		}
	}
	objptr->count=LIST_MAXPLUSONE;

	// RETURN RESULT
	*LSTp=objptr;
}

// Deinitialise a list symbol table
void LSTDone (
	LST_t **LSTp)												// list symbol table to be deinitialised
{
	// PRIMARY ERROR CHECK
	if (LSTp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise list symbol table with NULL argument pointer");
	}
	if (*LSTp==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise list symbol table with NULL argument");
	}
	if ((*LSTp)->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to deinitialise list symbol table with NULL entries");
	}
#ifdef PARANOID
	if ((*LSTp)->count>(*LSTp)->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// DEALLOCATE MEMORY FOR THE LST ENTRIES
	free((*LSTp)->entries);

	// DEALLOCATE MEMORY FOR THE LST
	free(*LSTp);

	// RETURN RESULT
	*LSTp=NULL;
}

/**********************************************************************************************************************************/
/**************************************************** List Symbol Table Usage *****************************************************/
/**********************************************************************************************************************************/

// Add new symbol into a list symbol table and return metrics
void LSTAddSymbolAndGetMetrics (
	LST_t *LST,													// list symbol table to add symbol to (if not already there)
	const char *name,											// lump name from which to derive flag, list and start flag
	type_t type,												// lump name type
	list_t *list,												// list identifier associated with lump name
	bool_t *start)												// start of list flag (FALSE means end of list) for lump name
{
	// VARIABLES
	size_t i,index;
	char nameflag[LIST_MAXFLAGLENGTH+1];
	bool_t is_font,is_start;
	LSTentry_t *entry;
	int check;

	// PRIMARY ERROR CHECK
	if (LST==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to add to list symbol table with NULL argument");
	}
	if (LST->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to add to list symbol table with NULL entries");
	}
#ifdef PARANOID
	if (LST->count>LST->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// SECONDARY ERROR CHECK
	if (strlen(name)>8)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to use name (\"%s\") as a WAD entry name (longer than 8 characters)",name);
	}

	// IS IT A NORMAL LIST OR A GRAPHIC FONT?
	is_font=(
		   (type==TYPE_HERIANFONTSTART)  ||
		   (type==TYPE_HERETICFONTSTART) ||
		   (type==TYPE_HEXENFONTSTART)   ||
		   (type==TYPE_HERIANFONTEND)    ||
		   (type==TYPE_HERETICFONTEND)   ||
		   (type==TYPE_HEXENFONTEND)
		 );

	// GET FLAG CORRESPONDING TO NAME
	if (is_font)
	{
		nameflag[0]=name[4];
		nameflag[1]=((name[5]=='_')?NUL:name[5]);
		nameflag[2]=NUL;
		is_start=(( ((name[5]=='_') && (name[6]=='S')) || ((name[6]=='_') && (name[7]=='S')) )?TRUE:FALSE);
	}
	else
	{
		nameflag[0]=name[0];
		nameflag[1]=((name[1]=='_')?NUL:name[1]);
		nameflag[2]=NUL;
		is_start=(( ((name[1]=='_') && (name[2]=='S')) || ((name[2]=='_') && (name[3]=='S')) )?TRUE:FALSE);
	}

	// FIND FLAG AND RETURN WITH ITS METRICS IF FOUND
	for (i=0;i<LST->count;i++)
	{
		if (
		     (
		       // The following two lines tell us why Pascal
		       // is a (any old garbage will do here as long
			   // as it is nonzero) programming language,
			   // while C is just a glorified assembler :-)
		       ( LST->entries[i].font &&  is_font) ||
		       (!LST->entries[i].font && !is_font)
		     )
		     &&
		     (strcmp(nameflag,LST->entries[i].flag)==0)
		   )
		{
			if (is_start)
			{
				LST->entries[i].seen++;
			}
			*list=i;
			*start=is_start;
			return;
		}
	}

	// RANGE CHECK ON NEW VALUE FOR LIST ID
	if (LST->count==LIST_MAXREALVALUE)
	{
		FatalAbort(1,__FILE__,__LINE__,"Could not grow list symbol table by 1 entry (from %ld entries)",LIST_MAXREALVALUE);
	}

	// REALLOCATE [IF NECESSARY] MEMORY FOR THE LST ENTRIES
	if (LST->count==LST->limit)
	{
		LSTentry_t *newptr;
		size_t size,sizeb,incrb;
		/**********************/
		size=(LST->limit)+LST_INCREMENT;
		sizeb=size*sizeof(LSTentry_t);
		incrb=LST_INCREMENT*sizeof(LSTentry_t);
		newptr=realloc(LST->entries,sizeb);
		if (newptr==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not grow list symbol table entries by %ld bytes (from %ld bytes)",incrb,sizeb);
		}
		LST->entries=newptr;
		(void) memset(&LST->entries[LST->limit],0,incrb);
		LST->limit=size;
	}

	// PERFORM THE OPERATION
	index=LST->count;
	entry=&LST->entries[index];
	entry->seen=1;
	entry->font=is_font;
	(void) strcpy(entry->flag,nameflag);
	check=snprintf(entry->code,LIST_MAXCODELENGTH+1,"LIST%04lu (%s_)",((unsigned long int)index),nameflag);
	if ( (check<0) || (check>LIST_MAXCODELENGTH) )
	{
		FatalAbort(1,__FILE__,__LINE__,"Buffer overflow avoided while adding symbol %ld code to list symbol table",index);
	}
	check=snprintf(entry->name,LIST_MAXNAMELENGTH+1,"unknown list #%lu (%s)",((unsigned long int)index),nameflag);
	if ( (check<0) || (check>LIST_MAXNAMELENGTH) )
	{
		FatalAbort(1,__FILE__,__LINE__,"Buffer overflow avoided while adding symbol %ld name to list symbol table",index);
	}
	check=snprintf(entry->desc,LIST_MAXDESCLENGTH+1,"unknown list #%lu (%s)",((unsigned long int)index),nameflag);
	if ( (check<0) || (check>LIST_MAXDESCLENGTH) )
	{
		FatalAbort(1,__FILE__,__LINE__,"Buffer overflow avoided while adding symbol %ld description to list symbol table",index);
	}
	LST->count++;

	// RETURN RESULTS
	*list=(LST->count)-1;
	*start=(( ((name[1]=='_') && (name[2]=='S')) || ((name[2]=='_') && (name[3]=='S')) )?TRUE:FALSE);
}

// Return metrics of an existing symbol via a lump name
void LSTGetMetricsViaLumpName (
	LST_t *LST,													// list symbol table to get list metrics from
	const char *name,											// lump name from which to identify list
	type_t type,												// lump name type
	list_t *list,												// list identifier associated with lump name
	bool_t *start)												// start of list flag (FALSE means end of list) for lump name
{
	// VARIABLES
	size_t i;
	char nameflag[LIST_MAXFLAGLENGTH+1];
	bool_t is_font,is_start;

	// PRIMARY ERROR CHECK
	if (LST==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get list metrics via lump name with NULL argument");
	}
	if (LST->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get list metrics via lump name with NULL entries");
	}
#ifdef PARANOID
	if (LST->count>LST->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// SECONDARY ERROR CHECK
	if (strlen(name)>8)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to use name (\"%s\") as a WAD entry name (longer than 8 characters)",name);
	}

	// IS IT A NORMAL LIST OR A GRAPHIC FONT?
	is_font=(
		   (type==TYPE_HERIANFONTSTART)  ||
		   (type==TYPE_HERETICFONTSTART) ||
		   (type==TYPE_HEXENFONTSTART)   ||
		   (type==TYPE_HERIANFONTEND)    ||
		   (type==TYPE_HERETICFONTEND)   ||
		   (type==TYPE_HEXENFONTEND)
		 );

	// GET FLAG CORRESPONDING TO NAME
	if (is_font)
	{
		nameflag[0]=name[4];
		nameflag[1]=((name[5]=='_')?NUL:name[5]);
		nameflag[2]=NUL;
		is_start=(( ((name[5]=='_') && (name[6]=='S')) || ((name[6]=='_') && (name[7]=='S')) )?TRUE:FALSE);
	}
	else
	{
		nameflag[0]=name[0];
		nameflag[1]=((name[1]=='_')?NUL:name[1]);
		nameflag[2]=NUL;
		is_start=(( ((name[1]=='_') && (name[2]=='S')) || ((name[2]=='_') && (name[3]=='S')) )?TRUE:FALSE);
	}

	// FIND FLAG AND RETURN WITH ITS METRICS
	for (i=0;i<LST->count;i++)
	{
		if (
		     (
		       // The following two lines tell us why Pascal
		       // is a (any old garbage will do here as long
			   // as it is nonzero) programming language,
			   // while C is just a glorified assembler :-)
		       ( LST->entries[i].font &&  is_font) ||
		       (!LST->entries[i].font && !is_font)
		     )
		     &&
		     (strcmp(nameflag,LST->entries[i].flag)==0)
		   )
		{
			if (is_start)
			{
				LST->entries[i].seen++;
			}
			*list=i;
			*start=is_start;
			return;
		}
	}

	// SHOULD HAVE BEEN FOUND BY NOW
	FatalAbort(1,__FILE__,__LINE__,"Could not find list metrics for lump name \"%s\"",name);
}

// Get a lump name from a list symbol table and name metrics
void LSTGetLumpName (
	const LST_t *LST,											// list symbol table to get flag from
	list_t list,												// list identifier associated with lump name
	bool_t start,												// start of list flag (FALSE means end of list) for lump name
	char *s)													// lump name to construct from flag, list and start flag
{
	// PRIMARY ERROR CHECK
	if (LST==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get from list symbol table with NULL argument");
	}
	if (LST->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to get from list symbol table with NULL entries");
	}
#ifdef PARANOID
	if (LST->count>LST->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// SECONDARY ERROR CHECK
	if (list>LST->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Unrecognised value %ld in variable \"list\"");
	}

	// GET NAME CORRESPONDING TO FLAG
	if (
		 (list==LIST_HERIANFONTA)  ||
		 (list==LIST_HERIANFONTAY) ||
		 (list==LIST_HERIANFONTB)  ||
		 (list==LIST_HERETICFONTA) ||
		 (list==LIST_HERETICFONTB) ||
		 (list==LIST_HEXENFONTA)   ||
		 (list==LIST_HEXENFONTAY)  ||
		 (list==LIST_HEXENFONTB)
		 )
	{
		(void) strcpy(s,"FONT");
		(void) strcat(s,LST->entries[list].flag);
		(void) strcat(s,(start?"_S":"_E"));
	}
	else
	{
		(void) strcpy(s,LST->entries[list].flag);
		(void) strcat(s,(start?"_START":"_END"));
	}
}

// Check if any list was found more than once
bool_t LSTCheckForDuplicates (
	const LST_t *LST)											// list symbol table to check for duplicate flags
{
	// VARIABLES
	size_t i;

	// PRIMARY ERROR CHECK
	if (LST==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check if any list was found more than once in list symbol table with NULL argument");
	}
	if (LST->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to check if any list was found more than once in list symbol table with NULL entries");
	}
#ifdef PARANOID
	if (LST->count>LST->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"List symbol table is corrupt");
	}
#endif

	// PERFORM OPERATION
	for (i=0;i<LST->count;i++)
	{
		if (LST->entries[i].seen>1)
		{
			return TRUE;
		}
	}
	return FALSE;
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
