/**********************************************************************************************************************************/
/********************************************************* Documentation **********************************************************/
/**********************************************************************************************************************************/

/*

WAD entry optimisation

*/

/**********************************************************************************************************************************/
/*********************************************************** Systemics ************************************************************/
/**********************************************************************************************************************************/

// Includes
#include <malloc.h>
#include <string.h>
#include <stdlib.h>
#include "services.h"
#include "filesys.h"
#include "waddef.h"
#include "wadlru.h"
#include "wadopt.h"

/**********************************************************************************************************************************/
/***************************************************** WAD Entry Optimization *****************************************************/
/**********************************************************************************************************************************/

// Check current block for duplicates or disorder
static bool_t BlockDuplicatesOrDisorder (const SINT16 *current_block, const SINT16 *current_number)
{
	// DOCUMENTATION
	/*
		Find duplicate or out-of-order linedef numbers in the
		block and stop if one is found. All linedef numbers
		before the current number are checked to see if they
		are less than the current number. If equal, then the
		current number is a duplicate of that number; if
		greater, then the data for the found linedef run occurs
		before that of the (lower-numbered) current linedef.
	*/
	// VARIABLES
	const SINT16 *p=current_block;
	SINT16 v=*current_number;
	// SKIP OVER THE DUMMY VALUE
	p++;
	// PROCESS
	while (p<current_number)
	{
		// PERFORM THE CHECK
		if (*p>=v)
		{
			// Found duplicate or out-of-order linedef
			// number. This shouldn't prevent the map
			// from being played, but it could indicate
			// trouble elsewhere in the input file.
			return TRUE;
		}
		// POINT AT NEXT LINEDEF NUMBER
		p++;
	}
	// RETURN
	return FALSE;
}

// Check current block for problems
static bool_t BlockAnyLinedefNumberProblems (const SINT16 *current_block)
{
	// VARIABLES
	const SINT16 *p=current_block;
	// SKIP OVER THE DUMMY VALUE
	p++;
	// VALIDATE THE ACTUAL LINEDEF LIST
	while (*p!=(-1))
	{
		// CHECK FOR DUPLICATES OR DISORDERED VALUES.
		if (BlockDuplicatesOrDisorder(current_block,p))
		{
			return TRUE;
		}
		// POINT TO NEXT IN BLOCK (OR END-OF-BLOCK FLAG)
		p++;
	}
	// RETURN
	return FALSE;
}

// Find block length (including dummy and terminator)
static size_t FindBlockLength (const SINT16 *current_block)
{
	// VARIABLES
	const SINT16 *p;
	size_t numbers_scanned;
	// GET POINTER TO FIRST NUMBER IN THE LINEDEF LIST
	p=current_block;
	// GET NUMBER COUNT CORRESPONDING TO POINTER
	numbers_scanned=1;
	//
	// POINTER P NOW POINTS AT THE FIRST NUMBER IN THE LIST
	// AND NUMBER_SCANNED IS NOW THE COUNT OF NUMBERS IN THE
	// BLOCK INCLUDING THE ONE POINTED AT BY P BUT NOT THE
	// NUMBERS BEFORE IT (PARTS OF POINTERS, HEADER, ETC).
	//
	// RECALL THAT THIS NUMBER IS A DUMMY NUMBER WITH A
	// VALUE OF ZERO AND THAT IF THE BLOCK ACTUALLY STARTS
	// AT LINEDEF ZERO THEN ANOTHER ZERO COMES AFTER IT AND
	// IT IS THAT ZERO WHICH IS THE ACTUAL LIST START. THIS
	// STRUCTURE WILL HAVE ALREADY BEEN VALIDATED BY NOW.
	//
	// SKIP OVER THE DUMMY VALUE
	//
	// NOTE THAT EVEN IF THIS IS AN EMPTY BLOCK, WE STILL
	// COUNT THE TERMINATOR FLAG (-1) AS PART OF THE BLOCK,
	// WHICH IS WHY WE UNCONDITIONALLY INCREMENT THE COUNT.
	//
	numbers_scanned++;
	p++;
	// SCAN THE REST OF THE BLOCK
	while (*p!=(-1))
	{
		// POINT TO NEXT IN BLOCK (OR END-OF-BLOCK FLAG)
		numbers_scanned++;
		p++;
	}
	// RETURN
	return numbers_scanned;
}

// QSort-compatible comparison f'n for lindef number sorting
int SortLinedefNumbersCompare (const void *x, const void *y)
{
	// VARIABLES
	const SINT16 p=*((SINT16*)x);
	const SINT16 q=*((SINT16*)y);
	//
	// No idea
	//
	// CHECK LINEDEF NUMBER INEQUALITY
	if (p<q)
	{
		return (-1);
	}
	if (p>q)
	{
		return 1;
	}
	//
	// Linedef numbers are equal
	//
	// SOMETHING WRONG HERE; WE DELIBERATELY SET THIS UP AS
	// A STABLE SORT, SO NO TWO ITEMS SHOULD EVER BE EQUAL.
	FatalAbort(1,__FILE__,__LINE__,"Sort preprocessing failed (found equal items in stable sort)");
	return 0;
}

// Rebuild a block (linedef list) in a blockmap
static void RebuildBlock (SINT16 *in_list, size_t in_length)
{
	// DOCUMENTATION
	/*
		The list should have at least two elements apart from
		the dummy zero value and the terminator, both of which
		should be included in the list. THIS IS NOT CHECKED!
	*/
	// VARIABLES
	//
	// These initialisers skip over the dummy zero value
	//
	size_t count_in=2;
	const SINT16 *p_in=in_list+1;
	size_t count_out=2;
	SINT16 *p_out=in_list+1;
	// REMOVE DUPLICATES FROM LIST
	while (count_in<in_length)
	{
		// VARIABLES
		bool_t found;
		const SINT16 *q;
		// POINT TO NEXT LINEDEF NUMBER
		count_in++;
		p_in++;
		// FIND THIS
		found=FALSE;
		for (q=in_list+1;q<=p_out;q++)
		{
			if (*q==*p_in)
			{
				found=TRUE;
				break;
			}
		}
		// IF NOT FOUND THEN NEW ITEM
		if (!found)
		{
			count_out++;
			p_out++;
			if (p_out!=p_in)
			{
				*p_out=*p_in;
			}
		}
	}
	// SORT (EXCLUDING DUMMY AND TERMINATOR) BY LINEDEF NUMBER
	qsort(in_list+/*for the dummy value zero*/1,(count_out-/*for the dummy value zero and the (-1)*/2),sizeof(SINT16),SortLinedefNumbersCompare);
}

// Rebuild blocks (linedef lists) in a blockmap that require it
static void RebuildLinedefBlocks (void *in, const char *mapname, size_t num_blocks, bool_t process_only)
{
	// DOCUMENTATION
	/*
		The blockmap and each of its linedef lists must have
		already been checked for all other inconsistencies
		before this function is called. THIS IS NOT CHECKED!

		There is no need to allocate fresh memory, because the
		rebuild cannot GROW the physical block data; it can
		only shrink it or leave it at the same size.
	*/

	// VARIABLES
	const UINT16 *inpointers=(UINT16*)in+(sizeof(blockmap_header_t)/sizeof(UINT16));
	size_t block,rebuilt_blocks;

	// INITIALISE STATISTICS
	rebuilt_blocks=0;

	// CHECK AND IF NEEDED THEN REBUILD THE BLOCK
	for (block=0;block<num_blocks;block++)
	{
		// VARIABLES
		size_t block_length;
		const SINT16 *block_data=(SINT16*)in+inpointers[block];
		SINT16 *block_data_notconst=(SINT16*)in+inpointers[block];
		// SKIP IF NO ERRORS IN BLOCK
		if (!BlockAnyLinedefNumberProblems(block_data))
		{
			continue;
		}
		// ACCOUNT FOR BLOCK TO BE REBUILT
		rebuilt_blocks++;
		// ISSUE DIAGNOSTIC IF REQUIRED
		if (!process_only)
		{
			EventState(VERBOSITY_DETAILS,"Rebuilding block (linedef list) %ld",block);
		}
		// FIND BLOCK LENGTH
		block_length=FindBlockLength(block_data);
		// REBUILD THE BLOCK
		RebuildBlock(block_data_notconst,block_length);
	}

	// LOG FULL BLOCK REBUILDS
	if (rebuilt_blocks==0)
	{
		FatalAbort(1,__FILE__,__LINE__,"Logic failure 001 during rebuild of blockmap of %s",mapname);
	}
	if (!process_only)
	{
		EventState(VERBOSITY_WARNINGS,"Blockmap from %s required %ld linedef lists (of %ld) to be rebuilt",mapname,rebuilt_blocks,num_blocks);
	}
}

#define BLOCKMAP_INCREMENT 1024

// Optimize a blockmap
static void OptimizeBlockmap (
	const void *in,
	void **out,
	size_t in_count,
	bool_t pack_blockmaps,
	bool_t unpack_blockmaps,
	bool_t process_only,
	size_t *size_out,
	bool_t *success)
{
	// DOCUMENTATION
	/*
		This does not need a "rebuild_blockmaps" parameter as
		it always does at least a rebuild. It may also pack or
		unpack, according to the other two blockmap options.
	*/

	// VARIABLES
	const UINT16 *inpointers=(UINT16*)in+(sizeof(blockmap_header_t)/sizeof(UINT16));
	void *out_pcopy;
	UINT16 *outpointers;
	size_t num_blocks,i,out_limit,out_count=0;

	// PRIMARY ERROR CHECK
	if (pack_blockmaps && unpack_blockmaps)
	{
		FatalAbort(1,__FILE__,__LINE__,"Contradictory blockmap processing flags should have been trapped by now");
	}

	// INITIALISE
	num_blocks=(size_t)(RDLend16UV(((blockmap_header_t*)in)->rows)*RDLend16UV(((blockmap_header_t*)in)->columns));
	out_pcopy=*out;
	out_limit=in_count;
//	out_count=0;

	// COPY OUT HEADER AND POINTER ARRAY
	out_count=sizeof(blockmap_header_t)+(num_blocks*sizeof(UINT16));
	if (out_count>out_limit)
	{
		void *newptr;
		size_t sizeb,incrb;
		/*****************/
		sizeb=out_count+BLOCKMAP_INCREMENT;
		incrb=sizeb-out_limit;
		newptr=realloc(out_pcopy,sizeb);
		if (newptr==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not grow blockmap pointer list by %ld bytes (from %ld bytes)",incrb,out_limit);
		}
		out_pcopy=newptr;
		out_limit=sizeb;
	}
	(void) memcpy(out_pcopy,in,out_count);

	// SET UP (C) POINTERS INTO BOTH (DOOM) POINTER ARRAYS
	/*****************************************************
	Note that some of the pointer arithmetic, like DOOM
	itself, relies on the blockmap header size being a
	multiple of the block data and pointer sizes. This is
	because DOOM pointers in a blockmap are like typed C
	pointers in that their values are in units of the size
	of what they point to, rather than bytes; that is each
	gives the offset of its block as the number of 16-BIT
	OBJECTS into the resource and not the number of bytes.
	******************************************************/
//	inpointers=(UINT16*)in+(sizeof(blockmap_header_t)/sizeof(UINT16));
	outpointers=(UINT16*)out_pcopy+(sizeof(blockmap_header_t)/sizeof(UINT16));

	// PROCESS EACH BLOCK
	for (i=0;i<num_blocks;i++)
	{
		// VARIABLES
		bool_t match;

		// ASSUME NO MATCH FOUND WHEN STARTING EACH BLOCK
		match=FALSE;

		// FIND AND COPY DUPLICATE POINTERS UNLESS UNPACKING;
		// I.E., MUST DO THIS IF REBUILDING AND/OR PACKING.
		if (!unpack_blockmaps)
		{
			/**************************************************
			Search the input pointer array up to just before
			this position. If we find one whose value is the
			same as the one in the current position, then the
			corresponding one in the output array can share the
			same data, so overwrite it with the one that was
			found and also make a note that a match WAS found.

			Note that we do not need to byte-swap, even if the
			host machine is big-endian, as we are only testing
			pointers for equality and possible moving "as is".
			**************************************************/
			size_t j;
			UINT16 inpointers_i;
			/******************/
			inpointers_i=inpointers[i];
			for (j=0;j<i;j++)
			{
				if (inpointers_i==inpointers[j])
				{
					// Input pointer j may have been matched by
					// a DATA match, earlier in the pointer
					// array, that was not picked up when the
					// blockmap was created but was detected in
					// a previous iteration of the parent loop.
					// In this case, the output pointer j will
					// have already been changed to one from
					// earlier on in the blockmap (which may
					// itself be a changed pointer). Since they
					// have the same data, we use the earlier
					// one (if any; if none, then it would not
					// have been changed anyway). In this way,
					// we always use the same pointers for each
					// identical block, irrespective of whether
					// we check for existing pointer reuse when
					// packing or not. This is vital, as it
					// makes the packing deterministic under
					// all circumstances.
					outpointers[i]=outpointers[j];
					match=TRUE;
					break;
				}
			}
		}

		// FIND POINTERS THAT ARE NOT DUPLICATES BUT SHOULD BE;
		// REPLACE EACH SUCH WITH POINTER TO IDENTICAL DATA.
		if (pack_blockmaps && !match)
		{
			/**************************************************
			If we are packing and there was no match for this
			pointer, then examine the data of each one before
			it and see if the actual data is identical. If so
			then we can discard (i.e., not copy out) the data
			for this pointer and just overwrite this pointer
			with the pointer that points to the identical data.

			It is probably worth pointing out that prev_pointer
			can safely use zero as a "rogue value" because it
			would never have that value, as that would in fact
			refer to the blockmap header and not block data.

			Variable curr_block is a C pointer to the block
			data pointed to by the DOOM pointer currently being
			analysed (i.e., inpointers[i]). Remember that such
			data is a list of linedefs that the block includes.

			As before, we do not need to byte-swap, even if the
			host machine is big-endian, whenever we are testing
			pointers for equality and possible moving "as is";
			however we do when we are accessing the pointer.

			We would also need to byte-swap when comparing with
			a value as in the code below; however, in this case
			we are comparing with a value (-1 or 0) that has
			the same bit pattern in both endian formats.
			**************************************************/
			size_t j;
			SINT16 *curr_block;
			/*****************/
			curr_block=(SINT16*)in+RDLend16UV(inpointers[i]);
			for (j=0;j<i;j++)
			{
				// Initialised variables
				UINT16 prev_pointer=0;
				// Uninitialised variables
				UINT16 curr_pointer;
				// Cache the current DOOM pointer in question
				curr_pointer=outpointers[j];
				// If not same pointer then see if same data
				if (prev_pointer!=curr_pointer)
				{
					// Variables
					SINT16 *p_in,*p_out;
					SINT16 curr_linedefnum;
					bool_t identical;
					// C pointer of current DOOM output pointer
					p_out=(SINT16*)out_pcopy+RDLend16UV(curr_pointer);
					// C pointer to current DOOM input pointer
					p_in=curr_block;
					// Assume different until found otherwise
					identical=FALSE;
					// Compare blocks same way as DOOM strings
					while (TRUE)
					{
						curr_linedefnum=*p_out++;
						if (curr_linedefnum!=*p_in++)
						{
							// mismatch
							break;
						}
						if (curr_linedefnum==(-1))
						{
							identical=TRUE;
							break;
						}
					}
					// If same then copy pointer and finish
					if (identical)
					{
						if (!process_only)
						{
							EventState(VERBOSITY_INTERNALS,"Blockmap block %u is identical to block %u",i,j);
						}
						outpointers[i]=outpointers[j];
						match=TRUE;
						break;
					}
				}
				// Make note of previous pointer checked
				prev_pointer=curr_pointer;
			}
		}

		// IF THIS POINTER IS NOT DUPLICATED BY ANOTHER POINTER,
		// COPY ITS DATA OUT, AS IT IS DISTINCT FROM OTHER DATA.
		if (!match)
		{
			// VARIABLES
			size_t num_copied,out_pointer;
			SINT16 *p_in,*p_out;
			// C POINTER EQUIVALENT OF INPOINTERS[I]
			p_in=(SINT16*)in+inpointers[i];
			// C POINTER TO NEXT AVAILABLE DATA SPACE
			p_out=(SINT16*)out_pcopy+(out_count/sizeof(SINT16));
			// SET DOOM POINTER CORRESPONDING TO THAT SPACE
			out_pointer=(out_count/sizeof(UINT16));
			if (out_pointer>((size_t)(UINT16_MAX)))
			{
				// Processed blockmap violates
				// blockmap structure limits.
				*out=out_pcopy;
				*success=FALSE;
				*size_out=in_count;
				return;
			}
			outpointers[i]=(UINT16)(out_pointer);
			// NO LINEDEF NUMBERS HAVE BEEN COPIED YET
			num_copied=0;
			// COPY THE DATA ONE LINDEFNUM AT A TIME
			do
			{
				// GET POSITION OF NEXT OUTPUT DATA SPACE
				out_count+=sizeof(SINT16);
				// ENSURE THAT THERE IS ENOUGH SPACE
				if (out_count>out_limit)
				{
					// VARIABLES
					void *newptr;
					size_t sizeb,incrb;
					// REALLOCATE THE OUPUT DATA SPACE
					sizeb=out_count+BLOCKMAP_INCREMENT;
					incrb=sizeb-out_limit;
					newptr=realloc(out_pcopy,sizeb);
					if (newptr==NULL)
					{
						ErrorAbort(1,__FILE__,__LINE__,"Could not grow blockmap pointer list by %ld bytes (from %ld bytes)",incrb,out_limit);
					}
					out_pcopy=newptr;
					out_limit=sizeb;
					// UPDATE C POINTER TO DOOM POINTER ARRAY
					outpointers=(UINT16*)out_pcopy+(sizeof(blockmap_header_t)/sizeof(UINT16));
					// UPDATE C POINTER TO OUTPUT DATA SPACE
					p_out=(SINT16*)out_pcopy+((out_count-/*for the increment*/sizeof(SINT16))/sizeof(SINT16));
				}
				// COPY THIS LINEDEF NUMBER
				*p_out++=*p_in;
				// COUNT LINEDEFS COPIED
				num_copied++;
			} /* STOP IF HIT END OF THIS BLOCK */
			while ((*p_in++)!=(-1));
		}
	}

	// RETURN RESULTS
	*out=out_pcopy;
	*success=TRUE;
	*size_out=out_count;
}

// Check current pixel run for duplicates or disorder
static bool_t PixelRunDuplicatesOrDisorder (const UINT08 *current_column, const UINT08 *current_run)
{
	// DOCUMENTATION
	/*
		Find duplicate or out-of-order pixels in the column and
		stop if one is found. All pixel runs before the current
		run are checked to see if the first row value is equal
		to or greater than that of the current run. If equal,
		then the current run is a duplicate of that run; if
		greater, then the data for the found pixel run occurs
		before that of the (lower-numbered) current run.
	*/
	// VARIABLES
	const UINT08 *q=current_column;
	// PROCESS
	while (q<current_run)
	{
		// VARIABLES
		size_t first_row,num_pixels;
		// GET FIRST ROW (AT START OF CURRENT RUN)
		first_row=(size_t)(*q);
		// PERFORM THE CHECK
		if (first_row>=(*current_run))
		{
			// Found duplicate or out-of-order row number. This
			// shouldn't prevent the picture from being
			// displayed, but it could indicate trouble
			// elsewhere in the input file.
			return TRUE;
		}
		// POINT AT NUMBER OF PIXELS
		q++;
		// GET NUMBER OF PIXELS
		num_pixels=(size_t)(*q);
		// POINT AT LAST DATA BYTE IN THIS RUN
		q+=(num_pixels+GRAPHIC_DUMMY_PIXEL_COUNT);
		// POINT AT NEXT PIXEL RUN
		q++;
	}
	// RETURN
	return FALSE;
}

// Check current column for all three problems
static bool_t ColumnAnyPixelRunProblems (const UINT08 *current_column)
{
	// DOCUMENTATION
	/*
		Find duplicate, out-of-order pixels or consecutive
		pixel runs in the column and stop if one is found.
	*/
	// VARIABLES
	const UINT08 *p;
	const UINT08 *first_row_ptr;
	size_t run_number,prev_first_row,prev_num_pixels;
	bool_t in_consecutive_sticky=FALSE;							// have already found at least one consecutive pixel run
	bool_t in_consecutive_current=FALSE;						// are in a consecutive pixel run group *right now*
	size_t current_consecutive_group=0;							// count (so far) of groups of consecutive pixel runs
	size_t current_consecutive_run=0;							// count (so far) of consecutive pixel runs in current group
	size_t current_consecutive_start=-1;						// last-found start row of consecutive pixel run
	size_t current_consecutive_count=0;							// count (so far) of (consecutive) pixels in current group
	// INITIALISE CONSECUTIVE RUN DETECTION LOGIC
	run_number=0;
	// GET POINTER TO PIXEL RUN LIST
	p=current_column;
	// INITIALISE POINTER TO CURRENT FIRST ROW
	first_row_ptr=NULL;
	// WE KNOW THAT THE FIRST VALUE IN THE PIXEL RUN IS
	// WITHIN THE PICTURE EXTENT, SINCE THE POINTER ARRAY
	// HAS ALREADY BEEN CHECKED, SO VALIDATE THE REST.
	while (
	       // After column rebuild by CleanWAD there will
	       // be only one group of consecutive pixel runs,
	       // if any at all, and it will only have two
	       // runs. CleanWAD always merges consecutive
	       // pixel runs. It only splits once and even
	       // then only if a pixel run size is > 255. It
	       // never splits twice for that reason as this
	       // would imply a start row number > 254
	       // (0-based) which is impossible in the doom
	       // picture format. We therefore use these
	       // constraints to optimize out any validation
	       // work that occurs after a point by which we
	       // know we would have to rebuild anyway.
	       (current_consecutive_group<2) &&						// not processed same way as CleanWAD or else wrong to begin with
	       (current_consecutive_run<3)							// not processed same way as CleanWAD or else wrong to begin with
	      )
	{
		// VARIABLES
		size_t first_row,num_pixels;
		//
		// ALREADY POINTING AT NEXT PIXEL RUN/END-OF-COLUMN
		// WHEN THIS LOOP STARTS; THAT IS HOW LOOP IS SET
		// UP BEFORE THE FIRST ITERATION AND DURING ALL.
		//
		// IF END OF COLUMN THEN DONE
		if (*p==(UINT08)255)
		{
			break;
		}
		// OK, FOUND ANOTHER RUN
		run_number++;
		// GET FIRST ROW
		first_row=(size_t)(*p);
		// KEEP POINTER TO CURRENT FIRST ROW
		first_row_ptr=p;
		// POINT AT NUMBER OF PIXELS
		p++;
		// GET NUMBER OF PIXELS
		num_pixels=(size_t)(*p);
		// POINT AT LAST DATA BYTE IN THIS RUN
		p+=(num_pixels+GRAPHIC_DUMMY_PIXEL_COUNT);
		// CHECK FOR DUPLICATES OR DISORDERED VALUES.
		if (PixelRunDuplicatesOrDisorder(current_column,first_row_ptr))
		{
			return TRUE;
		}
		// CHECK FOR CONSECUTIVE RUNS (SHOULD BE ONE RUN)
		// OR PREVIOUS RUN ENDING IN MIDDLE OF THIS ONE
		// (Z-CONFUSION: WHAT SHOULD GET PASTED OVERALL?).
		//
		// THIS CHECK COULD RETURN A FALSE NEGATIVE IF ANY
		// DISORDERED OR DUPLICATE ROW NUMBERS EXIST IN THE
		// COLUMN. THIS IS NOT IN FACT A PROBLEM, BECAUSE
		// THE LOGIC OF THE CHECK FOR THAT SITUATION IS NOT
		// DEPENDENT ON THIS CHECK AND IF DISORDERED OR
		// DUPLICATE ROW NUMBERS ARE DETECTED AT ANY POINT,
		// THEY WILL HAVE BEEN FLAGGED BY NOW ANYWAY.
		//
		if (run_number==1)
		{
			// NOTHING TO COMPARE, JUST INITIALISE CONTEXT
			prev_first_row=first_row;
			prev_num_pixels=num_pixels;
//			// CAN'T BE IN A CONSECUTIVE RUN RIGHT NOW
//			in_consecutive_current=FALSE;
		}
		else if ((prev_first_row+prev_num_pixels)>first_row)
		{
			// ONE RUN ENDS IN MIDDLE OF ANOTHER. THIS
			// REQUIRES A REBUILD, BECAUSE IT MAKES THE
			// RENDERING SEQUENCE POTENTIALLY AMBIGUOUS.
			return TRUE;
//			// CAN'T BE IN A CONSECUTIVE RUN RIGHT NOW
//			in_consecutive_current=FALSE;
		}
		else if ((prev_first_row+prev_num_pixels)==first_row)
		{
			// CONSECUTIVE PIXEL RUNS
			if (!in_consecutive_current)
			{
				current_consecutive_group++;					// a new group of consecutive pixel runs
				current_consecutive_run=1;						// account for the previous pixel-run
				current_consecutive_count=prev_num_pixels;		// account for the previous pixel run
				in_consecutive_sticky=TRUE;						// it may already BE true of course
				in_consecutive_current=TRUE;					// now in a sequence of consecutive pixel runs
			}
			current_consecutive_run++;
			current_consecutive_count+=num_pixels;
			current_consecutive_start=first_row;
//			// MUST BE IN A CONSECUTIVE RUN RIGHT NOW
//			in_consecutive_current=TRUE;
		}
		else
		{
			// NO ACTION NEEDED, JUST MAINTAIN CONTEXT
			prev_first_row=first_row;
			prev_num_pixels=num_pixels;
/* e */		// CAN'T BE IN A CONSECUTIVE RUN RIGHT NOW
/* e */		in_consecutive_current=FALSE;
		}
		// POINT AT NEXT PIXEL RUN OR END-OF-COLUMN
		p++;
		// DO NOT ADJUST BYTE COUNT AS THAT IS ALWAYS DONE
		// AT THE START OF THE NEXT ITERATION OF THE LOOP.
	}
	// UNDO ERROR FLAG IF ALLOWED CONSECUTIVE
	if (
	    (in_consecutive_sticky) &&								// found consecutive pixel runs
	    (current_consecutive_group==1) &&						// but only one group of them
	    (current_consecutive_run==2) &&							// and that has only two pixel runs
	    (current_consecutive_start==GRAPHIC_MAX_FIRST_ROW) &&	// which are split at the boundary
	    (current_consecutive_count>GRAPHIC_MAX_PIXEL_RUN_LENGTH)// and only because originally one overlong pixel run
	   )
	{
		in_consecutive_sticky=FALSE;
	}
	// STOP IF FOUND AN ERROR
	if (in_consecutive_sticky)
	{
		return TRUE;
	}
	// RETURN
	return FALSE;
}

// Initialise column render buffer
static void InitialiseColumnRenderBuffer (UINT16 *render_buffer, size_t render_buffer_size)
{
	// VARIABLES
	size_t row;
	// PROCESS
	for (row=0;row<render_buffer_size;row++)
	{
		render_buffer[row]=0x8000;
	}
}

// Render a column
static void RenderColumn (const UINT08 *column, UINT16 *render_buffer, size_t render_buffer_size, const char *picturename)
{
	// VARIABLES
	size_t total_pixels=0;
	const UINT08 *p=column;
	// PROCESS
	while (*p!=(UINT08)255)
	{
		// VARIABLES
		size_t first_row,num_pixels,pixel_num;
		UINT16 *row_ptr;
		// GET FIRST ROW
		first_row=(size_t)(*p++);
		// GET NUMBER OF PIXELS
		num_pixels=(size_t)(*p++);
		total_pixels+=num_pixels;
		// SKIP OVER FIRST DUMMY PIXEL TO FIRST REAL PIXEL
		p++;
		// COPY PIXEL DATA
		row_ptr=&render_buffer[first_row];
		for (pixel_num=0;pixel_num<num_pixels;pixel_num++)
		{
			// PARANOID ERROR CHECK
			if (row_ptr>&render_buffer[render_buffer_size-1])
			{
				FatalAbort(1,__FILE__,__LINE__,"Logic failure 002 during rebuild of picture %s",picturename);
			}
			// COPY THE PIXEL
			*row_ptr++=(((UINT16)(*p++)) & 0x00FF);
		}
		// SKIP OVER SECOND DUMMY PIXEL
		p++;
		// NOW AT START OF NEXT PIXEL RUN,
		// OR MAYBE AN END-OF-COLUMN FLAG.
	}
}

// Find column depth (from row 0 in lines)
static size_t FindColumnDepth (const UINT16 *render_buffer, size_t render_buffer_size)
{
	// VARIABLES
	size_t row_offset=render_buffer_size-1;
	// PROCESS
	while ( (row_offset>0) && (render_buffer[row_offset]==0x8000) )
	{
		row_offset--;
	}
	if ( (row_offset==0) && (render_buffer[row_offset]==0x8000) )
	{
		return 0;
	}
	else
	{
		return row_offset+1;
	}
}

//
// Used to decide where to split a pixel run if its length is
// too big to fit the length word (which, in the DOOM picture
// format, is an unsigned byte).
//
// Most WAD tools that can do this at all appear to use 254.
// This isn't surprising, as 254 is the maximum run-start row
// number (zero-based) supported by the DOOM picture format.
//
#define RUN_SPLIT_THRESHOLD 254

// PARANOID ERROR CHECK
#if (RUN_SPLIT_THRESHOLD<2) || (RUN_SPLIT_THRESHOLD>255)
#error "Pixel run split threshold must be between 2 and 255"
#endif

// Recode column data
static void RecodeColumnData (UINT08 *column_data, const UINT16 *render_buffer, size_t column_depth, const char *picturename)
{
	// VARIABLES
	size_t row=0;
	const UINT16 *in_ptr=&render_buffer[0];
	UINT08 *out_ptr=&column_data[0];
	// PROCESS
	while (row<column_depth)
	{
		// VARIABLES
		UINT08 pixel_count;
		UINT08 *pixel_count_ptr;
		// SKIP TO FIRST/NEXT RENDERED ROW
		while (*in_ptr==0x8000)
		{
			if (row>GRAPHIC_MAX_FIRST_ROW)
			{
				FatalAbort(1,__FILE__,__LINE__,"Logic failure 003 during rebuild of picture %s",picturename);
			}
			if (row>column_depth-1)
			{
				FatalAbort(1,__FILE__,__LINE__,"Logic failure 004 during rebuild of picture %s",picturename);
			}
			row++;
			in_ptr++;
		}
		// PARANOID ERROR CHECK
		if (row>GRAPHIC_MAX_FIRST_ROW)
		{
			FatalAbort(1,__FILE__,__LINE__,"Logic failure 005 during rebuild of picture %s",picturename);
		}
		// SET CURRENT ROW NUMBER
		*out_ptr=(UINT08)(row & 0xFF);
		// GET POINTER TO PIXEL COUNT
		out_ptr++;
		pixel_count_ptr=out_ptr;
		// CREATE FIRST DUMMY PIXEL
		out_ptr++;
		*out_ptr=(UINT08)0;
		// COPY PIXELS
		out_ptr++;
		pixel_count=0;
		while ((*in_ptr)!=0x8000)
		{
			pixel_count++;		// number of pixel that we are ***ABOUT*** to write out but haven't done yet
			if (pixel_count>RUN_SPLIT_THRESHOLD)
			{
				// CLOSE OFF CURRENT PIXEL RUN
				*pixel_count_ptr=pixel_count-1;
				*out_ptr++=(UINT08)0;
				// START NEW PIXEL RUN
				*out_ptr++=(UINT08)row;
				// REINITIALISE PIXEL COUNT
				pixel_count=1;
				pixel_count_ptr=out_ptr;
				out_ptr++;
				// WRITE FIRST DUMMY PIXEL
				*out_ptr++=(UINT08)0;
			}
			if (row>column_depth-1)
			{
				FatalAbort(1,__FILE__,__LINE__,"Logic failure 007 during rebuild of picture %s",picturename);
			}
			*out_ptr=(UINT08)((*in_ptr) & 0xFF);
			////////////////////////////////////
			*out_ptr++;
			row++;
			in_ptr++;
		}
		// CREATE TRAILING DUMMY PIXEL
		*out_ptr++=(UINT08)0;
		// CREATE PIXEL RUN COUNT
		*pixel_count_ptr=pixel_count;
	}
	*out_ptr=(UINT08)255;
}

// Rebuild any columns in a picture that require it
static void RebuildPixelColumns (void *in, const char *picturename, size_t num_columns, bool_t process_only)
{
	// DOCUMENTATION
	/*
		The picture and each of its pixel columns must have
		already been checked for all other inconsistencies
		before this function is called. THIS IS NOT CHECKED!

		Because of the way in which pixel runs could overlap
		or appear out-of-order, the only sensible way to fix
		them is to render the column the way DOOM would and
		then re-code the column back into pixel-run format.

		Note that the maximum picture height (column depth) is
		509 pixels. Row 255 is a "rogue value" meaning "end of
		column", so the maxium row number is 254. The highest
		physical row number in a run (1-based) is therefore 255
		and this gives a maximum of 254 rows before the run,
		plus 255 bytes in the run and 254+255==509.

		There is no need to allocate fresh memory, because the
		rebuild cannot GROW the physical column data; it can
		only shrink it or leave it at the same size.
	*/

	// METRICS
	#define MAX_COLUMN_DEPTH 509

	// VARIABLES
	const UINT32 *inpointers=(UINT32*)in+(sizeof(picture_header_t)/sizeof(UINT32));
	size_t column,rebuilt_columns;

	// INITIALISE STATISTICS
	rebuilt_columns=0;

	// CHECK AND IF NEEDED THEN REBUILD THE COLUMN
	for (column=0;column<num_columns;column++)
	{
		// VARIABLES
		size_t column_depth;
		UINT16 render_buffer[MAX_COLUMN_DEPTH];
		UINT08 *column_data=(UINT08*)in+inpointers[column];
		// SKIP IF NO ERRORS IN COLUMN
		if (!ColumnAnyPixelRunProblems(column_data))
		{
			continue;
		}
		// ACCOUNT FOR COLUMN TO BE REBUILT
		rebuilt_columns++;
		// ISSUE DIAGNOSTIC IF REQUIRED
		if (!process_only)
		{
			EventState(VERBOSITY_DETAILS,"Rebuilding column %ld",column);
		}
		// INITIALISE COLUMN RENDER BUFFER
		InitialiseColumnRenderBuffer(&render_buffer[0],MAX_COLUMN_DEPTH);
		// RENDER THE COLUMN
		RenderColumn(column_data,&render_buffer[0],MAX_COLUMN_DEPTH,picturename);
		// FIND COLUMN DEPTH
		column_depth=FindColumnDepth(&render_buffer[0],MAX_COLUMN_DEPTH);
		// REBUILD IS TRIVIAL IF COLUMN DEPTH IS ZERO
		if (column_depth==0)
		{
			*column_data=(UINT08)255;
			continue;
		}
		// RECODE THE COLUMN DATA
		RecodeColumnData (column_data,&render_buffer[0],column_depth,picturename);
	}

	// LOG FULL COLUMN REBUILDS
	if (rebuilt_columns==0)
	{
		FatalAbort(1,__FILE__,__LINE__,"Logic failure 008 during rebuild of picture %s",picturename);
	}
	if (!process_only)
	{
		EventState(VERBOSITY_WARNINGS,"Picture %s required %ld columns (of %ld) to be rebuilt",picturename,rebuilt_columns,num_columns);
	}
}

#define PICTURE_INCREMENT 1024

// Optimize a picture
static void OptimizePicture (
	const void *in,
	void **out,
	size_t in_count,
	bool_t pack_pictures,
	bool_t unpack_pictures,
	bool_t process_only,
	size_t *size_out)
//	bool_t *success)
{
	// DOCUMENTATION
	/*
		This does not need a "rebuild_pictures" parameter as it
		always does at least a rebuild. It may also pack or
		unpack, according to the other two blockmap options.
	*/

	// VARIABLES
	const UINT32 *inpointers=(UINT32*)in+(sizeof(picture_header_t)/sizeof(UINT32));
	void *out_pcopy;
	UINT32 *outpointers;
	size_t num_columns,i,out_limit,out_count;

	// PRIMARY ERROR CHECK
	if (pack_pictures && unpack_pictures)
	{
		FatalAbort(1,__FILE__,__LINE__,"Contradictory picture processing flags should have been trapped by now");
	}

	// INITIALISE
	num_columns=(size_t)(RDLend16SV(((picture_header_t*)in)->width));
	out_pcopy=*out;
	out_limit=in_count;
//	out_count=0;

	// COPY OUT HEADER AND POINTER ARRAY
	out_count=sizeof(picture_header_t)+(num_columns*sizeof(UINT32));
	if (out_count>out_limit)
	{
		void *newptr;
		size_t sizeb,incrb;
		/*****************/
		sizeb=out_count+PICTURE_INCREMENT;
		incrb=sizeb-out_limit;
		newptr=realloc(out_pcopy,sizeb);
		if (newptr==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not grow picture pointer list by %ld bytes (from %ld bytes)",incrb,out_limit);
		}
		out_pcopy=newptr;
		out_limit=sizeb;
	}
	(void) memcpy(out_pcopy,in,out_count);

	// SET UP (C) POINTERS INTO BOTH (DOOM) POINTER ARRAYS
	/*****************************************************
	Note that some of the pointer arithmetic, like DOOM
	itself, relies on the picture header size being a
	multiple of the pointer size. This is because DOOM
	picture pointers must also be pointed to. Although,
	unlike blockmap pointers, they refer to bytes and hold
	byte offsets, they themselves are four-byte integers.
	******************************************************/
//	inpointers=(UINT32*)in+(sizeof(picture_header_t)/sizeof(UINT32));
	outpointers=(UINT32*)out_pcopy+(sizeof(picture_header_t)/sizeof(UINT32));

	// PROCESS EACH COLUMN
	for (i=0;i<num_columns;i++)
	{
		// VARIABLES
		bool_t match;

		// ASSUME NO MATCH FOUND WHEN STARTING EACH COLUMN
		match=FALSE;

		// FIND AND COPY DUPLICATE POINTERS UNLESS UNPACKING;
		// I.E., MUST DO THIS IF REBUILDING AND/OR PACKING.
		if (!unpack_pictures)
		{
			/**************************************************
			Search the input pointer array up to just before
			this position. If we find one whose value is the
			same as the one in the current position, then the
			corresponding one in the output array can share the
			same data, so overwrite it with the one that was
			found and also make a note that a match WAS found.

			Note that we do not need to byte-swap, even if the
			host machine is big-endian, as we are only testing
			pointers for equality and possible moving "as is".
			**************************************************/
			size_t j;
			UINT32 inpointers_i;
			/******************/
			inpointers_i=inpointers[i];
			for (j=0;j<i;j++)
			{
				if (inpointers_i==inpointers[j])
				{
					// Input pointer j may have been matched by
					// a DATA match, earlier in the pointer
					// array, that was not picked up when the
					// picture was created but was detected in
					// a previous iteration of the parent loop.
					// In this case, the output pointer j will
					// have already been changed to one from
					// earlier on in the blockmap (which may
					// itself be a changed pointer). Since they
					// have the same data, we use the earlier
					// one (if any; if none, then it would not
					// have been changed anyway). In this way,
					// we always use the same pointers for each
					// identical column, irrespective of
					// whether we check for existing pointer
					// reuse when packing or not. This is
					// vital, as it makes the packing
					// deterministic under all circumstances.
					outpointers[i]=outpointers[j];
					match=TRUE;
					break;
				}
			}
		}

		// FIND POINTERS THAT ARE NOT DUPLICATES BUT SHOULD BE;
		// REPLACE EACH SUCH WITH POINTER TO IDENTICAL DATA.
		if (pack_pictures && !match)
		{
			size_t j;
			UINT08 *curr_column;
			/******************/
			curr_column=(UINT08*)in+RDLend32UV(inpointers[i]);
			for (j=0;j<i;j++)
			{
				// Initialised variables
				UINT32 prev_pointer=0;
				// Uninitialised variables
				UINT32 curr_pointer;
				// Cache the current DOOM pointer in question
				curr_pointer=outpointers[j];
				// If not same pointer then see if same data
				if (prev_pointer!=curr_pointer)
				{
					// Variables
					UINT08 *p_in,*p_out;
					bool_t identical;
					// C pointer of current DOOM output pointer
					p_out=(UINT08*)out_pcopy+RDLend32UV(curr_pointer);
					// C pointer to current DOOM input pointer
					p_in=curr_column;
					// Assume identical until found otherwise
					identical=TRUE;
					// Compare columns same way as DOOM strings
					while (TRUE)
					{
						// Variables
						size_t pixel_ctr,num_pixels;
						// Check if "row number" is
						// an end of column marker.
						if (*p_in==(UINT08)255)
						{
							// Hit end of column. Always break
							// at end of column whether same or
							// not so far. Must also check if
							// both end here; if not then
							// columns have different lengths
							// and are thus not identical.
							if (*p_out!=(UINT08)255)
							{
								// One column ends here,
								// but the other does not,
								// so they have different
								// lengths and therefore
								// different contents, so
								// they can't be a match.
								identical=FALSE;
							}
							break;
						}
						// Check the row numbers
						if (*p_out++!=*p_in++)
						{
							identical=FALSE;
							break;
						}
						// Same row as current column,
						// get number of pixels in run.
						num_pixels=(size_t)(*p_out);
						// Same number in both runs?
						if (num_pixels!=(size_t)(*p_in))
						{
							// Different lengths mean different
							// contents, so not identical.
							identical=FALSE;
							break;
						}
						//
						// Current output run and this output
						// run start at same row and have same
						// number of pixels. Compare contents.
						//
						// STEP TO FIRST PIXEL; BY TWO, AS
						// WE IGNORE DUMMY PIXEL IN EACH RUN.
						p_in+=2;
						p_out+=2;
						// COMPARE RUNS
						for (pixel_ctr=0;pixel_ctr<num_pixels;pixel_ctr++)
						{
							if (*p_out++!=*p_in++)
							{
								identical=FALSE;
								// Only breaks this for loop,
								// hence the "if(!identical)"
								// just after this for loop.
								break;
							}
						}
						// IF NOT IDENTICAL THEN DONE THIS
						if (!identical)
						{
							// Break this while loop if we
							// found a mismatching pixel.
							break;
						}
						// STEP OVER DUMMY PIXEL AT END OF RUN
						p_out++;
						p_in++;
					}
					if (identical)
					{
						if (!process_only)
						{
							EventState(VERBOSITY_INTERNALS,"Picture block %u is identical to block %u",i,j);
						}
						outpointers[i]=outpointers[j];
						match=TRUE;
						break;
					}
				}
				// Make note of previous pointer checked
				prev_pointer=curr_pointer;
			}
		}

		// IF THIS POINTER IS NOT DUPLICATED BY ANOTHER POINTER,
		// COPY ITS DATA OUT, AS IT IS DISTINCT FROM OTHER DATA.
		if (!match)
		{
			// VARIABLES
			UINT08 *p_in,*p_out;
			// C POINTER EQUIVALENT OF INPOINTERS[I]
			p_in=(UINT08*)in+inpointers[i];
			// C POINTER TO NEXT AVAILABLE DATA SPACE
			p_out=(UINT08*)out_pcopy+out_count;
			// SET DOOM POINTER CORRESPONDING TO THAT SPACE
			outpointers[i]=(UINT32)(out_count);
			// COPY THE DATA ONE ONE RUN AT A TIME
			while (TRUE)
			{
				// Variables
				size_t pixel_ctr,num_pixels;
				// ENSURE ENOUGH SPACE FOR AT LEAST A PIXEL
				// RUN HEADER OR THE END-OF-COLUMN MARKER.
				if (out_count+GRAPHIC_PIXEL_RUN_HEADER_SIZE>out_limit)
				{
					// VARIABLES
					void *newptr;
					size_t sizeb,incrb;
					// REALLOCATE THE OUPUT DATA SPACE
					sizeb=out_count+GRAPHIC_PIXEL_RUN_HEADER_SIZE+PICTURE_INCREMENT;
					incrb=sizeb-out_limit;
					newptr=realloc(out_pcopy,sizeb);
					if (newptr==NULL)
					{
						ErrorAbort(1,__FILE__,__LINE__,"Could not grow picture pointer list by %ld bytes (from %ld bytes)",incrb,out_limit);
					}
					out_pcopy=newptr;
					out_limit=sizeb;
					// UPDATE C POINTER TO DOOM POINTER ARRAY
					outpointers=(UINT32*)out_pcopy+(sizeof(picture_header_t)/sizeof(UINT32));
					// UPDATE C POINTER TO OUTPUT DATA SPACE
					p_out=(UINT08*)out_pcopy+out_count;
				}
				// IF END-OF-COLUMN THEN SET MARKER
				// IN ORDER TO FINISH THIS COLUMN.
				if (*p_in==(UINT08)255)
				{
					out_count++;
					*p_out=(UINT08)255;
					break;
				}
				// COPY PIXEL RUN OFFSET MEMBER
				out_count++;
				*p_out++=*p_in++;
				// GET NUMBER OF PIXELS IN RUN
				// INCLUDING THE DUMMY PIXELS.
				num_pixels=(size_t)(*p_in+(UINT08)GRAPHIC_DUMMY_PIXEL_COUNT);
				// COPY PIXEL COUNT MEMBER
				out_count++;
				*p_out++=*p_in++;
				// ENSURE ENOUGH SPACE FOR THE PIXEL RUN
				while (out_count+num_pixels>out_limit)
				{
					// VARIABLES
					void *newptr;
					size_t sizeb,incrb;
					// REALLOCATE THE OUPUT DATA SPACE
					sizeb=out_count+num_pixels+PICTURE_INCREMENT;
					incrb=sizeb-out_limit;
					newptr=realloc(out_pcopy,sizeb);
					if (newptr==NULL)
					{
						ErrorAbort(1,__FILE__,__LINE__,"Could not grow picture pointer list by %ld bytes (from %ld bytes)",incrb,out_limit);
					}
					out_pcopy=newptr;
					out_limit=sizeb;
					// UPDATE C POINTER TO DOOM POINTER ARRAY
					outpointers=(UINT32*)out_pcopy+(sizeof(picture_header_t)/sizeof(UINT32));
					// UPDATE C POINTER TO OUTPUT DATA SPACE
					p_out=(UINT08*)out_pcopy+out_count;
				}
				// COPY THE PIXEL RUN TSELF
				out_count+=num_pixels;
				for (pixel_ctr=0;pixel_ctr<num_pixels;pixel_ctr++)
				{
					*p_out++=*p_in++;
				}
			}
		}
	}

	// RETURN RESULTS
	*out=out_pcopy;
	*size_out=out_count;
}

/**********************************************************************************************************************************/
/********************************************** WAD Entry Copy Out with Optimization **********************************************/
/**********************************************************************************************************************************/

// Check for invalid blockmap block pointers
static bool_t ValidateBlockmapBlockPointers (
	const char *map_name,
	size_t lump_length,
	size_t numbers_length,
	UINT16 *inpointers,
	size_t num_blocks,
	bool_t process_only)
{
	// VARIABLES
	size_t i;
	// CHECK FOR VALID POINTER ARRAY
	for (i=0;i<num_blocks;i++)
	{
//
//		A 1-block map
//		Length(BYTES)=15
///     Numbers_Length(INTS)==RoundDown(15/2)==7
//		0 1 2 3 4 5 6 7
//	    012345678901234
//		H_H_H_H_0500FF
//
//		inpointers[0]=5 --> ok
//		inpointers[0]=6 --> bad: no room for minimum linedef list size
//		inpointers[0]>6 --> bad: points outside the lump
//
		if ((size_t)inpointers[i]==(size_t)(numbers_length-1))
		{
			if (!process_only)
			{
				EventState(VERBOSITY_DETAILS,"Block pointer (%ld) linedef list would end outside blockmap extent (%ld)",i,lump_length);
				EventState(VERBOSITY_WARNINGS,"Blockmap from %s has an invalid pointer array -- not processed",map_name);
			}
			return FALSE;
		}
		if ((size_t)inpointers[i]>(size_t)(numbers_length-1))
		{
			if (!process_only)
			{
				EventState(VERBOSITY_DETAILS,"Block pointer (%ld) points outside blockmap extent (%ld)",i,lump_length);
				EventState(VERBOSITY_WARNINGS,"Blockmap from %s has an invalid pointer array -- not processed",map_name);
			}
			return FALSE;
		}
	}
	// RETURN
	return TRUE;
}

// Check for invalid blockmap block extents
static bool_t ValidateBlockmapBlockExtents (
	const void *lump_ptr,
	const char *map_name,
	size_t lump_length,
	size_t numbers_length,
	const UINT16 *inpointers,
	size_t num_blocks,
	size_t num_linedefs,
	bool_t process_only)
{
	// VARIABLES
	size_t i;
	// CHECK FOR VALID LINEDEF LISTS
	for (i=0;i<num_blocks;i++)
	{
		// VARIABLES
		SINT16 *p;
		size_t numbers_scanned;

		// GET POINTER TO FIRST NUMBER IN THE LINEDEF LIST
		p=(SINT16*)lump_ptr+inpointers[i];

		// GET NUMBER COUNT CORRESPONDING TO POINTER
		numbers_scanned=(size_t)inpointers[i]+1;

		//
		// POINTER P NOW POINTS AT THE FIRST NUMBER IN THE LIST
		// AND NUMBER_SCANNED IS NOW THE COUNT OF NUMBERS IN THE
		// BLOCK INCLUDING THE ONE POINTED AT BY P AND ALSO ANY
		// NUMBERS BEFORE IT (PARTS OF POINTERS, HEADER, ETC).
		//

		// WE KNOW THAT THE FIRST VALUE IN THE LINEDEF LIST IS
		// WITHIN THE BLOCKMAP EXTENT, SINCE THE POINTER ARRAY
		// HAS ALREADY BEEN CHECKED; HOWEVER, THIS VALUE SHOULD
		// IN FACT BE A DUMMY VALUE OF ZERO, SO CHECK IT HERE.
		if (*p!=0)
		{
			if (!process_only)
			{
				EventState(VERBOSITY_DETAILS,"Invalid first value (%hd) in block %ld (value should be zero)",*p,i);
				EventState(VERBOSITY_WARNINGS,"Blockmap from %s includes an invalid linedef list -- not processed",map_name);
			}
			return FALSE;
		}

		// SKIP OVER THE DUMMY VALUE
		//
		// NOTE THAT EVEN IF THIS IS AN EMPTY BLOCK, WE STILL
		// COUNT THE TERMINATOR FLAG (-1) AS PART OF THE BLOCK,
		// WHICH IS WHY WE UNCONDITIONALLY INCREMENT THE COUNT.
		//
		numbers_scanned++;
		p++;

		// IF TERMINATOR THEN ENSURE IS WITHIN BLOCKMAP EXTENT
		//
		// MUST DO THIS HERE BECAUSE THE LOOP MUST NOT START IF
		// THIS IS THE TERMINATOR; TREAT IT AS A SPECIAL CASE.
		//
		if (*p==(-1))
		{
			if (numbers_scanned>numbers_length)
			{
				if (!process_only)
				{
					EventState(VERBOSITY_DETAILS,"Linedef list of block (%ld) runs outside blockmap extent (%ld)",i,lump_length);
					EventState(VERBOSITY_WARNINGS,"Blockmap from %s includes an invalid linedef list -- not processed",map_name);
				}
				return FALSE;
			}
		}

		// VALIDATE THE REST OF THE LINEDEF LIST
		while (*p!=(-1))
		{
			// VARIABLES
			SINT16 v;

			// ENSURE THAT IT IS WITHIN THE BLOCKMAP EXTENT
			if (numbers_scanned>numbers_length)
			{
				if (!process_only)
				{
					EventState(VERBOSITY_DETAILS,"Linedef list of block (%ld) runs outside blockmap extent (%ld)",i,lump_length);
					EventState(VERBOSITY_WARNINGS,"Blockmap from %s includes an invalid linedef list -- not processed",map_name);
				}
				return FALSE;
			}

			// GET THE VALUE
			v=*p;

			// CHECK FOR INVALID VALUES
			if (v<(-1))
			{
				if (!process_only)
				{
					EventState(VERBOSITY_DETAILS,"Invalid linedef number (%hd) in block %ld (value is less than -1)",v,i);
					EventState(VERBOSITY_WARNINGS,"Blockmap from %s includes an invalid linedef list -- not processed",map_name);
				}
				return FALSE;
			}
			if ( (v>(-1)) && (num_linedefs!=0) &&  ((size_t)(v)>(num_linedefs-1)) )
			{
				if (!process_only)
				{
					EventState(VERBOSITY_DETAILS,"Invalid linedef number (%hd) in block %ld (value is greater than highest linedef number (%ld))",v,i,(num_linedefs-1));
					EventState(VERBOSITY_WARNINGS,"Blockmap from %s has invalid linedef list -- not processed",map_name);
				}
				return FALSE;
			}

			// POINT TO NEXT IN BLOCK (OR END-OF-BLOCK FLAG)
			numbers_scanned++;
			p++;
		}
	}
	// RETURN
	return TRUE;
}

// Copy out a blockmap (optimized)
static void CheckAndProcessBlockmap (
	const void *lump_in,
	const WADDIR_t *WADDIR,
	size_t index,
	bool_t pack_blockmaps,
	bool_t unpack_blockmaps,
	bool_t flush_blockmaps,
	bool_t maintain_blockmaps,
	bool_t process_only,
	void **lump_out,
	size_t *size_out)
{
	// DOCUMENTATION
	/*
		In a BLOCKMAP, the data numbers and the internal
		pointers are all 16-bit (2-byte) integers, some signed
		and some unsigned. Therefore this function ASSUMES
		that (sizeof(UINT16)==sizeof(SINT16)). This assumption
		ought to be true by definition and so is not checked.
	*/

	// VARIABLES
	void *lump_out_temp;
	char mapname[9];
	UINT16 *inpointers;
	const WADDIRentry_t *entry;
	blockmap_header_t *header;
	bool_t success,must_rebuild_blocks,is_resized,do_writeit;
	size_t mapnameindex,linedefsindex,num_blocks,num_linedefs,numbers_length,size_out_temp;

	// PRIMARY ERROR CHECK
	if (pack_blockmaps && unpack_blockmaps)
	{
		FatalAbort(1,__FILE__,__LINE__,"Contradictory blockmap processing flags should have been trapped by now");
	}
	if (lump_in==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a blockmap with NULL lump_in argument");
	}
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a blockmap with NULL directory argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a blockmap with NULL directory entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (index>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a blockmap with entry index (%ld) greater directory entry count (%ld)",index,WADDIR->count);
	}

	// INITIALISE
	size_out_temp=WADDIR->entries[index].length;
	lump_out_temp=malloc(size_out_temp);
	if (lump_out_temp==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for blockmap buffer",size_out_temp);
	}
	(void) memmove(lump_out_temp,lump_in,size_out_temp);
	*lump_out=lump_out_temp;
	*size_out=size_out_temp;
	must_rebuild_blocks=FALSE;

	// FIND THE MAP NAME AND LINEDEFS FOR THIS BLOCKMAP
	if (index==0)
	{	// Should never happen -- parse phase should have caught this by now
		FatalAbort(1,__FILE__,__LINE__,"Corrupt map sequence -- BLOCKMAP entry with no map name header");
	}
	mapnameindex=index-1;
	linedefsindex=0;
	while ( (mapnameindex>0) && (WADDIR->entries[mapnameindex].type==TYPE_MAPDATA) )
	{
		if (wnmatch(WADDIR->entries[mapnameindex].name,"LINEDEFS"))
		{
			linedefsindex=mapnameindex;
		}
		mapnameindex--;
	}
	if (WADDIR->entries[mapnameindex].type!=TYPE_MAPNAME)
	{	// Should never happen -- parse phase should have caught this by now
		FatalAbort(1,__FILE__,__LINE__,"Corrupt map sequence -- BLOCKMAP entry with no map name header");
	}
	(void) strcpy(mapname,WADDIR->entries[mapnameindex].name);

	// GET NUMBER OF LINEDEFS IN THIS MAP
	num_linedefs=0;
	if (linedefsindex!=0)
	{
		num_linedefs=(WADDIR->entries[linedefsindex].length/sizeof(linedef_t));
	}

	// CREATE POINTER TO THE DIRECTORY ENTRY OF THIS BLOCKMAP
	entry=&WADDIR->entries[index];

	// STATEMENT
	if (!process_only)
	{
		EventState(VERBOSITY_DETAILS,"Processing blockmap from %s",mapname);
	}

	// IF NOT A VALID BLOCKMAP (CHECK 1) THEN DO NOT MODIFY IT
	if (entry->length<sizeof(blockmap_header_t))
	{
		if (!process_only)
		{
			EventState(VERBOSITY_WARNINGS,"Blockmap from %s is too short -- not processed",mapname);
		}
		return;
	}

	// WARN IF LENGTH IS NOT A MULTIPLE OF BLOCKMAP NUMBER SIZE
	if ((entry->length%sizeof(UINT16))!=0)
	{
		if (!process_only)
		{
			EventState(VERBOSITY_WARNINGS,"Blockmap from %s has size (%ld) that is not a multiple of %ld",mapname,entry->length,sizeof(UINT16));
		}
	}

	// CREATE POINTER TO THE BLOCKMAP HEADER
	header=(blockmap_header_t*)lump_in;

	// GET NUMBER OF BLOCKS IN BLOCKMAP
	num_blocks=(size_t)(RDLend16UV(header->rows)*RDLend16UV(header->columns));

	// IF NOT A VALID BLOCKMAP (CHECK 2) THEN DO NOT MODIFY IT
	/***********************************************************
	If header + pointer array is same size or bigger than the
	entry, then there is no room for the linedef lists, so this
	is not a valid BLOCKMAP.
	***********************************************************/
	if (sizeof(blockmap_header_t)+(num_blocks*sizeof(UINT16))>=(entry->length))
	{
		if (!process_only)
		{
			EventState(VERBOSITY_WARNINGS,"Blockmap from %s has an invalid header -- not processed",mapname);
		}
		return;
	}

	// CREATE POINTER TO THE LINEDEF LIST POINTERS ARRAY
	inpointers=(UINT16*)lump_in+(sizeof(blockmap_header_t)/sizeof(UINT16));

	// GET NUMBER OF WHOLE [US]INT16S (ROUND DOWN) IN LUMP
	numbers_length=(entry->length/sizeof(UINT16));

	// ENSURE THAT THE BLOCK POINTERS OF THIS BLOCKMAP ARE VALID
	if (!ValidateBlockmapBlockPointers(entry->name,entry->length,numbers_length,inpointers,num_blocks,process_only))
	{
		return;
	}

	// ENSURE THAT THE BLOCK EXTENTS OF THIS BLOCKMAP ARE VALID
	if (!ValidateBlockmapBlockExtents(lump_in,entry->name,entry->length,numbers_length,inpointers,num_blocks,num_linedefs,process_only))
	{
		return;
	}

	// ENSURE THAT THE LINEDEFS LISTS OF THIS BLOCKMAP ARE VALID
	if (!maintain_blockmaps)
	{
		// CHECK FOR DUPLICATE OR OUT-OF-ORDER LINEDEF NUMBERS.
		// THESE CAUSE BLOCKMAPS THAT FUNCTION IDENTICALLY TO
		// HAVE PHYSICALLY DIFFERENT DATA.
		size_t block;
		for (block=0;block<num_blocks;block++)
		{
			if (BlockAnyLinedefNumberProblems((SINT16*)lump_in+inpointers[block]))
			{
				must_rebuild_blocks=TRUE;
				break;
			}
		}
	}

	//
	// FROM THIS POINT ONWARD ALL PREVIOUS POINTERS INTO PLACES
	// INSIDE THE INPUT DATA ARE INVALID. YOU HAVE BEEN WARNED!
	//

	// OPTIMIZE THE BLOCKMAP
	if (!must_rebuild_blocks)
	{
		// USE ORIGINAL DATA
		OptimizeBlockmap(lump_in,&lump_out_temp,entry->length,pack_blockmaps,unpack_blockmaps,process_only,&size_out_temp,&success);
	}
	else
	{
		// VARIABLES
		void *temp;
		// CREATE A COPY AS MUST PREPROCESS IT
		temp=malloc(entry->length);
		if (temp==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for blockmap buffer copy",entry->length);
		}
		(void) memmove(temp,lump_in,entry->length);
		// REBUILD ANY LINEDEF LISTS THAT REQUIRE IT
		RebuildLinedefBlocks(temp,mapname,num_blocks,process_only);
		// PERFORM THE OPTIMIZATION
		OptimizeBlockmap(temp,&lump_out_temp,entry->length,pack_blockmaps,unpack_blockmaps,process_only,&size_out_temp,&success);
		// DEALLOCATE THE COPY
		free(temp);
//		temp=NULL;
	}

	// CHECK FOR AND HANDLE OPTIMIZATION FAILURE
	if (!success)
	{
		if (!process_only)
		{
			if (!pack_blockmaps && !unpack_blockmaps /* i.e. (rebuild_blockmaps) */)
			{
				EventState(VERBOSITY_WARNINGS,"Blockmap from %s would exceed structure limits if rebuilt -- not processed",mapname);
			}
			else if (pack_blockmaps)
			{
				EventState(VERBOSITY_WARNINGS,"Blockmap from %s would exceed structure limits if packed -- not processed",mapname);
			}
			else if (unpack_blockmaps)
			{
				EventState(VERBOSITY_WARNINGS,"Blockmap from %s would exceed structure limits if unpacked -- not processed",mapname);
			}
			else
			{
				FatalAbort(1,__FILE__,__LINE__,"Contradictory blockmap processing flags should have been trapped by now");
			}
		}
		free(lump_out_temp);
		size_out_temp=WADDIR->entries[index].length;
		lump_out_temp=malloc(size_out_temp);
		if (lump_out_temp==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for blockmap buffer",size_out_temp);
		}
		(void) memmove(lump_out_temp,lump_in,size_out_temp);
		*lump_out=lump_out_temp;
		*size_out=size_out_temp;
		return;
	}

	// IS THE BLOCKMAP CHANGED AND DO WE WRITE IT OUT
	is_resized=(entry->length!=size_out_temp)?TRUE:FALSE;
	do_writeit=(is_resized || (flush_blockmaps && !mamatch(lump_in,entry->length,lump_out_temp,size_out_temp)))?TRUE:FALSE;

	// SET POINTER TO THE MODIFIED BLOCKMAP IF MUST REWRITE IT
	// OTHERWISE, REPLACE IT WITH A COPY OF THE INPUT BLOCKMAP.
	if (!do_writeit)
	{
		free(lump_out_temp);
		size_out_temp=WADDIR->entries[index].length;
		lump_out_temp=malloc(size_out_temp);
		if (lump_out_temp==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for blockmap buffer",size_out_temp);
		}
		(void) memmove(lump_out_temp,lump_in,size_out_temp);
	}
	*lump_out=lump_out_temp;
	*size_out=size_out_temp;

	// NOTIFY USER IF ENTRY WAS MODIFIED AND MUST BE WRITTEN
	if (!process_only && do_writeit)
	{
		if (is_resized)
		{
			if (!pack_blockmaps && !unpack_blockmaps /* i.e. (rebuild_blockmaps) */)
			{
				EventState(VERBOSITY_ACTIONS,"Blockmap of %s rebuilt (length changed from %ld to %ld)",mapname,entry->length,(*size_out));
			}
			else if (pack_blockmaps)
			{
				EventState(VERBOSITY_ACTIONS,"Blockmap of %s packed (length changed from %ld to %ld)",mapname,entry->length,(*size_out));
			}
			else if (unpack_blockmaps)
			{
				EventState(VERBOSITY_ACTIONS,"Blockmap of %s unpacked (length changed from %ld to %ld)",mapname,entry->length,(*size_out));
			}
			else
			{
				FatalAbort(1,__FILE__,__LINE__,"Contradictory blockmap processing flags should have been trapped by now");
			}
		}
		else
		{
			if (!pack_blockmaps && !unpack_blockmaps /* i.e. (rebuild_blockmaps) */)
			{
				EventState(VERBOSITY_ACTIONS,"Blockmap of %s rebuilt (length unchanged at %ld)",mapname,entry->length);
			}
			else if (pack_blockmaps)
			{
				EventState(VERBOSITY_ACTIONS,"Blockmap of %s packed (length unchanged at %ld)",mapname,entry->length);
			}
			else if (unpack_blockmaps)
			{
				EventState(VERBOSITY_ACTIONS,"Blockmap of %s unpacked (length unchanged at %ld)",mapname,entry->length);
			}
			else
			{
				FatalAbort(1,__FILE__,__LINE__,"Contradictory blockmap processing flags should have been trapped by now");
			}
		}
	}
}

// Check for invalid picture column pointers
static bool_t ValidatePictureColumnPointers (
	const char *lump_name,
	size_t lump_length,
	const UINT32 *inpointers,
	size_t num_columns,
	bool_t process_only)
{
	// VARIABLES
	size_t i;
	// CHECK FOR VALID POINTER ARRAY
	for (i=0;i<num_columns;i++)
	{
		size_t actual;
		/************/
		actual=(inpointers[i]*sizeof(UINT08));
		if (actual>lump_length)
		{
			if (!process_only)
			{
				EventState(VERBOSITY_DETAILS,"Column pointer (%ld) points outside picture extent (%ld)",actual,lump_length);
				EventState(VERBOSITY_WARNINGS,"Picture %s has an invalid pointer array -- not processed",lump_name);
			}
			return FALSE;
		}
	}
	// RETURN
	return TRUE;
}

// Check for invalid picture column extents
static bool_t ValidatePictureColumnExtents (
	const void *lump_ptr,
	const char *lump_name,
	size_t lump_length,
	const UINT32 *inpointers,
	size_t num_columns,
	bool_t process_only)
{
	// VARIABLES
	size_t i;
	// CHECK FOR VALID PIXEL RUN LISTS
	for (i=0;i<num_columns;i++)
	{
		// VARIABLES
		UINT08 *p;
		size_t scanned;
		// GET POINTER TO PIXEL RUN LIST
		p=(UINT08*)lump_ptr+inpointers[i];
		// GET BYTE COUNT CORRESPONDING TO POINTER
		scanned=(inpointers[i]*sizeof(UINT08));
		// WE KNOW THAT THE FIRST VALUE IN THE PIXEL RUN IS
		// WITHIN THE PICTURE EXTENT, SINCE THE POINTER ARRAY
		// HAS ALREADY BEEN CHECKED, SO VALIDATE THE REST.
		while (TRUE)
		{
			// VARIABLES
			size_t num_pixels;
			//
			// ALREADY POINTING AT NEXT PIXEL RUN/END-OF-COLUMN
			// WHEN THIS LOOP STARTS; THAT IS HOW LOOP IS SET
			// UP BEFORE THE FIRST ITERATION AND DURING ALL.
			//
			// GET BYTE COUNT CORRESPONDING TO CURRENT POSITION
			scanned+=sizeof(UINT08);
			if (scanned>lump_length)
			{
				if (!process_only)
				{
					EventState(VERBOSITY_DETAILS,"Column (%ld) data runs outside picture extent (%ld)",i,lump_length);
					EventState(VERBOSITY_WARNINGS,"Picture %s includes an invalid column -- not processed",lump_name);
				}
				return FALSE;
			}
			// IF END OF COLUMN THEN DONE
			if (*p==(UINT08)255)
			{
				break;
			}
			// POINT AT NUMBER OF PIXELS
			p++;
			// GET BYTE COUNT CORRESPONDING TO CURRENT POSITION
			scanned+=sizeof(UINT08);
			if (scanned>lump_length)
			{
				if (!process_only)
				{
					EventState(VERBOSITY_DETAILS,"Column (%ld) data runs outside picture extent (%ld)",i,lump_length);
					EventState(VERBOSITY_WARNINGS,"Picture %s includes an invalid column -- not processed",lump_name);
				}
				return FALSE;
			}
			// GET NUMBER OF PIXELS
			num_pixels=(size_t)(*p);
			// ID SOFTWARE HAVE NEVER DEFINED WHAT TO DO WITH A
			// ZERO-LENGTH PIXEL RUN. IS IT AN ERROR? IS IT OK?
			if (num_pixels==0)
			{
				if (!process_only)
				{
					EventState(VERBOSITY_DETAILS,"Column (%ld) data includes undefined (zero) pixel run length)",i);
					EventState(VERBOSITY_WARNINGS,"Picture %s includes an invalid column -- not processed",lump_name);
				}
				return FALSE;
			}
			// POINT AT LAST DATA BYTE IN THIS RUN
			p+=(num_pixels+GRAPHIC_DUMMY_PIXEL_COUNT);
			// GET BYTE COUNT CORRESPONDING TO CURRENT POSITION
			scanned+=((num_pixels+GRAPHIC_DUMMY_PIXEL_COUNT)*sizeof(UINT08));
			// CHECK THAT RUN DOES NOT OVERFLOW PICTURE EXTENT
			if (scanned>lump_length)
			{
				if (!process_only)
				{
					EventState(VERBOSITY_DETAILS,"Column (%ld) data runs outside picture extent (%ld)",i,lump_length);
					EventState(VERBOSITY_WARNINGS,"Picture %s includes an invalid column -- not processed",lump_name);
				}
				return FALSE;
			}
			// POINT AT NEXT PIXEL RUN OR END-OF-COLUMN
			p++;
			// DO NOT ADJUST BYTE COUNT AS THAT IS ALWAYS DONE
			// AT THE START OF THE NEXT ITERATION OF THE LOOP.
		}
	}
	// RETURN
	return TRUE;
}

// Copy out a picture (optimized)
static void CheckAndProcessPicture (
	const void *lump_in,
	const WADDIR_t *WADDIR,
	size_t index,
	bool_t pack_pictures,
	bool_t unpack_pictures,
	bool_t flush_pictures,
	bool_t maintain_pictures,
	bool_t process_only,
	void **lump_out,
	size_t *size_out)
{
	// VARIABLES
	void *lump_out_temp;
	UINT32 *inpointers;
	const WADDIRentry_t *entry;
	picture_header_t *header;
	size_t num_columns,size_out_temp;
	bool_t is_resized,do_writeit,must_rebuild_columns;

	// PRIMARY ERROR CHECK
	if (pack_pictures && unpack_pictures)
	{
		FatalAbort(1,__FILE__,__LINE__,"Contradictory picture processing flags should have been trapped by now");
	}
	if (lump_in==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a picture with NULL lump_in argument");
	}
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a picture with NULL directory argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a picture with NULL directory entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (index>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a picture with entry index (%ld) greater directory entry count (%ld)",index,WADDIR->count);
	}

	// INITIALISE
	size_out_temp=WADDIR->entries[index].length;
	lump_out_temp=malloc(size_out_temp);
	if (lump_out_temp==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for picture buffer",size_out_temp);
	}
	(void) memmove(lump_out_temp,lump_in,size_out_temp);
	*lump_out=lump_out_temp;
	*size_out=size_out_temp;
	must_rebuild_columns=FALSE;

	// CREATE POINTER TO THE DIRECTORY ENTRY OF THIS PICTURE
	entry=&WADDIR->entries[index];

	// STATEMENT
	if (!process_only)
	{
		EventState(VERBOSITY_DETAILS,"Processing picture %s",entry->name);
	}

	// IF NOT A VALID PICTURE (CHECK 1) THEN DO NOT MODIFY IT
	if (entry->length<sizeof(picture_header_t))
	{
		if (!process_only)
		{
			EventState(VERBOSITY_WARNINGS,"Picture %s is too short -- not processed",entry->name);
		}
		return;
	}

	// CREATE POINTER TO THE PICTURE HEADER
	header=(picture_header_t*)lump_in;

	// IF NOT A VALID PICTURE (CHECK 2) THEN DO NOT MODIFY IT
	/***********************************************************
	If header + pointer array is same size or bigger than the
	entry, then there is no room for the pixel runs, so this is
	not a valid PICTURE.
	***********************************************************/
	if (sizeof(picture_header_t)+(RDLend16SV(header->width)*sizeof(UINT32))>=(entry->length))
	{
		if (!process_only)
		{
			EventState(VERBOSITY_WARNINGS,"Picture %s has an invalid header -- not processed",entry->name);
		}
		return;
	}

	// IF NOT A VALID PICTURE (CHECK 3) THEN DO NOT MODIFY IT
	/***********************************************************
	If the picture metrics are unreasonable then this is not a
	valid PICTURE.
	***********************************************************/
	if (
	     (RDLend16SV(header->width)       <=  0                     ) ||
	     (RDLend16SV(header->width)       >   GRAPHIC_MAX_WIDTH     ) ||
	     (RDLend16SV(header->height)      <=  0                     ) ||
	     (RDLend16SV(header->height)      >   GRAPHIC_MAX_HEIGHT    ) ||
	     (RDLend16SV(header->left_offset) >   GRAPHIC_MAX_LEFTOFFSET) ||
	     (RDLend16SV(header->left_offset) <  -GRAPHIC_MAX_LEFTOFFSET) ||
	     (RDLend16SV(header->top_offset)  >   GRAPHIC_MAX_TOPOFFSET ) ||
	     (RDLend16SV(header->top_offset)  <  -GRAPHIC_MAX_TOPOFFSET )
	   )
	{
		if (!process_only)
		{
			EventState(VERBOSITY_WARNINGS,"Picture %s has an invalid header -- not processed",entry->name);
		}
		return;
	}

	// GET NUMBER OF COLUMNS IN STANDARD FORMAT
	num_columns=(size_t)RDLend16SV(header->width);

	// CREATE POINTER TO THE COLUMN POINTERS ARRAY
	inpointers=(UINT32*)lump_in+(sizeof(picture_header_t)/sizeof(UINT32));

	// ENSURE THAT THE COLUMN POINTERS OF THIS PICTURE ARE VALID
	if (!ValidatePictureColumnPointers(entry->name,entry->length,inpointers,num_columns,process_only))
	{
		return;
	}

	// ENSURE THAT THE COLUMN EXTENTS OF THIS PICTURE ARE VALID
	if (!ValidatePictureColumnExtents(lump_in,entry->name,entry->length,inpointers,num_columns,process_only))
	{
		return;
	}

	// ENSURE THAT THE COLUMN DATA OF THIS PICTURE IS VALID
	if (!maintain_pictures)
	{
		// CHECK FOR DUPLICATE, OUT-OF-ORDER OR CONSECUTIVE
		// PIXEL RUNS. THESE CAUSE COLUMNS THAT DISPLAY
		// IDENTICALLY TO HAVE PHYSICALLY DIFFERENT DATA.
		size_t column;
		for (column=0;column<num_columns;column++)
		{
			if (ColumnAnyPixelRunProblems((UINT08*)lump_in+inpointers[column]))
			{
				must_rebuild_columns=TRUE;
				break;
			}
		}
	}

	// OPTIMIZE THE PICTURE
	if (!must_rebuild_columns)
	{
		// USE ORIGINAL DATA
		OptimizePicture(lump_in,&lump_out_temp,entry->length,pack_pictures,unpack_pictures,process_only,&size_out_temp);
	}
	else
	{
		// VARIABLES
		void *lump_in_temp;
		// CREATE A COPY AS MUST PREPROCESS IT
		lump_in_temp=malloc(entry->length);
		if (lump_in_temp==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for picture buffer copy",entry->length);
		}
		(void) memmove(lump_in_temp,lump_in,entry->length);
		// REBUILD PIXEL RUNS IN COLUMNS THAT REQUIRE IT
		RebuildPixelColumns(lump_in_temp,entry->name,num_columns,process_only);
		// PERFORM THE OPTIMIZATION
		OptimizePicture(lump_in_temp,&lump_out_temp,entry->length,pack_pictures,unpack_pictures,process_only,&size_out_temp);
		// DEALLOCATE THE COPY
		free(lump_in_temp);
//		temp=NULL;
	}

	// IS THE PICTURE CHANGED AND DO WE WRITE IT OUT
	is_resized=(entry->length!=size_out_temp)?TRUE:FALSE;
	do_writeit=(is_resized || (flush_pictures && !mamatch(lump_in,entry->length,lump_out_temp,size_out_temp)))?TRUE:FALSE;

	// SET POINTER TO THE MODIFIED PICTURE IF MUST REWRITE IT
	// OTHERWISE, REPLACE IT WITH A COPY OF THE INPUT PICTURE.
	if (!do_writeit)
	{
		free(lump_out_temp);
		size_out_temp=entry->length;
		lump_out_temp=malloc(size_out_temp);
		if (lump_out_temp==NULL)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for picture buffer",size_out_temp);
		}
		(void) memmove(lump_out_temp,lump_in,size_out_temp);
	}
	*lump_out=lump_out_temp;
	*size_out=size_out_temp;

	// NOTIFY USER IF ENTRY WAS MODIFIED AND MUST BE WRITTEN
	if (!process_only && do_writeit)
	{
		if (is_resized)
		{
			if (!pack_pictures && !unpack_pictures /* i.e. (rebuild_pictures) */)
			{
				EventState(VERBOSITY_ACTIONS,"Picture %s rebuilt (length changed from %ld to %ld)",entry->name,entry->length,(*size_out));
			}
			else if (pack_pictures)
			{
				EventState(VERBOSITY_ACTIONS,"Picture %s packed (length changed from %ld to %ld)",entry->name,entry->length,(*size_out));
			}
			else if (unpack_pictures)
			{
				EventState(VERBOSITY_ACTIONS,"Picture %s unpacked (length changed from %ld to %ld)",entry->name,entry->length,(*size_out));
			}
			else
			{
				FatalAbort(1,__FILE__,__LINE__,"Contradictory picture processing flags should have been trapped by now");
			}
		}
		else
		{
			if (!pack_pictures && !unpack_pictures /* i.e. (rebuild_pictures) */)
			{
				EventState(VERBOSITY_ACTIONS,"Picture %s rebuilt (length unchanged at %ld)",entry->name,entry->length);
			}
			else if (pack_pictures)
			{
				EventState(VERBOSITY_ACTIONS,"Picture %s packed (length unchanged at %ld)",entry->name,entry->length);
			}
			else if (unpack_pictures)
			{
				EventState(VERBOSITY_ACTIONS,"Picture %s unpacked (length unchanged at %ld)",entry->name,entry->length);
			}
			else
			{
				FatalAbort(1,__FILE__,__LINE__,"Contradictory picture processing flags should have been trapped by now");
			}
		}
	}
}

// Copy out a sound (optimized)
static void CheckAndProcessSound (
	const void *lump_in,
	const WADDIR_t *WADDIR,
	size_t index,
	bool_t process_only,
	void **lump_out,
	size_t *size_out)
{
	// VARIABLES
	void *lump_out_temp;
	size_t size_out_temp;
	sound_header_t *header;
	const WADDIRentry_t *entry;

	// PRIMARY ERROR CHECK
	if (lump_in==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a sound with NULL lump_in argument");
	}
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a sound with NULL directory argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a sound with NULL directory entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (index>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a sound with entry index (%ld) greater directory entry count (%ld)",index,WADDIR->count);
	}

	// INITIALISE
	size_out_temp=WADDIR->entries[index].length;
	lump_out_temp=malloc(size_out_temp);
	if (lump_out_temp==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for sound buffer",size_out_temp);
	}
	(void) memmove(lump_out_temp,lump_in,size_out_temp);
	*lump_out=lump_out_temp;
	*size_out=size_out_temp;

	// CREATE POINTER TO THE DIRECTORY ENTRY OF THIS SOUND
	entry=&WADDIR->entries[index];

	// STATEMENT
	if (!process_only)
	{
		EventState(VERBOSITY_DETAILS,"Processing sound %s",entry->name);
	}

	// IF NOT A VALID SOUND (CHECK 1) THEN DO NOT MODIFY IT
	if (entry->length<sizeof(sound_header_t))
	{
		if (!process_only)
		{
			EventState(VERBOSITY_WARNINGS,"Sound %s is not DOOM format or is invalid -- not processed",entry->name);
		}
		return;
	}

	// CREATE POINTER TO THE SOUND HEADER
	header=(sound_header_t*)lump_in;

	// IF NOT A VALID SOUND (CHECK 2) THEN DO NOT MODIFY IT
	/***********************************************************
	If header + samples array is allegedly bigger than the
	entry, then there is not enough room for ALL of the samples,
	so this is not a valid SOUND.
	***********************************************************/
	if (sizeof(sound_header_t)+(RDLend32UV(header->num_samples)*sizeof(UINT08))>(entry->length))
	{
		if (!process_only)
		{
			EventState(VERBOSITY_WARNINGS,"Sound %s is not DOOM format or is invalid -- not processed",entry->name);
		}
		return;
	}

	// IF NOT A VALID SOUND (CHECK 3) THEN DO NOT MODIFY IT
	/***********************************************************
	If magic number is wrong then this is not a valid SOUND.
	***********************************************************/
	if (RDLend16UV(header->magic)!=3)
	{
		if (!process_only)
		{
			EventState(VERBOSITY_WARNINGS,"Sound %s is not DOOM format or is invalid -- not processed",entry->name);
		}
		return;
	}

	// OPTIMIZE IT
	if (entry->length>(header->num_samples+sizeof(sound_header_t)))
	{
		if (!process_only)
		{
			EventState(VERBOSITY_ACTIONS,"Sound entry %s truncated to correct length",entry->name);
		}
//		size_out_temp=(size_t)(header->num_samples+8);
//		*size_out=size_out_temp;
		*size_out=(size_t)(header->num_samples+8);
	}
}

// Copy out generic lumps (not optimized)
static void CheckAndProcessLump (
	const void *lump_in,
	const WADDIR_t *WADDIR,
	size_t index,
	bool_t process_only,
	void **lump_out,
	size_t *size_out)
{
	// VARIABLES
	void *lump_out_temp;
	size_t size_out_temp;

	// PRIMARY ERROR CHECK
	if (lump_in==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a generic lump with NULL lump_in argument");
	}
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a generic lump with NULL directory argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a generic lump with NULL directory entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (index>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a generic lump with entry index (%ld) greater directory entry count (%ld)",index,WADDIR->count);
	}

	// INITIALISE
	size_out_temp=WADDIR->entries[index].length;
	lump_out_temp=malloc(size_out_temp);
	if (lump_out_temp==NULL)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for lump buffer",size_out_temp);
	}
	(void) memmove(lump_out_temp,lump_in,size_out_temp);
	*lump_out=lump_out_temp;
	*size_out=size_out_temp;

	// STATEMENT
	if (!process_only)
	{
		EventState(VERBOSITY_DETAILS,"Copying entry %s without modifications",WADDIR->entries[index].name);
	}
}

// Optimize an already-loaded WAD file directory entry
void OptimizeDirectoryEntry (
	const void *lump_in,										// lump to process
	const WADDIR_t *WADDIR,										// WAD file directory to take entry metrics from
	size_t index,												// index of lump in directory entries
	bool_t truncate_waves,										// remove wasted space at the end of sound resources
	bool_t rebuild_blockmaps,									// rebuild (no pack/unpack) blockmaps to remove wasted space
	bool_t pack_blockmaps,										// pack blockmaps to remove current and potential wasted space
	bool_t unpack_blockmaps,									// unpack blockmaps into normalised (but inefficient) format
	bool_t flush_blockmaps,										// write out modified blockmaps even if the size has not changed
	bool_t rebuild_pictures,									// rebuild (no pack/unpack) pictures to remove wasted space
	bool_t pack_pictures,										// pack pictures to remove current and potential wasted space
	bool_t unpack_pictures,										// unpack pictures into normalised (but inefficient) format
	bool_t flush_pictures,										// write out modified pictures even if the size has not changed
	bool_t maintain_blockmaps,									// maintain nonstandard blockmaps (preserve non-optimal blocks)
	bool_t maintain_pictures,									// maintain nonstandard pictures (preserve non-optimal columns)
	bool_t process_only,										// issue no messages except for fatal errors
	void **lump_out,											// processed lump
	size_t *size_out)											// size of processed lump
{
	// DOCUMENTATION
	/*
		This function optimizes a lump that is already loaded
		into memory and does so silently with no diagnostics.
		It is used exclusively when searching for new lump
		reuse and always finds the canonical form of the lump.
	*/

	// VARIABLES
	const WADDIRentry_t *entry;

	// PRIMARY ERROR CHECK
	if (lump_in==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a WAD file directory entry with NULL lump_in argument");
	}
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a WAD file directory with NULL directory argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a WAD file directory with NULL directory entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif
	if (index>WADDIR->count)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to optimize a WAD file directory entry with entry index (%ld) greater directory entry count (%ld)",index,WADDIR->count);
	}

	// CREATE POINTER TO THE DIRECTORY ENTRY
	entry=&WADDIR->entries[index];

	// PROCESS THE ENTRY
	if ( (rebuild_blockmaps || pack_blockmaps || unpack_blockmaps ) && (entry->type==TYPE_MAPDATA) && (strcmp(entry->name,"BLOCKMAP")==0) )
	{
		CheckAndProcessBlockmap(lump_in,WADDIR,index,pack_blockmaps,unpack_blockmaps,flush_blockmaps,maintain_blockmaps,process_only,lump_out,size_out);
	}
	else if ( (rebuild_pictures || pack_pictures || unpack_pictures ) && (entry->type==TYPE_GRAPHIC) )
	{
		CheckAndProcessPicture(lump_in,WADDIR,index,pack_pictures,unpack_pictures,flush_pictures,maintain_pictures,process_only,lump_out,size_out);
	}
	else if ( truncate_waves && ((entry->type==TYPE_SOUND) || (entry->type==TYPE_VOICE)) )
	{
		CheckAndProcessSound(lump_in,WADDIR,index,process_only,lump_out,size_out);
	}
	else
	{
		 CheckAndProcessLump (lump_in,WADDIR,index,process_only,lump_out,size_out);
	}
}

// Copy entries of WAD file directory (optimized)
void CopyDirectoryEntriesOptimized (
	FILE *infile,												// file to copy entries from
	FILE *outfile,												// file to copy entries to
	WADDIR_t *WADDIR,											// directory to copy entries of
	const LST_t *LST,											// symbol table resulting from initial scan
	bool_t truncate_waves,										// remove wasted space at the end of sound resources
	bool_t rebuild_blockmaps,									// rebuild (no pack/unpack) blockmaps to remove wasted space
	bool_t pack_blockmaps,										// pack blockmaps to remove current and potential wasted space
	bool_t unpack_blockmaps,									// unpack blockmaps into normalised (but inefficient) format
	bool_t flush_blockmaps,										// write out modified blockmaps even if the size has not changed
	bool_t rebuild_pictures,									// rebuild (no pack/unpack) pictures to remove wasted space
	bool_t pack_pictures,										// pack pictures to remove current and potential wasted space
	bool_t unpack_pictures,										// unpack pictures into normalised (but inefficient) format
	bool_t flush_pictures,										// write out modified pictures even if the size has not changed
	bool_t reuse_lumps,											// reuse lumps where directory entries specify same size and offset
	bool_t pack_lumps,											// share lumps with identical contents between directory entries
	bool_t unpack_lumps,										// output each lump explicitly even if it could be shared or reused
	bool_t unpack_conflicts,									// do not preserve ineligible lump reuse (if tolerated)
	bool_t align_resources,										// align resources on 32-bit boundaries for speed at cost of space
	bool_t align_directory,										// align directory on 32-bit boundaries for speed at cost of space
	bool_t maintain_blockmaps,									// maintain nonstandard blockmaps (preserve non-optimal blocks)
	bool_t maintain_pictures)									// maintain nonstandard pictures (preserve non-optimal columns)
{
	// DOCUMENTATION
	/*
		PRE:
		File must be positioned just after the WAD file header,
		which, by common usage, is the start of the data lumps
		in the file [any file CREATED with this library will
		always follow the convention].

		Input file entry positions are taken from the
		directory.

		POST:
		File is positioned just after the data lumps in the
		file which, by common usage, is where the directory is
		to be located [any file CREATED with this library will
		always follow the convention].

		Output file entry positions are the next available file
		position.

		The "start" member of each directory entry is updated.
	*/

	// VARIABLES
	size_t i;
	fofs_t fofs;
	WADDIR_t *OLDDIR=NULL;

	// PRIMARY ERROR CHECK
	if (infile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to copy entries in WAD file directory with NULL input file argument");
	}
	if (outfile==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to copy entries in WAD file directory with NULL output file argument");
	}
	if (WADDIR==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to copy entries in WAD file directory with NULL directory argument");
	}
	if (WADDIR->entries==NULL)
	{
		FatalAbort(1,__FILE__,__LINE__,"Attempt to copy entries in WAD file directory with NULL directory entries");
	}
#ifdef PARANOID
	if (WADDIR->count>WADDIR->limit)
	{
		FatalAbort(1,__FILE__,__LINE__,"WAD file directory is corrupt");
	}
#endif

	// GET CURRENT FILE POSITION FOR REFERENCE
	fofs=ftell(outfile);
	if (fofs<0)
	{
		ErrorAbort(1,__FILE__,__LINE__,"Could not tell position in output file");
	}

	// GET COPY OF DIRECTORY ENTRIES IF REQUIRED
	//
	// When handling existing lump reuse, we need a copy of the
	// WAD file directory as it was before updating with the
	// new lump file offsets and (maybe) sizes.
	//
	// This is safe because from now on, the number of entries
	// and their positions in the directory cannot change; only
	// their file offsets and (maybe) sizes can change here.
	//
	if (WADDIR->reuse)
	{
		WADInitDirectory(&OLDDIR);
		WADCopyDirectory(OLDDIR,WADDIR);
	}

	// PERFORM THE OPERATION
	for (i=0;i<WADDIR->count;i++)
	{
		// Variables
		void *lump_in;
		void *lump_out;
		bool_t reused;
		size_t reused_index,size_out;
		WADDIRentry_t *entry;
		static const char padding[3]={NUL,NUL,NUL};
		// Initialise pointers
		lump_in=NULL;
		lump_out=NULL;
		// Create pointer to the directory entry
		entry=&WADDIR->entries[i];
		// Handle existing lump reuse
		reused=FALSE;
		if (WADDIR->reuse)
		{
			WADHandleExistingReuseOnOutput(
				OLDDIR,											// WAD file directory to check for reuse in
				LST,											// symbol table resulting from the scan of that directory
				reuse_lumps,									// reuse lumps where directory entries specify same size and offset
				pack_lumps,										// share lumps with identical contents between directory entries
				unpack_lumps,									// output each lump explicitly even if it could be shared or reused
				unpack_conflicts,								// do not preserve ineligible lump reuse (if tolerated)
				i,												// index of entry to check for reuse of
				&reused);										// flag for this lump reuses a previous directory entry
		}
		// If there was any then go to the next entry
		if (reused)
		{
			// This lump reuses
			// a previous entry.
			continue;
		}
		//
		// If we get here then either there was no existing
		// lump reuse or else we are required to unpack it.
		//
		// Seek to lump location in the input file
		if (fseek(infile,entry->start,SEEK_SET)!=0)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not seek in input file");
		}
		// Allocate memory for the lump
		if (entry->length==0)
		{
			lump_in=malloc(1);
			if (lump_in==NULL)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not allocate 1 byte for (dummy) lump buffer (entry size is 0)");
			}
		}
		else
		{
			lump_in=malloc(entry->length);
			if (lump_in==NULL)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not allocate %ld bytes for lump buffer",entry->length);
			}
		}
		// Read the lump into memory
		if (entry->length==0)
		{
			*((char*)lump_in)=NUL;
		}
		else
		{
			if (fread(lump_in,entry->length,1,infile)<1)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not read from file");
			}
		}
		OptimizeDirectoryEntry (
			lump_in,											// lump to process
			WADDIR,												// WAD file directory to take entry metrics from
			i,													// index of lump in directory entries
			truncate_waves,										// remove wasted space at the end of sound resources
			rebuild_blockmaps,									// rebuild (no pack/unpack) blockmaps to remove wasted space
			pack_blockmaps,										// pack blockmaps to remove current and potential wasted space
			unpack_blockmaps,									// unpack blockmaps into normalised (but inefficient) format
			flush_blockmaps,									// write out modified blockmaps even if the size has not changed
			rebuild_pictures,									// rebuild (no pack/unpack) pictures to remove wasted space
			pack_pictures,										// pack pictures to remove current and potential wasted space
			unpack_pictures,									// unpack pictures into normalised (but inefficient) format
			flush_pictures,										// write out modified pictures even if the size has not changed
			maintain_blockmaps,									// maintain nonstandard blockmaps (preserve non-optimal blocks)
			maintain_pictures,									// maintain nonstandard pictures (preserve non-optimal columns)
			/*process_only=*/FALSE,								// issue no messages except for fatal errors
			&lump_out,											// processed lump
			&size_out);											// size of processed lump
		entry->length=size_out;
		// Find new lump reuse (if required to)
		reused=FALSE;
		reused_index=0;
		if (pack_lumps)
		{
			// Flush the file as it is open for write
			if (fflush(outfile)!=0)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not flush output file in order to navigate it");
			}
			// Find a valid previous copy of this lump if any
			WADIdentifyNewReuseOnOutput (
				outfile,										// WAD file to search in
				WADDIR,											// WAD file directory to look for potential reuse in
				LST,											// symbol table resulting from the scan of that directory
				lump_out,										// lump to check for potential reuse of
				i,												// index in WAD file directory of the lump
				OptimizeDirectoryEntry,							// lump canonicalization function
				truncate_waves,									// remove wasted space at the end of sound resources
				rebuild_blockmaps,								// rebuild (no pack/unpack) blockmaps to remove wasted space
				pack_blockmaps,									// pack blockmaps to remove current and potential wasted space
				unpack_blockmaps,								// unpack blockmaps into normalised (but inefficient) format
				flush_blockmaps,								// write out modified blockmaps even if the size has not changed
				rebuild_pictures,								// rebuild (no pack/unpack) pictures to remove wasted space
				pack_pictures,									// pack pictures to remove current and potential wasted space
				unpack_pictures,								// unpack pictures into normalised (but inefficient) format
				flush_pictures,									// write out modified pictures even if the size has not changed
				maintain_blockmaps,								// maintain nonstandard blockmaps (preserve non-optimal blocks)
				maintain_pictures,								// maintain nonstandard pictures (preserve non-optimal columns)
				/*process_only=*/TRUE,							// issue no messages except for fatal errors
				&reused,										// flag for this lump reuses a previous directory entry
				&reused_index);									// index of entry that this lump can now reuse
		}
		// If there was one then use it and go to the entry
		if (reused)
		{
			// Set the entry start to that of the copy
			entry->start=WADDIR->entries[reused_index].start;
			// Deallocate memory for modified lump (if any)
			free(lump_out);
			lump_out=NULL;
			// Deallocate memory for the lump (as read)
			free(lump_in);
			lump_in=NULL;
			// This lump (now) reuses a previous entry
			continue;
		}
		// If the entry has no lump then go to the next entry
		if (entry->length==0)
		{
			// By convention use next lump address
			if (i==WADDIR->count)
			{
				if (align_directory && ((fofs%4)!=0) )
				{
					entry->start=fofs+(4-(fofs%4));
				}
				else
				{
					entry->start=fofs;
				}
			}
			else
			{
				if (align_resources && ((fofs%4)!=0) )
				{
					entry->start=fofs+(4-(fofs%4));
				}
				else
				{
					entry->start=fofs;
				}
			}
			// Deallocate memory for modified lump
			free(lump_out);
			lump_out=NULL;
			// Deallocate memory for the lump (as read)
			free(lump_in);
			lump_in=NULL;
			// This entry has no lump to write
			continue;
		}
		// Align the lump in the output file if applicable
		if (align_resources && ((fofs%4)!=0) )
		{
			if (fwrite(padding,(size_t)(4-(fofs%4)),1,outfile)<1)
			{
				ErrorAbort(1,__FILE__,__LINE__,"Could not write to file");
			}
			fofs+=(4-(fofs%4));
		}
		// Update the directory entry with the new file position
		entry->start=fofs;
		// Write the lump to the output file
		if (fwrite((lump_out!=NULL)?lump_out:lump_in,entry->length,1,outfile)<1)
		{
			ErrorAbort(1,__FILE__,__LINE__,"Could not write to file");
		}
		// Set the next file position to write to
#ifdef _WIN64
		if (entry->length>SINT32_MAX)
		{
			FatalAbort(1,__FILE__,__LINE__,"Value in \"entry->length\" is > 32 bits");
		}
#endif
		#ifdef __VISUALC__
		#pragma warning( push )
		#pragma warning( disable : 4267 )
		#endif
		fofs+=(entry->length);
		#ifdef __VISUALC__
		#pragma warning( pop )
		#endif

		// Deallocate memory for modified lump
		free(lump_out);
		lump_out=NULL;
		// Deallocate memory for the lump (as read)
		free(lump_in);
		lump_in=NULL;
	}

	// DISCARD UNMODIFIED COPY OF DIRECTORY IF REQUIRED
	if (OLDDIR!=NULL)
	{
		WADDoneDirectory(&OLDDIR);
	}
}

/**********************************************************************************************************************************/
/********************************************************** End of File ***********************************************************/
/**********************************************************************************************************************************/
