#include "patch.h"

/************************************************************
 MAKSUBS.C
			 /-------------------------\
			||     MAK subroutines     ||
			 \-------------------------/
									
************************************************************/

// Advance a file pointer and the offset in the file
// to next longword alignment boundary

void Advance(int *off,int n,FILE *st)
{
	int m;

	m = ((n+3)>>2)<<2;
	*off += m;
	fseek(st,m-n,SEEK_CUR);
}

// Read a resource file and embed it in a wad without translation.
// memory balanced except directory extension

void WriteGenControl(DTex *list,int len,FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir)
{
	int p,n;
	FILE *st;
	char path[256];
	char *buffer = NULL;

	strupr(resn);
	if ((p=IsInDTex(resn,list,len))>=0)
	{
		strcpy(path,dirn);
		strcat(path,"\\");
		strcat(path,resn);
		strcat(path,extn);

		st = fopen(path,"rb");
		if (st!=NULL)
		{
			n = filelen(st);
			if (n>0)
			{
				buffer = malloc(n);
				assert(buffer);
				fread(buffer,n,1,st);
				fclose(st);

				fwrite(buffer,n,1,tst);
				free(buffer);

				AddDirEntry(dir,*offs,n,resn);
				Advance(offs,n,tst);
			}
		}
		else printf("Warning: %s not made\n",path);
	}
}

// Read a .BMP file and embed in a wad as a DOOM picture
// memory balanced except directory extension

void WritePicControl(DTex *list,int len,FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir,int pf)
{
	int p,n;
	FILE *st;
	char path[256];
	BYTE *out = NULL;
	int logw=0,logh=0,rwid=0;
	PicData pic;

	strupr(resn);
	if ((p=IsInDTex(resn,list,len))>=0)
	{
		strcpy(path,dirn);
		strcat(path,"\\");
		strcat(path,resn);
		strcat(path,extn);

		ReadBMP(&out,&logw,&rwid,&logh,path);	// allocs out
		if (logw>0)
		{
			PatchFormat=pf;						// allocs pic
			ConvertToPicture(out,logw,rwid,logh,list[p].num1,list[p].num2,&pic);
	
			WritePictureImage(tst,ftell(tst),&pic);
			n=GetPictureLen(&pic);
			free(pic.coloffs);
			free(pic.posts);

			AddDirEntry(dir,*offs,n,resn);
			Advance(offs,n,tst);
		}
		else printf("Warning: %s not made\n",path);
		free(out);
	}
}

// Read a .WAV file and embed in a wad as a DOOM sound
// memory balanced except directory extension

void WriteSoundControl(DTex *list,int len,FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir,int pf)
{
	int p,n;
	FILE *st;
	char path[256];
	char *buffer = NULL;
	WaveHeader whdr;
	RawHeader rhdr;

	strupr(resn);
	if ((p=IsInDTex(resn,list,len))>=0)
	{
		strcpy(path,dirn);
		strcat(path,"\\");
		strcat(path,resn);
		strcat(path,extn);

		st = fopen(path,"rb");
		if (st!=NULL)
		{
			fread(&whdr,sizeof(WaveHeader),1,st);
			if (strnicmp(whdr.riff,"RIFF",4)!=0 || strnicmp(whdr.wave,"WAVEfmt ",8)!=0)
				printf("Warning: not a valid .WAV file: %s\n",path);
			if ((whdr.s1!=0x2b11 && whdr.s1!=0x5622) || (whdr.s2!=0x2b11 && whdr.s2!=0x5622))
				printf("Warning: bad sample rate %s: %d %d\n",path,whdr.s1,whdr.s2);

			n = whdr.nsamps;

			if (n>0)
			{
				buffer = malloc(n);
				assert(buffer);
				fread(buffer,n,1,st);
				fclose(st);

				rhdr.three=3;
				rhdr.zero=0;
				rhdr.srate=11025;
				rhdr.nsamps=whdr.nsamps;

				fwrite(&rhdr,sizeof(RawHeader),1,tst);
				fwrite(buffer,n,1,tst);
				free(buffer);

				AddDirEntry(dir,*offs,n+sizeof(RawHeader),resn);
				Advance(offs,n+sizeof(RawHeader),tst);
			}
			else printf("Error: wave %s has negative number of samples, skipping\n",resn);

		}
		else printf("Warning: %s not made\n",path);
	}
}

// Read a .BMP file and embed in a wad as a DOOM flat
// memory balanced except directory extension

void WriteFlatControl(DTex *list,int len,FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir,int pf)
{
	int p,n,i;
	FILE *st;
	char path[256];
	char *buffer = NULL;
	BYTE *out = NULL;
	int logw=0,logh=0,rwid=0;

	strupr(resn);
	if ((p=IsInDTex(resn,list,len))>=0)
	{
		strcpy(path,dirn);
		strcat(path,"\\");
		strcat(path,resn);
		strcat(path,extn);

		ReadBMP(&out,&logw,&rwid,&logh,path);	// allocs out
		if (logw>0)
		{
			PatchFormat=pf;
			n = 4096;
			fwrite(out,n,1,tst);
			AddDirEntry(dir,*offs,n,resn);
			Advance(offs,n,tst);
		}
		else printf("Warning: %s not made\n",path);
		free(out);
	}
}

// Process a Lump for MAK
// no allocation

void WriteControlLump(FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir)
{
	WriteGenControl(clump,nclump,tst,resn,dirn,extn,offs,dir);
}

// Process a Sound for MAK
// no allocation

void WriteControlSound(FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir)
{
	if (resn[1]=='S')
		WriteSoundControl(csound,ncsoun,tst,resn,dirn,extn,offs,dir,0);
	else
		WriteGenControl(csound,ncsoun,tst,resn,dirn,extn,offs,dir);
}

// Process a Music for MAK
// no allocation

void WriteControlMusic(FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir)
{
	WriteGenControl(cmusic,ncmusi,tst,resn,dirn,extn,offs,dir);
}

// Process a Graphic for MAK
// no allocation

void WriteControlGraphic(FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir)
{
	WritePicControl(cgraphic,ncgrap,tst,resn,dirn,extn,offs,dir,0);
}

// Process a Sprite for MAK
// no allocation

void WriteControlSprite(FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir)
{
	WritePicControl(csprite,ncspri,tst,resn,dirn,extn,offs,dir,0);
}

// Process a Flat for MAK
// no allocation

void WriteControlFlat(FILE *tst,char *resn,char *dirn,char *extn,int *offs,WadDir *dir)
{
	WriteFlatControl(cflat,ncflat,tst,resn,dirn,extn,offs,dir,0);
}

// This routine embeds an external level in the MAK output wad
// memory balanced except for directory extension

void WriteControlLevel(FILE *tst,int p,char *dirn,char *extn,int *offs,WadDir *dir)
{
	int i,n,levn;
	FILE *st;
	char path[256],resn[256],*tok;
	char mapn[64],filn[256];
	char *buffer = NULL;
	WadDir wd;
	WadHeader wh;
	int gotmap=0;

	// if level control string is of form MAPnn=filename then parse out
	// filename and note level it is intended for
	strcpy(path,clevel[p].name);
	strupr(path);
	tok = strtok(path,"=");
	if (tok!=NULL)
	{
		strcpy(mapn,tok);
		strcpy(resn,tok);
		strcpy(filn,tok);
		tok = strtok(NULL,"=");
		if (tok!=NULL)
			strcpy(filn,tok);
	}
	else
	{
		strcpy(mapn,"");
		strcpy(filn,"");
		strcpy(resn,"");
	}
	strupr(mapn);

	if (Verbose)
		printf("mapn: %s filn: %s resn: %s\n",mapn,filn,resn);

	strcpy(path,dirn);
	strcat(path,"\\");
	strcat(path,filn);
	strcat(path,extn);

	st = fopen(path,"rb");
	if (st!=NULL)
	{
		fread(&wh,sizeof(WadHeader),1,st);
		InitDir(&wd);
		ReadDir(&wh,&wd,st);

		for (i=0;i<wd.nentries;i++)
		{
			if (IsMapResource(wd.dirp+i))
			{
				n = wd.dirp[i].reslen;
				if (n>0)
				{
					buffer = malloc(n);
					assert(buffer);
					fseek(st,wd.dirp[i].resptr,SEEK_SET);
					fread(buffer,n,1,st);

					fwrite(buffer,n,1,tst);
					free(buffer);
				}

				AddDirEntry(dir,*offs,n,wd.dirp[i].resname);
				Advance(offs,n,tst);
			}
			else if (IsMapName(wd.dirp+i))
			{
				if (gotmap) break;
				AddDirEntry(dir,*offs,0,mapn);
				gotmap=1;
			}
			else if (gotmap)
				break;
		}
		free(wd.dirp);
	}
	else
	{
		printf("Can't open level: %s\n",path);
		exit(1);
	}
	fclose(st);
}

// This subroutine actually writes the MAK output wad
// memory balanced except for directory extension

void WriteControlWad(char *tname,WadTextures *wt)
{
	int i,j,k,m,n,o,p;
	char *buffer=NULL,*q;
	int offs = 0,noend=1,npats;
	char bstr[16],ostr[16],pnam[128],mapn[8];
	PicData pic;
	int logw,logh,rwid;
	BYTE *out=NULL;
	FILE *tst,*st;	
	WadDir dir;		
	WadHeader hdr;	

	q = strrchr(tname,'.');
	if (q!=NULL) *q='\0';
	strcat(tname,".WAD");

	tst = fopen(tname,"wb");
	strncpy(hdr.wadtag,"PWAD",4);
	hdr.waddirlen=0;
	hdr.waddiraddr=0;
	fseek(tst,0,SEEK_SET);
	fwrite(&hdr,sizeof(WadHeader),1,tst);
	offs = sizeof(WadHeader);
	InitDir(&dir);

	// Write the lumps: PLAYPAL, COLORMAP, ENDDOOM, DEMO1-3, GENMIDI, DMXGUSC
	// Each is an exact binary image of the wad resource data

	if (DoLumps)
	{
		m=dir.nentries;
		for (i=I_PLAYPAL;i<=I_DEMO3;i++)
		{
			WriteControlLump(tst,resstr[i],"LUMPS",".LMP",&offs,&dir);
			if (dir.nentries>m)
				printf("Inserted %-1d LUMPS: %8s\r",dir.nentries-m,resstr[i]);
		}
		if (dir.nentries>m)
			printf("Inserted %-1d LUMPS  %s\n",dir.nentries-m,"        ");
	}

	// Now write the levels: MAP01 thru MAP32

	if (DoWads)
	{
		m=dir.nentries;
		for (i=1;i<=32;i++)
		{
			sprintf(mapn,"MAP%02d",i);
			if ((p=IsInDTexN(mapn,clevel,ncleve,5))>=0)
				WriteControlLevel(tst,p,"LEVELS",".WAD",&offs,&dir);
			if (dir.nentries!=m)
				printf("Inserted %-2d LEVELS: %5s\r",(dir.nentries-m)/11,mapn);
		}
		if (dir.nentries>m)
			printf("Inserted %-2d LEVELS  %s\n",(dir.nentries-m)/11,"     ");
	}

	// if there are pnames in the output WadTextures, write the PNAMES resource

	if (DoTextures)
	{
		// if textures exist in the output WadTextures, write the TEXTURE1 resource

		if (wt->t1.N)
		{
			fwrite(&wt->t1.N,sizeof(int),1,tst);
			fwrite(wt->t1.offs,sizeof(int),wt->t1.N,tst);
			for (j=0,npats=0;j<wt->t1.N;j++)
			{
				fwrite(wt->t1.textures+j,sizeof(TextureDesc)-sizeof(Patch *),1,tst);
				fwrite(wt->t1.textures[j].patches,sizeof(Patch),wt->t1.textures[j].npatches,tst);
				npats += wt->t1.textures[j].npatches;
			}
			n = sizeof(int) + sizeof(int) * wt->t1.N
				+ (sizeof(TextureDesc)-sizeof(Patch *)) * wt->t1.N
				+ npats * sizeof(Patch);
			AddDirEntry(&dir,offs,n,"TEXTURE1");
			Advance(&offs,n,tst);
			printf("Inserted TEXTURE1\n");
		}

		// if pnames exist write the PNAMES too

		if (wt->pnames.npnames)
		{
			fwrite(&wt->pnames.npnames,sizeof(int),1,tst);
			fwrite(wt->pnames.patchname[0],8,wt->pnames.npnames,tst);
			n = sizeof(int) + 8 * wt->pnames.npnames;
			AddDirEntry(&dir,offs,n,"PNAMES");
			Advance(&offs,n,tst);
			printf("Inserted PNAMES\n");
		}
	}

	if (DoLumps)
	{
		m=dir.nentries;
		for (i=I_GENMIDI;i<=I_DMXGUSC;i++)
		{
			WriteControlLump(tst,resstr[i],"LUMPS",".LMP",&offs,&dir);
			if (dir.nentries>m)
				printf("Inserted %-1d MIDI LUMPS: %8s\r",dir.nentries-m,resstr[i]);
		}
		if (dir.nentries>m)
			printf("Inserted %d MIDI LUMPS  %s        \n",dir.nentries-m,"        ");
	}

	// add the first sound block, if any

	if (DoEffects)
	{
		m = dir.nentries;
		for (i=I_DPPISTOL;i<=I_DSMETAL;i++)
		{
			WriteControlSound(tst,resstr[i],"SOUNDS",resstr[i][1]=='P'? ".TXT" : ".WAV",&offs,&dir);
			if (dir.nentries!=m)
				printf("Inserted %-3d SOUNDS: %8s\r",dir.nentries-m,resstr[i]);
		}
		if (dir.nentries>m)
			printf("Inserted %d SOUNDS  %s       \n",dir.nentries-m,"        ");
	}

	// add the musics, if any

	if (DoMusics)
	{
		m = dir.nentries;
		for (i=I_D_RUNNIN;i<=I_D_DM2INT;i++)
		{
			WriteControlMusic(tst,resstr[i],"MUSICS",".MUS",&offs,&dir);
			if (dir.nentries!=m)
				printf("Inserted %-2d MUSICS: %8s\r",dir.nentries-m,resstr[i]);
		}
		if (dir.nentries>m)
			printf("Inserted %d MUSICS  %s       \n",dir.nentries-m,"        ");
	}

	// add the second sound block, if any

	if (DoEffects)
	{
		m = dir.nentries;
		for (i=I_DPDSHTGN;i<=I_DSRADIO;i++)
		{
			WriteControlSound(tst,resstr[i],"SOUNDS",resstr[i][1]=='P'? ".TXT" : ".WAV",&offs,&dir);
			if (dir.nentries!=m)
				printf("Inserted %-3d SOUNDS: %8s\r",dir.nentries-m,resstr[i]);
		}
		if (dir.nentries>m)
			printf("Inserted %d SOUNDS  %s        \n",dir.nentries-m,"        ");
	}

	// add the graphics, if any

	if (DoGraphics)
	{
		m = dir.nentries;
		for (i=I_HELP;i<=I_INTERPIC;i++)
		{
			WriteControlGraphic(tst,resstr[i],"GRAPHICS",".BMP",&offs,&dir);
			if (dir.nentries!=m)
				printf("Inserted %-3d GRAPHICS: %8s\r",dir.nentries-m,resstr[i]);
		}
		if (dir.nentries>m)
			printf("Inserted %d GRAPHICS  %s          \n",dir.nentries-m,"        ");
	}

	// add the sprites, if any

	if (DoSprites)
	{
		if (ncspri>0)
		{
			if (MakeIwad || MakeRwad)
				AddDirEntry(&dir,offs,0,"S_START");
			else
				AddDirEntry(&dir,offs,0,"SS_START");
			m = dir.nentries;
			for (i=0;i<ncspri;i++)
			{
				WriteControlSprite(tst,csprite[i].name,"SPRITES",".BMP",&offs,&dir);
				if (ncspri>0)
					printf("Inserted %-4d SPRITES: %8s\r",dir.nentries-m,csprite[i].name);
			}
			if (dir.nentries>m)
				printf("Inserted %d SPRITES  %s          \n",dir.nentries-m,"        ");
			if (MakeIwad || MakeRwad)
				AddDirEntry(&dir,offs,0,"S_END");
			else
				AddDirEntry(&dir,offs,0,"SS_END");
		}
	}

	// add the patches, if any

	if (DoPatches)
	{
		if (wt->pnames.npnames)
		{
			if (MakeIwad || MakeRwad)
			{
				AddDirEntry(&dir,offs,0,"P_START");
				AddDirEntry(&dir,offs,0,"P1_START");
			}
			else
				AddDirEntry(&dir,offs,0,"PP_START"); // add marker
			m = dir.nentries;
			for (j=0;j<wt->pnames.npnames;j++)
			{
				strcpy(ostr,Str8(wt->pnames.patchname[j]));
				strupr(ostr);
				strcpy(pnam,"PATCHES\\");
				strcat(pnam,ostr);
				strcat(pnam,".BMP");

				ReadBMP(&out,&logw,&rwid,&logh,pnam);
				if (logw>0)
				{
					PatchFormat=1;
					ConvertToPicture(out,logw,rwid,logh,0,0,&pic);
					free(out);
					out=NULL;
			
					WritePictureImage(tst,ftell(tst),&pic);
					n=GetPictureLen(&pic);
					free(pic.coloffs);
					free(pic.posts);	   
					AddDirEntry(&dir,offs,n,ostr);
					Advance(&offs,n,tst);
					printf("Inserted %-4d PATCHES: %8s\r",dir.nentries-m,ostr);
				}
			}
			if (dir.nentries>m)
				printf("Inserted %d PATCHES  %s        \n",dir.nentries-m,"        ");
			if (MakeIwad || MakeRwad)
			{
				AddDirEntry(&dir,offs,0,"P1_END");
				AddDirEntry(&dir,offs,0,"P2_START");
				AddDirEntry(&dir,offs,0,"P2_END");
				AddDirEntry(&dir,offs,0,"P3_START");
				AddDirEntry(&dir,offs,0,"P3_END");
			 	AddDirEntry(&dir,offs,0,"P_END"); // and marker
			}
			else
			 	AddDirEntry(&dir,offs,0,"PP_END"); // and marker
		}
	}

	// Add the flats, if any

	if (DoFlats)
	{
		if (ncflat>0)
		{
			if (MakeIwad || MakeRwad)
			{
				AddDirEntry(&dir,offs,0,"F_START");
				AddDirEntry(&dir,offs,0,"F1_START");
			}
			else
				AddDirEntry(&dir,offs,0,"FF_START");
			m = dir.nentries;
			for (i=0;i<ncflat;i++)
			{
				WriteControlFlat(tst,cflat[i].name,"FLATS",".BMP",&offs,&dir);
				printf("Inserted %-3d FLATS: %8s\r",dir.nentries-m,cflat[i].name);
			}
			if (dir.nentries>m)
				printf("Inserted %d FLATS  %s         \n",dir.nentries-m,"        ");
			if (MakeIwad || MakeRwad)
			{
				AddDirEntry(&dir,offs,0,"F1_END");
				AddDirEntry(&dir,offs,0,"F2_START");
				AddDirEntry(&dir,offs,0,"F2_END");
				AddDirEntry(&dir,offs,0,"F3_START");
				AddDirEntry(&dir,offs,0,"F3_END");
				AddDirEntry(&dir,offs,0,"F_END");
			}
			else
				AddDirEntry(&dir,offs,0,"FF_END");
		}
	}

	// write the output wad's directory
	fwrite(dir.dirp,sizeof(WadDirEntry),dir.nentries,tst);
	printf("Inserted %d Directory Entries at %XH\n",dir.nentries,offs);

	// finally rewrite the output wad's header and close
	fseek(tst,0,SEEK_SET);
	if (MakeIwad) strncpy(hdr.wadtag,"IWAD",4);
	hdr.waddirlen = dir.nentries;
	hdr.waddiraddr = offs;
	fwrite(&hdr,sizeof(WadHeader),1,tst);
	fclose(tst);
	printf("Rewrote Directory Header, file closed\n");
}

// This routine implements MAK
// Read a control file, and create a wad containing the resources in it

void ReadResourceControl(char *ctlname,char *wadname)
{
	FILE *cst=NULL,*cst2=NULL;
	char buffer[256],*p,*q,*r,secname[128];
	char *tok,token1[16],token2[16],token3[16],token4[256];
	TextureDesc td;
	int i,j,ispatch=0,npts=0,redir=0;
	WadTextures wt;
	int lineno=0;
	char **toks=NULL;
	int ntoks=0;
	BYTE *out = NULL;
	int logw,logh;
	PicData pic;
	int lasttexwid=0;

	InitTextures(&wt);

	cst = fopen(ctlname,"r");
	if (cst!=NULL)
	{
		buffer[0]='\0';
		while ((p=fgets(buffer,256,cst))!=NULL && *p!='[')
			lineno++;
		if (p!=NULL)
		{
nextsec:
			q=strrchr(buffer,']');
			if (q!=NULL) *q='\0';
			strcpy(secname,p+1);
			if (strnicmp(secname,"TEXTURES",8)==0 || strnicmp(secname,"TEXTURE1",8)==0)
			{
finishit:
				while((p=fgets(buffer,256,redir? cst2 : cst))!=NULL && *p!='[')
				{
					if (!redir) lineno++;
					if (!isspace(*p) && *p!=';')
					{
						if (!redir && strstr(p,"TEXTURE1"))
						{
							if (strchr(p,'\\')!=NULL)
							{
								r = strrchr(p,'\n');
								if (r) *r='\0';
								cst2=fopen(p,"r");
							}
							else
								cst2=fopen("TEXTURES\\TEXTURE1.TXT","r");
							if (cst2!=NULL)
								redir=1;
							else
								printf("Warning: TEXTURES\\TEXTURE1.TXT not found\n");
							goto finishit;
						}
						ispatch=0;		// texture name or
						if (*p=='*')	// patch description line?
						{
							ispatch=1;
							p++;
						}
						if ((q=strstr(p,"="))==NULL)	// normal line
						{
							Tokenize(p," \t\n",&toks,&ntoks);
							AddDTex
							(
								&ctexture,&nctext,ispatch,
								ntoks<1? "" : toks[0],
								ntoks<2? 0 : atoi(toks[1]),
								ntoks<3? 0 : atoi(toks[2])
							);
							if (!ispatch) lasttexwid=atoi(toks[1]);
							FreeDynStr(&toks,&ntoks);
						}
						else	// text font sign descriptor string
						{
							TokenizeN(p,"= \t\n,:;",&toks,&ntoks,4);
							if (ntoks==4)
							{
								out = (BYTE *) malloc(65536);
								assert(out);
								for (i=0;i<65536;i++)
									out[i]=XPARENT;
								RenderString(ispatch? lasttexwid : 0,toks[3],out,&pic);
								printf("Created %s as %d X %d patch resource\n",toks[0],pic.pwid,pic.phgt);
								mkdir("PATCHES");
								strcpy(buffer,"PATCHES\\");
								strcat(buffer,toks[0]);
								strcat(buffer,".BMP");
								WriteBMP(out,pic.pwid,pic.phgt,buffer);
								free (out);
								if (ispatch)
								{
									AddDTex
									(
										&ctexture,&nctext,ispatch,
										toks[0],
										atoi(toks[1]),
										atoi(toks[2])
									);
								}
								else
								{
									AddDTex
									(
										&ctexture,&nctext,0,
										toks[0],
										pic.pwid,
										pic.phgt
									);
									AddDTex
									(
										&ctexture,&nctext,1,
										toks[0],
										atoi(toks[1]),
										atoi(toks[2])
									);
								}
								free (pic.coloffs);
								pic.coloffs = NULL;
								free (pic.posts);
								pic.posts = NULL;
								pic.pwid = 0;
							}
							else
							{
								printf("(%d) Syntax: texture/*patch=x0,y0 font-string-descriptor\n",lineno);
							}
							FreeDynStr(&toks,&ntoks);
						}
					}
				}
				if (redir)
				{
					redir=0;
					fclose(cst2);
					goto finishit;
				}
				if (*p=='[') goto nextsec;
			}
			else if (strnicmp(secname,"LEVELS",6)==0)
			{
				while((p=fgets(buffer,256,cst))!=NULL && *p!='[')
				{
					lineno++;
					if (!isspace(*p) && *p!=';')
					{
						strcpy(token1,"");
						tok = strtok(p," \t\n");
						if (tok!=NULL)
						{
							strcpy(token1,tok);
							AddDTex(&clevel,&ncleve,0,token1,0,0);
						}
					}
				}
				if (*p=='[') goto nextsec;
			}
			else if (strnicmp(secname,"SPRITES",7)==0)
			{
				while((p=fgets(buffer,256,cst))!=NULL && *p!='[')
				{
					lineno++;
					if (!isspace(*p) && *p!=';')
					{
						strcpy(token1,"");
						strcpy(token2,"0");
						strcpy(token3,"0");
						tok = strtok(p," \t\n");
						if (tok!=NULL)
						{
							strcpy(token1,tok);
							tok = strtok(NULL," \t\n");
							if (tok!=NULL)
							{
								strcpy(token2,tok);
								tok = strtok(NULL," \t\n");
								if (tok!=NULL)
									strcpy(token3,tok);
							}
							AddDTex(&csprite,&ncspri,0,token1,atoi(token2),atoi(token3));
						}
					}
				}
				if (*p=='[') goto nextsec;
			}
			else if (strnicmp(secname,"FLATS",5)==0)
			{
				while((p=fgets(buffer,256,cst))!=NULL && *p!='[')
				{
					lineno++;
					if (!isspace(*p) && *p!=';')
					{
						strcpy(token1,"");
						tok = strtok(p," \t\n");
						if (tok!=NULL)
						{
							strcpy(token1,tok);
							AddDTex(&cflat,&ncflat,0,token1,0,0);
						}
					}
				}
				if (*p=='[') goto nextsec;
			}
			else if (strnicmp(secname,"PATCHES",7)==0)
			{
				while((p=fgets(buffer,256,cst))!=NULL && *p!='[')
				{
					lineno++;
					if (!isspace(*p) && *p!=';')
					{
						strcpy(token1,"");
						tok = strtok(p," \t\n");
						if (tok!=NULL)
						{
							strcpy(token1,tok);
							AddDTex(&cpatch,&ncpatc,0,token1,0,0);
						}
					}
				}
				if (*p=='[') goto nextsec;
			}
			else if (strnicmp(secname,"GRAPHICS",8)==0)
			{
				while((p=fgets(buffer,256,cst))!=NULL && *p!='[')
				{
					lineno++;
					if (!isspace(*p) && *p!=';')
					{
						if ((q=strstr(p,"="))==NULL)	// normal line
						{
							Tokenize(p," \t\n",&toks,&ntoks);
							AddDTex
							(
								&cgraphic,&ncgrap,0,
								ntoks<1? "" : toks[0],
								ntoks<2? 0 : atoi(toks[1]),
								ntoks<3? 0 : atoi(toks[2])
							);
							FreeDynStr(&toks,&ntoks);
						}
						else	// text font sign descriptor string
						{
							TokenizeN(p,"= \t\n,:;",&toks,&ntoks,4);
							if (ntoks==4)
							{
								if (Verbose)
									printf("%s=%s,%s %s",toks[0],toks[1],toks[2],toks[3]);
								out = (BYTE *) malloc(65536);
								assert(out);
								for (i=0;i<65536;i++)
									out[i]=XPARENT;
								PatchFormat=0;
								RenderString(0,toks[3],out,&pic);
								PatchFormat=1;
								printf("Created %s as %d X %d graphic resource\n",toks[0],pic.pwid,pic.phgt);
								mkdir("GRAPHICS");
								strcpy(buffer,"GRAPHICS\\");
								strcat(buffer,toks[0]);
								strcat(buffer,".BMP");
								WriteBMP(out,pic.pwid,pic.phgt,buffer);
								free (out);
								AddDTex
								(
									&cgraphic,&ncgrap,0,
									toks[0],
									atoi(toks[1]),
									atoi(toks[2])
								);
								free (pic.coloffs);
								pic.coloffs = NULL;
								free (pic.posts);
								pic.posts = NULL;
								pic.pwid = 0;
							}
							else
							{
								printf("(%d) Syntax: graphic=x0,y0 font-string-descriptor\n",lineno);
							}
							FreeDynStr(&toks,&ntoks);
						}
					}
				}
				if (*p=='[') goto nextsec;
			}
			else if (strnicmp(secname,"SOUNDS",6)==0)
			{
				while((p=fgets(buffer,256,cst))!=NULL && *p!='[')
				{
					lineno++;
					if (!isspace(*p) && *p!=';')
					{
						strcpy(token1,"");
						tok = strtok(p," \t\n");
						if (tok!=NULL)
						{
							strcpy(token1,tok);
							AddDTex(&csound,&ncsoun,0,token1,0,0);
						}
					}
				}
				if (*p=='[') goto nextsec;
			}
			else if (strnicmp(secname,"MUSICS",6)==0)
			{
				while((p=fgets(buffer,256,cst))!=NULL && *p!='[')
				{
					lineno++;
					if (!isspace(*p) && *p!=';')
					{
						strcpy(token1,"");
						tok = strtok(p," \t\n");
						if (tok!=NULL)
						{
							strcpy(token1,tok);
							AddDTex(&cmusic,&ncmusi,0,token1,0,0);
						}
					}
				}
				if (*p=='[') goto nextsec;
			}
			else if (strnicmp(secname,"LUMPS",5)==0)
			{
				while((p=fgets(buffer,256,cst))!=NULL && *p!='[')
				{
					lineno++;
					if (!isspace(*p) && *p!=';')
					{
						strcpy(token1,"");
						tok = strtok(p," \t\n");
						if (tok!=NULL)
						{
							strcpy(token1,tok);
							AddDTex(&clump,&nclump,0,token1,0,0);
						}
					}
				}
				if (*p=='[') goto nextsec;
			}
			else
			{
				printf("Invalid section name \"%s\" in patch wad control file\n",secname);
				exit(1);
			}
		}
		fclose(cst);
	}
	else
	{
		printf("Cannot open patch control file: %s\n",ctlname);
		exit(1);
	}
	printf("Control lines read: %d\n",nctext);
	if (Verbose)
	{
		printf("Textures:\n");
		for (i=0;i<nctext;i++)
			printf("%s %s %d %d\n",ctexture[i].name,ctexture[i].isp? "Y" :"N",ctexture[i].num1,ctexture[i].num2);
		printf("\n");
	}

	// Now create the patchname array
	wt.pnames.patchname = NULL;
	wt.pnames.npnames = 0;
	for (i=0;i<nctext;i++)
		if (ctexture[i].isp)
			AddPName(&wt.pnames,ctexture[i].name);
	for (i=0;i<ncpatc;i++)
		AddPName(&wt.pnames,cpatch[i].name);
	printf("Patchnames found: %d\n",wt.pnames.npnames);

	// Armed with a PNames resource we now build the output TEXTURE1

	wt.t1.textures = NULL;
	wt.t1.N = 0;
	td.npatches = 0;
	td.patches = NULL;
	for (i=0;i<nctext;i++)
	{
		if (!ctexture[i].isp)
		{
			if (i)
			{
				AddTexture(&wt.t1,&td,1);
				npts += td.npatches;
			}
			strncpy(td.textname,ctexture[i].name,8);
			td.res1 = td.res2 = td.res3 = td.res4 = 0;
			td.twid = ctexture[i].num1;
			td.thgt = ctexture[i].num2;
			td.npatches = 0;
			td.patches = NULL;
		}
		else
		{
			td.patches = realloc(td.patches,(td.npatches+1)*sizeof(Patch));
			assert(td.patches);
			td.patches[td.npatches].xoff = ctexture[i].num1;
			td.patches[td.npatches].yoff = ctexture[i].num2;
			td.patches[td.npatches].pidx = GetPIdx(&wt.pnames,ctexture[i].name);
			td.patches[td.npatches].cmap = 0;
			td.patches[td.npatches].stepdir = 1;
			td.npatches++;
		}
	}
	if (td.npatches)
	{
		AddTexture(&wt.t1,&td,1);
		npts += td.npatches;
	}
	printf("Textures built: %d\n",wt.t1.N);

	// Now write a wad that corresponds to the control file
	WriteControlWad(wadname,&wt);
	printf("Control file converted to wad: %s\n",wadname);

	free (td.patches);
	td.patches=NULL;

	if (wt.pnames.npnames)
		free (wt.pnames.patchname[0]);
	free (wt.pnames.patchname);
	for (i=0;i<wt.t1.N;i++)
		free (wt.t1.textures[i].patches);
	free (wt.t1.textures);
	free (wt.t1.offs);

	FreeDTex();
}

