#include "system.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <math.h>
#include <limits.h>
#include <assert.h>
#include "blockmap.h"
#include "level.h"
#include "node.h"
#include "seg.h"
#include "structs.h"
#include "util.h"
#include "wad.h"

int block_x, block_y;
int block_w, block_h;
int block_count;
static uint16_g ** block_lines;
static uint16_g *block_ptrs;
static uint16_g *block_dups;
static int block_compression;

#define DUMMY_DUP  0xFFFF

int CheckLinedefInside(int xmin, int ymin, int xmax, int ymax,
    int x1, int y1, int x2, int y2)
{
  int count = 2;
  int tmp;

  for (;;)
  {
    if (y1 > ymax)
    {
      if (y2 > ymax)
        return FALSE;
        
      x1 = x1 + (x2-x1) * (double)(ymax-y1) / (double)(y2-y1);
      y1 = ymax;
      
      count = 2;
      continue;
    }

    if (y1 < ymin)
    {
      if (y2 < ymin)
        return FALSE;
      
      x1 = x1 + (x2-x1) * (double)(ymin-y1) / (double)(y2-y1);
      y1 = ymin;
      
      count = 2;
      continue;
    }

    if (x1 > xmax)
    {
      if (x2 > xmax)
        return FALSE;
        
      y1 = y1 + (y2-y1) * (double)(xmax-x1) / (double)(x2-x1);
      x1 = xmax;

      count = 2;
      continue;
    }

    if (x1 < xmin)
    {
      if (x2 < xmin)
        return FALSE;
        
      y1 = y1 + (y2-y1) * (double)(xmin-x1) / (double)(x2-x1);
      x1 = xmin;

      count = 2;
      continue;
    }

    count--;

    if (count == 0)
      break;

    tmp=x1;  x1=x2;  x2=tmp;
    tmp=y1;  y1=y2;  y2=tmp;
  }

  return TRUE;
}

#define BK_NUM    0
#define BK_MAX    1
#define BK_XOR    2
#define BK_FIRST  3
#define BK_QUANTUM  32

static void BlockAdd(int blk_num, int line_index)
{
  uint16_g *cur = block_lines[blk_num];
    
  if (! cur)
  {
    block_lines[blk_num] = cur = UtilCalloc(BK_QUANTUM * 
        sizeof(uint16_g));
    cur[BK_NUM] = 0;
    cur[BK_MAX] = BK_QUANTUM;
    cur[BK_XOR] = 0x1234;
  }

  if (BK_FIRST + cur[BK_NUM] == cur[BK_MAX])
  {
    cur[BK_MAX] += BK_QUANTUM;
    block_lines[blk_num] = cur = UtilRealloc(cur, cur[BK_MAX] * 
        sizeof(uint16_g));
  }
  cur[BK_XOR] = ((cur[BK_XOR] << 4) | (cur[BK_XOR] >> 12)) ^ line_index;

  cur[BK_FIRST + cur[BK_NUM]] = line_index;
  cur[BK_NUM]++;
}

void BlockAddLine(linedef_t *L)
{
  int x1 = (int) L->start->x;
  int y1 = (int) L->start->y;
  int x2 = (int) L->end->x;
  int y2 = (int) L->end->y;

  int bx1 = (MIN(x1,x2) - block_x) / 128;
  int by1 = (MIN(y1,y2) - block_y) / 128;
  int bx2 = (MAX(x1,x2) - block_x) / 128;
  int by2 = (MAX(y1,y2) - block_y) / 128;

  int bx, by;
  int line_index = L->index;

  if (bx1 < 0) bx1 = 0;
  if (by1 < 0) by1 = 0;
  if (bx2 >= block_w) bx2 = block_w - 1;
  if (by2 >= block_h) by2 = block_h - 1;

  if (bx2 < bx1 || by2 < by1)
    return;
  if (by1 == by2)
  {
    for (bx=bx1; bx <= bx2; bx++)
    {
      int blk_num = by1 * block_w + bx;
      BlockAdd(blk_num, line_index);
    }
    return;
  }

  if (bx1 == bx2)
  {
    for (by=by1; by <= by2; by++)
    {
      int blk_num = by * block_w + bx1;
      BlockAdd(blk_num, line_index);
    }
    return;
  }

  for (by=by1; by <= by2; by++)
  for (bx=bx1; bx <= bx2; bx++)
  {
    int blk_num = by * block_w + bx;
  
    int minx = block_x + bx * 128;
    int miny = block_y + by * 128;
    int maxx = minx + 127;
    int maxy = miny + 127;

    if (CheckLinedefInside(minx, miny, maxx, maxy, x1, y1, x2, y2))
    {
      BlockAdd(blk_num, line_index);
    }
  }
}

void CreateBlockmap(void)
{
  int i;
  block_lines = UtilCalloc(block_count * sizeof(uint16_g *));
  for (i=0; i < num_linedefs; i++)
  {
    linedef_t *L = LookupLinedef(i);
    if (L->zero_len)
      continue;
    BlockAddLine(L);
  }
}

int BlockCompare(const void *p1, const void *p2)
{
  int blk_num1 = ((const uint16_g *) p1)[0];
  int blk_num2 = ((const uint16_g *) p2)[0];
  const uint16_g *A = block_lines[blk_num1];
  const uint16_g *B = block_lines[blk_num2];
  if (A == B)
    return 0;
  if (A == NULL) return -1;
  if (B == NULL) return +1;
  if (A[BK_NUM] != B[BK_NUM])
  {
    return A[BK_NUM] - B[BK_NUM];
  }
  if (A[BK_XOR] != B[BK_XOR])
  {
    return A[BK_XOR] - B[BK_XOR];
  }
  return memcmp(A+BK_FIRST, B+BK_FIRST, A[BK_NUM] * sizeof(uint16_g));
}

void CompressBlockmap(void)
{
  int i;
  int cur_offset;
  int dup_count=0;

  int orig_size, new_size;

  block_ptrs = UtilCalloc(block_count * sizeof(uint16_g));
  block_dups = UtilCalloc(block_count * sizeof(uint16_g));


  for (i=0; i < block_count; i++)
    block_dups[i] = i;

  qsort(block_dups, block_count, sizeof(uint16_g), BlockCompare);
  cur_offset = 4 + block_count + 2;

  orig_size = 4 + block_count;
  new_size  = cur_offset;

  for (i=0; i < block_count; i++)
  {
    int blk_num = block_dups[i];
    int count;
    if (block_lines[blk_num] == NULL)
    {
      block_ptrs[blk_num] = 4 + block_count;
      block_dups[i] = DUMMY_DUP;

      orig_size += 2;
      continue;
    }

    count = 2 + block_lines[blk_num][BK_NUM];
    if (i+1 < block_count && 
        BlockCompare(block_dups + i, block_dups + i+1) == 0)
    {
      block_ptrs[blk_num] = cur_offset;
      block_dups[i] = DUMMY_DUP;
      UtilFree(block_lines[blk_num]);
      block_lines[blk_num] = NULL;
      
      dup_count++;

      orig_size += count;
      continue;
    }
    block_ptrs[blk_num] = cur_offset;

    cur_offset += count;

    orig_size += count;
    new_size  += count;
  }
  block_compression = (orig_size - new_size) * 100 / orig_size;

  if (block_compression < 0)
    block_compression = 0;
}


void WriteBlockmap(void)
{
  int i;
  
  raw_blockmap_header_t header;

  lump_t *lump = CreateLevelLump("BLOCKMAP");

  uint16_g null_block[2] = { 0x0000, 0xFFFF };
  uint16_g m_zero = 0x0000;
  uint16_g m_neg1 = 0xFFFF;

  header.x_origin = UINT16(block_x);
  header.y_origin = UINT16(block_y);
  header.x_blocks = UINT16(block_w);
  header.y_blocks = UINT16(block_h);
  
  AppendLevelLump(lump, &header, sizeof(header));
  for (i=0; i < block_count; i++)
  {
    uint16_g ptr = UINT16(block_ptrs[i]);
    AppendLevelLump(lump, &ptr, sizeof(uint16_g));
  }
  AppendLevelLump(lump, null_block, sizeof(null_block));
  for (i=0; i < block_count; i++)
  {
    int blk_num = block_dups[i];
    uint16_g *blk;
    if (blk_num == DUMMY_DUP)
      continue;
    blk = block_lines[blk_num];
    AppendLevelLump(lump, &m_zero, sizeof(uint16_g));
    AppendLevelLump(lump, blk + BK_FIRST, blk[BK_NUM] * sizeof(uint16_g));
    AppendLevelLump(lump, &m_neg1, sizeof(uint16_g));
  }
}

void FreeBlockmap(void)
{
  int i;
  for (i=0; i < block_count; i++)
  {
    if (block_lines[i])
      UtilFree(block_lines[i]);
  }
  UtilFree(block_lines);
  UtilFree(block_ptrs);
  UtilFree(block_dups);
}

void FindBlockmapLimits(bbox_t *bbox)
{
  int i;
  bbox->minx = bbox->miny = SHRT_MAX;
  bbox->maxx = bbox->maxy = SHRT_MIN;
  for (i=0; i < num_linedefs; i++)
  {
    linedef_t *L = LookupLinedef(i);
    if (! L->zero_len)
    {
      float_g x1 = L->start->x;
      float_g y1 = L->start->y;
      float_g x2 = L->end->x;
      float_g y2 = L->end->y;
      int lx = floor(MIN(x1, x2));
      int ly = floor(MIN(y1, y2));
      int hx = ceil(MAX(x1, x2));
      int hy = ceil(MAX(y1, y2));
      if (lx < bbox->minx) bbox->minx = lx;
      if (ly < bbox->miny) bbox->miny = ly;
      if (hx > bbox->maxx) bbox->maxx = hx;
      if (hy > bbox->maxy) bbox->maxy = hy;
    }
  }
}

void TruncateBlockmap(void)
{
  int orig_w = block_w;
  int orig_h = block_h;
  while (block_w * block_h > 64000)
  {
    block_w -= block_w / 8;
    block_h -= block_h / 8;
  }
  block_count = block_w * block_h;
  block_x += (block_w - orig_w) * 128 / 2;
  block_y += (block_h - orig_h) * 128 / 2;
}

void InitBlockmap(void)
{
  bbox_t map_bbox;
  FindBlockmapLimits(&map_bbox);
  PrintMsg("Map goes from (%d,%d) to (%d,%d)\n",
      map_bbox.minx, map_bbox.miny, map_bbox.maxx, map_bbox.maxy);
  block_x = map_bbox.minx - (map_bbox.minx & 0x7);
  block_y = map_bbox.miny - (map_bbox.miny & 0x7);
  block_w = ((map_bbox.maxx - block_x) / 128) + 1;
  block_h = ((map_bbox.maxy - block_y) / 128) + 1;
  block_count = block_w * block_h;
}

void PutBlockmap(void)
{
  if (block_count > 64000)
    TruncateBlockmap();
  CreateBlockmap();
  CompressBlockmap();
  WriteBlockmap();
  PrintMsg("Completed blockmap building (compression: %d%%)\n",
      block_compression);
  FreeBlockmap();
}