#include "system.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <math.h>
#include <limits.h>
#include <assert.h>
#include "blockmap.h"
#include "level.h"
#include "node.h"
#include "seg.h"
#include "structs.h"
#include "util.h"
#include "wad.h"

superblock_t *quick_alloc_supers = NULL;
int PointOnLineSide(seg_t *part, float_g x, float_g y)
{
  float_g perp = ComputePerpDist(part, x, y);
  
  if (fabs(perp) <= DIST_EPSILON)
    return 0;
  
  return (perp < 0) ? -1 : +1;
}
#define IFFY_LEN 0.000001
int BoxOnLineSide(superblock_t *box, seg_t *part)
{
  float_g x1 = (float_g)box->x1;
  float_g y1 = (float_g)box->y1 - IFFY_LEN;
  float_g x2 = (float_g)box->x2;
  float_g y2 = (float_g)box->y2 + IFFY_LEN;

  int p1, p2;

  if (part->pdx == 0)
  {
    p1 = (x1 > part->psx) ? +1 : -1;
    p2 = (x2 > part->psx) ? +1 : -1;

    if (part->pdy < 0)
    {
      p1 = -p1;
      p2 = -p2;
    }
  }
  else if (part->pdy == 0)
  {
    p1 = (y1 < part->psy) ? +1 : -1;
    p2 = (y2 < part->psy) ? +1 : -1;

    if (part->pdx < 0)
    {
      p1 = -p1;
      p2 = -p2;
    }
  }
  else if (part->pdx * part->pdy > 0)
  {
    p1 = PointOnLineSide(part, x1, y2);
    p2 = PointOnLineSide(part, x2, y1);
  }
  else
  {
    p1 = PointOnLineSide(part, x1, y1);
    p2 = PointOnLineSide(part, x2, y2);
  }

  if (p1 == p2)
    return p1;

  return 0;
}

superblock_t *NewSuperBlock(void)
{
  superblock_t *block;

  if (quick_alloc_supers == NULL)
    return UtilCalloc(sizeof(superblock_t));

  block = quick_alloc_supers;
  quick_alloc_supers = block->subs[0];
  memset(block, 0, sizeof(superblock_t));

  return block;
}

void FreeQuickAllocSupers(void)
{
  while (quick_alloc_supers)
  {
    superblock_t *block = quick_alloc_supers;
    quick_alloc_supers = block->subs[0];

    UtilFree(block);
  }
}

void FreeSuper(superblock_t *block)
{
  int num;

  if (block->segs)
    block->segs = NULL;
  for (num=0; num < 2; num++)
  {
    if (block->subs[num])
      FreeSuper(block->subs[num]);
  } 


  block->subs[0] = quick_alloc_supers;
  quick_alloc_supers = block;
}

void AddSegToSuper(superblock_t *block, seg_t *seg)
{
  for (;;)
  {
    int p1, p2;
    int child;

    int x_mid = (block->x1 + block->x2) / 2;
    int y_mid = (block->y1 + block->y2) / 2;

    superblock_t *sub;

    if (seg->linedef)
      block->real_num++;
   
    if (SUPER_IS_LEAF(block))
    {
      seg->next = block->segs;
      seg->block = block;

      block->segs = seg;
      return;
    }

    if (block->x2 - block->x1 >= block->y2 - block->y1)
    {

      p1 = seg->start->x >= x_mid;
      p2 = seg->end->x   >= x_mid;
    }
    else
    {

      p1 = seg->start->y >= y_mid;
      p2 = seg->end->y   >= y_mid;
    }

    if (p1 && p2)
      child = 1;
    else if (!p1 && !p2)
      child = 0;
    else
    {

      seg->next = block->segs;
      seg->block = block;

      block->segs = seg;
      return;
    }

    if (! block->subs[child])
    {
      block->subs[child] = sub = NewSuperBlock();
      sub->parent = block;

      if (block->x2 - block->x1 >= block->y2 - block->y1)
      {
        sub->x1 = child ? x_mid : block->x1;
        sub->y1 = block->y1;

        sub->x2 = child ? block->x2 : x_mid;
        sub->y2 = block->y2;
      }
      else
      {
        sub->x1 = block->x1;
        sub->y1 = child ? y_mid : block->y1;

        sub->x2 = block->x2;
        sub->y2 = child ? block->y2 : y_mid;
      }
    }

    block = block->subs[child];
  }
}

void SplitSegInSuper(superblock_t *block, seg_t *seg)
{
  do
  {
    if (seg->linedef)
      block->real_num++;
 
    block = block->parent;
  }
  while (block != NULL);
}

static seg_t *CreateOneSeg(linedef_t *line, vertex_t *start, vertex_t *end,
    sidedef_t *side, int side_num)
{
  seg_t *seg = NewSeg();
 
  seg->start   = start;
  seg->end     = end;
  seg->linedef = line;
  seg->side    = side_num;
  seg->sector  = side->sector;

  seg->source_line = seg->linedef;
  seg->index = -1;

  RecomputeSeg(seg);

  return seg;
}

superblock_t *CreateSegs(void)
{
  int i;
  superblock_t *block;
  PrintMsg("Creating Segs, wait\n");
  block = NewSuperBlock();
  block->x1 = block_x;
  block->y1 = block_y;
  block->x2 = block->x1 + 128 * RoundPOW2(block_w);
  block->y2 = block->y1 + 128 * RoundPOW2(block_h);

  for (i=0; i < num_linedefs; i++)
  {
    linedef_t *line = LookupLinedef(i);
    if (line->zero_len)
      continue;
    if (line->right)
      AddSegToSuper(block, CreateOneSeg(line, line->start, line->end, line->right, 0));
    if (line->left)
      AddSegToSuper(block,CreateOneSeg(line, line->end, line->start, line->left, 1));
  }
  return block;
}

static void DetermineMiddle(subsec_t *sub)
{
  seg_t *cur;

  float_g mid_x=0, mid_y=0;
  int total=0;
  for (cur=sub->seg_list; cur; cur=cur->next)
  {
    mid_x += cur->start->x + cur->end->x;
    mid_y += cur->start->y + cur->end->y;

    total += 2;
  }

  sub->mid_x = mid_x / total;
  sub->mid_y = mid_y / total;
}

void ClockwiseOrder(subsec_t *sub)
{
  seg_t *cur;
  seg_t ** array;
  seg_t *seg_buffer[32];

  int i;
  int total = 0;
  for (cur=sub->seg_list; cur; cur=cur->next)
    total++;
  if (total <= 32)
    array = seg_buffer;
  else
    array = UtilCalloc(total * sizeof(seg_t *));

  for (cur=sub->seg_list, i=0; cur; cur=cur->next, i++)
    array[i] = cur;

  i = 0;

  while (i+1 < total)
  {
    seg_t *A = array[i];
    seg_t *B = array[i+1];

    angle_g angle1, angle2;

    angle1 = ComputeAngle(A->start->x-sub->mid_x, A->start->y-sub->mid_y);
    angle2 = ComputeAngle(B->start->x-sub->mid_x, B->start->y-sub->mid_y);

    if (angle1 + ANG_EPSILON < angle2)
    {
      array[i] = B;
      array[i+1] = A;
      if (i > 0)
        i--;
    }
    else
    {
      i++;
    }
  }

  sub->seg_list = NULL;

  for (i=total-1; i >= 0; i--)
  {
    array[i]->next = sub->seg_list;
    sub->seg_list  = array[i];
  }
 
  if (total > 32)
    UtilFree(array);
}

void RenumberSubsecSegs(subsec_t *sub)
{
  seg_t *cur;

  sub->seg_count = 0;

  for (cur=sub->seg_list; cur; cur=cur->next)
  {
    cur->index = num_complete_seg;
    num_complete_seg++;
    sub->seg_count++;
  }
}


static void CreateSubsecWorker(subsec_t *sub, superblock_t *block)
{
  int num;

  while (block->segs)
  {
    seg_t *cur = block->segs;
    block->segs = cur->next;
    cur->next = sub->seg_list;
    cur->block = NULL;

    sub->seg_list = cur;
  }

  for (num=0; num < 2; num++)
  {
    superblock_t *A = block->subs[num];

    if (A)
    {
      CreateSubsecWorker(sub, A);

      FreeSuper(A);
      block->subs[num] = NULL;
    }
  }

  block->real_num = 0;
}

subsec_t *CreateSubsec(superblock_t *seg_list)
{
  subsec_t *sub = NewSubsec();
  sub->index = num_subsecs - 1;
  CreateSubsecWorker(sub, seg_list);

  DetermineMiddle(sub);

  return sub;
}

int ComputeHeight(node_t *node)
{
  if (node)
  {
    int left, right;
    
    right = ComputeHeight(node->r.node);
    left  = ComputeHeight(node->l.node);

    return MAX(left, right) + 1;
  }

  return 1;
}

glbsp_ret_e BuildNodes(superblock_t *seg_list, 
    node_t ** N, subsec_t ** S)
{
  node_t *node;
  seg_t *best;

  superblock_t *rights;
  superblock_t *lefts;

  intersection_t *cut_list;

  glbsp_ret_e ret;

  *N = NULL;
  *S = NULL;

  if (cur_comms->cancelled)
    return GLBSP_E_Cancelled;

  best = PickNode(seg_list);

  if (best == NULL)
  {
    if (cur_comms->cancelled)
      return GLBSP_E_Cancelled;

    *S = CreateSubsec(seg_list);
    return GLBSP_E_OK;
  }

  lefts  = (superblock_t *) NewSuperBlock();
  rights = (superblock_t *) NewSuperBlock();

  lefts->x1 = rights->x1 = seg_list->x1;
  lefts->y1 = rights->y1 = seg_list->y1;
  lefts->x2 = rights->x2 = seg_list->x2;
  lefts->y2 = rights->y2 = seg_list->y2;

  cut_list = NULL;

  SeparateSegs(seg_list, best, lefts, rights, &cut_list);

  *N = node = NewNode();

  node->x  = (int)best->psx;
  node->y  = (int)best->psy;
  node->dx = (int)best->pdx;
  node->dy = (int)best->pdy;

  FindLimits(lefts,  &node->l.bounds);
  FindLimits(rights, &node->r.bounds);

  ret = BuildNodes(lefts,  &node->l.node, &node->l.subsec);
  FreeSuper(lefts);

  if (ret != GLBSP_E_OK)
  {
    FreeSuper(rights);
    return ret;
  }

  ret = BuildNodes(rights, &node->r.node, &node->r.subsec);
  FreeSuper(rights);

  return ret;
}

void ClockwiseBspTree(node_t *root)
{
  int i;

  (void) root;

  for (i=0; i < num_subsecs; i++)
  {
    subsec_t *sub = LookupSubsec(i);

    ClockwiseOrder(sub);
    RenumberSubsecSegs(sub);
  }
}

void NormaliseSubsector(subsec_t *sub)
{
  seg_t *new_head = NULL;
  seg_t *new_tail = NULL;

  while (sub->seg_list)
  {
    seg_t *cur = sub->seg_list;
    sub->seg_list = cur->next;

    if (cur->linedef)
    {
      cur->next = NULL;

      if (new_tail)
        new_tail->next = cur;
      else
        new_head = cur;

      new_tail = cur;

      cur->index = -1;
    }
    else
      cur->index = 1<<24;
  }

  sub->seg_list = new_head;
}

void NormaliseBspTree(node_t *root)
{
  int i;

  (void) root;

  num_complete_seg = 0;

  for (i=0; i < num_subsecs; i++)
  {
    subsec_t *sub = LookupSubsec(i);

    NormaliseSubsector(sub);
    RenumberSubsecSegs(sub);
  }
}

void RoundOffSubsector(subsec_t *sub)
{
  seg_t *new_head = NULL;
  seg_t *new_tail = NULL;

  seg_t *cur;
  seg_t *last_real_degen = NULL;

  int real_total  = 0;
  int degen_total = 0;

  for (cur=sub->seg_list; cur; cur=cur->next)
  {
    if (cur->start->normal_dup)
      cur->start = cur->start->normal_dup;

    if (cur->end->normal_dup)
      cur->end = cur->end->normal_dup;
    if ((int)cur->start->x == (int)cur->end->x &&
        (int)cur->start->y == (int)cur->end->y)
    {
      cur->degenerate = 1;

      if (cur->linedef)
        last_real_degen = cur;
      
      degen_total++;
      continue;
    }
    
    if (cur->linedef)
      real_total++;
  }

  if (real_total == 0)
  {
    last_real_degen->end = NewVertexDegenerate(last_real_degen->start,
        last_real_degen->end);
    last_real_degen->degenerate = 0;
  }

  while (sub->seg_list)
  {
    cur = sub->seg_list;
    sub->seg_list = cur->next;
    if (! cur->degenerate)
    {
      cur->next = NULL;
      if (new_tail)
        new_tail->next = cur;
      else
        new_head = cur;
      new_tail = cur;
      cur->index = -1;
    }
    else
      cur->index = 1<<24;
  }
  sub->seg_list = new_head;
}

void RoundOffBspTree(node_t *root)
{
  int i;
  (void) root;
  num_complete_seg = 0;
  for (i=0; i < num_subsecs; i++)
  {
    subsec_t *sub = LookupSubsec(i);
    RoundOffSubsector(sub);
    RenumberSubsecSegs(sub);
  }
}