#include "system.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <math.h>
#include <limits.h>
#include <assert.h>
#include "blockmap.h"
#include "level.h"
#include "node.h"
#include "seg.h"
#include "structs.h"
#include "util.h"
#include "wad.h"

#define PRECIOUS_MULTIPLY  64

typedef struct eval_info_s
{
  int cost;
  int real_left;
  int real_right;
}
eval_info_t;


static intersection_t *quick_alloc_cuts = NULL;

angle_g ComputeAngle(float_g dx, float_g dy)
{
  double angle;

  if (dx == 0)
    return (dy > 0) ? 90.0 : 270.0;

  angle = atan2((double) dy, (double) dx) * 180.0 / M_PI;

  if (angle < 0) 
    angle += 360.0;

  return angle;
}

intersection_t *NewIntersection(void)
{
  intersection_t *cut;

  if (quick_alloc_cuts)
  {
    cut = quick_alloc_cuts;
    quick_alloc_cuts = cut->next;
  }
  else
  {
    cut = UtilCalloc(sizeof(intersection_t));
  }

  return cut;
}

void FreeQuickAllocCuts(void)
{
  while (quick_alloc_cuts)
  {
    intersection_t *cut = quick_alloc_cuts;
    quick_alloc_cuts = cut->next;

    UtilFree(cut);
  }
}

void RecomputeSeg(seg_t *seg)
{
  seg->psx = seg->start->x;
  seg->psy = seg->start->y;
  seg->pex = seg->end->x;
  seg->pey = seg->end->y;
  seg->pdx = seg->pex - seg->psx;
  seg->pdy = seg->pey - seg->psy;
  
  seg->p_length = ComputeDist(seg->pdx, seg->pdy);
  seg->p_angle = ComputeAngle(seg->pdx, seg->pdy);

  seg->p_perp =  seg->psy * seg->pdx - seg->psx * seg->pdy;
  seg->p_para = -seg->psx * seg->pdx - seg->psy * seg->pdy;
}

seg_t *SplitSeg(seg_t *old_seg, float_g x, float_g y)
{
  seg_t *new_seg;
  vertex_t *new_vert;

  if (old_seg->block)
    SplitSegInSuper(old_seg->block, old_seg);

  new_vert = NewVertexFromSplitSeg(old_seg, x, y);
  new_seg  = NewSeg();

  new_seg[0] = old_seg[0];
  new_seg->next = NULL;

  old_seg->end = new_vert;
  RecomputeSeg(old_seg);

  new_seg->start = new_vert;
  RecomputeSeg(new_seg);

  return new_seg;
}

void ComputeIntersection(seg_t *cur, seg_t *part,
  float_g perp_c, float_g perp_d, float_g *x, float_g *y)
{
  double ds;
  if (part->pdy == 0 && cur->pdx == 0)
  {
    *x = cur->psx;
    *y = part->psy;
    return;
  }
  if (part->pdx == 0 && cur->pdy == 0)
  {
    *x = part->psx;
    *y = cur->psy;
    return;
  }
  ds = perp_c / (perp_c - perp_d);

  if (cur->pdx == 0)
    *x = cur->psx;
  else
    *x = cur->psx + (cur->pdx * ds);

  if (cur->pdy == 0)
    *y = cur->psy;
  else
    *y = cur->psy + (cur->pdy * ds);
}

void AddIntersection(intersection_t ** cut_list,
    vertex_t *vert, seg_t *part)
{
  intersection_t *cut;
  intersection_t *after;

  for (cut=(*cut_list); cut; cut=cut->next)
  {
    if (vert == &cut->vertex)
      return;
  }

  cut = NewIntersection();

  cut->vertex = *vert;
  cut->along_dist = ComputeParallelDist(part, vert->x, vert->y);
  
  cut->l.open = VertexCheckOpen(vert, -part->pdx, -part->pdy,
      &cut->l.right, &cut->l.left);

  cut->r.open = VertexCheckOpen(vert, part->pdx, part->pdy,
      &cut->r.left, &cut->r.right);

  for (after=(*cut_list); after && after->next; after=after->next)
  { }

  while (after && cut->along_dist < after->along_dist) 
    after = after->prev;

  cut->next = after ? after->next : (*cut_list);
  cut->prev = after;

  if (after)
  {
    if (after->next)
      after->next->prev = cut;
    
    after->next = cut;
  }
  else
  {
    if (*cut_list)
      (*cut_list)->prev = cut;
    
    (*cut_list) = cut;
  }
}

int EvalPartitionWorker(superblock_t *seg_list, seg_t *part, 
    int best_cost, eval_info_t *info)
{
  seg_t *check;
  float_g a, b, fa, fb;
  int num, factor = cur_info->factor;
  num = BoxOnLineSide(seg_list, part);
  if (num < 0)
  {
    info->real_left += seg_list->real_num;
    return FALSE;
  }
  else if (num > 0)
  {
    info->real_right += seg_list->real_num;
    return FALSE;
  }

  #define ADD_LEFT()  \
        if (check->linedef) info->real_left += 1;

  #define ADD_RIGHT()  \
        if (check->linedef) info->real_right += 1; 

  for (check=seg_list->segs; check; check=check->next)
  { 
    if (info->cost > best_cost)
      return TRUE;

    if (check->source_line == part->source_line)
      a = b = fa = fb = 0;
    else
    {
      a = ComputePerpDist(part, check->psx, check->psy);
      b = ComputePerpDist(part, check->pex, check->pey);

      fa = fabs(a);
      fb = fabs(b);
    }

    if (fa <= DIST_EPSILON && fb <= DIST_EPSILON)
    {
      if (check->pdx*part->pdx + check->pdy*part->pdy < 0)
        {ADD_LEFT();} else  {ADD_RIGHT();}
    continue;
    }

    if (a > -DIST_EPSILON && b > -DIST_EPSILON){ ADD_RIGHT(); continue; }
    if (a < DIST_EPSILON && b < DIST_EPSILON){ ADD_LEFT(); continue; }

    if (check->linedef && check->linedef->is_precious)
      info->cost += 100 * factor * PRECIOUS_MULTIPLY;
    else
      info->cost += 100 * factor;
  }

  for (num=0; num < 2; num++)
    if (seg_list->subs[num])
    if (EvalPartitionWorker(seg_list->subs[num], part, best_cost, info))
      return TRUE;
  return FALSE;
}

int EvalPartition(superblock_t *seg_list, seg_t *part, 
    int best_cost)
{
  eval_info_t info;
  info.cost   = 0;
  info.real_left  = 0;
  info.real_right = 0;
  if (EvalPartitionWorker(seg_list, part, best_cost, &info))
    return -1;
  if (!info.real_left || !info.real_right)
    return -1;

  info.cost += 100 * ABS(info.real_left - info.real_right);
  if (part->pdx != 0 && part->pdy != 0)
    info.cost += 25;

  return info.cost;
}

boolean_g PickNodeWorker(superblock_t *part_list, 
    superblock_t *seg_list, seg_t ** best, int *best_cost)
{
  seg_t *part;
  int num, cost;
  for (part=part_list->segs; part&&part->linedef; part = part->next)
  {    
    cost = EvalPartition(seg_list, part, *best_cost);
    if (cost < 0 || cost >= *best_cost)
      continue;
    (*best_cost) = cost;
    (*best) = part;
  }

  for (num=0; num < 2; num++)
    if (part_list->subs[num])
      PickNodeWorker(part_list->subs[num], seg_list, best, best_cost);
  return TRUE;
}

seg_t *PickNode(superblock_t *seg_list)
{
  seg_t *best=NULL;

  int best_cost=INT_MAX;

  if (PickNodeWorker(seg_list, seg_list, &best, &best_cost)==0)
    return NULL;
  return best;
}

void DivideOneSeg(seg_t *cur, seg_t *part, 
    superblock_t *left_list, superblock_t *right_list,
    intersection_t ** cut_list)
{
  seg_t *new_seg;
  float_g x, y;
  float_g a = ComputePerpDist(part, cur->psx, cur->psy);
  float_g b = ComputePerpDist(part, cur->pex, cur->pey);
  if (cur->source_line == part->source_line)
    a = b = 0;
  if (fabs(a) <= DIST_EPSILON && fabs(b) <= DIST_EPSILON)
  {
    AddIntersection(cut_list, cur->start, part);
    AddIntersection(cut_list, cur->end, part);
    if (cur->pdx*part->pdx + cur->pdy*part->pdy < 0)
      AddSegToSuper(left_list, cur);
    else
      AddSegToSuper(right_list, cur);
    return;
  }

  if (a > -DIST_EPSILON && b > -DIST_EPSILON)
  {
    if (a < DIST_EPSILON)
      AddIntersection(cut_list, cur->start, part);
    else if (b < DIST_EPSILON)
      AddIntersection(cut_list, cur->end, part);
    AddSegToSuper(right_list, cur);
    return;
  }
  if (a < DIST_EPSILON && b < DIST_EPSILON)
  {
    if (a > -DIST_EPSILON)
      AddIntersection(cut_list, cur->start, part);
    else if (b > -DIST_EPSILON)
      AddIntersection(cut_list, cur->end, part);
    AddSegToSuper(left_list, cur);
    return;
  }
  ComputeIntersection(cur, part, a, b, &x, &y);

  new_seg = SplitSeg(cur, x, y);

  AddIntersection(cut_list, cur->end, part);

  if (a < 0)
  {
    AddSegToSuper(left_list,  cur);
    AddSegToSuper(right_list, new_seg);
  }
  else
  {
    AddSegToSuper(right_list, cur);
    AddSegToSuper(left_list,  new_seg);
  }
}

void SeparateSegs(superblock_t *seg_list, seg_t *part,
    superblock_t *lefts, superblock_t *rights,
    intersection_t ** cut_list)
{
  int num;
  while (seg_list->segs)
  {
    seg_t *cur = seg_list->segs;
    seg_list->segs = cur->next;
    cur->block = NULL;
    DivideOneSeg(cur, part, lefts, rights, cut_list);
  }
  for (num=0; num < 2; num++)
  {
    superblock_t *A = seg_list->subs[num];
    if (A)
    {
      SeparateSegs(A, part, lefts, rights, cut_list);
      FreeSuper(A);
      seg_list->subs[num] = NULL;
    }
  }

  seg_list->real_num = 0;
}


static void FindLimitWorker(superblock_t *block, bbox_t *bbox)
{
  seg_t *cur;
  int num;

  for (cur=block->segs; cur; cur=cur->next)
  {
    float_g x1 = cur->start->x;
    float_g y1 = cur->start->y;
    float_g x2 = cur->end->x;
    float_g y2 = cur->end->y;

    int lx = floor(MIN(x1, x2));
    int ly = floor(MIN(y1, y2));
    int hx = ceil(MAX(x1, x2));
    int hy = ceil(MAX(y1, y2));

    if (lx < bbox->minx) bbox->minx = lx;
    if (ly < bbox->miny) bbox->miny = ly;
    if (hx > bbox->maxx) bbox->maxx = hx;
    if (hy > bbox->maxy) bbox->maxy = hy;
  }

  for (num=0; num < 2; num++)
  {
    if (block->subs[num])
      FindLimitWorker(block->subs[num], bbox);
  }
}

void FindLimits(superblock_t *seg_list, bbox_t *bbox)
{
  bbox->minx = bbox->miny = SHRT_MAX;
  bbox->maxx = bbox->maxy = SHRT_MIN;

  FindLimitWorker(seg_list, bbox);
}