#include "system.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <math.h>
#include <limits.h>
#include <errno.h>

#include "blockmap.h"
#include "level.h"
#include "node.h"
#include "seg.h"
#include "structs.h"
#include "util.h"
#include "wad.h"


FILE *in_file = NULL;
FILE *out_file = NULL;
#define APPEND_BLKSIZE  256
#define LEVNAME_BUNCH   20
wad_t wad;

#define NUM_LEVEL_LUMPS  10

const char *level_lumps[NUM_LEVEL_LUMPS]=
{
  "THINGS", "LINEDEFS", "SIDEDEFS", "VERTEXES", "SEGS", 
  "SSECTORS", "NODES", "SECTORS", "REJECT", "BLOCKMAP"
};

int CheckMagic(const char type[4])
{
  if ((type[0] == 'I' || type[0] == 'P') && 
       type[1] == 'W' && type[2] == 'A' && type[3] == 'D')
  {
    return TRUE;
  }
  
  return FALSE;
}

int CheckLevelName(const char *name)
{
  int n;
  
  if (strlen(name) > 5)
    return FALSE;

  for (n=0; n < wad.num_level_names; n++)
  {
    if (strcmp(wad.level_names[n], name) == 0)
      return TRUE;
  }

  return FALSE;
}

int CheckLevelLumpName(const char *name)
{
  int i;

  for (i=0; i < NUM_LEVEL_LUMPS; i++)
  {
    if (strcmp(name, level_lumps[i]) == 0)
      return TRUE;
  }
  
  return FALSE;
}

void AddLevelName(const char *name)
{
  if ((wad.num_level_names % LEVNAME_BUNCH) == 0)
  {
    wad.level_names = (const char **) UtilRealloc(wad.level_names,
        (wad.num_level_names + LEVNAME_BUNCH) * sizeof(const char *));
  }

  wad.level_names[wad.num_level_names] = UtilStrDup(name);
  wad.num_level_names++;
}

lump_t *NewLump(char *name)
{
  lump_t *cur;

  cur = UtilCalloc(sizeof(lump_t));

  cur->name = name;
  cur->start = cur->new_start = -1;
  cur->flags = 0;
  cur->length = 0;
  cur->space = 0;
  cur->data = NULL;
  cur->level_list = NULL;
  cur->level_buddy = NULL;

  return cur;
}

void FreeLump(lump_t *lump)
{
  if (lump->flags & LUMP_IS_LEVEL)
  {
    while (lump->level_list)
    {
      lump_t *head = lump->level_list;
      lump->level_list = head->next;
      FreeLump(head);
    }
  }

  if (lump->data)
    UtilFree(lump->data);
  
  UtilFree(lump->name);
  UtilFree(lump);
}

boolean_g ReadHeader(const char *filename)
{
  int len;
  raw_wad_header_t header;
  char strbuf[1024];

  len = fread(&header, sizeof(header), 1, in_file);

  if (len != 1)
  {
    sprintf(strbuf, "Trouble reading wad header for %s : %s", 
      filename, strerror(errno));

    GlbspFree(cur_comms->message);
    cur_comms->message = GlbspStrDup(strbuf);
    
    return FALSE;
  }

  if (! CheckMagic(header.type))
  {
    sprintf(strbuf, "%s does not appear to be a wad file : bad magic", 
        filename);

    GlbspFree(cur_comms->message);
    cur_comms->message = GlbspStrDup(strbuf);
    
    return FALSE;
  }

  wad.kind = (header.type[0] == 'I') ? IWAD : PWAD;
  
  wad.num_entries = UINT32(header.num_entries);
  wad.dir_start   = UINT32(header.dir_start);

  wad.dir_head = NULL;
  wad.dir_tail = NULL;
  wad.current_level = NULL;
  wad.level_names = NULL;
  wad.num_level_names = 0;

  return TRUE;
}

void ReadDirEntry(void)
{
  int len;
  raw_wad_entry_t entry;
  lump_t *lump;

  len = fread(&entry, sizeof(entry), 1, in_file);

  if (len != 1)
    FatalError("Trouble reading wad directory");

  lump = NewLump(UtilStrNDup(entry.name, 8));

  lump->start = UINT32(entry.start);
  lump->length = UINT32(entry.length);

  lump->next = NULL;
  lump->prev = wad.dir_tail;

  if (wad.dir_tail)
    wad.dir_tail->next = lump;
  else
    wad.dir_head = lump;

  wad.dir_tail = lump;
}

void DetermineLevelNames(void)
{
  lump_t *L, *N;
  int i;

  for (L=wad.dir_head; L; L=L->next)
  {
    for (i=0, N=L->next; (i < 4) && N; i++, N=N->next)
      if (strcmp(N->name, level_lumps[i]) != 0)
        break;

    if (i != 4)
      continue;
      AddLevelName(L->name);
  }
}

void ProcessDirEntry(lump_t *lump)
{

  if (CheckLevelName(lump->name))
  {
    if (cur_info->load_all)
      lump->flags |= LUMP_READ_ME;
    else
      lump->flags |= LUMP_COPY_ME;

    lump->flags |= LUMP_IS_LEVEL;
    wad.current_level = lump;

    lump->next = NULL;
    lump->prev = wad.dir_tail;

    if (wad.dir_tail)
      wad.dir_tail->next = lump;
    else
      wad.dir_head = lump;

    wad.dir_tail = lump;

    return;
  }

  if (wad.current_level)
  {
    if (CheckLevelLumpName(lump->name))
    {
      if (FindLevelLump(lump->name))
      {
        FreeLump(lump);
        wad.num_entries--;
        return;
      }

      lump->flags |= LUMP_READ_ME;
      lump->next = wad.current_level->level_list;
      lump->prev = NULL;

      if (lump->next)
        lump->next->prev = lump;

      wad.current_level->level_list = lump;
      return;
    }
      
    wad.current_level = NULL;
  }

  if (cur_info->load_all)
    lump->flags |= LUMP_READ_ME;
  else
    lump->flags |= LUMP_COPY_ME;

  lump->next = NULL;
  lump->prev = wad.dir_tail;

  if (wad.dir_tail)
    wad.dir_tail->next = lump;
  else
    wad.dir_head = lump;

  wad.dir_tail = lump;
}

void ReadDirectory(void)
{
  int i;
  int total_entries = wad.num_entries;
  lump_t *prev_list;

  fseek(in_file, wad.dir_start, SEEK_SET);

  for (i=0; i < total_entries; i++)
  {
    ReadDirEntry();
  }

  DetermineLevelNames();

  prev_list = wad.dir_head;
  wad.dir_head = wad.dir_tail = NULL;
  
  while (prev_list)
  {
    lump_t *cur = prev_list;
    prev_list = cur->next;

    ProcessDirEntry(cur);
  }
}

void ReadLumpData(lump_t *lump)
{

  if (lump->length == 0)
    return;

  lump->data = UtilCalloc(lump->length);

  fseek(in_file, lump->start, SEEK_SET);

  fread(lump->data, lump->length, 1, in_file);

  lump->flags &= ~LUMP_READ_ME;
}

int ReadAllLumps(void)
{
  lump_t *cur, *L;
  int count = 0;

  for (cur=wad.dir_head; cur; cur=cur->next)
  {
    count++;

    if (cur->flags & LUMP_READ_ME)
      ReadLumpData(cur);

    if (cur->flags & LUMP_IS_LEVEL)
    {
      for (L=cur->level_list; L; L=L->next)
      {
        count++;

        if (L->flags & LUMP_READ_ME)
          ReadLumpData(L);
      }
    }
  }

  return count;
}

int CountLumpTypes(int flag_mask, int flag_match)
{
  lump_t *cur, *L;
  int count = 0;

  for (cur=wad.dir_head; cur; cur=cur->next)
  {
    if ((cur->flags & flag_mask) == flag_match)
      count++;

    if (cur->flags & LUMP_IS_LEVEL)
    {
      for (L=cur->level_list; L; L=L->next)
        if ((L->flags & flag_mask) == flag_match)
          count++;
    }
  }

  return count;
}

void WriteHeader(void)
{
  raw_wad_header_t header;

  switch (wad.kind)
  {
    case IWAD:
      strncpy(header.type, "IWAD", 4);
      break;

    case PWAD:
      strncpy(header.type, "PWAD", 4);
      break;
  }

  header.num_entries = UINT32(wad.num_entries);
  header.dir_start   = UINT32(wad.dir_start);

  fwrite(&header, sizeof(header), 1, out_file);
}

void SortLumps(lump_t ** list, const char **names, int count)
{
  int i;
  lump_t *cur;

  for (i=count-1; i >= 0; i--)
  {
    for (cur=(*list); cur; cur=cur->next)
    {
      if (strcmp(cur->name, names[i]) != 0)
        continue;
      if (cur->next)
        cur->next->prev = cur->prev;
      if (cur->prev)
        cur->prev->next = cur->next;
      else
        (*list) = cur->next;
      cur->next = (*list);
      cur->prev = NULL;
      if (cur->next)
        cur->next->prev = cur;
      (*list) = cur;
      break;
    }
  }
}


void RecomputeDirectory(void)
{
  lump_t *cur, *L;

  wad.num_entries = 0;
  wad.dir_start = sizeof(raw_wad_header_t);
  for (cur=wad.dir_head; cur; cur=cur->next)
  {
    if (cur->flags & LUMP_IGNORE_ME)
      continue;
    cur->new_start = wad.dir_start;
    wad.dir_start += cur->length;
    wad.num_entries++;
    if (cur->flags & LUMP_IS_LEVEL)
    {
      SortLumps(&cur->level_list, level_lumps, NUM_LEVEL_LUMPS);

      for (L=cur->level_list; L; L=L->next)
      {
        if (L->flags & LUMP_IGNORE_ME)
          continue;

        L->new_start = wad.dir_start;

        wad.dir_start += L->length;
        wad.num_entries++;
      }
    }
  }
}

void WriteLumpData(lump_t *lump)
{
  if (lump->length == 0)
    return;

  if (lump->flags & LUMP_COPY_ME)
  {
    lump->data = UtilCalloc(lump->length);

    fseek(in_file, lump->start, SEEK_SET);

    fread(lump->data, lump->length, 1, in_file);
  }

  fwrite(lump->data, lump->length, 1, out_file);  
  UtilFree(lump->data);
  lump->data = NULL;
}

int WriteAllLumps(void)
{
  lump_t *cur, *L;
  int count = 0;

  for (cur=wad.dir_head; cur; cur=cur->next)
  {
    if (cur->flags & LUMP_IGNORE_ME)
      continue;

    WriteLumpData(cur);
    count++;

    if (cur->flags & LUMP_IS_LEVEL)
    {
      for (L=cur->level_list; L; L=L->next)
      {
        if (L->flags & LUMP_IGNORE_ME)
          continue;

        WriteLumpData(L);
        count++;
      }
    }
  }
  fflush(out_file);
  return count;
}

void WriteDirEntry(lump_t *lump)
{
  raw_wad_entry_t entry;
  strncpy(entry.name, lump->name, 8);
  entry.start = UINT32(lump->new_start);
  entry.length = UINT32(lump->length);
  fwrite(&entry, sizeof(entry), 1, out_file);
}

int WriteDirectory(void)
{
  lump_t *cur, *L;
  int count = 0;

  for (cur=wad.dir_head; cur; cur=cur->next)
  {
    if (cur->flags & LUMP_IGNORE_ME)
      continue;
    WriteDirEntry(cur);
    count++;
    if (cur->flags & LUMP_IS_LEVEL)
    {
      for (L=cur->level_list; L; L=L->next)
      {
        if (cur->flags & LUMP_IGNORE_ME)
          continue;
        WriteDirEntry(L);
        count++;
      }
    }
  }
  fflush(out_file);
  return count;
}

int CheckExtension(const char *filename, const char *ext)
{
  int A = strlen(filename) - 1;
  int B = strlen(ext) - 1;
  for (; B >= 0; B--, A--)
  {
    if (A < 0)
      return FALSE;
    if (toupper(filename[A]) != toupper(ext[B]))
      return FALSE;
  }
  return (A >= 1) && (filename[A] == '.');
}

char *ReplaceExtension(const char *filename, const char *ext)
{
  char *dot_pos;
  char buffer[512];

  strcpy(buffer, filename);
  
  dot_pos = strrchr(buffer, '.');

  if (dot_pos)
    dot_pos[1] = 0;
  else
    strcat(buffer, ".");
  
  strcat(buffer, ext);

  return UtilStrDup(buffer);
}

lump_t *CreateLevelLump(const char *name)
{
  lump_t *cur;

  for (cur=wad.current_level->level_list; cur; cur=cur->next)
  {
    if (strcmp(name, cur->name) == 0)
      break;
  }
  
  if (cur)
  {
    if (cur->data)
      UtilFree(cur->data);
    
    cur->data = NULL;
    cur->length = 0;
    cur->space  = 0;

    return cur;
  }

  cur = NewLump(UtilStrDup(name));
  cur->next = wad.current_level->level_list;
  cur->prev = NULL;

  if (cur->next)
    cur->next->prev = cur;

  wad.current_level->level_list = cur;

  return cur;
}

void AppendLevelLump(lump_t *lump, void *data, int length)
{
  if (length == 0)
    return;
  if (lump->length == 0)
  {
    lump->space = MAX(length, APPEND_BLKSIZE);
    lump->data = UtilCalloc(lump->space);
  }
  else if (lump->space < length)
  {
    lump->space = MAX(length, APPEND_BLKSIZE);
    lump->data = UtilRealloc(lump->data, lump->length + lump->space);
  }
  memcpy(((char *)lump->data) + lump->length, data, length);
  lump->length += length;
  lump->space  -= length;
}

int CountLevels(void)
{
  lump_t *cur;
  int result = 0;
  for (cur=wad.dir_head; cur; cur=cur->next)
  {
    if (cur->flags & LUMP_IS_LEVEL)
      result++;
  }
  return result;
}

int FindNextLevel(void)
{
  lump_t *cur;
  
  if (wad.current_level)
    cur = wad.current_level->next;
  else
    cur = wad.dir_head;

  while (cur && ! (cur->flags & LUMP_IS_LEVEL))
    cur=cur->next;

  wad.current_level = cur;

  return (cur != NULL);
}

const char *GetLevelName(void){ return wad.current_level->name; }

lump_t *FindLevelLump(const char *name)
{
  lump_t *cur = wad.current_level->level_list;

  while (cur && (strcmp(cur->name, name) != 0))
    cur=cur->next;

  return cur;
}

int CheckLevelLumpZero(lump_t *lump)
{
  int i;
  
  if (lump->length == 0)
    return TRUE;

  for (i=0; i < lump->length; i++)
  {
    if (((uint8_g *)lump->data)[i])
      return FALSE;
  }

  return TRUE;
}

glbsp_ret_e ReadWadFile(const char *filename)
{
  char strbuf[1024];

  in_file = fopen(filename, "rb");

  if (! in_file)
  {
    sprintf(strbuf, "Cannot open WAD file %s : %s", filename, 
        strerror(errno));
    GlbspFree(cur_comms->message);
    cur_comms->message = GlbspStrDup(strbuf);
    return GLBSP_E_ReadError;
  }
  if (! ReadHeader(filename))
  {
    fclose(in_file);
    return GLBSP_E_ReadError;
  }
  PrintMsg("Opened %cWAD file : %s\n", (wad.kind == IWAD) ? 'I' : 'P', 
      filename); 
  PrintMsg("Reading %d dir entries at 0x%X\n", wad.num_entries, 
      wad.dir_start);
  ReadDirectory();
  sprintf(strbuf, "Reading: %s", filename);
  ReadAllLumps();  
  wad.current_level = NULL;
  return GLBSP_E_OK;
}

glbsp_ret_e WriteWadFile(const char *filename)
{
  char strbuf[1024];

  PrintMsg("\nSaving WAD as %s\n", filename);

  RecomputeDirectory();
  out_file = fopen(filename, "wb");

  if (! out_file)
  {
    sprintf(strbuf, "Cannot open output WAD file: %s", strerror(errno));
    GlbspFree(cur_comms->message);
    cur_comms->message = GlbspStrDup(strbuf);
    return GLBSP_E_WriteError;
  }

  WriteHeader();
  
  sprintf(strbuf, "Writing: %s", filename);
  WriteAllLumps();
  WriteDirectory();
  return GLBSP_E_OK;
}

void CloseWads(void)
{
  int i;
  if (in_file)
  {
    fclose(in_file);
    in_file = NULL;
  }
  
  if (out_file)
  {
    fclose(out_file);
    out_file = NULL;
  }
  
  while (wad.dir_head)
  {
    lump_t *head = wad.dir_head;
    wad.dir_head = head->next;

    FreeLump(head);
  }

  if (wad.level_names)
  {
    for (i=0; i < wad.num_level_names; i++)
      UtilFree((char *) wad.level_names[i]);

    UtilFree(wad.level_names);
    wad.level_names = NULL;
  }
}
