/*
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation. NO WARRANTY.
*/

#include "doomstat.h"
#include "w_wad.h"
#include "r_main.h"

typedef struct
{
  short originx;
  short originy;
  short patch;
  short stepdir;
  short colormap;
} mappatch_t;

typedef struct
{
  char       name[8];
  boolean    masked;
  short      width;
  short      height;
  char       pad[4];
  short      patchcount;
  mappatch_t patches[1];
} maptexture_t;

typedef struct
{
  int originx, originy;
  int patch;
} texpatch_t;

typedef struct
{
  char  name[8];
  int   next, index;
  short width, height;
  short patchcount;
  texpatch_t patches[1];
} texture_t;

int firstcolormaplump, lastcolormaplump;

int       firstflat, lastflat, numflats;
int       firstspritelump, lastspritelump, numspritelumps;
int       numtextures;
texture_t **textures;
int       *texturewidthmask;
fixed_t   *textureheight;
int       *texturecompositesize;
short     **texturecolumnlump;
unsigned  **texturecolumnofs;
byte      **texturecomposite;
int       *flattranslation;
int       *texturetranslation;

fixed_t   *spritewidth, *spriteoffset, *spritetopoffset;

static void R_DrawColumnInCache(const column_t *patch, byte *cache,
				int originy, int cacheheight, byte *marks)
{
  while (patch->topdelta != 0xff)
    {
      int count = patch->length;
      int position = originy + patch->topdelta;

      if (position < 0)
        {
          count += position;
          position = 0;
        }

      if (position + count > cacheheight)
        count = cacheheight - position;

      if (count > 0)
        {
          memcpy (cache + position, (byte *)patch + 3, count);
          memset (marks + position, 0xff, count);
        }

      patch = (column_t *)((byte *) patch + patch->length + 4);
    }
}

static void R_GenerateComposite(int texnum)
{
  byte *block = Z_Malloc(texturecompositesize[texnum], PU_STATIC,
                         (void **) &texturecomposite[texnum]);
  texture_t *texture = textures[texnum];
  texpatch_t *patch = texture->patches;
  short *collump = texturecolumnlump[texnum];
  unsigned *colofs = texturecolumnofs[texnum];
  int i = texture->patchcount;
  byte *marks = calloc(texture->width, texture->height), *source;

  for (; --i >=0; patch++)
    {
      patch_t *realpatch = W_CacheLumpNum(patch->patch, PU_CACHE);
      int x, x1 = patch->originx, x2 = x1 + SHORT(realpatch->width);
      const int *cofs = realpatch->columnofs - x1;

      if (x1 < 0)
        x1 = 0;
      if (x2 > texture->width)
        x2 = texture->width;
      for (x = x1; x < x2 ; x++)
        if (collump[x] == -1)
          R_DrawColumnInCache((column_t*)((byte*) realpatch + LONG(cofs[x])),
                              block + colofs[x], patch->originy,
			      texture->height, marks + x*texture->height);
    }

  source = malloc(texture->height);
  for (i=0; i < texture->width; i++)
    if (collump[i] == -1)
      {
        column_t *col = (column_t *)(block + colofs[i] - 3);
        const byte *mark = marks + i * texture->height;
        int j = 0;
        memcpy(source, (byte *) col + 3, texture->height);

        for (;;)
          {
	    unsigned len;

            while (j < texture->height && !mark[j])
              j++;

            if (j >= texture->height)
              {
                col->topdelta = -1;
                break;
              }

            col->topdelta = j;

	    for (len = 0; j < texture->height && mark[j]; j++)
              len++;

	    col->length = len;
            memcpy((byte *) col + 3, source + col->topdelta, len);
            col = (column_t *)((byte *) col + len + 4);
          }
      }
  free(source);
  free(marks);
  Z_ChangeTag(block, PU_CACHE);
}

static void R_GenerateLookup(int texnum, int *const errors)
{
  const texture_t *texture = textures[texnum];

  short *collump = texturecolumnlump[texnum];
  unsigned *colofs = texturecolumnofs[texnum];

  struct {
    unsigned patches, posts;
  } *count = calloc(sizeof *count, texture->width);

  const texpatch_t *patch = texture->patches;
  int i = texture->patchcount;

  while (--i >= 0)
    {
      int pat = patch->patch;
      const patch_t *realpatch = W_CacheLumpNum(pat, PU_CACHE);
      int x, x1 = patch++->originx, x2 = x1 + SHORT(realpatch->width);
      const int *cofs = realpatch->columnofs - x1;
      
      if (x2 > texture->width)
	x2 = texture->width;
      if (x1 < 0)
	x1 = 0;
      for (x = x1 ; x<x2 ; x++)
	{
	  count[x].patches++;
	  collump[x] = pat;
	  colofs[x] = LONG(cofs[x])+3;
	}
    }

  if (texture->patchcount > 1 && texture->height < 256)
    {
      unsigned limit = texture->height*3+3;
      int badcol = devparm;

      for (i = texture->patchcount, patch = texture->patches; --i >= 0;)
	{
	  int pat = patch->patch;
	  const patch_t *realpatch = W_CacheLumpNum(pat, PU_CACHE);
	  int x, x1 = patch++->originx, x2 = x1 + SHORT(realpatch->width);
	  const int *cofs = realpatch->columnofs - x1;
	  
	  if (x2 > texture->width)
	    x2 = texture->width;
	  if (x1 < 0)
	    x1 = 0;

	  for (x = x1 ; x<x2 ; x++)
	    if (count[x].patches > 1)
	      {
		const column_t *col =
		  (column_t*)((byte*) realpatch+LONG(cofs[x]));
		const byte *base = (const byte *) col;
		for (;col->topdelta != 0xff; count[x].posts++)
		  if ((unsigned)((byte *) col - base) <= limit)
		    col = (column_t *)((byte *) col + col->length + 4);
		  else
		    {
		      if (badcol)
			{
			  badcol = 0;
			  printf("\nWarning: Texture %8.8s "
				 "(height %d) has bad column(s)"
				 " starting at x = %d.",
				 texture->name, texture->height, x);
			}
		      break;
		    }
	      }
	}
    }

  texturecomposite[texnum] = 0;

  {
    int x = texture->width;
    int height = texture->height;
    int csize = 0, err = 0;

    while (--x >= 0)
      {
	if (!count[x].patches)
	  if (devparm)
	    {
	      printf("\nR_GenerateLookup:"
		     " Column %d is without a patch in texture %.8s",
		     x, texture->name);
	      ++*errors;
	    }
	  else
	    err = 1;

        if (count[x].patches > 1)
          {
            collump[x] = -1;
            colofs[x] = csize + 3;
            csize += 4*count[x].posts+5;
          }
        csize += height;
      }

    texturecompositesize[texnum] = csize;
    
    if (err)
      {
	printf("\nR_GenerateLookup: Column without a patch in texture %.8s",
	       texture->name);
	++*errors;
      }
  }
  free(count);
}

byte *R_GetColumn(int tex, int col)
{
  int lump = texturecolumnlump[tex][col &= texturewidthmask[tex]];
  int ofs  = texturecolumnofs[tex][col];

  if (lump > 0)
    return (byte *) W_CacheLumpNum(lump, PU_CACHE) + ofs;

  if (!texturecomposite[tex])
    R_GenerateComposite(tex);

  return texturecomposite[tex] + ofs;
}

void R_InitTextures (void)
{
  maptexture_t *mtexture;
  texture_t    *texture;
  mappatch_t   *mpatch;
  texpatch_t   *patch;
  int  i, j;
  int  *maptex;
  int  *maptex1, *maptex2;
  char name[9];
  char *names;
  char *name_p;
  int  *patchlookup;
  int  totalwidth;
  int  nummappatches;
  int  offset;
  int  maxoff, maxoff2;
  int  numtextures1, numtextures2;
  int  *directory;
  int  errors = 0;

  name[8] = 0;
  names = W_CacheLumpName("PNAMES", PU_STATIC);
  nummappatches = LONG(*((int *)names));
  name_p = names+4;
  patchlookup = malloc(nummappatches*sizeof(*patchlookup));

  for (i=0 ; i<nummappatches ; i++)
    {
      strncpy (name,name_p+i*8, 8);
      patchlookup[i] = W_CheckNumForName(name);
      if (patchlookup[i] == -1)
        {
          patchlookup[i] = (W_CheckNumForName)(name, ns_sprites);

          if (patchlookup[i] == -1 && devparm)
            printf("\nWarning: patch %.8s, index %d does not exist",name,i);
        }
    }
  Z_Free(names);

  maptex = maptex1 = W_CacheLumpName("TEXTURE1", PU_STATIC);
  numtextures1 = LONG(*maptex);
  maxoff = W_LumpLength(W_GetNumForName("TEXTURE1"));
  directory = maptex+1;

  if (W_CheckNumForName("TEXTURE2") != -1)
    {
      maptex2 = W_CacheLumpName("TEXTURE2", PU_STATIC);
      numtextures2 = LONG(*maptex2);
      maxoff2 = W_LumpLength(W_GetNumForName("TEXTURE2"));
    }
  else
    {
      maptex2 = NULL;
      numtextures2 = 0;
      maxoff2 = 0;
    }
  numtextures = numtextures1 + numtextures2;

  textures = Z_Malloc(numtextures*sizeof*textures, PU_STATIC, 0);
  texturecolumnlump =
    Z_Malloc(numtextures*sizeof*texturecolumnlump, PU_STATIC, 0);
  texturecolumnofs =
    Z_Malloc(numtextures*sizeof*texturecolumnofs, PU_STATIC, 0);
  texturecomposite =
    Z_Malloc(numtextures*sizeof*texturecomposite, PU_STATIC, 0);
  texturecompositesize =
    Z_Malloc(numtextures*sizeof*texturecompositesize, PU_STATIC, 0);
  texturewidthmask =
    Z_Malloc(numtextures*sizeof*texturewidthmask, PU_STATIC, 0);
  textureheight = Z_Malloc(numtextures*sizeof*textureheight, PU_STATIC, 0);

  totalwidth = 0;

  for (i=0 ; i<numtextures ; i++, directory++)
    {
      if (i == numtextures1)
        {
          maptex = maptex2;
          maxoff = maxoff2;
          directory = maptex+1;
        }

      offset = LONG(*directory);

      if (offset > maxoff)
        I_Error("R_InitTextures: mauvais texture directore");

      mtexture = (maptexture_t *) ( (byte *)maptex + offset);

      texture = textures[i] =
        Z_Malloc(sizeof(texture_t) +
                 sizeof(texpatch_t)*(SHORT(mtexture->patchcount)-1),
                 PU_STATIC, 0);

      texture->width = SHORT(mtexture->width);
      texture->height = SHORT(mtexture->height);
      texture->patchcount = SHORT(mtexture->patchcount);

      memcpy(texture->name, mtexture->name, sizeof(texture->name));
      mpatch = mtexture->patches;
      patch = texture->patches;

      for (j=0 ; j<texture->patchcount ; j++, mpatch++, patch++)
        {
          patch->originx = SHORT(mpatch->originx);
          patch->originy = SHORT(mpatch->originy);
          patch->patch = patchlookup[SHORT(mpatch->patch)];
          if (patch->patch == -1)
            {
              printf("\nR_InitTextures: Missing patch %d in texture %.8s",
                     SHORT(mpatch->patch), texture->name);
              ++errors;
            }
        }
      texturecolumnlump[i] =
        Z_Malloc(texture->width*sizeof**texturecolumnlump, PU_STATIC,0);
      texturecolumnofs[i] =
        Z_Malloc(texture->width*sizeof**texturecolumnofs, PU_STATIC,0);

      for (j=1; j*2 <= texture->width; j<<=1)
        ;
      texturewidthmask[i] = j-1;
      textureheight[i] = texture->height<<FRACBITS;

      totalwidth += texture->width;
    }
 
  free(patchlookup);

  Z_Free(maptex1);
  if (maptex2)
    Z_Free(maptex2);

  if (errors)
    I_Error("\n\n%d erreurs.", errors);

  for (i=0 ; i<numtextures ; i++)
    R_GenerateLookup(i, &errors);

  if (errors)
    I_Error("\n\n%d erreurs.", errors);

  texturetranslation =
    Z_Malloc((numtextures+1)*sizeof*texturetranslation, PU_STATIC, 0);

  for (i=0 ; i<numtextures ; i++)
    texturetranslation[i] = i;

  for (i = 0; i<numtextures; i++)
    textures[i]->index = -1;
  while (--i >= 0)
    {
      int j = W_LumpNameHash(textures[i]->name) % (unsigned) numtextures;
      textures[i]->next = textures[j]->index;
      textures[j]->index = i;
    }
}

void R_InitFlats(void)
{
  int i;

  firstflat = W_GetNumForName("F_START") + 1;
  lastflat  = W_GetNumForName("F_END") - 1;
  numflats  = lastflat - firstflat + 1;

  flattranslation =
    Z_Malloc((numflats+1)*sizeof(*flattranslation), PU_STATIC, 0);

  for (i=0 ; i<numflats ; i++)
    flattranslation[i] = i;
}

void R_InitSpriteLumps(void)
{
  int i;
  patch_t *patch;

  firstspritelump = W_GetNumForName("S_START") + 1;
  lastspritelump = W_GetNumForName("S_END") - 1;
  numspritelumps = lastspritelump - firstspritelump + 1;

  spritewidth = Z_Malloc(numspritelumps*sizeof*spritewidth, PU_STATIC, 0);
  spriteoffset = Z_Malloc(numspritelumps*sizeof*spriteoffset, PU_STATIC, 0);
  spritetopoffset =
    Z_Malloc(numspritelumps*sizeof*spritetopoffset, PU_STATIC, 0);

  for (i=0 ; i< numspritelumps ; i++)
    {
      patch = W_CacheLumpNum(firstspritelump+i, PU_CACHE);
      spritewidth[i] = SHORT(patch->width)<<FRACBITS;
      spriteoffset[i] = SHORT(patch->leftoffset)<<FRACBITS;
      spritetopoffset[i] = SHORT(patch->topoffset)<<FRACBITS;
    }
}

void R_InitColormaps(void)
{
  int i;
  firstcolormaplump = W_GetNumForName("C_START");
  lastcolormaplump  = W_GetNumForName("C_END");
  numcolormaps = lastcolormaplump - firstcolormaplump;
  colormaps = Z_Malloc(sizeof(*colormaps) * numcolormaps, PU_STATIC, 0);

  colormaps[0] = W_CacheLumpNum(W_GetNumForName("COLORMAP"), PU_STATIC);

  for (i=1; i<numcolormaps; i++)
    colormaps[i] = W_CacheLumpNum(i+firstcolormaplump, PU_STATIC);
}

int R_ColormapNumForName(const char *name)
{
  register int i = 0;
  if (strncasecmp(name,"COLORMAP",8))
    if ((i = (W_CheckNumForName)(name, ns_colormaps)) != -1)
      i -= firstcolormaplump;
  return i;
}

int tran_filter_pct = 100;

#define TSC 12

void R_InitTranMap(int progress)
{
  int lump = W_CheckNumForName("TRANMAP");

  if (lump != -1)
    main_tranmap = W_CacheLumpNum(lump, PU_STATIC);
  else
    {
      unsigned char *playpal = W_CacheLumpName("PLAYPAL", PU_STATIC);
      char fname[PATH_MAX+1], *D_DoomExeDir(void);
      struct {
        unsigned char pct;
        unsigned char playpal[256];
      } cache;
      FILE *cachefp = fopen(strcat(strcpy(fname, D_DoomExeDir()),
                                   "/tranmap.dat"),"r+b");
      main_tranmap = Z_Malloc(256*256, PU_STATIC, 0);

      if (!cachefp ? cachefp = fopen(fname,"wb") , 1 :
          fread(&cache, 1, sizeof cache, cachefp) != sizeof cache ||
          cache.pct != tran_filter_pct ||
          memcmp(cache.playpal, playpal, sizeof cache.playpal) ||
          fread(main_tranmap, 256, 256, cachefp) != 256 )
        {
          long pal[3][256], tot[256], pal_w1[3][256];
          long w1 = ((unsigned long) tran_filter_pct<<TSC)/100;
          long w2 = (1l<<TSC)-w1;
          {
            register int i = 255;
            register const unsigned char *p = playpal+255*3;
            do
              {
                register long t,d;
                pal_w1[0][i] = (pal[0][i] = t = p[0]) * w1;
                d = t*t;
                pal_w1[1][i] = (pal[1][i] = t = p[1]) * w1;
                d += t*t;
                pal_w1[2][i] = (pal[2][i] = t = p[2]) * w1;
                d += t*t;
                p -= 3;
                tot[i] = d << (TSC-1);
              }
            while (--i>=0);
          }
          {
            int i,j;
            byte *tp = main_tranmap;
            for (i=0;i<256;i++)
              {
                long r1 = pal[0][i] * w2;
                long g1 = pal[1][i] * w2;
                long b1 = pal[2][i] * w2;

                for (j=0;j<256;j++,tp++)
                  {
                    register int color = 255;
                    register long err;
                    long r = pal_w1[0][j] + r1;
                    long g = pal_w1[1][j] + g1;
                    long b = pal_w1[2][j] + b1;
                    long best = LONG_MAX;
                    do
                      if ((err = tot[color] - pal[0][color]*r
                          - pal[1][color]*g - pal[2][color]*b) < best)
                        best = err, *tp = color;
                    while (--color >= 0);
                  }
              }
          }
          if (cachefp)
            {
              cache.pct = tran_filter_pct;
              memcpy(cache.playpal, playpal, 256);
              fseek(cachefp, 0, SEEK_SET);
              fwrite(&cache, 1, sizeof cache, cachefp);
              fwrite(main_tranmap, 256, 256, cachefp);
            }
        }

      if (cachefp) fclose(cachefp);
      Z_ChangeTag(playpal, PU_CACHE);
    }
}

void R_InitData(void)
{
  R_InitTextures();
  R_InitFlats();
  R_InitSpriteLumps();
  if (tran_filter_pct<100)
    R_InitTranMap(1);
  R_InitColormaps();
}

int R_FlatNumForName(const char *name)
{
  int i = (W_CheckNumForName)(name, ns_flats);
  if (i == -1)
    I_Error("R_FlatNumForName: %.8s ne finde", name);
  return i - firstflat;
}

int R_CheckTextureNumForName(const char *name)
{
  int i = 0;
  if (*name != '-')
    {
      i = textures[W_LumpNameHash(name) % (unsigned) numtextures]->index;
      while (i >= 0 && strncasecmp(textures[i]->name,name,8))
        i = textures[i]->next;
    }
  return i;
}

int R_TextureNumForName(const char *name)
{
  int i = R_CheckTextureNumForName(name);
  if (i == -1)
    I_Error("R_TextureNumForName: %.8s ne finde", name);
  return i;
}