//
// Copyright(C) 1999,2007 Simon Howard
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307, USA.
//

/***************************** FraggleScript ******************************/
                // Copyright(C) 1999 Simon Howard 'Fraggle' //
//
// Operators
//
// Handler code for all the operators. The 'other half'
// of the parsing.
//

/* includes ************************/

#include <stdlib.h>
#include <string.h>

#include "parse.h"
#include "variable.h"
#include "main.h"

#define evaluate_leftnright(a, b, c) {\
        left = evaluate_expression((a), (b)-1); \
        right = evaluate_expression((b)+1, (c)); }\

svalue_t OPequals(int, int, int);           // =
svalue_t OPor(int, int, int);               // |
svalue_t OPand(int, int, int);              // &
svalue_t OPcmp(int, int, int);              // ==
svalue_t OPnotcmp(int, int, int);           // !=
svalue_t OPplus(int, int, int);             // +
svalue_t OPminus(int, int, int);            // -
svalue_t OPmultiply(int, int, int);         // *
svalue_t OPdivide(int, int, int);           // /
svalue_t OPremainder(int, int, int);        // %
svalue_t OPnot(int, int, int);              // !
svalue_t OPlessthan(int, int, int);         // <
svalue_t OPgreaterthan(int, int, int);      // >
svalue_t OPincrement(int, int, int);        // ++
svalue_t OPdecrement(int, int, int);        //--

operator_t operators[]=
{
        {"=",   OPequals,               backward},
        {"||",  OPor,                   forward},
        {"&&",  OPand,                  forward},
        {"|",   OPor,                   forward},
        {"&",   OPand,                  forward},
        {"==",  OPcmp,                  forward},
        {"!=",  OPnotcmp,               forward},
        {"<",   OPlessthan,             forward},
        {">",   OPgreaterthan,          forward},
        {"+",   OPplus,                 forward},
        {"-",   OPminus,                forward},
        {"*",   OPmultiply,             forward},
        {"/",   OPdivide,               forward},
        {"%",   OPremainder,            forward},
        {"!",   OPnot,                  forward},
        {"++",  OPincrement,            forward},
        {"--",  OPdecrement,            forward},
};

int num_operators = sizeof(operators) / sizeof(operator_t);

/***************** logic *********************/
                    // = operator
svalue_t OPequals(int start, int n, int stop)
{
        svariable_t *var;
        svalue_t evaluated;

        var = find_variable(tokens[start]);
                    
        if(var)
        {
                evaluated = evaluate_expression(n+1, stop);
                setvariablevalue(var, evaluated);
        }
        else
        {
                script_error("unknown variable '%s'", tokens[start]);                
                return nullvar;
        }

        return evaluated;
}


svalue_t OPor(int start, int n, int stop)
{
        svalue_t returnvar;
        int exprtrue = false;
        svalue_t eval;

                // if first is true, do not evaluate the second

        eval = evaluate_expression(start, n-1);

        if(intvalue(eval))
                exprtrue = true;
        else
        {
                eval = evaluate_expression(n+1, stop);
                exprtrue = !!intvalue(eval);
        }

        returnvar.type = vt_int;
        returnvar.value.i = exprtrue;
        return returnvar;

        return returnvar;
}


svalue_t OPand(int start, int n, int stop)
{
        svalue_t returnvar;
        int exprtrue = true;
        svalue_t eval;

                // if first is false, do not eval second

        eval = evaluate_expression(start, n-1);

        if(!intvalue(eval) )
                exprtrue = false;
        else
        {
                eval = evaluate_expression(n+1, stop);
                exprtrue = !!intvalue(eval);
        }

        returnvar.type = vt_int;
        returnvar.value.i = exprtrue;

        return returnvar;
}

svalue_t OPcmp(int start, int n, int stop)
{
        svalue_t left, right, returnvar;

        evaluate_leftnright(start, n, stop);

        returnvar.type = vt_int;        // always an int returned

        if(left.type == vt_int)
        {
            if(right.type == vt_int)
            {
                returnvar.value.i = left.value.i == right.value.i;
                return returnvar;
            }
            if(right.type == vt_string)
            {
                returnvar.value.i = left.value.i == atoi(right.value.s);
                return returnvar;
            }
        }
        if(left.type == vt_string)
        {
            if(right.type == vt_int)
            {
                returnvar.value.i = right.value.i == atoi(left.value.s);
                return returnvar;
            }
            if(right.type == vt_string)
            {
                returnvar.value.i = !strcmp(left.value.s, right.value.s);
                return returnvar;
            }
        }

        return nullvar;
}

svalue_t OPnotcmp(int start, int n, int stop)
{
        svalue_t returnvar;

        returnvar = OPcmp(start, n, stop);
        returnvar.value.i = !returnvar.value.i;

        return returnvar;
}

svalue_t OPlessthan(int start, int n, int stop)
{
        svalue_t left, right, returnvar;

        evaluate_leftnright(start, n, stop);

        returnvar.type = vt_int;
        returnvar.value.i = intvalue(left) < intvalue(right);
        return returnvar;
}

svalue_t OPgreaterthan(int start, int n, int stop)
{
        svalue_t left, right, returnvar;

        evaluate_leftnright(start, n, stop);

        returnvar.type = vt_int;
        returnvar.value.i = intvalue(left) > intvalue(right);
        return returnvar;
}

svalue_t OPnot(int start, int n, int stop)
{
        svalue_t right, returnvar;

        right = evaluate_expression(n+1, stop);

        returnvar.type = vt_int;
        returnvar.value.i = !intvalue(right);
        return returnvar;
}

/************** simple math ***************/

svalue_t OPplus(int start, int n, int stop)
{
        svalue_t left, right, returnvar;

        evaluate_leftnright(start, n, stop);

        returnvar.type = vt_int;
        returnvar.value.i = intvalue(left) + intvalue(right);
        return returnvar;
}

svalue_t OPminus(int start, int n, int stop)
{
        svalue_t left, right, returnvar;

                // do they mean minus as in '-1' rather than '2-1'?
        if(start == n)
        {
                        // kinda hack, hehe
                left.value.i = 0; left.type = vt_int;
                right = evaluate_expression(n+1, stop);
        }
        else
                evaluate_leftnright(start, n, stop);

        returnvar.type = vt_int;
        returnvar.value.i = intvalue(left) - intvalue(right);
        return returnvar;
}

svalue_t OPmultiply(int start, int n, int stop)
{
        svalue_t left, right, returnvar;

        evaluate_leftnright(start, n, stop);

        returnvar.type = vt_int;
        returnvar.value.i = intvalue(left) * intvalue(right);
        return returnvar;
}

svalue_t OPdivide(int start, int n, int stop)
{
        svalue_t left, right, returnvar;

        evaluate_leftnright(start, n, stop);


        returnvar.type = vt_int;
        returnvar.value.i = intvalue(left) / intvalue(right);
        return returnvar;
}

svalue_t OPremainder(int start, int n, int stop)
{
        svalue_t left, right, returnvar;

        evaluate_leftnright(start, n, stop);

        returnvar.type = vt_int;
        returnvar.value.i = intvalue(left) % intvalue(right);
        return returnvar;
}

        // ++
svalue_t OPincrement(int start, int n, int stop)
{
        if(start == n)          // ++n
        {
                svalue_t value;
                svariable_t *var;

                var = find_variable(tokens[stop]);
                if(!var)
                {
                     script_error("unknown variable '%s'", tokens[stop]);
                     return nullvar;
                }
                value = getvariablevalue(var);

                value.value.i = intvalue(value) + 1;
                value.type = vt_int;
                setvariablevalue(var, value);

                return value;
        }
        else if(stop == n)     // n++
        {
                svalue_t origvalue, value;
                svariable_t *var;

                var = find_variable(tokens[start]);
                if(!var)
                {
                        script_error("unknown variable '%s'", tokens[start]);
                        return nullvar;
                }
                origvalue = getvariablevalue(var);

                value.type = vt_int;
                value.value.i = intvalue(origvalue) + 1;
                setvariablevalue(var, value);

                return origvalue;
        }

        script_error("incorrect arguments to ++ operator");
        return nullvar;
}

        // --
svalue_t OPdecrement(int start, int n, int stop)
{
        if(start == n)          // ++n
        {
                svalue_t value;
                svariable_t *var;

                var = find_variable(tokens[stop]);
                if(!var)
                {
                    script_error("unknown variable '%s'", tokens[stop]);
                    return nullvar;
                }
                value = getvariablevalue(var);

                value.value.i = intvalue(value) - 1;
                value.type = vt_int;
                setvariablevalue(var, value);

                return value;
        }
        else if(stop == n)   // n++
        {
                svalue_t origvalue, value;
                svariable_t *var;

                var = find_variable(tokens[start]);
                if(!var)
                {
                    script_error("unknown variable '%s'", tokens[start]);
                    return nullvar;
                }
                origvalue = getvariablevalue(var);

                value.type = vt_int;
                value.value.i = intvalue(origvalue) - 1;
                setvariablevalue(var, value);

                return origvalue;
        }

        script_error("incorrect arguments to ++ operator");
        return nullvar;
}
