//----------------------------------------------------------------------------
//  EDGE DJGPP CD Handling Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
// Based heavily on:
//   Brennan's CD-ROM Audio Playing Library by Brennan Underwood.
//   (Unable to give email as it now appears to be dead).
//
// -ACB- 1999/11/09 Modified extensively for use with the EDGE Engine
//

#include "..\i_defs.h"
#include "i_sysinc.h"

#pragma pack(1)

#define RESET_ERROR (_error = _error_code = 0)

#define ERROR_BIT (1 << 15)
#define BUSY_BIT (1 << 9)

typedef struct
{
  int is_audio;
  int start;
  int end;
  int len;
}
cdtrack_t;

/* I know 'typedef struct {} bleh' is a bad habit, but... */
typedef struct
{
  byte len              __attribute__((packed));
  byte unit             __attribute__((packed));
  byte command          __attribute__((packed));
  unsigned short status __attribute__((packed));
  byte reserved[8]      __attribute__((packed));
}
reqhead_t;

typedef struct
{
  reqhead_t request_header __attribute__((packed));
  byte descriptor          __attribute__((packed));
  unsigned long address    __attribute__((packed));
  unsigned short len       __attribute__((packed));
  unsigned short secnum    __attribute__((packed));
  unsigned long ptr        __attribute__((packed));
}
iocontrol_t;

typedef struct
{
  byte control  __attribute__((packed));
  byte lowest   __attribute__((packed));
  byte highest  __attribute__((packed));
  byte total[4] __attribute__((packed));
}
diskinfo_t;

typedef struct
{
  byte control      __attribute__((packed));
  byte track_number __attribute__((packed));
  byte start[4]     __attribute__((packed));
  byte info         __attribute__((packed));
}
trackinfo_t;

typedef struct
{
  reqhead_t request   __attribute__((packed));
  byte mode           __attribute__((packed));
  unsigned long start __attribute__((packed));
  unsigned long len   __attribute__((packed));
}
playreq_t;

typedef struct
{
  reqhead_t request __attribute__((packed));
}
stopreq_t;

typedef struct
{
  reqhead_t request __attribute__((packed));
}
resumereq_t;

typedef struct
{
  byte control __attribute__((packed));
  byte input0  __attribute__((packed));
  byte volume0 __attribute__((packed));
  byte input1  __attribute__((packed));
  byte volume1 __attribute__((packed));
  byte input2  __attribute__((packed));
  byte volume2 __attribute__((packed));
  byte input3  __attribute__((packed));
  byte volume3 __attribute__((packed));
}
volreq_t;

typedef struct
{
  byte control __attribute__((packed));
  byte fn      __attribute__((packed));
}
lockreq_t;

typedef struct
{
  byte control         __attribute__((packed));
  unsigned long status __attribute__((packed));
}
statusreq_t;

typedef struct
{
  byte control      __attribute__((packed));
  byte mode         __attribute__((packed));
  unsigned long loc __attribute__((packed));
}
posreq_t;

#pragma pack()

/* BCD Status Bits */
#define BCD_DOOR_OPEN		1
#define BCD_DOOR_UNLOCKED	2
#define BCD_SUPPORT_COOKED	4
#define BCD_READ_ONLY		8
#define BCD_DATA_READ_ONLY	16
#define BCD_SUPPORT_INTERLEAVE	32

static int mscdex_version;
static int first_drive;
static int num_tracks;
static int lowest_track, highest_track;
static int audio_length;
static cdtrack_t *tracks = NULL;

static int dos_mem_segment, dos_mem_selector = -1;

static int _status, _error, _error_code;
static char *_bcd_error = NULL;

// ========================== INTERNALS ===============================

//
// IOControl
//
// DOS IOCTL with command block
//
static void IOControl(iocontrol_t *ioctrl, void *command, int len)
{
  int ioctrllen;
  unsigned long command_address;
  __dpmi_regs regs;

  ioctrllen = sizeof(iocontrol_t);
  command_address = dos_mem_segment << 4;
  memset(&regs, 0, sizeof(__dpmi_regs));

  regs.x.es = (__tb >> 4) & 0xffff;
  regs.x.ax = 0x1510;
  regs.x.bx = __tb & 0xf;
  regs.x.cx = first_drive;

  ioctrl->address = dos_mem_segment << 16;
  ioctrl->len = len;

  dosmemput(ioctrl, ioctrllen, __tb);          /* put ioctl into dos area */
  dosmemput(command, len, command_address);    /* and command too */

  if (__dpmi_int(0x2f, &regs) == -1)
  {
    _bcd_error = "__dpmi_int() failed";
    return;
  }

  dosmemget(__tb, ioctrllen, ioctrl);  /* retrieve results */
  dosmemget(command_address, len, command);

  _status = ioctrl->request_header.status;

  if (_status & ERROR_BIT)
  {
    _error = true;
    _error_code = _status & 0xff;
  }
  else
  {
    _error = false;
    _error_code = 0;
  }
}

//
// IOControl2
//
// DOS IOCTL without command block
//
static void IOControl2(void *cmd, int len)
{
  __dpmi_regs regs;

  memset(&regs, 0, sizeof(__dpmi_regs));

  regs.x.es = (__tb >> 4) & 0xffff;
  regs.x.ax = 0x1510;
  regs.x.bx = __tb & 0xf;
  regs.x.cx = first_drive;

  dosmemput(cmd, len, __tb); /* put ioctl block in dos arena */

  if (__dpmi_int(0x2f, &regs) == -1)
  {
    _bcd_error = "__dpmi_int() failed";
    return;
  }

  /* I hate to have no error capability for ioctl2 but the command block
     doesn't necessarily have a status field */
  RESET_ERROR;
}

//
// Red2Hsg
//
static int Red2Hsg(char *r)
{
  return r[0] + r[1]*75 + r[2]*4500 - 150;
}

//
// GetStatusWord
//
static int GetStatusWord(void)
{
  iocontrol_t ioctrl;
  diskinfo_t disk_info;

  /* get cd info as an excuse to get a look at the status word */
  memset(&disk_info, 0, sizeof(diskinfo_t));
  memset(&ioctrl, 0, sizeof(iocontrol_t));

  ioctrl.request_header.len = 26;
  ioctrl.request_header.command = 3;
  ioctrl.len = 7;

  disk_info.control = 10;
  IOControl(&ioctrl, &disk_info, sizeof(diskinfo_t));

  return _status;
}

//
// PlayCDSection
//
// Plays a part of the CD starting at location for length frames
//
static boolean_t PlayCDSection(int location, int frames)
{
  playreq_t cmd;
  memset(&cmd, 0, sizeof(playreq_t));

  _bcd_error = NULL;

  cmd.request.len = sizeof(playreq_t);
  cmd.request.command = 132;
  cmd.start = location;
  cmd.len = frames;

  IOControl2(&cmd, sizeof(playreq_t));

  if (_error)
    return 0;

  return 1;
}

//
// LockCD
//
static boolean_t LockCD(boolean_t lockit)
{
  iocontrol_t ioctrl;
  lockreq_t req;

  _bcd_error = NULL;

  memset(&ioctrl, 0, sizeof(iocontrol_t));
  memset(&req, 0, sizeof(lockreq_t));

  ioctrl.request_header.len = sizeof(iocontrol_t);
  ioctrl.request_header.command = 12;
  ioctrl.len = sizeof(lockreq_t);

  req.control = 1;
  req.fn = lockit ? 1 : 0;

  IOControl(&ioctrl, &req, sizeof(lockreq_t));

  if (_error)
    return false;

  return true;
}

//
// GetTrackInfo
//
// Internal function to get track info
//
static void GetTrackInfo(int n, cdtrack_t *t)
{
  iocontrol_t ioctrl;
  trackinfo_t info;

  memset(&ioctrl, 0, sizeof(iocontrol_t));
  memset(&info, 0, sizeof(trackinfo_t));

  ioctrl.request_header.len = sizeof(iocontrol_t);
  ioctrl.request_header.command = 3;
  info.control = 11;
  info.track_number = n;

  IOControl(&ioctrl, &info, sizeof(trackinfo_t));

  t->start = Red2Hsg(info.start);

  if (info.info & 64)
    t->is_audio = 0;
  else
    t->is_audio = 1;
}

//
// DeviceStatus
//
static int DeviceStatus(void)
{
  iocontrol_t ioctrl;
  statusreq_t req;
  _bcd_error = NULL;

  memset(&ioctrl, 0, sizeof(iocontrol_t));
  memset(&req, 0, sizeof(statusreq_t));

  ioctrl.request_header.len = sizeof(iocontrol_t); // ok
  ioctrl.request_header.command = 3;
  ioctrl.len = sizeof(statusreq_t);

  req.control = 6;

  IOControl(&ioctrl, &req, sizeof(statusreq_t));

  return req.status;
}

//
// GetAudioInfo
//
static int GetAudioInfo(void)
{
  iocontrol_t ioctrl;
  diskinfo_t disk_info;
  int i;

  _bcd_error = NULL;

  if (tracks)
    free(tracks);

  tracks = NULL;

  memset(&disk_info, 0, sizeof(diskinfo_t));
  memset(&ioctrl, 0, sizeof(iocontrol_t));

  ioctrl.request_header.len = 26;
  ioctrl.request_header.command = 3;
  ioctrl.len = 7;

  disk_info.control = 10;

  IOControl(&ioctrl, &disk_info, sizeof(diskinfo_t));

  if (_error)
    return 0;

  lowest_track = disk_info.lowest;
  highest_track = disk_info.highest;
  num_tracks = disk_info.highest - disk_info.lowest + 1;

  tracks = calloc(highest_track + 1, sizeof(cdtrack_t));
  if (tracks == NULL)
  {
    _bcd_error = "Out of memory allocating tracks\n";
    return 0;
  }

  // get track starts
  for (i = lowest_track; i <= highest_track; i++)
    GetTrackInfo(i, tracks+i);
 
  // figure out track ends
  for (i = lowest_track; i < highest_track; i++)
    tracks[i].end = tracks[i+1].start-1;

  audio_length = Red2Hsg(disk_info.total);

  tracks[i].end = audio_length;

  for (i = lowest_track; i <= highest_track; i++)
    tracks[i].len = tracks[i].end - tracks[i].start;

  return num_tracks;
}

// ======================= END OF INTERNALS =========================

//
// BCD_Startup
//
// Initialisation Code
//
boolean_t BCD_Startup(void)
{
  __dpmi_regs regs;
  _bcd_error = NULL;

  /* disk I/O wouldn't work anyway if you set sizeof tb this low, but... */
  if (_go32_info_block.size_of_transfer_buffer < 4096)
  {
    _bcd_error = "Transfer buffer too small";
    return false;
  }

  memset(&regs, 0, sizeof (__dpmi_regs));
  regs.x.ax = 0x1500;
  regs.x.bx = 0x0;
  __dpmi_int(0x2f, &regs);

  if (regs.x.bx == 0) /* abba no longer lives */
  {     
    _bcd_error = "MSCDEX not found";
    return false;
  }

  first_drive = regs.x.cx; /* use the first drive */

  /* check for mscdex at least 2.0 */
  memset(&regs, 0, sizeof(__dpmi_regs));
  regs.x.ax = 0x150C;
  __dpmi_int(0x2f, &regs);

  if (regs.x.bx == 0)
  {
    _bcd_error = "MSCDEX version < 2.0";
    return false;
  }

  mscdex_version = regs.x.bx;

  /* allocate 256 bytes of dos memory for the command blocks */
  if ((dos_mem_segment = __dpmi_allocate_dos_memory(16, &dos_mem_selector))<0)
  {
    _bcd_error = "Could not allocate 256 bytes of DOS memory";
    return 0;
  }

  return true;
}

//
// BCD_PlayTrack
//
// Plays CD-Audio Track
//
boolean_t BCD_PlayTrack(int tracknum)
{
  _bcd_error = NULL;

  if (!GetAudioInfo())
    return false;

  if (tracknum < lowest_track || tracknum > highest_track)
  {
    _bcd_error = "Track out of range";
    return false;
  }

  if (!tracks[tracknum].is_audio)
  {
    _bcd_error = "Not an audio track";
    return false;
  }

  return PlayCDSection(tracks[tracknum].start, tracks[tracknum].len);
}

//
// BCD_ResumeTrack
//
boolean_t BCD_ResumeTrack(void)
{
  resumereq_t cmd;
  _bcd_error = NULL;

  memset(&cmd, 0, sizeof(resumereq_t));

  cmd.request.len = sizeof(resumereq_t);
  cmd.request.command = 136;

  IOControl2(&cmd, sizeof(resumereq_t));

  if (_error)
    return false;

  return true;
}

//
// BCD_StopTrack
//
// Stops current track from playing. This function is
// the same as pausing.
//
boolean_t BCD_StopTrack(void)
{
  stopreq_t cmd;
  _bcd_error = NULL;

  memset(&cmd, 0, sizeof(stopreq_t));

  cmd.request.len = sizeof(stopreq_t);
  cmd.request.command = 133;

  IOControl2(&cmd, sizeof(stopreq_t));

  if (_error)
    return false;

  return true;
}

//
// BCD_IsPlaying
//
boolean_t BCD_IsPlaying(void)
{
  _bcd_error = NULL;

  /* If the door is open, then the head is busy, and so the busy bit is
     on. It is not, however, playing audio. */
  if (DeviceStatus() & BCD_DOOR_OPEN)
    return false;

  GetStatusWord();

  return (_status & BUSY_BIT) ? true : false;
}

//
// BCD_SetVolume
//
// Sets the volume
//
boolean_t BCD_SetVolume(int volume)
{
  iocontrol_t ioctrl;
  volreq_t v;

  _bcd_error = NULL;

  if (volume > 255)
    volume = 255;
  else if (volume < 0)
    volume = 0;

  memset(&ioctrl, 0, sizeof(iocontrol_t));

  ioctrl.request_header.len = sizeof(iocontrol_t);
  ioctrl.request_header.command = 12;
  ioctrl.len = sizeof(volreq_t);

  v.control = 3;
  v.volume0 = volume;
  v.input0 = 0;
  v.volume1 = volume;
  v.input1 = 1;
  v.volume2 = volume;
  v.input2 = 2;
  v.volume3 = volume;
  v.input3 = 3;
  
  IOControl(&ioctrl, &v, sizeof(volreq_t));

  if (_error)
    return false;

  return true;
}

//
// BCD_Shutdown
//
// Shuts down CD-ROM audio interface
//
void BCD_Shutdown(void)
{
  _bcd_error = NULL;

  if (dos_mem_selector != -1)
  {
    __dpmi_free_dos_memory(dos_mem_selector);
    dos_mem_selector = -1;
  }

  if (tracks)
    free(tracks);

  tracks = NULL;

  RESET_ERROR;

  return;
}

//
// BCD_ReturnError
//
// Returns error string if something goes pair-shaped
//
char *BCD_ReturnError(void)
{
  static char retstr[132];

  static char *errorcodes[] =
  {
    "Write-protect violation",
    "Unknown unit",
    "Drive not ready",
    "Unknown command",
    "CRC error",
    "Bad drive request structure length",
    "Seek error",
    "Unknown media",
    "Sector not found",
    "Printer out of paper: world coming to an end", /* I mean really, on a CD? */
    "Write fault",
    "Read fault",
    "General failure",
    "Reserved",
    "Reserved",
    "Invalid disk change"
  };

  *retstr = 0;

  if (_error != 0)
  {
    strcat(retstr, "Device error: ");

    if (_error_code < 0 || _error_code > 0xf)
      strcat(retstr, "Invalid error");
    else
      strcat(retstr, errorcodes[_error_code]);

    strcat(retstr, "  ");
  }

  if (_bcd_error != NULL)
  {
    if (*retstr)
      strcat(retstr, ", ");

    strcat(retstr, _bcd_error);
  }

  return retstr;
}


