//----------------------------------------------------------------------------
//  EDGE DJGPP Input Systems
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------

#include "..\i_defs.h"
#include "i_sysinc.h"

#include "..\e_main.h" // This should not be here!!!
#include "..\z_zone.h" // This should not be here!!!

// -ES- 1999/0623 Boosted up keyboard queue size - 128 bytes ought to be enough for anyone :-)
#define KBDQUESIZE 128
static volatile byte keyboardque[KBDQUESIZE];
static volatile unsigned int kbdhead;
static unsigned int kbdtail;

// What we got?
static boolean_t mouseinited = false;

/****** Keyboard Functions ******/

//
// KeyboardISR
//
// Keyboard callback function. Just stores the raw scancode in a queue, it will
// be translated later.
//
static void KeyboardISR(int key)
{
  // Get the scan code
  keyboardque[kbdhead & (KBDQUESIZE - 1)] = key;
  kbdhead++;
}
static END_OF_FUNCTION(KeyboardISR)

//
// StartupKeyboard
//
static void StartupKeyboard(void)
{
  LOCK_FUNCTION(KeyboardISR);
  LOCK_VARIABLE(keyboardque);
  LOCK_VARIABLE(kbdhead);

  keyboard_lowlevel_callback = KeyboardISR;

  install_keyboard();
}

//
// ScanCode2EDGECode
//
// -MH- 1998/07/02  USABLE_EXTKEY, I_EDGECode2ScanCode, I_ScanCode2EDGECode
// -MH- 1998/08/18  Placed up here so defined _before_ use
//
// Doom seems to have some of the special keys usable as control keys and
// others not. Whether or not a special key is is determined by the two
// functions mentioned above and a LONG if statement further down in this
// module. For example, if the functions have them but they are not in the
// if-statement, the DEFAULT.CFG logic can handle them but EDGE can't; that
// is to say, you can't assign them in the CONTROLS menu.
//
// I have changed the contents of the if-statement into a macro to make it
// more manageable and I have move the functions from i_video.c to here so
// that they are all defined together. Who knows what the hell they were
// doing in i_video.c since they are nothing to do with video logic.
//
// I have cleaned up the functions and the macro and made these changes
// because the menu won't recognise some of the keys especially insert and
// delete. I guess we should really do same for function keys as well, but
// one possible bug-introducer at a time, eh :-)
//
static int ScanCode2EDGECode(int a)
{
  switch (a)
  {
    case KEY_UP:       return KEYD_UPARROW;
    case KEY_DOWN:     return KEYD_DOWNARROW;
    case KEY_LEFT:     return KEYD_LEFTARROW;
    case KEY_RIGHT:    return KEYD_RIGHTARROW;
    case KEY_BACKSPACE:return KEYD_BACKSPACE;
    case KEY_ENTER:    return KEYD_ENTER;
    case KEY_TAB:      return KEYD_TAB;
    case KEY_SPACE:    return KEYD_SPACE;
    case KEY_ESC:      return KEYD_ESCAPE;
    case KEY_TILDE:    return KEYD_TILDE;
    case KEY_EQUALS:   return KEYD_EQUALS;
    case KEY_MINUS:    return KEYD_MINUS;
    case KEY_F1:       return KEYD_F1;
    case KEY_F2:       return KEYD_F2;
    case KEY_F3:       return KEYD_F3;
    case KEY_F4:       return KEYD_F4;
    case KEY_F5:       return KEYD_F5;
    case KEY_F6:       return KEYD_F6;
    case KEY_F7:       return KEYD_F7;
    case KEY_F8:       return KEYD_F8;
    case KEY_F9:       return KEYD_F9;
    case KEY_F10:      return KEYD_F10;
    case KEY_F11:      return KEYD_F11;
    case KEY_F12:      return KEYD_F12;
    case KEY_LSHIFT:
    case KEY_RSHIFT:   return KEYD_RSHIFT;
    case KEY_LCONTROL:
    case KEY_RCONTROL: return KEYD_RCTRL;
    case KEY_ALTGR:    return KEYD_RALT;
    case KEY_ALT:      return KEYD_LALT;
    case KEY_PGUP:     return KEYD_PGUP;
    case KEY_PGDN:     return KEYD_PGDN;
    case KEY_HOME:     return KEYD_HOME;
    case KEY_END:      return KEYD_END;
    case KEY_INSERT:   return KEYD_INSERT;
    case KEY_DEL:      return KEYD_DELETE;
    case KEY_PRTSCR:   return KEYD_PRTSCR;  // -MH- 1998/08/18 Added "Print Screen"
    case KEY_PAUSE:    return KEYD_PAUSE;
    case KEY_SCRLOCK:  return KEYD_SCRLOCK;
    case KEY_NUMLOCK:  return KEYD_NUMLOCK;
    case KEY_CAPSLOCK: return KEYD_CAPSLOCK;
  }

  if ((key_shifts & KB_NUMLOCK_FLAG) && (a == KEY_5_PAD))
    return '5';

  if (a >= 0x100)
    return a;
  // with allegros newer than 3.12, use scancode_to_ascii
#if defined ALLEGRO_DATE && ALLEGRO_DATE > 19991200
  else if (scancode_to_ascii(a) > 8)
    return scancode_to_ascii(a);
#else
  else if (key_ascii_table[a] > 8)
    return key_ascii_table[a];
#endif
  else
    return a + 0x80;
}

//
// ReadKeyboard
//
// -KM- 1998/09/01 Handles keyboard
//
static void ReadKeyboard(void)
{
  event_t event;
  int k;

  while (kbdtail < kbdhead)
  {
    k = keyboardque[kbdtail & (KBDQUESIZE - 1)];
    kbdtail++;

    if (k & 0x80)
    {
      event.type = ev_keyup;
      k &= 0x7f;  // mask away top bit
    }
    else
    {
      event.type = ev_keydown;
    }

    event.value.key = ScanCode2EDGECode(k);
    E_PostEvent(&event);
  }
}

//
// ShutdownKeyboard
//
// -ACB- 1999/09/22 Kills Keyboard
//
static void ShutdownKeyboard(void)
{
  remove_keyboard();
}

/****** Mouse Functions ******/

//
// StartupMouse
//
static void StartupMouse(void)
{
  int retval;

  retval = install_mouse();

  if (retval != -1)
  {
    mouseinited = true;
    show_mouse(NULL);
  }
  else
  {
    mouseinited = false;
  }
}

//
// ReadMouse
//
// -KM- 1998/09/01 Handles Mouse
// -KM- 1998/12/21 Boosted mouse speed 10 times.
//
static void ReadMouse()
{
  event_t event;
  int xmickeys, ymickeys;
  static int lastbuttons = 0;
  int buttons = 0;

  if (!mouseinited || !usemouse)
    return;
    
  get_mouse_mickeys(&xmickeys, &ymickeys);

  event.type = ev_analogue;

  if (xmickeys != 0)
  {
    event.value.analogue.axis = mouse_xaxis;
    event.value.analogue.amount = xmickeys * (mouseSensitivity + 5);
    E_PostEvent(&event);
  }
  
  if (ymickeys != 0)
  {
    event.value.analogue.axis = mouse_yaxis;
    event.value.analogue.amount = (-ymickeys) * (mouseSensitivity + 5);
    if (invertmouse)
      event.value.analogue.amount = -event.value.analogue.amount;

    E_PostEvent(&event);
  }
  
  // now, do buttons
  buttons = mouse_b;

  if (buttons != lastbuttons)
  {
    int j = 1, i;

    for (i = 0; i < 3; i++)
    {
      if ((buttons & j) && !(lastbuttons & j))
      {
        event.type = ev_keydown;
        event.value.key = KEYD_MOUSE1 + i;
        E_PostEvent(&event);
      }
      if (!(buttons & j) && (lastbuttons & j))
      {
        event.type = ev_keyup;
        event.value.key = KEYD_MOUSE1 + i;
        E_PostEvent(&event);
      }
      j <<= 1;
    }
  }
  lastbuttons = buttons;
}

//
// ShutdownMouse
//
// -ACB- 1999/09/22 Kills mouse
//
static void ShutdownMouse(void)
{
  if (mouseinited)
  {
    remove_mouse();
    mouseinited = false;
  }
}

/****** Joystick Functions ******/

//
// Count Axis
//
static int CountAxis(int joynum)
{
  int sticks;
  int axis = 0;

  for (sticks = 0; sticks < joy[joynum].num_sticks; sticks++)
    axis += joy[joynum].stick[sticks].num_axis;

  return axis;
}

//
// I_CalibrateJoystick
//
// -KM- (-ACB- It changes from my base, must be new) Joystick Config
// -KM- 1999/01/31 Use Allegro keyboard routines as install_keyboard has
//  been called.
//
void I_CalibrateJoystick(int ch)
{
/*
  EXTERNIALISE THIS CODE!!!

  static int phase = 0;
  static char message[64];
  static const char *joy_centre = NULL;

  if (!joy_centre)
    joy_centre = DDF_LanguageLookup("JoystickCentre");

  if (graphicsmode)
  {
    // -KM- 1998/07/31 Added escape functionality.
    if (ch == KEYD_ESCAPE)
    {
      phase = 0;
      usejoystick = false;
      return;
    }

    switch (phase)
    {
      case 0:
        usejoystick = false;
        remove_joystick();
        strcpy(message, joy_centre);
        phase = 1;
        break;
      case 1:
        install_joystick(JOY_TYPE_AUTODETECT);
        if (joy[0].flags & JOYFLAG_CALIBRATE)
        {
          sprintf(message, "%s\n\npress a key", calibrate_joystick_name(0));
          phase = 2;
          break;
        }
      case -1:
        usejoystick = true;
        phase = 0;
        return;
      case 2:
        if (!calibrate_joystick(0))
        {
          if (joy[0].flags & JOYFLAG_CALIBRATE)
            sprintf(message, "%s\n\npress a key", calibrate_joystick_name(0));
          else
          {
            strcpy(message, joy_centre);
            phase = -1;
          }
        }
        else
          phase = -2;
        break;
    }

    if (phase == -2)
    {
      phase = 0;
      return;
    }

    M_StartMessage(message, I_CalibrateJoystick, false);
    return;
  }
  else
  {
    if (load_joystick_data(NULL))
    {
      clear_keybuf();
      CON_MessageLDF("JoystickCentreT");
      while (!keypressed() && !joy_b1 && !joy_b2)
        poll_joystick();
      install_joystick(JOY_TYPE_AUTODETECT);
      clear_keybuf();
      while (joy_b1 || joy_b2)
        poll_joystick();
      I_Printf("\n");

      while (joy[0].flags & JOYFLAG_CALIBRATE)
      {
        I_Printf("%s and press a key", calibrate_joystick_name(0));
        while (!keypressed() && !joy_b1 && !joy_b2)
          poll_joystick();
        clear_keybuf();
        while (joy_b1 || joy_b2)
          poll_joystick();
        I_Printf("\n");
        if (calibrate_joystick(0))
        {
          I_Printf("Error calibrating joystick.\n");
          usejoystick = false;
          return;
        }
      }
    }
  }
  */
}

//
// StartupJoystick
//
static void StartupJoystick(void)
{
  if (usejoystick)
    I_CalibrateJoystick(0);
}

//
// ReadJoystick
//
// -KM- 1998/09/01 Handles Joystick.  Hat support added
//
static void ReadJoystick()
{
  static boolean_t *oldbuttons = NULL;
  static int numButtons = -1;

  static int *axis = NULL;
  static int numAxis = -1;
  int i, a, s;
  event_t event;

  if (usejoystick)
  {
    if (numButtons != joy[0].num_buttons)
    {
      oldbuttons = Z_ReMalloc(oldbuttons, sizeof(boolean_t) * joy[0].num_buttons);
      numButtons = joy[0].num_buttons;
      memset(oldbuttons, false, sizeof(boolean_t) * numButtons);
    }

    i = CountAxis(0);
    if (numAxis != i)
    {
      char name[32];

      numAxis = i;
      axis = Z_ReMalloc(axis, sizeof(int) * numAxis);

      for (i = 0; i < numAxis - 2; i++)
      {
        sprintf(name, "MiscAxis%d", i + 1);
        axis[i] = get_config_int("Joystick", name, AXIS_DISABLE);
      }
    }

    poll_joystick();

    event.type = ev_analogue;
    event.value.analogue.axis = joy_xaxis;
    event.value.analogue.amount = abs(joy_x) <= 8 ? 0 : joy_x;

    E_PostEvent(&event);

    event.type = ev_analogue;
    event.value.analogue.axis = joy_yaxis;
    event.value.analogue.amount = abs(joy_y) <= 8 ? 0 : joy_y;

    E_PostEvent(&event);

    a = 0;
    for (s = 1; s < joy[0].num_sticks; s++)
    {
      for (i = 0; i < joy[0].stick[s].num_axis; i++)
      {
        event.type = ev_analogue;
        event.value.analogue.axis = axis[a++];
        event.value.analogue.amount =
            abs(joy[0].stick[s].axis[i].pos) <= 8 ? 0 : joy[0].stick[s].axis[i].pos;
        if (joy[0].stick[s].flags & JOYFLAG_UNSIGNED)
          event.value.analogue.amount -= 128;

        E_PostEvent(&event);
      }
    }

    //now, do buttons
    // -KM- 1998/12/16 Do all 8 buttons.
    for (i = 0; i < numButtons; i++)
    {
      if (joy[0].button[i].b && !oldbuttons[i])
      {
        event.type = ev_keydown;
        event.value.key = KEYD_JOYBASE + i;
        E_PostEvent(&event);
      }

      if (!joy[0].button[i].b && oldbuttons[i])
      {
        event.type = ev_keyup;
        event.value.key = KEYD_JOYBASE + i;
        E_PostEvent(&event);
      }

      oldbuttons[i] = joy[0].button[i].b;
    }
  }
}

//
// ShutdownJoystick
//
// -ACB- 1999/09/22 Saves Joystick Data on exit
//
static void ShutdownJoystick(void)
{
  if (usejoystick)
    save_joystick_data(NULL);
}

/****** Input Event Generation ******/

//
// I_StartupControl
//
void I_StartupControl(void)
{
  I_Printf("I_StartupControl:\n");

  StartupJoystick();
  I_Printf("  StartupJoystick\n");

  StartupMouse();
  I_Printf("  StartupMouse\n");

  StartupKeyboard();
  I_Printf("  StartupKeyboard\n");

  return;
}

//
// I_ControlGetEvents
//
void I_ControlGetEvents(void)
{
  ReadMouse();
  ReadJoystick();
  ReadKeyboard();
}

//
// I_ShutdownControl
//
// Return input devices to system control
//
void I_ShutdownControl(void)
{
  ShutdownJoystick();
  ShutdownMouse();
  ShutdownKeyboard();
}

