//----------------------------------------------------------------------------
//  EDGE DJGPP MUS Support Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
#include "..\i_defs.h"
#include "i_sysinc.h"

// Speed of the MUS player system.  Default is BPS_TO_TIMER(140)
#define TIC_SPEED BPS_TO_TIMER(35)
#define MUS_TIC_SPEED BPS_TO_TIMER(140)

// MUS Header MAGIC
const char musmagic[4] = { 0x4D, 0x55, 0x53, 0x1A };

typedef struct
{
  char ID[4]                  __attribute__((packed)); // identifier - "MUS" 0x1A
  unsigned short scorelen     __attribute__((packed)); // Score Length
  unsigned short scorestart   __attribute__((packed)); // Score Start Offset in bytes
  unsigned short channels     __attribute__((packed)); // number of primary channels
  unsigned short sec_channels __attribute__((packed)); // number of secondary channels
  unsigned short instrcnt     __attribute__((packed)); // Number of instruments
  unsigned short dummy        __attribute__((packed));
}
musheader_t;

// What we doing then?
musheader_t *song       = NULL;  // The song.
byte* playdata;                  // The current play data.
int playpos;                     // The current play position.
boolean_t midiavailable = false; // Available?
boolean_t playing       = false; // The song is playing.
boolean_t looping       = false; // The song is looping.
signed int waittics     = 0;     // Must be signed
int datalength          = 0;
byte channelvols[16];      // Last volume for each channel.

//
// Some bytes that when sent to the MIDI device stops all its sound...
// useful for when a song stops to cut off hanging notes; also resets
// controllers to defaults
//
static byte allsoundoff[6] = {0xB0, 123, 0, 0xB0, 121, 0};

// ================ INTERNALS =================

//
// GetMUSTime
//
// Reads a MIDI/MUS packed value
//
long GetMUSTime(void)
{
  long retval;
  int i;

  // reads a variable length byte:
  // maximum 4 bytes read or bit 7 not set
  i=4;
  retval =0;
  while((playdata[playpos] & 128) && i)
  {
    retval += playdata[playpos++] & 127;
    retval <<= 7;
    i--;
  }
  retval += playdata[playpos++] & 127;

  return retval;
};
static END_OF_FUNCTION(GetMUSTime);

//
// HandleEvent
//
// Where the real work gets done.  Takes a MUS encoded event, decodes it,
// reencodes it as a MIDI event, then sends it to the MIDI driver (in Allegro)
// (which decodes it...)
//
void HandleEvent(void)
{
  unsigned char channel, lastevent = 0;
  long pitchwheel;
  unsigned char midimsg[4];

  // Are we playing?
  if (!playing)
  {
    waittics = 0;  // Just Ended a song
    return;
  }

  // Sanity check...
  if (!song)
  {
    playing = false;
    waittics = 0;  // Just Ended a song
    return;
  }

  // Do all events; until last event bit set
  do
  {
    // Store the last_event bit for comparison later
    lastevent |= playdata[playpos] & 0x80;

    // Isolate the channel
    midimsg[0] = channel = playdata[playpos] & 0xF;

    // Switch channels 16 (MUS drum) & 10 (MIDI Drum)
    if (midimsg[0] == 0xF)
    {
      midimsg[0] = 0x9;
      channel = 0x9;
    }
    else if (midimsg[0] == 0x9)
    {
      midimsg[0] = 0xF;
      channel = 0xF;
    }

    // Check the event
    switch (playdata[playpos++] & 0x70)
    {

      case 0:  // Note Off
      {
        midimsg[0] |= 0x80;                // Note off
        midimsg[1] = playdata[playpos++];  // Note Number
        midimsg[2] = 0x40;                 // Velocity (64)
        midi_out(midimsg, 3);
        break;
      }

      case 0x10:  // Note On
      {
        midimsg[0] |= 0x90;  // Note On
        midimsg[1] = (playdata[playpos] & 0x7F);  // Note Number

        // Check for volume data
        if ((playdata[playpos++] & 0x80) == 0x80)
          channelvols[channel] = playdata[playpos++];

        midimsg[2] = channelvols[channel];  // Velocity

        midi_out(midimsg, 3);
        break;
      } 

      case 0x20:  // Pitch Wheel
      {
        // Scale to MIDI Pitch range
        pitchwheel = playdata[playpos++];
        pitchwheel *= 16384;
        pitchwheel /= 256;

        // Assemble to 14-bit MIDI pitch value
        midimsg[0] |= 0xE0;
        midimsg[1] = (pitchwheel & 7);
        midimsg[2] = (pitchwheel & 0x3F80) >> 7;
        midi_out(midimsg, 3);

        break;
      }

      case 0x30:  // System event
      {
        // A control change event with no extra data byte
        midimsg[0] |= 0xB0;

        switch (playdata[playpos++])
        {
          case 10:
            midimsg[1] = 0x78;
            break;
          case 11:
            midimsg[1] = 0x7b;
            break;
          case 12:
            midimsg[1] = 0x7e;
            break;
          case 13:
            midimsg[1] = 0x7f;
            break;
          case 14:
            midimsg[1] = 0x79;
            break;
        }
        midimsg[2] = 0;
        midi_out(midimsg, 3);
        break;
      }

      case 0x40:  // Controller Change
      {
        // Map MUS controllers to MIDI controllers
        channel = 3;  // 3 data bytes

        midimsg[3] = playdata[playpos++];
        midimsg[2] = playdata[playpos++];
        midimsg[1] = midimsg[2];
        midimsg[0] |= 0xb0;  // probably control change

        switch (midimsg[3])
        {
          // Program Change
          case 0:
            midimsg[0] = 0xC0 + (midimsg[0] & 0xF);
            channel = 2;
            break;

          // Bank change
          case 1:
            midimsg[1] = 0;
            break;

          case 2:
            midimsg[1] = 1;
            break;

          // Volume controller
          case 3:
            midimsg[1] = 7;
            break;

          // Pan controller
          case 4:
            midimsg[1] = 0xa;
            break;

          case 5:
            midimsg[1] = 0xb;
            break;
          case 6:
            midimsg[1] = 0x5b;
            break;
          case 7:
            midimsg[1] = 0x5d;
            break;
          case 8:
            midimsg[1] = 0x40;
            break;
          case 9:
            midimsg[1] = 0x43;
            break;

            // Uh Oh....
          default:
            channel = -1;
            break;
        }

        if (channel > 0)
          midi_out(midimsg, channel);

        break;
      }

      //
      // Running unhandled events (types 5 and 7) into an end of
      // song effectively stops the player from crashing.
      // If we run into a event type 5 of 7 we will gracefully loop to
      // start.
      //
      case 0x60:  // End of song
      default:
      {
        //
        // If the loop flag is set, change the datapointer to the start
        // of the score; else set the datapointer to the start of the 
        // score and clear the play bit
        //
        playpos = 0;
        // Unless looping, stop play.
        if (!looping)
        {
          playing = false;
          lastevent = 0x80;
        }
        break;
      }

    }  // Switch event_type

    // If the last_event bit was set we can leave
  }
  while (!(lastevent & 0x80));

  // And get the time until the next event
  if (playpos != 0)
    waittics = GetMUSTime();
  else
    waittics = 0;  // Just Ended a song

  waittics *= MUS_TIC_SPEED;
  return;
}
static END_OF_FUNCTION(HandleEvent);

//
// MUSTicker
//
void MUSTicker(void)
{
  // if we're not playing there is no need to continue
  if (!playing)
    return;

  if (waittics <= 0)
    HandleEvent();
  else
    waittics -= TIC_SPEED;

  //
  // Changing the speed of the interrupt takes load off the CPU and transfers
  // it to hardware that was designed for it. (Programmable Interval Timer)
  //
  if (waittics > 0)
  {
    install_int_ex(MUSTicker, waittics);
    waittics = 0;
  }
  else
  {
    install_int_ex(MUSTicker, TIC_SPEED);
  }

  return;
}
static END_OF_FUNCTION(MUSTicker);

//
// SongStartAddress
//
// Address of the start of the MUS track.
//
static byte *SongStartAddress(void)
{
  byte* track;

  if (!song)
    return 0;

  track = (byte*)song;
  track += song->scorestart;

  return track;
}

// ============ END OF INTERNALS ==============

//
// I_StartupMUS
//
// Returns true if no problems.
//
boolean_t I_StartupMUS(void)
{
  if (midiavailable)
    return true; // Already initialized.

  if (install_int_ex(MUSTicker, TIC_SPEED))
  {
    // I_PostMusicError()?
    return false;
  }

  // Load the MIDI Patches
  load_midi_patches();

  // Kill all sound, reset all controllers
  allsoundoff[0] = 0xb0;
  allsoundoff[3] = 0xb0;
  while(allsoundoff[0] <= 0xbf)
  {
    midi_out(allsoundoff, sizeof(byte)*6);
    allsoundoff[0]++;
    allsoundoff[3]++;
  }

  LOCK_FUNCTION(GetMUSTime);
  LOCK_FUNCTION(HandleEvent);
  LOCK_VARIABLE(song);
  LOCK_VARIABLE(playdata);
  LOCK_VARIABLE(playpos);
  LOCK_VARIABLE(midiavailable);
  LOCK_VARIABLE(playing);
  LOCK_VARIABLE(looping);
  LOCK_VARIABLE(waittics);
  LOCK_VARIABLE(datalength);
  LOCK_VARIABLE(channelvols);

  midiavailable = true;
  song          = NULL;
  playdata      = NULL;
  playpos       = 0;
  playing       = false;

  return true;
}

//
// I_MUSPlayTrack
//
int I_MUSPlayTrack(byte *data, int length, boolean_t loopy)
{
  if (!midiavailable)
    return -1;

  // Kill any previous music
  if (song)
    I_MUSStop();

  song = malloc(length*sizeof(byte));
  if (!song)
  {
    I_PostMusicError("Unable to allocate for MUS Song");
    return -1;
  }

  memcpy(song, data, length);

  if (song->ID[0] != musmagic[0] ||
      song->ID[1] != musmagic[1] ||
      song->ID[2] != musmagic[2] ||
      song->ID[3] != musmagic[3] )
  {
    free(song);
    song = NULL;
    return -1;
  }

  _go32_dpmi_lock_data(song, (unsigned long)length);

  playdata   = SongStartAddress();       // Go to the beginning of the song.
  playpos    = 0;
  playing    = true;
  looping    = loopy;
  datalength = length;

  return 1;
}

//
// I_MUSPause
//
void I_MUSPause(void)
{
  playing = false;

  // Kill all sound, reset all controllers
  allsoundoff[0] = 0xb0;
  allsoundoff[3] = 0xb0;
  while(allsoundoff[0] <= 0xbf)
  {
    midi_out(allsoundoff, sizeof(byte)*6);
    allsoundoff[0]++;
    allsoundoff[3]++;
  }
}

//
// I_MUSResume
//
void I_MUSResume(void)
{
  playing = true;
}

//
// I_MUSStop
//
void I_MUSStop(void)
{
  playpos = 0;
  playing = false;
  playdata = NULL;

  // Kill all sound, reset all controllers
  allsoundoff[0] = 0xb0;
  allsoundoff[3] = 0xb0;
  while(allsoundoff[0] <= 0xbf)
  {
    midi_out(allsoundoff, sizeof(byte)*6);
    allsoundoff[0]++;
    allsoundoff[3]++;
  }

  _unlock_dpmi_data(song, datalength);

  // Free mem
  free(song);
  song = NULL;
}

//
// I_MUSPlaying
//
boolean_t I_MUSPlaying(void)
{
  return playing;
}

//
// I_ShutdownMUS
//
void I_ShutdownMUS(void)
{
  if (playing)
    I_MUSStop();

  if (song)
    free(song);

  remove_int(MUSTicker);

  midiavailable = false;
}

//
// I_MUSSetVolume
//
// Vol is from 0 to 255.
//
void I_MUSSetVolume(int vol)
{
  vol = vol * 17;

  // (sfxvol, musvol). -1 means don't change
  set_volume(-1, vol);
}


