//----------------------------------------------------------------------------
//  EDGE DJGPP Music Subsystems
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
// -ACB- 1999/11/13 Written
// -ACB- 2001/01/14 Replaced I_Warning() with I_PostMusicError()
//

#include "..\i_defs.h"
#include "i_sysinc.h"

// #defines for handle information
#define GETLIBHANDLE(_handle) (_handle&0xFF)
#define GETLOOPBIT(_handle)   ((_handle&0x10000)>>16)
#define GETTYPE(_handle)      ((_handle&0xFF00)>>8)

#define MAKEHANDLE(_type,_loopbit,_libhandle) \
   (((_loopbit&1)<<16)+(((_type)&0xFF)<<8)+(_libhandle))

typedef enum
{
  support_CD   = 0x01,
  support_MIDI = 0x02,
  support_MP3  = 0x04,
  support_MUS  = 0x08
}
mussupport_e;

static byte capable;
static boolean_t musicpaused;

static MIDI *mididata;

#define MUSICERRLEN 256
static char errordesc[MUSICERRLEN];

// ========================= INTERNALS ==========================

//
// StartCDPlayback
//
static int StartCDPlayback(int track, boolean_t looping)
{
  if (capable & support_CD)
  {
    if (!BCD_PlayTrack(track))
    {
      I_PostMusicError(BCD_ReturnError());
      return -1;
    }

    return MAKEHANDLE(MUS_CD, looping, track);
  }

  I_PostMusicError("StartCDPlayback: CD Music format is currently unsupported.\n");
  return -1;
}

//
// StartMIDIDataPlayback
//
static int StartMIDIDataPlayback(char *data, boolean_t looping)
{
  short shortnum;
  int intnum;
  int pos;
  int num_tracks;
  int i;

  // Alloc some data using normal malloc: I won't lock data on the heap
  mididata = malloc(sizeof(MIDI));
  if (!mididata)
  {
    I_PostMusicError("StartMIDIDataPlayback: MIDI data malloc failed\n");
    return -1;
  }

  // Clear the tracks...
  for (i=0; i<MIDI_TRACKS; i++)
  {
    mididata->track[i].data = NULL;
    mididata->track[i].len = 0;
  }

  // Check header..
  if (memcmp(data, "MThd", 4))
  {
    I_PostMusicError("StartMIDIDataPlayback: Lump is not a MIDI file\n");
    free(mididata);
    return -1;
  }

  // Check type..
  if ((data[9] != 0) && (data[9] != 1))
  {
    I_PostMusicError("StartMIDIDataPlayback: MIDI Type is unsupported\n");
    free(mididata);
    return -1;
  }

  // check tracks..
  num_tracks = data[11];
  if ((num_tracks < 1) || (num_tracks > MIDI_TRACKS))
  {
    I_PostMusicError("StartMIDIDataPlayback: MIDI Type had invalid num of tracks\n");
    free(mididata);
    return -1;
  }

  // Divisions...
  shortnum = (data[12]<<8)&&0xFF00;
  shortnum += data[13];
  mididata->divisions = ABS(shortnum);

  pos=14;

  // read each track
  for (i=0; i<num_tracks; i++)
  {
    // read track header
    if (memcmp(&data[pos], "MTrk", 4))
    {
      I_PostMusicError("StartMIDIDataPlayback: MIDI Header Invalid");
      free(mididata);
      return -1;
    }
    pos+=4;

    intnum = (data[pos]<<24)&0xFF000000;
    intnum = (data[pos+1]<<16)&0xFF0000;
    intnum = (data[pos+2]<<8)&0xFF00;
    intnum = (data[pos+3])&0xFF;

    pos+=4;

    mididata->track[i].len = intnum;
    mididata->track[i].data = malloc(intnum*sizeof(byte));    // allocate memory

    if (!mididata->track[i].data)
    {
      I_PostMusicError("StartMIDIDataPlayback: MIDI Track malloc failed\n");
      free(mididata);
      return -1;
    }

    memcpy(mididata->track[i].data, &data[pos], sizeof(byte)*intnum);
    pos+=intnum;
  }

  lock_midi(mididata);

  if (play_midi(mididata, looping))
  {
    I_PostMusicError("StartMIDIFilePlayback: MIDI Error occured\n");
    return -1;
  }

  return MAKEHANDLE(MUS_MIDI, looping, 0);
}

//
// SetCDVolume
//
static void SetCDVolume(int vol)
{
  vol &= 0xF;

  if (!BCD_SetVolume(vol*17))
    I_PostMusicError(BCD_ReturnError());
}

//
// SetMIDIVolume
//
static void SetMIDIVolume(int vol)
{
  vol &= 0xF;

  // The -1 is to indicate to allegro that digital volume should be left unchanged
  set_volume(-1, vol*17);
}


// ====================== END OF INTERNALS ======================

//
// I_StartupMusic
//
boolean_t I_StartupMusic(void *sysinfo)
{
  // assume we can support the lot: then disprove
  capable = support_CD | support_MUS | support_MIDI | support_MP3;

  // CD Startup
  if (!BCD_Startup())
    capable &= ~support_CD;

  // if allegro has started the sound then we can't support MIDI or MUS either
  if (nosound || midi_driver->id == MIDI_NONE)
  {
    capable &= ~support_MIDI;
    capable &= ~support_MUS;
  }
  else
  {
    // We need to startup MUS Code
    if (!I_StartupMUS())
      capable &= ~support_MUS;
  }

  return true;
}

//
// I_MusicPlayback
//
// This attempts to play music of a given type. The strdata parameter can be
// a track number, filename or lumpname.
//
int I_MusicPlayback(i_music_info_t *musdat, int type, boolean_t looping)
{
  int handle;
  int track;

  if (!(capable & support_CD)   && type == MUS_CD)   return -1;
  if (!(capable & support_MIDI) && type == MUS_MIDI) return -1;
  if (!(capable & support_MP3)  && type == MUS_MP3)  return -1;
  if (!(capable & support_MUS)  && type == MUS_MUS)  return -1;

  switch (type)
  {
    // CD Support...
    case MUS_CD:
    {
      handle = StartCDPlayback(musdat->info.cd.track, looping);
      break;
    }

    case MUS_MIDI:
    {
      handle = StartMIDIDataPlayback(musdat->info.data.ptr, looping);
      break;
    }

    case MUS_MUS:
    {
      track = I_MUSPlayTrack((byte*)musdat->info.data.ptr, musdat->info.data.size, looping);

      if (track == -1)
        handle = -1;
      else
        handle = MAKEHANDLE(MUS_MUS, looping, track);

      break;
    }

    case MUS_MP3:
    {
      I_Warning("I_MusicPlayback: Music format '%d' is unsupported.\n",type);
      handle = -1;
      break;
    }

    case MUS_UNKNOWN:
    {
      I_Warning("I_MusicPlayback: Unknown format type given.\n");
      handle = -1;
      break;
    }

    default:
    {
      I_Warning("I_MusicPlayback: Weird Format '%d' given.\n", type);
      handle = -1;
      break;
    }
  }

  return handle;
}

//
// I_MusicPause
//
void I_MusicPause(int *handle)
{
  int type;

  type = GETTYPE(*handle);
  musicpaused = true;

  switch (type)
  {
    case MUS_CD:
    {
      if (!BCD_StopTrack())
      {
        I_PostMusicError(BCD_ReturnError());
        return;
      }
      break;
    }

    case MUS_MIDI:
    {
      midi_pause();
      break;
    }

    case MUS_MUS:
    {
      I_MUSPause();
      break;
    }

    default:
      break;
  }

  return;
}

//
// I_MusicResume
//
void I_MusicResume(int *handle)
{
  int type;

  type = GETTYPE(*handle);
  musicpaused = false;

  switch (type)
  {
    case MUS_CD:
    {
      if (!BCD_ResumeTrack())
      {
        I_PostMusicError(BCD_ReturnError());
        *handle = -1;
        return;
      }
      break;
    }

    case MUS_MIDI:
    {
      midi_resume();
      break;
    }

    case MUS_MUS:
    {
      I_MUSPause();
      break;
    }

    default:
      break;
  }

  return;
}

//
// I_MusicKill
//
// You can't stop the rock!! This does...
//
void I_MusicKill(int *handle)
{
  int type;

  type = GETTYPE(*handle);

  switch (type)
  {
    case MUS_CD:
    {
      if (!BCD_StopTrack())
        I_PostMusicError(BCD_ReturnError());

      break;
    }

    case MUS_MIDI:
    {
      play_midi(NULL, false); // Passing a NULL means stop
      destroy_midi(mididata); // kill data;
      break;
    }

    case MUS_MUS:
    {
      I_MUSStop();
      break;
    }

    default:
      break;
  }

  *handle = -1;
  return;
}

//
// I_MusicTicker
//
void I_MusicTicker(int *handle)
{
  boolean_t looping;
  int type;
  int libhandle;

  libhandle = GETLIBHANDLE(*handle);
  looping = GETLOOPBIT(*handle);
  type = GETTYPE(*handle);

  switch (type)
  {
    // CD Playback
    case MUS_CD:
    {
      if (!BCD_IsPlaying() && !musicpaused)
      {
        // not playing? but it is set to loop: start playback again.
        if (looping)
          *handle = StartCDPlayback(libhandle, looping);
        else
          *handle = -1; // clear handle as no music is now playing
      }
      break;
    }

    // MIDI Playback
    case MUS_MIDI:
    {
      if (midi_pos == -1) // if it has finished, kill.
      {
        destroy_midi(mididata);
        *handle = -1;
      }

      break;
    }

    case MUS_MUS: // handled by an interrupt ticker in I_MUS.C
    default:
      break;
  }

  return;
}

//
// I_SetMusicVolume
//
void I_SetMusicVolume(int *handle, int volume)
{
  int type;
  int handleint;

  handleint = *handle;

  type = GETTYPE(handleint);

  switch (type)
  {
    case MUS_CD:
    {
      SetCDVolume(volume);
      break;
    }

    case MUS_MIDI:
    {
      SetMIDIVolume(volume);
      break;
    }

    case MUS_MUS:
    {
      I_MUSSetVolume(volume);
      break;
    }

    default:
      break;
  }

  return;
}

//
// I_ShutdownMusic
//
void I_ShutdownMusic(void)
{
  if (capable & support_CD)
    I_ShutdownMUS();

  if (capable & support_CD)
    BCD_Shutdown();
}

//
// I_PostMusicError
//
void I_PostMusicError(char *message)
{
  memset(errordesc, 0, MUSICERRLEN*sizeof(char));

  if (strlen(message) > MUSICERRLEN)
    strncpy(errordesc, message, sizeof(char)*MUSICERRLEN);
  else
    strcpy(errordesc, message);

  return;
}

//
// I_MusicReturnError
//
char *I_MusicReturnError(void)
{
  return errordesc;
}

