//----------------------------------------------------------------------------
//  EDGE DJGPP Sound Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
#include "..\i_defs.h"
#include "i_sysinc.h"

// List of sfx for playback
static SAMPLE **storedsfx = NULL;
static unsigned int numstoredsfx = 0;

// System up and running?
static boolean_t inited = false;

// Error Description
static char errordesc[256];
static char scratcherror[256];

// Channel Control
typedef struct chancontrol_s
{
  int syshandle;
  int endtime;
  int pausetime;
  unsigned int sfxhandle;
  boolean_t looping;
  boolean_t playing;
}
chancontrol_t;

static chancontrol_t *chanctrl;
static unsigned int chancount;

//
// I_StartupSound
//
// Exactly what it says on the label
//
// -ACB- 1999/10/06 Written
//
boolean_t I_StartupSound(void *sysinfo)
{
  if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL) == -1)
  {
    I_Printf("I_StartupSound: Unable to install sound.\n");
    I_Warning("%s\n",allegro_error);
    inited = false;
  }
  else
  {
    I_Printf("I_StartupSound: Sound System Interface Started\n");
    chanctrl = NULL;
    chancount = 0;
    inited = true;
  }

  return inited;
}

//
// I_LoadSfx
//
// Loads a sound effect from data. In this DJGPP/Allegro
// version it stores sfx in a SAMPLE for playback
//
// -ACB- 1999/10/05 Written
// -ES- 2000/02/06 Added handle parameter. Return success.
//
boolean_t I_LoadSfx(const unsigned char *data, unsigned int length, unsigned int freq, unsigned int handle)
{
  unsigned int i;
  SAMPLE **oldlist;
  
  if (handle >= numstoredsfx)
  {
    i = numstoredsfx;
    numstoredsfx = handle + 1;

    if (storedsfx)
    {
      oldlist = storedsfx;
      storedsfx = (SAMPLE**)malloc(sizeof(SAMPLE *) * numstoredsfx);
      if (!storedsfx)
      {
        strcpy(errordesc, "I_LoadSFX: Alloc failed on storedsfx_t**");
        return false;
      }
      memcpy(storedsfx, oldlist, sizeof(SAMPLE*)*i);
      free(oldlist);
    }
    else
    {
      storedsfx = (SAMPLE**)malloc(sizeof(SAMPLE *) * numstoredsfx);
      if (!storedsfx)
      {
        strcpy(errordesc, "I_LoadSFX: Alloc failed on first storedsfx_t**");
        return false;
      }
    }

    // clear any new elements
    while (i < numstoredsfx)
      storedsfx[i++] = NULL;
  }

  DEV_ASSERT(storedsfx[handle] == NULL, ("I_LoadSfx: Sound already loaded"));

  storedsfx[handle] = create_sample(8, FALSE, freq, length);
  if (storedsfx[handle] == NULL)
  {
    strcpy(errordesc, "I_LoadSFX: Unable to create_sample");
    return false;
  }

  storedsfx[handle]->bits       = 8;
  storedsfx[handle]->freq       = freq;
  storedsfx[handle]->len        = length;
  storedsfx[handle]->priority   = 255;
  storedsfx[handle]->loop_start = 0;
  storedsfx[handle]->loop_end   = length;
  storedsfx[handle]->param      = -1;

  memcpy(storedsfx[handle]->data, data, length);

  return true;
}

//
// I_UnloadSfx
//
// -ACB- 2000/02/19 Check for playing sound effects and kills them
//
boolean_t I_UnloadSfx(unsigned int handle)
{
  unsigned int i;

  DEV_ASSERT(handle < numstoredsfx, ("I_UnloadSfx: %d out of range", handle));
  DEV_ASSERT(storedsfx[handle], ("I_UnloadSfx: NULL sample"));

  // Kill playing sound effects
  for (i = 0; i < chancount && chanctrl[i].syshandle != -1; i++)
  {
    if (chanctrl[i].sfxhandle == handle)
      I_SoundKill(i);
  }

  // this will also stop the sound from playing
  destroy_sample(storedsfx[handle]);
  storedsfx[handle] = NULL;

  return true;
}

//
// I_SoundPlayback
//
// Sets up sound for playback.
//
// -ACB- 1999/10/05 Written.
//
int I_SoundPlayback(unsigned int soundid, int pan, int vol, boolean_t looping)
{
  unsigned int i;
  int ctrlid;
  chancontrol_t *newctrllist;

  // -ES- 2000/03/07 These errors can only be resulted by quite fatal bugs,
  // so abort
  DEV_ASSERT(inited, ("I_SoundPlayback: Sound System Not Inited!"));
  DEV_ASSERT(soundid > 0 && soundid < numstoredsfx,
           ("I_SoundPlayback: Sound %d does not exist!", soundid));

  // Lets find us a spare channel control
  i=0;
  while (i<chancount && chanctrl[i].syshandle != -1)
    i++;

  // if i is equal to the count, we didn't find one...
  if (i==chancount)
  {
    newctrllist = malloc(sizeof(chancontrol_t) * (chancount+1));
    if (!newctrllist)
    {
      strcpy(errordesc,"I_SoundPlayback: Unable to create new control list");
      return -1;
    }

    memset(newctrllist, 0, sizeof(chancontrol_t) * (chancount+1));

    // copy across old info if there is any
    if (chanctrl)
    {
      memcpy(newctrllist, chanctrl, sizeof(chancontrol_t)*(chancount));
      free(chanctrl);
    }

    chanctrl = newctrllist;
    ctrlid = chancount;
    chancount++;
  }
  else
  {
    ctrlid = i; // 'i' must be a spare channel control
  }

  // Nice simple allegro: allocate a voice with given sample...
  chanctrl[i].syshandle = allocate_voice(storedsfx[soundid]);

  if (chanctrl[i].syshandle == -1)
  {
    strcpy(errordesc, "I_SoundPlayback: Unable to allocate voice for sample");
    return -1;
  }

  chanctrl[ctrlid].playing   = true;
  chanctrl[ctrlid].looping   = looping;
  chanctrl[ctrlid].sfxhandle = soundid;
  chanctrl[ctrlid].endtime   = I_GetTime() + (storedsfx[soundid]->len/(storedsfx[soundid]->freq/TICRATE));

  voice_set_frequency(chanctrl[ctrlid].syshandle, storedsfx[soundid]->freq);
  voice_set_pan(chanctrl[ctrlid].syshandle, pan);
  voice_set_playmode(chanctrl[ctrlid].syshandle, (looping)?PLAYMODE_LOOP:PLAYMODE_PLAY);
  voice_set_volume(chanctrl[ctrlid].syshandle, vol);
  voice_start(chanctrl[ctrlid].syshandle);

  return ctrlid;
}

//
// I_SoundKill
//
// Stops sound from playing any further.
//
// -ACB- 1999/10/05 Written
//
boolean_t I_SoundKill(unsigned int chanid)
{
  if (!inited)
  {
    strcpy(errordesc, "I_SoundKill: Sound System Not Inited!");
    return false;
  }

  deallocate_voice(chanctrl[chanid].syshandle);
  chanctrl[chanid].playing = false;
  chanctrl[chanid].looping = false;
  chanctrl[chanid].pausetime = 0;
  chanctrl[chanid].endtime = 0;
  chanctrl[chanid].syshandle = -1;
  chanctrl[chanid].sfxhandle = -1;

  return true;
}

//
// I_SoundCheck
//
// Checks to see if a channel is playing
//
// -ACB- 1999/10/06 Written.
//
boolean_t I_SoundCheck(unsigned int chanid)
{
  if (!inited)
  {
    strcpy(errordesc, "I_SoundCheck: Sound System Not Inited!");
    return false;
  }

  return chanctrl[chanid].playing;
}

//
// I_SoundPause
//
boolean_t I_SoundPause(unsigned int chanid)
{
  if (!inited)
  {
    strcpy(errordesc, "I_SoundPause: Sound System Not Inited!");
    return false;
  }

  voice_stop(chanctrl[chanid].syshandle);

  // if a pausetime exists, the sound is considered to be paused
  chanctrl[chanid].pausetime = I_GetTime();

  return true;
}

//
// I_ResumeSound
//
boolean_t I_SoundResume(unsigned int chanid)
{
  if (!inited)
  {
    strcpy(errordesc, "I_SoundResume: Sound System Not Inited!");
    return false;
  }

  voice_start(chanctrl[chanid].syshandle);
  chanctrl[chanid].endtime += (I_GetTime()-chanctrl[chanid].pausetime);
  chanctrl[chanid].pausetime = 0;

  return true;
}

//
// I_SoundTicker
//
// -ACB- 1999/10/06 Written.
//
void I_SoundTicker(void)
{
  unsigned int i;
  int currtime;

  if (!inited)
  {
    strcpy(errordesc, "I_SoundUpdate: Sound System Not Inited!");
    return;
  }

  currtime = I_GetTime();

  for (i = 0; i < chancount; i++)
  {
    if (chanctrl[i].playing && !chanctrl[i].looping)
    {
      // We make sure that resources are returned ASAP with the time control sys
      if (!chanctrl[i].pausetime && chanctrl[i].endtime <= currtime)
        I_SoundKill(i);
    }
  }
}

//
// I_SoundAlter
//
// Alters the parameters of a currently playing sound
//
// -ACB- 1999/10/06
//
boolean_t I_SoundAlter(unsigned int chanid, int pan, int vol)
{
  if (!inited)
  {
    strcpy(errordesc, "I_SoundAlter: Sound System Not Inited!");
    return false;
  }

  voice_set_volume(chanctrl[chanid].syshandle, vol);
  voice_set_pan(chanctrl[chanid].syshandle, pan);
  return true;
}

//
// I_SoundStopLooping
//
boolean_t I_SoundStopLooping(unsigned int chanid)
{
  if (!inited)
  {
    strcpy(errordesc, "I_SoundAlter: Sound System Not Inited!");
    return false;
  }

  voice_set_playmode(chanctrl[chanid].syshandle, PLAYMODE_PLAY);
  return false;
}

//
// I_ShutdownSound
//
// Brings the sound system to a halt
//
// -ACB- 1999/10/06 Written.
//
void I_ShutdownSound(void)
{
  unsigned int i;

  if (!inited)
    return;
    
  // kill all the playing channels
  for (i = 0; i<chancount; i++)
  {
    if (chanctrl[i].syshandle != -1)
      I_SoundKill(i);
  }

  // clear the channel watch array
  chanctrl = NULL;
  chancount = 0;

  // kill all the stored effects
/*
  for (i = 0; i < numstoredsfx; i++)
  {
    if (storedsfx[i])
      destroy_sample(storedsfx[i]);
  }
*/

  free(storedsfx);

  // clear the sfx array
  storedsfx = NULL;
  numstoredsfx = 0;

  remove_sound();
  nosound = true;
  return;
}

//
// I_SoundReturnError
//
const char *I_SoundReturnError(void)
{
  memcpy(scratcherror, errordesc, sizeof(char)*256);
  memset(errordesc, '\0', sizeof(char)*256);
  return scratcherror;
}
