//----------------------------------------------------------------------------
//  EDGE DJGPP Misc System Interface Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//

#include "..\i_defs.h"

#include "..\con_main.h"
#include "..\dm_defs.h"
#include "..\e_main.h"
#include "..\e_net.h"
#include "..\g_game.h"
#include "..\m_argv.h"
#include "..\m_menu.h"
#include "..\m_misc.h"
#include "..\r_draw1.h"
#include "..\r_draw2.h"
#include "..\s_sound.h"
#include "..\version.h"
#include "..\w_wad.h"
#include "..\z_zone.h"

#include "i_sysinc.h"

#define BGCOLOUR         4
#define FGCOLOUR         0xf

#ifdef DEVELOPERS
static boolean_t traceback_on_error = false;
#endif

// Mem setting
int mb_used = 10;

// msgbuf for I_Warning, I_Printf
#define MSGBUFSIZE 4096
static char msgbuf[4096];

// Standard timer stuff
volatile int mselapsed = 0;

// Micro timer stuff
unsigned long microtimer_granularity = 35;

//
// SysTimer
//
static void SysTimer(void)
{
  mselapsed++;
}
static END_OF_FUNCTION(SysTimer);

//
// I_SystemStartup
//
// -ACB- 1998/07/11 Reformatted the code.
//
boolean_t I_SystemStartup(void)
{
  allegro_init();

  //init timer
  LOCK_VARIABLE(mselapsed);
  LOCK_FUNCTION(SysTimer);

  i_love_bill = (!M_CheckParm("-ihatebill"));

  install_timer();
  if (install_int_ex(SysTimer, BPS_TO_TIMER(TICRATE)))
    I_Error("Could not install timer!");

  I_StartupControl();
  I_StartupGraphics();

  if (!nosound)
  {
    nosound = !I_StartupSound(NULL);
    I_StartupMusic(NULL);
  }

#ifdef DEVELOPERS
  if (M_CheckParm("-traceback"))
    traceback_on_error = true;

#ifdef INTOLERANT_MATH
  // -ES- 2000/03/24 Crash on various strange FP errors
  _control87(0, EM_ZERODIVIDE | EM_INVALID);
#endif /* INTOLERANT_MATH */

#endif /* DEVELOPERS */

  return true;
}

//
// I_EDGELoop
//
// DOS Version will loop forever.
//
void I_EDGELoop(void)
{
  while(1)
    E_EDGELoopRoutine();

  return;
}

//
// I_DisplayExitScreen
//
void I_DisplayExitScreen(void)
{
  const char *s;

  s = W_CacheLumpName("ENDOOM");
  // Intentional Const Override: libc function lacks 'const' keyword
  puttext(1, 1, 80, 25, (void *)s);
  W_DoneWithLump(s);
  gotoxy(1, 25);
}

//
// I_CloseProgram
//
void I_CloseProgram(int exitnum)
{
  exit(exitnum);
}

//
// I_TraceBack
//
// Aborts the program, and displays a traceback if possible.
// In DJGPP this is done by writing to an illegal address.
//
void I_TraceBack(void)
{
#ifdef DEVELOPERS
  __djgpp_traceback_exit(SIGQUIT);
#endif
  I_CloseProgram(-1);
}

//
// I_GetTime
//
// returns time in 1/TICRATE second tics
//
int I_GetTime(void)
{
  return mselapsed;
}

//
// I_WaitVBL
//
void I_WaitVBL(int count)
{
}

//
// I_ReadMicroSeconds
//
unsigned long I_ReadMicroSeconds(void)
{
  return mselapsed * 1000000 / 35;
}

//
// I_Warning
//
// -AJA- 1999/09/10: added this.
//
void I_Warning(const char *warning,...)
{
  va_list argptr;

  va_start(argptr, warning);
  vsprintf(msgbuf, warning, argptr);

  if (debugfile)
  {
    fprintf(debugfile, "WARNING: %s", msgbuf);
    fflush(debugfile);
  }

  CON_Printf("%s", msgbuf);
  va_end(argptr);
}

//
// I_Error
//
void I_Error(const char *error,...)
{
  va_list argptr;
  // -ES- 2000/01/07 Avoid infinite recursion.
  static boolean_t firsttime = true;

  if (firsttime)
    firsttime = false;
  else
    return;

  // Bring the system to a close
  I_SystemShutdown();

  // Message last, so it actually prints on the screen
  va_start(argptr, error);
  if (debugfile)
  {
    fprintf(debugfile, "\nERROR: ");
    vfprintf(debugfile, error, argptr);
    fflush(debugfile);
  }
  fprintf(stdout, "\n");
  vfprintf(stdout, error, argptr);
  fprintf(stdout, "\n");

  va_end(argptr);
  fflush(stdout);

#ifdef DEVELOPERS
//  if (traceback_on_error)
    I_TraceBack();
#endif

  I_CloseProgram(-1);
}

//
// I_Printf
//
// System specific printf() function
//
void I_Printf(const char *message,...)
{
  va_list argptr;
  char *string;
  char printbuf[MSGBUFSIZE];

  // clear the buffer
  memset(printbuf, 0, MSGBUFSIZE);

  string = printbuf;

  va_start(argptr, message);

  // Print the message into a text string
  vsprintf(printbuf, message, argptr);

  // Clean up \n\r combinations
  while (*string)
  {
    if (*string == '\n')
    {
      memmove(string + 2, string + 1, strlen(string));
      string[1] = '\r';
      string++;
    }
    string++;
  }

  // If debuging enabled, print to the debugfile
  L_WriteDebug("%s", printbuf);
  
  // Send the message to the console.
  CON_Printf(printbuf);

  // And the text screen if in text mode
  if (!graphicsmode)
    cprintf(printbuf);

  va_end(argptr);
}

//
// I_PutTitle
//
// -KM- 1998/10/29 Use all of screen, not just first 25 rows.
//
void I_PutTitle(const char *title)
{
  char string[81];
  int centre;

  memset(string, ' ', 80);
  string[80] = 0;

  // Build the title
  centre = (80 - strlen(title)) / 2;
  memcpy(&string[centre], title, strlen(title));

  // Print the title
  textattr(0x07);
  clrscr();
  textattr((BGCOLOUR << 4) + FGCOLOUR);
  cprintf("%s\n", string);
  textattr(0x07);
  window(1, 2, 80, ScreenRows());
}

//
// I_PathIsAbsolute
//
// Returns true if the path should be treated as an absolute path.
//
// -ES- 2000/01/01 Written.
//
boolean_t I_PathIsAbsolute(const char *path)
{
  if (path[0] == '\\' || path[0] == '/' || path[0] == '.' ||
      (path[0] <= 'z' && path[0] >= 'A' && path[1] == ':'))
    return true;
  else
    return false;
}

//
// I_PreparePath
//
// Prepares the end of the path name, so it will be possible to concatenate
// a DIRSEPARATOR and a file name to it.
// Allocates and returns the new string.
//
char *I_PreparePath(const char *path)
{
  int len = strlen(path);
  char *s;

  if (len == 0)
  {
    // empty string means ".\"
    return Z_StrDup(".");
  }

  if (path[0] >= 'A' && path[0] <= 'z' && path[1] == ':' && len == 2)
  {
    // special case: "c:" turns into "c:."
    s = Z_Malloc(4);
    s[0] = path[0];
    s[1] = path[1];
    s[2] = '.';
    s[3] = 0;
    return s;
  }

  if (path[len - 1] == '\\' || path[len - 1] == '/')
  {
    // cut off the last separator
    s = Z_Malloc(len);
    memcpy(s, path, len - 1);
    s[len - 1] = 0;

    return s;
  }

  return Z_StrDup(path);
}

//
// I_PureRandom
//
long I_PureRandom(void)
{
  return time(NULL);
}

//
// I_PathIsDirectory
//
boolean_t I_PathIsDirectory(const char *path)
{
  return (!access(path, D_OK)?true:false);
}

//
// I_Access
//
boolean_t I_Access(const char *filename)
{
  return (access(filename, R_OK) == 0) ? true : false;
}


//
// I_GetModifiedTime
//
// -ACB- 2001/06/14
//
boolean_t I_GetModifiedTime(const char *filename, i_time_t *t)
{
/*
  struct stat buf;
  struct tm timeinf;

  // Check the sanity of the coders...
  if (!t)
    return false;

  memset(t,0,sizeof(i_time_t));

  // Check the file is invalid
  if (stat(filename, &buf))			
    return false;

  // Convert the 'time_t' of the modified time into something more human
  if(!localtime_r(&buf.st_mtime, &timeinf))
    return false;

  t->secs    = timeinf.tm_sec;
  t->minutes = timeinf.tm_min;
  t->hours   = timeinf.tm_hour;
  t->day     = timeinf.tm_mday;
  t->month   = timeinf.tm_mon;
  t->year    = timeinf.tm_year;
*/
  // Check the sanity of the coders...
  if (!t)
    return false;

  if (!filename)
    return false;

  memset(t,0,sizeof(i_time_t));

  t->secs    = 0;
  t->minutes = 0;
  t->hours   = 0;
  t->day     = 1;
  t->month   = 1;
  t->year    = 1900;

  return true;
}

//
// I_SystemShutdown
//
// -ACB- 1998/07/11 Tidying the code
//
void I_SystemShutdown(void)
{
  // -ES- 2000/01/23 Avoid infinite recursion.
  static boolean_t firsttime = true;

  if (firsttime)
    firsttime = false;
  else
    return;

  if (!nosound)
  {
    I_ShutdownMusic();
    I_ShutdownSound();
  }

  I_ShutdownGraphics();
  I_ShutdownControl();

  remove_timer();

  allegro_exit();

  // -KM- 1999/01/31 Close the debugfile
#ifdef DEVELOPERS
  if (debugfile)
    fclose(debugfile);
#endif
}



