//----------------------------------------------------------------------------
//  EDGE Networking Code (OS independend parts)
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------
//
// -MH- 1998/07/02 "shootupdown" --> "true3dgameplay"
//
// -ACB- 1998/07/25 DEVELOPERS define for debugging
//                  Extended Settings output more descriptive
//
// -ACB- 1998/09/06 Removed the startmap/startepisode stuff
//
// -KM- 1998/12/21 3 monitor view works.
// TODO: Make sure DDF/RTS files are the same on all PC's on a network
//

#include "i_defs.h"
#include "e_net.h"

#include "con_main.h"
#include "ddf_main.h"
#include "g_game.h"
#include "gui_main.h"
#include "dm_defs.h"
#include "dm_state.h"
#include "e_main.h"
#include "m_argv.h"
#include "m_fixed.h"
#include "m_menu.h"
#include "p_local.h"
#include "p_spec.h"
#include "p_bot.h"
#include "s_sound.h"
#include "version.h"
#include "z_zone.h"

#define NCMD_EXIT 0x80000000
#define NCMD_RETRANSMIT 0x40000000
#define NCMD_SETUP 0x20000000
#define NCMD_KILL 0x10000000  // kill game
#define NCMD_CHECKSUM 0x0fffffff

typedef struct
{
  int tics;
  // set false as nodes leave game
  boolean_t in_game;
  // set when local needs tics
  boolean_t remoteresend;
  // set when remote needs tics
  int resendto;
  int resendcount;
} netnode_t;

netnode_t *netnodes;
netnode_t *localnode;

doomcom_t *doomcom;
static doomdata_t *netbuffer;  // points inside doomcom

//
// NETWORKING
//
// gametic is the tic about to (or currently being) run
// maketic is the tick that hasn't had control made for it yet
// netnode->tics has the maketics for a node
//
// a gametic cannot be run until tics > gametic for all netnodes
//
#define RESENDCOUNT 10
#define PL_DRONE 0x80  // bit flag in doomdata->player

int ticdup;
int maketic;

static int skiptics;
static int maxsend;  // BACKUPTICS/(2*ticdup)-1

int gametime;

// -KM- 1998/01/29 New Setup code.
static int setupflags;

// -ES- The server's random seed in netgames.
static long net_seed;

// Local communication (send packets to self) is always done through this,
// to simplify the system specifics.
static doomdata_t *localcom_queue;
static int localcom_size;

// true if there may be other nodes on the network
static int network_active = false;

//
// NetbufferSize
//
// Returns the size of netbuffer. This depends on the number of tics to send
static int NetBufferSize(void)
{
  return (int)&(((doomdata_t *) 0)->cmds[netbuffer->numtics]);
}

//
// NetbufferChecksum
//
// Calculates a checksum for netbuffer.
static unsigned int NetBufferChecksum(void)
{
  unsigned c;
  int i, l;

  c = 0x1234567;

  l = (NetBufferSize() - (int)&(((doomdata_t *) 0)->retransmitfrom)) / 4;
  for (i = 0; i < l; i++)
    c += ((unsigned *)&netbuffer->retransmitfrom)[i] * (i + 1);

  return c & NCMD_CHECKSUM;
}

//
// ExpandTics
//
static int ExpandTics(int low)
{
  int delta;

  delta = low - (maketic & 0xff);

  if (delta >= -64 && delta <= 64)
    return (maketic & ~0xff) + low;
  if (delta > 64)
    return (maketic & ~0xff) - 256 + low;
  if (delta < -64)
    return (maketic & ~0xff) + 256 + low;

  I_Error("ExpandTics: strange value %i at maketic %i", low, maketic);
  return 0;
}

//
// HSendPacket
//
static void HSendPacket(int node, int flags)
{
#if (DEBUG_NET)
  int i;
  int realretrans;
#endif

  netbuffer->checksum = NetBufferChecksum() | flags;

  if (node == consoleplayer->netnode)
  {
    // Send packet to self.
    localcom_size++;
    Z_Resize(localcom_queue, doomdata_t, localcom_size);
    localcom_queue[localcom_size - 1] = *netbuffer;
    return;
  }

  if (demoplayback)
    return;

  if (!netgame)
    I_Error("Tried to transmit to another node");

  doomcom->command = CMD_SEND;
  doomcom->remotenode = node;
  doomcom->datalength = NetBufferSize();

#if (DEBUG_NET)
  if (netbuffer->checksum & NCMD_RETRANSMIT)
    realretrans = ExpandTics(netbuffer->retransmitfrom);
  else
    realretrans = -1;

  L_WriteDebug("HSendPacket: send %d (%i + %i, R %i) [%i] ",
      node,
      ExpandTics(netbuffer->starttic),
      netbuffer->numtics, realretrans, doomcom->datalength);

  for (i = 0; i < doomcom->datalength; i++)
    L_WriteDebug("%i ", ((byte *) netbuffer)[i]);

  L_WriteDebug("\n");
#endif

  I_NetCmd();
}

// -KM- 1999/01/29 Recieves a setup packet.  Should change settings instantly.
//  works at startup, needs a bit of work to fix for in-game changes.
static boolean_t gotinfo[MAXNETNODES];
static void E_ReceiveSetupPacket(void)
{
  setupdata_t *packet = (setupdata_t *) & doomcom->data;

  if (packet->info.version != DEMOVERSION)
    I_Error("EDGE Versions are incompatable!");

  packet->player &= 0x7f;

  if (packet->info.drone)
  {
    if (packet->player)
      doomcom->drone |= 1 << packet->player;
    else
      doomcom->drone = packet->info.drone;
  }

  if ((packet->checksum & ~NCMD_CHECKSUM) != (unsigned int)setupflags)
    return;

  // This is an ACK Packet
  gotinfo[packet->player] = true;
  if (packet->player)
    return;

  global_flags = packet->info.setupflags;
  level_flags  = packet->info.setupflags;
  startskill = packet->info.skill;
  deathmatch = packet->info.deathmatch;

  if (gamestate != -1)
    I_Error("E_ReceiveSetupPacket: Can only be called at startup!");

  startmap = Z_StrDup(packet->info.startmap);

  net_seed = packet->info.random_seed;
//    else if (!currentmap || strcmp(packet->startmap, currentmap->ddf.name))
//      G_InitNew(packet->skill, DDF_LevelMapLookup(packet->startmap), packet->random_seed);

  // ACK setup packet
  packet->player = (byte)doomcom->consoleplayer;
  packet->info.drone = drone;

  HSendPacket(doomcom->remotenode, packet->checksum & ~NCMD_CHECKSUM);
}

//
// HGetPacket
// Returns false if no packet is waiting
//
static boolean_t E_HGetPacket(void)
{
#if (DEBUG_NET)
  int realretrans;
  int i;
#endif

  if (localcom_size)
  {
    localcom_size--;
    *netbuffer = localcom_queue[localcom_size];
    Z_Resize(localcom_queue, doomdata_t, localcom_size);
    return true;
  }

  if (!netgame)
    return false;

  if (demoplayback)
    return false;

  if (!network_active)
    return false;
    
  doomcom->command = CMD_GET;
  I_NetCmd();

  // Some consistency checks
  if (doomcom->datalength != NetBufferSize())
  {
#if (DEBUG_NET)
    L_WriteDebug("E_HGetPacket: bad packet length %i\n",
doomcom->datalength);
#endif
    return false;
  }

  if (NetBufferChecksum() != (netbuffer->checksum & NCMD_CHECKSUM))
  {
#if (DEBUG_NET)
    L_WriteDebug("E_HGetPacket: bad packet checksum %i\n",
doomcom->datalength);
#endif
    return false;
  }

  if (doomcom->remotenode == -1)
    return false;

  if (netbuffer->checksum & NCMD_SETUP)
  {
    L_WriteDebug("E_HGetPacket: Setup packet %d\n", doomcom->remotenode);
    E_ReceiveSetupPacket();
    return true;
  }

#if (DEBUG_NET)
  if (netbuffer->checksum & NCMD_RETRANSMIT)
    realretrans = ExpandTics(netbuffer->retransmitfrom);
  else
    realretrans = -1;

  L_WriteDebug("E_HGetPacket: get %i = (%i + %i, R %i)[%i] ",
      doomcom->remotenode,
      ExpandTics(netbuffer->starttic),
      netbuffer->numtics, realretrans, doomcom->datalength);

  for (i = 0; i < doomcom->datalength; i++)
    L_WriteDebug("%i ", ((byte *) netbuffer)[i]);

  L_WriteDebug("\n");
#endif

  return true;
}

//
// GetPackets
//
static void GetPackets(void)
{
  player_t *pl;
  netnode_t *netnode;
  int realend;
  int realstart;

  while (E_HGetPacket())
  {
    if (netbuffer->checksum & NCMD_SETUP)
      continue;  // extra setup packet

    pl = playerlookup[netbuffer->player & ~PL_DRONE];
    if (! pl)
      continue;

    if (netbuffer->player & PL_DRONE)
      doomcom->drone |= 1 << pl->pnum;

    netnode = &netnodes[doomcom->remotenode];

    // to save bytes, only the low byte of tic numbers are sent
    // Figure out what the rest of the bytes are
    realstart = ExpandTics(netbuffer->starttic);
    realend = (realstart + netbuffer->numtics);

    // check for exiting the game
    if (netbuffer->checksum & NCMD_EXIT)
    {
      if (!netnode->in_game)
        continue;
      netnode->in_game = false;
      P_RemovePlayerFromGame(pl);

      // -KM- 1998/12/21 Player numbers are 0 based, add 1.
      CON_Printf("Player %d left the game", pl->pnum + 1);

      if (demorecording)
        G_CheckDemoStatus();
      continue;
    }

    // check for a remote game kill
    if (netbuffer->checksum & NCMD_KILL)
      I_Error("Killed by network driver");

    pl->netnode = netnode - netnodes;

    // check for retransmit request
    if (netnode->resendcount <= 0
        && (netbuffer->checksum & NCMD_RETRANSMIT))
    {
      netnode->resendto = ExpandTics(netbuffer->retransmitfrom);

#if (DEBUG_NET)
      L_WriteDebug("GetPackets: retransmit from %i\n", netnode->resendto);
#endif
      netnode->resendcount = RESENDCOUNT;
    }
    else
      netnode->resendcount--;

    // check for out of order / duplicated packet  
    if (realend == netnode->tics)
      continue;

    if (realend < netnode->tics)
    {

#if (DEBUG_NET)
      L_WriteDebug("out of order packet (%i + %i)\n",
          realstart, netbuffer->numtics);
#endif
      continue;
    }

    // check for a missed packet
    if (realstart > netnode->tics)
    {
      // stop processing until the other system resends the missed tics

#if (DEBUG_NET)
      L_WriteDebug(
          "missed tics from %i (%i - %i)\n",
          (int)(netnode-netnodes), realstart, netnode->tics);
#endif
      netnode->remoteresend = true;
      continue;
    }

    // update command store from the packet
    {
      int start;
      ticcmd_t *cmd;

      netnode->remoteresend = false;

      start = netnode->tics - realstart;
      cmd = &netbuffer->cmds[start];

      while (netnode->tics < realend)
      {
        pl->netcmds[netnode->tics % BACKUPTICS] = *cmd;
        netnode->tics++;
        cmd++;
      }
    }
  }
}

//
// E_NetUpdate
// Builds ticcmds for console player,
// sends out a packet
//
void E_NetUpdate(void)
{
  int nowtime;
  int newtics;
  int i, j;
  int realstart;
  int gameticdiv;
  ticcmd_t *cmd;
  netnode_t *netnode;
  player_t *p;

  // check time
  nowtime = I_GetTime() / ticdup;
  newtics = nowtime - gametime;
  gametime = nowtime;

  // -ACB- 1998/07/17 removed goto - educational note: GOTO's are best avoided.
  if (newtics <= 0)
  {
    // nothing new to update
    GetPackets();
    return;
  }

  if (skiptics <= newtics)
  {
    newtics -= skiptics;
    skiptics = 0;
  }
  else
  {
    skiptics -= newtics;
    newtics = 0;
  }

  // build new ticcmds for console player
  gameticdiv = gametic / ticdup;
  for (i = 0; i < newtics; i++)
  {
    I_ControlGetEvents();
    E_ProcessEvents();
    if (maketic - gameticdiv >= BACKUPTICS / 2 - 1)
      break;  // can't hold any more

    // think
    for (p = players; p; p = p->next)
    {
      if (p->thinker)
      {
        cmd = &p->netcmds[maketic % BACKUPTICS];
        p->thinker(p, p->data, cmd);
        cmd->consistency = p->consistency[maketic % BACKUPTICS];
      }
    }

    maketic++;
  }

  if (singletics)
    return;  // singletic update is syncronous

  // send packets from all local players
  for (p = players; p; p = p->next)
  {
    // thinker is only set on local players, net players will be updated over
    // the network.
    if (p->thinker)
    {
      // Prepare and send the packet to the other nodes
      for (i = 0, netnode = netnodes; i < doomcom->numnodes; i++, netnode++)
      {
        if (netnode->in_game)
        {
          netbuffer->player = p->pnum;
          netbuffer->starttic = realstart = netnode->resendto;
          netbuffer->numtics = maketic - realstart;
          if (netbuffer->numtics > BACKUPTICS)
            I_Error("NetUpdate: netbuffer->numtics > BACKUPTICS");
          for (j = 0; j < netbuffer->numtics; j++)
            netbuffer->cmds[j] = p->netcmds[(realstart + j) % BACKUPTICS];
          netnode->resendto = maketic - doomcom->extratics;
    
          if (netnode->remoteresend)
          {
            netbuffer->retransmitfrom = netnode->tics;
            HSendPacket(i, NCMD_RETRANSMIT);
          }
          else
          {
            netbuffer->retransmitfrom = 0;
            HSendPacket(i, 0);
          }
        }
      }
    }
  }

  GetPackets();
}

//
// E_CheckAbort
//
static void E_CheckAbort(void)
{
  event_t *ev;
  int stoptic;

  stoptic = I_GetTime() + 2;
  while (I_GetTime() < stoptic)
    I_ControlGetEvents();

  I_ControlGetEvents();
  for (; eventtail != eventhead; eventtail = (++eventtail) & (MAXEVENTS - 1))
  {
    ev = &events[eventtail];
    if (ev->type == ev_keydown && ev->value.key == KEYD_ESCAPE)
      I_Error("Network game synchronization aborted.");
  }
}

static void E_PrintSetupPacket(void)
{
  // -ACB- 1998/07/25 Display the current settings

  I_Printf("             True3D: %s\n",
      global_flags.true3dgameplay ? "On" : "Off");
  I_Printf("   Enemy Respawning:\n");
  if (global_flags.res_respawn)
    I_Printf("       Resurrection:");
  else
    I_Printf("           Teleport:");
  I_Printf(global_flags.respawn ? "On\n" : "Off\n");

  // Shows enabled even if not set with altdeath
  I_Printf("       Item Respawn: %s\n",
      global_flags.itemrespawn ? "On" : "Off");

  I_Printf("      Gravity Level: %1.1f\n",
    (float_t)global_flags.menu_grav / (float_t)MENU_GRAV_NORMAL);

  I_Printf("           Fastparm: %s\n", global_flags.fastparm ? "On" : "Off");
  I_Printf("         More Blood: %s\n", global_flags.more_blood ? "On" : "Off");
  I_Printf("            Jumping: %s\n", global_flags.jump ? "On" : "Off");
  I_Printf("          Crouching: %s\n", global_flags.crouch ? "On" : "Off");
  I_Printf("              Mlook: %s\n", global_flags.mlook ? "On" : "Off");
  I_Printf("       Translucency: %s\n", global_flags.trans ? "On" : "Off");
  I_Printf("           Monsters: %s\n", global_flags.nomonsters ? "Off" : "On");
  I_Printf("             Extras: %s\n", global_flags.have_extra ? "On" : "Off");
}

void E_SendSetupPacket(int flags)
{
  setupdata_t *packet = (setupdata_t *) & doomcom->data;
  int i;
  boolean_t done;

  net_seed = I_PureRandom();

  // Only player 0 can send setup packets
  if (doomcom->consoleplayer)
    return;

  memset(gotinfo, false, sizeof(boolean_t)*MAXNETNODES);
  setupflags = flags;

  do
  {
    E_CheckAbort();
    L_WriteDebug("Sending Setup Packet...\n");
    done = true;
    for (i = 1; i < doomcom->numnodes; i++)
    {
      if (gotinfo[i])
        continue;

      done = false;
      packet->info.version = DEMOVERSION;
      packet->player = (byte)doomcom->consoleplayer;
      packet->info.deathmatch = deathmatch;
      packet->info.drone = (byte)doomcom->drone;
      packet->info.random_seed = net_seed;
      
      if ((int)gamestate == -1)
      {
        unsigned int len;

        len = sizeof(doomdata_t) - sizeof(setupdata_t);
        if (len < strlen(startmap))
          I_Error("Startmap name '%s' too long (max is %d chars)", startmap, len);

        strcpy(packet->info.startmap, startmap);
        packet->info.skill = startskill;
        packet->info.setupflags = global_flags;
      }
      else
      {
        I_Error("E_SendSetupPacket: Can only be called at startup");
//        packet->info.skill = gameskill;
//        if (currentmap)
//          strcpy(packet->startmap, currentmap->ddf.name);  // FIXME
//        else
//          packet->startmap[0] = 0;
//
//        packet->info.setupflags = level_flags;
      }
      // number of ticcmds is total_size/ticcmd_size rounded up
      packet->numtics = (sizeof(setup_info_t) + strlen(packet->info.startmap) +
         sizeof(ticcmd_t) - 1) / sizeof(ticcmd_t);
      HSendPacket(i, flags);
    }

    while (E_HGetPacket())
      ;
  }
  while (!done);
}

//
// E_ArbitrateNetStart
//
// -ACB- 1998/07/25 Cleaned up the extended settings; Output more descriptive.
//
static void E_ArbitrateNetStart(void)
{
  autostart = true;

  if (doomcom->consoleplayer != 0)
  {
    // listen for setup info from key player
    CON_Message("ListenNet");
    memset(gotinfo, false, sizeof(gotinfo));
    setupflags = NCMD_SETUP;
    while (!gotinfo[0])
    {
      E_HGetPacket();
      E_CheckAbort();
    }
    E_PrintSetupPacket();
    memset(gotinfo, false, sizeof(gotinfo));
    setupflags = NCMD_SETUP | NCMD_EXIT;
    do
    {
      E_HGetPacket();
      E_CheckAbort();
    }
    while (!gotinfo[0]);
  }
  else
  {
    // key player, send the setup info
    CON_Message("SendNet");
    E_PrintSetupPacket();
    E_SendSetupPacket(NCMD_SETUP);
    E_SendSetupPacket(NCMD_EXIT | NCMD_SETUP);
  }
}

//
// E_QuitNetGame
// Called before quitting to leave a net game
// without hanging the other players
//
void E_QuitNetGame(void)
{
  int i, j;

  if (!netgame || !usergame || consoleplayer == NULL || demoplayback)
    return;

  // send a bunch of packets for security
  netbuffer->player = consoleplayer->pnum;
  if (drone)
    netbuffer->player |= PL_DRONE;
  netbuffer->numtics = 0;
  for (i = 0; i < 4; i++)
  {
    for (j = 1; j < doomcom->numnodes; j++)
      if (netnodes[j].in_game)
        HSendPacket(j, NCMD_EXIT);
    I_WaitVBL(1);
  }
}

//
// E_TryRunTics
//
void E_TryRunTics(void)
{
  static int oldentertics;
  static int frameskip[4] = {0,0,0,0};
  static int oldnettics = -1;
  static int frameon = 0;
  int i;
  player_t *p;
  int lowtic;
  int entertic;
  int realtics;
  int availabletics;
  int counts;
  netnode_t *netnode;

  // get real tics  
  entertic = I_GetTime() / ticdup;
  realtics = entertic - oldentertics;
  oldentertics = entertic;

  // get available tics
  E_NetUpdate();

  lowtic = INT_MAX;

  for (i = 0, netnode = netnodes; i < doomcom->numnodes; i++, netnode++)
  {
    if (netnode->in_game)
    {
      if (netnode->tics < lowtic)
        lowtic = netnode->tics;
    }
  }
  availabletics = lowtic - gametic / ticdup;

  // decide how many tics to run
  if (realtics < availabletics - 1)
    counts = realtics + 1;
  else if (realtics < availabletics)
    counts = realtics;
  else
    counts = availabletics;

  if (counts < 1)
    counts = 1;

  frameon++;

#if (DEBUG_NET)
  L_WriteDebug("=======real: %i  avail: %i  game: %i\n",
      realtics, availabletics, counts);
#endif

  if (!demoplayback)
  {
    // ideally nettics[0] should be 1 - 3 tics above lowtic
    // if we are consistantly slower, speed up time
    for (p = players; p; p = p->next)
    {
      if (p->in_game)
        break;
    }

    DEV_ASSERT2(p);

    // the key player does not adapt
    if (consoleplayer != p)
    {
      i = netnodes[p->netnode].tics;
      if (netnodes[0].tics <= i)
        gametime--;
      frameskip[frameon & 3] = (oldnettics > i);
      oldnettics = netnodes[0].tics;
      if (frameskip[0] && frameskip[1] && frameskip[2] && frameskip[3])
        skiptics = 1;
    }
  }  // demoplayback

  // wait for new tics if needed
  while (lowtic < gametic / ticdup + counts)
  {
    E_NetUpdate();
    lowtic = INT_MAX;

    for (i = 0, netnode = netnodes; i < doomcom->numnodes; i++, netnode++)
      if (netnode->in_game && netnode->tics < lowtic)
        lowtic = netnode->tics;

    if (lowtic < gametic / ticdup)
      I_Error("TryRunTics: lowtic < gametic");

    // don't stay in here forever -- give the menu a chance to work
    if (I_GetTime() / ticdup - entertic >= 20)
    {
      GUI_MainTicker();
      M_Ticker();
      return;
    }

  }

  // run the count * ticdup dics
  while (counts--)
  {
    for (i = 0; i < ticdup; i++)
    {
      if (gametic / ticdup > lowtic)
        I_Error("gametic>lowtic");
    
      if (advancedemo)
        E_DoAdvanceDemo();
      
      GUI_MainTicker();
      M_Ticker();
      G_Ticker();
      S_SoundTicker(); // -ACB- 1999/10/11 Improved sound update routines
      S_MusicTicker(); // -ACB- 1999/11/13 Improved music update routines
      gametic++;

      // modify command for duplicated tics
      if (i != ticdup - 1)
      {
        ticcmd_t *cmd;
        int buf;

        buf = (gametic / ticdup) % BACKUPTICS;
        for (p = players; p; p = p->next)
        {
          cmd = &p->netcmds[buf];
          cmd->chatchar = 0;
          if (cmd->buttons & BT_SPECIAL)
            cmd->buttons = 0;
        }
      }
    }
    E_NetUpdate();  // check for new console commands
  }
}

//
// E_NetGetRandomSeed
//
// Returns the random seed for the level. In multiplayer, this is received
// from the server, in single player it's created.
//
long E_NetGetRandomSeed(void)
{
  if (network_active)
    return net_seed;
  else
    return I_PureRandom();
}

//
// E_CheckNetGame
//
// Works out player numbers among the net participants
//
// -KM- 1999/01/29 Fixed network play for 3 view and normal play.
//
boolean_t E_CheckNetGame(void)
{
  int i;
  float_t f;
  const char *s;

  //
  // -ES- Major fixme: Clean up netnodes. They should be more dynamic,
  // players should not depend on different DDF entries (they should all
  // use the same entry), and btw C/S networking should be used instead.
  //
  netnodes = Z_ClearNew(netnode_t, MAXPLAYERS);

  for (i = 0; i < MAXPLAYERS; i++)
  {
    netnodes[i].in_game = false;
    netnodes[i].tics = 0;
    netnodes[i].remoteresend = false;  // set when local needs tics

    netnodes[i].resendto = 0;  // which tic to start sending
  }

  doomcom = Z_ClearNew(doomcom_t, 1);

  netgame = false;
  doomcom->id = DOOMCOM_ID;
  doomcom->numplayers = doomcom->numnodes = 1;
  doomcom->deathmatch = 0;
  doomcom->consoleplayer = 0;
  doomcom->extratics = 0;
  doomcom->ticdup = 1;

  // I_InitNetwork sets doomcom and netgame if networking will be done
  I_InitNetwork();

  if (doomcom->id != DOOMCOM_ID)
    I_Error("Doomcom buffer invalid!");

  if (doomcom->numnodes > 1)
    network_active = true;

  // Additional players will be paralysed unless they are bots
  if (doomcom->numplayers == 1)
  {
    s = M_GetParm("-players");
    if (s)
    {
      doomcom->numplayers = atoi(s);
    }
  }
  else
    netgame = true;

  if (doomcom->numplayers > MAXPLAYERS)
    I_Error("E_CheckNetGame: Only %d player entries in DDF, %d required", MAXPLAYERS, doomcom->numplayers);

  for (i = 0; i < doomcom->numplayers; i++)
    if (! playerlookup[i])
      I_Error("E_CheckNetGame: Missing DDF entry for player %d!", i);

  // -ES- FIXME: Need a better bot adding system when multiplayer over network
  // is possible
  s = M_GetParm("-bots");
  if (s)
  {
    I_Printf("  Bots: ");
    i = atoi(s);
    if (i > doomcom->numplayers-1)
      I_Error("E_CheckNetGame: More bots than available players!");
    for (; i > 0; i--)
    {
      P_BotCreate(playerlookup[i]);
      I_Printf("%d", i + 1);
      if (i > 1)
        I_Printf(", ");
    }
    I_Printf("\n");
    netgame = true;
  }

  netbuffer = &doomcom->data;
  consoleplayer = displayplayer = playerlookup[doomcom->consoleplayer];

  // -ES- Fixme: This belongs somewhere else (around G_PlayerReborn).
  // Needs a big cleanup though.
  consoleplayer->thinker = P_ConsolePlayerThinker;
  consoleplayer->data = NULL;

  //
  // -viewangle <angle> [num]
  //
  // Offsets the viewangle by <angle> degrees.
  // Set the displayplayer to num if passed.
  //
  i = M_CheckParm("-viewangle");
  if (i && i + 1 < M_GetArgCount())
  {
    sscanf(M_GetArgument(i + 1), "%f", &f);
    viewanglebaseoffset = ANG1 * f;
    if (netgame)
    {
      drone = true;
      doomcom->drone |= 1 << consoleplayer->pnum;
    }

    // -KM- 1998/12/21 Find the display player.
    //  EDGE has enough network players for
    //  more than one persone with 3 monitors each.
    if (i + 1 >= M_GetArgCount() || (1 != sscanf(M_GetArgument(i + 2), "%d", &i)))
      i = 0;
      
    displayplayer = playerlookup[i];
  }

  if (netgame)
    E_ArbitrateNetStart();

  I_Printf("  startmap: %s  startskill: %i  deathmatch: %i\n", startmap, startskill, deathmatch);

  // read values out of doomcom
  ticdup = doomcom->ticdup;
  maxsend = BACKUPTICS / (2 * ticdup) - 1;
  if (maxsend < 1)
    maxsend = 1;

  for (i = 0; i < doomcom->numplayers; i++)
    if (playerlookup[i])
      P_AddPlayerToGame(playerlookup[i]);

  for (i = 0; i < doomcom->numnodes; i++)
    netnodes[i].in_game = true;

  I_Printf("  Console: Player %i of %i (%i nodes)\n",
      consoleplayer->pnum + 1, doomcom->numplayers, doomcom->numnodes);

  return true;
}
