#ifndef __L_EPI_H__
#define __L_EPI_H__

//
// Engine links
//

void ENGINE_Main(void);

//
// From l_main.c
//

typedef void shutdown_f(void *data);
typedef void text_receiver_f(const char *str);

void L_Startup(void);
void L_RegisterShutdownRoutine(shutdown_f *f, void *data);
void L_Shutdown(void);
void L_Error(const char *message, ...);

//
// From l_mem.c
//

typedef enum
{
  L_UrgencyLow     = 1,
  L_UrgencyMedium  = 2,
  L_UrgencyHigh    = 3,
  L_UrgencyExtreme = 4
}
flusher_urgency_e;

// A cache flusher is a function that can find and free unused memory.
typedef void cache_flusher_f(flusher_urgency_e urge);

void L_RegisterCacheFlusher(cache_flusher_f *f);
void L_UnregisterCacheFlusher(cache_flusher_f *f);
void *L_Calloc2(int size);
void *L_Malloc2(int size);
void *L_ReMalloc2(void *ptr, int size);
void L_Free(void *ptr);
void L_CheckHeap(void);
int L_FreeMemory(int level);
int L_UsedMemory(int level);

void L_InitMemorySystem(void);
void L_ShutdownMemorySystem(void);

// -ES- 1999/12/16 Leak Hunt
#ifdef LEAK_HUNT
void *L_RegisterMalloc(void *ptr, const char *file, int row);
void *L_UnRegisterTmpMalloc(void *ptr, const char *func);
void L_DumpLeakInfo(int level);
#define L_Malloc(size) L_RegisterMalloc(L_Malloc2(size),__FILE__,__LINE__)
#define L_Calloc(size) L_RegisterMalloc(L_Calloc2(size),__FILE__,__LINE__)
#define L_ReMalloc(ptr,size) L_RegisterMalloc(L_ReMalloc2(ptr,size),__FILE__,__LINE__)
#else
#define L_Malloc L_Malloc2
#define L_Calloc L_Calloc2
#define L_ReMalloc L_ReMalloc2
#endif

//
// L_New
//
// Allocates num elements of type. Use this instead of L_Malloc whenever
// possible.
#define L_New(type, num) ((type *) L_Malloc((num) * sizeof(type)))

//
// L_Resize
//
// Reallocates a block. Use instead of L_ReMalloc wherever possible.
// Unlike normal L_ReMalloc, the pointer parameter is assigned the new
// value, and there is no return value.
#define L_Resize(ptr,type,n)  \
    (void)((ptr) = (type *) L_ReMalloc((void *)(ptr), (n) * sizeof(type)))

//
// L_ClearNew
//
// The L_Calloc version of L_New. Allocates mem and clears it to zero.
#define L_ClearNew(type, num) ((type *) L_Calloc((num) * sizeof(type)))

//
// From l_memhlp.c
//

typedef struct stack_array_s stack_array_t;

struct stack_array_s
{
  // points to the actual array pointer.
  // the stack_array is defined to be uninitialised if this one is NULL.
  void ***ptr;

  // the size of each element in ptr.
  int elem_size;

  // The number of currently used objects.
  // Elements above this limit will be freed automatically if memory gets
  // tight. Otherwise it will stay there, saving some future L_ReMalloc calls.
  int num;

  // The number of allocated objects.
  int max;

  // alloc_bunch elements will be allocated at a time, saving some memory block
  // overheads.
  // If this is negative, the array will work in a different way:
  // *ptr will be an array of elements rather than an array of pointers.
  // This means that no pointer except *ptr may point to an element
  // of the array, since it can be reallocated any time.
  int alloc_bunch;

  // Shows the number of users who have locked this grow_array.
  int locked;

  stack_array_t *next;
};

// Stack array functions.
void L_LockStackArray(stack_array_t *a);
void L_UnlockStackArray(stack_array_t *a);
void L_ClearStackArray(stack_array_t *a);
void L_DeleteStackArray(stack_array_t *a);
void L_InitStackArray(stack_array_t *a, void ***ptr, int elem_size, int alloc_bunch);
stack_array_t *L_CreateStackArray(void ***ptr, int elem_size, int alloc_bunch);
void L_SetArraySize(stack_array_t *a, int num);

// Generic helper functions.
char *L_StrDup(const char *s);

// Initialisation
void L_InitMemoryHelpers(void);

//
// stdio wrappers
//

//
// L_MemSet
//
// Clears memory to ch.
#define L_MemSet(ptr, ch, type, num)  \
    memset((void *)(ptr), (ch) + ((ptr) - ((type *)(ptr))), (num) * sizeof(type))

//
// L_Clear
//
// Clears memory to zero.
#define L_Clear(ptr, type, num) L_MemSet(ptr, 0, type, num)

//
// L_MoveMem
//
// Moves data from src to dest.
#define L_MoveMem(dest, src, type, num)  \
    memmove((void *)(dest), (void *)(src), (num) * sizeof(type) + ((src) - (type *)(src)) + ((dest) - (type *)(dest)))

//
// L_CopyMem
//
// Copies data from src to dest. Faster than L_MoveMem, but the memory areas
// must be disjunctive.
#define L_CopyMem(dest, src, type, num)  \
    memcpy((void *)(dest), (void *)(src), (num) * sizeof(type) + ((src) - (type *)(src)) + ((dest) - (type *)(dest)))

//
// L_StrNCpy
//
// Copies up to max characters of src into dest, and then applies a
// terminating zero (so dest must hold at least max+1 characters).
// The terminating zero is always applied (there is no reason not to)
#define L_StrNCpy(dest, src, max) \
    (void)(strncpy((dest), (src), (max)), (dest)[(max)] = 0)

//
// L_StrCpy
//
#define L_StrCpy(dest, src) \
    (void)(strcpy((dest), (src)))

//
// l_compen.c
//

#ifndef L_StrUpr
void L_StrUpr(char *str);
#endif
#ifndef L_StrCaseCmp
int L_StrCaseCmp(const char *s1, const char *s2);
#endif
#ifndef L_StrNCaseCmp
int L_StrNCaseCmp(const char *s1, const char *s2, int max);
#endif

//
// l_string.c
//
int L_StrNCaseCmpWild(const char *s1, const char *s2, int n);

#endif // __L_EPI_H__
