//----------------------------------------------------------------------------
//  EDGE MP3 Interface
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------

#include "i_defs.h"
#include "l_mp3.h"

#include "z_zone.h"

#include "xingmp3/mhead.h"

// sound parameters
static int current_rate = -1;
static int current_bits = -1;
static int current_chan = -1;
static int bytes_per_sample = -1;

// Maximum samples that decoding from MP3 will ever produce.
// Also the maximum samples that the platform code can ever request.
#define MAX_OVERFLOW  8192

// the ring buffer
static byte *ring_buffer = NULL;
static int ring_samples;

// the bitstream buffer
static byte *bs_buffer = NULL;
static int bs_total;
static int bs_pos;
static int bs_len;

// info from the MP3 file
static int file_handle = -1;
static int file_pos;
static int file_startpos;
static int file_endpos;   // < 0 for files, >= 0 for lumps.

static MPEG_HEAD file_head;
static DEC_INFO file_info;
static int file_framebytes;
static boolean_t file_eof;

// Current read/write positions in the ring, in samples.  When
// read_pos == write_pos, then the ring buffer is empty.  When
// write_pos == (read_pos - 1), the ring buffer is completely full.
//
// NOTE: the volatile is necessary, as these values can be
// used/updated in different threads/execution contexts.

static volatile int r_read_pos;
static volatile int r_write_pos;

#define RING_SIZE()   \
    (((r_write_pos - r_read_pos) + ring_samples) % ring_samples)

#define RING_SPACE()  \
    (ring_samples - RING_SIZE())


// stuff needed by XingMP3

float_t equalizer[32] = 
{
  1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
  1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
  1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
  1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
};
int enableEQ = 0;

//
// L_MP3Init
//
// Initialise this MP3 interface.  Should be called right after the
// platform sound code is successfully initialised, passing the sound
// device's sample rate, bits and mono/stereo-ness.
//
void L_MP3Init(int sample_rate, int sample_bits, int channels)
{
  DEV_ASSERT2(4000 <= sample_rate && sample_rate <= 44100);
  DEV_ASSERT2(sample_bits == 8 || sample_bits == 16);
  DEV_ASSERT2(channels == 1 || channels == 2);

  current_rate = sample_rate;
  current_bits = sample_bits;
  current_chan = channels;

  bytes_per_sample = (current_bits/8) * current_chan;

  // setup ring buffer.  Allocate enough space in the ring buffer for
  // at least one complete second of sound.

  ring_samples = MAX(current_rate, 4 * MAX_OVERFLOW);
  ring_buffer = Z_New(byte, (ring_samples + MAX_OVERFLOW) * bytes_per_sample);
}

//
// L_MP3Shutdown
//
// Shuts down the MP3 interface.  The platform code *must* ensure
// (before calling this routine) that any thread/interrupt/signal
// handler which calls L_MP3ReadBuffer() or L_MP3ReadAdvance() has
// already been disabled.
//
void L_MP3Shutdown(void)
{
  DEV_ASSERT2(current_rate > 0);

  current_rate = -1;

  if (file_handle >= 0)
  {
    if (file_endpos < 0)
      close(file_handle);

    file_handle = -1;
  }
  
  if (bs_buffer)
  {
    Z_Free(bs_buffer);
    bs_buffer = NULL;
  }

  if (ring_buffer)
  {
    byte *temp = ring_buffer;

    ring_buffer = NULL;
    Z_Free(temp);
  }
}

//
// TryReadInfo
//
// Attempts to parse the MP3 header information at the beginning of
// the MP3 file.  Returns true if all went well, otherwise false.
// When successful, the bitstream buffer will be ready for reading.
//
static boolean_t TryReadInfo(int reduction_code)
{
  int bitrate;
  int freq_limit = 24000;
  int convert_code = ((current_chan == 1) ? 1 : 0) +
      ((current_bits == 8) ? 8 : 0);
  int start_idx;

  // seek to beginning
  lseek(file_handle, file_startpos, SEEK_SET);

  // read in initial amount
  bs_len = read(file_handle, bs_buffer, bs_total);

  if (bs_len <= 0)
  {
    I_Warning("L_MP3SetMusicFile: Couldn't read MP3 file.\n");
    return false;
  }

  file_pos = file_startpos + bs_len;

  // parse mpeg header
  file_framebytes = head_info3(bs_buffer, bs_len, &file_head, 
      &bitrate, &start_idx);

  if (file_framebytes == 0)
  {
    I_Warning("L_MP3SetMusicFile: Bad MP3 file !\n");
    return false;
  }

  L_WriteDebug("MP3: Framebytes: %d  Bitrate: %d  StartIdx: %d\n", 
      file_framebytes, bitrate, start_idx);

  bs_pos += start_idx;
  file_startpos += start_idx;

  if (file_endpos > 0)
    file_endpos -= start_idx;

  // initialise decoder
  if (!audio_decode_init(&file_head, file_framebytes,
      reduction_code, 0, convert_code, freq_limit))
  {
    I_Warning("L_MP3SetMusicFile: Decoder init fail !\n");
    return false;
  }

  // read info
  audio_decode_info(&file_info);

  L_WriteDebug("MP3: channels %d  samprate %ld  bits %d  "
      "framebytes %d  type %d\n", file_info.channels, file_info.samprate,
      file_info.bits, file_info.framebytes, file_info.type);

  L_WriteDebug("MP3: sync %d  id %d  opt %d  prot %d  mode %d\n",
      file_head.sync, file_head.id, file_head.option,
      file_head.prot, file_head.mode);

  if (file_info.channels < 1 || file_info.channels > 2 ||
      (file_info.bits != 8 && file_info.bits != 16))
  {
    I_Warning("L_MP3SetMusicFile: Weird channels or bitsize !\n");
    return false;
  }
 
  return true;
}

//
// L_MP3SetMusicFile
//
// Attempt to begin decoding the given MP3 file.  Returns true if all
// systems are go, and returns false is something went wrong.  When
// successful, sets the `sample_rate' parameter to what the MP3 file
// is (a reduction value is chosen to make it as close to the sound
// device's sample_rate as possible).
//
boolean_t L_MP3SetMusicFile(const char *filename, int *sample_rate)
{
  int handle;

  DEV_ASSERT2(bs_buffer == NULL);
  DEV_ASSERT2(ring_buffer);

  // open file
  handle = open(filename, O_RDONLY | O_BINARY);

  if (handle < 0)
  {
    I_Warning("L_MP3SetMusicFile: couldn't open file: %s\n", filename);
    return false;
  }

  L_WriteDebug("MP3: Opened file: %s\n", filename);

  if (! L_MP3SetMusicLump(handle, 0, -1, sample_rate))
  {
    close(handle);
    return false;
  }

  return true;
}

//
// L_MP3SetMusicLump
//
// Like L_MP3SetMusicFile() above, but for a lump in a PWAD.  `handle'
// is the file handle as returned from open() (and stored by the
// w_wad.c code in data_file_t).
//
boolean_t L_MP3SetMusicLump(int handle, int position, int size, 
    int *sample_rate)
{
  int reduction_code = 0;

  DEV_ASSERT2(handle >= 0);
  DEV_ASSERT2(bs_buffer == NULL);
  DEV_ASSERT2(ring_buffer);

  file_handle = handle;
  file_eof = false;
  file_startpos = position;
  file_endpos = (size < 0) ? -1 : (position + size);
  file_pos = 0;

  if (size >= 0)
    L_WriteDebug("MP3: Using lump from %d..%d in file %d\n",
      file_startpos, file_endpos, handle);

  //
  // -ACB- 2000/08/23: Read position and write position are matched
  //                   so that the ring buffer is considered to be
  //                   empty.
  //
  r_read_pos = r_write_pos = 0;

  // setup bitstream buffer
  bs_total  = 80 * 1024;
  bs_buffer = Z_New(byte, bs_total);
  bs_pos    = 0;
  bs_len    = 0;

  // read initial information
  if (!TryReadInfo(reduction_code))
  {
    Z_Free(bs_buffer);
    bs_buffer = NULL;

    file_handle = -1;
    return false;
  }
  
  while (reduction_code < 2 && 
      (file_info.samprate >> (reduction_code+1)) >= current_rate)
  {
    reduction_code++;
  }

  L_WriteDebug("MP3: Reduction code chosen = %d\n", reduction_code);

  if (reduction_code > 0)
  {
    if (! TryReadInfo(reduction_code))
    {
      Z_Free(bs_buffer);
      bs_buffer = NULL;

      file_handle = -1;
      return false;
    }
  }

  (*sample_rate) = file_info.samprate;
  
  return true;
}

//
// L_MP3ClearMusicFile
//
// Clear any data associated with a previously successful
// L_MP3SetMusicFile call.  The platform code *must* ensure (before
// calling this routine) that any thread/interrupt/signal handler will
// no longer call L_MP3ReadBuffer() or L_MP3ReadAdvance().
//
// Note: There is no shutdown function for XingMP3
//
void L_MP3ClearMusicFile(void)
{
  DEV_ASSERT2(file_handle >= 0);
  DEV_ASSERT2(bs_buffer);

  if (file_endpos < 0)
    close(file_handle);

  file_handle = -1;

  Z_Free(bs_buffer);
  bs_buffer = NULL;
}

//
// DecodeIntoRingBuffer
//
// Reads data from the file (when necessary) and calls XingMP3 to
// convert a frame of MP3 to audio data in the ring buffer.
//
// NOTE: channel number differences and bitsize differences are
// handled by XingMP3 itself (via convert_code), and sample rate
// differences need to be handled in the platform sound code.
//
static boolean_t DecodeIntoRingBuffer(void)
{
  IN_OUT inout;
 
  volatile byte *write_pos = ring_buffer + (r_write_pos * bytes_per_sample);

  DEV_ASSERT2(bs_buffer);
  DEV_ASSERT2(0 <= bs_len && bs_len <= bs_total);
  DEV_ASSERT2(0 <= bs_pos && bs_pos <= bs_total);

  // fill bitstream buffer, if needed
  if (bs_len < file_framebytes)
  {
    int num = bs_total - bs_len;
    
    if (bs_len > 0)
      memmove(bs_buffer, bs_buffer + bs_pos, bs_len);
      
    if (file_endpos >= 0 && num > file_endpos - file_pos)
      num = file_endpos - file_pos;

    if (num > 0)
    {
      // NOTE WELL: must do a seek when using lumps, as the w_wad code
      // may have read other lumps (and hence moved the read position)
      // since the last time we were here.

      if (file_endpos >= 0)
        lseek(file_handle, file_pos, SEEK_SET);

      num = read(file_handle, bs_buffer + bs_len, num);
    }

    if (num <= 0 || bs_len + num < file_framebytes)
    {
      L_WriteDebug("MP3: EOF !\n");
      file_eof = true;
      return 0;
    }

    bs_pos = 0;
    bs_len += num;
    file_pos += num;
  }

  inout = audio_decode(bs_buffer + bs_pos, (short *) write_pos);

#if 0
  L_WriteDebug("MP3: IN=%d  OUT=%d\n", inout.in_bytes, inout.out_bytes);
  L_WriteDebug("MP3: Data: %02X %02X %02X %02X %02X %02X %02X %02X\n",
      write_pos[0], write_pos[1], write_pos[2], write_pos[3],
      write_pos[4], write_pos[5], write_pos[6], write_pos[7]);
#endif

  //
  // -ACB- 2000/08/23 Errors Suck: If the input bytes are zero
  //                               or less, just return zero - this
  //                               will stop the current MP3 playing.
  //                               Some MP3's have bad sync at
  //                               the end of them for termination
  //                               reasons.
  //
  if (inout.in_bytes <= 0)
  {
    L_WriteDebug("MP3: OUT OF SYNC !\n");
    file_eof = true;
    return 0;
  }
 
  bs_pos += inout.in_bytes;
  bs_len -= inout.in_bytes;

  return inout.out_bytes / file_info.channels;
}

//
// L_MP3FillBuffer
//
// Fill the ring buffer with data decoded from the MP3 file.  Returns
// true if all went well, or false if the file's EOF was reached.
//
boolean_t L_MP3FillBuffer(void)
{
  int space;
  int samples;
  boolean_t got_some = false;
 
  DEV_ASSERT2(file_handle >= 0);
  DEV_ASSERT2(bs_buffer);
  DEV_ASSERT2(ring_buffer);

  if (file_eof)
    return false;

  space = RING_SPACE();

  while (space >= MAX_OVERFLOW)
  {
    samples = DecodeIntoRingBuffer();

    if (!samples)
      return got_some;
    
    got_some = true;

    // handle case of overflowing ring buffer
    if (r_write_pos + samples > ring_samples)
    {
      // must copy samples to the bottom of the ring buffer
      int over_num = (r_write_pos + samples - ring_samples);

      I_MoveData(ring_buffer, 
          ring_buffer + (ring_samples * bytes_per_sample),
          over_num * bytes_per_sample);
    }

    r_write_pos = (r_write_pos + samples) % ring_samples;
    space -= samples;
  }

  return true;
}

//
// L_MP3RestartMusicFile
//
// Cause the MP3 file to start playing from the beginning again.
// Usually called after L_MP3FillBuffer() returns an EOF indication.
//
void L_MP3RestartMusicFile(void)
{
  DEV_ASSERT2(file_handle >= 0);
  DEV_ASSERT2(bs_buffer);
  DEV_ASSERT2(ring_buffer);

  L_WriteDebug("MP3: Restarting...\n");

  // seek to the beginning of the file
  lseek(file_handle, file_startpos, SEEK_SET);

  // clear bitstream buffer
  bs_pos = 0;
  bs_len = 0;

  file_eof = false;
  file_pos = file_startpos;
}

//
// L_MP3ReadBuffer
//
// Attempt to read a block of samples of the given size, returning a
// pointer to it, or NULL if there wasn't enough data to satisfy the
// request.
//
byte *L_MP3ReadBuffer(int samples)
{
  DEV_ASSERT2(samples <= MAX_OVERFLOW);

  if (!ring_buffer)
    return NULL;
  
  if (RING_SIZE() < samples)
  {
    // Ouch, we ran out of data
    return NULL;
  }

  // does this request overflow the ring buffer ?
  if ((r_read_pos+samples) > ring_samples)
  {
    // must copy samples from bottom of ring buffer
    int over_num = ((r_read_pos + samples) - ring_samples);

    I_MoveData(ring_buffer + (ring_samples * bytes_per_sample),
        ring_buffer, over_num * bytes_per_sample);
  }
  
  return ring_buffer + (r_read_pos * bytes_per_sample);
}

//
// L_MP3ReadAdvance
//
// Move the read position within the ring buffer by the given number
// of samples.  Should be called after a successful L_MP3ReadBuffer()
// call, using the same number of samples.
//
void L_MP3ReadAdvance(int samples)
{
  DEV_ASSERT2(samples <= MAX_OVERFLOW);
  DEV_ASSERT2(ring_buffer);
  DEV_ASSERT2(RING_SIZE() >= samples);

  r_read_pos = (r_read_pos + samples) % ring_samples;
}

