//----------------------------------------------------------------------------
//  EDGE Linux FM Patches
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  DESCRIPTION:
//    Default OPL2 patches, for use when GENMIDI is unavailable (e.g.
//    when using our EPI in another project).
//
//  SOURCE: 
//    Allegro.  More specially: instrument patches by Jorrit Rouwe,
//    and percussion patches partly by Jorrit Rouwe and partly from
//    the SBTimbre utility.
//
//----------------------------------------------------------------------------

#include "../i_defs.h"
#include "i_sysinc.h"


#define N_INSTR  175

typedef struct FM_INSTRUMENT
{
  byte characteristic1;
  byte characteristic2;
  byte level1;
  byte level2;
  byte attackdecay1;
  byte attackdecay2;
  byte sustainrelease1;
  byte sustainrelease2;
  byte wave1;
  byte wave2;
  byte feedback;
  byte freq;
  byte key;
  byte type;
  
  char *name;
} 
FM_INSTRUMENT;

#define FM_HH  1
#define FM_CY  2
#define FM_TT  4
#define FM_SD  8
#define FM_BD  16

static const byte genmidi_magic[8] = "#OPL_II#";

static const FM_INSTRUMENT allegro_fm_patches[N_INSTR] = {

{ 0x21,0x21,0x8f,0xc,0xf2,0xf2,0x45,0x76,0,0,8,0,0,0,"Acoustic Grand" },
{ 0x31,0x21,0x4b,9,0xf2,0xf2,0x54,0x56,0,0,8,0,0,0,"Bright Acoustic" },
{ 0x31,0x21,0x49,9,0xf2,0xf2,0x55,0x76,0,0,8,0,0,0,"Electric Grand" },
{ 0xb1,0x61,0xe,9,0xf2,0xf3,0x3b,0xb,0,0,6,0,0,0,"Honky-Tonk" },
{ 1,0x21,0x57,9,0xf1,0xf1,0x38,0x28,0,0,0,0,0,0,"Electric Piano 1" },
{ 1,0x21,0x93,9,0xf1,0xf1,0x38,0x28,0,0,0,0,0,0,"Electric Piano 2" },
{ 0x21,0x36,0x80,0x17,0xa2,0xf1,1,0xd5,0,0,8,0,0,0,"Harpsichord" },
{ 1,1,0x92,9,0xc2,0xc2,0xa8,0x58,0,0,0xa,0,0,0,"Clav" },
{ 0xc,0x81,0x5c,9,0xf6,0xf3,0x54,0xb5,0,0,0,0,0,0,"Celesta" },
{ 7,0x11,0x97,0x89,0xf6,0xf5,0x32,0x11,0,0,2,0,0,0,"Glockenspiel" },
{ 0x17,1,0x21,9,0x56,0xf6,4,4,0,0,2,0,0,0,"Music Box" },
{ 0x18,0x81,0x62,9,0xf3,0xf2,0xe6,0xf6,0,0,0,0,0,0,"Vibraphone" },
{ 0x18,0x21,0x23,9,0xf7,0xe5,0x55,0xd8,0,0,0,0,0,0,"Marimba" },
{ 0x15,1,0x91,9,0xf6,0xf6,0xa6,0xe6,0,0,4,0,0,0,"Xylophone" },
{ 0x45,0x81,0x59,0x89,0xd3,0xa3,0x82,0xe3,0,0,0xc,0,0,0,"Tubular Bells" },
{ 3,0x81,0x49,0x89,0x74,0xb3,0x55,5,1,0,4,0,0,0,"Dulcimer" },
{ 0x71,0x31,0x92,9,0xf6,0xf1,0x14,7,0,0,2,0,0,0,"Drawbar Organ" },
{ 0x72,0x30,0x14,9,0xc7,0xc7,0x58,8,0,0,2,0,0,0,"Percussive Organ" },
{ 0x70,0xb1,0x44,9,0xaa,0x8a,0x18,8,0,0,4,0,0,0,"Rock Organ" },
{ 0x23,0xb1,0x93,9,0x97,0x55,0x23,0x14,1,0,4,0,0,0,"Church Organ" },
{ 0x61,0xb1,0x13,0x89,0x97,0x55,4,4,1,0,0,0,0,0,"Reed Organ" },
{ 0x24,0xb1,0x48,9,0x98,0x46,0x2a,0x1a,1,0,0xc,0,0,0,"Accoridan" },
{ 0x61,0x21,0x13,9,0x91,0x61,6,7,1,0,0xa,0,0,0,"Harmonica" },
{ 0x21,0xa1,0x13,0x92,0x71,0x61,6,7,0,0,6,0,0,0,"Tango Accordian" },
{ 2,0x41,0x9c,0x89,0xf3,0xf3,0x94,0xc8,1,0,0xc,0,0,0,"Acoustic Guitar(nylon)" },
{ 3,0x11,0x54,9,0xf3,0xf1,0x9a,0xe7,1,0,0xc,0,0,0,"Acoustic Guitar(steel)" },
{ 0x23,0x21,0x5f,9,0xf1,0xf2,0x3a,0xf8,0,0,0,0,0,0,"Electric Guitar(jazz)" },
{ 3,0x21,0x87,0x89,0xf6,0xf3,0x22,0xf8,1,0,6,0,0,0,"Electric Guitar(clean)" },
{ 3,0x21,0x47,9,0xf9,0xf6,0x54,0x3a,0,0,0,0,0,0,"Electric Guitar(muted)" },
{ 0x23,0x21,0x4a,0xe,0x91,0x84,0x41,0x19,1,0,8,0,0,0,"Overdriven Guitar" },
{ 0x23,0x21,0x4a,9,0x95,0x94,0x19,0x19,1,0,8,0,0,0,"Distortion Guitar" },
{ 9,0x84,0xa1,0x89,0x20,0xd1,0x4f,0xf8,0,0,8,0,0,0,"Guitar Harmonics" },
{ 0x21,0xa2,0x1e,9,0x94,0xc3,6,0xa6,0,0,2,0,0,0,"Acoustic Bass" },
{ 0x31,0x31,0x12,9,0xf1,0xf1,0x28,0x18,0,0,0xa,0,0,0,"Electric Bass(finger)" },
{ 0x31,0x31,0x8d,9,0xf1,0xf1,0xe8,0x78,0,0,0xa,0,0,0,"Electric Bass(pick)" },
{ 0x31,0x32,0x5b,9,0x51,0x71,0x28,0x48,0,0,0xc,0,0,0,"Fretless Bass" },
{ 1,0x21,0x8b,0x49,0xa1,0xf2,0x9a,0xdf,0,0,8,0,0,0,"Slap Bass 1" },
{ 0x21,0x21,0x8b,0x11,0xa2,0xa1,0x16,0xdf,0,0,8,0,0,0,"Slap Bass 2" },
{ 0x31,0x31,0x8b,9,0xf4,0xf1,0xe8,0x78,0,0,0xa,0,0,0,"Synth Bass 1" },
{ 0x31,0x31,0x12,9,0xf1,0xf1,0x28,0x18,0,0,0xa,0,0,0,"Synth Bass 2" },
{ 0x31,0x21,0x15,9,0xdd,0x56,0x13,0x26,1,0,8,0,0,0,"Violin" },
{ 0x31,0x21,0x16,9,0xdd,0x66,0x13,6,1,0,8,0,0,0,"Viola" },
{ 0x71,0x31,0x49,9,0xd1,0x61,0x1c,0xc,1,0,8,0,0,0,"Cello" },
{ 0x21,0x23,0x4d,0x89,0x71,0x72,0x12,6,1,0,2,0,0,0,"Contrabass" },
{ 0xf1,0xe1,0x40,9,0xf1,0x6f,0x21,0x16,1,0,2,0,0,0,"Tremolo Strings" },
{ 2,1,0x1a,0x89,0xf5,0x85,0x75,0x35,1,0,0,0,0,0,"Pizzicato Strings" },
{ 2,1,0x1d,0x89,0xf5,0xf3,0x75,0xf4,1,0,0,0,0,0,"Orchestral Strings" },
{ 0x10,0x11,0x41,9,0xf5,0xf2,5,0xc3,1,0,2,0,0,0,"Timpani" },
{ 0x21,0xa2,0x9b,0xa,0xb1,0x72,0x25,8,1,0,0xe,0,0,0,"String Ensemble 1" },
{ 0xa1,0x21,0x98,9,0x7f,0x3f,3,7,1,1,0,0,0,0,"String Ensemble 2" },
{ 0xa1,0x61,0x93,9,0xc1,0x4f,0x12,5,0,0,0xa,0,0,0,"SynthStrings 1" },
{ 0x21,0x61,0x18,9,0xc1,0x4f,0x22,5,0,0,0xc,0,0,0,"SynthStrings 2" },
{ 0x31,0x72,0x5b,0x8c,0xf4,0x8a,0x15,5,0,0,0,0,0,0,"Choir Aahs" },
{ 0xa1,0x61,0x90,9,0x74,0x71,0x39,0x67,0,0,0,0,0,0,"Voice Oohs" },
{ 0x71,0x72,0x57,9,0x54,0x7a,5,5,0,0,0xc,0,0,0,"Synth Voice" },
{ 0x90,0x41,0,9,0x54,0xa5,0x63,0x45,0,0,8,0,0,0,"Orchestra Hit" },
{ 0x21,0x21,0x92,0xa,0x85,0x8f,0x17,9,0,0,0xc,0,0,0,"Trumpet" },
{ 0x21,0x21,0x94,0xe,0x75,0x8f,0x17,9,0,0,0xc,0,0,0,"Trombone" },
{ 0x21,0x61,0x94,9,0x76,0x82,0x15,0x37,0,0,0xc,0,0,0,"Tuba" },
{ 0x31,0x21,0x43,9,0x9e,0x62,0x17,0x2c,1,1,2,0,0,0,"Muted Trumpet" },
{ 0x21,0x21,0x9b,9,0x61,0x7f,0x6a,0xa,0,0,2,0,0,0,"French Horn" },
{ 0x61,0x22,0x8a,0xf,0x75,0x74,0x1f,0xf,0,0,8,0,0,0,"Brass Section" },
{ 0xa1,0x21,0x86,0x8c,0x72,0x71,0x55,0x18,1,0,0,0,0,0,"SynthBrass 1" },
{ 0x21,0x21,0x4d,9,0x54,0xa6,0x3c,0x1c,0,0,8,0,0,0,"SynthBrass 2" },
{ 0x31,0x61,0x8f,9,0x93,0x72,2,0xb,1,0,8,0,0,0,"Soprano Sax" },
{ 0x31,0x61,0x8e,9,0x93,0x72,3,9,1,0,8,0,0,0,"Alto Sax" },
{ 0x31,0x61,0x91,9,0x93,0x82,3,9,1,0,0xa,0,0,0,"Tenor Sax" },
{ 0x31,0x61,0x8e,9,0x93,0x72,0xf,0xf,1,0,0xa,0,0,0,"Baritone Sax" },
{ 0x21,0x21,0x4b,9,0xaa,0x8f,0x16,0xa,1,0,8,0,0,0,"Oboe" },
{ 0x31,0x21,0x90,9,0x7e,0x8b,0x17,0xc,1,1,6,0,0,0,"English Horn" },
{ 0x31,0x32,0x81,9,0x75,0x61,0x19,0x19,1,0,0,0,0,0,"Bassoon" },
{ 0x32,0x21,0x90,9,0x9b,0x72,0x21,0x17,0,0,4,0,0,0,"Clarinet" },
{ 0xe1,0xe1,0x1f,9,0x85,0x65,0x5f,0x1a,0,0,0,0,0,0,"Piccolo" },
{ 0xe1,0xe1,0x46,9,0x88,0x65,0x5f,0x1a,0,0,0,0,0,0,"Flute" },
{ 0xa1,0x21,0x9c,9,0x75,0x75,0x1f,0xa,0,0,2,0,0,0,"Recorder" },
{ 0x31,0x21,0x8b,9,0x84,0x65,0x58,0x1a,0,0,0,0,0,0,"Pan Flute" },
{ 0xe1,0xa1,0x4c,9,0x66,0x65,0x56,0x26,0,0,0,0,0,0,"Blown Bottle" },
{ 0x62,0xa1,0xcb,9,0x76,0x55,0x46,0x36,0,0,0,0,0,0,"Skakuhachi" },
{ 0x62,0xa1,0xa2,9,0x57,0x56,7,7,0,0,0xb,0,0,0,"Whistle" },
{ 0x62,0xa1,0x9c,9,0x77,0x76,7,7,0,0,0xb,0,0,0,"Ocarina" },
{ 0x22,0x21,0x59,9,0xff,0xff,3,0xf,2,0,0,0,0,0,"Lead 1 (square)" },
{ 0x21,0x21,0xe,9,0xff,0xff,0xf,0xf,1,1,0,0,0,0,"Lead 2 (sawtooth)" },
{ 0x22,0x21,0x46,0x89,0x86,0x64,0x55,0x18,0,0,0,0,0,0,"Lead 3 (calliope)" },
{ 0x21,0xa1,0x45,9,0x66,0x96,0x12,0xa,0,0,0,0,0,0,"Lead 4 (chiff)" },
{ 0x21,0x22,0x8b,9,0x92,0x91,0x2a,0x2a,1,0,0,0,0,0,"Lead 5 (charang)" },
{ 0xa2,0x61,0x9e,0x49,0xdf,0x6f,5,7,0,0,2,0,0,0,"Lead 6 (voice)" },
{ 0x20,0x60,0x1a,9,0xef,0x8f,1,6,0,2,0,0,0,0,"Lead 7 (fifths)" },
{ 0x21,0x21,0x8f,0x86,0xf1,0xf4,0x29,9,0,0,0xa,0,0,0,"Lead 8 (bass+lead)" },
{ 0x77,0xa1,0xa5,9,0x53,0xa0,0x94,5,0,0,2,0,0,0,"Pad 1 (new age)" },
{ 0x61,0xb1,0x1f,0x89,0xa8,0x25,0x11,3,0,0,0xa,0,0,0,"Pad 2 (warm)" },
{ 0x61,0x61,0x17,9,0x91,0x55,0x34,0x16,0,0,0xc,0,0,0,"Pad 3 (polysynth)" },
{ 0x71,0x72,0x5d,9,0x54,0x6a,1,3,0,0,0,0,0,0,"Pad 4 (choir)" },
{ 0x21,0xa2,0x97,9,0x21,0x42,0x43,0x35,0,0,8,0,0,0,"Pad 5 (bowed)" },
{ 0xa1,0x21,0x1c,9,0xa1,0x31,0x77,0x47,1,1,0,0,0,0,"Pad 6 (metallic)" },
{ 0x21,0x61,0x89,0xc,0x11,0x42,0x33,0x25,0,0,0xa,0,0,0,"Pad 7 (halo)" },
{ 0xa1,0x21,0x15,9,0x11,0xcf,0x47,7,1,0,0,0,0,0,"Pad 8 (sweep)" },
{ 0x3a,0x51,0xce,9,0xf8,0x86,0xf6,2,0,0,2,0,0,0,"FX 1 (rain)" },
{ 0x21,0x21,0x15,9,0x21,0x41,0x23,0x13,1,0,0,0,0,0,"FX 2 (soundtrack)" },
{ 6,1,0x5b,9,0x74,0xa5,0x95,0x72,0,0,0,0,0,0,"FX 3 (crystal)" },
{ 0x22,0x61,0x92,0x8c,0xb1,0xf2,0x81,0x26,0,0,0xc,0,0,0,"FX 4 (atmosphere)" },
{ 0x41,0x42,0x4d,9,0xf1,0xf2,0x51,0xf5,1,0,0,0,0,0,"FX 5 (brightness)" },
{ 0x61,0xa3,0x94,0x89,0x11,0x11,0x51,0x13,1,0,6,0,0,0,"FX 6 (goblins)" },
{ 0x61,0xa1,0x8c,0x89,0x11,0x1d,0x31,3,0,0,6,0,0,0,"FX 7 (echoes)" },
{ 0xa4,0x61,0x4c,9,0xf3,0x81,0x73,0x23,1,0,4,0,0,0,"FX 8 (sci-fi)" },
{ 2,7,0x85,0xc,0xd2,0xf2,0x53,0xf6,0,1,0,0,0,0,"Sitar" },
{ 0x11,0x13,0xc,0x89,0xa3,0xa2,0x11,0xe5,1,0,0,0,0,0,"Banjo" },
{ 0x11,0x11,6,9,0xf6,0xf2,0x41,0xe6,1,2,4,0,0,0,"Shamisen" },
{ 0x93,0x91,0x91,9,0xd4,0xeb,0x32,0x11,0,1,8,0,0,0,"Koto" },
{ 4,1,0x4f,9,0xfa,0xc2,0x56,5,0,0,0xc,0,0,0,"Kalimba" },
{ 0x21,0x22,0x49,9,0x7c,0x6f,0x20,0xc,0,1,6,0,0,0,"Bagpipe" },
{ 0x31,0x21,0x85,9,0xdd,0x56,0x33,0x16,1,0,0xa,0,0,0,"Fiddle" },
{ 0x20,0x21,4,0x8a,0xda,0x8f,5,0xb,2,0,6,0,0,0,"Shanai" },
{ 5,3,0x6a,0x89,0xf1,0xc3,0xe5,0xe5,0,0,6,0,0,0,"Tinkle Bell" },
{ 7,2,0x15,9,0xec,0xf8,0x26,0x16,0,0,0xa,0,0,0,"Agogo" },
{ 5,1,0x9d,9,0x67,0xdf,0x35,5,0,0,8,0,0,0,"Steel Drums" },
{ 0x18,0x12,0x96,9,0xfa,0xf8,0x28,0xe5,0,0,0xa,0,0,0,"Woodblock" },
{ 0x10,0,0x86,0xc,0xa8,0xfa,7,3,0,0,6,0,0,0,"Taiko Drum" },
{ 0x11,0x10,0x41,0xc,0xf8,0xf3,0x47,3,2,0,4,0,0,0,"Melodic Tom" },
{ 1,0x10,0x8e,9,0xf1,0xf3,6,2,2,0,0xe,0,0,0,"Synth Drum" },
{ 0xe,0xc0,0,9,0x1f,0x1f,0,0xff,0,3,0xe,0,0,0,"Reverse Cymbal" },
{ 6,3,0x80,0x91,0xf8,0x56,0x24,0x84,0,2,0xe,0,0,0,"Guitar Fret Noise" },
{ 0xe,0xd0,0,0xe,0xf8,0x34,0,4,0,3,0xe,0,0,0,"Breath Noise" },
{ 0xe,0xc0,0,9,0xf6,0x1f,0,2,0,3,0xe,0,0,0,"Seashore" },
{ 0xd5,0xda,0x95,0x49,0x37,0x56,0xa3,0x37,0,0,0,0,0,0,"Bird Tweet" },
{ 0x35,0x14,0x5c,0x11,0xb2,0xf4,0x61,0x15,2,0,0xa,0,0,0,"Telephone ring" },
{ 0xe,0xd0,0,9,0xf6,0x4f,0,0xf5,0,3,0xe,0,0,0,"Helicopter" },
{ 0x26,0xe4,0,9,0xff,0x12,1,0x16,0,1,0xe,0,0,0,"Applause" },
{ 0,0,0,9,0xf3,0xf6,0xf0,0xc9,0,2,0xe,0,0,0,"Gunshot" },
{ 0x10,0x11,0x22,0xb,0xf8,0xf3,0x77,6,2,0,0x38,0x87,6,FM_BD,"Acoustic Bass Drum" },
{ 0x10,0x11,0x22,0xb,0xf8,0xf3,0x77,6,2,0,0x38,0x87,6,FM_BD,"Bass Drum 1" },
{ 2,0x11,7,0xb,0xf9,0xf8,0xff,0xff,0,0,0x38,0x60,0xb,FM_TT,"Side Stick" },
{ 0,0,0,0xb,0xfc,0xfa,5,0x17,2,0,0x3e,0xad,0xa,FM_SD,"Acoustic Snare" },
{ 0,1,2,0xb,0xff,0xff,7,8,0,0,0x30,0x63,0xe,FM_SD,"Hand Clap" },
{ 0,0,0,0xb,0xfc,0xfa,5,0x17,2,0,0x3e,0xad,0xe,FM_SD,"Electric Snare" },
{ 1,1,8,8,0xc6,0xc6,0x98,0x98,0,0,5,0x57,0xd,FM_TT,"Low Floor Tom" },
{ 1,1,3,3,0xb8,0xb8,0xb5,0xb5,1,1,0xd,0x57,0x11,FM_CY,"Closed Hi-Hat" },
{ 1,1,0,8,0xc6,0xc6,0x98,0x98,0,0,5,0xb0,0xd,FM_TT,"High Floor Tom" },
{ 1,1,8,8,0x88,0x88,0xa5,0xa5,1,1,3,0x57,0x11,FM_CY,"Pedal Hi-Hat" },
{ 1,1,8,8,0xc6,0xc6,0x98,0x98,0,0,5,2,0xe,FM_TT,"Low Tom" },
{ 1,1,9,9,0x86,0x86,0xa5,0xa5,1,1,3,0x57,0x11,FM_CY,"Open Hi-Hat" },
{ 1,1,8,8,0xc6,0xc6,0x98,0x98,0,0,5,0x57,0x11,FM_TT,"Low-Mid Tom" },
{ 1,1,8,8,0xc6,0xc6,0x98,0x98,0,0,5,0xb0,0x11,FM_TT,"Hi-Mid Tom" },
{ 0xb7,1,0x41,0,0x37,0xff,0x90,6,1,0,0,0x57,0x11,FM_CY,"Crash Cymbal 1" },
{ 1,1,8,8,0xc6,0xc6,0x98,0x98,0,0,5,2,0x12,FM_TT,"High Tom" },
{ 1,1,0x82,0x82,0xf6,0xf6,0xd5,0xd5,1,1,3,0x57,0x11,FM_CY,"Ride Cymbal 1" },
{ 0xb7,1,0x41,0,0x37,0xff,0x90,6,1,0,0,0x57,0x11,FM_CY,"Chinese Cymbal" },
{ 0xe,0xe,7,7,0xb5,0xb5,0x15,0x15,1,1,0xb,0x57,0x11,FM_CY,"Ride Bell" },
{ 1,1,7,7,0x77,0x77,0x73,0x73,1,1,0xb,0x57,0x11,FM_CY,"Tambourine" },
{ 0xb7,1,0x41,0,0x37,0xff,0x90,6,1,0,0,0x57,0x11,FM_CY,"Splash Cymbal" },
{ 0x10,0x10,0,0,0x96,0x96,0x67,0x67,0,0,0xa,0xb0,0x11,FM_SD,"Cowbell" },
{ 0xb7,1,0x41,0,0x37,0xff,0x90,6,1,0,0,0x57,0x11,FM_CY,"Crash Cymbal 2" },
{ 1,1,7,7,0xf9,0xf9,0xb5,0xb5,0,0,0xb,0x57,0x11,FM_SD,"Vibraslap" },
{ 0xd1,0xd1,5,5,0xe7,0xe7,0x65,0x65,1,1,0xd,0x57,0x11,FM_SD,"Ride Cymbal 2" },
{ 1,1,0,0,0xe7,0xe7,0x94,0x94,0,0,3,2,0x12,FM_TT,"Hi Bongo" },
{ 1,1,0,0,0xe7,0xe7,0x94,0x94,0,0,0xb,0x81,0x11,FM_TT,"Low Bongo" },
{ 1,1,0,0,0x96,0x96,0x67,0x67,0,0,0xa,2,0x12,FM_TT,"Mute Hi Conga" },
{ 1,1,0,0,0xb4,0xb4,0x26,0x26,0,0,0xa,2,0x12,FM_TT,"Open Hi Conga" },
{ 1,1,0,0,0xb4,0xb4,0x26,0x26,0,0,0xa,0x57,0x11,FM_TT,"Low Conga" },
{ 1,1,8,8,0xc6,0xc6,0x98,0x98,0,0,5,0x57,0x11,FM_TT,"High Timbale" },
{ 1,1,8,8,0xc6,0xc6,0x98,0x98,0,0,5,2,0xe,FM_TT,"Low Timbale" },
{ 5,0xa7,0,0x7f,0xf6,0x33,0x56,0x90,0,3,0,0x63,0xe,FM_TT,"High Agogo" },
{ 5,0xa7,0,0x7f,0xf6,0x33,0x56,0x90,0,3,0,0x93,0xb,FM_TT,"Low Agogo" },
{ 0,0,0x2e,0x2e,0x8a,0x8a,0xec,0xec,2,2,0xa,0x57,0x11,FM_HH,"Cabasa" },
{ 0,0,0x2e,0x2e,0x8b,0x8b,0xe6,0xe6,2,2,0xa,0x57,0x11,FM_HH,"Maracas" },
{ 0x52,0x52,0x27,0x27,0x59,0x59,0x15,0x15,3,3,0xb,0x82,0x16,FM_TT,"Short Whistle" },
{ 0x52,0x52,0x27,0x27,0x48,0x48,0x14,0x14,3,3,0xb,0x82,0x16,FM_TT,"Long Whistle" },
{ 0,0,0xe,0xe,0x5a,0x5a,0xd6,0xd6,1,1,0xa,0x57,0x11,FM_HH,"Short Guiro" },
{ 0x10,0x10,0xe,0xe,0x53,0x53,0x9f,0x9f,1,1,0xa,0x57,0x11,FM_HH,"Long Guiro" },
{ 0x11,0x11,0,0,0xf5,0xf5,0x75,0x75,0,0,2,2,0x1a,FM_TT,"Claves" },
{ 4,4,0,0,0xf8,0xf8,0xb6,0xb6,1,1,3,0x57,0x11,FM_TT,"Hi Wood Block" },
{ 4,4,0,0,0xf8,0xf8,0xb7,0xb7,1,1,3,0xca,0xd,FM_TT,"Low Wood Block" },
{ 1,1,0xb,0xb,0x5e,0x5e,0xdc,0xdc,1,1,0xa,0x57,0x11,FM_HH,"Mute Cuica" },
{ 0,0,7,7,0x5c,0x5c,0xdc,0xdc,1,1,0xa,0x57,0x11,FM_HH,"Open Cuica" },
{ 0xc5,0xd5,0x4f,0x10,0xf6,0xf6,0x84,0x84,0,0,8,2,0x16,FM_BD,"Mute Triangle" },
{ 0xc5,0xd5,0x4f,0x10,0xf6,0xf6,0x84,0x72,0,0,8,2,0x16,FM_BD,"Open Triangle" }

};


void ConvertPatch(byte *dest, const FM_INSTRUMENT *src, boolean_t drum)
{
  memset(dest, 0, 36);

  // flags
  dest[0] = drum ? 1 : 0;
  dest += 2;

  // finetune and note
  dest[0] = src->key;
  dest[1] = src->freq;
  dest += 2;
  
  // operators
  dest[0]  = dest[16] = src->characteristic1;
  dest[1]  = dest[17] = src->attackdecay1;
  dest[2]  = dest[18] = src->sustainrelease1;
  dest[3]  = dest[19] = src->wave1;
  dest[4]  = dest[20] = src->level1 & 0xC0;
  dest[5]  = dest[21] = src->level1 & 0x3F;
  dest[6]  = dest[22] = src->feedback;
  dest[7]  = dest[23] = src->characteristic2;
  dest[8]  = dest[24] = src->attackdecay2;
  dest[9]  = dest[25] = src->sustainrelease2;
  dest[10] = dest[26] = src->wave2;
  dest[11] = dest[27] = src->level2 & 0xC0;
  dest[12] = dest[28] = src->level2 & 0x3F;

//  dest[14] = dest[30] = src->key;  //!!! src->freq ?
}

void ConvertName(byte *dest, const FM_INSTRUMENT *src)
{
  memset(dest, 0, 32);

  strncpy(dest, src->name, 31);
}

//
// I_CreateGENMIDI
//
// Convert the Allegro patch information into GENMIDI format.
//
void I_CreateGENMIDI(byte *dest)
{
  int i;
  byte *orig = dest;

  memcpy(dest, genmidi_magic, 8);
  dest += 8;

  for (i=0; i < N_INSTR; i++)
  {
    ConvertPatch(dest, allegro_fm_patches + i, i >= 128);
    dest += 36;
  }

  for (i=0; i < N_INSTR; i++)
  {
    ConvertName(dest, allegro_fm_patches + i);
    dest += 32;
  }

  DEV_ASSERT2(dest == (orig + 11908));
}

