//-----------------------------------------------------------------------------
//  EDGE LINUX Music Server (musserv) Interface
//-----------------------------------------------------------------------------
//
//  Copyright (c) 1999-2001  The EDGE Team.
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  DESCRIPTION:
//	Play music via `musserv' : the linux music server.
//
//-----------------------------------------------------------------------------
//
// -AJA- 2000/03/22: wrote this file, based heavily on the code in
//                   Colin Phipp's linux doom port LxDoom (not to
//                   mention Andy Baker's linux/i_cd.c code).
//
#include "../i_defs.h"
#include "i_sysinc.h"

#include "../m_argv.h"
#include "../w_wad.h"

#define MUSSERV_PATH1  homedir
#define MUSSERV_PATH2  "."
#define MUSSERV_PATH3  "/usr/local/games"
#define MUSSERV_PATH4  "/usr/games"

#define MUSSERV_NAME1  "musserv"

// pipe to music server, or NULL if inactive
static FILE *musserver = NULL;

static boolean_t MusservTestEXE(char *cmdbuf, const char *path)
{
  char *new_path = I_PreparePath(path);
  boolean_t result;

  sprintf(cmdbuf, "%s%c%s", new_path, DIRSEPARATOR, MUSSERV_NAME1);

  result = (access(cmdbuf, X_OK) == 0);
  
  Z_Free(new_path);
  return result;
}

//
// I_StartupMusserv
//
boolean_t I_StartupMusserv(void)
{
  char cmdbuf[4096];

  const byte *genmidi_data;
  int genmidi_lump;
  int genmidi_len;
  boolean_t alleg_patches;

  L_WriteDebug("I_StartupMusserv: Initializing...\n");

  if (! MusservTestEXE(cmdbuf, MUSSERV_PATH1) &&
      ! MusservTestEXE(cmdbuf, MUSSERV_PATH2) &&
      ! MusservTestEXE(cmdbuf, MUSSERV_PATH3) &&
      ! MusservTestEXE(cmdbuf, MUSSERV_PATH4))
  {
    I_PostMusicError("Could not find linux music server\n");
    nomusic = true;
    return false;
  }
  
  // FIXME: add options here
  
  musserver = popen(cmdbuf, "w");

  if (! musserver)
  {
    I_PostMusicError("Failed to run linux music server\n");
    nomusic = true;
    return false;
  }

  alleg_patches = M_CheckParm("-allegfm");
  
  if (!alleg_patches)
  {
    genmidi_lump = W_CheckNumForName("GENMIDI");

    if (genmidi_lump < 0)
    {
      I_Warning("Missing GENMIDI lump !  Using in-built patches...\n");
      alleg_patches = true;
    }
  }

  if (alleg_patches)
  {
    genmidi_len  = 8 + (175 * 36) + (175 * 32);
    genmidi_data = (const byte *) Z_Malloc(genmidi_len);

    // Intentional Const Override
    I_CreateGENMIDI((byte *)genmidi_data);
  }
  else
  {
    genmidi_len  = W_LumpLength(genmidi_lump);
    genmidi_data = W_CacheLumpNum(genmidi_lump);
  }

  fwrite(genmidi_data, genmidi_len, 1, musserver);
  fflush(musserver);

  // check for error
  if (ferror(musserver))
  {
    L_WriteDebug("I_StartupMusserv: Init FAILED\n");
    pclose(musserver);
    musserver = NULL;
    return false;
  }

  if (alleg_patches)
    Z_Free((byte *)genmidi_data);
  else
    W_DoneWithLump(genmidi_data);

  L_WriteDebug("I_StartupMusserv: Init OK\n");
  return true;	
}

static boolean_t SendMusFile(const byte *data, int length)
{
  //
  // CPhipps - must use the length as specified by the MUS data, not
  // the lump length, as lxmusserver expects the former.
  //
  // -AJA- This blows very hard, this should be handled by the server
  //       itself.  Ideally, files sent from client to server would be
  //       better encapsulated (e.g. a four byte network order length
  //       before the data).  But for now we need to be compatible
  //       with the existing linux music servers.
  //
  struct musheader_s
  {
      char sig[4];
      unsigned short scorelen;
      unsigned short scorestart;
      unsigned short channels;
      unsigned short sec_channels;
      unsigned short instrumentcount;
      unsigned short pad;
  }
  __attribute__ ((packed)) *musheader = (void*)data;

  int mus_len = sizeof(struct musheader_s) + sizeof(unsigned short) * 
      musheader->instrumentcount + musheader->scorelen;

  if (mus_len < length)
  {
    I_Warning("I_MusicStartPlayback: excess data in MUS lump "
        "(%d > %d), correcting header\n", length, mus_len);

    // Correct the MUS header, by Gady Kozma <gady@math.tau.ac.il>
    musheader->scorelen += length - mus_len; 
    mus_len = length;
  } 
  else if (mus_len > length)
  {
    I_Printf("I_MusicStartPlayback: incomplete MUS lump "
        "(%d < %d)\n", length, mus_len);
    return false;
  }

  clearerr(musserver);

  fprintf(musserver, "N%d\n", 1 /* FIXME: looping_current */);
  fwrite(data, mus_len, 1, musserver);
  fflush(musserver);

  if (ferror(musserver))
  {
    I_Printf("I_MusicStartPlayback: Error transferring data.\n");
    return false;
  }

  return true;
}

//
// I_MusservStartPlayback
//
boolean_t I_MusservStartPlayback(const char *data, int len)
{
  boolean_t result = false;

  if (!musserver)
    return false;

  result = SendMusFile(data, len);

  L_WriteDebug("I_MusservStartPlayback: %s\n", result ? "OK" : "Failed");

  return result;
}

//
// I_MusservPausePlayback
//
void I_MusservPausePlayback(void)
{
  if (!musserver)
    return;

  L_WriteDebug("I_MusservPausePlayback called\n");

  fputc('P', musserver);
  fflush(musserver);
}

//
// I_MusservresumePlayback
//
void I_MusservResumePlayback(void)
{
  if (!musserver)
    return;

  L_WriteDebug("I_MusservResumePlayback called\n");

  fputc('R', musserver);
  fflush(musserver);
}

//
// I_MusservStopPlayback
//
void I_MusservStopPlayback(void)
{
  if (!musserver)
    return;

  L_WriteDebug("I_MusservStopPlayback called\n");

  fputc('S', musserver);
  fflush(musserver);
  // FIXME: check pipe
}

//
// I_MusservSetVolume
//
void I_MusservSetVolume(int vol)
{
  if (!musserver)
    return;

  if (vol < 0 || vol > 15)
    return;
   
  L_WriteDebug("I_MusservSetVolume: changing volume to %d...\n", vol);

  fprintf(musserver, "V%d R\n", vol);
  fflush(musserver);
}

//
// I_ShutdownMusserv
//
void I_ShutdownMusserv()
{
  if (!musserver)
    return;

  L_WriteDebug("I_ShutdownMusserv: killing the server...\n");

  fputc('Q', musserver);
  fflush(musserver);

  pclose(musserver);
  musserver = NULL;
}
