//----------------------------------------------------------------------------
//  EDGE Linux Sound System, Using OSS
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2000  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  -AJA- 2000/03/25: Began work on Linux/OSS support.
//

#include "../i_defs.h"
#include "i_sysinc.h"

#include "../m_argv.h"
#include "../l_mp3.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/soundcard.h>

#define SOUND_DEVICE  "/dev/dsp"

// File descriptor for sound device
static int fd = -1;

// Sound device info
#define SYNC_MILLISECONDS  20

// We use a 22.10 fixed point for sound offsets.  It's a reasonable
// compromise between longest sound and accumulated round-off error.
typedef long fixed22_t;

static int dev_freq;
static int dev_bits;
static int dev_stereo;
static int dev_unsigned;
static int dev_frag_size;
static int dev_frag_pairs;
static int dev_running = 0;
static void *dev_buffer = NULL;
static volatile int dev_lock = 0;
static struct itimerval dev_itimer;

// Storage info
typedef struct mix_sound_s
{
  int length;

  // offset delta value, higher values mean higher pitch
  fixed22_t delta;

  // Note: signed 8 bit values
  char *data_L;
  char *data_R;
}
mix_sound_t;

static mix_sound_t **stored_sfx = NULL;
static int stored_sfx_num = 0;

// Channel info
typedef struct mix_channel_s
{
  int priority;

  int volume_L;
  int volume_R;
  boolean_t looping;

  mix_sound_t *sound;
  int paused;

  fixed22_t offset;
  fixed22_t delta;
}
mix_channel_t;

#define PRI_NOSOUND   -1
#define PRI_FINISHED  -2

// Mixing info
#define MIX_CHANNELS  64

static mix_channel_t mix_chan[MIX_CHANNELS];
static int *mix_buffer_L = NULL;
static int *mix_buffer_R = NULL;

// MP3 details
static int mp3_volume;
static int mp3_rate;
static int mp3_cur_track;
static boolean_t mp3_inited = false;

// Error Description
static char errordesc[256] = "FOO";
static char scratcherror[256];

// Signal Stuff
void InternalSoundTicker(void);

static void SoundSignalHandler(int sig)
{
  signal(SIGPROF, SIG_IGN);

  if (dev_running)
  {
    if (dev_lock == 0)
      InternalSoundTicker();
    
    signal(SIGPROF, SoundSignalHandler);
  }
}

// very basic locking (all we need though)
#define OSS_LOCK()    \
    do { dev_lock++; DEV_ASSERT2(dev_lock <= 4); } while(0)
    
#define OSS_UNLOCK()  \
    do { dev_lock--; DEV_ASSERT2(dev_lock >= 0); } while(0)

//
// I_StartupSound
//
boolean_t I_StartupSound(void *sysinfo)
{
  int fragment;
  int format;

  int want_freq;
  int want_bits;
  int want_stereo;

  const char *dev_name;
  const char *p;
  int i;
  
  if (nosound)
    return true;

  // clear channels
  for (i=0; i < MIX_CHANNELS; i++)
    mix_chan[i].priority = PRI_NOSOUND;

  p = M_GetParm("-sounddev");

  if (p)
    dev_name = p;
  else
    dev_name = SOUND_DEVICE;
  
  fd = open(dev_name, O_WRONLY, 0);

  if (fd < 0)
  {
    I_Printf("I_StartupSound: Unable to open sound device: %s\n",
        SOUND_DEVICE);
    nosound = true;
    return false;
  }

  // 5 fragments, 512 bytes each:
  fragment = 0x00040009;
  
  ioctl(fd, SNDCTL_DSP_SETFRAGMENT, &fragment);
  ioctl(fd, SNDCTL_DSP_GETBLKSIZE, &dev_frag_size);

  format = AFMT_U8;
  dev_bits = 8;
  dev_unsigned = 1;

  ioctl(fd, SNDCTL_DSP_SETFMT, &format);

  // FIXME: support 16 bits
  if (format != AFMT_U8)
  {
    close(fd);
    I_Printf("I_StartupSound: 8 bits not supported !\n");
    nosound = true;
    return false;
  }

  want_stereo = (M_CheckParm("-mono") > 0) ? 0 : 1;

  dev_stereo = want_stereo;
  ioctl(fd, SNDCTL_DSP_STEREO, &dev_stereo);

  if (want_stereo && !dev_stereo)
    I_Printf("I_StartupSound: stereo sound not available.\n");
  else if (!want_stereo && dev_stereo)
    I_Printf("I_StartupSound: mono sound not available.\n");
 
  p = M_GetParm("-freq");

  if (p)
    want_freq = atoi(p);
  else
    want_freq = 11025;

  dev_freq = want_freq;
  ioctl(fd, SNDCTL_DSP_SPEED, &dev_freq);

  if (dev_freq < (want_freq - want_freq/100) || 
      dev_freq > (want_freq + want_freq/100))
  {
    I_Printf("I_StartupSound: %d Hz sound not available.\n", want_freq);
  }
  
  // show some info...
  I_Printf("I_StartupSound: OSS Started: %d Hz, %d bits, %s\n",
      dev_freq, dev_bits, dev_stereo ? "Stereo" : "Mono");

  // allocate output buffer
  dev_buffer = malloc(dev_frag_size);

  if (dev_buffer == NULL)
  {
    close(fd);
    sprintf(errordesc, "I_StartupSound: Out of memory.");
    nosound = true;
    return false;
  }

  dev_frag_pairs = dev_frag_size;

  if (dev_stereo)
    dev_frag_pairs /= 2;
  
  if (dev_bits == 16)
    dev_frag_pairs /= 2;
  
  // allocate mixing buffers
  mix_buffer_L = (int *) malloc(dev_frag_pairs * sizeof(int));

  if (! mix_buffer_L)
  {
    close(fd);
    sprintf(errordesc, "I_StartupSound: Out of memory.\n");
    nosound = true;
    return false;
  }

  mix_buffer_R = (int *) malloc(dev_frag_pairs * sizeof(int));

  if (! mix_buffer_R)
  {
    close(fd);
    sprintf(errordesc, "I_StartupSound: Out of memory.\n");
    nosound = true;
    return false;
  }

  // start interval timer
  dev_running = 1;

  dev_itimer.it_interval.tv_sec  = 0;
  dev_itimer.it_interval.tv_usec = SYNC_MILLISECONDS * 1000;
  dev_itimer.it_value = dev_itimer.it_interval;
  
  signal(SIGPROF, SoundSignalHandler);
  setitimer(ITIMER_PROF, &dev_itimer, NULL);
  
  return true;
}

static fixed22_t ComputeDelta(int data_freq, int device_freq)
{
  // sound data's frequency close enough ?
  if (data_freq > device_freq - device_freq/100 &&
      data_freq < device_freq + device_freq/100)
  {
    return 1 << 10;
  }

  return floor((float_t)data_freq * 1024.0 / device_freq);
}

//
// I_ShutdownSound
//
void I_ShutdownSound(void)
{
  if (nosound)
    return;

  // let signal handler know we are shutting down
  dev_running = 0;

  if (fd >= 0)
  {
    struct itimerval loc_itimer;

    ioctl(fd, SNDCTL_DSP_RESET, 0);
    close(fd);
    fd = -1;

    // stop interval timer
    loc_itimer.it_interval.tv_sec  = 0;
    loc_itimer.it_interval.tv_usec = 0;
    loc_itimer.it_value = loc_itimer.it_interval;
    
    signal(SIGPROF, SIG_IGN);
    setitimer(ITIMER_PROF, &loc_itimer, NULL);
  }

  if (dev_buffer)
  {
    free(dev_buffer);
    dev_buffer = NULL;
  }

  if (mix_buffer_L)
  {
    free(mix_buffer_L);
    mix_buffer_L = NULL;
  }

  if (mix_buffer_R)
  {
    free(mix_buffer_R);
    mix_buffer_L = NULL;
  }

  // FIXME: free sounds
 
  nosound = true;
}

//
// I_LoadSfx
//
boolean_t I_LoadSfx(const unsigned char *data, unsigned int length,
    unsigned int freq, unsigned int handle)
{
  unsigned int i;
  mix_sound_t *sfx;

  OSS_LOCK();

  if (handle >= stored_sfx_num)
  {
    i = stored_sfx_num;
    stored_sfx_num = handle + 1;

    stored_sfx = (mix_sound_t **) realloc(stored_sfx, stored_sfx_num *
        sizeof(mix_sound_t *));
    
    // FIXME: do the old-list new-list thing

    if (! stored_sfx)
    {
      OSS_UNLOCK();
      I_Error("Out of memory in sound code.\n");
    }
  
    // clear any new elements
    for (; i < stored_sfx_num; i++)
      stored_sfx[i] = NULL;
  }

  DEV_ASSERT2(stored_sfx[handle] == NULL);

  sfx = stored_sfx[handle] = (mix_sound_t *)
      malloc(sizeof(mix_sound_t));
  
  if (! sfx)
  {
    OSS_UNLOCK();
    sprintf(errordesc, "Out of memory in sound code.");
    return false;
  }

  sfx->length = length;
  sfx->delta  = ComputeDelta(freq, dev_freq);

  sfx->data_R = NULL;
  sfx->data_L = malloc(length);

  if (! sfx->data_L)
  {
    stored_sfx[handle] = NULL;
    OSS_UNLOCK();
    sprintf(errordesc, "Out of memory in sound code.");
    return false;
  }

  // convert to signed format
  for (i=0; i < length; i++)
  {
    sfx->data_L[i] = (char) (data[i] ^ 0x80);
  }

  OSS_UNLOCK();
  return true;
}

//
// I_UnloadSfx
//
boolean_t I_UnloadSfx(unsigned int handle)
{
  int i;
  mix_sound_t *sfx;
  
  DEV_ASSERT(handle < stored_sfx_num,
      ("I_UnloadSfx: %d out of range", handle));
  
  sfx = stored_sfx[handle];

  DEV_ASSERT(sfx, ("I_UnloadSfx: NULL sample"));

  OSS_LOCK();
  
  // Kill playing sound effects
  for (i = 0; i < MIX_CHANNELS; i++)
  {
    if (mix_chan[i].priority != PRI_NOSOUND && mix_chan[i].sound == sfx)
      I_SoundKill(i);
  }

  // free sound data
  free(sfx->data_L);
  stored_sfx[handle] = NULL;

  OSS_UNLOCK();
  return true;
}

//
// I_SoundPlayback
//
int I_SoundPlayback(unsigned int soundid, int pan, int vol, boolean_t looping)
{
  int i;
  int lvol, rvol;

  mix_sound_t *sfx;

  if (nosound)
    return true;

  DEV_ASSERT(soundid > 0 && soundid < stored_sfx_num,
        ("I_SoundPlayback: Sound %d does not exist!", soundid));

  sfx = stored_sfx[soundid];

  OSS_LOCK();

  for (i=0; i < MIX_CHANNELS; i++)
  {
    if (mix_chan[i].priority == PRI_NOSOUND)
      break;
  }

  if (i >= MIX_CHANNELS)
  {
    OSS_UNLOCK();
    sprintf(errordesc, "I_SoundPlayback: Unable to allocate voice.");
    return -1;
  }

  // compute panning
  lvol = rvol = vol;
  
  if (dev_stereo)
  {
    DEV_ASSERT2(0 <= pan && pan <= 255);

    // strictly linear equations
    lvol = (lvol * (255 - pan)) / 255;
    rvol = (rvol * (0   + pan)) / 255;
  }
 
  mix_chan[i].volume_L = lvol << (16 - dev_bits);
  mix_chan[i].volume_R = rvol << (16 - dev_bits);
  mix_chan[i].looping  = looping;
  mix_chan[i].sound    = sfx;
  mix_chan[i].paused   = 0;
  mix_chan[i].offset   = 0;
  mix_chan[i].priority = 1;
 
  OSS_UNLOCK();
  return i;
}

//
// I_SoundStopLooping
//
boolean_t I_SoundStopLooping(unsigned int chanid)
{
  return false;
}



//
// I_SoundKill
//
boolean_t I_SoundKill(unsigned int chanid)
{
  OSS_LOCK();

  if (mix_chan[chanid].priority == PRI_NOSOUND)
  {
    OSS_UNLOCK();
    sprintf(errordesc, "I_SoundKill: channel not playing.\n");
    return false;
  }
 
  mix_chan[chanid].priority = PRI_NOSOUND;
  mix_chan[chanid].volume_L = 0;
  mix_chan[chanid].volume_R = 0;
  mix_chan[chanid].looping  = false;
  mix_chan[chanid].sound    = NULL;
  mix_chan[chanid].paused   = 0;
  mix_chan[chanid].offset   = 0;

  OSS_UNLOCK();
  return true;
}

//
// I_SoundCheck
//
boolean_t I_SoundCheck(unsigned int chanid)
{
  return (mix_chan[chanid].priority >= 0);
}

//
// I_SoundAlter
//
boolean_t I_SoundAlter(unsigned int chanid, int pan, int vol)
{
  int lvol, rvol;

  if (mix_chan[chanid].priority == PRI_NOSOUND)
  {
    sprintf(errordesc, "I_SoundAlter: channel not playing.\n");
    return false;
  }

  // compute panning
  lvol = rvol = vol;
  
  if (dev_stereo)
  {
    DEV_ASSERT2(0 <= pan && pan <= 255);

    // strictly linear equations
    lvol = (lvol * (255 - pan)) / 255;
    rvol = (rvol * (0   + pan)) / 255;
  }
  
  mix_chan[chanid].volume_L = lvol << (16 - dev_bits);
  mix_chan[chanid].volume_R = rvol << (16 - dev_bits);
  
  return true;
}

//
// I_SoundPause
//
boolean_t I_SoundPause(unsigned int chanid)
{
  if (nosound)
    return true;

  if (mix_chan[chanid].priority == PRI_NOSOUND)
  {
    sprintf(errordesc, "I_SoundPause: channel not playing.\n");
    return false;
  }

  mix_chan[chanid].paused = 1;
  return true;
}

//
// I_SoundResume
//
boolean_t I_SoundResume(unsigned int chanid)
{
  if (nosound)
    return true;

  if (mix_chan[chanid].priority == PRI_NOSOUND)
  {
    sprintf(errordesc, "I_SoundResume: channel not playing.\n");
    return false;
  }

  mix_chan[chanid].paused = 0;
  return true;
}

static void MixChannel(mix_channel_t *chan)
{
  int i;
  int want = dev_frag_pairs;
  
  int *dest_L = mix_buffer_L;
  int *dest_R = mix_buffer_R;

  char *src_L = chan->sound->data_L;

  while (want > 0)
  {
    int count = MIN(chan->sound->length - (chan->offset >> 10), want);

    DEV_ASSERT2(count > 0);

    for (i=0; i < count; i++)
    {
      char src_sample = src_L[chan->offset >> 10];
      
      *dest_L++ += src_sample * chan->volume_L;
      *dest_R++ += src_sample * chan->volume_R;

      chan->offset += chan->sound->delta;
    }

    want -= count;

    // return if sound hasn't finished yet
    if ((chan->offset >> 10) < chan->sound->length)
      return;
 
    if (! chan->looping)
    {
      chan->priority = PRI_FINISHED;
      return;
    }

    // loop back to beginning
    src_L = chan->sound->data_L;
    chan->offset = 0;
  }
}

static void MixMP3Buffer(byte *mp3_buf)
{
  int i;
  
  int *dest_L = mix_buffer_L;
  int *dest_R = mix_buffer_R;

  for (i=0; i < dev_frag_pairs; i++)
  {
    *dest_L++ += (char)(*mp3_buf++ ^ 0x80) * mp3_volume;
    *dest_R++ += (char)(*mp3_buf++ ^ 0x80) * mp3_volume;
  }
}

static INLINE unsigned char ClipSample(int value)
{
  if (value < 0) return 0;
  if (value > 0xFF) return 0xFF;

  return value;
}

static void BlitToSoundDevice(void)
{
  int i;
  unsigned char *dest = (unsigned char *) dev_buffer;

  if (dev_stereo)
  {
    for (i=0; i < dev_frag_pairs; i++)
    {
      *dest++ = ClipSample((mix_buffer_L[i] >> 16) + 0x80);
      *dest++ = ClipSample((mix_buffer_R[i] >> 16) + 0x80);
    }
  }
  else
  {
    for (i=0; i < dev_frag_pairs; i++)
    {
      *dest++ = ClipSample((mix_buffer_L[i] >> 16) + 0x80);
    }
  }

  // write block to sound device
  write(fd, dev_buffer, dev_frag_pairs * (dev_stereo ? 2 : 1) *
      ((dev_bits == 16) ? 2 : 1));
}

//
// InternalSoundTicker
//
void InternalSoundTicker(void)
{
  int i;
  audio_buf_info info;
  byte *mp3_buf;
  
  if (nosound || fd < 0)
    return;
  
  // check how many fragments the sound device can accept.  When
  // greather than 0, it is time to mix some channels and write some
  // sound data to it.

  ioctl(fd, SNDCTL_DSP_GETOSPACE, &info);

  for (; info.fragments > 0; info.fragments--)
  {
    // clear mixer buffer
    memset(mix_buffer_L, 0, sizeof(int) * dev_frag_pairs);
    memset(mix_buffer_R, 0, sizeof(int) * dev_frag_pairs);
  
    // add each channel
    for (i=0; i < MIX_CHANNELS; i++)
    {
      if (mix_chan[i].priority >= 0)
        MixChannel(&mix_chan[i]);
    }

    if (mp3_cur_track && !musicpaused)
    {
      // add MP3 chunk (if any)
      mp3_buf = L_MP3ReadBuffer(dev_frag_pairs);

      if (mp3_buf)
      {
        MixMP3Buffer(mp3_buf);
        L_MP3ReadAdvance(dev_frag_pairs);
      }
    }
    
    BlitToSoundDevice();
  }
}

//
// I_SoundTicker
//
void I_SoundTicker(void)
{
  if (nosound)
    return;

  if (mp3_cur_track)
  {
    if (! L_MP3FillBuffer())
      L_MP3RestartMusicFile();
  }
}

//
// I_SoundReturnError
//
const char *I_SoundReturnError(void)
{
  memcpy(scratcherror, errordesc, sizeof(scratcherror));
  memset(errordesc, '\0', sizeof(errordesc));

  return scratcherror;
}


//  ======  MP3 STUFF  ======


//
// I_StartupMP3
//
boolean_t I_StartupMP3(void)
{
  if (nosound)
    return false;

  // initialise MP3 system
  L_MP3Init(dev_freq, dev_bits, dev_stereo ? 2 : 1);

  mp3_volume = 240 << (16 - dev_bits);
  mp3_cur_track = 0;
  mp3_inited = true;

  return true;
}

//
// I_ShutdownMP3
//
void I_ShutdownMP3(void)
{
  if (mp3_inited)
  {
    L_MP3Shutdown();
    mp3_inited = false;
  }
}

//
// I_MP3PlayTrack
//
int I_MP3PlayTrack(const char *filename, boolean_t looping)
{
  boolean_t result;

  if (mp3_cur_track)
  {
    L_MP3ClearMusicFile();
    mp3_cur_track = 0;
  }

  // FIXME: looping not honoured yet

  result = L_MP3SetMusicFile(filename, &mp3_rate);

  L_WriteDebug("I_MP3StartPlayback: %s\n", result ? "OK" : "Failed");
 
  if (result)
  {
    mp3_cur_track = 1;
    return 1;
  }
  
  return -1;
}

//
// I_MP3StopTrack
//
void I_MP3StopTrack(int track)
{
  L_WriteDebug("I_MP3StopPlayback called\n");

  if (track == mp3_cur_track)
  {
    L_MP3ClearMusicFile();
    mp3_cur_track = 0;
  }
}

//
// I_MP3SetVolume
//
void I_MP3SetVolume(int vol)
{
  if (vol < 0 || vol > 15)
    return;
   
  L_WriteDebug("I_MP3SetVolume: changing volume to %d...\n", vol);

  mp3_volume = (vol * 17) << (16 - dev_bits);
}

