//----------------------------------------------------------------------------
//  EDGE Sound System for SDL
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  -AJA- 2000/07/07: Began work on SDL sound support.
//

#include <SDL/SDL.h>

#include "../i_defs.h"
#include "i_sysinc.h"

#include "../l_mp3.h"
#include "../m_argv.h"
#include "../w_wad.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>

// We use a 22.10 fixed point for sound offsets.  It's a reasonable
// compromise between longest sound and accumulated round-off error.
typedef long fixed22_t;

static SDL_AudioSpec mydev;
static int dev_bits;
static int dev_bytes_per_sample;
static int dev_frag_pairs;

// Storage info
typedef struct mix_sound_s
{
  int length;

  // offset delta value, higher values mean higher pitch
  fixed22_t delta;

  // Note: signed 8 bit values
  char *data_L;
  char *data_R;
}
mix_sound_t;

static mix_sound_t **stored_sfx = NULL;
static int stored_sfx_num = 0;

// Channel info
typedef struct mix_channel_s
{
  int priority;

  int volume_L;
  int volume_R;
  boolean_t looping;

  mix_sound_t *sound;
  int paused;

  fixed22_t offset;
  fixed22_t delta;
}
mix_channel_t;

#define PRI_NOSOUND   -1
#define PRI_FINISHED  -2

// Mixing info
#define MIX_CHANNELS  64

static mix_channel_t mix_chan[MIX_CHANNELS];
static int *mix_buffer_L = NULL;
static int *mix_buffer_R = NULL;

// MP3 details
static int mp3_volume;
static int mp3_rate;
static int mp3_cur_track;
static boolean_t mp3_inited = false;

// Error Description
static char errordesc[256] = "FOO";
static char scratcherror[256];

// Callback Stuff
void InternalSoundFiller(void *udata, Uint8 *stream, int len);

//
// I_StartupSound
//
boolean_t I_StartupSound(void *sysinfo)
{
  int i;
  const char *p;
  SDL_AudioSpec firstdev;

  int want_freq;
//!!!  int want_bits;
  int want_stereo;
  
  if (nosound)
    return true;

  // clear channels
  for (i=0; i < MIX_CHANNELS; i++)
    mix_chan[i].priority = PRI_NOSOUND;

  p = M_GetParm("-freq");
  if (p)
    want_freq = atoi(p);
  else
    want_freq = 11025;

  want_stereo = (M_CheckParm("-mono") > 0) ? 0 : 1;

  firstdev.freq = want_freq;
  firstdev.format = AUDIO_U8;
  firstdev.channels = want_stereo ? 2 : 1;
  firstdev.samples = 512;
  firstdev.callback = InternalSoundFiller;

  if (SDL_OpenAudio(&firstdev, &mydev) < 0)
  {
    I_Printf("I_StartupSound: Couldn't open sound: %s\n", SDL_GetError());
    nosound = true;
    return false;
  }

#ifdef DEVELOPERS
  // get round SDL's signal handlers
  signal(SIGFPE,  SIG_DFL);
  signal(SIGSEGV, SIG_DFL);
#endif

  // check stuff...

  // FIXME: support 16 bits
  if (mydev.format != AUDIO_U8)
  {
    I_Printf("I_StartupSound: 8 bits not supported !\n");
    nosound = true;
    return false;
  }
  dev_bits = 8;

  if (want_stereo && mydev.channels != 2)
    I_Printf("I_StartupSound: stereo sound not available.\n");
  else if (!want_stereo && mydev.channels != 1)
    I_Printf("I_StartupSound: mono sound not available.\n");

  if (mydev.freq < (want_freq - want_freq/100) || 
      mydev.freq > (want_freq + want_freq/100))
  {
    I_Printf("I_StartupSound: %d Hz sound not available.\n", want_freq);
  }

  // display some useful stuff
  I_Printf("I_StartupSound: SDL Started: %d Hz, %d bits, %s\n",
      mydev.freq, dev_bits, (mydev.channels==2) ? "Stereo" : "Mono");

  dev_bytes_per_sample = (mydev.channels) * (dev_bits / 8);
  DEV_ASSERT2(dev_bytes_per_sample > 0);
  
  dev_frag_pairs = mydev.size / dev_bytes_per_sample;
  DEV_ASSERT2(dev_frag_pairs > 0);

  // allocate mixing buffers
  mix_buffer_L = (int *) malloc(dev_frag_pairs * sizeof(int));

  if (! mix_buffer_L)
  {
    SDL_CloseAudio();
    I_Error("I_StartupSound: Out of memory.\n");
    return false;
  }

  mix_buffer_R = (int *) malloc(dev_frag_pairs * sizeof(int));

  if (! mix_buffer_R)
  {
    SDL_CloseAudio();
    I_Error("I_StartupSound: Out of memory.\n");
    return false;
  }

  // okidoke, start things rolling
  SDL_PauseAudio(0);
  
  return true;
}

static fixed22_t ComputeDelta(int data_freq, int device_freq)
{
  // sound data's frequency close enough ?
  if (data_freq > device_freq - device_freq/100 &&
      data_freq < device_freq + device_freq/100)
  {
    return 1 << 10;
  }

  return floor((float_t)data_freq * 1024.0 / device_freq);
}

//
// I_ShutdownSound
//
void I_ShutdownSound(void)
{
  if (nosound)
    return;

  SDL_CloseAudio();

  if (mix_buffer_L)
  {
    free(mix_buffer_L);
    mix_buffer_L = NULL;
  }

  if (mix_buffer_R)
  {
    free(mix_buffer_R);
    mix_buffer_L = NULL;
  }

  // FIXME: free sounds
 
  nosound = true;
}

//
// I_LoadSfx
//
boolean_t I_LoadSfx(const unsigned char *data, unsigned int length,
    unsigned int freq, unsigned int handle)
{
  unsigned int i;
  mix_sound_t *sfx;

  SDL_LockAudio();

  if (handle >= stored_sfx_num)
  {
    i = stored_sfx_num;
    stored_sfx_num = handle + 1;

    stored_sfx = (mix_sound_t **) realloc(stored_sfx, stored_sfx_num *
        sizeof(mix_sound_t *));
    
    // FIXME: do the old-list new-list thing

    if (! stored_sfx)
    {
      SDL_UnlockAudio();
      I_Error("Out of memory in sound code.\n");
      return false;
    }
  
    // clear any new elements
    for (; i < stored_sfx_num; i++)
      stored_sfx[i] = NULL;
  }

  DEV_ASSERT2(stored_sfx[handle] == NULL);

  sfx = stored_sfx[handle] = (mix_sound_t *)
      malloc(sizeof(mix_sound_t));
  
  if (! sfx)
  {
    SDL_UnlockAudio();
    I_Error("Out of memory in sound code.\n");
    return false;
  }

  sfx->length = length;
  sfx->delta  = ComputeDelta(freq, mydev.freq);

  sfx->data_R = NULL;
  sfx->data_L = malloc(length);

  if (! sfx->data_L)
  {
    SDL_UnlockAudio();
    I_Error("Out of memory in sound code.\n");
    return false;
  }

  // convert to signed format
  for (i=0; i < length; i++)
  {
    sfx->data_L[i] = (char) (data[i] ^ 0x80);
  }

  SDL_UnlockAudio();
  return true;
}

//
// I_UnloadSfx
//
boolean_t I_UnloadSfx(unsigned int handle)
{
  int i;
  mix_sound_t *sfx;
  
  DEV_ASSERT(handle < stored_sfx_num,
      ("I_UnloadSfx: %d out of range", handle));
  
  sfx = stored_sfx[handle];

  DEV_ASSERT(sfx, ("I_UnloadSfx: NULL sample"));

  // note: assumes locking is recursive
  SDL_LockAudio();

  // Kill playing sound effects
  for (i = 0; i < MIX_CHANNELS; i++)
  {
    if (mix_chan[i].priority != PRI_NOSOUND && mix_chan[i].sound == sfx)
      I_SoundKill(i);
  }

  // free sound data
  free(sfx->data_L);
  stored_sfx[handle] = NULL;

  SDL_UnlockAudio();
  return true;
}

//
// I_SoundPlayback
//
int I_SoundPlayback(unsigned int soundid, int pan, int vol, boolean_t looping)
{
  int i;
  int lvol, rvol;

  mix_sound_t *sfx;

  if (nosound)
    return true;

  DEV_ASSERT(soundid > 0 && soundid < stored_sfx_num,
        ("I_SoundPlayback: Sound %d does not exist!", soundid));

  sfx = stored_sfx[soundid];

  SDL_LockAudio();

  for (i=0; i < MIX_CHANNELS; i++)
  {
    if (mix_chan[i].priority == PRI_NOSOUND)
      break;
  }

  if (i >= MIX_CHANNELS)
  {
    SDL_UnlockAudio();
    sprintf(errordesc, "I_SoundPlayback: Unable to allocate voice.");
    return -1;
  }

  // compute panning
  lvol = rvol = vol;
  
  if (mydev.channels == 2)
  {
    DEV_ASSERT2(0 <= pan && pan <= 255);

    // strictly linear equations
    lvol = (lvol * (255 - pan)) / 255;
    rvol = (rvol * (0   + pan)) / 255;
  }
 
  mix_chan[i].volume_L = lvol << (16 - dev_bits);
  mix_chan[i].volume_R = rvol << (16 - dev_bits);
  mix_chan[i].looping  = looping;
  mix_chan[i].sound    = sfx;
  mix_chan[i].paused   = 0;
  mix_chan[i].offset   = 0;
  mix_chan[i].priority = 1;  // set priority last
 
  SDL_UnlockAudio();
  return i;
}

//
// I_SoundKill
//
boolean_t I_SoundKill(unsigned int chanid)
{
  SDL_LockAudio();

  if (mix_chan[chanid].priority == PRI_NOSOUND)
  {
    SDL_UnlockAudio();
    sprintf(errordesc, "I_SoundKill: channel not playing.\n");
    return false;
  }
 
  mix_chan[chanid].priority = PRI_NOSOUND;
  mix_chan[chanid].volume_L = 0;
  mix_chan[chanid].volume_R = 0;
  mix_chan[chanid].looping  = false;
  mix_chan[chanid].sound    = NULL;
  mix_chan[chanid].paused   = 0;
  mix_chan[chanid].offset   = 0;

  SDL_UnlockAudio();
  return true;
}

//
// I_SoundCheck
//
boolean_t I_SoundCheck(unsigned int chanid)
{
  return (mix_chan[chanid].priority >= 0);
}

//
// I_SoundAlter
//
boolean_t I_SoundAlter(unsigned int chanid, int pan, int vol)
{
  int lvol, rvol;

  if (mix_chan[chanid].priority == PRI_NOSOUND)
  {
    sprintf(errordesc, "I_SoundAlter: channel not playing.\n");
    return false;
  }

  // compute panning
  lvol = rvol = vol;
  
  if (mydev.channels == 2)
  {
    DEV_ASSERT2(0 <= pan && pan <= 255);

    // strictly linear equations
    lvol = (lvol * (255 - pan)) / 255;
    rvol = (rvol * (0   + pan)) / 255;
  }
    
  mix_chan[chanid].volume_L = lvol << (16 - dev_bits);
  mix_chan[chanid].volume_R = rvol << (16 - dev_bits);

  return true;
}

//
// I_SoundPause
//
boolean_t I_SoundPause(unsigned int chanid)
{
  if (nosound)
    return true;

  if (mix_chan[chanid].priority == PRI_NOSOUND)
  {
    sprintf(errordesc, "I_SoundPause: channel not playing.\n");
    return false;
  }

  mix_chan[chanid].paused = 1;
  return true;
}

//
// I_SoundStopLooping
//
boolean_t I_SoundStopLooping(unsigned int chanid)
{
  if (nosound)
    return true;

  mix_chan[chanid].looping = false;
  return true;
}


//
// I_SoundResume
//
boolean_t I_SoundResume(unsigned int chanid)
{
  if (nosound)
    return true;

  if (mix_chan[chanid].priority == PRI_NOSOUND)
  {
    sprintf(errordesc, "I_SoundResume: channel not playing.\n");
    return false;
  }

  mix_chan[chanid].paused = 0;
  return true;
}

static void MixChannel(mix_channel_t *chan, int want)
{
  int i;
 
  int *dest_L = mix_buffer_L;
  int *dest_R = mix_buffer_R;

  char *src_L = chan->sound->data_L;

  while (want > 0)
  {
    int count = MIN(chan->sound->length - (chan->offset >> 10), want);

    DEV_ASSERT2(count > 0);

    for (i=0; i < count; i++)
    {
      char src_sample = src_L[chan->offset >> 10];
      
      *dest_L++ += src_sample * chan->volume_L;
      *dest_R++ += src_sample * chan->volume_R;

      chan->offset += chan->sound->delta;
    }

    want -= count;

    // return if sound hasn't finished yet
    if ((chan->offset >> 10) < chan->sound->length)
      return;
 
    if (! chan->looping)
    {
      chan->priority = PRI_FINISHED;
      return;
    }

    // loop back to beginning
    src_L = chan->sound->data_L;
    chan->offset = 0;
  }
}

static void MixMP3Buffer(byte *mp3_buf)
{
  int i;
  
  int *dest_L = mix_buffer_L;
  int *dest_R = mix_buffer_R;

  for (i=0; i < dev_frag_pairs; i++)
  {
    *dest_L++ += (char)(*mp3_buf++ ^ 0x80) * mp3_volume;
    *dest_R++ += (char)(*mp3_buf++ ^ 0x80) * mp3_volume;
  }
}

static INLINE unsigned char ClipSample(int value)
{
  if (value < 0) return 0;
  if (value > 0xFF) return 0xFF;

  return value;
}

static void BlitToSoundDevice(unsigned char *dest, int pairs)
{
  int i;

  DEV_ASSERT2(dev_bits == 8);

  if (mydev.channels == 2)
  {
    for (i=0; i < pairs; i++)
    {
      *dest++ = ClipSample((mix_buffer_L[i] >> 16) + 0x80);
      *dest++ = ClipSample((mix_buffer_R[i] >> 16) + 0x80);
    }
  }
  else
  {
    for (i=0; i < pairs; i++)
    {
      *dest++ = ClipSample((mix_buffer_L[i] >> 16) + 0x80);
    }
  }
}

//
// InternalSoundFiller
//
void InternalSoundFiller(void *udata, Uint8 *stream, int len)
{
  int i;
  int pairs = len / dev_bytes_per_sample;
  byte *mp3_buf;

  // check that we're not getting too much data
  DEV_ASSERT2(pairs <= dev_frag_pairs);

  if (nosound || pairs <= 0)
    return;
  
  // clear mixer buffer
  memset(mix_buffer_L, 0, sizeof(int) * pairs);
  memset(mix_buffer_R, 0, sizeof(int) * pairs);

  // add each channel
  for (i=0; i < MIX_CHANNELS; i++)
  {
    if (mix_chan[i].priority >= 0)
      MixChannel(&mix_chan[i], pairs);
  }

  // add MP3 chunk (if any)
  if (mp3_cur_track && !musicpaused)
  {
    mp3_buf = L_MP3ReadBuffer(dev_frag_pairs);

    if (mp3_buf)
    {
      MixMP3Buffer(mp3_buf);
      L_MP3ReadAdvance(dev_frag_pairs);
    }
  }

  BlitToSoundDevice(stream, pairs);
}

//
// I_SoundTicker
//
void I_SoundTicker(void)
{
  if (nosound)
    return;

  if (mp3_cur_track)
  {
    if (! L_MP3FillBuffer())
      L_MP3RestartMusicFile();
  }
}

//
// I_SoundReturnError
//
const char *I_SoundReturnError(void)
{
  memcpy(scratcherror, errordesc, sizeof(scratcherror));
  memset(errordesc, '\0', sizeof(errordesc));

  return scratcherror;
}


//  ======  MP3 STUFF  ======


//
// I_StartupMP3
//
boolean_t I_StartupMP3(void)
{
  if (nosound)
    return false;

  L_WriteDebug("I_StartupMP3: FREQ %d Hz  BITS %d\n", mydev.freq, dev_bits);
  
  // initialise MP3 system
  L_MP3Init(mydev.freq, dev_bits, mydev.channels);

  mp3_volume = 240 << (16 - dev_bits);
  mp3_cur_track = 0;
  mp3_inited = true;

  return true;
}

//
// I_ShutdownMP3
//
void I_ShutdownMP3(void)
{
  if (mp3_inited)
  {
    L_WriteDebug("I_ShutdownMP3: Shutting down\n");

    L_MP3Shutdown();
    mp3_inited = false;
  }
}

//
// I_MP3PlayTrack
//
int I_MP3PlayTrack(const char *filename, boolean_t looping)
{
  boolean_t result;

  if (mp3_cur_track)
  {
    L_MP3ClearMusicFile();
    mp3_cur_track = 0;
  }

  // FIXME: looping not honoured yet

  result = L_MP3SetMusicFile(filename, &mp3_rate);

  L_WriteDebug("I_MP3StartPlayback: %s\n", result ? "OK" : "Failed");
 
  if (result)
  {
    mp3_cur_track = 1;
    return 1;
  }
  
  return -1;
}

//
// I_MP3StopTrack
//
void I_MP3StopTrack(int track)
{
  L_WriteDebug("I_MP3StopPlayback called\n");

  if (track == mp3_cur_track)
  {
    L_MP3ClearMusicFile();
    mp3_cur_track = 0;
  }
}

//
// I_MP3SetVolume
//
void I_MP3SetVolume(int vol)
{
  if (vol < 0 || vol > 15)
    return;
   
  L_WriteDebug("I_MP3SetVolume: changing volume to %d...\n", vol);

  mp3_volume = (vol * 17) << (16 - dev_bits);
}
