//----------------------------------------------------------------------------
//  EDGE Linux GGI Video Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2000  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------

#include "i_defs.h"
#include "./i_trans.h"

#include "version.h"
#include "dm_state.h"
#include "dm_defs.h"
#include "dm_type.h"
#include "m_argv.h"
#include "st_stuff.h"
#include "v_video1.h"
#include "v_video2.h"
#include "v_res.h"
#include "v_colour.h"
#include "v_screen.h"

#include "e_main.h"
#include "e_event.h"
#include "z_zone.h"
#include "w_wad.h"

#include "s_sound.h"


#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include <stdarg.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <signal.h>

#include <ggi/ggi.h>

#ifndef DISABLE_WMH
#include <ggi/wmh.h>
#endif

short hicolortransmask1, hicolortransmask2;

// Dummy screen... 
// mainscreen is a full-size subscreen of this one.
static screen_t dummy_screen;

unsigned char *thepalette;

static int graphics_shutdown = 0;

static ggi_visual_t my_vis;
static ggi_mode my_mode;

static char *my_targetname = NULL;
static char *my_modename = "";

static const ggi_directbuffer *my_dbuf;
static const ggi_pixelformat *my_pixfmt;
static boolean_t my_fmt_ok;

static ggi_color my_pal[256];

static colourshift_t redshift, greenshift, blueshift;

static int scaling;

// Possible Screen Modes
typedef struct
{
  int width;
  int height;
  int bpp;
}
possresmode_t;

static possresmode_t possresmode[] =
{
  { 320, 200, 8},
  { 320, 240, 8},
  { 400, 300, 8},
  { 512, 384, 8},
  { 640, 400, 8},
  { 640, 480, 8},
  { 800, 600, 8},
  {1024, 768, 8},
  {1280,1024, 8},

  { 320, 200, 16},
  { 320, 240, 16},
  { 400, 300, 16},
  { 512, 384, 16},
  { 640, 400, 16},
  { 640, 480, 16},
  { 800, 600, 16},
  {1024, 768, 16},
  {1280,1024, 16},

  {  -1,  -1, -1}
};


// ====================== INTERNALS ======================

//
// MakeCol
//
static inline unsigned long MakeCol(long r, long g, long b)
{
  r = (r >> (8 - redshift.bits))   << redshift.shift;
  g = (g >> (8 - greenshift.bits)) << greenshift.shift;
  b = (b >> (8 - blueshift.bits))  << blueshift.shift;

  return r | g | b;
}

//
// CountLeadingZeros
//
static inline unsigned long CountLeadingZeros(unsigned long x)
{
  unsigned long r = 0;

  if (x == 0)
    return 0;

  while (!(x & 1))
  {
    x >>= 1;
    r++;
  }

  return r;
}

//
// CountBitsInMask
//
static inline unsigned long CountBitsInMask(unsigned long x)
{
  unsigned long r = 0;

  while (!(x & 1))
    x >>= 1;

  while (x & 1)
  {
    x >>= 1;
    r++;
  }

  if (x)
    I_Error ("GGI Version: Mask is not contiguous.\n");

  return r;
}

//
// SetColourShift
//
// Constructs the shifts necessary to set one element of an RGB colour value
//
void SetColourShift(unsigned long mask, colourshift_t * ps)
{
  ps->mask  = mask;
  ps->shift = CountLeadingZeros(mask);
  ps->bits  = CountBitsInMask(mask);
}

//
// FastBlitToScreen
//
// Uses DirectBuffer to directly copy to the screen
//
static inline void FastBlitToScreen(void)
{
  register int y;
  register unsigned char *s1, *d1;
  register unsigned int stride = my_dbuf->buffer.plb.stride;

  s1 = (void *) dummy_screen.data;
  d1 = (void *) my_dbuf->write;

  for (y=0; y < SCREENHEIGHT; y++)
  {
    memcpy(d1, s1, SCREENWIDTH * BPP);
    
    s1 += SCREENDEPTH;
    d1 += stride;
  }
}

//
// SlowBlitToScreen
//
// Uses the ggiPutHLine method to copy to the screen
//
static inline void SlowBlitToScreen(void)
{
  int x, y, w, i, dx, dy;

  unsigned char *s1;
  unsigned char buffer[64 * 4 * sizeof(ggi_pixel)];

  s1 = (void *) dummy_screen.data;

  if (scaling == 1)
  {
    for (y=0; y < SCREENHEIGHT; y++, s1 += SCREENDEPTH)
      ggiPutHLine(my_vis, 0, y, SCREENWIDTH, s1);

    return;
  }

  for (y=0; y < SCREENHEIGHT; y++, s1 += SCREENDEPTH)
  for (x=0; x < SCREENWIDTH; x += 64)
  {
    w = MIN(64, SCREENWIDTH - x);

    DEV_ASSERT2(w > 0);
    
    if (BPP == 1)
    {
      unsigned char *dest = buffer;

      for (i=0; i < w; i++)
      for (dx=0; dx < scaling; dx++)
      {
        *dest++ = s1[x + i];
      }
    }
    else /* BPP == 2 */
    {
      unsigned short *dest = (unsigned short *)buffer;

      for (i=0; i < w; i++)
      for (dx=0; dx < scaling; dx++)
      {
        *dest++ = ((short *)s1)[x + i];
      }
    }

    for (dy=0; dy < scaling; dy++)
      ggiPutHLine(my_vis, x * scaling, y * scaling + dy, w * scaling, buffer);
  }
}

//
// SnailBlitToScreen
//
// Very slow fallback method, when nothing else is possible
//
static inline void SnailBlitToScreen(void)
{
  int i, w, x, y;
  unsigned char *s1;

  ggi_color col_buf[64];
  unsigned char buffer[64 * sizeof(ggi_pixel)];

  s1 = (void *) dummy_screen.data;

  for (y=0; y < SCREENHEIGHT; y++, s1 += SCREENDEPTH)
  for (x=0; x < SCREENWIDTH; x += 64)
  {
    w = MIN(64, SCREENWIDTH - x);

    DEV_ASSERT2(w > 0);
    
    for (i=0; i < w; i++)
    {
      unsigned short pix = ((unsigned short *)s1)[x + i];

      col_buf[i].r = (int)((pix >> 10) & 0x1F) * 0xFFFF / 0x1F;
      col_buf[i].g = (int)((pix >>  5) & 0x1F) * 0xFFFF / 0x1F;
      col_buf[i].b = (int)((pix)       & 0x1F) * 0xFFFF / 0x1F;
    }

    ggiPackColors(my_vis, buffer, col_buf, w);

    ggiPutHLine(my_vis, x, y, w, buffer);
  }
}

//
// DetectAvailableModes
//
static void DetectAvailableModes(void)
{
  int i;
  int resadded = 0;

  int snail = M_CheckParm("-ggisnail");

  // -ACB- 2000/03/16 Detect Possible Resolutions
  for (i=0; possresmode[i].width != -1; i++)
  {
    ggi_mode template_mode;
    ggi_mode test_mode;

    // option to allow any mode (no matter how slow)
    if (snail > 0)
    {
      V_AddAvailableResolution(possresmode[i].width / scaling, 
          possresmode[i].height / scaling, possresmode[i].bpp);
      resadded++;
      continue;
    }

    ggiParseMode("", &template_mode);

    GT_SETSIZE(template_mode.graphtype, possresmode[i].bpp);

    template_mode.visible.x = possresmode[i].width;
    template_mode.visible.y = possresmode[i].height;

    test_mode = template_mode;
    test_mode.virt.x = V_GetDepth(possresmode[i].width, 
        possresmode[i].bpp/8) / (possresmode[i].bpp/8);

    if (ggiCheckMode(my_vis, &test_mode) >= 0)
    {
      V_AddAvailableResolution(possresmode[i].width / scaling, 
          possresmode[i].height / scaling, possresmode[i].bpp);
      resadded++;
      continue;
    }

    // -AJA- We don't REQUIRE the computed virt.x to be available,
    //       that is just the optimal value to prevent cache thrashing
    //       in the column drawers.  So test again, allowing virt.x to
    //       be decided by the hardware.

    test_mode = template_mode;
    
    if (ggiCheckMode(my_vis, &test_mode) >= 0)
    {
      V_AddAvailableResolution(possresmode[i].width / scaling, 
          possresmode[i].height / scaling, possresmode[i].bpp);
      resadded++;
      continue;
    }
  }

  if (resadded == 0)
    I_Error("I_StartupGraphics: Found NO supported resolutions !\n");
}


// =================== END OF INTERNALS ===================

//
// I_StartupGraphics
//
void I_StartupGraphics(void)
{
  const char *val_str;

  scaling = 1;

  // check for command-line options
  val_str = M_GetParm("-ggitarget");
  if (val_str)
    my_targetname = strdup(val_str);
  else if (getenv("EDGE_GGI_TARGET"))
    my_targetname = getenv("EDGE_GGI_TARGET");

  val_str = M_GetParm("-ggimode");
  if (val_str)
    my_modename = strdup(val_str);
  else if (getenv("EDGE_GGI_MODE"))
    my_modename = getenv("EDGE_GGI_MODE");

  if (M_CheckParm("-2"))
    scaling = 2;
  else if (M_CheckParm("-3"))
    scaling = 3;
  else if (M_CheckParm("-4"))
    scaling = 4;

  if (ggiInit() < 0)
    I_Error("Couldn't init LibGGI !\n");

  my_vis = ggiOpen(my_targetname, NULL);

  if (my_vis == NULL)
    I_Error("Couldn't open LibGGI target !\n");

  ggiSetFlags(my_vis, GGIFLAG_ASYNC);

  DetectAvailableModes();

#ifndef DISABLE_WMH
  if (ggiWmhInit() < 0)
    I_Error("Couldn't init LibWMH !\n");
#endif

#ifdef DEVELOPERS
  // get round LibGGI's signal handlers
  signal(SIGFPE,  SIG_DFL);
  signal(SIGSEGV, SIG_DFL);
#endif
}


//
// I_GetTruecolInfo
//
void I_GetTruecolInfo(truecol_info_t * info)
{
  info->red_bits = redshift.bits;
  info->red_shift = redshift.shift;
  info->red_mask = redshift.mask;

  info->green_bits = greenshift.bits;
  info->green_shift = greenshift.shift;
  info->green_mask = greenshift.mask;

  info->blue_bits = blueshift.bits;
  info->blue_shift = blueshift.shift;
  info->blue_mask = blueshift.mask;

  info->grey_mask = 0x7FFFFFFF;

  // check for RGB 5:6:5 mode (should be more general !)
  if (info->red_bits == 5 && info->green_bits == 6 &&
      info->blue_bits == 5)
  {
    info->grey_mask = 0xFFDF;
  }
}

//
// I_SetScreenSize
//
boolean_t I_SetScreenSize(int width, int height, int bpp)
{
  char title[80];

  ggiParseMode("", &my_mode);

  GT_SETSIZE(my_mode.graphtype, BPP * 8);

  my_mode.visible.x = SCREENWIDTH  * scaling;
  my_mode.visible.y = SCREENHEIGHT * scaling;
  my_mode.virt.x = V_GetDepth(SCREENWIDTH, BPP) / BPP;

  if (ggiSetMode(my_vis, &my_mode) < 0)
  {
    I_Warning("Requested mode not available !\n");

    if (ggiSetMode(my_vis, &my_mode) < 0)
    {
      I_Warning("ARGH ! Mode still not available (bug in LibGGI)\n");

      ggiParseMode("", &my_mode);

      if (ggiSetMode(my_vis, &my_mode) < 0)
      {
        I_Printf("Holy crap !!  LibGGI has suffered a meltdown\n");
        return false;
      }
    }
  }

  BPP = (GT_SIZE(my_mode.graphtype) + 7) / 8;
  SCREENWIDTH = my_mode.visible.x  / scaling;
  SCREENHEIGHT = my_mode.visible.y / scaling;

  my_pixfmt = ggiGetPixelFormat(my_vis);
  my_fmt_ok = true;

  // read directbuffer info

  my_dbuf = ggiDBGetBuffer(my_vis, 0);

  // FIXME: support scaling with direct buffer
  if (scaling > 1)
  {
    my_dbuf = NULL;
  }
  else if (! my_dbuf)
  {
    I_Printf("I_SetScreenSize: No DirectBuffer available.\n");
  }
  else
  {
    if (! (my_dbuf->type & GGI_DB_SIMPLE_PLB)) 
    {
      I_Printf("I_SetScreenSize: Non-standard DirectBuffer.\n");
      my_dbuf = NULL;
    }
  }

  // check pixel format

  if (GT_SIZE(my_mode.graphtype) != (BPP * 8))
  {
    I_Printf("I_SetScreenSize: Non-standard pixel size.\n");
    my_fmt_ok = false;
  }

  if (BPP == 1)
  {
    if (GT_SCHEME(my_mode.graphtype) != GT_PALETTE)
    {
      I_Printf("I_SetScreenSize: Mode has no palette.\n");
      my_fmt_ok = false;
    }
  }
  else if (BPP == 2)
  {
    if (GT_SCHEME(my_mode.graphtype) != GT_TRUECOLOR)
    {
      I_Printf("I_SetScreenSize: Mode is not truecolor.\n");
      my_fmt_ok = false;
    }

    SetColourShift(my_pixfmt->red_mask,   &redshift);
    SetColourShift(my_pixfmt->green_mask, &greenshift);
    SetColourShift(my_pixfmt->blue_mask,  &blueshift);
  }
  else
  {
    I_Printf("I_SetScreenSize: 24/32 bit modes not supported.\n");
    my_fmt_ok = false;
  }

  if (! my_fmt_ok)
  {
    // FIXME: support scaling in snail mode
    scaling = 1;

    // Unknown format, don't continue unless we are in "snail" mode
    if (M_CheckParm("-ggisnail") <= 0)
      return false;

    // Force the main code to RGB 5:5:5 format, which we'll use to
    // convert the colors.
    BPP = 2;

    SetColourShift(0x1F << 10, &redshift);
    SetColourShift(0x1F <<  5, &greenshift);
    SetColourShift(0x1F,       &blueshift);

    my_dbuf = NULL;
  }

#ifdef DEVELOPERS
  // get round LibGGI's signal handlers
  signal(SIGFPE,  SIG_DFL);
  signal(SIGSEGV, SIG_DFL);
#endif

  SCREENDEPTH = V_GetDepth(my_mode.virt.x / scaling, BPP);

  dummy_screen.width = SCREENWIDTH;
  dummy_screen.height = SCREENHEIGHT;
  dummy_screen.depth = SCREENDEPTH;
  dummy_screen.bpp = BPP;
  dummy_screen.parent = NULL;
  dummy_screen.data = malloc(SCREENDEPTH * SCREENHEIGHT);

  main_scr = V_CreateSubScreen(&dummy_screen, 0, 0, SCREENWIDTH, SCREENHEIGHT);

#ifndef DISABLE_WMH
  if (ggiWmhAttach(my_vis) < 0)
    I_Error("Unable to attach WMH extension to LibGGI visual !\n");

  sprintf(title, "EDGE v" EDGEVERSTR);

  ggiWmhSetTitle(my_vis, title);
#endif

  return true;
}

//
// I_StartFrame
//
void I_StartFrame(void)
{
  if (my_dbuf && scaling == 1)
  {
    // Acquire DirectBuffer before we use it.
    if (ggiResourceAcquire(my_dbuf->resource, GGI_ACTYPE_WRITE) != 0)
      I_Error("Error acquiring DirectBuffer\n");
  }
}

//
// I_FinishFrame
//
void I_FinishFrame(void)
{
  if (my_dbuf && scaling == 1)
  {
    FastBlitToScreen();

    // Release DirectBuffer when done with it.
    ggiResourceRelease(my_dbuf->resource);
  }
  else if (my_fmt_ok)
  {
      SlowBlitToScreen();
  }
  else
  {
      SnailBlitToScreen();
  }
  
  ggiFlush(my_vis);
}

//
// I_SetPalette
//
void I_SetPalette(byte palette[256][3])
{
  int i;

  const byte *gtable = gammatable[usegamma];

  if (BPP != 1)
    return;

  for (i = 0; i < 256; i++)
  {
    int r = gtable[palette[i][0]];
    int g = gtable[palette[i][1]];
    int b = gtable[palette[i][2]];

    my_pal[i].r = (r << 8) + r;
    my_pal[i].g = (g << 8) + g;
    my_pal[i].b = (b << 8) + b;
  }

  ggiSetPalette(my_vis, 0, 256, my_pal);
}

//
// I_Colour2Pixel
//
long I_Colour2Pixel(byte palette[256][3], int col)
{
  if (BPP == 1)
    return col;

  return MakeCol(palette[col][0], palette[col][1], palette[col][2]);
}

//
// I_ShutdownGraphics
//
void I_ShutdownGraphics(void)
{
  if (graphics_shutdown)
    return;

  graphics_shutdown = 1;

  fprintf(stderr, "Shutting down graphics...\n");

#ifndef DISABLE_WMH
  ggiWmhDetach(my_vis);
  ggiWmhExit();
#endif

  ggiClose(my_vis);
  ggiExit();
}

//
//   INPUT CODE 
//
#define GGI_EVENTS  (emKey | emPointer)

//
// Translates a key from GGI -> EDGE
//
// Returns -1 if no suitable translation exists.

static int TranslateGGIKey(gii_event *ev)
{
  int label = ev->key.label;

  if (GII_KTYP(label) == GII_KT_PAD)
    label = ev->key.sym;

  switch (label) 
  {
    case GIIUC_Tab: return KEYD_TAB;
    case GIIUC_Linefeed:
    case GIIUC_Return: return KEYD_ENTER;
    case GIIUC_Escape: return KEYD_ESCAPE;
    case GIIUC_Delete:
    case GIIUC_BackSpace: return KEYD_BACKSPACE;

    case GIIK_Up:    return KEYD_UPARROW;
    case GIIK_Down:  return KEYD_DOWNARROW;
    case GIIK_Left:  return KEYD_LEFTARROW;
    case GIIK_Right: return KEYD_RIGHTARROW;

    case GIIK_Find:   return KEYD_HOME;
    case GIIK_Select: return KEYD_END;
    case GIIK_ScrollBack:
    case GIIK_PageUp:   return KEYD_PGUP;
    case GIIK_ScrollForw:
    case GIIK_PageDown:   return KEYD_PGDN;
    case GIIK_Insert: return KEYD_INSERT;
    case GIIK_Pause: return KEYD_PAUSE;

    case GIIK_F1:  return KEYD_F1;
    case GIIK_F2:  return KEYD_F2;
    case GIIK_F3:  return KEYD_F3;
    case GIIK_F4:  return KEYD_F4;
    case GIIK_F5:  return KEYD_F5;
    case GIIK_F6:  return KEYD_F6;
    case GIIK_F7:  return KEYD_F7;
    case GIIK_F8:  return KEYD_F8;
    case GIIK_F9:  return KEYD_F9;
    case GIIK_F10: return KEYD_F10;
    case GIIK_F11: return KEYD_F11;
    case GIIK_F12: return KEYD_F12;

    case GIIK_PPlus:  return '+';
    case GIIK_PMinus: return '-';
    case GIIK_PDecimal: return '.';
    case GIIK_PEnter: return KEYD_ENTER;
    case GIIK_PSlash: return '/';
    case GIIK_PStar:  return '*';
    case GIIK_PEqual: return '=';

    case GIIK_NumLock:    return KEYD_NUMLOCK;
    case GIIK_ScrollLock: return KEYD_SCRLOCK;
    case GIIK_CapsLock:   return KEYD_CAPSLOCK;

    case GIIK_ShiftL:
    case GIIK_ShiftR: return KEYD_RSHIFT;
    case GIIK_CtrlL:
    case GIIK_CtrlR:  return KEYD_RCTRL;
    case GIIK_MetaL:
    case GIIK_AltL:   return KEYD_LALT;
    case GIIK_MetaR:
    case GIIK_AltR:   return KEYD_RALT;
  }

  if (GII_KTYP(label) == GII_KT_LATIN1)
  {
    label = GII_KVAL(label);

    if (label <= 0x7F)
      return tolower(label);
  }

  return -1;
}

static void HandleKeyEvent(gii_event * ev)
{
  event_t event;

  if (ev->any.type != evKeyPress && ev->any.type != evKeyRelease) 
    return;

  event.type  = (ev->any.type == evKeyPress) ? ev_keydown : ev_keyup;
  event.value.key = TranslateGGIKey(ev);

  if (event.value.key < 0)
    return;
  
  E_PostEvent(&event);
}

static void HandleMouseEvent(gii_event * ev)
{
  event_t event;

  if (ev->any.type == evPtrRelative)
  {
    event.type = ev_analogue;

    if (ev->pmove.x)
    {
      event.value.analogue.axis = mouse_xaxis;
      event.value.analogue.amount = ev->pmove.x * mouseSensitivity;
      E_PostEvent(&event);
    }
    
    if (ev->pmove.y)
    {
      event.value.analogue.axis = mouse_yaxis;
      event.value.analogue.amount = ev->pmove.y * mouseSensitivity;

      if (invertmouse)
        event.value.analogue.amount = -event.value.analogue.amount;

      E_PostEvent(&event);
    }

    // handle the mouse wheel
    if (ev->pmove.wheel == 0)
      return;

    event.type = ev_keydown;
    event.value.key = (ev->pmove.wheel < 0) ? KEYD_MWHEEL_UP : KEYD_MWHEEL_DN;
    
    E_PostEvent(&event);

    event.type = ev_keyup;
    event.value.key = (ev->pmove.wheel < 0) ? KEYD_MWHEEL_UP : KEYD_MWHEEL_DN;
    
    E_PostEvent(&event);
    return;
  }

  if (ev->any.type == evPtrButtonPress) 
    event.type = ev_keydown;
  else if (ev->any.type == evPtrButtonRelease) 
    event.type = ev_keyup;
  else 
    return;

  switch (ev->pbutton.button)
  {
    case GII_PBUTTON_PRIMARY:
      event.value.key = KEYD_MOUSE1; break;
      
    case GII_PBUTTON_SECONDARY:
      event.value.key = KEYD_MOUSE2; break;
      
    case GII_PBUTTON_TERTIARY:
      event.value.key = KEYD_MOUSE3; break;

    default:
      return;
  }

  E_PostEvent(&event);
}

//
// I_ControlGetEvents
//
void I_ControlGetEvents (void)
{
  gii_event ev;

  for (;;) 
  {
    struct timeval tv = {0, 0};

    if (ggiEventPoll(my_vis, GGI_EVENTS, &tv) == 0)
      break;

    ggiEventRead(my_vis, &ev, GGI_EVENTS);

    HandleKeyEvent(&ev);
    HandleMouseEvent(&ev);
  }
}

//
// I_EDGELoop
//
void I_EDGELoop(void)
{
  while (1)
    E_EDGELoopRoutine();
}
