//----------------------------------------------------------------------------
//  EDGE Linux X-Windows Video code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2000  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------

#include "i_defs.h"
#include "./i_trans.h"

#include "dm_state.h"
#include "dm_defs.h"
#include "dm_type.h"
#include "m_argv.h"
#include "m_menu.h"
#include "st_stuff.h"
#include "v_video1.h"
#include "v_video2.h"
#include "v_res.h"
#include "v_colour.h"
#include "v_screen.h"

#include "e_main.h"
#include "e_event.h"
#include "w_wad.h"
#include "z_zone.h"

#include "s_sound.h"

#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include <netinet/in.h>
#include <signal.h>

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xutil.h>

#include <X11/extensions/XShm.h>
// Had to dig up XShm.c for this one.
// It is in the libXext, but not in the XFree86 headers.
int XShmGetEventBase(Display * dpy);  // problems with g++?
int XShmQueryExtension(Display * dpy);  // CP added


#define DEBUG  0

// POLL_POINTER means do a XQueryPointer once pre frame, to measure
// mouse motion. The alternative is an XGrabPointer which instructs
// XFree86 to send events when the pointer moves - however X seems to
// buffer these events, making it unusable

#define POLL_POINTER  1

// AGGR_FOCUS is short for 'aggressive focusing'. That means that
// whenever the mouse is grabbed, grab the focus too. The alternative
// is passive focusing, which queries if we have the input focus and
// sets the mouse grab state to match.
//
// AGGR_FOCUS is more likely to be asynchronous, but could make it
// harder to get focus away from lxedge in some window managers.

#define AGGR_FOCUS  0


static void I_AutodetectBPP(void);


short hicolortransmask1, hicolortransmask2;

// Dummy screen... 
// mainscreen is a full-size subscreen of this one.
static screen_t dummy_screen;


static boolean_t in_game_now = false;

unsigned char *thepalette;

static int graphics_shutdown = 0;


static Display *X_display = NULL;
static int X_screen = -1;
static Window X_mainWindow;
static Colormap X_cmap;
static Visual *X_visual;
static GC X_gc;
static XEvent X_event;
static XImage *X_image;
static Atom X_deletewin;

#define INITIALISED  (X_display != NULL)

// MIT SHared Memory extension.
static boolean_t doShm;
static XShmSegmentInfo X_shminfo;
static int X_shmeventtype;

// array of XColor structs used for setting the 256-colour palette 
// in 256 colour mode.
static XColor *colours = NULL;

colourshift_t redshift, greenshift, blueshift;

// Common const strings
static const char *lcase_lxedge = "lxedge";
static const char *ucase_lxedge = "LXEDGE";


// input related vars

static event_t event;  // CPhipps - static because several subs need it 
                       // so save having several scattered around

static boolean_t shmFinished = true;  // Are we awaiting an MITShm completion?

static int vis_flag = VisibilityUnobscured;  // Is lxedge visible?

// Mouse handling
extern int usemouse;  // config file var

static boolean_t grabMouse;  // internal var

static boolean_t grabbed = false;  // Is the mouse currently grabbed

// Little struct to keep things together
typedef struct
  {
    int x, y;
  }
mouse_t;

// Mouse status last time & this time (respectively)
static mouse_t lastmouse, newmouse;

#if (! POLL_POINTER)
static int buttons;
#endif


static void I_GetEvent(void);

static boolean_t QueryPointer(mouse_t * p, unsigned int *pmask);
static void VerifyPointerGrabState(void);
static void InitInputs(void);


static inline unsigned long MakeCol(long r, long g, long b)
{
  r = (r >> (8 - redshift.bits))   << redshift.shift;
  g = (g >> (8 - greenshift.bits)) << greenshift.shift;
  b = (b >> (8 - blueshift.bits))  << blueshift.shift;

  return r | g | b;
}

static inline unsigned long CountLeadingZeros(unsigned long x)
{
  unsigned long r = 0;

  if (x == 0)
    return 0;

  while (!(x & 1))
    {
      x >>= 1;
      r++;
    }
  return r;
}

static inline unsigned long CountBitsInMask(unsigned long x)
{
  unsigned long r = 0;

  while (!(x & 1))
    x >>= 1;

  while (x & 1)
    {
      x >>= 1;
      r++;
    }

  if (x)
    I_Error ("X Version: Mask is not contiguous.\n");

  return r;
}


//
// SetColourShift
//
// Constructs the shifts necessary to set one element of an RGB colour
// value.

void SetColourShift(unsigned long mask, colourshift_t *c)
{
  c->mask  = mask;
  c->shift = CountLeadingZeros(mask);
  c->bits  = CountBitsInMask(mask);
}

//
// I_StartupGraphics
//
void I_StartupGraphics (void)
{
  char *displayname = NULL;
  const char *val_str;

  // check for command-line display name
  if ((val_str = M_GetParm("-disp")) ||
      (val_str = M_GetParm("-display")) ||
      (val_str = M_GetParm("-displayname")))
  {
    displayname = strdup(val_str);
  }
  else if (getenv("DISPLAY"))
  {
    displayname = getenv("DISPLAY");
  }

  // open the display
  X_display = XOpenDisplay(displayname);

  if (!X_display)
  {
    if (displayname)
      I_Error("Could not open display: %s\n", displayname);
    else
      I_Error("Could not open default display.\n");
  }

  // use the default visual 
  X_screen = DefaultScreen(X_display);

  // -AJA- What the fuck is going on here ?
  X_deletewin = XInternAtom(X_display, "WM_DELETE_WINDOW", False);

  // check for the MITSHM extension.
  // even if it's available, make sure it's a local connection
  doShm = XShmQueryExtension(X_display);

  I_Printf("X-version: %sMITShm extension detected.\n",
      doShm ? "" : "no ");

  I_AutodetectBPP();
}

static void CreateWindow(void)
{
  // setup attributes for main window
  unsigned long attribmask;
  XSetWindowAttributes attribs;

  attribmask = CWEventMask | CWColormap | CWBorderPixel;
  attribs.event_mask = KeyPressMask | KeyReleaseMask | ExposureMask;

  attribs.colormap = X_cmap;
  attribs.border_pixel = 0;

  // create the main window
  X_mainWindow = XCreateWindow(X_display, RootWindow(X_display, X_screen),
      0, 0, SCREENWIDTH, SCREENHEIGHT, 0, BPP * 8,
      InputOutput, X_visual, attribmask, &attribs);
}

static void SetHints(void)
{
  XClassHint c_hint;
  XTextProperty x_name;
  XSizeHints s_hint;
  XWMHints wm_hint;
  char lcase_buf[10], ucase_buf[10];
  char *str;
  
  int myargc;
  const char **myargv;

  strcpy (str = lcase_buf, lcase_lxedge);
  XStringListToTextProperty (&str, 1, &x_name);

  s_hint.flags = PSize | PMinSize | PMaxSize;
  s_hint.min_width = s_hint.max_width = s_hint.base_width = SCREENWIDTH;
  s_hint.min_height = s_hint.max_height = s_hint.base_height = SCREENHEIGHT;

  wm_hint.input = True;
  wm_hint.window_group = X_mainWindow;
  wm_hint.flags = InputHint | WindowGroupHint;

  strcpy (c_hint.res_name = lcase_buf, lcase_lxedge);
  strcpy (c_hint.res_class = ucase_buf, ucase_lxedge);

  XSetWMProtocols (X_display, X_mainWindow, &X_deletewin, 1);

  myargv = M_GetArguments(&myargc);

  // Intentional Const Override
  XSetWMProperties (X_display, X_mainWindow, &x_name, &x_name,
                    (char **) myargv, myargc, &s_hint, &wm_hint, &c_hint);

  XFlush (X_display);
}

/////////////////////////////////////////////////////////////////////////
// Internal helper functions

static Cursor 
I_XCreateNullCursor (Display * display, Window root)
{
  Pixmap cursormask;
  XGCValues xgc;
  GC gc;
  XColor dummycolour;
  Cursor cursor;

  cursormask = XCreatePixmap (display, root, 1, 1, 1);
  // NOTE: depth in the above call is 1 because the image is monochrone, 
  // i.e a mask
  xgc.function = GXclear;
  gc = XCreateGC (display, cursormask, GCFunction, &xgc);
  XFillRectangle (display, cursormask, gc, 0, 0, 1, 1);
  dummycolour.pixel = 0;
  dummycolour.red = 0;
  dummycolour.flags = 04;
  cursor = XCreatePixmapCursor (display, cursormask, cursormask,
                                &dummycolour, &dummycolour, 0, 0);
  XFreePixmap (display, cursormask);
  XFreeGC (display, gc);
  return cursor;
}


static void SetContext(Void)
{
  XGCValues xgcvalues;
  int valuemask;

  // Hide pointer while over this window
  XDefineCursor (X_display, X_mainWindow,
                 I_XCreateNullCursor (X_display, X_mainWindow));

  // create the GC
  valuemask = GCGraphicsExposures;
  xgcvalues.graphics_exposures = False;
  X_gc = XCreateGC (X_display, X_mainWindow, valuemask, &xgcvalues);

  // name the window
  XStoreName (X_display, X_mainWindow, "lxedge");

  // map the window
  XMapWindow (X_display, X_mainWindow);

  // wait until it is OK to draw
  do
    {
      XNextEvent (X_display, &X_event);
    }
  while (!(X_event.type == Expose && !X_event.xexpose.count));

}


static void 
I_PrintMode (const char *s, boolean_t shm)
{
  fprintf (stderr, "%s %d BPP %s %s\n", s, BPP,
           (BPP != 1) ? "TrueColor" : "Palettised",
           shm ? " via MITShm" : "");
}

static Visual *
I_FindMode (int screen, boolean_t shm)
{
  XVisualInfo X_visualinfo;

  fprintf (stderr, "I_FindMode: ");

  if (!XMatchVisualInfo (X_display, screen, BPP * 8,
                         (BPP == 1) ? PseudoColor : TrueColor,
                         &X_visualinfo))
  {
    I_Error ("Unable to find supported visual settings");
  }

  // set up colour shifts
  if (BPP != 1)
  {
    SetColourShift(X_visualinfo.red_mask,   &redshift);
    SetColourShift(X_visualinfo.green_mask, &greenshift);
    SetColourShift(X_visualinfo.blue_mask,  &blueshift);
  }

  I_PrintMode ("using", shm);

  return X_visualinfo.visual;
}


//
// This function is probably redundant,
//  if XShmDetach works properly.
// ddt never detached the XShm memory,
//  thus there might have been stale
//  handles accumulating.
//
static void XShmGrabSharedMemory (size_t size)
{
  int key = ('d' << 24) | ('o' << 16) | ('o' << 8) | 'm';
  struct shmid_ds shminfo;
  int minsize = 320 * 200;
  int id;
  int rc;
  // UNUSED int done=0;
  int pollution = 5;

  // try to use what was here before
  do
    {
      id = shmget ((key_t) key, minsize, 0777);  // just get the id

      if (id != -1)
        {
          rc = shmctl (id, IPC_STAT, &shminfo);  // get stats on it

          if (!rc)
            {
              if (shminfo.shm_nattch)
                {
                  fprintf (stderr, "User %d appears to be running "
                           "DOOM.  Is that wise?\n", shminfo.shm_cpid);
                  key++;
                }
              else
                {
                  if (getuid () == shminfo.shm_perm.cuid)
                    {
                      rc = shmctl (id, IPC_RMID, 0);
                      if (!rc)
                        fprintf (stderr,
                                 "Was able to kill my old shared memory\n");
                      else
                        I_Error ("Was NOT able to kill my old shared memory");

                      id = shmget ((key_t) key, size, IPC_CREAT | 0777);
                      if (id == -1)
                        I_Error ("Could not get shared memory");

                      rc = shmctl (id, IPC_STAT, &shminfo);

                      break;

                    }
                  if (size >= shminfo.shm_segsz)
                    {
                      fprintf (stderr,
                               "will use %d's stale shared memory\n",
                               shminfo.shm_cpid);
                      break;
                    }
                  else
                    {
                      fprintf (stderr,
                               "warning: can't use stale "
                               "shared memory belonging to id %d, "
                               "key=0x%x\n",
                               shminfo.shm_cpid, key);
                      key++;
                    }
                }
            }
          else
            {
              I_Error ("could not get stats on key=%d", key);
            }
        }
      else
        {
          id = shmget ((key_t) key, size, IPC_CREAT | 0777);
          if (id == -1)
            {
              extern int errno;
              fprintf (stderr, "errno=%d\n", errno);
              I_Error ("Could not get any shared memory");
            }
          break;
        }
    }
  while (--pollution);

  if (!pollution)
  {
    I_Error ("Sorry, system too polluted with stale shared memory segments.\n");
  }

  X_shminfo.shmid = id;

  // attach to the shared memory segment
  X_image->data = X_shminfo.shmaddr = shmat (id, 0, 0);

// fprintf(stderr, "shared memory id=%d, addr=0x%x\n", id, (int) (image->data));
}


static void *OpenWindow(void)
{
  void *out_buffer;

  if ((X_visual = I_FindMode (X_screen, doShm)) == NULL)
    I_Error ("Unsupported visual");

  // create the colormap
  X_cmap = XCreateColormap (X_display, RootWindow (X_display, X_screen),
                            X_visual, (BPP == 1) ? AllocAll : AllocNone);

  CreateWindow();
  SetHints();
  SetContext();

  if (doShm)
  {
    X_shmeventtype = XShmGetEventBase (X_display) + ShmCompletion;

    // create the image
    X_image = XShmCreateImage (X_display, X_visual,
        BPP * 8, ZPixmap, 0, &X_shminfo, SCREENDEPTH / BPP,
        SCREENHEIGHT);

    XShmGrabSharedMemory (X_image->bytes_per_line * X_image->height);

    out_buffer = X_image->data;

    if (!out_buffer)
    {
      I_Error ("X-version: shmat() failed.");
    }

    // get the X server to attach to it
    if (!XShmAttach (X_display, &X_shminfo))
      I_Error ("XShmAttach() failed in InitGraphics()");
  }
  else
  {
    out_buffer = malloc (SCREENDEPTH * SCREENHEIGHT);

    X_image = XCreateImage (X_display, X_visual,
        BPP * 8, ZPixmap, 0, (char *) out_buffer, SCREENDEPTH / BPP,
        SCREENHEIGHT, BPP * 8, SCREENDEPTH);
  }

  InitInputs();

  in_game_now = true;

  return out_buffer;
}

void I_GetTruecolInfo (truecol_info_t * info)
{
  info->red_bits = redshift.bits;
  info->red_shift = redshift.shift;
  info->red_mask = redshift.mask;

  info->green_bits = greenshift.bits;
  info->green_shift = greenshift.shift;
  info->green_mask = greenshift.mask;

  info->blue_bits = blueshift.bits;
  info->blue_shift = blueshift.shift;
  info->blue_mask = blueshift.mask;

  info->grey_mask = 0x7FFFFFFF;

  // check for RGB 5:6:5 mode (should be more general !)
  if (info->red_bits == 5 && info->green_bits == 6 &&
      info->blue_bits == 5)
  {
    info->grey_mask = 0xFFDF;
  }
}

void I_GetResolution (void)
{
  const char *val_str;

  val_str = M_GetParm("-width");
  if (val_str)
    SCREENWIDTH = atoi(val_str);

  val_str = M_GetParm("-height");
  if (val_str)
    SCREENHEIGHT = atoi(val_str);
}

static void I_AutodetectBPP(void)
{
  const char *val_str;

  BPP = DefaultDepth(X_display, X_screen) / 8;

  if (M_CheckParm("-hicolour"))
    BPP = 2;

  val_str = M_GetParm("-bpp");
  if (val_str)
    BPP = atoi(val_str);
}

boolean_t I_SetScreenSize(int width, int height, int bpp)
{
  void *out_buffer;

  SCREENDEPTH = V_GetDepth(SCREENWIDTH, BPP);

  out_buffer = OpenWindow();

  dummy_screen.width = SCREENWIDTH;
  dummy_screen.height = SCREENHEIGHT;
  dummy_screen.depth = SCREENDEPTH;
  dummy_screen.bpp = BPP;
  dummy_screen.parent = NULL;
  dummy_screen.data = out_buffer;

  main_scr = V_CreateSubScreen (&dummy_screen, 0, 0, SCREENWIDTH, SCREENHEIGHT);

  return true;
}


//----------------------------------------------------------------------------

//
//  PALETTE STUFF
//

//
// UploadNewPalette
//
// This is used to replace the current 256 colour cmap with a new one.
// Used by 256 colour PseudoColor modes.

static void UploadNewPalette(const byte palette[256][3])
{
  int i;

  const byte *gtable = gammatable[usegamma];

  if (colours == NULL)
  {
    // first call - allocate and prepare colour array
    colours = malloc(256 * sizeof (*colours));

    for (i = 0; i < 256; i++)
    {
      colours[i].pixel = i;
      colours[i].flags = DoRed | DoGreen | DoBlue;
    }
  }

  // set the X colormap entries
  for (i = 0; i < 256; i++)
  {
    int r = gtable[palette[i][0]];
    int g = gtable[palette[i][1]];
    int b = gtable[palette[i][2]];

    colours[i].red   = r * 0x101;
    colours[i].green = g * 0x101;
    colours[i].blue  = b * 0x101;
  }

  // store the colors to the current colormap
  XStoreColors(X_display, X_cmap, colours, 256);
}


void I_ShutdownGraphics (void)
{
  if (graphics_shutdown)
    return;

  graphics_shutdown = 1;

  fprintf (stderr, "I_ShutdownGraphics : ");

  if (doShm)
  {
    // Detach from X server
    if (!XShmDetach(X_display, &X_shminfo))
      I_Error("XShmDetach() failed in I_ShutdownGraphics()");

    fprintf(stderr, "Released XShm shared memory.\n");

    // Release shared memory.
    shmdt(X_shminfo.shmaddr);
    shmctl(X_shminfo.shmid, IPC_RMID, 0);

    // Paranoia.
    X_image->data = NULL;
  }
  else
  {
    X_image->data = NULL;  // Prevent server trying to free image data

    XDestroyImage(X_image);
    fprintf (stderr, "Released XImage memory\n");
  }

  XDestroyWindow(X_display, X_mainWindow);
  XCloseDisplay(X_display);
  X_display = NULL;
}


//
// WaitShm
//
static void WaitShm(void)
{
  while (! shmFinished)
  {
    if (XPending(X_display))
      I_GetEvent();
    else
      usleep(10000);
  }
}

//
// I_StartFrame
//
void I_StartFrame(void)
{
  // nothing to do
}

//
// I_FinishFrame
//
void I_FinishFrame(void)
{
  // protect against calls before I_InitGraphics
  if (!INITIALISED)
    return;

  if (doShm)
  {
    WaitShm();

    if (! XShmPutImage(X_display, X_mainWindow, X_gc, X_image,
        0, 0, 0, 0, SCREENWIDTH, SCREENHEIGHT, True))
    {
      I_Error("XShmPutImage() failed.\n");
    }

    // If we will reuse this buffer in next rendering, wait for it to finish
    shmFinished = false;
    WaitShm();

    return;
  }

  // draw the image
  XPutImage (X_display, X_mainWindow, X_gc, X_image,
      0, 0, 0, 0, SCREENWIDTH, SCREENHEIGHT);

  // sync up with server
  XSync (X_display, False);
}

//
// I_SetPalette
//
void I_SetPalette(byte palette[256][3])
{
  if (!INITIALISED || BPP != 1)
    return;

  UploadNewPalette(palette);
}

//
// I_Colour2Pixel
//
long I_Colour2Pixel(byte palette[256][3], int col)
{
  if (BPP == 1)
    return col;

  return MakeCol(palette[col][0], palette[col][1], palette[col][2]);
}


//----------------------------------------------------------------------------

//
//  INPUT CODE 
//

//
// QueryPointer
//
static boolean_t QueryPointer(mouse_t * p, unsigned int *pmask)
{
  int rootx, rooty;
  Window root, child;
  mouse_t altmouse;
  unsigned int altmask;

  // If NULL parameter, use a dummy instead
  if (p == NULL)
    p = &altmouse;

  if (pmask == NULL)
    pmask = &altmask;

  return XQueryPointer(X_display, X_mainWindow, &root, &child,
         &rootx, &rooty, &(p->x), &(p->y), pmask) ? true : false;
}

//
// VerifyPointerGrabState
//
static void VerifyPointerGrabState(void)
{
#if (! AGGR_FOCUS)
  boolean_t focused;  // Are we the focused window?
  {
    Window fwin = None;
    int revert_to = 0;

    XGetInputFocus (X_display, &fwin, &revert_to);

    // WARNING: non-portable?
    focused = (fwin == X_mainWindow) ? true : false;
  }
#endif

  if (!grabMouse)
    return;

  // Fix for EDGE premature mouse grabbing
  if (!in_game_now)
    return;

  // CPhipps - do not grab the mouse if paused
  // or not playing a level or the window is obscured
  // or during demo playback
  // DO grab in menus, because needed for key bindings screen
  if (paused || (gamestate != GS_LEVEL) ||
      (vis_flag != VisibilityUnobscured)
#if (! AGGR_FOCUS)
      || !focused
#endif
      || demoplayback)
  {

    if (grabbed)
      {
        // Release the mouse
        XUngrabPointer (X_display, CurrentTime);

        // Post a Doom event saying there is no movement and no buttons
        event.type = ev_analogue;
        event.value.analogue.axis = AXIS_DISABLE;
      }

    grabbed = false;
  }
  else
  {
#if AGGR_FOCUS
    XSetInputFocus (X_display, X_mainWindow, RevertToParent, CurrentTime);
#endif

    // grabs the pointer so it is restricted to this window
    if (!grabbed && (vis_flag == VisibilityUnobscured))
    {
      XGrabPointer (X_display, X_mainWindow, True,
#if (! POLL_POINTER)
          ButtonPressMask | ButtonReleaseMask | PointerMotionMask,
#else
          0,
#endif
          GrabModeAsync, GrabModeAsync,
          X_mainWindow, None, CurrentTime);

      QueryPointer (&newmouse, NULL);
    }

    grabbed = true;
    // Warp the pointer back to the middle of the window
    //  or it could end up stuck on an edge
#define POINTER_WARP_TRIGGER 100
#ifdef POINTER_WARP_TRIGGER
    // Only warping if needed
    if (abs (newmouse.x - SCREENWIDTH / 2) > POINTER_WARP_TRIGGER ||
        abs (newmouse.y - SCREENHEIGHT / 2) > POINTER_WARP_TRIGGER)
#endif
    {
      XWarpPointer (X_display, None, X_mainWindow,
          0, 0, 0, 0, SCREENWIDTH / 2, SCREENHEIGHT / 2);

      lastmouse.x = SCREENWIDTH / 2;
      lastmouse.y = SCREENHEIGHT / 2;
    }
    else
    {
      lastmouse = newmouse;
    }
  }
}

//
// InitInputs
//
static void InitInputs(void)
{
  // Make sure we have focus

//ALT:  XSetInputFocus(X_display, X_mainWindow, RevertToParent, CurrentTime);

  // check if the user wants to grab the mouse (quite unnice)
  grabMouse = M_CheckParm("-nomouse") ? false : usemouse;

  VerifyPointerGrabState();
}

//
// TranslateXKey
//
// Translates the key currently in X_event.

static int TranslateXKey(XEvent * pEvent)
{
  int rc;

  rc = XKeycodeToKeysym(X_display, pEvent->xkey.keycode, 0);

  switch (rc)
  {
#ifdef NETWINDER
      /* CPhipps - Jamie's hacks to work with NetWinder X server. */
    case 65430: return KEYD_LEFTARROW;
    case 65432: return KEYD_RIGHTARROW;
    case 65433: return KEYD_DOWNARROW;
    case 65431: return KEYD_UPARROW;
#endif
    case XK_Left: return KEYD_LEFTARROW;
    case XK_Right: return KEYD_RIGHTARROW;
    case XK_Down: return KEYD_DOWNARROW;
    case XK_Up: return KEYD_UPARROW;
    case XK_Escape: return KEYD_ESCAPE;
    case XK_Tab: return KEYD_TAB;
    case XK_F1: return KEYD_F1;
    case XK_F2: return KEYD_F2;
    case XK_F3: return KEYD_F3;
    case XK_F4: return KEYD_F4;
    case XK_F5: return KEYD_F5;
    case XK_F6: return KEYD_F6;
    case XK_F7: return KEYD_F7;
    case XK_F8: return KEYD_F8;
    case XK_F9: return KEYD_F9;
    case XK_F10: return KEYD_F10;
    case XK_F11: return KEYD_F11;
    case XK_F12: return KEYD_F12;

    case XK_KP_Enter:
    case XK_Return: 
      return KEYD_ENTER;

    case XK_BackSpace:
    case XK_Delete:
      return KEYD_BACKSPACE;

    case XK_Pause:
      return KEYD_PAUSE;

    case XK_KP_Equal:
    case XK_equal:
      return KEYD_EQUALS;

    case XK_KP_Subtract:
    case XK_minus:
      return KEYD_MINUS;

    case XK_Shift_L:
    case XK_Shift_R:
      return KEYD_RSHIFT;

    case XK_Control_L:
    case XK_Control_R:
      return KEYD_RCTRL;

    case XK_Alt_L:
    case XK_Meta_L:
    case XK_Alt_R:
    case XK_Meta_R:
      return KEYD_RALT;
  }

  if (rc >= XK_space && rc <= XK_asciitilde)
    rc = rc - XK_space + ' ';

  if (rc >= 'A' && rc <= 'Z')
    rc = rc - 'A' + 'a';

  return rc;
}


//
// I_GetEvent
//
static void I_GetEvent(void)
{
  // CPhipps - make this local
  XEvent X_event;

  // put event-grabbing stuff in here
  XNextEvent(X_display, &X_event);
  switch (X_event.type)
  {
    case KeyPress:
      event.type = ev_keydown;
      event.value.key = TranslateXKey(&X_event);
      E_PostEvent (&event);

      #if DEBUG
      fprintf(stderr, "kd%02X ", event.value.key);
      #endif

      break;
    case KeyRelease:
      event.type = ev_keyup;
      event.value.key = TranslateXKey(&X_event);
      E_PostEvent (&event);

      #if DEBUG
      fprintf(stderr, "ku%02X ", event.value.key);
      #endif

      break;

#if (! POLL_POINTER)
    case ButtonPress:
      if (X_event.xbutton.button == Button1)
      {
        event.type = ev_keydown;
        event.value.key = KEYD_MOUSE1;
        E_PostEvent (&event);
        // fprintf (stderr, "mb1");
      }
      if (X_event.xbutton.button == Button2)
      {
        event.type = ev_keydown;
        event.value.key = KEYD_MOUSE2;
        E_PostEvent (&event);
        // fprintf (stderr, "mb2");
      }
      if (X_event.xbutton.button == Button3)
      {
        event.type = ev_keydown;
        event.value.key = KEYD_MOUSE3;
        E_PostEvent (&event);
        // fprintf (stderr, "mb3");
      }
      break;
    case ButtonRelease:
      if (X_event.xbutton.button == Button1)
      {
        event.type = ev_keyup;
        event.value.key = KEYD_MOUSE1;
        E_PostEvent (&event);
      }
      if (X_event.xbutton.button == Button2)
      {
        event.type = ev_keyup;
        event.value.key = KEYD_MOUSE2;
        E_PostEvent (&event);
      }
      if (X_event.xbutton.button == Button3)
      {
        event.type = ev_keyup;
        event.value.key = KEYD_MOUSE3;
        E_PostEvent (&event);
      }
      break;
    case MotionNotify:
      buttons =
        (X_event.xmotion.state & Button1Mask ? 1 : 0)
        | (X_event.xmotion.state & Button2Mask ? 2 : 0)
        | (X_event.xmotion.state & Button3Mask ? 4 : 0);
      newmouse.x = X_event.xmotion.x;
      newmouse.y = X_event.xmotion.y;

      break;
#endif
    case ClientMessage:
      // CPhipps - allow WM quit
      if (X_event.xclient.data.l[0] == X_deletewin)
        {
          S_StartSound (NULL, sfx_swtchn);
          M_QuitEDGE (0);
        }
      break;

    case Expose:
    case ConfigureNotify:
      break;

    default:
      if (doShm && X_event.type == X_shmeventtype)
        shmFinished = true;
      break;
  }
}


//
// I_ControlGetEvents
//
void I_ControlGetEvents (void)
{
  if (!X_display)
    return;

#if (! POLL_POINTER)
  newmouse = lastmouse;
#endif

  while (XPending (X_display))
    I_GetEvent ();

  if (grabbed)
    {
      unsigned int mask = 0;
#if (! POLL_POINTER)
      if (newmouse.y != lastmouse.y || newmouse.x != lastmouse.x)
        {
          event.type = ev_analogue;
          event.value.data1 = buttons;
#else
      if (QueryPointer (&newmouse, &mask))
        {
//          // It only reaches here if pointer is on our screen
//          event.value.data1 = ((mask & Button1Mask) ? 1 : 0)
//            | ((mask & Button2Mask) ? 2 : 0)
//            | ((mask & Button3Mask) ? 4 : 0);
#endif
//          event.type = ev_analogue;

//          event.value.data1 = mouse_xaxis;
//          event.value.data3 = mouse_yaxis;
 //         event.value.data2 = (newmouse.x - lastmouse.x) * mouseSensitivity;
//          event.value.data4 = (newmouse.y - lastmouse.y) * mouseSensitivity;

//          if (invertmouse)
//            event.value.data4 = -event.value.data4;

          if ((newmouse.x - lastmouse.x))
          {
            event.type = ev_analogue;
            event.value.analogue.axis = mouse_xaxis;
            event.value.analogue.amount = (newmouse.x - lastmouse.x) * mouseSensitivity;
            E_PostEvent(&event);
          }

          if ((newmouse.y - lastmouse.y))
          {
            event.type = ev_analogue;
            event.value.analogue.axis = mouse_yaxis;
            event.value.analogue.amount = (newmouse.y - lastmouse.y) * mouseSensitivity;

            if (invertmouse)
              event.value.analogue.amount = -event.value.analogue.amount;

            E_PostEvent(&event);
          }

///          E_PostEvent (&event);

          {  // CPhipps - EDGE mouse button handling

            static unsigned int buttonstate = 0;
            unsigned int chmask = buttonstate ^ mask;  // XOR prev and current

            if (chmask & Button1Mask)
              {
                event.type = (mask & Button1Mask) ? ev_keydown : ev_keyup;
                event.value.key = KEYD_MOUSE1;
                E_PostEvent (&event);
              }
            if (chmask & Button2Mask)
              {
                event.type = (mask & Button2Mask) ? ev_keydown : ev_keyup;
                event.value.key = KEYD_MOUSE2;
                E_PostEvent (&event);
              }
            if (chmask & Button3Mask)
              {
                event.type = (mask & Button3Mask) ? ev_keydown : ev_keyup;
                event.value.key = KEYD_MOUSE3;
                E_PostEvent (&event);
              }

            buttonstate = mask;
          }
        }
    }

  VerifyPointerGrabState ();
}

//
// I_EDGELoop
//
void I_EDGELoop(void)
{
  while (1)
  {
    E_EDGELoopRoutine();
  }
}
