//----------------------------------------------------------------------------
//  EDGE Null Misc System Interface code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2000  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------

#include "../i_defs.h"

#include "../con_main.h"
#include "../dm_defs.h"
#include "../e_main.h"
#include "../e_net.h"
#include "../g_game.h"
#include "../m_argv.h"
#include "../m_menu.h"
#include "../m_misc.h"
#include "../r_draw1.h"
#include "../r_draw2.h"
#include "../w_wad.h"
#include "../z_zone.h"
#include "../version.h"

#include "i_sysinc.h"

// msgbuf for I_Warning, I_WriteDebug & I_Printf
#define MSGBUFSIZE 4096
static char msgbuf[MSGBUFSIZE];

//
// I_SystemStartup
//
// Startup system. Returns success.
//
boolean_t I_SystemStartup(void)
{
  I_StartupControl();
  I_StartupGraphics();
  I_StartupSound(NULL);    // -ACB- 1999/09/20 Sets nosound directly
  I_StartupMusic(NULL);
  
  return true;
}

//
// I_WaitVBL
//
// Waits for count/70 seconds.
void I_WaitVBL(int count)
{
  int end = time(NULL) + count / 70;

  while (time(NULL) < end)
    ;
}

//
// I_EDGELoop
//
// Will loop forever.
void I_EDGELoop(void)
{
  while(1)
    E_EDGELoopRoutine();
}

//
// I_DisplayExitScreen
//
// Displays the exit screen, from the lump ENDOOM.
void I_DisplayExitScreen(void)
{
  int i;
  const char *out = W_CacheLumpName("ENDOOM");

  putch('\n');
  for (i = 0; i < 80 * 25; i++)
  {
    putch(*out);
    out += 2;
  }

  W_DoneWithLump(out);
}

//
// I_CloseProgram
//
// Closes the program
void I_CloseProgram(int exitnum)
{
  exit(exitnum);
}

//
// I_TraceBack
//
// Like I_CloseProgram, but may display some sort of debugging information
// on some systems (typically the function call stack).
void I_TraceBack(void)
{
  I_CloseProgram(-1);
}


//
// I_GetTime
//
// returns time in 1/TICRATE second tics
int I_GetTime(void)
{
  // In this ANSI C version, we have a very lousy precision.
  return TICRATE * time(0);
}


unsigned long microtimer_granularity = 1;
unsigned long I_ReadMicroSeconds(void)
{
  return time(0) * 1000000;
}

//
// I_Warning
//
// -AJA- 1999/09/10: added this.
//
void I_Warning(const char *warning,...)
{
  va_list argptr;

  va_start(argptr, warning);
  vsprintf(msgbuf, warning, argptr);

  if (debugfile)
    fprintf(debugfile, "%s", msgbuf);

  CON_Printf("%s", msgbuf);
  va_end(argptr);
}

//
// I_Error
//
void I_Error(const char *error,...)
{
  va_list argptr;

  // Bring the system to a close
  I_SystemShutdown();

  // Message last, so it actually prints on the screen
  va_start(argptr, error);
  fprintf(stdout, "\n");
  vfprintf(stdout, error, argptr);
  fprintf(stdout, "\n");

  va_end(argptr);
  fflush(stdout);

  I_CloseProgram(-1);
}

//
// I_Printf
//
void I_Printf(const char *message,...)
{
  va_list argptr;
  char *string;
  char printbuf[MSGBUFSIZE];

  // clear the buffer
  memset(printbuf, 0, MSGBUFSIZE);

  string = printbuf;

  va_start(argptr, message);

  // Print the message into a text string
  vsprintf(printbuf, message, argptr);

  // Clean up \n\r combinations
  while (*string)
  {
    if (*string == '\n')
    {
      memmove(string + 2, string + 1, strlen(string));
      string[1] = '\r';
      string++;
    }
    string++;
  }

  // Send the message to the console.
  CON_Printf(printbuf);

  // And the text screen if in text mode
  if (!graphicsmode)
    cprintf(printbuf);

  va_end(argptr);
}

// -KM- 1998/10/29 Use all of screen, not just first 25 rows.
void I_PutTitle(const char *title)
{
  char string[81];
  int centre;

  memset(string, ' ', 80);
  string[80] = 0;

  // Build the title
  // 23-6-98 KM Changed edgever to hex to allow versions such as 0.65a
  centre = (80 - strlen(title)) / 2;
  memcpy(&string[centre], title, strlen(title));

  // Print the title
  printf("%s\n", string);
}

//
// I_SystemShutdown
//
void I_SystemShutdown(void)
{
  E_EngineShutdown();

  I_ShutdownMusic();
  I_ShutdownSound();
  I_ShutdownGraphics();
  I_ShutdownControl();

  // -KM- 1999/01/31 Close the debugfile
  if (debugfile)
    fclose(debugfile);
}

//
// I_PathIsAbsolute
//
// Returns true if the path should be treated as an absolute path.
// -ES- 2000/01/01 Written.
boolean_t I_PathIsAbsolute(const char *path)
{
  // We assume POSIX.
  if (path[0] == '/')
    return true;
  else
    return false;
}

//
// I_PreparePath
//
// Prepares the end of the path name, so it will be possible to concatenate
// a DIRSEPARATOR and a file name to it.
// Allocates and returns the new string.
//
char *I_PreparePath(const char *path)
{
/*
  int len = strlen(path);
  char *s;

  // We assume POSIX.
  if (len == 0)
  {
    // empty string means "./"
    return Z_StrDup(".");
  }

  if (path[len - 1] == '/')
  {
    // cut off the last separator
    s = Z_Malloc(len);
    memcpy(s, path, len - 1);
    s[len - 1] = 0;

    return s;
  }

  return Z_StrDup(path);
*/
}

//
// I_Access
//
boolean_t I_Access(const char *filename)
{
  return (access(filename, R_OK) == 0) ? true : false;
}


//
// I_GetModifiedTime
//
// Fills in 'i_time_t' to match the modified time of 'filename'. Returns true on success.
//
// -ACB- 2001/06/14
//
boolean_t I_GetModifiedTime(const char *filename, i_time_t *t)
{
/*
  struct stat buf;
  struct tm timeinf;

  // Check the sanity of the coders...
  if (!t)
    return false;

  memset(t,0,sizeof(i_time_t));

  // Check the file is invalid
  if (stat(filename, &buf))			
    return false;

  // Convert the 'time_t' of the modified time into something more human
  if(!localtime_r(&buf.st_mtime, &timeinf))
    return false;

  t->secs    = timeinf.tm_sec;
  t->minutes = timeinf.tm_min;
  t->hours   = timeinf.tm_hour;
  t->day     = timeinf.tm_mday;
  t->month   = timeinf.tm_mon+1;
  t->year    = timeinf.tm_year+1900;
*/
  // Check the sanity of the coders...
  if (!t)
    return false;

  if (!filename)
    return false;

  memset(t,0,sizeof(i_time_t));

  t->secs    = 0;
  t->minutes = 0;
  t->hours   = 0;
  t->day     = 1;
  t->month   = 1;
  t->year    = 1900;

  return true;
}


