//----------------------------------------------------------------------------
//  EDGE Play Simulation Action routines
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------
//
// Notes:
//  All Procedures here are never called directly, except possibly
//  by another P_Act* Routine. Otherwise the procedure is called
//  by referencing an code pointer from the states[] table. The only
//  exception to these rules are P_ActMissileContact and
//  P_ActSlammedIntoObject that requiring "acting" on the part
//  of an obj.
//
// This file was created for all action code by DDF.
//
// -KM- 1998/09/27 Added sounds.ddf capability
// -KM- 1998/11/25 Visibility is now a fixed_t.
// -KM- 1998/12/21 New smooth visibility.
// -AJA- 1999/07/21: Replaced some non-critical P_Randoms with M_Random.
// -AJA- 1999/08/08: Replaced some P_Random()-P_Random() stuff.
//

#include "i_defs.h"
#include "p_action.h"

#include "con_main.h"
#include "dm_defs.h"
#include "dm_state.h"
#include "g_game.h"
#include "m_inline.h"
#include "m_misc.h"
#include "m_random.h"
#include "p_local.h"
#include "p_weapon.h"
#include "r_state.h"
#include "rad_trig.h"
#include "s_sound.h"
#include "w_wad.h"
#include "z_zone.h"

#define TRACEANGLE 0xc000000

void P_DoAttack(mobj_t * object);
void P_ActRangeAttack(mobj_t * object);
void P_ActMeleeAttack(mobj_t * object);

//-----------------------------------------
//--------------MISCELLANOUS---------------
//-----------------------------------------

//
// P_ActActivateLineType
//
// Allows things to also activate linetypes, bringing them into the
// fold with radius triggers, which can also do it.  There's only two
// parameters needed: linetype number & tag number, which are stored
// in the state's `action_par' field as a pointer to two integers.
// 
// -AJA- 2000/01/09: written.
//
void P_ActActivateLineType(mobj_t * object)
{
  int *values;
  
  if (!object->state || !object->state->action_par)
    return;

  values = (int *) object->state->action_par;
  
  // Note the `NULL' here: this prevents the activation from failing
  // because the object isn't a PLAYER, for example.
  P_RemoteActivation(NULL, values[0], values[1], 0, line_Any);
}

//
// P_ActEnableRadTrig
// P_ActDisableRadTrig
//
// Allows things to enable or disable radius triggers (by tag number),
// like linetypes can do already.
//
// -AJA- 2000/01/09: written.
//
void P_ActEnableRadTrig(mobj_t * object)
{
  int *value;
  
  if (!object->state || !object->state->action_par)
    return;

  value = (int *) object->state->action_par;
  RAD_EnableByTag(object, value[0], false);
}

void P_ActDisableRadTrig(mobj_t * object)
{
  int *value;
  
  if (!object->state || !object->state->action_par)
    return;

  value = (int *) object->state->action_par;
  RAD_EnableByTag(object, value[0], true);
}

//
// P_ActLookForTargets
//
// Looks for targets: used in the same way as enemy things look
// for players
//
// TODO: Write a decent procedure.
// -KM- 1999/01/31 Added sides. Still has to search every mobj on the
//  map to find a target.  There must be a better way...
//
boolean_t P_ActLookForTargets(mobj_t * object)
{
  mobj_t *currmobj;

  currmobj = mobjlisthead;

  while (currmobj != NULL)
  {
    if ((currmobj->side & object->side) && !object->supportobj)
    {
      if (currmobj->supportobj != object && P_CheckSight(object, currmobj))
      {
        P_MobjSetSupportObj(object, currmobj);
        if (object->info->meander_state)
          P_SetMobjStateDeferred(object, object->info->meander_state, 0);
        return true;
      }
    }
    if ((((currmobj->target == object->supportobj || currmobj->target == object)
                && currmobj->target)
            || (object->side && !(currmobj->side & object->side)))
        && ((currmobj != object) &&
            (currmobj != object->supportobj) &&
            (object->info != currmobj->info || (object->extendedflags & EF_DISLOYALTYPE)) &&
            (object->supportobj != currmobj->supportobj)))
    {
      if ((currmobj->flags & MF_SHOOTABLE) && P_CheckSight(object, currmobj))
      {
        P_MobjSetTarget(object, currmobj);
        if (object->info->chase_state)
          P_SetMobjStateDeferred(object, object->info->chase_state, 0);
        return true;
      }
    }
    currmobj = currmobj->next;
  }

  return false;
}

//
// DecideMeleeAttack
//
// This is based on P_CheckMeleeRange, except that it relys upon
// info from the objects close combat attack, the original code
// used a set value for all objects which was MELEERANGE + 20,
// this code allows different melee ranges for different objects.
//
// -ACB- 1998/08/15
// -KM- 1998/11/25 Added attack parameter.
//
static boolean_t DecideMeleeAttack(mobj_t * object, const attacktype_t * attack)
{
  mobj_t *target;
  float_t distance;
  float_t meleedist;

  target = object->target;

  if (!target)
    return false;

  if (!P_CheckSight(object, target))
    return false;

  if (!attack)
    return false;  // cannot evaluate range with no attack range

  distance = P_ApproxDistance(target->x - object->x, target->y - object->y);
  if (level_flags.true3dgameplay)
    distance = P_ApproxDistance(target->z - object->z, distance);
  meleedist = attack->range - 20 + target->info->radius;

  if (distance >= meleedist)
    return false;

  return true;
}

//
// P_ActDecideRangeAttack
//
// This is based on P_CheckMissileRange, contrary the name it does more
// than check the missile range, it makes a decision of whether or not an
// attack should be made or not depending on the object with the option
// to attack. A return of false is mandatory if the object cannot see its
// target (makes sense, doesn't it?), after this the distance is calculated,
// it will eventually be check to see if it is greater than a number from
// the Random Number Generator; if so the procedure returns true. Essentially
// the closer the object is to its target, the more chance an attack will
// be made (another logical decision).
//
// -ACB- 1998/08/15
//
static boolean_t P_ActDecideRangeAttack(mobj_t * object)
{
  percent_t chance;
  float_t distance;
  const attacktype_t *attack;

  if (! object->target)
    return false;

  if (!P_CheckSight(object, object->target))
    return false;

  if (object->info->rangeattack)
    attack = object->info->rangeattack;
  else
    return false;  // cannot evaluate range with no attack range

  // Just been hit (and have felt pain), so in true tit-for-tat
  // style, the object - without regard to anything else - hits back.
  if (object->flags & MF_JUSTHIT)
  {
    object->flags &= ~MF_JUSTHIT;
    return true;
  }

  // Bit slow on the up-take: the object hasn't had time to
  // react his target.
  if (object->reactiontime)
    return false;

  // Get the distance, a basis for our decision making from now on
  distance = P_ApproxDistance(object->x - object->target->x,
      object->y - object->target->y);

  // If no close-combat attack, increase the chance of a missile attack
  if (!object->info->melee_state)
    distance -= 192;
  else
    distance -= 64;

  // Object is too far away to attack?
  if (attack->range && distance >= attack->range)
    return false;

  // Object is too close to target
  if (attack->tooclose && attack->tooclose >= distance)
    return false;

  // Object likes to fire? if so, double the chance of it happening
  if (object->extendedflags & EF_TRIGGERHAPPY)
    distance /= 2;

  // The chance in the object is one given that the attack will happen, so
  // we inverse the result (since its one in 255) to get the chance that
  // the attack will not happen.
  chance = 1.0 - object->info->minatkchance;
  chance = MIN(distance / 255.0, chance);

  // now after modifing distance where applicable, we get the random number and
  // check if it is less than distance, if so no attack is made.
  if (P_RandomTest(chance))
    return false;

  return true;
}

//
// P_ActFaceTarget
//
// Look at the prey......
//
void P_ActFaceTarget(mobj_t * object)
{
  mobj_t *target;
  float_t dist, dz;

  target = object->target;

  if (!target || (target->extendedflags & EF_DUMMYMOBJ))
    return;

  if (object->flags & MF_STEALTH)
    object->vis_target = VISIBLE;

  object->flags &= ~MF_AMBUSH;

  object->angle = R_PointToAngle(object->x, object->y, target->x, target->y);

  dist = R_PointToDist(object->x, object->y, target->x, target->y);

  if (dist >= 0.1)
  {
    dz = MO_MIDZ(target) - MO_MIDZ(object);

    object->vertangle = dz / dist;
  }

  if (target->flags & MF_FUZZY)
  {
    object->angle += P_RandomNegPos() << (ANGLEBITS - 11);
    object->vertangle += P_RandomNegPos() / 1024.0;
  }

  if (target->visibility < VISIBLE)
  {
    float_t amount = (VISIBLE - target->visibility);

    object->angle += P_RandomNegPos() * (ANGLEBITS - 12) * amount;
    object->vertangle += P_RandomNegPos() * amount / 2048.0;
  }

  // don't look up/down too far...
  if (object->vertangle < LOOKDOWNLIMIT)
    object->vertangle = LOOKDOWNLIMIT;
  else if (object->vertangle > LOOKUPLIMIT)
    object->vertangle = LOOKUPLIMIT;
}

//
// P_ActMakeIntoCorpse
//
// Gives the effect of the object being a corpse....
//
void P_ActMakeIntoCorpse(mobj_t * mo)
{
  if (mo->flags & MF_STEALTH)
    mo->vis_target = VISIBLE;  // dead and very visible

  // object is on ground, it can be walked over
  mo->flags &= ~MF_SOLID;
}

//
// P_BringCorpseToLife
//
// Bring a corpse back to life (the opposite of the above routine).
// Handles players too !
//
void P_BringCorpseToLife(mobj_t * corpse)
{
  const mobjinfo_t *info = corpse->info;

  corpse->flags = info->flags;
  corpse->health = info->spawnhealth;
  corpse->radius = info->radius;
  corpse->height = info->height;
  corpse->extendedflags = info->extendedflags;
  corpse->vis_target = PERCENT_2_FLOAT(info->translucency);

  if (corpse->player)
  {
    corpse->player->playerstate = PST_LIVE;
    corpse->player->health = corpse->health;
    corpse->player->std_viewheight = corpse->height * 
        PERCENT_2_FLOAT(info->viewheight);
  }

  if (info->overkill_sound)
    S_StartSound(corpse, info->overkill_sound);

  if (info->raise_state)
    P_SetMobjState(corpse, info->raise_state);
  else if (info->meander_state)
    P_SetMobjState(corpse, info->meander_state);
  else if (info->idle_state)
    P_SetMobjState(corpse, info->idle_state);
  else
    I_Error("Object %s has no RESURRECT states.\n", info->ddf.name);
}

//
// P_ActResetSpreadCount
//
// Resets the spreader count for fixed-order spreaders, normally used at the
// beginning of a set of missile states to ensure that an object fires in
// the same object each time.
//
void P_ActResetSpreadCount(mobj_t * object)
{
  object->spreadcount = 0;
}

//-------------------------------------------------------------------
//-------------------VISIBILITY HANDLING ROUTINES--------------------
//-------------------------------------------------------------------

//
// P_ActTransSet
//
void P_ActTransSet(mobj_t * object)
{
  const state_t *st;
  float_t value = VISIBLE;

  st = object->state;

  if (st && st->action_par)
  {
    value = ((percent_t *)st->action_par)[0];
    value = MAX(0.0, MIN(1.0, value));
  }

  object->visibility = object->vis_target = value;
}

//
// P_ActTransFade
//
void P_ActTransFade(mobj_t * object)
{
  const state_t *st;
  float_t value = INVISIBLE;

  st = object->state;

  if (st && st->action_par)
  {
    value = ((percent_t *)st->action_par)[0];
    value = MAX(0.0, MIN(1.0, value));
  }

  object->vis_target = value;
}

//
// P_ActTransLess
//
void P_ActTransLess(mobj_t * object)
{
  const state_t *st;
  float_t value = 0.05;

  st = object->state;

  if (st && st->action_par)
  {
    value = ((percent_t *)st->action_par)[0];
    value = MAX(0.0, MIN(1.0, value));
  }

  object->vis_target -= value;

  if (object->vis_target < INVISIBLE)
    object->vis_target = INVISIBLE;
}

//
// P_ActTransMore
//
void P_ActTransMore(mobj_t * object)
{
  const state_t *st;
  float_t value = 0.05;

  st = object->state;

  if (st && st->action_par)
  {
    value = ((percent_t *)st->action_par)[0];
    value = MAX(0.0, MIN(1.0, value));
  }

  object->vis_target += value;

  if (object->vis_target > VISIBLE)
    object->vis_target = VISIBLE;
}

//
// P_ActTransAlternate
//
// Alters the translucency of an item, EF_LESSVIS is used
// internally to tell the object if it should be getting
// more visible or less visible; EF_LESSVIS is set when an
// object is to get less visible (because it has become
// to a level of lowest translucency) and the flag is unset
// if the object has become as highly translucent as possible.
//
void P_ActTransAlternate(mobj_t * object)
{
  const state_t *st;
  float_t value = 0.05;

  st = object->state;

  if (st && st->action_par)
  {
    value = ((percent_t *)st->action_par)[0];
    value = MAX(0.0, MIN(1.0, value));
  }

  if (object->extendedflags & EF_LESSVIS)
  {
    object->vis_target -= value;
    if (object->vis_target <= INVISIBLE)
    {
      object->vis_target = INVISIBLE;
      object->extendedflags &= ~EF_LESSVIS;
    }
  }
  else
  {
    object->vis_target += value;
    if (object->vis_target >= VISIBLE)
    {
      object->vis_target = VISIBLE;
      object->extendedflags |= EF_LESSVIS;
    }
  }
}

//
// P_ActDLightSet
//
void P_ActDLightSet(mobj_t * mo)
{
  const state_t *st = mo->state;

  if (st && st->action_par)
  {
    mo->dlight_qty = MAX(0, ((int *)st->action_par)[0]);
    mo->dlight_target = mo->dlight_qty;
  }
}

//
// P_ActDLightFade
//
void P_ActDLightFade(mobj_t * mo)
{
  const state_t *st = mo->state;

  if (st && st->action_par)
  {
    mo->dlight_target = MAX(0, ((int *)st->action_par)[0]);
  }
}

//
// P_ActDLightRandom
//
void P_ActDLightRandom(mobj_t * mo)
{
  const state_t *st = mo->state;

  if (st && st->action_par)
  {
    int low  = ((int *)st->action_par)[0];
    int high = ((int *)st->action_par)[1];

    // Note: using M_Random so that gameplay is unaffected
    int qty = low + (high - low) * M_Random() / 255;
    
    mo->dlight_qty = MAX(0, qty);
    mo->dlight_target = mo->dlight_qty;
  }
}


//-------------------------------------------------------------------
//------------------- MOVEMENT ROUTINES -----------------------------
//-------------------------------------------------------------------

void P_ActFaceDir(mobj_t * object)
{
  const state_t *st = object->state;

  if (st && st->action_par)
    object->angle = *(angle_t *)st->action_par;
  else
    object->angle = 0;
}

void P_ActTurnDir(mobj_t * object)
{
  const state_t *st = object->state;
  angle_t turn = ANG180;

  if (st && st->action_par)
  {
    int par = *(int *)st->action_par;
    turn = FLOAT_2_ANG((float_t) par);
  }

  object->angle += turn;
}

void P_ActTurnRandom(mobj_t * object)
{
  const state_t *st = object->state;
  int turn = 359;

  if (st && st->action_par)
  {
    turn = *(int *)st->action_par;
  }

  turn = turn * P_Random() / 90;  // 10 bits of angle
   
  if (turn < 0)
    object->angle -= (angle_t)((-turn) << (ANGLEBITS - 10));
  else
    object->angle += (angle_t)(turn << (ANGLEBITS - 10));
}

void P_ActMlookFace(mobj_t * object)
{
  const state_t *st = object->state;

  if (st && st->action_par)
    object->vertangle = *(float_t *)st->action_par;
  else
    object->vertangle = 0.0;
}

void P_ActMlookTurn(mobj_t * object)
{
  const state_t *st = object->state;

  if (st && st->action_par)
    object->vertangle = *(float_t *)st->action_par;
}

void P_ActMoveFwd(mobj_t * object)
{
  const state_t *st = object->state;

  if (st && st->action_par)
  {
    float_t amount = *(float_t *)st->action_par;
    
    float_t dx = M_Cos(object->angle);
    float_t dy = M_Sin(object->angle);

    object->mom.x += dx * amount;
    object->mom.y += dy * amount;
  }
}

void P_ActMoveRight(mobj_t * object)
{
  const state_t *st = object->state;

  if (st && st->action_par)
  {
    float_t amount = *(float_t *)st->action_par;
    
    float_t dx = M_Cos(object->angle - ANG90);
    float_t dy = M_Sin(object->angle - ANG90);

    object->mom.x += dx * amount;
    object->mom.y += dy * amount;
  }
}

void P_ActMoveUp(mobj_t * object)
{
  const state_t *st = object->state;

  if (st && st->action_par)
    object->mom.z += *(float_t *)st->action_par;
}

void P_ActStopMoving(mobj_t * object)
{
  object->mom.x = object->mom.y = object->mom.z = 0;
}


//-------------------------------------------------------------------
//-------------------SOUND CAUSING ROUTINES--------------------------
//-------------------------------------------------------------------

//
// P_ActPlaySound
//
// Generate an arbitrary sound.
//
void P_ActPlaySound(mobj_t * mo)
{
  sfx_t *sound = NULL;

  if (mo->state && mo->state->action_par)
    sound = (sfx_t *) mo->state->action_par;

  if (! sound)
  {
    M_WarnError("P_ActPlaySound: missing sound name in %s.\n", 
        mo->info->ddf.name);
    return;
  }

  S_StartSound(mo, sound);
}

//
// P_ActKillSound
//
// Kill any current sounds from this thing.
//
void P_ActKillSound(mobj_t * mo)
{
  S_StopSound(mo);
}

//
// P_ActMakeAmbientSound
//
// Just a sound generating procedure that cause the sound ref
// in seesound to be generated.
//
void P_ActMakeAmbientSound(mobj_t * object)
{
  if (object->info->seesound)
    S_StartSound(object, object->info->seesound);

#ifdef DEVELOPERS
  else
    L_WriteDebug("%s has no ambient sound\n", object->info->ddf.name);
#endif
}

//
// P_ActMakeAmbientSoundRandom
//
// Give a small "random" chance that this object will make its
// ambient sound. Currently this is a set value of 50, however
// the code that drives this, should allow for the user to set
// the value, note for further DDF Development.
//
void P_ActMakeAmbientSoundRandom(mobj_t * object)
{
  if (object->info->seesound)
  {
    if (M_Random() < 50)
      S_StartSound(object, object->info->seesound);
  }
#ifdef DEVELOPERS
  else
  {
    L_WriteDebug("%s has no ambient sound\n", object->info->ddf.name);
    return;
  }
#endif

}

//
// P_ActMakeActiveSound
//
// Just a sound generating procedure that cause the sound ref
// in seesound to be generated.
//
// -KM- 1999/01/31
//
void P_ActMakeActiveSound(mobj_t * object)
{
  if (object->info->activesound)
    S_StartSound(object, object->info->activesound);

#ifdef DEVELOPERS
  else
    L_WriteDebug("%s has no ambient sound\n", object->info->ddf.name);
#endif
}

//
// P_ActMakeDyingSound
//
// This procedure is like everyother sound generating
// procedure with the exception that if the object is
// a boss (EF_BOSSMAN extended flag) then the sound is
// generated at full volume (source = NULL).
//
void P_ActMakeDyingSound(mobj_t * object)
{
  sfx_t *sound;

  sound = object->info->deathsound;

  if (sound)
  {
    if (object->info->extendedflags & EF_BOSSMAN)
      S_StartSound(NULL, sound);
    else
      S_StartSound(object, sound);
    return;
  }

#ifdef DEVELOPERS
  L_WriteDebug("%s has no death sound\n", object->info->ddf.name);
#endif
}

//
// P_ActMakePainSound (Ow!! it hurts!)
//
void P_ActMakePainSound(mobj_t * object)
{
  if (object->info->painsound)
  {
    if (object->info->extendedflags & EF_BOSSMAN)
      S_StartSound(object, object->info->painsound);
    else
      S_StartSound(object, object->info->painsound);
  }
#ifdef DEVELOPERS
  else
  {
    L_WriteDebug("%s has no pain sound\n", object->info->ddf.name);
  }
#endif
}

//
// P_ActMakeOverKillSound
//
// -AJA- 1999/12/01: made user definable.
//
void P_ActMakeOverKillSound(mobj_t * object)
{
  if (object->info->overkill_sound)
  {
    if (object->info->extendedflags & EF_BOSSMAN)
      S_StartSound(NULL, object->info->overkill_sound);
    else
      S_StartSound(object, object->info->overkill_sound);
  }
#ifdef DEVELOPERS
  else
    L_WriteDebug("%s has no overkill sound\n", object->info->ddf.name);
#endif
}

//
// P_ActMakeCloseAttemptSound
//
// Attempting close combat sound
//
void P_ActMakeCloseAttemptSound(mobj_t * object)
{
  sfx_t *sound;

  if (! object->info->closecombat)
    I_Error("Object [%s] used CLOSEATTEMPTSND action, "
        "but has no CLOSE_ATTACK\n", object->info->ddf.name);
   
  sound = object->info->closecombat->initsound;

  if (sound)
  {
    S_StartSound(object, sound);
  }
#ifdef DEVELOPERS
  else
    L_WriteDebug("%s has no close combat attempt sound\n", object->info->ddf.name);
#endif
}

//
// P_ActMakeRangeAttemptSound
//
// Attempting attack at range sound
//
void P_ActMakeRangeAttemptSound(mobj_t * object)
{
  sfx_t *sound;

  if (! object->info->rangeattack)
    I_Error("Object [%s] used RANGEATTEMPTSND action, "
        "but has no RANGE_ATTACK\n", object->info->ddf.name);     

  sound = object->info->rangeattack->initsound;

  if (sound)
    S_StartSound(object, sound);
#ifdef DEVELOPERS
  else
    L_WriteDebug("%s has no range attack attempt sound\n", object->info->ddf.name);
#endif
}

//-------------------------------------------------------------------
//-------------------EXPLOSION DAMAGE ROUTINES-----------------------
//-------------------------------------------------------------------

//
// P_ActDamageExplosion
//
// Radius Attack damage set by info->damage. Used for the original Barrels
//
void P_ActDamageExplosion(mobj_t * object)
{
  float_t damage;
  
  DAMAGE_COMPUTE(damage, &object->info->damage);

#ifdef DEVELOPERS
  if (!damage)
  {
    L_WriteDebug("%s caused no explosion damage\n", object->info->ddf.name);
    return;
  }
#endif

  P_RadiusAttack(object, object->source, damage, damage,
      &object->info->damage, false);
}

//
// P_ActThrust
//
// Thrust set by info->damage.
//
// -AJA- 1999/11/06: written.
//
void P_ActThrust(mobj_t * object)
{
  float_t damage;
  
  DAMAGE_COMPUTE(damage, &object->info->damage);

#ifdef DEVELOPERS
  if (!damage)
  {
    L_WriteDebug("%s caused no thrust\n", object->info->ddf.name);
    return;
  }
#endif

  P_RadiusAttack(object, object->source, damage, damage,
      &object->info->damage, true);
}

//-------------------------------------------------------------------
//-------------------MISSILE HANDLING ROUTINES-----------------------
//-------------------------------------------------------------------

//
// P_ActExplode
//
// The object blows up, like a missile.
//
// -AJA- 1999/08/21: Replaced P_ActExplodeMissile (which was identical
//       to p_mobj's P_ExplodeMissile) with this.
//
void P_ActExplode(mobj_t * object)
{
  P_MobjExplodeMissile(object);
}

//
// P_ActCheckMissileSpawn
//
// This procedure handles a newly spawned missile, it moved
// by half the amount of momentum and then checked to see
// if the move is possible, if not the projectile is
// exploded. Also the number of initial tics on its
// current state is taken away from by a random number
// between 0 and 3, although the number of tics will never
// go below 1.
//
// -ACB- 1998/08/04
//
// -AJA- 1999/08/22: Fixed a bug that occasionally caused the game to
//       go into an infinite loop.  NOTE WELL: don't fiddle with the
//       object's x & y directly, use P_TryMove instead, or
//       P_ChangeThingPosition.
//
static void CheckMissileSpawn(mobj_t * projectile)
{
  projectile->tics -= P_Random() & 3;

  if (projectile->tics < 1)
    projectile->tics = 1;

  projectile->z += projectile->mom.z / 2;

  if (!P_TryMove(projectile,
          projectile->x + projectile->mom.x / 2,
          projectile->y + projectile->mom.y / 2))
  {
    P_MobjExplodeMissile(projectile);
  }
}

//
// P_ActLaunchProjectile
//
// This procedure launches a project the direction of the target mobj.
// * source - the source of the projectile
// * target - the target of the projectile
// * type   - the mobj type of the projectile
//
// For all sense and purposes it is possible for the target to be a dummy
// mobj, just to act as a carrier for a set of target co-ordinates.
//
// Missiles can be spawned at different locations on and around
// the mobj. Traditionally an mobj would fire a projectile
// at a height of 32 from the centerpoint of that
// mobj, this was true for all creatures from the Cyberdemon to
// the Imp. The currentattack holds the height and x & y
// offsets that dictates the spawning location of a projectile.
//
// Traditionally: Height   = 4*8
//                x-offset = 0
//                y-offset = 0
//
// The exception to this rule is the revenant, which did increase
// its z value by 16 before firing: This was a hack
// to launch a missile at a height of 48. The revenants
// height was reduced to normal after firing, this new code
// makes that an unnecesary procedure.
//
// projx, projy & projz are the projectiles spawn location
//
// NOTE: may return NULL.
//
// -ACB- 1998/08/04
// -KM-  1998/11/25 Accuracy is now a fixed_t
//
static mobj_t *LaunchProjectile(mobj_t * source, mobj_t * target,
    const mobjinfo_t * type)
{
  const attacktype_t *attack;
  float_t projx, projy, projz;
  angle_t angle;
  float_t slope;
  mobj_t *projectile;
  float_t yoffset;

  attack = source->currentattack;

  if (! attack)
    return NULL;

  // -AJA- prevent possible crashes.  Prolly not the best solution,
  //       more like a baid-aid solution.
  if (!target)
    target = P_MapTargetTheory(source);

  // -AJA- projz now handles crouching
  projx = source->x;
  projy = source->y;
  projz = source->z + attack->height * source->height /
      source->info->height;

  angle = source->angle;

  projx += attack->xoffset * M_Cos(source->angle + ANG90);
  projy += attack->xoffset * M_Sin(source->angle + ANG90);

  if (attack->yoffset)
    yoffset = attack->yoffset;
  else
    yoffset = source->radius - 0.5;

  projx += yoffset * M_Cos(angle);
  projy += yoffset * M_Sin(angle);
  projz += yoffset * source->vertangle;

  projectile = P_MobjCreateObject(projx, projy, projz, type);

  // currentattack is held so that when a collision takes place
  // with another object, we know whether or not the object hit
  // can shake off the attack or is damaged by it.
  //
  projectile->currentattack = attack;
  P_MobjSetRealSource(projectile, source);

  // check for blocking lines between source and projectile
  if (P_MapCheckBlockingLine(source, projectile))
  {
    P_MobjExplodeMissile(projectile);
    return NULL;
  }

  if (projectile->info && projectile->info->seesound)
  {
    if (projectile->info->extendedflags & EF_BOSSMAN)
      S_StartSound(NULL, projectile->info->seesound);
    else
      S_StartSound(projectile, projectile->info->seesound);
  }

  angle = R_PointToAngle(projx, projy, target->x, target->y);

  // Now add the fact that the target may be difficult to spot and
  // make the projectile's target the same as the sources. Only
  // do these if the object is not a dummy object, otherwise just
  // flag the missile not to trace: you cannot track a target that
  // does not exist...
  //
  if (target->extendedflags & EF_DUMMYMOBJ)
  {
    projectile->extendedflags |= EF_NOTRACE;
    P_MobjSetTarget(projectile, NULL);
    target->z += attack->height;
  }
  else
  {
    P_MobjSetTarget(projectile, target);
    projectile->extendedflags |= EF_FIRSTCHECK;

    if (!attack->flags & AF_Player)
    {
      if (target->flags & MF_FUZZY)
        angle += P_RandomNegPos() << (ANGLEBITS - 12);

      if (target->visibility < VISIBLE)
        angle += P_RandomNegPos() * 64 * (VISIBLE - target->visibility);
    }
  }

  // Calculate slope
  slope = P_ApproxSlope(target->x - projx, target->y - projy,
      MO_MIDZ(target) - projz);

  // -AJA- 1999/09/11: add in attack's angle & slope offsets.
  angle -= attack->angle_offset;
  slope += attack->slope_offset;
  
  // is the attack not accurate?
  if (!source->player || source->player->refire > 0)
  {
    if (attack->accuracy_angle > 0)
      angle += (attack->accuracy_angle >> 8) * P_RandomNegPos();
    if (attack->accuracy_slope > 0)
      slope += attack->accuracy_slope * (P_RandomNegPos() / 255.0);
  }

  P_SetMobjDirAndSpeed(projectile, angle, slope, projectile->speed);
  CheckMissileSpawn(projectile);

  return projectile;
}

//
// P_ActLaunchSmartProjectile
//
// This procedure has the same effect as
// LaunchProjectile, but it calculates a point where the target
// and missile will intersect.  This comes from the fact that to shoot
// something, you have to aim slightly ahead of it.  It will also put
// an end to circle-strafing.  :-)
//
// -KM- 1998/10/29
// -KM- 1998/12/16 Fixed it up.  Works quite well :-)
//
static void LaunchSmartProjectile(mobj_t * source, mobj_t * target, 
    const mobjinfo_t * type)
{
  float_t a, b, c;
  float_t t1 = -1, t2 = -1, t;
  float_t dx, dy;
  float_t mx, my;
  float_t s;

  // -AJA- prevent possible crashes.  Prolly not the best solution,
  //       more like a baid-aid solution.
  if (!target)
    target = P_MapTargetTheory(source);

  mx = target->mom.x;
  my = target->mom.y;

  dx = source->x - target->x;
  dy = source->y - target->y;

  s = type->speed;
  if (level_flags.fastparm)
    s *= type->fast;

  a = mx * mx + my * my - s * s;
  b = 2 * (dx * mx + dy * my);
  c = dx * dx + dy * dy;

  if (a && ((b * b - 4 * a * c) >= 0))
  {
    t1 = -b + sqrt(b * b - 4 * a * c);
    t1 /= 2 * a;

    t2 = -b - sqrt(b * b - 4 * a * c);
    t2 /= 2 * a;
  }

  if (t1 < 0)
    t1 = t2;

  if (t2 < 0)
    t2 = t1;

  t = t1 < t2 ? t1 : t2;

  if (t > 0)
  {
    static mobj_t spot;
    mobj_t *projectile;

    spot.x = target->x + mx * t;
    spot.y = target->y + my * t;
    
    // -KM- 1999/01/31 Calculate the target for grenades.
    if (type->flags & MF_NOGRAVITY)
      spot.z = target->z + 2 * target->height / 3 - source->currentattack->height;
    else
      spot.z = target->z - source->z +
          target->subsector->sector->props.gravity * t * t / 16.0;

    spot.height = 0;
    spot.extendedflags = EF_DUMMYMOBJ;

    projectile = LaunchProjectile(source, &spot, type);

    if (projectile)
      source->angle = projectile->angle;

    return;
  }

  LaunchProjectile(source, target, type);
}

//
// P_ActMissileContact
//
// Called by PIT_CheckRelThing when a missile comes into
// contact with another object. Placed here with
// the other missile code for cleaner code.
//
// Returns true if damage was done.
//
// -ACB- 1998/08/10
//
boolean_t P_ActMissileContact(mobj_t * object, mobj_t * objecthit)
{
  mobj_t *source;
  const damage_t *damtype;
  float_t damage;

  source = object->source;

  if (source)
  {
    if (source->info == objecthit->info)
    {
      if (!(objecthit->extendedflags & EF_DISLOYALTYPE))
        return false;
    }

    if (object->currentattack != NULL &&
        !(objecthit->extendedflags & EF_OWNATTACKHURTS))
    {
      if (object->currentattack == objecthit->info->rangeattack)
        return false;
      if (object->currentattack == objecthit->info->closecombat)
        return false;
    }
  }

  // check for immunity against the attack
  if (object->currentattack && BITSET_EMPTY ==
      (object->currentattack->attack_class & ~objecthit->info->immunity))
  {
    return false;
  }

  // support for "tunnelling" missiles, which should only do damage at
  // the first impact.
  if (object->extendedflags & EF_TUNNEL)
  {
    // this hash is very basic, but should work OK
    unsigned long hash = (unsigned long)objecthit;

    if (object->tunnel_hash[0] == hash || object->tunnel_hash[1] == hash)
      return false;
    
    object->tunnel_hash[0] = object->tunnel_hash[1];
    object->tunnel_hash[1] = hash;
  }

  // transitional hack
  if (object->currentattack)
    damtype = &object->currentattack->damage;
  else
    damtype = &object->info->damage;

  DAMAGE_COMPUTE(damage, damtype);

  if (!damage)
  {
#ifdef DEVELOPERS
    L_WriteDebug("%s missile did zero damage.\n", 
      object->info->ddf.name);
#endif
    return false;
  }

  P_DamageMobj(objecthit, object, object->source, damage, damtype);
  return true;
}

//
// P_ActBulletContact
//
// Called by PTR_ShootTraverse when a bullet comes into contact with
// another object.  Needed so that the "DISLOYAL" special will behave
// in the same manner for bullets as for missiles.  Note: also used
// for close combat attacks.
//
// Returns true if damage was done.
//
// -AJA- 2000/02/17: written.
//
boolean_t P_ActBulletContact(mobj_t * object, mobj_t * objecthit, 
    float_t damage, const damage_t *damtype)
{
  if (object->info == objecthit->info)
  {
    if (! (objecthit->extendedflags & EF_DISLOYALTYPE))
      return false;
  }

  if (object->currentattack != NULL &&
      !(objecthit->extendedflags & EF_OWNATTACKHURTS))
  {
    if (object->currentattack == objecthit->info->rangeattack)
      return false;
    if (object->currentattack == objecthit->info->closecombat)
      return false;
  }

  // check for immunity against the attack
  if (object->currentattack && BITSET_EMPTY ==
      (object->currentattack->attack_class & ~objecthit->info->immunity))
  {
    return false;
  }

  if (!damage)
  {
#ifdef DEVELOPERS
    L_WriteDebug("%s's shoot/combat attack did zero damage.\n", 
      object->info->ddf.name);
#endif
    return false;
  }

  P_DamageMobj(objecthit, object, object, damage, damtype);
  return true;
}

//
// P_ActCreateSmokeTrail
//
// Just spawns smoke behind an mobj: the smoke is
// risen by giving it z momentum, in order to
// prevent the smoke appearing uniform (which obviously
// does not happen), the number of tics that the smoke
// mobj has is "randomly" reduced, although the number
// of tics never gets to zero or below.
//
// -ACB- 1998/08/10 Written
// -ACB- 1999/10/01 Check thing's current attack has a smoke projectile
//
void P_ActCreateSmokeTrail(mobj_t * projectile)
{
  mobj_t *smoke;
  float_t z;
  const attacktype_t *attack;

  attack = projectile->currentattack;

  if (attack == NULL)
    return;

  if (attack->puff == NULL)
  {
    M_WarnError("P_ActCreateSmokeTrail: attack %s has no PUFF object\n",
        attack->ddf.name);
    return;
  }
  
  // -AJA- 1999/12/07: center puff vertically
  z = MO_MIDZ(projectile) - attack->puff->height/2;

  // spawn a puff of smoke behind the rocket
  smoke = P_MobjCreateObject(
      projectile->x - projectile->mom.x, 
      projectile->y - projectile->mom.y, z, attack->puff);

  smoke->mom.z = smoke->info->float_speed;
  smoke->tics -= M_Random() & 3;

  if (smoke->tics < 1)
    smoke->tics = 1;
}

//
// P_ActRandomHomingProjectile
//
// This projectile will alter its course to intercept its
// target, if is possible for this procedure to be called
// and nothing results because of a chance that the
// projectile will not chase its target.
//
// As this code is based on the revenant tracer, it did use
// a bit check on the current gametic - which was why every so
// often a revenant fires a missile straight and not one that
// homes in on its target: If the gametic has bits 1+2 on
// (which boils down to 1 in every 4 tics), the trick in this
// is that - in conjuntion with the tic count for the
// tracing object's states - the tracing will always fail or
// pass the check: if it passes first time, it will always
// pass and vice versa. The problem with this was two fold:
// demos will go out of sync if the starting gametic if different
// from when the demo was recorded (which admittly is easily
// fixable), the second is that for someone designing a new
// tracing projectile it would be more than a bit confusing to
// joe "dooming" public.
//
// The new system that affects the original gameplay slightly is
// to get a random chance of the projectile not homing in on its
// target and working this out first time round, the test result
// is recorded (in the form of the presence or lack of the
// extended flag: EF_NOTRACE) and everytime this procedure is
// called, it will check for the flag and act accordingly.
//
// Chance calculated is in percentage terms. The procedure below
// this one gives the original gameplay.
//
// -ACB- 1998/08/10
//
void P_ActRandomHomingProjectile(mobj_t * projectile)
{
  angle_t exact;
  float_t slope;
  mobj_t *destination;
  const attacktype_t *attack;

  attack = projectile->currentattack;

  if (attack == NULL)
    return;

  if (projectile->extendedflags & EF_NOTRACE)
    return;

  if (projectile->extendedflags & EF_FIRSTCHECK)
  {
    // if either value is zero, the projectile will trace
    if (P_RandomTest(attack->notracechance))
    {
      projectile->extendedflags |= EF_NOTRACE;
      return;
    }

    projectile->extendedflags &= ~EF_FIRSTCHECK;
  }

  if (attack->flags & AF_TraceSmoke)
    P_ActCreateSmokeTrail(projectile);

  destination = projectile->target;

  if (!destination || destination->health <= 0)
    return;

  // change angle
  exact = R_PointToAngle(projectile->x, projectile->y,
      destination->x, destination->y);

  if (exact != projectile->angle)
  {
    if (exact - projectile->angle > 0x80000000)
    {
      projectile->angle -= TRACEANGLE;

      if (exact - projectile->angle < 0x80000000)
        projectile->angle = exact;
    }
    else
    {
      projectile->angle += TRACEANGLE;

      if (exact - projectile->angle > 0x80000000)
        projectile->angle = exact;
    }
  }

  exact = projectile->angle;
  projectile->mom.x = projectile->speed * M_Cos(exact);
  projectile->mom.y = projectile->speed * M_Sin(exact);

  // change slope
  slope = P_ApproxSlope(destination->x - projectile->x,
      destination->y - projectile->y,
      MO_MIDZ(destination) - projectile->z);

  slope *= projectile->speed;

  if (slope < projectile->mom.z)
    projectile->mom.z -= 0.125;
  else
    projectile->mom.z += 0.125;
}

//
// P_ActFixedHomingProjectile
//
// This projectile will alter its course to intercept its
// target, if is possible for this procedure to be called
// and nothing results because of a chance that the
// projectile will not chase its target.
//
// Same as above, but more this is for purists; the above
// procedure gives a random chance; the one here is based on
// a modulas result from gametic (subtracting the levelstarttic
// to make sure the results are the same when playing back the
// demos: it is possible for gametic to be different when playing
// a demo), the test result is recorded (in the form of the
// presence or lack of the extended flag: EF_NOTRACE) and
// everytime this procedure is called, it will check for the
// flag and act accordingly. Although a Purist function, it does
// allow a dual firing object thats launchs two tracer-types
// in the same tic to generate either tracers or normal missiles.
//
// -ACB- 1998/08/20
//
void P_ActFixedHomingProjectile(mobj_t * projectile)
{
  angle_t exact;
  float_t slope;
  mobj_t *destination;
  const attacktype_t *attack;

  attack = projectile->currentattack;

  if (attack == NULL)
    return;

  if (projectile->extendedflags & EF_NOTRACE)
    return;

  if (projectile->extendedflags & EF_FIRSTCHECK)
  {
    projectile->extendedflags &= ~EF_FIRSTCHECK;

    if (P_RandomTest(attack->notracechance))
    {
      projectile->extendedflags |= EF_NOTRACE;
      return;
    }
  }

  if (attack->flags & AF_TraceSmoke)
    P_ActCreateSmokeTrail(projectile);

  destination = projectile->target;

  if (!destination || destination->health <= 0)
    return;

  // change angle
  exact = R_PointToAngle(projectile->x, projectile->y,
      destination->x, destination->y);

  if (exact != projectile->angle)
  {
    if (exact - projectile->angle > 0x80000000)
    {
      projectile->angle -= TRACEANGLE;

      if (exact - projectile->angle < 0x80000000)
        projectile->angle = exact;
    }
    else
    {
      projectile->angle += TRACEANGLE;

      if (exact - projectile->angle > 0x80000000)
        projectile->angle = exact;
    }
  }

  projectile->mom.x = projectile->speed * M_Cos(exact);
  projectile->mom.y = projectile->speed * M_Sin(exact);

  // change slope
  slope = P_ApproxSlope(destination->x - projectile->x,
      destination->y - projectile->y,
      MO_MIDZ(destination) - projectile->z);
  
  slope *= projectile->speed;

  if (slope < projectile->mom.z)
    projectile->mom.z -= 0.125;
  else
    projectile->mom.z += 0.125;
}

//
// P_ActHomeToSpot
//
// This projectile will alter its course to intercept its
// target, or explode if it has reached it.  Used by the bossbrain
// cube.
//
// -AJA- 1999/09/15: written.
//
void P_ActHomeToSpot(mobj_t * projectile)
{
  float_t dx, dy, dz;
  float_t angle, slope;
  float_t ck_radius, ck_height;

  mobj_t *target = projectile->target;
  
  if (!target)
  {
    P_MobjExplodeMissile(projectile);
    return;
  }

  dx = target->x - projectile->x;
  dy = target->y - projectile->y;
  dz = target->z - projectile->z;

  ck_radius = target->radius + projectile->radius + 2;
  ck_height = target->height + projectile->height + 2;
  
  // reached target ?
  if (fabs(dx) <= ck_radius && fabs(dy) <= ck_radius && fabs(dz) <= ck_height)
  {
    P_MobjExplodeMissile(projectile);
    return;
  }

  // calculate new angles
  angle = R_PointToAngle(0, 0, dx, dy);
  slope = P_ApproxSlope(dx, dy, dz);
  
  P_SetMobjDirAndSpeed(projectile, angle, slope, projectile->speed);
}

//
// P_ActLaunchOrderedSpread
//
// Due to the unique way of handling that the mancubus fires, it is necessary
// to write a single procedure to handle the firing. In real terms it amounts
// to a glorified hack; The table holds the angle modifier and the choice of
// whether the firing object or the projectile is affected. This procedure
// should NOT be used for players as it will alter the player's mobj, bypassing
// the normal player controls; The only reason for its existance is to maintain
// the original mancubus behaviour. Although it is possible to make this generic,
// the benefits of doing so are minimal. Purist function....
//
// -ACB- 1998/08/15
//
void P_ActLaunchOrderedSpread(mobj_t * object)
{
  // left side = angle modifier
  // right side = object or projectile (true for object).
  static int spreadorder[] =
  {
      (ANG90 / 8), true,
      (ANG90 / 8), false,
      -(ANG90 / 8), true,
      -(ANG90 / 4), false,
      -(ANG90 / 16), false,
      (ANG90 / 16), false
  };

  mobj_t *projectile;
  angle_t angle;
  int count;
  const attacktype_t *attack;

  attack = object->currentattack;

  if (attack == NULL)
    return;

  count = object->spreadcount;

  if (count < 0 || count > 12)
    count = object->spreadcount = 0;

  // object or projectile? - if true is the object, else it is the projectile
  if (spreadorder[count + 1])
  {
    object->angle += spreadorder[count];
    LaunchProjectile(object, object->target, attack->atk_mobj);
  }
  else
  {
    projectile = LaunchProjectile(object, object->target,
        attack->atk_mobj);

    if (projectile == NULL)
      return;

    projectile->angle += spreadorder[count];
    angle = projectile->angle;

    projectile->mom.x = projectile->speed * M_Cos(angle);
    projectile->mom.y = projectile->speed * M_Sin(angle);
  }

  object->spreadcount += 2;
}

//
// P_ActLaunchRandomSpread
//
// This is a the generic function that should be used for a spreader like
// mancubus, although its random nature would certainly be a change to the
// ordered method used now. The random number is bit shifted to the right
// and then the ANG90 is divided by it, the first bit of the RN is checked
// to detemine if the angle is change is negative or not (approx 50% chance).
// The result is the modifier for the projectile's angle.
//
// -ACB- 1998/08/15
//
void P_ActLaunchRandomSpread(mobj_t * object)
{
  mobj_t *projectile;
  angle_t spreadangle;
  angle_t angle;
  int i;

  if (object->currentattack == NULL)
    return;

  projectile = LaunchProjectile(object, object->target,
      object->currentattack->atk_mobj);

  if (projectile == NULL)
    return;

  i = P_Random() % 128;

  if (i >> 1)
  {
    spreadangle = (ANG90 / (i >> 1));

    if (i & 1)
      spreadangle -= spreadangle << 1;

    projectile->angle += spreadangle;
  }

  angle = projectile->angle;

  projectile->mom.x = projectile->speed * M_Cos(angle);
  projectile->mom.y = projectile->speed * M_Sin(angle);
}

//-------------------------------------------------------------------
//-------------------LINEATTACK ATTACK ROUTINES-----------------------
//-------------------------------------------------------------------

// -KM- 1998/11/25 Added uncertainty to the z component of the line.
static void ShotAttack(mobj_t * object)
{
  int i;
  int count;
  angle_t angle;
  angle_t objangle;
  float_t slope, objslope;
  float_t damage;
  const attacktype_t *attack;

  attack = object->currentattack;
  count = attack->count;

  if (! attack)
    return;

  // -ACB- 1998/09/05 Remember to use the object angle, fool!
  objangle = object->angle;
  if (object->player && (! object->target || 
      (object->target->extendedflags & EF_DUMMYMOBJ)))
    objslope = object->vertangle;
  else
    objslope = P_AimLineAttack(object, objangle, MISSILERANGE);

  if (attack->sound)
    S_StartSound(object, attack->sound);

  // -AJA- 1999/09/10: apply the attack's angle offsets.
  objangle -= attack->angle_offset;
  objslope += attack->slope_offset;
  
  for (i = 0; i < count; i++)
  {
    angle = objangle;
    slope = objslope;

    // is the attack not accurate?
    if (!object->player || object->player->refire > 0)
    {
      if (attack->accuracy_angle > 0)
        angle += (attack->accuracy_angle >> 8) * P_RandomNegPos();
      if (attack->accuracy_slope > 0)
        slope += attack->accuracy_slope * (P_RandomNegPos() / 255.0);
    }

    DAMAGE_COMPUTE(damage, &attack->damage);

    P_LineAttack(object, angle, MISSILERANGE, slope, damage,
        &attack->damage, attack->puff);
  }
}

// -KM- 1998/11/25 BFG Spray attack.  Must be used from missiles.
//   Will do a BFG spray on every monster in sight.
static void SprayAttack(mobj_t * mo)
{
  int i;
  angle_t an;
  const attacktype_t *attack;
  mobj_t *m;
  float_t damage;

  attack = mo->currentattack;

  if (! attack)
    return;

  // offset angles from its attack angle
  for (i = 0; i < 40; i++)
  {
    an = mo->angle - ANG90 / 2 + (ANG90 / 40) * i;

    // mo->source is the originator (player)
    //  of the missile
    P_AimLineAttack(mo->source ? mo->source : mo, an, attack->range);

    if (!linetarget)
      continue;

    m = P_MobjCreateObject(linetarget->x, linetarget->y,
        linetarget->z + linetarget->height / 4,
        attack->atk_mobj);

    P_MobjSetTarget(m, mo->target);

    // check for immunity against the attack
    if (BITSET_EMPTY == (attack->attack_class & ~linetarget->info->immunity))
    {
      return;
    }

    DAMAGE_COMPUTE(damage, &attack->damage);

    if (damage)
      P_DamageMobj(linetarget, NULL, mo->source, damage, &attack->damage);
  }
}

//-------------------------------------------------------------------
//--------------------TRACKER HANDLING ROUTINES----------------------
//-------------------------------------------------------------------

//
// A Tracker is an object that follows its target, by being on top of
// it. This is the attack style used by an Arch-Vile. The first routines
// handle the tracker itself, the last two are called by the source of
// the tracker.
//

//
// P_ActTrackerFollow
//
// Called by the tracker to follow its target.
//
// -ACB- 1998/08/22
//
void P_ActTrackerFollow(mobj_t * tracker)
{
  mobj_t *destination;
  angle_t angle;

  destination = tracker->target;

  if (!destination || !tracker->source)
    return;

  // check for dummy object
  if (destination->extendedflags & EF_DUMMYMOBJ)
    return;

  // Can the source of the tracker, see the destination target?
  if (!P_CheckSight(tracker->source, destination))
    return;

  angle = destination->angle;

  P_ChangeThingPosition(tracker,
      destination->x + 24 * M_Cos(angle),
      destination->y + 24 * M_Sin(angle),
      destination->z);
}

//
// P_ActTrackerActive
//
// Called by the tracker to make its active sound: also tracks
//
// -ACB- 1998/08/22
//
void P_ActTrackerActive(mobj_t * tracker)
{
  if (tracker->info->activesound)
    S_StartSound(tracker, tracker->info->activesound);

  P_ActTrackerFollow(tracker);
}

//
// P_ActTrackerStart
//
// Called by the tracker to make its launch (see) sound: also tracks
//
// -ACB- 1998/08/22
//
void P_ActTrackerStart(mobj_t * tracker)
{
  if (tracker->info->seesound)
    S_StartSound(tracker, tracker->info->seesound);

  P_ActTrackerFollow(tracker);
}

//
// LaunchTracker
//
// This procedure starts a tracking object off and links
// the tracker and the object together.
//
// -ACB- 1998/08/22
//
static void LaunchTracker(mobj_t * object)
{
  mobj_t *tracker;
  mobj_t *target;
  const attacktype_t *attack;

  attack = object->currentattack;
  target = object->target;

  if (!attack || !target || (target->extendedflags & EF_DUMMYMOBJ))
    return;

  tracker = P_MobjCreateObject(target->x, target->y, target->z,
      attack->atk_mobj);

  // link the tracker to the object
  P_MobjSetTracer(object, tracker);

  // tracker source is the object
  P_MobjSetRealSource(tracker, object);

  // tracker's target is the object's target
  P_MobjSetTarget(tracker, target);

  P_ActTrackerFollow(tracker);
}

//
// P_ActEffectTracker
//
// Called by the object that launched the tracker to
// cause damage to its target and a radius attack
// (explosion) at the location of the tracker.
//
// -ACB- 1998/08/22
//
void P_ActEffectTracker(mobj_t * object)
{
  mobj_t *tracker;
  mobj_t *target;
  const attacktype_t *attack;
  angle_t angle;
  float_t damage;

  if (!object->target || !object->currentattack)
    return;

  attack = object->currentattack;
  target = object->target;

  if (attack->flags & AF_FaceTarget)
    P_ActFaceTarget(object);

  if (attack->flags & AF_NeedSight)
  {
    if (!P_CheckSight(object, target))
      return;
  }

  if (attack->sound)
    S_StartSound(object, attack->sound);

  angle = object->angle;
  tracker = object->tracer;

  DAMAGE_COMPUTE(damage, &attack->damage);

  if (damage)
    P_DamageMobj(target, object, object, damage, &attack->damage);
#ifdef DEVELOPERS
  else
    L_WriteDebug("%s + %s attack has zero damage\n",
        object->info->ddf.name, tracker->info->ddf.name);
#endif

  // -ACB- 2000/03/11 Check for zero mass
  if (target->info->mass)
    target->mom.z = 1000 / target->info->mass;
  else
    target->mom.z = 2000;

  if (!tracker)
    return;

  // move the tracker between the object and the object's target

  P_ChangeThingPosition(tracker,
      target->x - 24 * M_Cos(angle),
      target->y - 24 * M_Sin(angle),
      target->z);

#ifdef DEVELOPERS
  if (!tracker->info->damage.nominal)
    L_WriteDebug("%s + %s explosion has zero damage\n",
        object->info->ddf.name, tracker->info->ddf.name);
#endif

  DAMAGE_COMPUTE(damage, &tracker->info->damage);

  P_RadiusAttack(tracker, object, damage, damage,
      &tracker->info->damage, false);
}

//-----------------------------------------------------------------
//--------------------BOSS HANDLING PROCEDURES---------------------
//-----------------------------------------------------------------

static void ShootToSpot(mobj_t * object)
{
  // Note: using a static int here for better randomness.
  static int current_spot = 0;

  if (! object->currentattack)
    return;

  if (brain_spots.number == 0)
  {
    if (! object->info->spitspot)
    {
      M_WarnError("Thing [%s] used SHOOT_TO_SPOT attack, but has no "
          "SPIT_SPOT\n", object->info->ddf.name);
      return;
    }

    P_LookForShootSpots(object->info->spitspot);
  }

  DEV_ASSERT2(brain_spots.targets);
  DEV_ASSERT2(brain_spots.number > 0);

  current_spot += P_Random();
  current_spot %= brain_spots.number;
  
  LaunchProjectile(object, brain_spots.targets[current_spot],
      object->currentattack->atk_mobj);
}

//-------------------------------------------------------------------
//-------------------OBJECT-SPAWN-OBJECT HANDLING--------------------
//-------------------------------------------------------------------

//
// P_ActObjectSpawning
//
// An Object spawns another object and is spawned in the state specificed
// by attack->objinitstate. The procedure is based on the A_PainShootSkull
// which is the routine for shooting skulls from a pain elemental. In
// this the object being created is decided in the attack. This
// procedure also used the new blocking line check to see if
// the object is spawned across a blocking line, if so the procedure
// terminates.
//
// -ACB- 1998/08/23
//
static void ObjectSpawning(mobj_t * object, angle_t angle)
{
  float_t spawnx;
  float_t spawny;
  float_t spawnz;
  float_t slope;
  float_t prestep;
  const attacktype_t *attack;
  const mobjinfo_t *shoottype;
  mobj_t *newobject;

  attack = object->currentattack;
  if (! attack)
    return;

  shoottype = attack->spawnedobj;

  if (! shoottype)
    I_Error("Object [%s] uses spawning attack [%s], but no object "
        "specified.\n", object->info->ddf.name, attack->ddf.name);

  // -AJA- 1999/09/10: apply the angle offset of the attack.
  angle -= attack->angle_offset;
  slope = object->vertangle + attack->slope_offset;
  
  if (attack->flags & AF_PrestepSpawn)
  {
    prestep = 4 + 1.5 * object->radius + shoottype->radius;
    spawnx = object->x + prestep * M_Cos(angle);
    spawny = object->y + prestep * M_Sin(angle);
  }
  else
  {
    spawnx = object->x;
    spawny = object->y;
  }

  spawnz = object->z + attack->height;

  newobject = P_MobjCreateObject(spawnx, spawny, spawnz, shoottype);

  // Blocking line detected between object and spawnpoint?
  if (P_MapCheckBlockingLine(object, newobject))
  {
    // -KM- 1999/01/31 Explode objects over remove them.
    // -AJA- 2000/02/01: Remove now the default.

    if (attack->flags & AF_KillFailedSpawn)
      P_KillMobj(object, newobject, NULL);
    else
      P_RemoveMobj(newobject);

    return;
  }

  if (attack->sound)
    S_StartSound(object, attack->sound);

  // If the object cannot move from its position, remove it or kill it.
  if (!P_TryMove(newobject, newobject->x, newobject->y))
  {
    if (attack->flags & AF_KillFailedSpawn)
      P_KillMobj(object, newobject, NULL);
    else
      P_RemoveMobj(newobject);

    return;
  }

  P_MobjSetTarget(newobject, object->target);
  P_MobjSetSupportObj(newobject, object);
  newobject->side = object->side;

  // -AJA- 1999/09/25: Set the initial direction & momentum when
  //       the ANGLED_SPAWN attack special is used.
  if (attack->flags & AF_AngledSpawn)
    P_SetMobjDirAndSpeed(newobject, angle, slope, attack->assault_speed);

  P_SetMobjStateDeferred(newobject, attack->objinitstate, 0);
}

//
// P_ActObjectTripleSpawn
//
// Spawns three objects at 90, 180 and 270 degrees. This is essentially
// another purist function to support the death sequence of the Pain
// elemental. However it could be used as in conjunction with radius
// triggers to generate a nice teleport spawn invasion.
//
// -ACB- 1998/08/23 (I think....)
//

static void ObjectTripleSpawn(mobj_t * object)
{
  ObjectSpawning(object, object->angle + ANG90);
  ObjectSpawning(object, object->angle + ANG180);
  ObjectSpawning(object, object->angle + ANG270);
}

//-------------------------------------------------------------------
//-------------------SKULLFLY HANDLING ROUTINES----------------------
//-------------------------------------------------------------------

//
// P_ActSkullFlyAssault
//
// This is the attack procedure for objects that launch themselves
// at their target like a missile.
//
// -ACB- 1998/08/16
//
static void SkullFlyAttack(mobj_t * object)
{
  mobj_t *destination;
  float_t slope;
  float_t speed;
  sfx_t *sound;

  if (!object->currentattack)
    return;

  if (!object->target)
  {
    // -AJA- 2000/09/29: fix for the zombie lost soul bug
    // -AJA- 2000/10/22: monsters only !  Don't stuff up gibs/missiles.
    if (object->extendedflags & EF_MONSTER)
      object->flags |= MF_SKULLFLY;
    return;
  }

  speed = object->currentattack->assault_speed;

  // -KM- 1999/01/31 Fix skulls in nightmare mode
  if (level_flags.fastparm)
    speed *= object->info->fast;

  destination = object->target;
  sound = object->currentattack->initsound;
  object->flags |= MF_SKULLFLY;

  if (sound)
    S_StartSound(object, sound);

  slope = P_ApproxSlope(destination->x - object->x,
      destination->y - object->y,
      MO_MIDZ(destination) - object->z);

  P_SetMobjDirAndSpeed(object, object->angle, slope, speed);
}

//
// P_ActSlammedIntoObject
//
// Used when a flying object hammers into another object when on the
// attack. Replaces the code in PIT_Checkthing.
//
// -ACB- 1998/07/29: Written
//
// -AJA- 1999/09/12: Now uses P_SetMobjStateDeferred, since this
//                   routine can be called by TryMove/PIT_CheckRelThing.
//
void P_ActSlammedIntoObject(mobj_t * object, mobj_t * objecthit)
{
  sfx_t *sound;
  float_t damage;

  if (object->currentattack)
  {
    if (objecthit != NULL)
    {
      // -KM- 1999/01/31 Only hurt shootable objects...
      if (objecthit->flags & MF_SHOOTABLE)
      {
        DAMAGE_COMPUTE(damage, &object->currentattack->damage);

        P_DamageMobj(objecthit, object, object, damage,
            &object->currentattack->damage);
      }
    }

    sound = object->currentattack->sound;
    if (sound)
      S_StartSound(object, sound);
  }

  object->flags &= ~MF_SKULLFLY;
  object->mom.x = object->mom.y = object->mom.z = 0;

  P_SetMobjStateDeferred(object, object->info->idle_state, 0);
}

//
// P_ActUseThing
//
// Called when this thing is attempted to be used (i.e. by pressing
// the spacebar near it) by the player.  Returns true if successfully
// used, or false if other things should be checked.
//
// -AJA- 2000/02/17: written.
//
boolean_t P_ActUseThing(mobj_t * user, mobj_t * thing, float_t open_bottom,
    float_t open_top)
{
  // item is disarmed ?
  if (!(thing->flags & MF_TOUCHY))
    return false;
  
  // can be reached ?
  open_top    = MIN(open_top, thing->z + thing->height);
  open_bottom = MAX(open_bottom, thing->z);
  
  if (user->z >= open_top || 
      (user->z + user->height + USE_Z_RANGE < open_bottom))
    return false;
  
  // OK, disarm and put into touch states  
  DEV_ASSERT2(thing->info->touch_state > 0);

  thing->flags &= ~MF_TOUCHY;
  P_SetMobjStateDeferred(thing, thing->info->touch_state, 0);

  return true;
}

//
// P_ActTouchyContact
//
// Used whenever a thing comes into contact with a TOUCHY object.
//
// -AJA- 1999/08/21 written.
// -AJA- 1999/09/12: Now uses P_SetMobjStateDeferred, since this
//       routine can be called by TryMove/PIT_CheckRelThing.
//
void P_ActTouchyContact(mobj_t *touchy, mobj_t *victim)
{
  // dead thing touching. Can happen with a sliding player corpse.
  if (victim->health <= 0)
    return;

  // don't harm the grenadier...
  if (touchy->source == victim)
    return;

  P_MobjSetTarget(touchy, victim);
  touchy->flags &= ~MF_TOUCHY;  // disarm

  if (touchy->info->touch_state)
    P_SetMobjStateDeferred(touchy, touchy->info->touch_state, 0);
  else
    P_MobjExplodeMissile(touchy);
}

//
// P_ActTouchyRearm    
// P_ActTouchyDisarm
//
// -AJA- 1999/08/22 written.
//
void P_ActTouchyRearm(mobj_t * touchy)
{
  touchy->flags |= MF_TOUCHY;
}

void P_ActTouchyDisarm(mobj_t * touchy)
{
  touchy->flags &= ~MF_TOUCHY;
}

//
// P_ActBounceRearm
// P_ActBounceDisarm
//
// -AJA- 1999/10/18 written.
//
void P_ActBounceRearm(mobj_t * mo)
{
  mo->extendedflags &= ~EF_JUSTBOUNCED;
}

void P_ActBounceDisarm(mobj_t * mo)
{
  mo->extendedflags |= EF_JUSTBOUNCED;
}

//
// P_ActDropItem
//
// -AJA- 2000/10/20: added.
//
void P_ActDropItem(mobj_t * mo)
{
  const mobjinfo_t *info = mo->info->dropitem;
  mobj_t *item;

  float_t dx, dy;

  if (mo->state && mo->state->action_par)
    info = (const mobjinfo_t *) mo->state->action_par;

  if (! info)
  {
    M_WarnError("P_ActDropItem: %s specifies no item to drop.\n", 
        mo->info->ddf.name);
    return;
  }

  // unlike normal drops, these ones are displaced randomly

  dx = P_RandomNegPos() * mo->info->radius / 255.0;
  dy = P_RandomNegPos() * mo->info->radius / 255.0;

  item = P_MobjCreateObject(mo->x + dx, mo->y + dy, mo->floorz, info);

  // -ES- 1998/07/18 NULL check to prevent crashing
  if (item)
  {
    item->flags |= MF_DROPPED;
    item->flags &= ~MF_SOLID;

    item->angle = mo->angle;

    // allow respawning
    item->spawnpoint.x = item->x;
    item->spawnpoint.y = item->y;
    item->spawnpoint.z = item->z;
    item->spawnpoint.angle = item->angle;
    item->spawnpoint.slope = item->vertangle;
    item->spawnpoint.info  = info;
    item->spawnpoint.flags = 0;
  }
}

//
// P_ActPathCheck
//
// Checks if the creature is a path follower, and if so enters the
// meander states.
//
// -AJA- 2000/02/17: wrote this & PathFollow.
//
void P_ActPathCheck(mobj_t * mo)
{
  if (! mo->path_trigger || ! mo->info->meander_state)
    return;
 
  P_SetMobjStateDeferred(mo, mo->info->meander_state, 0);

  mo->movedir = mo->movecount = 0;
}

//
// P_ActPathFollow
//
// For path-following creatures (spawned via RTS), makes the creature
// follow the path by trying to get to the next node.
//
void P_ActPathFollow(mobj_t * mo)
{
  float_t dx, dy;
  angle_t diff;

  if (! mo->path_trigger)
    return;

  if (RAD_CheckReachedTrigger(mo))
  {
    mo->movedir = 0;
    return;
  }

  // maybe reached the end ?
  if (! mo->path_trigger)
    return;

  dx = mo->path_trigger->x - mo->x;
  dy = mo->path_trigger->y - mo->y;

  diff = mo->angle - R_PointToAngle(0, 0, dx, dy);

  // movedir value: 
  //   0 for slow turning.
  //   1 for fast turning.
  //   2 for walking.
  //   3 for evasive maneouvres.
  
  if (mo->movedir < 2)
  {
    angle_t step = (mo->movedir == 1) ? ANG1*30 : ANG1*7;
    
    // if not facing the path node, turn towards it
    if (diff > ANG1*15 && diff < ANG180)
    {
      mo->angle -= P_Random() * (step >> 8);
      return;
    }
    else if (diff >= ANG180 && diff < ANG1*345U)
    {
      mo->angle += P_Random() * (step >> 8);
      return;
    }

    mo->movedir = 2;
  }

  if (mo->movedir == 2)
  {
    if (diff < ANG1*30)
      mo->angle -= ANG1;
    else if (diff >= ANG1*330U)
      mo->angle += ANG1;
    else
      mo->movedir = 0;

    if (! P_Move(mo, true))
    {
      mo->movedir = 3;
      mo->angle = P_Random() << (ANGLEBITS - 8);
      mo->movecount = 1 + (P_Random() & 7);
    }
    return;
  }

  // make evasive maneouvres
  mo->movecount--;
  if (mo->movecount <= 0)
  {
    mo->movedir = 1;
    return;
  }

  P_Move(mo, true);
}

//-------------------------------------------------------------------
//--------------------ATTACK HANDLING PROCEDURES---------------------
//-------------------------------------------------------------------

//
// P_DoAttack
//
// When an object goes on the attack, it current attack is handled here;
// the attack type is discerned and the assault is launched.
//
// -ACB- 1998/08/07
//
void P_DoAttack(mobj_t * object)
{
  const attacktype_t *attack;
  mobj_t *target;
  float_t damage;

  attack = object->currentattack;
  target = object->target;

  DEV_ASSERT2(attack);

  switch (attack->attackstyle)
  {
    case ATK_CLOSECOMBAT:
    {
      // -KM- 1998/12/21 Use Line attack so bullet puffs are spawned.
      float_t slope;

      if (!DecideMeleeAttack(object, attack))
      {
        P_LineAttack(object, object->angle, attack->range, object->vertangle, 
            1, NULL, attack->puff);
        return;
      }

      if (attack->sound)
        S_StartSound(object, attack->sound);

      DAMAGE_COMPUTE(damage, &attack->damage);

      // -KM- 1998/11/25 Berserk ability

      if (object->player && object->player->powers[PW_Berserk] > 0)
        damage *= 10.0;

      slope = P_AimLineAttack(object, object->angle, attack->range);

      P_LineAttack(object, object->angle, attack->range, slope,
          damage, &attack->damage, attack->puff);
      break;
    }

    case ATK_PROJECTILE:
    {
      LaunchProjectile(object, target, attack->atk_mobj);
      break;
    }

    case ATK_SMARTPROJECTILE:
    {
      LaunchSmartProjectile(object, target, attack->atk_mobj);
      break;
    }

    case ATK_RANDOMSPREAD:
    {
      P_ActLaunchRandomSpread(object);
      break;
    }

    case ATK_SHOOTTOSPOT:
    {
      ShootToSpot(object);
      break;
    }

    case ATK_SHOT:
    {
      ShotAttack(object);
      break;
    }

    case ATK_SKULLFLY:
    {
      SkullFlyAttack(object);
      break;
    }

    case ATK_SPAWNER:
    {
      ObjectSpawning(object, object->angle);
      break;
    }

    case ATK_SPREADER:
    {
      P_ActLaunchOrderedSpread(object);
      break;
    }

    case ATK_TRACKER:
    {
      LaunchTracker(object);
      break;
    }

    case ATK_TRIPLESPAWNER:
    {
      ObjectTripleSpawn(object);
      break;
    }

    // -KM- 1998/11/25 Added spray attack
    case ATK_SPRAY:
    {
      SprayAttack(object);
      break;
    }

    default:  // THIS SHOULD NOT HAPPEN
    {
#ifdef DEVELOPERS
      I_Error("P_DoAttack: %s has an unknown attack type.\n", object->info->ddf.name);
#endif
      break;
    }
  }
}

//
// P_ActComboAttack
//
// This is called at end of a set of states that can result in
// either a closecombat or ranged attack. The procedure checks
// to see if the target is within melee range and picks the
// approiate attack.
//
// -ACB- 1998/08/07
//
void P_ActComboAttack(mobj_t * object)
{
  const attacktype_t *attack;

  if (!object->target)
    return;

  if (DecideMeleeAttack(object, object->info->closecombat))
    attack = object->info->closecombat;
  else
    attack = object->info->rangeattack;

  if (attack)
  {
    if (attack->flags & AF_FaceTarget)
      P_ActFaceTarget(object);

    if (attack->flags & AF_NeedSight)
    {
      if (!P_CheckSight(object, object->target))
        return;
    }

    object->currentattack = attack;
    P_DoAttack(object);
  }
#ifdef DEVELOPERS
  else
  {
    if (!object->info->closecombat)
      M_WarnError("%s hasn't got a close combat attack\n", object->info->ddf.name);
    else
      M_WarnError("%s hasn't got a range attack\n", object->info->ddf.name);
  }
#endif

}

//
// P_ActMeleeAttack
//
// Setup a close combat assault
//
// -ACB- 1998/08/07
//
void P_ActMeleeAttack(mobj_t * object)
{
  const attacktype_t *attack;

  attack = object->info->closecombat;

  // -AJA- 1999/08/10: Multiple attack support.
  if (object->state && object->state->action_par)
    attack = (const attacktype_t *) object->state->action_par;

  if (!attack)
  {
    M_WarnError("P_ActMeleeAttack: %s has no close combat attack.\n", 
        object->info->ddf.name);
    return;
  }

  if (attack->flags & AF_FaceTarget)
    P_ActFaceTarget(object);

  if (attack->flags & AF_NeedSight)
  {
    if (!object->target || !P_CheckSight(object, object->target))
      return;
  }

  object->currentattack = attack;
  P_DoAttack(object);
}

//
// P_ActRangeAttack
//
// Setup an attack at range
//
// -ACB- 1998/08/07
//
void P_ActRangeAttack(mobj_t * object)
{
  const attacktype_t *attack;

  attack = object->info->rangeattack;

  // -AJA- 1999/08/10: Multiple attack support.
  if (object->state && object->state->action_par)
    attack = (const attacktype_t *) object->state->action_par;

  if (!attack)
  {
    M_WarnError("P_ActRangeAttack: %s hasn't got a range attack.\n", 
        object->info->ddf.name);
    return;
  }

  if (attack->flags & AF_FaceTarget)
    P_ActFaceTarget(object);

  if (attack->flags & AF_NeedSight)
  {
    if (!object->target || !P_CheckSight(object, object->target))
      return;
  }

  object->currentattack = attack;
  P_DoAttack(object);
}

//
// P_ActSpareAttack
//
// Setup an attack that is not defined as close or range. can be
// used to act as a follow attack for close or range, if you want one to
// add to the others.
//
// -ACB- 1998/08/24
//
void P_ActSpareAttack(mobj_t *object)
{
  const attacktype_t *attack;

  attack = object->info->spareattack;

  // -AJA- 1999/08/10: Multiple attack support.
  if (object->state && object->state->action_par)
    attack = (const attacktype_t *) object->state->action_par;

  if (attack)
  {
    if ((attack->flags & AF_FaceTarget) && object->target)
      P_ActFaceTarget(object);

    if ((attack->flags & AF_NeedSight) && object->target)
    {
      if (!P_CheckSight(object, object->target))
        return;
    }

    object->currentattack = attack;
    P_DoAttack(object);
  }
#ifdef DEVELOPERS
  else
  {
    M_WarnError("P_ActSpareAttack: %s hasn't got a spare attack\n", object->info->ddf.name);
    return;
  }
#endif

}

//
// P_ActRefireCheck
//
// This procedure will be called inbetween firing on an object
// that will fire repeatly (Chaingunner/Arachontron etc...), the
// purpose of this is to see if the object should refire and
// performs checks to that effect, first there is a check to see
// if the object will keep firing regardless and the others
// check if the the target exists, is alive and within view. The
// only other code here is a stealth check: a object with stealth
// capabilitys will lose the ability while firing.
//
// -ACB- 1998/08/10
//
void P_ActRefireCheck(mobj_t * object)
{
  mobj_t *target;
  const attacktype_t *attack;

  attack = object->currentattack;

  if (! attack)
    return;

  if (attack->flags & AF_FaceTarget)
    P_ActFaceTarget(object);

  // Random chance that object will keep firing regardless
  if (P_RandomTest(attack->keepfirechance))
    return;

  target = object->target;

  if (!target || (target->health <= 0) || !P_CheckSight(object, target))
  {
    if (object->info->chase_state)
      P_SetMobjStateDeferred(object, object->info->chase_state, 0);
  }
  else if (object->flags & MF_STEALTH)
  {
    object->vis_target = VISIBLE;
  }
}

//---------------------------------------------
//-----------LOOKING AND CHASING---------------
//---------------------------------------------

//
// SelectTarget
//
// Search the things list for a target
//
// -ACB- 2000/06/20 Re-written and Simplified
//
static mobj_t *SelectTarget(boolean_t newlev)
{
  static mobj_t *targetobj;
  int count;

  // Setup target object
  if (newlev)
    targetobj = mobjlisthead;

  // Nothing?
  if (!targetobj)
    return NULL;

  // Find mobj in list 
  count = P_Random();
  while (count)
  {
    if (targetobj)
      targetobj = targetobj->next;
    else
      targetobj = mobjlisthead;

    count--;
  }

  // Found end of list
  if (!targetobj)
    return NULL;

  // Not a valid obj?
  if (!(targetobj->info->extendedflags&EF_MONSTER) || targetobj->health<=0)
    return NULL;

  return targetobj;
}

//
// CreateAggression
//
// Sets an object up to target a previously stored object.
//
// -ACB- 2000/06/20 Re-written and Simplified
//
static boolean_t CreateAggression(mobj_t * object)
{
  static const mapstuff_t *mapcheck = NULL;
  static mobj_t *target = NULL;
  static int count = 0;
  const mobjinfo_t *targinfo;
  const mobjinfo_t *objinfo;

  count++;

  // New map of no map - setup the procedure for next time
  if (mapcheck == NULL || mapcheck != currentmap)
  {
    mapcheck = currentmap;
    target = SelectTarget(true);
    count = 0;
    return false;
  }

  // No target or target dead
  if (target == NULL)
  {
    target = SelectTarget(false);
    count = 0;
    return false;
  }

  if (!(target->info->extendedflags&EF_MONSTER) || target->health <= 0)
  {
    target = SelectTarget(false);
    count = 0;
    return false;
  }

  // Don't target self...
  if (object == target)
    return false;

  // This object has been checked too many times, try a another one.
  if (count > 127)
  {
    target = SelectTarget(false);
    count = 0;
    return false;
  }

  objinfo = object->info;
  targinfo = target->info;

  if (!P_CheckSight(object, target))
    return false;

  if ((targinfo == objinfo) && (!(objinfo->extendedflags & EF_DISLOYALTYPE)))
    return false;

  // -ACB- 2000/07/20 Remove checking of attacks 
  if (!(objinfo->extendedflags & EF_OWNATTACKHURTS))
  {
    // Type the same and it can't hurt own kind - not good.
    if (targinfo == objinfo)
      return false;
  }

  P_MobjSetTarget(object, target);

  if (object->info->chase_state)
    P_SetMobjStateDeferred(object, object->info->chase_state, 0);

  return true;
}


//
// P_ActStandardLook
//
// Standard Lookout procedure
//
// -ACB- 1998/08/22
//
void P_ActStandardLook(mobj_t * object)
{
  int targ_pnum;
  mobj_t *targ = NULL;

  object->threshold = 0;  // any shot will wake up

  targ_pnum = object->subsector->sector->sound_player;

  if (targ_pnum >= 0 && targ_pnum < MAXPLAYERS && 
      playerlookup[targ_pnum])
  {
    targ = playerlookup[targ_pnum]->mo;
  }

  if (object->flags & MF_STEALTH)
    object->vis_target = VISIBLE;

  if (infight)
  {
    if (CreateAggression(object))
      return;
  }

  if (targ && (targ->flags & MF_SHOOTABLE))
  {
    P_MobjSetTarget(object, targ);

    if (object->flags & MF_AMBUSH)
    {
      if (!P_CheckSight(object, object->target) && 
          !P_LookForPlayers(object, object->info->sight_angle))
        return;
    }
  }
  else
  {
    if (!P_LookForPlayers(object, object->info->sight_angle))
      return;
  }

  if (object->info->seesound)
  {
    if (object->info->extendedflags & EF_BOSSMAN)
      S_StartSound(NULL, object->info->seesound);
    else
      S_StartSound(object, object->info->seesound);
  }

  if (object->info->chase_state)
    P_SetMobjStateDeferred(object, object->info->chase_state, 0);
}

//
// P_ActPlayerSupportLook
//
// Player Support Lookout procedure
//
// -ACB- 1998/09/05
//
void P_ActPlayerSupportLook(mobj_t * object)
{
  object->threshold = 0;  // any shot will wake up

  if (object->flags & MF_STEALTH)
    object->vis_target = VISIBLE;

  if (!object->supportobj)
  {
    if (! P_ActLookForTargets(object))
      return;

    if (object->info->seesound)
    {
      if (object->info->extendedflags & EF_BOSSMAN)
        S_StartSound(NULL, object->info->seesound);
      else
        S_StartSound(object, object->info->seesound);
    }
  }

  if (object->info->meander_state)
    P_SetMobjStateDeferred(object, object->info->meander_state, 0);
}

//
// P_ActStandardMeander
//
void P_ActStandardMeander(mobj_t * object)
{
  int delta;

  object->threshold = 0;  // any shot will wake up

  // move within supporting distance of player
  if (--object->movecount < 0 || !P_Move(object, false))
    P_NewChaseDir(object);

  // turn towards movement direction if not there yet
  if (object->movedir < 8)
  {
    object->angle &= (7 << 29);
    delta = object->angle - (object->movedir << 29);

    if (delta > 0)
      object->angle -= ANG45;
    else if (delta < 0)
      object->angle += ANG45;
  }
}

//
// P_ActPlayerSupportMeander
//
void P_ActPlayerSupportMeander(mobj_t * object)
{
  int delta;

  object->threshold = 0;  // any shot will wake up

  // move within supporting distance of player
  if (--object->movecount < 0 || !P_Move(object, false))
    P_NewChaseDir(object);

  // turn towards movement direction if not there yet
  if (object->movedir < 8)
  {
    object->angle &= (7 << 29);
    delta = object->angle - (object->movedir << 29);

    if (delta > 0)
      object->angle -= ANG45;
    else if (delta < 0)
      object->angle += ANG45;
  }

  //
  // we have now meandered, now check for a support object, if we don't
  // look for one and return; else look for targets to take out, if we
  // find one, go for the chase.
  //
  /*  if (!object->supportobj)
     {
     P_ActPlayerSupportLook(object);
     return;
     } */

  P_ActLookForTargets(object);
}

//
// P_ActStandardChase
//
// Standard AI Chase Procedure
//
// -ACB- 1998/08/22 Procedure Written
// -ACB- 1998/09/05 Added Support Object Check
//
void P_ActStandardChase(mobj_t * object)
{
  int delta;
  sfx_t *sound;

  if (object->reactiontime)
    object->reactiontime--;

  // object has a pain threshold, while this is true, reduce it. while
  // the threshold is true, the object will remain intent on its target.
  if (object->threshold)
  {
    if (!object->target || object->target->health <= 0)
      object->threshold = 0;
    else
      object->threshold--;
  }

  // A Chasing Stealth Creature becomes less visible
  if (object->flags & MF_STEALTH)
    object->vis_target = INVISIBLE;

  // turn towards movement direction if not there yet
  if (object->movedir < 8)
  {
    object->angle &= (7 << 29);
    delta = object->angle - (object->movedir << 29);

    if (delta > 0)
      object->angle -= ANG45;
    else if (delta < 0)
      object->angle += ANG45;
  }

  if (!object->target || !(object->target->flags & MF_SHOOTABLE))
  {
    if (P_ActLookForTargets(object))
      return;

    // -ACB- 1998/09/06 Target is not relevant: NULLify.
    P_MobjSetTarget(object, NULL);

    P_SetMobjStateDeferred(object, object->info->idle_state, 0);
    return;
  }

  // do not attack twice in a row
  if (object->flags & MF_JUSTATTACKED)
  {
    object->flags &= ~MF_JUSTATTACKED;

    // -KM- 1998/12/16 Nightmare mode set the fast parm.
    if (!level_flags.fastparm)
      P_NewChaseDir(object);

    return;
  }

  sound = object->info->attacksound;

  // check for melee attack
  if (object->info->melee_state && DecideMeleeAttack(object, object->info->closecombat))
  {
    if (sound)
      S_StartSound(object, sound);

    if (object->info->melee_state)
      P_SetMobjStateDeferred(object, object->info->melee_state, 0);
    return;
  }

  // check for missile attack
  if (object->info->missile_state)
  {
    // -KM- 1998/12/16 Nightmare set the fastparm.
    if (!(!level_flags.fastparm && object->movecount))
    {
      if (P_ActDecideRangeAttack(object))
      {
        if (object->info->missile_state)
          P_SetMobjStateDeferred(object, object->info->missile_state, 0);
        object->flags |= MF_JUSTATTACKED;
        return;
      }
    }
  }

  // possibly choose another target
  // -ACB- 1998/09/05 Object->support->object check, go for new targets
  if (!P_CheckSight(object, object->target) && !object->threshold)
  {
    if (P_ActLookForTargets(object))
      return;
  }

  // chase towards player
  if (--object->movecount < 0 || !P_Move(object, false))
    P_NewChaseDir(object);

  // make active sound
  if (object->info->activesound && M_Random() < 3)
    S_StartSound(object, object->info->activesound);
}

//
// P_ActResurrectChase
//
// Before undertaking the standard chase procedure, the object
// will check for a nearby corpse and raises one if it exists.
//
// -ACB- 1998/08/22 Procedure written
// -ACB- 1998/09/05 Support Check: Raised object supports raiser's supportobj
//
void P_ActResurrectChase(mobj_t * object)
{
  mobj_t *corpse;

  corpse = P_MapFindCorpse(object);

  if (corpse)
  {
    object->angle = R_PointToAngle(object->x, object->y, corpse->x, corpse->y);
    if (object->info->res_state)
      P_SetMobjStateDeferred(object, object->info->res_state, 0);

    // corpses without raise states should be skipped
    DEV_ASSERT2(corpse->info->raise_state);

    P_BringCorpseToLife(corpse);

    // -ACB- 1998/09/05 Support Check: Res creatures to support that object
    if (object->supportobj)
    {
      P_MobjSetSupportObj(corpse, object->supportobj);
      P_MobjSetTarget(corpse, object->target);
    }
    else
    {
      P_MobjSetSupportObj(corpse, NULL);
      P_MobjSetTarget(corpse, NULL);
    }

    // -AJA- Resurrected creatures are on Archvile's side (like MBF)
    corpse->side = object->side;
    return;
  }

  P_ActStandardChase(object);
}

//
// P_ActWalkSoundChase
//
// Make a sound and then chase...
//
void P_ActWalkSoundChase(mobj_t * object)
{
  if (!object->info->walksound)
  {
    M_WarnError("WALKSOUND_CHASE: %s hasn't got a walksound.\n", 
        object->info->ddf.name);
    return;
  }

  S_StartSound(object, object->info->walksound);
  P_ActStandardChase(object);
}

//
// P_ActCheckBlood
//
// -KM- 1999/01/31 Part of the extra blood option, makes blood stick around...
// -AJA- 1999/10/02: ...but not indefinitely.
//
void P_ActCheckBlood(mobj_t * mo)
{
  if (level_flags.more_blood && mo->tics >= 0)
  {
    int val = P_Random();

    // exponential formula
    mo->tics = ((val * val * val) >> 18) * TICRATE + TICRATE;
  }
}

//
// P_ActJump
//
// Jumps to the given label, possibly randomly.  Note: nothing to do
// with monsters physically jumping.
//
void P_ActJump(mobj_t * object)
{
  act_jump_info_t *jump;
  
  if (!object->state || !object->state->action_par)
  {
    M_WarnError("JUMP action used in [%s] without a label !\n",
        object->info->ddf.name);
    return;
  }

  jump = (act_jump_info_t *) object->state->action_par;

  DEV_ASSERT2(jump->chance >= 0);
  DEV_ASSERT2(jump->chance <= 1);

  if (P_RandomTest(jump->chance))
  {
    object->next_state = (object->state->jumpstate == S_NULL) ?
        NULL : (states + object->state->jumpstate);
  }
}

//
// P_ActPlayerAttack
//
// -AJA- 1999/08/08: New attack flag FORCEAIM, which fixes chainsaw.
//
void P_ActPlayerAttack(mobj_t * p_obj, const attacktype_t * attack)
{
  mobj_t *target;
  angle_t diff = ANG180 / 32;
  float_t range;

  DEV_ASSERT2(attack);

  range = (attack->range > 0) ? attack->range : MISSILERANGE;
  p_obj->currentattack = attack;

  // see which target is to be aimed at
  target = P_MapTargetAutoAim(p_obj, p_obj->angle,
      range, attack->flags & AF_ForceAim);
  P_MobjSetTarget(p_obj, target);

  if (leveltime & 1)
    diff = 0 - diff;

  // -KM- 1998/12/16 If that is a miss, aim slightly right or slightly 
  //      left of the target.
  if (target->extendedflags & EF_DUMMYMOBJ)
  {
    target = P_MapTargetAutoAim(p_obj, p_obj->angle + diff,
        range, attack->flags & AF_ForceAim);
  }

  if (target->extendedflags & EF_DUMMYMOBJ)
  {
    target = P_MapTargetAutoAim(p_obj, p_obj->angle - diff,
        range, attack->flags & AF_ForceAim);
  }

  if (! (target->extendedflags & EF_DUMMYMOBJ))
    P_MobjSetTarget(p_obj, target);

  if (attack->flags & AF_FaceTarget)
    P_ActFaceTarget(p_obj);

  P_DoAttack(p_obj);
}

