//----------------------------------------------------------------------------
//  EDGE True BSP Rendering (BSP Traversal)
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------
//
// -AJA- 1999/08/31: Wrote this file.
//
// TODO HERE:
//   + optimise first subsector: ignore floors out of view.
//   + fix the single-pixel-gap sprite clipping bug.
//   + draw halos.
//   + split up: r2_seg.c and r2_mobj.c.
//

#include "i_defs.h"

#include "dm_defs.h"
#include "dm_state.h"
#include "m_bbox.h"
#include "p_local.h"
#include "p_mobj.h"
#include "p_spec.h"
#include "r_defs.h"
#include "r_main.h"
#include "r_plane.h"
#include "r_sky.h"
#include "r_state.h"
#include "r_things.h"
#include "r2_defs.h"
#include "v_colour.h"
#include "v_res.h"
#include "z_zone.h"


#define DEBUG  0
#define DEBUG_OVERDRAW  0


// common stuff

extern sector_t *frontsector;
extern sector_t *backsector;

static subsector_t *cur_sub;
static seg_t *cur_seg;

static int cur_x_min, cur_x_max;

static boolean_t cur_upper_sky;
static boolean_t cur_lower_sky;
static boolean_t cur_upper_invis;
static boolean_t cur_lower_invis;

static subsector_t *drawsubs;


int detail_level = 1;
boolean_t use_dlights = false;
int sprite_kludge = 0;


#define APPROX_DIST2(dx,dy)  \
    ((dx) + (dy) - 0.5 * MIN((dx),(dy)))

#define APPROX_DIST3(dx,dy,dz)  \
    APPROX_DIST2(APPROX_DIST2(dx,dy),dz)


// The minimum distance between player and a visible sprite.
#define MINZ  2.0


#define PART_TEX_W(part)  (part->image ? IM_WIDTH( part->image) : 0)
#define PART_TEX_H(part)  (part->image ? IM_HEIGHT(part->image) : 0)


//
// R2_AddDLights
//
// Increases the array of light level at the points (x,y,z).
//
void R2_AddDLights(int num, int *level, 
    float_t *x, float_t *y, float_t *z, mobj_t *mo)
{
  int qty = (int)ceil(mo->dlight_qty);

  float_t mo_z = mo->z + mo->height * PERCENT_2_FLOAT(mo->info->dlight.height);
  float_t dist;

  DEV_ASSERT2(num > 0);
  DEV_ASSERT2(qty > 0);

  switch (mo->info->dlight.type)
  {
    case DLITE_None:
      I_Error("R2_AddDLights: bad dynamic light\n");

    case DLITE_Constant:
      for (; num > 0; num--, level++, x++, y++, z++)
        (*level) += qty;
      break;
  
    case DLITE_Linear:
      for (; num > 0; num--, level++, x++, y++, z++)
      {
        dist = APPROX_DIST3(fabs((*x) - mo->x), fabs((*y) - mo->y), 
            fabs((*z) - mo_z));

        (*level) += qty * 32.0 / MAX(1.0, dist);
      }
      break;

    case DLITE_Quadratic:
      for (; num > 0; num--, level++, x++, y++, z++)
      {
        dist = APPROX_DIST3(fabs((*x) - mo->x), fabs((*y) - mo->y), 
            fabs((*z) - mo_z));
        
        (*level) += qty * 1024.0 / MAX(1, dist * dist);
      }
      break;
  }
}

//
// R2_AddColourDLights
//
// Increases the arrays of colour levels at the points (x,y,z).
//
void R2_AddColourDLights(int num, int *r, int *g, int *b, 
    float_t *x, float_t *y, float_t *z, mobj_t *mo)
{
  int base_qty = (int)ceil(mo->dlight_qty);
  int qty;

  float_t mo_z = mo->z + mo->height * PERCENT_2_FLOAT(mo->info->dlight.height);
  float_t dist;
    
  int R = (mo->info->dlight.colour >> 16) & 0xFF;
  int G = (mo->info->dlight.colour >>  8) & 0xFF;
  int B = (mo->info->dlight.colour      ) & 0xFF;

  DEV_ASSERT2(num > 0);
  DEV_ASSERT2(base_qty > 0);

  switch (mo->info->dlight.type)
  {
    case DLITE_None:
      I_Error("R2_AddColourDLights: bad dynamic light\n");

    case DLITE_Constant:
      for (; num > 0; num--, r++, g++, b++, x++, y++, z++)
      {
        (*r) += base_qty * R / 255;
        (*g) += base_qty * G / 255;
        (*b) += base_qty * B / 255;
      }
      break;
  
    case DLITE_Linear:
      for (; num > 0; num--, r++, g++, b++, x++, y++, z++)
      {
        dist = APPROX_DIST3(fabs((*x) - mo->x), fabs((*y) - mo->y), 
            fabs((*z) - mo_z));

        qty = base_qty * 32.0 / MAX(1.0, dist);

        (*r) += qty * R / 255;
        (*g) += qty * G / 255;
        (*b) += qty * B / 255;
      }
      break;

    case DLITE_Quadratic:
      for (; num > 0; num--, r++, g++, b++, x++, y++, z++)
      {
        dist = APPROX_DIST3(fabs((*x) - mo->x), fabs((*y) - mo->y), 
            fabs((*z) - mo_z));
        
        qty = base_qty * 1024.0 / MAX(1, dist * dist);

        (*r) += qty * R / 255;
        (*g) += qty * G / 255;
        (*b) += qty * B / 255;
      }
      break;
  }
}

//
// R2_WallEvent
//
// Handle a single section of wall.
//
static void R2_WallEvent(drawfloor_t *dfloor, int x1, int x2,
    float_t top, float_t bottom, float_t tex_top_h, 
    surface_t *part, boolean_t is_masked, float_t x_offset)
{
  drawwall_t *wall;
  screenline_t *area;

  float_t yt, yt2, yt_step;
  float_t yb, yb2, yb_step;

  int x;
  float_t scale1, scale2, scale_step;

  boolean_t maybe_slider = is_masked;
  boolean_t solid, connect_low, connect_high;

#if (DEBUG >= 3)
    L_WriteDebug("WALL X:%d..%d (%d,%d,%d) -> (%d,%d,%d)\n", x1, x2,
        (int)cur_seg->v1->x, (int)cur_seg->v1->y, (int)top,
        (int)cur_seg->v2->x, (int)cur_seg->v2->y, (int)bottom);
#endif
  
  scale1 = cur_seg->scale1;
  scale2 = cur_seg->scale2;

  scale_step = (x1==x2) ? 0 : (scale2 - scale1) / (x2 - x1);
  
  wall = R2_GetDrawWall();
  area = &wall->area;

  wall->next = NULL;
  wall->seg  = cur_seg;

  wall->part  = part;
  wall->props = part->override_p ? part->override_p : dfloor->props;
  wall->is_masked = is_masked;

  wall->scale1 = scale1 + scale_step * (x1 - cur_seg->x1);
  wall->scale_step = scale_step;

  wall->distance = cur_seg->rw_distance;
  wall->x_offset = cur_seg->rw_offset + x_offset;
  wall->angle = cur_seg->angle;
  wall->slide_type = SLIDE_None;
  
  // horizontal slider hack
  if (maybe_slider && cur_seg->linedef->special && 
      cur_seg->linedef->special->s.type != SLIDE_None)
  {
    slider_move_t *smov = cur_seg->linedef->slider_move;

    if (smov)
    {
      wall->slide_type = smov->info->type;
      wall->opening = smov->opening;
      wall->line_len = smov->line_len;
    }

    wall->side = 0;

    // seg is on left side of linedef ?
    if ((cur_seg->v2->x - cur_seg->v1->x) * cur_seg->linedef->dx < 0 ||
        (cur_seg->v2->y - cur_seg->v1->y) * cur_seg->linedef->dy < 0)
    {
      wall->side = 1;
#if 0
      if (wall->slide_type == SLIDE_Left)
        wall->slide_type = SLIDE_Right;
      else if (wall->slide_type == SLIDE_Right)
        wall->slide_type = SLIDE_Left;
#endif
    }
  }

  DEV_ASSERT2(x2 >= x1);

  solid = !is_masked && (part->translucency > 0.99);

  connect_high = solid && !dfloor->higher && 
      (top >= cur_sub->sector->c_h);

  connect_low = solid && !dfloor->lower &&
      (bottom <= cur_sub->sector->f_h);

  // calculate area
  area->x1 = x1;
  area->x2 = x2;

  area->ranges = R2_GetOpenings(x2 - x1 + 1);

  area->y_offset = tex_top_h - top;

  top    -= viewz;
  bottom -= viewz;

  yt  = focusyfrac - top * scale1;
  yt2 = focusyfrac - top * scale2;

  yb  = focusyfrac - bottom * scale1;
  yb2 = focusyfrac - bottom * scale2;

  yt_step = (x1==x2) ? 0 : (yt2 - yt) / (x2 - x1);
  yb_step = (x1==x2) ? 0 : (yb2 - yb) / (x2 - x1);

  area->y = yt;
  area->step = yt_step;

  // quit early if totally off-screen
  if ((yb < 0 && yb2 < 0) || (yt >= viewheight && yt2 >= viewheight))
  {
    R2_CommitOpenings(0);
    R2_CommitDrawWall(0);
    return;
  }
  
  for (x=x1; x <= x2; x++, yt += yt_step, yb += yb_step)
  {
    int y1 = MAX((int)floor(yt), 0);
    int y2 = MIN((int)floor(yb), viewheight-1);

    if (y1 > y2)
    {
      y1 = 1;
      y2 = 0;
    }

    area->ranges[x - x1].y1 = y1;
    area->ranges[x - x1].y2 = y2;
  }

  R2_1DOcclusionClose(x1, x2, area->ranges);
  R2_2DOcclusionClose(x1, x2, area->ranges, connect_low, connect_high, solid);
 
  //
  // Dynamic lighting
  //
  wall->extra_light[0] = 0;
  wall->extra_light[1] = 0;

  if (use_dlights)
  {
    drawthing_t *dl;
    divline_t div;
    float_t wx[2], wy[2], wz[2];

    wx[0] = cur_seg->v1->x;
    wy[0] = cur_seg->v1->y;

    wx[1] = cur_seg->v2->x;
    wy[1] = cur_seg->v2->y;
    
    wz[0] = wz[1] = (top+bottom) / 2.0;

    div.x  = cur_seg->v1->x;
    div.y  = cur_seg->v1->y;
    div.dx = cur_seg->v2->x - div.x;
    div.dy = cur_seg->v2->y - div.y;

    for (dl=dfloor->dlights; dl; dl=dl->next)
    {
      // light behind seg ?    
      if (P_PointOnDivlineSide(dl->mo->x, dl->mo->y, &div) != 0)
        continue;
       
      R2_AddDLights(2, wall->extra_light, wx, wy, wz, dl->mo);
    }
  }

  // link it in
  DEV_ASSERT2(!cur_seg->miniseg);
  if (!cur_seg->backsector || connect_low || connect_high)
  {
    R2_DrawWall(cur_sub, wall);
    R2_CommitOpenings(0);
    R2_CommitDrawWall(0);
  }
  else
  {
    wall->next = dfloor->walls;
    dfloor->walls = wall;

    R2_CommitOpenings(x2 - x1 + 1);
    R2_CommitDrawWall(1);
  }
}


//
// R2_BuildWalls
//
// Analyses floor/ceiling heights, and add corresponding walls/floors
// to the drawfloor.  Returns true if the whole region was "solid".
//
static boolean_t R2_BuildWalls(drawfloor_t *floor)
{
  side_t *sd = cur_seg->sidedef;
   
  float_t f1 = floor->f_h;
  float_t c1 = floor->c_h;

  float_t f, c, tex_top_h;
  float_t x_offset;

  int j;
  wall_tile_t *wt;

#if (DEBUG >= 3)
    L_WriteDebug("   BUILD WALLS %1.1f .. %1.1f\n", f1, c1);
#endif

  // handle TRANSLUCENT + THICK floors (a bit of a hack)
  if (floor->ef && floor->higher &&
      (floor->ef->ef_info->type & EXFL_Thick) &&
      (floor->ef->top->translucency <= 0.99))
  {
    c1 = floor->ef->top_h;
  }

  for (j=0; j < sd->tile_used; j++)
  {
    wt = sd->tiles + j;

    c = MIN(c1, wt->z2);
    f = MAX(f1, wt->z1);

    // not visible ?
    if (f >= c)
      continue;

    DEV_ASSERT2(wt->surface->image);
 
    tex_top_h = wt->tex_z + wt->surface->offset.y;

    if (wt->flags & WTILF_Extra)
      x_offset = cur_seg->sidedef->middle.offset.x;
    else
      x_offset = wt->surface->offset.x;
    
    R2_WallEvent(floor, cur_seg->x1, cur_seg->x2, c, f, tex_top_h,
        wt->surface, (wt->flags & WTILF_MidMask), x_offset);
  }

  if (cur_seg->sidedef->middle.image == NULL)
  {
    // -AJA- hack for transparent doors (this test would normally be
    // above this block, not inside it).
    //
    if (f1 >= c1)
      return true;

    return false;
  }

  // handle sliders that are totally solid and closed
  if (cur_seg->linedef->special &&
      cur_seg->linedef->special->s.type != SLIDE_None &&
      ! cur_seg->linedef->special->s.see_through &&
      ! cur_seg->linedef->slider_move)
  {
    return true;
  }
   
  return false;
}


//
// R2_WalkWall
//
static void R2_WalkWall(seg_t *seg)
{
  drawfloor_t *dfloor;

  cur_seg = seg;

  frontsector = seg->front_sub->sector;
  backsector  = NULL;

  if (seg->back_sub)
    backsector = seg->back_sub->sector;

  cur_upper_sky = (backsector && IS_SKY(frontsector->ceil) &&
      IS_SKY(backsector->ceil));

  cur_lower_sky = (backsector && IS_SKY(frontsector->floor) &&
      IS_SKY(backsector->floor));

  cur_upper_invis = cur_lower_invis = false;

  DEV_ASSERT2(!seg->miniseg);

  // -AJA- hack to allow transparent doors
  if (backsector)
  {
    cur_lower_invis = (backsector->f_h > frontsector->f_h) &&
        (seg->sidedef->bottom.image == NULL);

    cur_upper_invis = (backsector->c_h < frontsector->c_h) &&
        (seg->sidedef->top.image == NULL);
  }

  // --- handle each floor ---
  
  for (dfloor=cur_sub->floors; dfloor; dfloor=dfloor->next)
  {
    R2_BuildWalls(dfloor);
  }

  // cookie-cut the seg from the 1D occlusion buffer
  if (seg->linedef->blocked)
    R2_1DOcclusionSet(seg->x1, seg->x2);
}

//
// R2_WalkSeg
//
// Visit a single seg (aka. line) that forms part of the current
// subsector.
//
// seg->back will be true if the seg faces away from the camera.
//
// seg->visible will be true if the seg (front or back) is visible.
// Examples of non-visible: totally off left/right screen edge,
// totally occluded by 1D buffer.
//
// Visible segs (front or back) have valid info (x1, x2, angle1/2,
// scale1/2, etc).  All segs have value translated coords (for sprite
// clipping).
//
static void R2_WalkSeg(seg_t *seg)
{
  angle_t angle1, angle2, temp;
  angle_t span, tspan1, tspan2, distangle;
  float_t hyp;

  int x1, x2;

  seg->visible = false;
  seg->back = false;

  // translate coordinates
  {
    float_t tx1 = seg->v1->x - viewx;
    float_t ty1 = seg->v1->y - viewy;
    float_t tx2 = seg->v2->x - viewx;
    float_t ty2 = seg->v2->y - viewy;

    seg->tx1 = tx1 * viewsin - ty1 * viewcos;
    seg->tz1 = tx1 * viewcos + ty1 * viewsin;

    seg->tx2 = tx2 * viewsin - ty2 * viewcos;
    seg->tz2 = tx2 * viewcos + ty2 * viewsin;
  }

  // compute sprite clipping info
  seg->orientation = 0;

  if (seg->back_sub && sprite_kludge <= 1 &&
      (seg->miniseg || !seg->linedef->blocked) &&
      (seg->tz1 > MINZ || seg->tz2 > MINZ) &&
      fabs(seg->tz1 - seg->tz2) > 0.2)
  {
    seg->orientation = ((seg->angle - (viewangle + ANG90)) <= 
        ANG180) ? -1 : +1;
  }

#if (DEBUG >= 3)
    L_WriteDebug("TRANS %p: (%1.1f,%1.1f) -> (%1.1f,%1.1f) "
        "orien=%d\n", seg, seg->tx1, seg->tz1, seg->tx2, 
        seg->tz2, seg->orientation);
#endif

  angle1 = R_PointToAngle(viewx, viewy, seg->v1->x, seg->v1->y);
  angle2 = R_PointToAngle(viewx, viewy, seg->v2->x, seg->v2->y);

  // Clip to view edges.
  // -ES- 1999/03/20 Replaced clipangle with clipscope/leftclipangle/rightclipangle

  span = angle1 - angle2;

  // back side ?
  if (span >= ANG180)
  {
    temp   = angle1; 
    angle1 = angle2; 
    angle2 = temp;
    
    seg->back = true;
    span = 0 - span;
  }

  // Global angle needed by segcalc.
  rw_angle1 = angle1;

  angle1 -= viewangle;
  angle2 -= viewangle;

#if (DEBUG >= 3)
    L_WriteDebug("ANGLE1 = %1.2f  ANGLE2 = %1.2f\n", 
        ANG_2_FLOAT(angle1), ANG_2_FLOAT(angle2));
#endif

  tspan1 = angle1 - rightclipangle;
  tspan2 = leftclipangle - angle2;

  if (tspan1 > clipscope)
  {
    // Totally off the left edge?
    if (tspan2 >= ANG180)
      return;

    angle1 = leftclipangle;
  }

  if (tspan2 > clipscope)
  {
    // Totally off the left edge?
    if (tspan1 >= ANG180)
      return;

    angle2 = rightclipangle;
  }

#if (DEBUG >= 3)
    L_WriteDebug("ANGLE1 = %1.2f  ANGLE2 = %1.2f\n", 
        ANG_2_FLOAT(angle1), ANG_2_FLOAT(angle2));
#endif

  seg->angle1 = angle1;
  seg->angle2 = angle2;

  // The seg is in the view range,
  // but not necessarily visible.

  angle1 = (angle1 + ANG90) >> ANGLETOFINESHIFT;
  angle2 = (angle2 + ANG90) >> ANGLETOFINESHIFT;

  // clip to screen
  x1 = MAX(viewangletox[angle1], 0);
  x2 = MIN(viewangletox[angle2], viewwidth) - 1;

  if (x1 <= x2)
  {
    seg->visible = ! R2_1DOcclusionTestShrink(&x1, &x2);
  }

  seg->x1 = x1;
  seg->x2 = x2;

#if (DEBUG >= 2)
    L_WriteDebug("  %sSEG %p (%1.1f, %1.1f) -> (%1.1f, %1.1f) %s %s\n",
        seg->miniseg ? "MINI" : "", seg, seg->v1->x, seg->v1->y, 
        seg->v2->x, seg->v2->y, seg->back ? "back" : "front", 
        seg->visible ?  "visible" : "non-vis");
#endif

  if (!seg->visible)
    return;

  // update screen bounds of subsector
  {
    cur_x_min = MIN(cur_x_min, x1);
    cur_x_max = MAX(cur_x_max, x2);
  }

  // mark the segment as visible for auto map
  if (seg->visible && !seg->miniseg)
    seg->linedef->flags |= ML_Mapped;

  // determine angle that is normal to the seg line
  rw_normalangle = seg->angle + (seg->back ? ANG270 : ANG90);

  // distangle is the angle that lies between the seg line and the
  // line from the camera to the seg's first point.

  distangle = rw_normalangle - rw_angle1;

  if (distangle > ANG180)
    distangle = 0 - distangle;
    
  distangle = (distangle >= ANG90) ? 0 : ANG90 - distangle;

  // rw_distance is the distance from the camera to the seg line along
  // a line which is normal to the seg line.

  hyp = R_PointToDist(viewx, viewy, (seg->back ? seg->v2 : seg->v1)->x, 
      (seg->back ? seg->v2 : seg->v1)->y);

  seg->rw_distance = hyp * M_Sin(distangle);
  seg->rw_offset   = hyp * M_Cos(distangle);

  if (rw_normalangle - rw_angle1 < ANG180)
    seg->rw_offset = 0 - seg->rw_offset;

  seg->rw_offset += seg->offset;

  // calculate scale at both ends and step
  rw_distance = seg->rw_distance;

  seg->scale1 = R_ScaleFromGlobalAngle(viewangle + xtoviewangle[x1]);
  seg->scale2 = R_ScaleFromGlobalAngle(viewangle + xtoviewangle[x2]);

#if (DEBUG >= 2)
    L_WriteDebug("    %d..%d  scale1=%1.8f  scale2=%1.8f  "
        "rw_dist=%1.2f  rw_off=%1.2f\n", seg->x1, seg->x2, 
        seg->scale1, seg->scale2, seg->rw_distance, seg->rw_offset);
#endif

#if 0
  // -AJA- this code is no longer needed, but I want to keep it in
  //       case later on we decide to implement slopes.  It computes
  //       the world coords of the start and end of the wall part.
  {
    angle_t angle1 = viewangle - seg->angle + xtoviewangle[x1] - ANG90;
    angle_t angle2 = viewangle - seg->angle + xtoviewangle[x2] - ANG90;

    float_t x_dist1 = rw_offset2 - seg->offset;
    float_t x_dist2 = rw_offset2 - seg->offset;

    float_t x1 = seg->v1->x;
    float_t y1 = seg->v1->y;
    angle_t angle = seg->angle;

    if (! (ANG90-ANG1  <= angle1 && angle1 <= ANG90+ANG1) &&
        ! (ANG270-ANG1 <= angle1 && angle1 <= ANG270+ANG1))
      x_dist1 -= M_Tan(angle1) * rw_distance;
    
    if (! (ANG90-ANG1  <= angle2 && angle2 <= ANG90+ANG1) &&
        ! (ANG270-ANG1 <= angle2 && angle2 <= ANG270+ANG1))
      x_dist2 -= M_Tan(angle2) * rw_distance;

    if (cur_is_back)
    {
      x1 = seg->x2;
      y1 = seg->y2;

      angle += ANG180;
    }
    
    cur_world_x1 = x1 + M_Cos(angle) * x_dist1;
    cur_world_y1 = y1 + M_Sin(angle) * x_dist1;
    cur_world_x2 = x1 + M_Cos(angle) * x_dist2;
    cur_world_y2 = y1 + M_Sin(angle) * x_dist2;
  }
#endif
}

//
// R2_CheckBBox
//
// Checks BSP node/subtree bounding box.
// Returns true if some part of the bbox might be visible.
//
// Placed here to be close to R2_WalkSeg(), which contains similiar
// logic with respect to the angle clipping stuff.
//
extern int checkcoord[12][4];

boolean_t R2_CheckBBox(float_t *bspcoord)
{
  int boxx;
  int boxy;
  int boxpos;

  float_t x1, y1, x2, y2;

  angle_t angle1, angle2;
  angle_t span, tspan1, tspan2;

  int sx1, sx2;

  // Find the corners of the box
  // that define the edges from current viewpoint.
  if (viewx <= bspcoord[BOXLEFT])
    boxx = 0;
  else if (viewx < bspcoord[BOXRIGHT])
    boxx = 1;
  else
    boxx = 2;

  if (viewy >= bspcoord[BOXTOP])
    boxy = 0;
  else if (viewy > bspcoord[BOXBOTTOM])
    boxy = 1;
  else
    boxy = 2;

  boxpos = (boxy << 2) + boxx;
  if (boxpos == 5)
    return true;

  x1 = bspcoord[checkcoord[boxpos][0]];
  y1 = bspcoord[checkcoord[boxpos][1]];
  x2 = bspcoord[checkcoord[boxpos][2]];
  y2 = bspcoord[checkcoord[boxpos][3]];

  // check clip list for an open space
  angle1 = R_PointToAngle(viewx, viewy, x1, y1) - viewangle;
  angle2 = R_PointToAngle(viewx, viewy, x2, y2) - viewangle;

  span = angle1 - angle2;

  // Sitting on a line?
  if (span >= ANG180)
    return true;

  // -ES- 1999/03/20 Replaced clipangle with clipscope/leftclipangle/rightclipangle

  tspan1 = angle1 - rightclipangle;
  tspan2 = leftclipangle - angle2;

  if (tspan1 > clipscope)
  {
    // Totally off the left edge?
    if (tspan2 >= ANG180)
      return false;

    angle1 = leftclipangle;
  }

  if (tspan2 > clipscope)
  {
    // Totally off the right edge?
    if (tspan1 >= ANG180)
      return false;

    angle2 = rightclipangle;
  }

  // Find the first clippost
  //  that touches the source post
  //  (adjacent pixels are touching).
  angle1 = (angle1 + ANG90) >> ANGLETOFINESHIFT;
  angle2 = (angle2 + ANG90) >> ANGLETOFINESHIFT;
  sx1 = viewangletox[angle1];
  sx2 = viewangletox[angle2] - 1;

  // Does not cross a pixel.
  if (sx1 > sx2)
    return false;

  return ! R2_1DOcclusionTest(sx1, sx2);
}

//
// R2_PlaneEvent
//
// face_dir is +1 for upwards (floors), -1 for downwards (ceilings).
//
static void R2_PlaneEvent(drawfloor_t *dfloor, int x1, int x2,
    float_t h, surface_t *info, int face_dir)
{
  drawplane_t *plane;
  screenline_t *area;
  seg_t *seg;
  int vis_count;

  int x, sx, ex;
  float_t y1, y2, step, h2;

  boolean_t sky, solid, connect_low, connect_high;

  connect_low  = (face_dir > 0) && (!dfloor->lower);
  connect_high = (face_dir < 0) && (!dfloor->higher);
  
  sky = connect_high && IS_SKY(*info);

  // ignore non-facing planes
  if (!sky && (viewz > h) != (face_dir > 0))
    return;

  // ignore dud regions (dfloor > ceiling)
  if (dfloor->f_h > dfloor->c_h)
    return;

  // ignore invisible planes
  if (info->translucency < 0.02)
    return;

  // ignore missing texture (maybe causing HOM)
  if (info->image == NULL)
    return;

  if (hom_detect)
    connect_low = connect_high = false;

  plane = R2_GetDrawPlane();
  area = &plane->area;

  // setup prototype plane
  plane->next  = NULL;
  plane->h     = h;
  plane->info  = info;
  plane->props = info->override_p ? info->override_p : dfloor->props;
  plane->face_dir = face_dir;

  h -= viewz;

  // setup area
  area->x1 = x1;
  area->x2 = x2;

  area->ranges = R2_GetOpenings(x2 - x1 + 1);
  
  solid = (info->translucency > 0.99) && info->image->solid;

  // setup initial shape.  The connect_xxx stuff in here is for
  // flat-flooding (an old Doom trick used e.g. for deep water).

  if (face_dir < 0)
  {
    for (x=x1; x <= x2; x++)
    {
      area->ranges[x - x1].y1 = connect_high ? Screen_clip[x].y1 : 0;
      area->ranges[x - x1].y2 = -1;
    }
  }
  else
  {
    for (x=x1; x <= x2; x++)
    {
      area->ranges[x - x1].y1 = viewheight;
      area->ranges[x - x1].y2 = connect_low ? Screen_clip[x].y2 : viewheight-1;
    }
  }
 
  vis_count=0;

  // handle sky specially (for compatibility)
  if (sky)
  {
    boolean_t double_sky;

    for (seg=cur_sub->segs; seg; seg=seg->sub_next)
    {
      if (!seg->visible || seg->back || seg->miniseg)
        continue;

      sx = MAX(x1, seg->x1);
      ex = MIN(x2, seg->x2);

      if (sx > ex)
        continue;

      double_sky = (seg->backsector && IS_SKY(seg->backsector->ceil));

      h2 = h;
      if (double_sky)
      {
        h2 = MIN(h2, seg->backsector->c_h - viewz);  // MAX ???
      }
      y1 = focusyfrac - h2 * seg->scale1;
      y2 = focusyfrac - h2 * seg->scale2;

      if (y1 < 0 && y2 < 0)
        continue;

      step = (seg->x1==seg->x2) ? 0 : (y2 - y1) / (seg->x2 - seg->x1);

      vis_count++;

      y1 += step * (sx - seg->x1);

      for (x=sx; x <= ex; x++, y1 += step)
      {
        int y = MIN((int)floor(y1), viewheight-1);

        if (y < 0)
          continue;

        area->ranges[x - x1].y2 = y;
      }

      R2_1DOcclusionClose(sx, ex, area->ranges + (sx-x1));
      R2_2DOcclusionClose(sx, ex, area->ranges + (sx-x1), 
          false, !double_sky, false);
    }

    if (vis_count > 0)
      R2_DrawPlane(cur_sub, plane);

    R2_CommitOpenings(0);
    R2_CommitDrawPlane(0);
    
    return;
  }
 
  // traverse the segs, computing the on-screen columns
  for (seg=cur_sub->segs; seg; seg=seg->sub_next)
  {
    if (!seg->visible)
      continue;
    
#if 0
    // allow vertical flooding when back equivalent to front
    if ((connect_low || connect_high) && seg->miniseg)
      continue;
#endif

    sx = MAX(x1, seg->x1);
    ex = MIN(x2, seg->x2);

    if (sx > ex)
      continue;

    y1 = focusyfrac - h * seg->scale1;
    y2 = focusyfrac - h * seg->scale2;

    step = (seg->x1==seg->x2) ? 0 : (y2 - y1) / (seg->x2 - seg->x1);

    // ignore parts of segs that are totally off-screen
    if (((face_dir < 0) && y1 < 0 && y2 < 0) ||
        ((face_dir > 0) && y1 >= viewheight && y2 >= viewheight))
    {
      continue;
    }

    vis_count++;

    y1 += step * (sx - seg->x1);

    if (face_dir < 0)
    {
      if (seg->back && connect_high)
        continue;

      for (x=sx; x <= ex; x++, y1 += step)
      {
        int y = MIN((int)floor(y1), viewheight-1);

        if (y < 0)
          continue;

        if (!seg->back)
          area->ranges[x - x1].y2 = y;
        else
          area->ranges[x - x1].y1 = y + 1;
      }
    }
    else
    {
      if (seg->back && connect_low)
        continue;

      for (x=sx; x <= ex; x++, y1 += step)
      {
        int y = MAX((int)floor(y1), 0);

        if (y >= viewheight)
          continue;

        if (!seg->back)
          area->ranges[x - x1].y1 = y;
        else
          area->ranges[x - x1].y2 = y - 1;
      }
    }
  }
  
  // quit early if totally off-screen
  if (vis_count == 0)
  {
    R2_CommitOpenings(0);
    R2_CommitDrawPlane(0);
    return;
  }

  R2_1DOcclusionClose(x1, x2, area->ranges);
  R2_2DOcclusionClose(x1, x2, area->ranges, connect_low, connect_high, solid);

#if 0  // TEMPORARY DISABLED
  //
  // Dynamic lighting
  //
  plane->min_y = plane->max_y = -1;

  plane->extra_light[0] = 0;
  plane->extra_light[1] = 0;

  if (use_dlights && ! IS_SKY(*info))
  {
    drawthing_t *dl;

    int mid_x = (x1 + x2) / 2;
    float_t dist1, dist2;
    angle_t angle;

    float_t wx[2], wy[2], wz[2];
    
    plane->min_y = SCREENHEIGHT-1;
    plane->max_y = 0;

    for (x=x1; x <= x2; x++)
    {
      if (area->ranges[x - x1].y1 > area->ranges[x - x1].y2)
        continue;
      
      plane->min_y = MIN(plane->min_y, area->ranges[x - x1].y1);
      plane->max_y = MAX(plane->max_y, area->ranges[x - x1].y2);
    }

    if (plane->min_y <= plane->max_y)
    {
      DEV_ASSERT2(0 <= plane->min_y < SCREENHEIGHT);
      DEV_ASSERT2(0 <= plane->max_y < SCREENHEIGHT);

      // compute map coordinates for top & bottom of plane

      dist1 = fabs(info->h - viewz) * yslope[plane->min_y];
      dist2 = fabs(info->h - viewz) * yslope[plane->max_y];
      angle = viewangle + xtoviewangle[mid_x];

      dist1 *= distscale[mid_x];
      dist2 *= distscale[mid_x];

      wx[0] = viewx + M_Cos(angle) * dist1;
      wy[0] = viewy + M_Sin(angle) * dist1;

      wx[1] = viewx + M_Cos(angle) * dist2;
      wy[1] = viewy + M_Sin(angle) * dist2;

      wz[0] = wz[1] = info->h;

      for (dl=dfloor->dlights; dl; dl=dl->next)
      {
        // light behind the plane ?    
        if ((dl->tz > info->h) != (face_dir > 0))
          continue;
       
        R2_AddDLights(2, plane->extra_light, wx, wy, wz, dl->mo);
      }
    }
  }
#endif

  // link it in
  if (connect_low || connect_high)
  {
    R2_DrawPlane(cur_sub, plane);
    R2_CommitOpenings(0);
    R2_CommitDrawPlane(0);
  }
  else
  {
    plane->next = dfloor->planes;
    dfloor->planes = plane;

    R2_CommitOpenings(x2 - x1 + 1);
    R2_CommitDrawPlane(1);
  }
}

//
// R2_GetThingSprite
//
// Can return NULL, for no image.
//
const image_t * R2_GetThingSprite(mobj_t *mo, boolean_t *flip)
{
  spritedef_t *sprite;
  spriteframe_t *frame;

  angle_t ang;
  int rot;

  // decide which patch to use for sprite relative to player

#ifdef DEVELOPERS
  // this shouldn't happen
  if ((unsigned int)mo->sprite >= (unsigned int)numsprites)
    I_Error("R2_GetThingSprite: invalid sprite number %i.\n", mo->sprite);
#endif

  sprite = sprites + mo->sprite;

  if (mo->frame >= sprite->numframes ||
      !sprite->frames[mo->frame].finished)
  {
#if 1
    // -AJA- 2001/08/04: allow the patch to be missing
    (*flip) = false;
    return W_ImageForDummySprite();
#else
    // -ACB- 1998/06/29 More Informative Error Message
    I_Error("R2_GetThingSprite: Invalid sprite frame %s:%c",
        sprite->name, 'A' + mo->frame);
#endif
  }

  frame = sprite->frames + mo->frame;

  if (frame->rotated)
  {
    // choose a different rotation based on player view
    ang = R_PointToAngle(viewx, viewy, mo->x, mo->y) -
        mo->angle + ANG180 + (ANG45 / (frame->extended ? 4 : 2));
    rot = ANG_2_ROT(ang);
  }
  else
  {
    // use single rotation for all views
    rot = 0;
  }

  DEV_ASSERT2(0 <= rot && rot < 16);

  (*flip) = frame->flip[rot] ? true : false;

  return frame->images[rot];
}

//
// R2_GetOtherSprite
//
// Used for non-object stuff, like weapons and finale.
//
const image_t * R2_GetOtherSprite(int spritenum, int framenum, boolean_t *flip)
{
  spritedef_t *sprite;
  spriteframe_t *frame;

#ifdef DEVELOPERS
  // this shouldn't happen
  if ((unsigned int)spritenum >= (unsigned int)numsprites)
    I_Error("R2_GetThingSprite: invalid sprite number %i.\n", spritenum);
#endif

  sprite = sprites + spritenum;

  if (framenum >= sprite->numframes ||
      !sprite->frames[framenum].finished)
  {
#if 1
    return NULL;
#else
    // -ACB- 1998/06/29 More Informative Error Message
    I_Error("R2_GetOtherSprite: Invalid sprite frame %s:%c",
        sprite->name, 'A' + framenum);
#endif
  }

  frame = sprite->frames + framenum;

  *flip = frame->flip[0] ? true : false;

  return frame->images[0];
}


//
// R2_WalkThing
//
// Visit a single thing that exists in the current subsector.
//
static void R2_WalkThing(mobj_t *mo)
{
  drawthing_t *dthing;
  screenline_t *area;
  
  float_t tr_x;
  float_t tr_y;

  float_t tx, tx1, tx2;
  float_t tz;

  float_t gzb, gzt;

  float_t xscale;
  float_t yscale;
  float_t dist_scale;

  int x1, x2;
  int offset = 0;

  const image_t *image;
  int sprite_height;
  int top_offset;

  boolean_t spr_flip;
  int clip_vert = 0;

  // ignore invisible things
  if (mo->visibility == INVISIBLE)
      return;

  // transform the origin point
  tr_x = mo->x - viewx;
  tr_y = mo->y - viewy;

  tz = tr_x * viewcos + tr_y * viewsin;

  // thing is behind view plane?
  if (tz < MINZ)
    return;

  // -ES- 1999/03/14 Use distunit for y and x scale.
  xscale = x_distunit / tz;
  yscale = y_distunit / tz;
  dist_scale = yscale;

  tx = tr_x * viewsin - tr_y * viewcos;

  // too far off the side?
  // -ES- 1999/03/13 Fixed clipping to work with large FOVs (up to 176 deg)
  // rejects all sprites where angle>176 deg (arctan 32), since those
  // sprites would result in overflow in future calculations
  if (fabs(tx) / 32 > tz)
    return;
  
  image = R2_GetThingSprite(mo, &spr_flip);

  if (!image)
    return;

  // calculate edges of the shape
  tx1 = tx - IM_OFFSETX(image) * mo->info->xscale;

  x1 = floor(focusxfrac + tx1 * xscale);

  // off the right side ?
  if (x1 >= viewwidth)
  {
    return;
  }
  else if (x1 < 0)
  {
    offset = 0 - x1;
    x1 = 0;
  }

  tx2 = tx1 + IM_WIDTH(image) * mo->info->xscale;

  x2 = floor(focusxfrac + tx2 * xscale) - 1;

  // off the left side ?
  if (x2 < 0)
    return;

  x2 = MIN(x2, viewwidth-1);

  // too narrow to be seen
  if (x1 > x2)
    return;

  xscale *= mo->info->xscale;
  yscale *= mo->info->yscale;

  sprite_height = IM_HEIGHT(image);
  top_offset = IM_OFFSETY(image);

  gzt = mo->z + top_offset * mo->info->yscale;
  gzb = gzt - sprite_height * mo->info->yscale;

  // fix for sprites that sit wrongly into the floor/ceiling

  if (sprite_kludge==0 && gzb < mo->floorz)
  {
    // explosion ?
    if (mo->info->flags & MF_MISSILE)
    {
      clip_vert = +1;
    }
    else
    {
      gzt += mo->floorz - gzb;
      gzb = mo->floorz;
    }
  }
  else if (sprite_kludge==0 && gzt > mo->ceilingz)
  {
    // explosion ?
    if (mo->info->flags & MF_MISSILE)
    {
      clip_vert = +1;
    }
    else
    {
      gzb -= gzt - mo->ceilingz;
      gzt = mo->ceilingz;
    }
  }

  if (gzb >= gzt)
    return;

  // create new draw thing

  dthing = R2_GetDrawThing();
  R2_CommitDrawThing(1);

  area = &dthing->area;

  dthing->mo = mo;
  dthing->clipped_left = dthing->clipped_right = false;
  dthing->props = cur_sub->floors->props;

  dthing->image  = image;
  dthing->flip   = spr_flip;
  dthing->bright = mo->bright ? true : false;
  dthing->clip_vert = clip_vert;
  dthing->is_halo = false;
  dthing->is_shadow = false;
  
  dthing->xscale = xscale;
  dthing->yscale = yscale;
  dthing->ixscale = 1.0f / xscale;
  dthing->iyscale = 1.0f / yscale;
  dthing->dist_scale = dist_scale;
  dthing->xfrac = offset * dthing->ixscale;

  dthing->tx = tx;
  dthing->tz = tz;
  dthing->tx1 = tx1;
  dthing->tx2 = tx2;

  dthing->top = gzt;
  dthing->bottom = gzb;

  // setup area.  NOTE: ranges are created later.
  area->x1 = x1;
  area->x2 = x2;
  area->ranges = NULL;  // note used for drawthings

  area->y = 0;
  area->step = 0;
  area->y_offset = 0;

  // translation support
  if (mo->info->palremap)
    dthing->trans_table = V_GetTranslationTable(mo->info->palremap);
  else
    dthing->trans_table = NULL;

  // link it in
  dthing->next = cur_sub->raw_things;
  dthing->prev = NULL;

  if (cur_sub->raw_things)
    cur_sub->raw_things->prev = dthing;

  cur_sub->raw_things = dthing;

  // create shadow
  if (level_flags.shadows && mo->info->shadow_trans > 0 &&
      mo->floorz < viewz && ! IS_SKY(mo->subsector->sector->floor))
  {
    drawthing_t *dshadow = R2_GetDrawThing();
    R2_CommitDrawThing(1);

    dshadow[0] = dthing[0];

    dshadow->is_shadow = true;
    dshadow->clip_vert = -1;
    dshadow->trans_table = NULL;

    // shadows are 1/4 the height
    dshadow->yscale /= 4.0;
    dshadow->iyscale *= 4.0;

    gzb = mo->floorz;
    gzt = gzb + sprite_height / 4.0 * mo->info->yscale;

    dshadow->top = gzt;
    dshadow->bottom = gzb;

    // link it in
    dshadow->next = cur_sub->raw_things;
    dshadow->prev = NULL;

    if (cur_sub->raw_things)
      cur_sub->raw_things->prev = dshadow;

    cur_sub->raw_things = dshadow;
  }
}

//
// R2_ClipSpriteHorizontally
//
// Checks to see if the sprite crosses the given clipseg.  If it does,
// then the current drawthing is split, and a new drawthing for the
// other half is added to the corresponding drawsub.  If the _whole_
// drawthing was moved to the other drawsub, then `true' is returned,
// otherwise returns false.  Note: `dthing' must be unlinked from any
// list.
//
#define SX_FUDGE  1

static boolean_t R2_ClipSpriteHorizontally(subsector_t *dsub, 
  drawthing_t *dthing, seg_t *clipper)
{
  float_t x, dx, dz;
  int new_x;

  drawthing_t *dnew;
  subsector_t *border;

  // quick distance check
  if (MIN(clipper->tz1, clipper->tz2) > dthing->tz ||
      MAX(clipper->tz1, clipper->tz2) < dthing->tz)
  {
    return false;
  }

  // compute intersection point
  dx = clipper->tx2 - clipper->tx1;
  dz = clipper->tz2 - clipper->tz1;

  // dz guaranteed to be != 0 (since orientation != 0)
  DEV_ASSERT2(clipper->orientation != 0);
  CHECKVAL(dz);

  x = clipper->tx1 + (dthing->tz - clipper->tz1) * dx / dz;

  new_x = floor(focusxfrac + x * dthing->xscale / 
      dthing->mo->info->xscale);  // OPTIMISE

  // check if visible part of sprite is totally in this subsector, or
  // totally in the other subsector.  These checks also guarantee that
  // any split point is not too close to the edge.

  if ((clipper->orientation > 0 && new_x-SX_FUDGE <= dthing->area.x1) ||
      (clipper->orientation < 0 && new_x+SX_FUDGE >= dthing->area.x2))
  {
    return false;
  }

  border = clipper->back_sub;

  DEV_ASSERT2(border);
  DEV_ASSERT2(subsectors_seen[border - subsectors]);

  // just moving the whole sprite (no clipping) ?
  if ((clipper->orientation > 0 && new_x+SX_FUDGE >= dthing->area.x2) ||
      (clipper->orientation < 0 && new_x-SX_FUDGE <= dthing->area.x1))
  {
    // prevent cycling
    if (clipper->orientation > 0 && dthing->area.x2 > new_x-SX_FUDGE)
      dthing->clipped_right = true;

    if (clipper->orientation < 0 && dthing->area.x1 < new_x+SX_FUDGE)
      dthing->clipped_left = true;

    dthing->next = border->raw_things;
    dthing->prev = NULL;

    if (border->raw_things)
      border->raw_things->prev = dthing;

    border->raw_things = dthing;
    return true;
  }

  // OK, we need to split the sprite at the seg boundary, and put the
  // new half into the correct drawsub.  The current drawthing is
  // shortened.
  
  dnew = R2_GetDrawThing();
  R2_CommitDrawThing(1);

  dnew[0] = dthing[0];

  if (clipper->orientation > 0)
  {
    // new piece is on the left side
    dthing->xfrac += (new_x - dthing->area.x1) * dthing->ixscale;
    dthing->area.x1 = new_x;
    dthing->clipped_left = true;

    dnew->area.x2 = new_x - 1;
    dnew->clipped_right = true;
  }
  else
  {
    // new piece is on the right side
    dnew->xfrac += (new_x - dnew->area.x1) * dnew->ixscale;
    dnew->area.x1 = new_x;
    dnew->clipped_left = true;

    dthing->area.x2 = new_x - 1;
    dthing->clipped_right = true;
  }

  // put new piece where it belongs

  dnew->next = border->raw_things;
  dnew->prev = NULL;

  if (border->raw_things)
    border->raw_things->prev = dnew;

  border->raw_things = dnew;
  return false;
}

static INLINE void LinkDrawthingIntoDrawfloor(drawthing_t *dthing,
    drawfloor_t *dfloor)
{
#ifndef USE_GL
  mobj_t *mo = dthing->mo;
#endif

  dthing->props = dfloor->props;
  dthing->next  = dfloor->things;
  dthing->prev  = NULL;

  if (dfloor->things)
    dfloor->things->prev = dthing;

  dfloor->things = dthing;

  //
  // Dynamic Lighting
  //
  dthing->extra_light = 0;

#ifndef USE_GL  // GL renderer uses its own method
  if (!(mo->flags & MF_FUZZY) && !(dthing->is_shadow | dthing->is_halo))
  {
    drawthing_t *dl;

    float_t mid_z = MO_MIDZ(mo);

    for (dl=dfloor->dlights; dl; dl=dl->next)
    {
      if (mo == dl->mo)
        continue;

      R2_AddDLights(1, &dthing->extra_light, &mo->x, &mo->y, &mid_z, dl->mo);
    }
  }
#endif
}

//
// R2_ClipSpriteVertically
//
#define SY_FUDGE  2

void R2_ClipSpriteVertically(subsector_t *dsub, drawthing_t *dthing)
{
  drawfloor_t *dfloor, *df_orig;
  drawthing_t *dnew, *dt_orig;

  float_t z;
  float_t f1, c1;
  float_t f1_orig, c1_orig;
  
  // find the thing's nominal region.  This section is equivalent to
  // the R_PointInVertRegion() code (but using drawfloors).

  if (dthing->is_shadow)
    z = dthing->bottom + 0.5;
  else
    z = (dthing->top + dthing->bottom) / 2.0;

  for (dfloor = dsub->z_floors; dfloor->higher; dfloor = dfloor->higher)
  {
    if (z <= dfloor->top_h)
      break;
  }

  DEV_ASSERT2(dfloor);

  // link in sprite.  We'll shrink it if it gets clipped.
  LinkDrawthingIntoDrawfloor(dthing, dfloor);

  // handle never-clip things 
  if (dthing->clip_vert < 0)
    return;
    
  // Note that sprites are not clipped by the lowest floor or
  // highest ceiling, OR by *solid* extrafloors (even translucent
  // ones) -- UNLESS clip_vert is > 0.

  f1 = dfloor->f_h;
  c1 = dfloor->c_h;

  // handle TRANSLUCENT + THICK floors (a bit of a hack)
  if (dfloor->ef && dfloor->ef->ef_info && dfloor->higher &&
      (dfloor->ef->ef_info->type & EXFL_Thick) &&
      (dfloor->ef->top->translucency <= 0.99))
  {
    c1 = dfloor->top_h;
  }

  df_orig = dfloor;
  dt_orig = dthing;
  f1_orig = f1;
  c1_orig = c1;

  // Two sections here: Downward clipping (for sprite's bottom) and
  // Upward clipping (for sprite's top).  Both use the exact same
  // algorithm:
  //     
  //    WHILE (current must be clipped)
  //    {
  //       new := COPY OF current
  //       clip both current and new to the clip height
  //       current := new
  //       floor := NEXT floor
  //       link current into floor
  //    }
 
  // ---- downward section ----

  for (;;)
  {
    if (!dfloor->lower)
      break;

    if ((dthing->bottom >= f1 - SY_FUDGE) ||
        (dthing->top    <  f1 + SY_FUDGE))
      break;

    DEV_ASSERT2(dfloor->lower->ef && dfloor->lower->ef->ef_info);

    if (! (dfloor->lower->ef->ef_info->type & EXFL_Liquid))
      break;
    
    // sprite's bottom must be clipped.  Make a copy.

    dnew = R2_GetDrawThing();
    R2_CommitDrawThing(1);

    dnew[0] = dthing[0];

    // shorten current sprite
    
    dthing->bottom = f1;
    
    DEV_ASSERT2(dthing->bottom < dthing->top);

    // shorten new sprite
    
    dnew->area.y_offset += (dnew->top - f1);
    dnew->top = f1;

    DEV_ASSERT2(dnew->bottom < dnew->top);

    // time to move on...

    dthing = dnew;
    dfloor = dfloor->lower;

    f1 = dfloor->f_h;
    c1 = dfloor->c_h;

    // handle TRANSLUCENT + THICK floors (a bit of a hack)
    if (dfloor->ef && dfloor->ef->ef_info && dfloor->higher &&
        (dfloor->ef->ef_info->type & EXFL_Thick) &&
        (dfloor->ef->top->translucency <= 0.99))
    {
      c1 = dfloor->top_h;
    }

    // link new piece in
    LinkDrawthingIntoDrawfloor(dthing, dfloor);
  }

  // when clip_vert is > 0, we must clip to solids
  if (dthing->clip_vert > 0 &&
      dthing->bottom <  f1 - SY_FUDGE &&
      dthing->top    >= f1 + SY_FUDGE)
  {
    // shorten current sprite
    
    dthing->bottom = f1;
    
    DEV_ASSERT2(dthing->bottom < dthing->top);
  }

  dfloor = df_orig;
  dthing = dt_orig;
  f1 = f1_orig;
  c1 = c1_orig;
  
  // ---- upward section ----
  
  for (;;)
  {
    if (!dfloor->higher)
      break;

    if ((dthing->bottom >= c1 - SY_FUDGE) ||
        (dthing->top    <  c1 + SY_FUDGE))
      break;

    DEV_ASSERT2(dfloor->ef && dfloor->ef->ef_info);

    if (! (dfloor->ef->ef_info->type & EXFL_Liquid))
      break;
 
    // sprite's top must be clipped.  Make a copy.

    dnew = R2_GetDrawThing();
    R2_CommitDrawThing(1);

    dnew[0] = dthing[0];

    // shorten current sprite
    
    dthing->area.y_offset += (dthing->top - c1);
    dthing->top = c1;
    
    DEV_ASSERT2(dthing->bottom < dthing->top);

    // shorten new sprite
    
    dnew->bottom = c1;

    DEV_ASSERT2(dnew->bottom < dnew->top);

    // time to move on...

    dthing = dnew;
    dfloor = dfloor->higher;

    f1 = dfloor->f_h;
    c1 = dfloor->c_h;

    // handle TRANSLUCENT + THICK floors (a bit of a hack)
    if (dfloor->ef && dfloor->ef->ef_info && dfloor->higher &&
        (dfloor->ef->ef_info->type & EXFL_Thick) &&
        (dfloor->ef->top->translucency <= 0.99))
    {
      c1 = dfloor->top_h;
    }

    // link new piece in
    LinkDrawthingIntoDrawfloor(dthing, dfloor);
  }

  // when clip_vert is > 0, we must clip to solids
  if (dthing->clip_vert > 0 &&
      dthing->bottom <  c1 - SY_FUDGE &&
      dthing->top    >= c1 + SY_FUDGE)
  {
    // shorten current sprite
 
    dthing->area.y_offset += dthing->top - c1;
    dthing->top = c1;

    DEV_ASSERT2(dthing->bottom < dthing->top);
  }
}

//
// R2_ClipOneSprite
//
// Checks if drawthing needs to be clipped against any clipsegs in the
// drawsub, clipping the remaining part (if any) onto the appropriate
// drawfloors.  Note: `dthing' must be unlinked from any list.
//
static void R2_ClipOneSprite(subsector_t *dsub, drawthing_t *dthing)
{
  seg_t *clipper;
 
  for (clipper=dsub->segs; clipper; clipper=clipper->sub_next)
  {
    // don't clip against 1-sided lines (or invisible back subs)
    if (!clipper->back_sub || 
        !subsectors_seen[clipper->back_sub - subsectors])
    {
      continue;
    }
    
    // ignore clipsegs that wouldn't be used
    if (clipper->orientation == 0 ||
       (clipper->orientation > 0 && dthing->clipped_left) ||
       (clipper->orientation < 0 && dthing->clipped_right))
    {
      continue;
    }

    if (R2_ClipSpriteHorizontally(dsub, dthing, clipper))
      return;
 
    // all clipped out ? :)
    if (dthing->clipped_left && dthing->clipped_right)
      break;
  }

  // the remaining piece has now been clipped against all the
  // clipsegs.  Now clip it vertically, which stores the sprite in the
  // appropriate drawfloor(s).

  R2_ClipSpriteVertically(dsub, dthing);
}

//
// R2_ClipSprites
//
static void R2_ClipSprites(void)
{
  subsector_t *cur;
  drawthing_t *dthing;

  for (;;)
  {
    int count = 0;

    for (cur=drawsubs; cur; cur=cur->rend_next)
    {
      if (! cur->raw_things)
        continue;
      
      count++;

      while (cur->raw_things)
      {
        dthing = cur->raw_things;
        cur->raw_things = dthing->next;

        if (cur->raw_things)
          cur->raw_things->prev = NULL;

        R2_ClipOneSprite(cur, dthing);
      }
    }

    if (count == 0)
      break;
  }
}

//
// R2_FindDLights
//
void R2_FindDLights(subsector_t *sub, drawfloor_t *dfloor)
{
  int xl, xh, yl, yh;
  int bx, by;

  xl = (sub->bbox[BOXLEFT]  -bmaporgx-MAXDLIGHTRADIUS) / MAPBLOCKUNITS;
  xh = (sub->bbox[BOXRIGHT] -bmaporgx+MAXDLIGHTRADIUS) / MAPBLOCKUNITS;
  yl = (sub->bbox[BOXBOTTOM]-bmaporgy-MAXDLIGHTRADIUS) / MAPBLOCKUNITS;
  yh = (sub->bbox[BOXTOP]   -bmaporgy+MAXDLIGHTRADIUS) / MAPBLOCKUNITS;

  for (bx = xl; bx <= xh; bx++)
  for (by = yl; by <= yh; by++)
  {
    mobj_t *mo;
    drawthing_t *dl;

    if (bx < 0 || by < 0 || bx >= bmapwidth || by >= bmapheight)
      continue;

    for (mo=blocklights[by * bmapwidth + bx]; mo; mo = mo->dlnext)
    {
      if (! mo->bright || mo->dlight_qty <= 0)
        continue;

      if (mo->ceilingz <= dfloor->f_h || mo->floorz >= dfloor->top_h)
      {
        continue;
      }

      dl = R2_GetDrawThing();
      R2_CommitDrawThing(1);

      dl->mo = mo;
      dl->tz = mo->z + mo->height * PERCENT_2_FLOAT(mo->info->dlight.height);
      
      dl->next = dfloor->dlights;
      dl->prev = NULL;  // NOTE: not used (singly linked)

      dfloor->dlights = dl;
    }
  }
}

static void LightUpPlayerWeapon(player_t *p, drawfloor_t *dfloor)
{
  drawthing_t *dl;
  float_t mid_z;

  mid_z = p->mo->z + p->mo->height *
      PERCENT_2_FLOAT(p->mo->info->shotheight);

  for (dl=dfloor->dlights; dl; dl=dl->next)
  {
    R2_AddDLights(1, &extra_psp_light, &p->mo->x, &p->mo->y,
        &mid_z, dl->mo);
  }
}


static INLINE void AddNewDrawFloor(extrafloor_t *ef,
    float_t f_h, float_t c_h, float_t top_h,
    surface_t *floor, surface_t *ceil,
    region_properties_t *props)
{
  drawfloor_t *dfloor;
  drawfloor_t *tail;

  dfloor = R2_GetDrawFloor();
  R2_CommitDrawFloor(1);

  Z_Clear(dfloor, drawfloor_t, 1);

  dfloor->f_h   = f_h;
  dfloor->c_h   = c_h;
  dfloor->top_h = top_h;
  dfloor->floor = floor;
  dfloor->ceil  = ceil;
  dfloor->ef    = ef;
  dfloor->props = props;
  
  // link it in
  // (order is very important)

  if (cur_sub->floors == NULL || f_h > viewz)
  {
    // add to head
    dfloor->next = cur_sub->floors;
    dfloor->prev = NULL;

    if (cur_sub->floors)
      cur_sub->floors->prev = dfloor;

    cur_sub->floors = dfloor;
  }
  else
  {
    // add to tail
    for (tail=cur_sub->floors; tail->next; tail=tail->next)
    { /* nothing here */ }

    dfloor->next = NULL;
    dfloor->prev = tail;

    tail->next = dfloor;
  }

  // add to tail of height order list (for sprite clipping)
  for (tail=cur_sub->z_floors; tail && tail->higher; tail=tail->higher)
  { /* nothing here */ }

  dfloor->higher = NULL;
  dfloor->lower = tail;

  if (tail)
    tail->higher = dfloor;
  else
    cur_sub->z_floors = dfloor;

  if (use_dlights && blocklights)
  {
    R2_FindDLights(cur_sub, dfloor);

    if (cur_sub == viewsubsector && f_h <= viewz && viewz <= c_h)
      LightUpPlayerWeapon(consoleplayer, dfloor);
  }
}
 

//
// R2_WalkSubsector
//
// Visit a subsector, and collect information, such as where the
// walls, planes (ceilings & floors) and things need to be drawn.
//
static void R2_WalkSubsector(int num)
{
  mobj_t *mo;
  seg_t *seg;
  sector_t *sector;
  surface_t *floor_s;
  float_t floor_h;

  extrafloor_t *S, *L, *C;

  drawfloor_t *dfloor;

#if (DEBUG >= 1)
    L_WriteDebug("\nVISITING SUBSEC %d\n\n", num);
#endif

  subsectors_seen[num] = 1;

  cur_sub = subsectors + num;
  sector = cur_sub->sector;

  cur_sub->ranges = NULL;
  cur_sub->floors = cur_sub->z_floors = NULL;
  cur_sub->raw_things = NULL;

  // add in each extrafloor, traversing strictly upwards

  floor_s = &sector->floor;
  floor_h = sector->f_h;

  S = sector->bottom_ef;
  L = sector->bottom_liq;

  while (S || L)
  {
    if (!L || (S && S->bottom_h < L->bottom_h))
    {
      C = S;  S = S->higher;
    }
    else
    {
      C = L;  L = L->higher;
    }

    DEV_ASSERT2(C);

    // ignore liquids in the middle of THICK solids, or below real
    // floor or above real ceiling
    //
    if (C->bottom_h < floor_h || C->bottom_h > sector->c_h)
      continue;
    
    AddNewDrawFloor(C, floor_h, C->bottom_h, C->top_h, floor_s, C->bottom, C->p);

    floor_s = C->top;
    floor_h = C->top_h;
  }

  AddNewDrawFloor(NULL, floor_h, sector->c_h, sector->c_h, 
      floor_s, &sector->ceil, sector->p);
 
#if (DEBUG >= 1)
  L_WriteDebug("\nANALYSING SUBSEC %d\n\n", num);
#endif

  cur_x_min = viewwidth-1;
  cur_x_max = 0;

  // first pass over segs.  Used to determine all back-facing segs
  // (for planes) and other non-visible segs.  Also computes the width
  // of the subsector (cur_x_min..cur_x_max), as well as clipping info
  // for sprites.  Does NOT do any drawing or update the 1D/2D
  // buffers.
 
  for (seg=cur_sub->segs; seg; seg=seg->sub_next)
  {
    R2_WalkSeg(seg);
  }

  if (cur_x_min > cur_x_max)
    return;

  // remember the occlusion buffer, so we can clip sprites later
  
  cur_sub->x_min = cur_x_min;
  cur_sub->x_max = cur_x_max;
  cur_sub->ranges = R2_GetOpenings(cur_x_max - cur_x_min + 1);

  R2_CommitOpenings(cur_x_max - cur_x_min + 1);
  R2_2DOcclusionCopy(cur_x_min, cur_x_max, cur_sub->ranges);

  cur_sub->clip_left = (cur_x_min <= 0 ||
      R2_1DOcclusionTest(cur_x_min-1, cur_x_min-1));
  cur_sub->clip_right = (cur_x_max >= viewwidth-1 ||
      R2_1DOcclusionTest(cur_x_max+1, cur_x_max+1));

  // handle each sprite in the subsector.
  // This must be done before segs, since the wall/plane code will
  // update the 1D/2D occlusion buffers.
  
  for (mo=cur_sub->thinglist; mo; mo=mo->snext)
  {
    R2_WalkThing(mo);
  }

  // now handle the planes in each floor.  Extrafloor planes use the
  // back-facing seg information, whereas the lowest floor / highest
  // ceiling can rely on the floor/ceiling clip values (i.e. just like
  // in the old Doom renderer -- and as an added bonus :) the old
  // flat-flooding trick should work again).  E.F planes are stored,
  // but the lowest F / highest C can be drawn immediately (updating
  // the 1D/2D buffers as they go).
  //
  // NOTE: traversal must go from closest plane to furthest plane
  // (from the viewheight) for correct updating of the 1D/2D buffers.
 
  if (cur_sub->floors)
  {
    for (dfloor=cur_sub->floors; dfloor->next; dfloor=dfloor->next)
    { /* nothing */ }

    for (; dfloor; dfloor=dfloor->prev)
    {
      R2_PlaneEvent(dfloor, cur_x_min, cur_x_max, dfloor->c_h, dfloor->ceil,  -1);
      R2_PlaneEvent(dfloor, cur_x_min, cur_x_max, dfloor->f_h, dfloor->floor, +1);
    }
  }

  // second pass over segs.  Draw certain solid stuff (e.g. one-sided
  // walls), updating the 1D and 2D occlusion buffers.  Add the rest
  // (like midmasked textures) to the drawing lists to draw later.
 
  for (seg=cur_sub->segs; seg; seg=seg->sub_next)
  {
    if (!seg->miniseg && seg->visible && !seg->back)
      R2_WalkWall(seg);
  }

  // update 1D buffer wherever the 2D buffer closed up
  R2_2DUpdate1D(cur_x_min, cur_x_max);
 
  // add drawsub to list
  // (add to head, thus the eventual order is furthest -> closest)
  
  cur_sub->rend_next = drawsubs;
  drawsubs = cur_sub;
}

//
// R2_WalkBSPNode
//
// Walks all subsectors below a given node, traversing subtree
// recursively, collecting information.  Just call with BSP root.
//
static void R2_WalkBSPNode(int bspnum)
{
  node_t *node;
  int side;

  // Found a subsector?
  if (bspnum & NF_SUBSECTOR)
  {
    R2_WalkSubsector(bspnum & (~NF_SUBSECTOR));
    return;
  }

  node = &nodes[bspnum];

  // Decide which side the view point is on.
  side = P_PointOnDivlineSide(viewx, viewy, &node->div);

#if (DEBUG >= 2)
    L_WriteDebug("NODE %d (%1.1f, %1.1f) -> (%1.1f, %1.1f)  SIDE %d\n",
        bspnum, node->div.x, node->div.y, node->div.x +
        node->div.dx, node->div.y + node->div.dy, side);
#endif

  // Recursively divide front space.
  if (R2_CheckBBox(node->bbox[side]))
    R2_WalkBSPNode(node->children[side]);

  // Recursively divide back space.
  if (R2_CheckBBox(node->bbox[side ^ 1]))
    R2_WalkBSPNode(node->children[side ^ 1]);
}


//
// R2_RenderTrueBSP
//
// True BSP rendering.  Initialises all structures, then walks the BSP
// tree collecting information and drawing solid walls and outermost
// planes, then renders the remaining stuff in each subsector (sprites
// and extrafloors) from back to front.
//
void R2_RenderTrueBSP(void)
{
  subsector_t *cur;
  
  // initialise
  R2_ClearBSP();
  R2_1DOcclusionClear(0, viewwidth-1);
  R2_2DOcclusionClear(0, viewwidth-1);
  R_InitSkyMap();
  
  extra_psp_light = 0;

  drawsubs = NULL;

  if (hom_detect)
  {
    V_DrawBox(main_scr, viewwindowx, viewwindowy, 
        viewwindowwidth, viewwindowheight, pal_green);
  }

  // walk the bsp tree
  R2_WalkBSPNode(root_node);

  R2_ClipSprites();

  // draw all the stored stuff
  for (cur=drawsubs; cur; cur=cur->rend_next)
  {
    R2_DrawSubsector(cur);
  }

#if (DEBUG > 0)
    L_WriteDebug("\n--------------------------------\n\n");
#endif
}
