//----------------------------------------------------------------------------
//  EDGE True BSP Rendering (Drawing code)
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------
//
// -AJA- 1999/08/31: Wrote this file.
//
// TODO HERE:
//   + R2_DrawPSprite: clean up.
//   - implement Fuzzy_MIP column and HoleyTrans span drawers.
//   - WALL SKEW is wrong when image_t is scaled.
//   - FLOOR ALIGNMENT is wrong when image_t is scaled.
//

#include "i_defs.h"

#include "dm_state.h"
#include "m_inline.h"
#include "m_swap.h"
#include "e_search.h"
#include "r_data.h"
#include "r_defs.h"
#include "r_main.h"
#include "r_plane.h"
#include "r_sky.h"
#include "r_state.h"
#include "r_things.h"
#include "v_colour.h"
#include "v_res.h"
#include "v_ctx.h"
#include "w_image.h"
#include "w_wad.h"
#include "w_textur.h"
#include "z_zone.h"

#include "r2_defs.h"
#include "r_draw1.h"


#define SHOW_WALLS   2
#define SHOW_PLANES  2
#define SHOW_THINGS  2
#define SHOW_SKY     2
#define SHOW_WEAPON  2

#define SHOW_OVERDRAW  0
#define SHOW_FIRST_IN  0


static void R2_DrawSkyPlane(drawplane_t *plane);

// FIXME:
extern float_t basexscale;
extern float_t baseyscale;


#if (SHOW_WALLS<2 || SHOW_PLANES<2 || SHOW_THINGS<2 || SHOW_SKY<2 || SHOW_WEAPON<2)

// Temporary routine
static void BOGUS_Column16(screen_t * scr, int x, int y, int h, int col)
{
  short *dest;

  x += viewwindowx;
  y += viewwindowy;

  dest = (short *)(scr->data + y * scr->pitch) + x;

  for (; h > 0; h--)
  {
      dest[0] = col;
      dest += scr->pitch / 2;
  }
}

// Temporary routine
static void BOGUS_Column(screen_t * scr, int x, int y, int h, int col)
{
  byte *dest;

  if (BPP == 2)
  {
    BOGUS_Column16(scr, x, y, h, pixel_values[col & 0xFF]);
    return;
  }
  else if (BPP != 1)
    return;

  x += viewwindowx;
  y += viewwindowy;

  dest = scr->data + y * scr->pitch + x;

  for (; h > 0; h--, dest += scr->pitch)
  {
#if SHOW_OVERDRAW
    if (col > 5 && *dest != 1) 
    {
      if (SHOW_FIRST_IN)
        *dest = col;
      else if (*dest >= 192)
        *dest += 2;
      else
        *dest = 192;
    }
    else 
#endif
      *dest = col;
  }
}

static void BOGUS_DrawArea(screenline_t *area, int col)
{
  int x, y1, y2;

  for (x=area->x1; x <= area->x2; x++)
  {
    if (x < 0 || x >= viewwidth)
      continue;

    y1 = area->ranges[x - area->x1].y1;
    y2 = area->ranges[x - area->x1].y2;

    y1 = MAX(y1, 0);
    y2 = MIN(y2, viewheight-1);

    if (y1 > y2)
      continue;

    BOGUS_Column(main_scr, x, y1, y2-y1+1, col);
  }
}
#endif

//
// R2_DrawMaskedWPost
//
// Render a single column of a texture with transparent parts.  Used
// for both sprites and mid-masked textures on 2S lines.
//
static float_t mask_top;
static float_t mask_scale;
static float_t mask_yl;
static float_t mask_yh;
static float_t mask_translucency;

static void R2_DrawMaskedWPost(const w_post_t * column)
{
  int yp;

  float_t raw_yl;
  float_t raw_yh;
  float_t raw_top;

  CHECKVAL(mask_scale);
  dc_ystep = M_FloatToFixed(1.0 / mask_scale) - 1;
  dc_translucency = M_FloatToFixed(mask_translucency);

  // handle each post in turn
  for (yp=0; WPOST_SKIP(column) != P_SENTINEL; column=WPOST_NEXT(column))
  {
    // calculate unclipped screen coordinates for post
    yp += WPOST_SKIP(column);
    raw_yl = mask_top + yp * mask_scale;
    raw_top = raw_yl;

    yp += WPOST_LEN(column);
    raw_yh = mask_top + yp * mask_scale;

    // clip post to given area (which must lie within the screen)
    if (raw_yl < mask_yl)
      raw_yl = mask_yl;

    if (raw_yh > mask_yh+1)
      raw_yh = mask_yh+1;

    dc_yl = floor(raw_yl);
    dc_yh = floor(raw_yh) - 1;

    if (dc_yh >= dc_yl)
    {
      dc_source = WPOST_DATA(column);
      dc_yfrac = FLOAT_2_FIX((dc_yl + 1 - raw_top) / mask_scale);

      (* colfunc)();
    }
  }
}

static float_t wall_scale_x;
static float_t wall_scale_y;
static int wall_base_lite;
static int wall_curlightlev;

static struct
{
  int mip;
  const cached_image_t *cim;
}
wall_image_cache;

static void R2_MapWallColumn(drawwall_t *wall, int x, int offset,
    fixed_t texcol, fixed_t texcol_next, int y1, int y2)
{
  float_t yt, yp, scale;

  int lightlev, lightdist;
  int column;

  const cached_image_t *cim;
  const byte *cim_block = NULL;
  int mip;

  mip = 0;

  if (use_mipmapping && ! wall->is_masked)
  {
    fixed_t tmp_ystep, tmp_zstep, tmp_step;

    tmp_zstep = ABS(texcol_next - texcol);

    scale = wall->scale1 + (x - wall->area.x1) * wall->scale_step;
    scale *= wall_scale_y * wall->part->y_mat.y;

    CHECKVAL(scale);

    tmp_ystep = FLOAT_2_FIX(1.0 / ABS(scale));
    tmp_step = MAX(tmp_ystep, tmp_zstep);

    for (; tmp_step > FRACSQRT2; tmp_step >>= 1)
      mip++;

    // make sure that mip values don't *decrease* as we go along the
    // wall.  The column drawing order should guarantee increases.
    mip = MAX(mip, wall_image_cache.mip);
  }

  dc_width  = MIP_SIZE(wall->part->image->total_w, mip);
  dc_height = MIP_SIZE(wall->part->image->total_h, mip);

  // see if this image (at this mip) is in the cache
  if (wall_image_cache.cim && mip == wall_image_cache.mip)
  {
    cim = wall_image_cache.cim;
  }
  else
  {
    if (wall_image_cache.cim)
      W_ImageDone(wall_image_cache.cim);

    cim = W_ImageCache(wall->part->image, 
        wall->is_masked ? IMG_Post : IMG_Block, mip, true);

    wall_image_cache.cim = cim;
    wall_image_cache.mip = mip;
  }

  if (! wall->is_masked)
    cim_block = W_ImageGetBlock(cim);
  
  // ------ //
  
  dc_x  = x;
  dc_yl = y1;
  dc_yh = y2;

  if (dc_yh < dc_yl || dc_yh < 0 || dc_yl >= viewheight)
    return;

  if (dc_yl < 0)
    dc_yl = 0;

  if (dc_yh >= viewheight) 
    dc_yh = viewheight-1;

  yt = wall->area.y + offset * wall->area.step;
  yp = wall->area.y_offset;
  
  scale = wall->scale1 + offset * wall->scale_step;
  CHECKVAL(scale);
  CHECKVAL(y_distunit);

  lightdist = (int) (scale * 160.0 * 16.0 / y_distunit);
  lightdist = MIN(MAXLIGHTSCALE-1, lightdist);

  // skew support
  yp += (FIX_2_FLOAT(texcol) - wall->part->offset.x) * wall->part->y_mat.x;
  
  // scale support
  scale *= wall_scale_y * wall->part->y_mat.y;
  yp    /= wall_scale_y * wall->part->y_mat.y;

  column = texcol >> FRACBITS;

  // determine colourmap

  lightlev = wall_base_lite;

  DEV_ASSERT2(currentmap);
  if (currentmap->lighting <= LMODEL_Doomish)
  {
    lightlev = scalelight[lightlev >> LIGHTSEGSHIFT][lightdist];
  }
  
#if 0  // DLIGHT stuff
  lightlev += e_lite_0 + e_lite_diff * offset / width;
#endif
  lightlev = MIN(255, MAX(0, lightlev));

  if (lightlev != wall_curlightlev)
  {
    wall_curlightlev = lightlev;
    dc_colourmap = V_GetColtable(wall->props->colourmap, lightlev, 0);
  }

  if (! wall->is_masked)
  {
    CHECKVAL(scale);

    dc_ystep = FLOAT_2_FIX(1.0 / scale);
    dc_yfrac = FLOAT_2_FIX((dc_yl + 1 - yt) / scale + yp);

    dc_yfrac /= (wall->part->image->total_h / dc_height);
    dc_ystep /= (wall->part->image->total_h / dc_height);

    column &= (wall->part->image->total_w - 1);
    column /= (wall->part->image->total_w / dc_width);
    
    dc_source = cim_block + dc_height * column;

    // -AJA- screen coords are now absolute
    dc_x  += viewwindowx;
    dc_yl += viewwindowy;
    dc_yh += viewwindowy;

    (* colfunc)();
  }
  else
  {
    const w_post_t *cim_col;

    if (wall->slide_type != SLIDE_None || (wall->seg && 
        wall->seg->linedef && wall->seg->linedef->special && 
        wall->seg->linedef->special->s.type != SLIDE_None))
    {
      if (wall->side == 1)
        column = wall->line_len - column;
    }

    if (wall->slide_type == SLIDE_Right)
    {
      column -= floor(wall->opening);
      if (column < 0)
        return;
    }
    else if (wall->slide_type == SLIDE_Left)
    {
      column += floor(wall->opening);
      if (column > wall->line_len)
        return;
    }
    else if (wall->slide_type == SLIDE_Center)
    {
      float_t pos1 = (wall->line_len - wall->opening) / 2;
      float_t pos2 = (wall->line_len + wall->opening) / 2;

      if (pos1 <= column && column <= pos2)
        return;
      
      if (column < pos1)
        column += wall->opening/2;
      else
        column -= wall->opening/2;
    }

    // -AJA- screen coords are now absolute
    dc_x  += viewwindowx;
    dc_yl += viewwindowy;
    dc_yh += viewwindowy;

    mask_yl = (float_t) dc_yl;
    mask_yh = (float_t) dc_yh;
    mask_scale = scale;
    mask_top = yt + viewwindowy - yp * mask_scale;

    cim_col = W_ImageGetPost(cim, column);

    if (cim_col)
      R2_DrawMaskedWPost(cim_col);
  }
}

//
// R2_DrawWall
//
// Render a single wall of a subsector.
//
void R2_DrawWall(subsector_t *dsub, drawwall_t *wall)
{
#if (SHOW_WALLS == 2)
  int x;
  int x_start, x_end, x_step;
  int offset;
  int width;

  float_t xp;
  int e_lite_0, e_lite_diff;
  fixed_t texcol1, texcol2;

  angle_t angle;

  wall_base_lite = wall->props->lightlevel;

  DEV_ASSERT2(currentmap);
  if (currentmap->lighting == LMODEL_Doom)
  {
    if (wall->seg->v1->y == wall->seg->v2->y)
      wall_base_lite += 8;
    else if (wall->seg->v1->x == wall->seg->v2->x)
      wall_base_lite -= 8;
    
    wall_base_lite = MAX(0, MIN(255, wall_base_lite));
  }
  
  mask_translucency = 1.0;
  colfunc = R_DrawColumn_MIP;

  if (wall->part->translucency <= 0.99 && level_flags.trans)
  {
    colfunc = R_DrawTranslucentColumn_MIP;
    mask_translucency = wall->part->translucency;
    dc_translucency = FLOAT_2_FIX(mask_translucency);
  }

  DEV_ASSERT2(wall->area.x1 <= wall->area.x2);

  // draw columns Left->Right to ensure increasing distance
  if (wall->seg->scale1 < wall->seg->scale2)
  {
    x_start = wall->area.x2;
    x_end   = wall->area.x1 - 1;
    x_step  = -1;
  }
  else
  {
    x_start = wall->area.x1;
    x_end   = wall->area.x2 + 1;
    x_step  = +1;
  }

  // calculate initial texture column
  xp = wall->x_offset;
  angle = viewangle - wall->angle + xtoviewangle[x_start] - ANG90;

  if (! (ANG90-ANG1  <= angle && angle <= ANG90+ANG1) &&
      ! (ANG270-ANG1 <= angle && angle <= ANG270+ANG1))
  {
    xp -= M_Tan(angle) * wall->distance;
  }

  wall_scale_x = (float_t)0x0100 / wall->part->image->scale_x;
  wall_scale_y = (float_t)0x0100 / wall->part->image->scale_y;

  CHECKVAL(wall->part->x_mat.x);
  texcol1 = M_FloatToFixed(xp / wall_scale_x / wall->part->x_mat.x);

  wall_curlightlev = -999;
  e_lite_0    = wall->extra_light[0];
  e_lite_diff = wall->extra_light[1] - e_lite_0;
  width = wall->area.x2 - wall->area.x1 + 1;

  wall_image_cache.cim = NULL;
  wall_image_cache.mip = 0;

  // --- draw the columns ---
  
  for (x = x_start; x != x_end; x += x_step, texcol1=texcol2)
  {
    offset = x - wall->area.x1;

    // calculate _next_ texture column
    // NOTE: assumes xtoviewangle[] array is 1 bigger than screen
    xp = wall->x_offset;
    angle = viewangle - wall->angle + xtoviewangle[x+x_step] - ANG90;

    if (! (ANG90-ANG1  <= angle && angle <= ANG90+ANG1) &&
        ! (ANG270-ANG1 <= angle && angle <= ANG270+ANG1))
    {
      xp -= M_Tan(angle) * wall->distance;
    }

    texcol2 = M_FloatToFixed(xp / wall_scale_x / wall->part->x_mat.x);

    R2_MapWallColumn(wall, x, offset, texcol1, texcol2,
        wall->area.ranges[offset].y1, wall->area.ranges[offset].y2);
  }

  if (wall_image_cache.cim)
    W_ImageDone(wall_image_cache.cim);

#elif (SHOW_WALLS == 1)
  BOGUS_DrawArea(&wall->area, 
      BROWN + BROWN_LEN * (255 - wall->props->lightlevel) / 256);
#endif /* SHOW_WALLS */
}


static const colourmap_t *plane_colourmap;
static surface_t *plane_data;
static float_t plane_h;
static float_t plane_translucency;
static int plane_base_light;
static int plane_cur_light;

static struct
{
  int mip;
  const cached_image_t *cim;
  const byte *block;
}
plane_image_cache[2];

static void R2_MapPlaneRow(drawplane_t *p, int cach_num, 
    int y, int x1, int x2)
{
  angle_t angle;
  float_t distance;
  float_t length;
  float_t SCL = 1.0;  //!!!! FIXME: 64.0 / plane_data->scale;
  float_t dx, dy;
  float_t scale_x, scale_y;
  int lightlev, lightdist;
  int index;
#if 0  // DLIGHT stuff
  int e_lite_0, e_lite_diff;
#endif

  const cached_image_t *cim;
  int mip;

#ifdef DEVELOPERS
  if (x2 < x1 || x1 < 0 || x2 >= viewwidth || y < 0 || y >= viewheight)
    I_Error("R2_MapPlaneRow: y=%d x=%d..%d", y, x1, x2);
#endif

  scale_x = (float_t)0x0100 / plane_data->image->scale_x;
  scale_y = (float_t)0x0100 / plane_data->image->scale_y;

  distance = fabs(plane_h - viewz) * yslope[y];
  length = distance * distscale[x1];
  angle = viewangle + xtoviewangle[x1];

  mip = 0;

  if (use_mipmapping)
  {
    int y2;
    float_t dist_next;
    fixed_t tmp_xstep, tmp_zstep, tmp_step;

    DEV_ASSERT2(0 <= y && y < SCREENHEIGHT);

    // compute tmp_zstep, which is the maximum distance in texture
    // space that we stepping from this span to the next one.
    if (plane_h < viewz)
      y2 = MIN(y+1, SCREENHEIGHT-1);
    else
      y2 = MAX(y-1, 0);

    dist_next = fabs(plane_h - viewz) * yslope[y2];

    tmp_xstep = M_FloatToFixed(ABS(distance * SCL / x_distunit));
    tmp_zstep = M_FloatToFixed(ABS(distance - dist_next) * SCL);
    tmp_step = MAX(tmp_xstep, tmp_zstep);

    for (; tmp_step > FRACSQRT2; tmp_step >>= 1)
      mip++;
  }

  ds_width  = MIP_SIZE(plane_data->image->total_w, mip);
  ds_height = MIP_SIZE(plane_data->image->total_h, mip);

  // see if this image (at this mip) is in the cache
  if (plane_image_cache[cach_num].cim && mip ==
      plane_image_cache[cach_num].mip)
  {
    cim = plane_image_cache[cach_num].cim;
    ds_source = plane_image_cache[cach_num].block;
  }
  else if (plane_image_cache[cach_num ^ 1].cim && mip ==
      plane_image_cache[cach_num ^ 1].mip)
  {
    cim = plane_image_cache[cach_num ^ 1].cim;
    ds_source = plane_image_cache[cach_num ^ 1].block;
  }
  else
  {
    if (plane_image_cache[cach_num].cim)
      W_ImageDone(plane_image_cache[cach_num].cim);

    cim = W_ImageCache(plane_data->image, IMG_Block, mip, true);
    ds_source = W_ImageGetBlock(cim);

    plane_image_cache[cach_num].cim = cim;
    plane_image_cache[cach_num].mip = mip;
    plane_image_cache[cach_num].block = ds_source;
  }

  ds_y  = y;
  ds_x1 = x1;
  ds_x2 = x2;

  ds_xstep = M_FloatToFixed(distance * SCL / scale_x * basexscale);
  ds_ystep = M_FloatToFixed(distance * SCL / scale_y * baseyscale);

  // -AJA- 1999/07/16: Add offsets (thanks Lee).
  // -AJA- NB: Using M_Cos/M_Sin here really slows things down.

  dx = FIX_2_FLOAT(finecosine[angle >> ANGLETOFINESHIFT]) * length;
  dy = FIX_2_FLOAT(finesine  [angle >> ANGLETOFINESHIFT]) * length;
 
  // FIXME: clean up !!!
  { 
    float_t tx = viewx + plane_data->offset.x + dx;
    float_t ty = viewy + plane_data->offset.y + dy;

    dx = tx ; //!!!! * M_Cos(plane_data->angle) + ty * M_Sin(plane_data->angle);
    dy = ty ; //!!!! * M_Cos(plane_data->angle) - tx * M_Sin(plane_data->angle);
  }
  
  // Doom flats are inverted W.R.T. the Y coordinate
  ds_xfrac =  M_FloatToFixed(dx * SCL / scale_x);
  ds_yfrac = -M_FloatToFixed(dy * SCL / scale_y);

  lightlev = plane_base_light;
  
  DEV_ASSERT2(currentmap);
  if (currentmap->lighting <= LMODEL_Doomish)
  {
    index = 1 << (LIGHTZSHIFT - FRACBITS);
    CHECKVAL(index);

    lightdist = distance / index;
    lightdist = MAX(0, MIN(MAXLIGHTZ-1, lightdist));
    
    lightlev = zlight[lightlev >> LIGHTSEGSHIFT][lightdist];
  }

#if 0  // DLIGHT stuff, disabled for now
  if (use_dlights && p->min_y >= 0)
  {
    int offset = y - p->min_y;
    int height = p->max_y - p->min_y + 1;

    DEV_ASSERT2(0 <= offset && offset < height);

    e_lite_0    = p->extra_light[0];
    e_lite_diff = p->extra_light[1] - e_lite_0;

    lightlev += (e_lite_0 + e_lite_diff * offset / height);
    lightlev = MAX(0, MIN(255, lightlev));
  }
#endif

  if (lightlev != plane_cur_light)
  {
    plane_cur_light = lightlev;
    ds_colourmap = V_GetColtable(plane_colourmap, lightlev, 0);
  }

  ds_xfrac /= (plane_data->image->total_w / ds_width);
  ds_xstep /= (plane_data->image->total_w / ds_width);

  ds_yfrac /= (plane_data->image->total_h / ds_height);
  ds_ystep /= (plane_data->image->total_h / ds_height);

  // -AJA- screen coords are now absolute
  ds_y  += viewwindowy;
  ds_x1 += viewwindowx;
  ds_x2 += viewwindowx;

  (* spanfunc)();
}

static INLINE void R2_MakeSpans(drawplane_t *p, 
    int x, int t1, int b1, int t2, int b2)
{
  for (; t1 < t2 && t1 <= b1; t1++)
    R2_MapPlaneRow(p, 0, t1, spanstart[t1], x-1);

  for (; b1 > b2 && t1 <= b1; b1--)
    R2_MapPlaneRow(p, 1, b1, spanstart[b1], x-1);

  for (; t2 < t1 && t2 <= b2; t2++)
    spanstart[t2] = x;

  for (; b2 > b1 && t2 <= b2; b2--)
    spanstart[b2] = x;
}

//
// R2_DrawPlane
//
// Render a single plane of a subsector.
//
void R2_DrawPlane(subsector_t *dsub, drawplane_t *plane)
{
#if (SHOW_PLANES == 2)
  int x;
  int L_yt = 9999, L_yb = 0;
  int R_yt, R_yb;

  if (IS_SKY(plane->info[0]))
  {
    R2_DrawSkyPlane(plane);
    return;
  }

  plane_h = plane->h;
  plane_data = plane->info;
  plane_translucency = plane_data->translucency;

  if (plane_translucency < 0.02)
    // transparent
    return;

  dc_translucency = FLOAT_2_FIX(plane_translucency);

  plane_cur_light = -999;
  plane_base_light = plane->props->lightlevel;
  plane_colourmap = plane->props->colourmap;
  
  plane_image_cache[0].cim = NULL;
  plane_image_cache[1].cim = NULL;
 
  // choose the appropriate span function
  if (! plane_data->image->solid)
    spanfunc = R_DrawHoleySpan_MIP;
  else if (plane_translucency <= 0.99 && level_flags.trans)
    spanfunc = R_DrawTranslucentSpan_MIP;
  else
    spanfunc = R_DrawSpan_MIP;

  basexscale =  M_Cos(viewangle - ANG90 /* !!!! - plane_data->angle */) / x_distunit;
  baseyscale = -M_Sin(viewangle - ANG90 /* !!!! - plane_data->angle */) / x_distunit;

  // draw the spans
  for (x = plane->area.x1; x <= plane->area.x2; x++)
  {
    int offset = x - plane->area.x1;

    R_yt = plane->area.ranges[offset].y1;
    R_yb = plane->area.ranges[offset].y2;

    if (R_yt < 0)
      R_yt = 0;

    if (R_yb >= viewheight)
      R_yb = viewheight-1;

    if (R_yt > R_yb)
    {
      R_yt = 9999;
      R_yb = 0;
    }
    
    R2_MakeSpans(plane, x, L_yt, L_yb, R_yt, R_yb);

    L_yt = R_yt;
    L_yb = R_yb;
  }

  R2_MakeSpans(plane, x, L_yt, L_yb, 9999, 0);

  if (plane_image_cache[0].cim)
    W_ImageDone(plane_image_cache[0].cim);

  if (plane_image_cache[1].cim)
    W_ImageDone(plane_image_cache[1].cim);

#elif (SHOW_PLANES == 1)
  BOGUS_DrawArea(&plane->area,
      GREEN + GREEN_LEN * (255 - plane->props->lightlevel) / 256);
#endif /* SHOW_PLANES */
}

//
// R2_DrawThing
//
// Render a single thing of a subsector.
//
extern lighttable_t *spritezlights;

#define SX_FUDGE  2

void R2_DrawThing(subsector_t *dsub, drawthing_t *dthing)
{
#if (SHOW_THINGS == 2)
  int x, px;
  int yt, yb;

  int index;
  int offset;
  int column;
  int patch_width;

  const cached_image_t *cim;
  const w_post_t *cim_col;

  patch_width = dthing->image->actual_w;
  
  // calculate lighting
  if (dthing->is_shadow)
  {
    dc_colourmap = V_GetColtable(shadow_map, 0, 0);
  }
  else if (dthing->is_halo)
  {
    dc_colourmap = NULL;
  }
  else if (dthing->bright)
  {
    dc_colourmap = V_GetColtable(dthing->props->colourmap, 255, 0);
  }
  else if (dthing->mo->flags & MF_FUZZY)
  {
    dc_colourmap = fuzz_coltable;
  }
  else
  {
    index = MIN(255, dthing->props->lightlevel + dthing->extra_light);

    DEV_ASSERT2(currentmap);
    if (currentmap->lighting <= LMODEL_Doomish)
    {
      index >>= LIGHTSEGSHIFT;

      if (index < 0)
        spritezlights = zlight[0];
      else if (index >= LIGHTLEVELS)
        spritezlights = zlight[LIGHTLEVELS - 1];
      else
        spritezlights = zlight[index];

      index = dthing->tz / (1 << (LIGHTZSHIFT - FRACBITS));

      if (index >= MAXLIGHTZ)
        index = MAXLIGHTZ - 1;
      
      index = spritezlights[index];
    }

    dc_colourmap = V_GetColtable(dthing->props->colourmap, index, 0);
  }

  // handle translucency, translation & shadow effect

  index = 0;
  mask_translucency = 1.0;

  if (dthing->is_shadow && level_flags.trans)
  {
    index |= 1;
    mask_translucency = dthing->mo->visibility *
        PERCENT_2_FLOAT(dthing->mo->info->shadow_trans);
  }
  else if (dthing->mo->visibility < VISIBLE && level_flags.trans)
  {
    index |= 1;
    mask_translucency = dthing->mo->visibility;
  }

  if (dthing->trans_table)
  {
    index |= 2;
    dc_translation = dthing->trans_table;
  }
  
  if (dthing->is_halo)
  {
    colfunc = R_DrawHaloColumn8_MIP;  //!!!
  }
  else if ((dthing->mo->flags & MF_FUZZY) && !dthing->is_shadow)
  {
    colfunc = R_DrawFuzzColumn;
  } 
  else
  {
    DEV_ASSERT2(index <= 3);
    
    switch (index)
    {
      case 0: colfunc = R_DrawColumn; break;
      case 1: colfunc = R_DrawTranslucentColumn; break;
      case 2: colfunc = R_DrawTranslatedColumn; break;
      case 3: colfunc = R_DrawTranslucentTranslatedColumn; break;
    }
  }

  yt = floor(focusyfrac - (dthing->top    - viewz) * dthing->dist_scale);
  yb = floor(focusyfrac - (dthing->bottom - viewz) * dthing->dist_scale);

  cim = W_ImageCache(dthing->image, IMG_Post, 0, true);

  // draw the columns
  for (x = dthing->area.x1; x <= dthing->area.x2; x++)
  {
    if (x < dsub->x_min-SX_FUDGE || x > dsub->x_max+SX_FUDGE ||
        (x < dsub->x_min && dsub->clip_left) ||
        (x > dsub->x_max && dsub->clip_right))
    {
      continue;
    }
    
    px = MAX(dsub->x_min, MIN(dsub->x_max, x));

    dc_x  = x;
    dc_yl = MAX(yt,   dsub->ranges[px - dsub->x_min].y1);
    dc_yh = MIN(yb-1, dsub->ranges[px - dsub->x_min].y2);

    if (dc_yl >= dc_yh)
      continue;
    
    DEV_ASSERT2(dc_yh >= 0 && dc_yl < viewheight);

    if (dc_yl < 0)
      dc_yl = 0;

    if (dc_yh >= viewheight) 
      dc_yh = viewheight-1;

    offset = x - dthing->area.x1;

    column = floor(dthing->xfrac + offset * dthing->ixscale);

    if (dthing->flip)
      column = patch_width - 1 - column;

    if (column < 0 || column >= patch_width)
      continue;

    // -AJA- screen coords are now absolute
    dc_x  += viewwindowx;
    dc_yl += viewwindowy;
    dc_yh += viewwindowy;

    mask_yl = (float_t) dc_yl;
    mask_yh = (float_t) dc_yh;
    mask_scale = dthing->yscale;
    mask_top = yt + viewwindowy - (dthing->area.y_offset * mask_scale
        / dthing->mo->info->yscale);  // OPTIMISE

    cim_col = W_ImageGetPost(cim, column);

    if (cim_col)
      R2_DrawMaskedWPost(cim_col);
  }

  W_ImageDone(cim);

#elif (SHOW_THINGS == 1)
  int x, px, y1, y2;
  int yt, yb;
  int col = RED + RED_LEN * (255 - dthing->props->lightlevel) / 256;

  yt = floor(focusyfrac - (dthing->top    - viewz) * dthing->dist_scale);
  yb = floor(focusyfrac - (dthing->bottom - viewz) * dthing->dist_scale);

  for (x=dthing->area.x1; x <= dthing->area.x2; x++)
  {
    if (x < dsub->x_min-SX_FUDGE || x > dsub->x_max+SX_FUDGE ||
        (x < dsub->x_min && dsub->clip_left) ||
        (x > dsub->x_max && dsub->clip_right))
    {
      continue;
    }
    
    px = MAX(dsub->x_min, MIN(dsub->x_max, x));

    y1 = MAX(yt, dsub->ranges[px - dsub->x_min].y1);
    y2 = MIN(yb, dsub->ranges[px - dsub->x_min].y2);

    if (y1 > y2)
      continue;
    
    BOGUS_Column(main_scr, x, y1, y2-y1+1, col);
  }
#endif /* SHOW_THINGS */
}

//
// R2_DrawSortThings
//
static drawthing_t **mapdrawthings = NULL;
static int mapdrawthing_num = 0;

static void R2_DrawSortThings(subsector_t *dsub, drawfloor_t *dfloor)
{
  drawthing_t *thing;

  int i, count=0;

  for (thing=dfloor->things; thing; thing=thing->next)
    count++;

  if (count == 0)
    return;

  // maybe reallocate array
  if (count > mapdrawthing_num)
  {
    Z_Resize(mapdrawthings, drawthing_t *, count);
    mapdrawthing_num = count;
  }

  // fill array
  count=0;

  for (thing=dfloor->things; thing; thing=thing->next)
    mapdrawthings[count++] = thing;

  // sort array.  Shadows are drawn first, Halos last
  //
  #define CMP(a,b)  ((a)->is_shadow > (b)->is_shadow ||  \
       ((a)->is_shadow == (b)->is_shadow &&  \
        ((a)->is_halo < (b)->is_halo ||                  \
         ((a)->is_halo == (b)->is_halo &&  \
          (a)->tz > (b)->tz))))
  QSORT(drawthing_t *, mapdrawthings, count, CUTOFF);
  #undef CMP

  // draw array
  for (i=0; i < count; i++)
    R2_DrawThing(dsub, mapdrawthings[i]);
}

//
// R2_DrawFloor
//
// Draw a single floor of a subsector.
//
void R2_DrawFloor(subsector_t *dsub, drawfloor_t *dfloor)
{
  drawwall_t  *wall;
  drawplane_t *plane;

  for (plane=dfloor->planes; plane; plane=plane->next)
  {
    R2_DrawPlane(dsub, plane);
  }

  for (wall=dfloor->walls; wall; wall=wall->next)
  {
    R2_DrawWall(dsub, wall);
  }

  R2_DrawSortThings(dsub, dfloor);
}

//
// R2_DrawSubsector
//
// Draw a subsector based on all the information collected when
// traversing the BSP tree.
//
void R2_DrawSubsector(subsector_t *dsub)
{
  drawfloor_t *dfloor;

  for (dfloor=dsub->floors; dfloor; dfloor=dfloor->next)
  {
    R2_DrawFloor(dsub, dfloor);
  }
}

//
// R2_DrawSkyPlane
//
static void R2_DrawSkyPlane(drawplane_t *plane)
{
#if (SHOW_SKY == 2)
  int x;
  int column;

  float_t scale_x = (float_t)0x0100 / sky_image->scale_x;
  float_t scale_y = (float_t)0x0100 / sky_image->scale_y;

  const cached_image_t *cim;

  cim = W_ImageCache(sky_image, IMG_Block, 0, true);

  dc_width  = sky_image->total_w;
  dc_height = sky_image->total_h;

  dc_ystep = M_FloatToFixed(skytexturescale / scale_y);
  dc_colourmap = V_GetColtable(normal_map, 255, VCOL_Sky);

  if ((sky_image->total_w * scale_x) < 300)
    scale_x *= 2.0;
  
  for (x=plane->area.x1; x <= plane->area.x2; x++)
  {
    dc_x = x;

    dc_yl = plane->area.ranges[x - plane->area.x1].y1;
    dc_yh = plane->area.ranges[x - plane->area.x1].y2;

    if (dc_yl > dc_yh)
      continue;
    
    column = ((viewangle + xtoviewangle[x]) >> (ANGLETOSKYSHIFT)) / scale_x;
    column &= (sky_image->total_w - 1);

    dc_yfrac = M_FloatToFixed(skytexturemid / scale_y) + dc_yl * dc_ystep;

    dc_source = W_ImageGetBlock(cim) + dc_height * column;

    // -AJA- screen coords are now absolute
    dc_x  += viewwindowx;
    dc_yl += viewwindowy;
    dc_yh += viewwindowy;

    R_DrawColumn_MIP();
  }
  W_ImageDone(cim);

#elif (SHOW_SKY == 1)
  BOGUS_DrawArea(&plane->area, CYAN+CYAN_LEN/2);
#endif /* SHOW_SKY */
}

//
// R2_DrawPSprite
//
// -ACB- 1998/09/02 Fixed Error Messages to display correct procedure.
//
static void R2_DrawPSprite(pspdef_t * psp, int which,
    player_t * player, region_properties_t *props, const state_t *state)
{
#if (SHOW_WEAPON >= 1)
  int texcol;
  int trans = 0;
  fixed_t frac;
  fixed_t dfrac;

  const image_t *image;
  const cached_image_t *cim;
  const w_post_t *cim_col;

  int x, x1, x2;
  int sx1, sx2;
  int sy1, sy2;
  boolean_t flip;
  const colourmap_t *nominal;

  float_t tx, ty1, ty2;
  float_t visibility;
  float_t xiscale, yscale;
  float_t startfrac, dist_scale;

  // decide which image to use

#ifdef DEVELOPERS
  if ((unsigned int)state->sprite >= (unsigned int)numsprites)
    I_Error("DrawPSprite: invalid sprite number %i ", state->sprite);
#endif

  image = R2_GetOtherSprite(state->sprite, state->frame, &flip);

  if (!image)
    return;

  // calculate edges of the shape
  tx = psp->sx - BASEXCENTER;

  tx -= image->offset_x;
  x1 = focusxfrac + tx * pspritescale;

  // off the right side
  if (x1 >= viewwidth)
    return;

  tx += IM_WIDTH(image);
  x2 = focusxfrac + tx * pspritescale - 1;

  // off the left side
  if (x2 <= 0)
    return;

  ty1 = psp->sy - IM_OFFSETY(image);
  ty2 = ty1 + IM_HEIGHT(image);

  sx1 = MAX(x1, 0);
  sx2 = MIN(x2, viewwindowwidth - 1);

  if (sx1 >= sx2)
    return;

  yscale = pspritescale2;
  dist_scale = pspritescale2;

  if (flip)
  {
    xiscale = -pspriteiscale;
    // -ES- Fixme
    startfrac = IM_WIDTH(image) - 0.0001;
  }
  else
  {
    xiscale = pspriteiscale;
    startfrac = 0;
  }

  if (sx1 > x1)
    startfrac += xiscale * (sx1 - x1);

  visibility = player->mo->visibility * psp->visibility;

  // calculate lighting
  // -AJA- 1999/07/10: Updated for colmap.ddf.
  nominal = props->colourmap;

  DEV_ASSERT2(currentmap);

  if (player->powers[PW_PartInvis] > 4 * 32 ||
      fmod(player->powers[PW_PartInvis], 16) >= 8)
  {
////    mobjflags |= MF_FUZZY;
    dc_colourmap = fuzz_coltable;
  }
  else if (state->bright)
  {
    // full bright
    dc_colourmap = V_GetColtable(nominal, 255, 0);
  }
  else if (currentmap->lighting <= LMODEL_Doomish)
  {
    // local light
    dc_colourmap = V_GetColtable(nominal, extra_psp_light +
        spritezlights[FLAT_LIGHTZ], 0);
  }
  else
  {
    // FLAT lighting
    dc_colourmap = V_GetColtable(nominal, extra_psp_light +
        props->lightlevel, 0);
  }

  if (player->mo->flags & MF_FUZZY)
  {
    trans |= 1;
  }

  if (visibility < VISIBLE && level_flags.trans)
  {
    trans |= 2;
    dc_translucency = M_FloatToFixed(visibility);
  }

//!!!  if (trans_table)
//!!!  {
//!!!    dc_translation = trans_table;
//!!!    trans |= 4;
//!!!  }

  colfunc = R_DrawColumn_MIP;

  if (trans & 1)
    colfunc = R_DrawFuzzColumn;  //!!! FIXME
  else
    switch (trans)
    {
      default:
        break;
      case 2:
        colfunc = R_DrawTranslucentColumn_MIP;
        break;

        //!!!! FIXME: use V_GetTranslatedColtable for these
      case 4:
        colfunc = R_DrawTranslatedColumn;
        break;
      case 6:
        colfunc = R_DrawTranslucentTranslatedColumn;
        break;
    }

  // -ES- 1999/03/04 Fixed aspect ratio
  masked_translucency = 1.0;  // -AJA- 1999/06/30

  frac = M_FloatToFixed(startfrac);
  dfrac = M_FloatToFixed(xiscale);
  
  sy1 = (int) (viewwindowheight * ty1 / 200.0);
  sy2 = (int) (viewwindowheight * ty2 / 200.0);

  cim = W_ImageCache(image, IMG_Post, 0, true);
  dc_height = image->total_h;

  if (sy1 < 0)
    sy1 = 0;

  if (sy2 >= viewwindowheight)
    sy2 = viewwindowheight-1;

  if (sy1 >= sy2)
    return;

#if (SHOW_WEAPON == 1)
  V_DrawBox(main_scr, viewwindowx + sx1, viewwindowy + dc_yl, sx2-sx1, dc_yh-dc_yl, which ? YELLOW : DBLUE);
  return;
#endif  /* SHOW_WEAPON */

  for (x = sx1; x <= sx2; x++, frac += dfrac)
  {
    texcol = frac >> FRACBITS;

    // allow a single pixel error at either side of sprite
    if (texcol == -1 || texcol == image->actual_w)
      continue;

#ifdef DEVELOPERS
    if (texcol < -1 || texcol > image->actual_w)
      I_Error("R_DrawVisSprite: bad texcol %d (image %dx%d)", texcol, image->actual_w, image->actual_h);
#endif

    // -AJA- screen coords are now absolute
    dc_x = x + viewwindowx;
    dc_yl = sy1 + viewwindowy;
    dc_yh = sy2 + viewwindowy;

    mask_yl = (float_t) dc_yl;
    mask_yh = (float_t) dc_yh;
    mask_scale = yscale;
    mask_top = mask_yl;

    cim_col = W_ImageGetPost(cim, texcol);

    if (cim_col)
      R2_DrawMaskedWPost(cim_col);
  }

  W_ImageDone(cim);
#endif  /* SHOW_WEAPON */
}

//
// R2_DrawPlayerSprites
//
void R2_DrawPlayerSprites(player_t * p)
{
  int i;
  int lightnum;

  DEV_ASSERT2(p->in_game);

  // get light level
  lightnum = view_props->lightlevel >> LIGHTSEGSHIFT;

  // -AJA- 1999/07/10: Now uses zlight[] instead of scalelight[].
  if (lightnum < 0)
    spritezlights = zlight[0];
  else if (lightnum >= LIGHTLEVELS)
    spritezlights = zlight[LIGHTLEVELS - 1];
  else
    spritezlights = zlight[lightnum];

  // special handling for zoom: show viewfinder
  if (viewiszoomed)
  {
    weaponinfo_t *w;
    pspdef_t *psp = &p->psprites[ps_weapon];

    if ((p->ready_wp < 0) || (psp->state == S_NULL))
      return;

    w = p->weapons[p->ready_wp].info;

    if (w->zoom_state <= 0)
      return;

    R2_DrawPSprite(psp, ps_weapon, p, view_props,
        states + w->zoom_state);

    return;
  }
  
  // add all active psprites
  // Note: order is significant
  for (i = 0; i < NUMPSPRITES; i++)
  {
    pspdef_t *psp = &p->psprites[i];

    if (psp->state == S_NULL)
      continue;

    R2_DrawPSprite(psp, i, p, view_props, psp->state);
  }
}


//----------------------------------------------------------------------------
//
//  VIDEO CONTEXT STUFF
//

//
// Automap clipping of lines.
//
// Based on Cohen-Sutherland clipping algorithm but with a slightly
// faster reject and precalculated slopes.  If the speed is needed,
// use a hash algorithm to handle  the common cases.
//
boolean_t R2_ClipLine(int *x1, int *y1, int *x2, int *y2,
    int cl_x1, int cl_y1, int cl_x2, int cl_y2)
{
  enum
  {
    LEFT = 1,
    RIGHT = 2,
    BOTTOM = 4,
    TOP = 8
  };

  int outcode1 = 0;
  int outcode2 = 0;
  int outside;

  int dx, dy;
  int tmp_x, tmp_y;

#define DO_OUTCODE(oc, xx, yy)  \
    do {                                       \
      (oc) = 0;                                \
      if ((yy) < cl_y1) (oc) |= TOP;           \
      else if ((yy) >= cl_y2) (oc) |= BOTTOM;  \
      if ((xx) < cl_x1) (oc) |= LEFT;          \
      else if ((xx) >= cl_x2) (oc) |= RIGHT;   \
    } while (0)

  // do trivial rejects and outcodes
  DO_OUTCODE(outcode1, (*x1), (*y1));
  DO_OUTCODE(outcode2, (*x2), (*y2));

  if (outcode1 & outcode2)
    return false;  // trivially outside

  while (outcode1 | outcode2)
  {
    // may be partially inside box
    // find an outside point
    if (outcode1)
      outside = outcode1;
    else
      outside = outcode2;

    DEV_ASSERT2(outside != 0);

    // clip to each side
    if (outside & TOP)
    {
      dy = (*y1) - (*y2);
      dx = (*x2) - (*x1);
      tmp_x = (*x1) + ((*y1) - cl_y1) * dx / dy;
      tmp_y = cl_y1;
    }
    else if (outside & BOTTOM)
    {
      dy = (*y1) - (*y2);
      dx = (*x2) - (*x1);
      tmp_x = (*x1) + ((*y1) - cl_y2) * dx / dy;
      tmp_y = cl_y2 - 1;
    }
    else if (outside & LEFT)
    {
      dy = (*y2) - (*y1);
      dx = (*x2) - (*x1);
      tmp_y = (*y1) + (cl_x1 - (*x1)) * dy / dx;
      tmp_x = cl_x1;
    }
    else
    {
      DEV_ASSERT2(outside & RIGHT);

      dy = (*y2) - (*y1);
      dx = (*x2) - (*x1);
      tmp_y = (*y1) + (cl_x2 - 1 - (*x1)) * dy / dx;
      tmp_x = cl_x2 - 1;
    }

    if (outside == outcode1)
    {
      (*x1) = tmp_x;
      (*y1) = tmp_y;

      DO_OUTCODE(outcode1, (*x1), (*y1));
    }
    else
    {
      (*x2) = tmp_x;
      (*y2) = tmp_y;

      DO_OUTCODE(outcode2, (*x2), (*y2));
    }

    if (outcode1 & outcode2)
      return false;  // trivially outside
  }

#undef DO_OUTCODE

  return true;
}

// 
// The video context structure.  There can only be one, either for
// software rendering (here) or for GL rendering.
//
#ifndef USE_GL
video_context_t vctx;
#endif

// static int clipper_x1;  ...

//
// R2_NewScreenSize
//
void R2_NewScreenSize(int width, int height, int bpp)
{
  //...
}

//
// R2_RenderScene
//
void R2_RenderScene(int x, int y, int w, int h, vid_view_t *view)
{
  //!!!
  R2_RenderTrueBSP();
}

//
// R2_BeginDraw
//
void R2_BeginDraw(int x1, int y1, int x2, int y2)
{
  //...
}

//
// R2_EndDraw
//
void R2_EndDraw(void)
{
  //...
}


static void DrawImagePost(int x, int y, int w, int h, const image_t *image, 
    float_t tx1, float_t ty1, float_t tx2, float_t ty2,
    const colourmap_t *colmap, float_t alpha)
{
  const cached_image_t *cim;
  const w_post_t *cim_col;

  int column;

  cim = W_ImageCache(image, IMG_Post, 0, false);

  dc_colourmap = V_GetColtable(colmap, 255, 0);
  dc_height = image->total_h;
  dc_translucency = M_FloatToFixed(alpha);

  if (alpha < 0.99)
    colfunc = R_DrawTranslucentColumn_MIP;
  else
    colfunc = R_DrawColumn_MIP;

  for (dc_x=x; dc_x < x+w; dc_x++)
  {
    float_t tx = tx1 + (tx2 - tx1) * (dc_x - x) / w;
    
    if (dc_x < 0 || dc_x >= SCREENWIDTH)
      continue;

    column = (int)(tx * image->total_w);
    
    dc_yl = y;
    dc_yh = y + h - 1;

    if (dc_yl < 0)
      dc_yl = 0;

    if (dc_yh >= SCREENHEIGHT) 
      dc_yh = SCREENHEIGHT-1;

    if (dc_yh < dc_yl || dc_yh < 0 || dc_yl >= SCREENHEIGHT)
      return;

    mask_yl = (float_t) dc_yl;
    mask_yh = (float_t) dc_yh;

    CHECKVAL(ty2 - ty1);
    mask_scale = (float_t)h / (ty2 - ty1) / image->total_h;
    mask_top = mask_yl - ty1 * mask_scale;

    cim_col = W_ImageGetPost(cim, column);

    if (cim_col)
      R2_DrawMaskedWPost(cim_col);
  }

  W_ImageDone(cim);
}

static void DrawImageBlock(int x, int y, int w, int h, const image_t *image, 
    float_t tx1, float_t ty1, float_t tx2, float_t ty2,
    const colourmap_t *colmap, float_t alpha)
{
  const cached_image_t *cim;

  float_t scale_x, scale_y;

  cim = W_ImageCache(image, IMG_Block, 0, false);
  ds_source = W_ImageGetBlock(cim);

  ds_colourmap = V_GetColtable(colmap, 255, 0);
  ds_width  = image->total_w;
  ds_height = image->total_h;
  dc_translucency = M_FloatToFixed(alpha);

  if (alpha < 0.99)
    spanfunc = R_DrawTranslucentSpan_MIP;
  else
    spanfunc = R_DrawSpan_MIP;

  scale_x = image->scale_x / (float_t)0x0100;
  scale_y = image->scale_y / (float_t)0x0100;
  
  for (ds_y=y; ds_y < y+h; ds_y++)
  {
    float_t ty = ty1 + (ty2 - ty1) * (ds_y - y) / h;

    if (ds_y < 0 || ds_y >= SCREENHEIGHT)
      continue;

    ds_x1 = x;
    ds_x2 = x + w - 1;

    if (ds_x1 < 0)
      ds_x1 = 0;

    if (ds_x2 >= SCREENWIDTH) 
      ds_x2 = SCREENWIDTH-1;

    if (ds_x2 < ds_x1 || ds_x2 < 0 || ds_x1 >= SCREENWIDTH)
      return;

    ds_yfrac = M_FloatToFixed(ty  * ds_height);
    ds_xfrac = M_FloatToFixed(tx1 * ds_width);

    CHECKVAL(w);
    ds_ystep = 0;
    ds_xstep = M_FloatToFixed(ds_width * (tx2 - tx1) / w);

    (* spanfunc)();
  }

  W_ImageDone(cim);
}

//
// R2_DrawImage
//
void R2_DrawImage(int x, int y, int w, int h, const image_t *image, 
    float_t tx1, float_t ty1, float_t tx2, float_t ty2,
    const colourmap_t *colmap, float_t alpha)
{
  if (x+w <= 0 || x >= SCREENWIDTH)
    return;

  if (y+h <= 0 || y > SCREENHEIGHT)
    return;

  if (! colmap)
    colmap = normal_map;

  if (image->solid)
    DrawImageBlock(x, y, w, h, image, tx1, ty1, tx2, ty2, colmap, alpha);
  else
    DrawImagePost(x, y, w, h, image, tx1, ty1, tx2, ty2, colmap, alpha);
}

//
// R2_SolidBox
//
void R2_SolidBox(int x, int y, int w, int h, int colour, float_t alpha)
{
  // FIXME: implement translucent box drawing
  
  if (alpha < 0.02)
    return;

  if (alpha >= 0.98)
  {
    V_DrawBox(main_scr, x, y, w, h, colour);
  }
  else
  {
    V_DrawBoxAlpha(main_scr, x, y, w, h, colour, M_FloatToFixed(alpha));
  }
}

//
// R2_SolidLine
//
void R2_SolidLine(int x1, int y1, int x2, int y2, int colour)
{
  if (R2_ClipLine(&x1, &y1, &x2, &y2, 0, 0, SCREENWIDTH, SCREENHEIGHT))
  {
    V_DrawLine(main_scr, x1, y1, x2, y2, colour);
  }
}

//
// R2_ReadScreen
//
void R2_ReadScreen(int x, int y, int w, int h, byte *rgb_buffer)
{
  //...
}

// -ACB- 2000/07/22 Do we not like this with OpenGL
#ifndef USE_GL
//
// R2_Init
//
void R2_Init(void)
{
  I_Printf("TrueBSP: Initialising...\n");

  // setup video context

#ifndef USE_GL
  vctx.double_buffered = false;

  vctx.NewScreenSize = R2_NewScreenSize;
  vctx.RenderScene = R2_RenderScene;
  vctx.BeginDraw = R2_BeginDraw;
  vctx.EndDraw = R2_EndDraw;
  vctx.DrawImage = R2_DrawImage;
  vctx.SolidBox = R2_SolidBox;
  vctx.SolidLine = R2_SolidLine;
  vctx.ReadScreen = R2_ReadScreen;

  R2_InitUtil();
#endif
}
#endif /* !USE_GL */


