//----------------------------------------------------------------------------
//  EDGE Column/Span Drawing for 8-bit Colour Code
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------
//
// -ACB- 1998/09/10 Cleaned up.
//

#include "i_defs.h"
#include "r_draw1.h"

#include "con_cvar.h"
#include "con_defs.h"
#include "dm_defs.h"
#include "dm_state.h"
#include "m_random.h"
#include "r_local.h"
#include "v_res.h"
#include "v_screen.h"
#include "v_colour.h"
#include "w_image.h"
#include "w_wad.h"
#include "z_zone.h"

funclist_t drawcol8_funcs;
funclist_t drawspan8_funcs;

//
// All drawing to the view buffer is accomplished in this file.
//
// The other refresh files only know about ccordinates, not the
// architecture of the frame buffer.
//
// Conveniently, the frame buffer is a linear one, and we need
// only the base address, and the total size == width*height*depth/8.
//

// -ES- 1999/02/12 Converted BLF stuff to 8-bit
// Detail level. High detail levels look better, but
// consume more memory. For each detail increase by one,
// the BLF table needs approx. four times as much memory.
#define BLFshift 2
#define BLFsz (1<<BLFshift)
#define BLFmax (BLFsz-1)

static void ColSmoothingOn(funclist_t * fl GCCATTR(unused))
{
  dc_usesmoothing = true;
}
static void ColSmoothingOff(funclist_t * fl GCCATTR(unused))
{
  dc_usesmoothing = false;
}
static void SpanSmoothingOn(funclist_t * fl GCCATTR(unused))
{
  ds_usesmoothing = true;
}
static void SpanSmoothingOff(funclist_t * fl GCCATTR(unused))
{
  ds_usesmoothing = false;
}

// Table with 256 longs containing translucency table-style fix-point RGBs.
typedef unsigned long BLF8LUT;

// Totally 8*8*256 BLF16LUTs
static BLF8LUT *BLFTab[BLFsz][BLFsz];

static void BLF_Init8(funclist_t * fl)
{
  static boolean_t firsttime = true;
  unsigned int i;

  unsigned long r, g, b, x, y, xy;

  const unsigned char *thepalette;

  // Array of all the used BLF16LUTs
  BLF8LUT *BLFBuf;

  // Stores which index to BLFBuf a certain x*y should use (x and y are 31.1 fix point)
  int BLFCheck[4 * BLFsz * BLFsz];

  int count = 0;

  if (fl->dest == &R_DrawColumn)
    ColSmoothingOn(fl);
  else if (fl->dest == &R_DrawSpan)
    SpanSmoothingOn(fl);
  if (!firsttime)
    return;
  firsttime = false;

  I_Printf("BLF_Init: Init Bilinear Filtering");

  // Init BLFCheck
  for (x = 0; x < 4 * BLFsz * BLFsz; x++)
    BLFCheck[x] = -1;
  for (x = 1; x < 2 * BLFsz; x += 2)
    for (y = 1; y < 2 * BLFsz; y += 2)
      if (BLFCheck[x * y] == -1)
      {
        BLFCheck[x * y] = count;
        count++;
      }

  // Allocate the memory if it isn't already allocated. Use 32-byte alignment.
  BLFBuf = BLFTab[0][0];
  if (!BLFBuf)
  {
    // allocate memory
    BLFBuf = (BLF8LUT *) Z_Malloc(count * 256 * sizeof(BLF8LUT) + 31);

    // align by 32
    BLFBuf = (BLF8LUT *) (((long)BLFBuf + 31) & ~31);
  }

  for (x = 0; x < BLFsz; x++)
    for (y = 0; y < BLFsz; y++)
      BLFTab[x][y] = &BLFBuf[256 * BLFCheck[(2 * x + 1) * (2 * y + 1)]];

  thepalette = W_CacheLumpName("PLAYPAL");
  for (xy = 0; xy < 4 * BLFsz * BLFsz; xy++)
    if (BLFCheck[xy] != -1)
    {
      for (i = 0; i < 256; i++)
      {
        r = thepalette[i * 3 + 0];
        g = thepalette[i * 3 + 1];
        b = thepalette[i * 3 + 2];
        r = ((r * xy) << 2) >> (2 + 2 * BLFshift);
        g = ((g * xy) << 4) >> (2 + 2 * BLFshift);
        b = ((b * xy) << 2) >> (2 + 2 * BLFshift);
        (&BLFBuf[256 * BLFCheck[xy]])[i] = (r << 22) + (g << 10) + b;
      }
    }
  W_DoneWithLump(thepalette);
  I_Printf("\n");
}

void resinit_r_draw_c8(void)
{
}

//
// R_DrawColumn8_CVersion
//
// A column is a vertical slice/span from a wall texture that,
// given the DOOM style restrictions on the view orientation,
// will always have constant z depth.
//
// Thus a special case loop for very fast rendering can
// be used. It has also been used with Wolfenstein 3D.
//
// -ES- 1999/04/19 Optimised slightly
void R_DrawColumn8_CVersion(void)
{
  int count;
  byte *dest;
  unsigned long frac;
  unsigned long fracstep;

  count = dc_yh - dc_yl + 1;

  // Zero length, column does not exceed a pixel.
  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  //
  // Framebuffer destination address.
  // Use ylookup LUT to avoid multiply with ScreenWidth.
  //
  dest = ylookup[dc_yl] + columnofs[dc_x];

  // Determine scaling, which is the only mapping to be done.
  fracstep = dc_ystep << (FRACBITS - 7);
  frac = dc_yfrac << (FRACBITS - 7);

  // Inner loop that does the actual texture mapping,
  // e.g. a DDA-lile scaling. This is as fast as it gets.
  do
  {
    // Re-map colour indices from wall texture column
    // using a lighting/special effects LUT.
    *dest = dc_colourmap[dc_source[frac >> (32 - 7)]];

    dest += vb_pitch;
    frac += fracstep;

  }
  while (--count);
}

//
// R_DrawColumn8_MIP
//
void R_DrawColumn8_MIP(void)
{
  int count;
  byte *dest;

  fixed_t yfrac;
  fixed_t ystep;
  fixed_t ymask;

  count = dc_yh - dc_yl + 1;

  // Zero length, column does not exceed a pixel.
  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  //
  // Framebuffer destination address.
  // Use ylookup LUT to avoid multiply with ScreenWidth.
  //
  dest = ylookup[dc_yl] + columnofs[dc_x];

  // Determine scaling, which is the only mapping to be done.
  yfrac = dc_yfrac;
  ystep = dc_ystep;
  ymask = (dc_height - 1) << FRACBITS;

  // Inner loop that does the actual texture mapping,
  // e.g. a DDA-lile scaling. This is as fast as it gets.
  do
  {
    // Re-map colour indices from wall texture column
    // using a lighting/special effects LUT.
    *dest = dc_colourmap[dc_source[(yfrac & ymask) >> FRACBITS]];

    dest += vb_pitch;
    yfrac += ystep;
  }
  while (--count);
}

//
// R_DrawTranslucentColumn8_MIP
//
void R_DrawTranslucentColumn8_MIP(void)
{
  int count;
  byte *dest;

  fixed_t yfrac;
  fixed_t ystep;
  fixed_t ymask;

  fixed_t fglevel, bglevel;
  unsigned long *fg2rgb, *bg2rgb;
  unsigned long fg;  // current colours

  count = dc_yh - dc_yl + 1;

  // Zero length, column does not exceed a pixel.
  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  fglevel = dc_translucency;
  fglevel = fglevel & ~0x3ff;
  bglevel = FRACUNIT - fglevel;

  fg2rgb = col2rgb8[fglevel >> 10];
  bg2rgb = col2rgb8[bglevel >> 10];

  //
  // Framebuffer destination address.
  // Use ylookup LUT to avoid multiply with ScreenWidth.
  //
  dest = ylookup[dc_yl] + columnofs[dc_x];

  // Determine scaling, which is the only mapping to be done.
  yfrac = dc_yfrac;
  ystep = dc_ystep;
  ymask = (dc_height - 1) << FRACBITS;

  // Inner loop that does the actual texture mapping,
  // e.g. a DDA-lile scaling. This is as fast as it gets.
  do
  {
    // Re-map colour indices from wall texture column
    // using a lighting/special effects LUT.
    fg = (fg2rgb[dc_colourmap[dc_source[(yfrac & ymask) >> FRACBITS]]] +
        bg2rgb[*dest]) | 0x07c1fc1f;

    *dest = rgb_32k[0][0][fg & (fg >> 17)];

    dest += vb_pitch;
    yfrac += ystep;
  }
  while (--count);
}

//
// R_DrawHaloColumn8_MIP
//
void R_DrawHaloColumn8_MIP(void)
{
  int count;
  byte *dest;

  fixed_t yfrac;
  fixed_t ystep;
  fixed_t ymask;

  fixed_t fglevel, bglevel;
  unsigned long fg;  // current colours

  int A = 1, B = 2;
  byte halo_col = 0x50;
    
  count = dc_yh - dc_yl + 1;

  // Zero length, column does not exceed a pixel.
  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  fglevel = dc_translucency;
  fglevel = fglevel & ~0x3ff;
  bglevel = FRACUNIT - fglevel;

  //
  // Framebuffer destination address.
  // Use ylookup LUT to avoid multiply with ScreenWidth.
  //
  dest = ylookup[dc_yl] + columnofs[dc_x];

  // Determine scaling, which is the only mapping to be done.
  yfrac = dc_yfrac;
  ystep = dc_ystep;
  ymask = (dc_height - 1) << FRACBITS;

  // Inner loop that does the actual texture mapping,
  // e.g. a DDA-lile scaling. This is as fast as it gets.
  do
  {
    byte val = dc_source[(yfrac & ymask) >> FRACBITS];

    fg = (col2rgb8[val * A / B][halo_col] +
          col2rgb8[64 - val * A / B][*dest]) | 0x07c1fc1f;

    *dest = rgb_32k[0][0][fg & (fg >> 17)];

    dest += vb_pitch;
    yfrac += ystep;
  }
  while (--count);
}

// These two just return without drawing anything. Used for profiling.
void R_DrawColumn8_dontdraw(void)
{
  return;
}
void R_DrawSpan8_dontdraw(void)
{
  return;
}

#if 0
// -ES- 1999-04-07 Mipmapping test
void R_DrawColumn8_MIP(void)
{
  int count;
  byte *dest;
  fixed_t frac;
  fixed_t fracstep;
  unsigned long mask;
  fixed_t i;

  count = dc_yh - dc_yl + 1;

  // Zero length, column does not exceed a pixel.
  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  //
  // Framebuffer destination address.
  // Use ylookup LUT to avoid multiply with ScreenWidth.
  //
  dest = ylookup[dc_yl] + columnofs[dc_x];

  // Determine scaling, which is the only mapping to be done.
  fracstep = dc_ystep;
  frac = dc_yfrac;

  // create mask, so that scale is at least 1.
  mask = 0xffff0000;
  for (i = abs(fracstep); i > FRACUNIT; i >>= 1)
  {
    mask <<= 1;
  }
  // the masking truncates the frac, so it's a good idea to add this value
  // so that it's rounded instead.
  frac += (0xFFFFFFFF - mask + 1 - FRACUNIT) / 2;
  mask &= 127 * FRACUNIT;
  // Inner loop that does the actual texture mapping,
  // e.g. a DDA-lile scaling. This is as fast as it gets.
  do
  {
    // Re-map colour indices from wall texture column
    // using a lighting/special effects LUT.
    *dest = dc_colourmap[dc_source[((unsigned long)(frac & mask)) >> FRACBITS]];

    dest += vb_pitch;
    frac += fracstep;

  }
  while (--count);
}
#endif

// Original algorithm invented by David Finch (mef@wave.net)
// DrawColumn fixed by KM
// Added by -ES- 1999/04/19
// 4-19-00: rewritten by David Finch to look like Unreal
void R_DrawColumn8_vfi(void)
{
  int count;
  byte *dest;
  fixed_t frac;
  fixed_t fracstep;

#ifdef NOSMOOTHING
  R_DrawColumn8_CVersion();
#else


  count = dc_yh - dc_yl + 1;

  // Zero length, column does not exceed a pixel.
  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 ||
      dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  //
  // Framebuffer destination address.
  // Use ylookup LUT to avoid multiply with ScreenWidth.
  //
  dest = ylookup[dc_yl] + columnofs[dc_x];


  // Determine scaling, which is the only mapping to be done.
  fracstep = dc_ystep;
  frac = dc_yfrac;

  // Inner loop that does the actual texture mapping,
  // e.g. a DDA-lile scaling. This is as fast as it gets.
  if (fracstep > 50000)
  {
    do
    {
      // Re-map colour indices from wall texture column
      // using a lighting/special effects LUT.
      *dest = dc_colourmap[dc_source[(frac >> FRACBITS) & 127]];

      dest += vb_pitch;
      frac += fracstep;

    }
    while (--count);
  }
  else
  {
    // This routine by David Finch <mef@wave.net>.
 // 4-19-00: rewritten to look like Unreal
    const byte *dcsrc1,*dcsrc2;
    int xa1,ya1,xa2,ya2;


    dc_xfrac &= 0xffff;

// () left out for a reason
#define _a1 10922+32768
#define _a2 32767+32768


 if(dc_x&1) {
  if(dc_yl&1) {
   xa1=-_a2;
   ya1=-_a1;
   xa2=+_a2;
   ya2=+_a1;
  } else {
   xa2=-_a2;
   ya2=-_a1;
   xa1=+_a2;
   ya1=+_a1;
  }
 } else {
  if(dc_yl&1) {
   xa1=_a1;
   ya1=-_a2;
   xa2=-_a1;
   ya2=_a2;
  } else {
   xa2=_a1;
   ya2=-_a2;
   xa1=-_a1;
   ya1=_a2;
  }
 }
 if(xa1+dc_xfrac>65535) dcsrc1=dc_source2; else dcsrc1=dc_source;
 if(xa2+dc_xfrac>65535) dcsrc2=dc_source2; else dcsrc2=dc_source;

 if(count&1) {
  *dest = dc_colourmap[dcsrc1[((frac + ya1) >> 16) & 127]];
  dest += vb_pitch;
  frac += fracstep;
  if(--count) {
   do
   {
    *dest = dc_colourmap[dcsrc2[((frac + ya2) >> 16) & 127]];

    dest += vb_pitch;
    frac += fracstep;

    *dest = dc_colourmap[dcsrc1[((frac + ya1) >> 16) & 127]];
    dest += vb_pitch;
    frac += fracstep;
   }
   while (count-=2);
  }
 } else {
  do
  {
   *dest = dc_colourmap[dcsrc1[((frac + ya1) >> 16) & 127]];
   dest += vb_pitch;
   frac += fracstep;

   *dest = dc_colourmap[dcsrc2[((frac + ya2) >> 16) & 127]];

   dest += vb_pitch;
   frac += fracstep;
  }
  while (count-=2);
 }
  }
#endif
}




// Original algorithm invented by David Finch (mef@wave.net)
// Converted to span routine by KM
// Added and improved by -ES- 1999/04/19
// 4-19-00: rewritten by David Finch to look like Unreal
#define TXTP(x,y) ((y)&(63<<6)) | ((x)&63)
void R_DrawSpan8_vfi(void)
{
  fixed_t xfrac;
  fixed_t yfrac;
  byte *dest;
  int count;
  int spot;

#ifdef DEVELOPERS
  if (ds_x2 < ds_x1 || ds_x1 < 0 || ds_x2 >= SCREENWIDTH ||
      (unsigned int)ds_y > (unsigned int)SCREENHEIGHT)
    I_Error("R_DrawSpan: %i to %i at %i", ds_x1, ds_x2, ds_y);
#endif

  xfrac = ds_xfrac;
  yfrac = ds_yfrac;

  dest = ylookup[ds_y] + columnofs[ds_x1];

  count = ds_x2 - ds_x1;

  if (abs(ds_xstep) + abs(ds_ystep) > 65535)
    do
    {
      spot = ((yfrac >> (16 - 6)) & (63 * 64)) + ((xfrac >> 16) & 63);

      *dest++ = ds_colourmap[ds_source[spot]];

      xfrac += ds_xstep;
      yfrac += ds_ystep;
    }
    while (count--);
  else
  {
    int xa1,ya1,xa2,ya2;

    // This routine by David Finch <mef@wave.net>.
 // 4-19-00: rewritten to look like Unreal

    xfrac -= FRACUNIT / 2;
    yfrac -= FRACUNIT / 2;

 if(ds_y&1) {
  if(ds_x1&1) {
   xa1=-_a2;
   ya1=-_a1;
   xa2=+_a2;
   ya2=+_a1;
  } else {
   xa2=-_a2;
   ya2=-_a1;
   xa1=+_a2;
   ya1=+_a1;
  }
 } else {
  if(ds_x1&1) {
   xa1=_a1;
   ya1=-_a2;
   xa2=-_a1;
   ya2=_a2;
  } else {
   xa2=_a1;
   ya2=-_a2;
   xa1=-_a1;
   ya1=_a2;
  }
 }
    do
    {
      *dest++ = ds_colourmap[ds_source[(((xfrac + xa1) >> 16) & 63) |
(((yfrac + ya1) >> 10) & (63 * 64))]];
   xfrac += ds_xstep;
      yfrac += ds_ystep;

   if(!count--) break;

      *dest++ = ds_colourmap[ds_source[(((xfrac + xa2) >> 16) & 63) |
(((yfrac + ya2) >> 10) & (63 * 64))]];
   xfrac += ds_xstep;
      yfrac += ds_ystep;

    }
    while (count--);
  }
}


// -ES- 1998/12/18 Converted this to 8-bit
void R_DrawColumn8_KM()
{
#ifdef NOSMOOTHING
  R_DrawColumn8_CVersion();
#else
  int count;
  byte *dest;
  fixed_t frac;
  fixed_t fracstep;

  count = dc_yh - dc_yl + 1;

  // Zero length, column does not exceed a pixel.
  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  // Framebuffer destination address.
  // Use ylookup LUT to avoid multiply with ScreenWidth.
  dest = ylookup[dc_yl] + columnofs[dc_x];

  // Determine scaling, which is the only mapping to be done.
  fracstep = dc_ystep;
  frac = dc_yfrac;

  // Inner loop that does the actual texture mapping, e.g. a DDA-lile scaling.
  if (fracstep > 0x12000)
  {
    do
    {
      // Re-map colour indices from wall texture column
      // using a lighting/special effects LUT.
      *dest = dc_colourmap[dc_source[(frac >> FRACBITS) & 127]];

      dest += vb_pitch;
      frac += fracstep;

    }
    while (--count);
  }
  else
  {
    unsigned long spot[4];
    unsigned long level[4];
    unsigned long c;
    int i;

    dc_xfrac &= 0xffff;
    do
    {
      level[3] = (frac & 0xffff) * dc_xfrac;
      level[2] = (FRACUNIT - (frac & 0xffff)) * dc_xfrac;
      level[1] = (frac & 0xffff) * (FRACUNIT - dc_xfrac - 1);
      level[0] = (FRACUNIT - (frac & 0xffff)) * (FRACUNIT - dc_xfrac - 1);
      spot[0] = dc_source[(frac >> FRACBITS) & 127];
      spot[1] = dc_source[((frac >> FRACBITS) + 1) & 127];
      spot[2] = dc_source2[(frac >> FRACBITS) & 127];
      spot[3] = dc_source2[((frac >> FRACBITS) + 1) & 127];

      for (i = 0, c = 0; i < 4; i++)
      {
        c += col2rgb8[level[i] >> 26][spot[i]];
      }

      // -ES- 1999/03/13 Fixed the RGB conversion
      c |= 0x07c1fc1f;
      *dest = dc_colourmap[rgb_32k[0][0][c & (c >> 17)]];

      dest += vb_pitch;
      frac += fracstep;
    }
    while (--count);
  }
#endif
}

// -ES- 1999/03/29 Added this
void R_DrawColumn8_BLF()
{
#ifdef NOSMOOTHING
  R_DrawColumn8_CVersion();
#else
  int count;
  byte *dest;
  fixed_t yfrac;
  fixed_t ystep;
  unsigned long col1, col2, col3, col4;
  unsigned long x1, x2, y1, y2;

  if (dc_ystep > 0x20000)
    R_DrawColumn8_CVersion();

  count = dc_yh - dc_yl + 1;

  // Zero length, column does not exceed a pixel.
  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  // Framebuffer destination address.
  // Use ylookup LUT to avoid multiply with ScreenWidth.
  dest = ylookup[dc_yl] + columnofs[dc_x];

  // Determine scaling, which is the only mapping to be done.
  ystep = dc_ystep;
  yfrac = dc_yfrac - FRACUNIT / 2;

  dc_xfrac &= 0xffff;

  // x position is constant
  x1 = (dc_xfrac >> (16 - BLFshift));
  x2 = BLFmax - x1;

  // Inner loop that does the actual texture mapping, e.g. a DDA-lile scaling.
  do
  {
    col1 = dc_source[(yfrac >> FRACBITS) & 127];
    col2 = dc_source2[(yfrac >> FRACBITS) & 127];
    col3 = dc_source[((yfrac >> FRACBITS) + 1) & 127];
    col4 = dc_source2[((yfrac >> FRACBITS) + 1) & 127];

    // Get the texture sub-coordinates
    y1 = (yfrac >> (16 - BLFshift)) & BLFmax;
    y2 = BLFmax - y1;

    col1 = BLFTab[x2][y2][col1]
        + BLFTab[x1][y2][col2]
        + BLFTab[x2][y1][col3]
        + BLFTab[x1][y1][col4];

    // Convert to usable RGB
    col1 |= 0x07c1fc1f;

    // Store pixel
    *dest = dc_colourmap[rgb_32k[0][0][col1 & (col1 >> 17)]];

    dest += vb_pitch;
    yfrac += ystep;
  }
  while (--count);
#endif
}

//
// R_DrawFuzzColumn8
//
// Spectre/Invisibility.
//
// Creates a fuzzy image by opying pixels from above/below.
//
// Used with an all black colourmap, this creates the FUZZY effect,
// i.e. spectres and invisible players.

void R_DrawFuzzColumn8(void)
{
  int count;
  byte *dest;
  int fuzzpos = framecount;

  // Adjust borders. Low...
  if (dc_yl < 6)
    dc_yl = 6;

  count = dc_yh - dc_yl;

  // Zero length.
  if (count < 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  // Does not work with blocky mode.
  dest = ylookup[dc_yl] + columnofs[dc_x];

  do
  {
    fuzzpos = (fuzzpos+1) & 7;

    // Lookup framebuffer, and retrieve a pixel that is zero to seven
    // pixels above the current one.

    *dest = dc_colourmap[dest[-fuzzpos * vb_pitch]];

    dest += vb_pitch;
  }
  while (count--);
}

// -ES- 1998/10/29 New translucency
//
// -KM- 1998/11/25 Modified rest of EDGE to work with this.
//
// -ES- 1998/02/12 New col2rgb format

void R_DrawTranslucentColumn8()
{
  int count;
  byte *dest;
  fixed_t frac;
  fixed_t fracstep;

  fixed_t fglevel, bglevel;
  unsigned long *fg2rgb, *bg2rgb;
  unsigned long fg;  // current colours

  fglevel = dc_translucency;
  fglevel = fglevel & ~0x3ff;
  bglevel = FRACUNIT - fglevel;

  fg2rgb = col2rgb8[fglevel >> 10];
  bg2rgb = col2rgb8[bglevel >> 10];

  count = dc_yh - dc_yl + 1;

  // Zero length, column does not exceed a pixel.
  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  dest = ylookup[dc_yl] + columnofs[dc_x];

  fracstep = dc_ystep;
  frac = dc_yfrac;

  do
  {
    /*

       New translucency algorithm, by Erik Sandberg:

       Basically, we compute the red, green and blue values for each pixel, and
       then use a RGB table to check which one of the palette colours that best
       represents those RGB values. The RGB table is 8k big, with 4 R-bits,
       5 G-bits and 4 B-bits. A 4k table gives a bit too bad precision, and a 32k
       table takes up more memory and results in more cache misses, so an 8k
       table seemed to be quite ultimate.

       The computation of the RGB for each pixel is accelerated by using two
       1k tables for each translucency level.
       The xth element of one of these tables contains the r, g and b values for
       the colour x, weighted for the current translucency level (for example,
       the weighted rgb values for background colour at 75% translucency are 1/4
       of the original rgb values). The rgb values are stored as three
       low-precision fixed point values, packed into one long per colour:
       Bit 0-4:   Frac part of blue  (5 bits)
       Bit 5-8:   Int  part of blue  (4 bits)
       Bit 9-13:  Frac part of red   (5 bits)
       Bit 14-17: Int  part of red   (4 bits)
       Bit 18-22: Frac part of green (5 bits)
       Bit 23-27: Int  part of green (5 bits)

       The point of this format is that the two colours now can be added, and
       then be converted to a RGB table index very easily: First, we just set
       all the frac bits and the four upper zero bits to 1. It's now possible
       to get the RGB table index by ORing the current value >> 7 with the
       current value >> 23, and then mask away some high bits by anding it
       with 0x1FFF.

     */

    fg = (fg2rgb[dc_colourmap[dc_source[(frac >> FRACBITS) & 127]]] +
        bg2rgb[*dest]) | 0x07c1fc1f;

    *dest = rgb_32k[0][0][fg & (fg >> 17)];

    dest += vb_pitch;
    frac += fracstep;
  }
  while (--count);
}

//
// R_DrawTranslatedColumn8
//
// Uses the translation tables to remap one set of palette colours to
// another. One prime example is the player greens to the other player
// colours.
//
// Could be used with different translation tables, e.g. the lighter coloured
// version of the BaronOfHell, the HellKnight, uses identical sprites,
// kinda brightened up.
//
// 8-Bit Colour Version.
//
void R_DrawTranslatedColumn8(void)
{
  int count;
  byte *dest;
  fixed_t frac;
  fixed_t fracstep;

  count = dc_yh - dc_yl;

  if (count < 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  dest = ylookup[dc_yl] + columnofs[dc_x];

  // Looks familiar.
  fracstep = dc_ystep;
  frac = dc_yfrac;

  // Here we do an additional index re-mapping.
  do
  {
    //
    // Translation tables are used to map certain colourramps to other ones,
    // used with PLAY sprites. Thus the "green" ramp of the player 0 sprite
    // is mapped to gray, red, black/indigo.
    //
    *dest = dc_colourmap[dc_translation[dc_source[frac >> FRACBITS]]];
    dest += vb_pitch;

    frac += fracstep;
  }
  while (count--);
}

// -ES- 1999/02/12 New col2rgb format
void R_DrawTranslucentTranslatedColumn8()
{
  int count;
  byte *dest;
  fixed_t frac;
  fixed_t fracstep;

  fixed_t fglevel, bglevel;
  unsigned long *fg2rgb, *bg2rgb;
  unsigned long fg;  // current colours

  fglevel = dc_translucency;
  fglevel = fglevel & ~0x3ff;
  bglevel = FRACUNIT - fglevel;

  fg2rgb = col2rgb8[fglevel >> 10];
  bg2rgb = col2rgb8[bglevel >> 10];

  count = dc_yh - dc_yl + 1;

  if (count <= 0)
    return;

#ifdef DEVELOPERS
  if ((unsigned int)dc_x >= (unsigned int)SCREENWIDTH || dc_yl < 0 || dc_yh >= SCREENHEIGHT)
    I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

  dest = ylookup[dc_yl] + columnofs[dc_x];

  fracstep = dc_ystep;
  frac = dc_yfrac;

  do
  {
    //
    // This is R_DrawTranslucentColumn with an extra translation table lookup
    //
    fg = (fg2rgb[dc_colourmap[dc_translation[dc_source[(frac >> FRACBITS) & 127]]]] +
        bg2rgb[*dest]) | 0x07c1fc1f;

    *dest = rgb_32k[0][0][fg & (fg >> 17)];

    dest += vb_pitch;
    frac += fracstep;
  }
  while (--count);
}

//
// R_DrawSpan8_CVersion
//
// With DOOM style restrictions on view orientation, the floors and ceilings
// consist of horizontal slices or spans with constant z depth.
//
// However, rotation around the world z axis is possible, thus this mapping,
// while simpler and faster than perspective correct texture mapping, has to
// traverse the texture at an angle in all but a few cases.
//
// In consequence, flats are not stored by column (like walls), and the inner
// loop has to step in texture space u and v.
//
// Draws the actual span.
//
// -ES- 1999/04/19 Optimised slightly
void R_DrawSpan8_CVersion(void)
{
  unsigned long xfrac;
  unsigned long yfrac;
  byte *dest;
  int count;
  int spot;

#ifdef DEVELOPERS
  if (ds_x2 < ds_x1 || ds_x1 < 0 || ds_x2 >= SCREENWIDTH || (unsigned int)ds_y > (unsigned int)SCREENHEIGHT)
    I_Error("R_DrawSpan: %i to %i at %i", ds_x1, ds_x2, ds_y);
#endif

  xfrac = ds_xfrac << 10;
  ds_xstep <<= 10;
  yfrac = ds_yfrac;

  dest = ylookup[ds_y] + columnofs[ds_x1];

  count = ds_x2 - ds_x1 + 1;

  do
  {
    // Current texture index in u,v.
    spot = ((yfrac >> (16 - 6)) & (63 * 64)) + (xfrac >> 26);

    // Lookup pixel from flat texture tile,
    // re-index using light/colourmap.
    *dest++ = ds_colourmap[ds_source[spot]];

    // Next step in u,v.
    xfrac += ds_xstep;
    yfrac += ds_ystep;
  }
  while (--count);
}

void R_DrawSpan8_MIP(void)
{
  fixed_t xfrac, yfrac;
  fixed_t xmask, ymask;
  fixed_t xstep, ystep;

  byte *dest;
  int count;
  int spot;

#ifdef DEVELOPERS
  if (ds_x2 < ds_x1 || ds_x1 < 0 || ds_x2 >= SCREENWIDTH || (unsigned int)ds_y > (unsigned int)SCREENHEIGHT)
    I_Error("R_DrawSpan8_MIP: %i to %i at %i", ds_x1, ds_x2, ds_y);
#endif

  xfrac = ds_xfrac;
  yfrac = ds_yfrac * ds_width;

  xmask = (ds_width-1) << FRACBITS;
  ymask = ((ds_height-1) * ds_width) << FRACBITS;

  xstep = ds_xstep;
  ystep = ds_ystep * ds_width;

  dest = ylookup[ds_y] + columnofs[ds_x1];

  // We do not check for zero spans here?
  count = ds_x2 - ds_x1;

  do
  {
    // Current texture index in u,v.  Maximum texture size is 128x128.
    spot = ((yfrac & ymask) + (xfrac & xmask)) >> FRACBITS;

    // Lookup pixel from flat texture tile, re-index using light/colourmap.
    *dest++ = ds_colourmap[ds_source[spot]];

    // Next step in u,v.
    xfrac += xstep;
    yfrac += ystep;
  }
  while (count--);
}

void R_DrawTranslucentSpan8_MIP(void)
{
  fixed_t xfrac, yfrac;
  fixed_t xmask, ymask;
  fixed_t xstep, ystep;

  byte *dest;
  int count;
  int spot;

  fixed_t fglevel, bglevel;
  unsigned long *fg2rgb, *bg2rgb;
  unsigned long fg;  // current colours

#ifdef DEVELOPERS
  if (ds_x2 < ds_x1 || ds_x1 < 0 || ds_x2 >= SCREENWIDTH || (unsigned int)ds_y > (unsigned int)SCREENHEIGHT)
    I_Error("R_DrawSpan8_MIP: %i to %i at %i", ds_x1, ds_x2, ds_y);
#endif

  fglevel = dc_translucency;
  fglevel = fglevel & ~0x3ff;
  bglevel = FRACUNIT - fglevel;

  fg2rgb = col2rgb8[fglevel >> 10];
  bg2rgb = col2rgb8[bglevel >> 10];

  xfrac = ds_xfrac;
  yfrac = ds_yfrac * ds_width;

  xmask = (ds_width-1) << FRACBITS;
  ymask = ((ds_height-1) * ds_width) << FRACBITS;

  xstep = ds_xstep;
  ystep = ds_ystep * ds_width;

  dest = ylookup[ds_y] + columnofs[ds_x1];

  // We do not check for zero spans here?
  count = ds_x2 - ds_x1;

  do
  {
    // Current texture index in u,v.  Maximum texture size is 128x128.
    spot = ((yfrac & ymask) + (xfrac & xmask)) >> FRACBITS;

    // Lookup pixel from flat texture tile, re-index using light/colourmap.
    fg = (fg2rgb[ds_colourmap[ds_source[spot]]] +
        bg2rgb[*dest]) | 0x07c1fc1f;

    *dest++ = rgb_32k[0][0][fg & (fg >> 17)];

    // Next step in u,v.
    xfrac += xstep;
    yfrac += ystep;
  }
  while (count--);
}

void R_DrawHoleySpan8_MIP(void)
{
  fixed_t xfrac, yfrac;
  fixed_t xmask, ymask;
  fixed_t xstep, ystep;

  byte *dest;
  int count;
  int spot;

#ifdef DEVELOPERS
  if (ds_x2 < ds_x1 || ds_x1 < 0 || ds_x2 >= SCREENWIDTH || (unsigned int)ds_y > (unsigned int)SCREENHEIGHT)
    I_Error("R_DrawSpan8_MIP: %i to %i at %i", ds_x1, ds_x2, ds_y);
#endif

  xfrac = ds_xfrac;
  yfrac = ds_yfrac * ds_width;

  xmask = (ds_width-1) << FRACBITS;
  ymask = ((ds_height-1) * ds_width) << FRACBITS;

  xstep = ds_xstep;
  ystep = ds_ystep * ds_width;

  dest = ylookup[ds_y] + columnofs[ds_x1];

  // We do not check for zero spans here?
  count = ds_x2 - ds_x1;

  do
  {
    // Current texture index in u,v.  Maximum texture size is 128x128.
    spot = ((yfrac & ymask) + (xfrac & xmask)) >> FRACBITS;

    // Lookup pixel from flat texture tile, re-index using light/colourmap.
    if (ds_source[spot] != TRANS_PIXEL)
      *dest++ = ds_colourmap[ds_source[spot]];

    // Next step in u,v.
    dest++;
    xfrac += xstep;
    yfrac += ystep;
  }
  while (count--);
}


// -AJA- 1999/08/16: Added this.
void R_DrawTranslucentSpan8(void)
{
  unsigned long xfrac;
  unsigned long yfrac;
  byte *dest;
  int count;
  int spot;

  fixed_t fglevel, bglevel;
  unsigned long *fg2rgb, *bg2rgb;
  unsigned long fg;  // current colours

#ifdef DEVELOPERS
  if (ds_x2 < ds_x1 || ds_x1 < 0 || ds_x2 >= SCREENWIDTH || (unsigned int)ds_y > (unsigned int)SCREENHEIGHT)
    I_Error("R_DrawTranslucentSpan8: %i to %i at %i", ds_x1, ds_x2, ds_y);
#endif

  fglevel = dc_translucency;
  fglevel = fglevel & ~0x3ff;
  bglevel = FRACUNIT - fglevel;

  fg2rgb = col2rgb8[fglevel >> 10];
  bg2rgb = col2rgb8[bglevel >> 10];

  xfrac = ds_xfrac << 10;
  ds_xstep <<= 10;
  yfrac = ds_yfrac;

  dest = ylookup[ds_y] + columnofs[ds_x1];

  count = ds_x2 - ds_x1 + 1;

  do
  {
    spot = ((yfrac >> (16 - 6)) & (63 * 64)) + (xfrac >> 26);

    // Lookup pixel from flat texture tile,
    // re-index using light/colourmap, and apply translucency.
    fg = (fg2rgb[ds_colourmap[ds_source[spot]]] +
        bg2rgb[*dest]) | 0x07c1fc1f;

    *dest++ = rgb_32k[0][0][fg & (fg >> 17)];

    xfrac += ds_xstep;
    yfrac += ds_ystep;
  }
  while (--count);
}

#if 0
// -ES- 1999-04-07 Mipmapping test
void R_DrawSpan8_MIP(void)
{
  fixed_t xfrac;
  fixed_t yfrac;
  byte *dest;
  int count;
  int spot;

  unsigned long xmask, ymask;
  unsigned long i;

#ifdef DEVELOPERS
  if (ds_x2 < ds_x1 || ds_x1 < 0 || ds_x2 >= SCREENWIDTH || (unsigned int)ds_y > (unsigned int)SCREENHEIGHT)
    I_Error("R_DrawSpan: %i to %i at %i", ds_x1, ds_x2, ds_y);
#endif

  xfrac = ds_xfrac;
  yfrac = ds_yfrac;

  dest = ylookup[ds_y] + columnofs[ds_x1];

  count = ds_x2 - ds_x1;

  xmask = 0xffff0000;
  for (i = abs(ds_xstep); i > FRACUNIT; i >>= 1)
    xmask <<= 1;
  xmask &= 63 * FRACUNIT;
  ymask = 0xffff0000;
  for (i = abs(ds_ystep); i > FRACUNIT; i >>= 1)
    ymask <<= 1;
  ymask &= 63 * FRACUNIT;
  do
  {
    spot = ((yfrac & ymask) >> (16 - 6)) + ((xfrac & xmask) >> 16);

    *dest++ = ds_colourmap[ds_source[spot]];

    // Next step in u,v.
    xfrac += ds_xstep;
    yfrac += ds_ystep;
  }
  while (count--);
}
#endif

// -ES- 1998/12/18 Added this one
void R_DrawSpan8_KM(void)
{
  unsigned long xfrac;
  unsigned long yfrac;
  unsigned char *dest;
  int count;
  int spot;

#ifdef DEVELOPERS
  if (ds_x2 < ds_x1 || ds_x1 < 0 || ds_x2 >= SCREENWIDTH || (unsigned int)ds_y > (unsigned int)SCREENHEIGHT)
    I_Error("R_DrawSpan: %i to %i at %i", ds_x1, ds_x2, ds_y);
#endif

  xfrac = ds_xfrac;
  yfrac = ds_yfrac;

  dest = ylookup[ds_y] + columnofs[ds_x1];

  count = ds_x2 - ds_x1;

  if (((ds_xstep & 0xffff) > FRACUNIT) && ((ds_ystep & 0xffff) > FRACUNIT))
  {
    do
    {
      spot = ((yfrac >> (FRACBITS - 6)) & (63 * 64)) + ((xfrac >> FRACBITS) & 63);

      *dest++ = dc_colourmap[ds_source[spot]];

      xfrac += ds_xstep;
      yfrac += ds_ystep;

    }
    while (count--);
  }
  else
  {
    unsigned long spot[4];
    unsigned long level[4];

    // current colour
    unsigned long c;

    int i;

    xfrac -= FRACUNIT / 2;
    yfrac -= FRACUNIT / 2;

    do
    {
      spot[0] = ((yfrac >> 16) & 63) * 64 + ((xfrac >> 16) & 63);
      spot[1] = ((yfrac >> 16) & 63) * 64 + (((xfrac >> 16) + 1) & 63);
      spot[2] = (((yfrac >> 16) + 1) & 63) * 64 + ((xfrac >> 16) & 63);
      spot[3] = (((yfrac >> 16) + 1) & 63) * 64 + (((xfrac >> 16) + 1) & 63);

      level[3] = ((yfrac & 0xffff) * (xfrac & 0xffff));
      level[1] = (FRACUNIT - (yfrac & 0xffff) - 1) * (xfrac & 0xffff);
      level[2] = (yfrac & 0xffff) * (FRACUNIT - (xfrac & 0xffff) - 1);
      level[0] = (FRACUNIT - (yfrac & 0xffff) - 1) * (FRACUNIT - (xfrac & 0xffff) - 1);

      for (i = 0, c = 0; i < 4; i++)
      {
        spot[i] = (unsigned long)(ds_source[spot[i]]);
        c += col2rgb8[level[i] >> 26][spot[i]];
      }

      // -ES- 1999/03/13 Fixed the RGB conversion
      c |= 0x07c1fc1f;
      *dest++ = ds_colourmap[rgb_32k[0][0][c & (c >> 17)]];

      xfrac += ds_xstep;
      yfrac += ds_ystep;
    }
    while (count--);
  }
}

//------------------------------------------------------------
// Bilinear Filtering
// 16-bit version originally written by Vitek Kavan vit.kavan@usa.net
// Improved & converted to 8-bit colour by -ES- 1999/02/12

void R_DrawSpan8_BLF(void)
{
  int count;
  unsigned long col1, col2, col3, col4;
  unsigned long x1, x2, y1, y2;
  unsigned char *dest = (ylookup[ds_y] + columnofs[ds_x1]);
  unsigned long xfrac = ds_xfrac - FRACUNIT / 2;
  unsigned long yfrac = ds_yfrac - FRACUNIT / 2;

  count = ds_x2 - ds_x1;

  do
  {
    // Get the texture coordinates
    y1 = ((yfrac >> 10) & (63 * 64));
    x1 = ((xfrac >> 16) & 63);
    y2 = (y1 + 64) & (63 * 64);
    x2 = (x1 + 1) & 63;

    // Get the colours of the four corners
    col1 = ds_source[y1 + x1];
    col2 = ds_source[y1 + x2];
    col3 = ds_source[y2 + x1];
    col4 = ds_source[y2 + x2];

    // Get the texture sub-coordinates
    x1 = (xfrac >> (16 - BLFshift)) & BLFmax;
    y1 = (yfrac >> (16 - BLFshift)) & BLFmax;
    x2 = BLFmax - x1;
    y2 = BLFmax - y1;

    // Get the fixed-point RGB value
    col1 = BLFTab[x2][y2][col1]
        + BLFTab[x1][y2][col2]
        + BLFTab[x2][y1][col3]
        + BLFTab[x1][y1][col4];

    // Convert to usable RGB
    col1 |= 0x07c1fc1f;

    // Store pixel
    *dest++ = ds_colourmap[rgb_32k[0][0][col1 & (col1 >> 17)]];

    // Next step
    xfrac += ds_xstep;
    yfrac += ds_ystep;
  }
  while (count--);
}

//
// R_InitFunctions_Draw1
//
void R_InitFunctions_Draw1(void)
{
  CON_InitFunctionList(&drawcol8_funcs, "col8", R_DrawColumn8_CVersion, ColSmoothingOff);
  CON_InitFunctionList(&drawspan8_funcs, "span8", R_DrawSpan8_CVersion, SpanSmoothingOff);

  CON_AddFunctionToList(&drawcol8_funcs, "BLF", "Bilinear Filtering", R_DrawColumn8_BLF, BLF_Init8);
  CON_AddFunctionToList(&drawcol8_funcs, "VFI", "Very fast Interpolation", R_DrawColumn8_vfi, ColSmoothingOn);
  CON_AddFunctionToList(&drawcol8_funcs, "KM", "Kester's Smoothing", R_DrawColumn8_KM, ColSmoothingOn);
  CON_AddFunctionToList(&drawcol8_funcs, "None", "Dummy routine, doesn't draw anything", R_DrawColumn8_dontdraw, ColSmoothingOff);

  CON_AddFunctionToList(&drawspan8_funcs, "BLF", "Bilinear Filtering", R_DrawSpan8_BLF, BLF_Init8);
  CON_AddFunctionToList(&drawspan8_funcs, "VFI", "Very fast Interpolation", R_DrawSpan8_vfi, SpanSmoothingOn);
  CON_AddFunctionToList(&drawspan8_funcs, "KM", "Kester's Smoothing", R_DrawSpan8_KM, SpanSmoothingOn);
  CON_AddFunctionToList(&drawspan8_funcs, "None", "Dummy routine, doesn't draw anything", R_DrawSpan8_dontdraw, SpanSmoothingOff);
}

//
// R_FillBackScreen8
//
// Fills the back screen with a pattern for variable screen sizes
// Also draws a beveled edge.
//
void R_FillBackScreen8(void)
{
#if 0  // OLD CODE
  const byte *src;
  byte *dest;
  int x;
  int y;
  const patch_t *patch;

  if ((viewwindowwidth == SCREENWIDTH) && (viewwindowheight == SCREENHEIGHT))
    return;

  src = W_CacheLumpName(currentmap->surround);

  for (y = 0; y < back_scr->height; y++)
  {
    dest = back_scr->data + back_scr->pitch * y;
    for (x = 0; x < back_scr->width / 64; x++)
    {
      Z_MoveData(dest, src + ((y & 63) << 6), byte, 64);
      dest += 64;
    }

    if (back_scr->width & 63)
    {
      Z_MoveData(dest, src + ((y & 63) << 6), byte, back_scr->width & 63);
    }
  }

  W_DoneWithLump(src);

  if (SCREENWIDTH == viewwindowwidth)
    return;

  if ((viewwindowy - 8) >= 0)
  {
    patch = W_CacheLumpName("brdr_t");
    for (x = 0; x < viewwindowwidth; x += 8)
      V_DrawPatch(back_scr, viewwindowx + x, viewwindowy - 8, patch);
    W_DoneWithLump(patch);
  }

  if ((viewwindowy + viewwindowheight + 8) < (SCREENHEIGHT - SBARHEIGHT))
  {
    patch = W_CacheLumpName("brdr_b");
    for (x = 0; x < viewwindowwidth; x += 8)
      V_DrawPatch(back_scr, viewwindowx + x, viewwindowy + viewwindowheight, patch);
    W_DoneWithLump(patch);
  }

  if ((viewwindowx - 8) >= 0)
  {
    patch = W_CacheLumpName("brdr_l");
    for (y = 0; y < viewwindowheight; y += 8)
      V_DrawPatch(back_scr, viewwindowx - 8, viewwindowy + y, patch);
    W_DoneWithLump(patch);
  }

  if ((viewwindowx + viewwindowwidth + 8) < SCREENWIDTH)
  {
    patch = W_CacheLumpName("brdr_r");
    for (y = 0; y < viewwindowheight; y += 8)
      V_DrawPatch(back_scr, viewwindowx + viewwindowwidth, viewwindowy + y, patch);
    W_DoneWithLump(patch);
  }

  // Draw beveled edge.
  if (((viewwindowx - 8) >= 0) && ((viewwindowy - 8) >= 0))
  {
    V_DrawPatchName(back_scr, viewwindowx - 8, viewwindowy - 8, "brdr_tl");
  }

  if (((viewwindowx + viewwindowwidth + 8) < SCREENWIDTH) && ((viewwindowy - 8) >= 0))
  {
    V_DrawPatchName(back_scr, viewwindowx + viewwindowwidth, viewwindowy - 8, "brdr_tr");
  }

  if (((viewwindowx - 8) >= 0) && ((viewwindowy + viewwindowheight + 8) < (SCREENHEIGHT - SBARHEIGHT)))
  {
    V_DrawPatchName(back_scr, viewwindowx - 8, viewwindowy + viewwindowheight, "brdr_bl");
  }

  if (((viewwindowx + viewwindowwidth + 8) < SCREENWIDTH) &&
      ((viewwindowy + viewwindowheight + 8) < (SCREENHEIGHT - SBARHEIGHT)))
  {
    V_DrawPatchName(back_scr, viewwindowx + viewwindowwidth, viewwindowy + viewwindowheight, "brdr_br");
  }
#endif
}

//
// R_VideoErase8
//
// Linear Frame Buffer copy.
//
void R_VideoErase8(unsigned ofs, int count)
{
#if 0  // OLD CODE
  Z_MoveData(main_scr->data + ofs, back_scr->data + ofs, byte, count);
#endif
}

//
// R_DrawViewBorder8
//
// Draws the border around the view for different size windows?
// 
// -ES- 1999/07/18 Fixed stuff regarding SCREENPITCH
void R_DrawViewBorder8(void)
{
#if 0  // OLD CODE
  int side;
  int ofs;
  int i;

  //
  // if screenwidth>320, draw stuff around status bar, even if
  // viewwindowwidth==SCREENWIDTH
  //
  if ((SCREENWIDTH > 320) && (SCREENHEIGHT != viewwindowheight))
  {
    ofs = (SCREENHEIGHT - SBARHEIGHT) * SCREENPITCH;
    side = (SCREENWIDTH - 320) / 2;
    for (i = 0; i < SBARHEIGHT; i++)
    {
      R_VideoErase8(ofs, side);
      R_VideoErase8(ofs + SCREENWIDTH - side, side);
      ofs += SCREENPITCH;
    }
  }

  // don't draw stuff over and under viewwindow if there is no space to draw it
  if ((viewwindowy == 0) && ((viewwindowy + viewwindowheight) >= (SCREENHEIGHT - SBARHEIGHT)))
    return;

  // copy top
  for (i = 0; i < viewwindowy; i++)
    R_VideoErase8(i * SCREENPITCH, SCREENWIDTH);

  // copy left and right side
  for (i = viewwindowy; i < viewwindowy + viewwindowheight; i++)
  {
    // left
    R_VideoErase8(i * SCREENPITCH, viewwindowx);
    // right
    R_VideoErase8(i * SCREENPITCH + viewwindowx + viewwindowwidth,
        SCREENWIDTH - (viewwindowx + viewwindowwidth));
  }

  // copy bottom
  for (i = viewwindowy + viewwindowheight; i < SCREENHEIGHT - SBARHEIGHT; i++)
    R_VideoErase8(i * SCREENPITCH, SCREENWIDTH);

  V_MarkRect(0, 0, SCREENWIDTH, SCREENHEIGHT - SBARHEIGHT);
#endif
}
