//----------------------------------------------------------------------------
//  EDGE OpenGL Rendering (BSP Traversal)
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------
//
// -AJA- 2000/01/03: Wrote this file (by cleaning up r2_bsp.c).
//
// TODO HERE:
//   * use IM_WIDTH(),IM_BOTTOM(),etc where needed.
//   + optimise first subsector: ignore floors out of view.
//   + split up: rgl_seg.c and rgl_mobj.c.
//   + handle scaling better.
//   - implement halos.
//

// this conditional applies to the whole file
#ifdef USE_GL

#include "i_defs.h"

#include "dm_defs.h"
#include "dm_state.h"
#include "e_search.h"
#include "m_bbox.h"
#include "m_random.h"
#include "p_local.h"
#include "p_mobj.h"
#include "r_defs.h"
#include "r_main.h"
#include "r_plane.h"
#include "r_sky.h"
#include "r_state.h"
#include "r_things.h"
#include "r2_defs.h"
#include "rgl_defs.h"
#include "v_colour.h"
#include "v_res.h"
#include "w_textur.h"
#include "w_wad.h"
#include "v_ctx.h"
#include "z_zone.h"


#define DEBUG  0

#define HALOS    0


// colour of the player's weapon
int rgl_weapon_r;
int rgl_weapon_g;
int rgl_weapon_b;

//!!!
extern int glpar_invuln;


// common stuff

extern sector_t *frontsector;
extern sector_t *backsector;

static subsector_t *cur_sub;
static seg_t *cur_seg;

static boolean_t cur_upper_sky;
static boolean_t cur_lower_sky;
static boolean_t cur_upper_invis;
static boolean_t cur_lower_invis;

static boolean_t solid_mode;

//!!! HACK FIXME
extern boolean_t ren_allbright;
extern float_t ren_red_mul;
extern float_t ren_grn_mul;
extern float_t ren_blu_mul;

static subsector_t *drawsubs_head;
static subsector_t *drawsubs_tail;

int rgl_currtimeval;  // hack..


// The minimum distance between player and a visible sprite.
#define MINZ        (4.0)


//
// LIGHTING RULES:
//
//   1. N/S/E/W changes and infrared are applied before darkening.
//   2. extralight and dlights are applied after darkening.
//   3. for gamma, we rely on the gamma function g() having the
//      property: g(X * Y) = g(X) * g(Y).
//

static INLINE int RGL_Light(int nominal)
{
  if (ren_allbright)
    return 255;
  
  if (effect_infrared)
    nominal += (int)(effect_strength * 255);

  DEV_ASSERT2(currentmap);
  if (currentmap->lighting <= LMODEL_Doomish)
  {
    nominal = rgl_light_map[MIN(255, nominal)];
  }
  
  nominal += extralight * 16;
  
  return GAMMA_CONV(MIN(255, nominal));
}

#define LT_RED(light)  (MIN(255,light) * ren_red_mul / 255.0)
#define LT_GRN(light)  (MIN(255,light) * ren_grn_mul / 255.0)
#define LT_BLU(light)  (MIN(255,light) * ren_blu_mul / 255.0)


typedef struct wall_plane_data_s
{
  vec3_t normal;
  divline_t div;

  int col[3];
  float_t trans;

  drawthing_t *dlights;

  float_t tx, tdx;
  float_t ty, ty_top, ty_mul, ty_skew;

  vec2_t x_mat, y_mat;
}
wall_plane_data_t;


void WallCoordFunc(vec3_t *src, local_gl_vert_t *vert, void *d)
{
  wall_plane_data_t *data = (wall_plane_data_t *)d;

  int R = data->col[0];
  int G = data->col[1];
  int B = data->col[2];

  float_t x = src->x;
  float_t y = src->y;
  float_t z = src->z;

  float_t tx, ty;
  float_t along;

  // compute texture coord
  if (fabs(data->div.dx) > fabs(data->div.dy))
  {
    CHECKVAL(data->div.dx);
    along = (x - data->div.x) / data->div.dx;
  }
  else
  {
    CHECKVAL(data->div.dy);
    along = (y - data->div.y) / data->div.dy;
  }

  tx = data->tx + along * data->tdx;
  ty = data->ty + (data->ty_top - z) * data->ty_mul +
       along * data->ty_skew;
   
  // dynamic lighting
  if (use_dlights)
  {
    drawthing_t *dl;

    for (dl=data->dlights; dl; dl=dl->next)
    {
      // light behind seg ?    
      if (P_PointOnDivlineSide(dl->mo->x, dl->mo->y, &data->div) != 0)
        continue;
       
      R2_AddColourDLights(1, &R, &G, &B, &x, &y, &z, dl->mo);
    }
  }

  SET_COLOR(LT_RED(R), LT_GRN(G), LT_BLU(B), data->trans);
  SET_TEXCOORD(tx, ty);
  SET_NORMAL(data->normal.x, data->normal.y, data->normal.z);
  SET_EDGE_FLAG(GL_TRUE);
  SET_VERTEX(x, y, z);
}


void PlaneCoordFunc(vec3_t *src, local_gl_vert_t *vert, void *d)
{
  wall_plane_data_t *data = (wall_plane_data_t *)d;

  float_t x = src->x;
  float_t y = src->y;
  float_t z = src->z;

  float_t rx = (x + data->tx) / 64;
  float_t ry = (y + data->ty) / 64;

  float_t tx = rx * data->x_mat.x + ry * data->x_mat.y;
  float_t ty = rx * data->y_mat.x + ry * data->y_mat.y;

  int R = data->col[0];
  int G = data->col[1];
  int B = data->col[2];

  // dynamic lighting
  if (use_dlights)
  {
    drawthing_t *dl;

    for (dl=data->dlights; dl; dl=dl->next)
    {
      // light behind the plane ?    
      if ((dl->tz > z) != (data->normal.z > 0))
        continue;
 
      R2_AddColourDLights(1, &R, &G, &B, &x, &y, &z, dl->mo);
    }
  }

  SET_COLOR(LT_RED(R), LT_GRN(G), LT_BLU(B), data->trans);
  SET_TEXCOORD(tx, ty);
  SET_NORMAL(data->normal.x, data->normal.y, data->normal.z);
  SET_EDGE_FLAG(GL_TRUE);
  SET_VERTEX(x, y, z);
}


//
// RGL_DrawWall
//
// Note: mid_masked is 2 & 3 for horiz. sliding door.
//
static void RGL_DrawWall(drawfloor_t *dfloor, float_t top,
    float_t bottom, float_t tex_top_h, surface_t *part,
    int mid_masked, boolean_t opaque, float_t x_offset)
{
  float_t x1 = cur_seg->v1->x;
  float_t y1 = cur_seg->v1->y;
  float_t x2 = cur_seg->v2->x;
  float_t y2 = cur_seg->v2->y;

  float_t xy_ofs = cur_seg->offset;
  float_t xy_len = cur_seg->length;

  float_t tex_x1 = 0, tex_y1 = 0;
  float_t tex_x2 = 0, tex_y2 = 0, tex_dy = 0;

  int w, h;
  float_t t_right, t_bottom;
  float_t trans = part->translucency;

  boolean_t blended;

  GLuint tex_id;
  const cached_image_t *cim;

  raw_polyquad_t *poly;
  wall_plane_data_t data;

  region_properties_t *props = part->override_p ? part->override_p : 
      dfloor->props;
  int lit_Nom = props->lightlevel;
  const colourmap_t *colmap = props->colourmap;
  float_t c_r, c_g, c_b;

  // ignore non-solid walls in solid mode (& vice versa)
  blended = (part->translucency <= 0.99) ? true : false;
  if ((blended || mid_masked) == solid_mode)
    return;
  
  DEV_ASSERT2(currentmap);
  
  // do the N/S/W/E bizzo...
  if (currentmap->lighting == LMODEL_Doom)
  {
    if (cur_seg->v1->y == cur_seg->v2->y)
      lit_Nom -= 8;
    else if (cur_seg->v1->x == cur_seg->v2->x)
      lit_Nom += 8;

    // limit to 0..255 range
    lit_Nom = MAX(0, MIN(255, lit_Nom));
  }

  V_GetColmapRGB(colmap, &c_r, &c_g, &c_b, false);

  lit_Nom = RGL_Light(lit_Nom);

  data.col[0] = lit_Nom * c_r;
  data.col[1] = lit_Nom * c_g;
  data.col[2] = lit_Nom * c_b;

  data.trans = trans;

  data.dlights = dfloor->dlights;

  DEV_ASSERT2(part->image);
  cim = W_ImageCache(part->image, IMG_OGL, 0, true);
  tex_id = W_ImageGetOGL(cim);

  // Note: normally this would be wrong, since we're using the GL
  // texture ID later on (after W_ImageDone).  The W_LockImagesOGL
  // call saves us though.
  W_ImageDone(cim);

  w = part->image->actual_w;
  h = part->image->actual_h;
  t_right  = w / (float_t)part->image->total_w;
  t_bottom = h / (float_t)part->image->total_h;

  x_offset += xy_ofs;

  tex_x1 = x_offset;
  tex_x2 = tex_x1 + xy_len;
  
  tex_y1 = tex_top_h - top;
  tex_y2 = tex_top_h - bottom;
  tex_dy = part->y_mat.x * xy_len;

  // horizontal sliding door hack
  if (mid_masked >= 2)
  {
    slider_move_t *smov = cur_seg->linedef->slider_move;
    line_t *L;

    float_t start, end;
    float_t seg_start, seg_end;

    // which side is seg on ?
    int side = 0;

    if ((cur_seg->v2->x - cur_seg->v1->x) * cur_seg->linedef->dx < 0 ||
        (cur_seg->v2->y - cur_seg->v1->y) * cur_seg->linedef->dy < 0)
    {
      side = 1;
    }

    if (side == 0)
    {
      seg_start = xy_ofs;
      seg_end   = seg_start + xy_len;
    }
    else
    {
      seg_end   = smov->line_len - xy_ofs;
      seg_start = seg_end - xy_len;
    }

    DEV_ASSERT2(smov);

    switch (smov->info->type * 2 + (mid_masked & 1))
    {
      case (SLIDE_Left * 2 + 0):
        start = 0;
        end = smov->line_len - smov->opening;
        break;

      case (SLIDE_Right * 2 + 0):
        start = smov->opening;
        end = smov->line_len;
        break;

      case (SLIDE_Center * 2 + 0):
        start = 0;
        end = (smov->line_len - smov->opening) / 2;
        break;

      case (SLIDE_Center * 2 + 1):
        start = (smov->line_len + smov->opening) / 2;
        end = smov->line_len;
        break;
        
      default:
        return;
    }

    start = MAX(start, seg_start);
    end = MIN(end, seg_end);

    // no overlap ?
    if (end <= start + 1)
      return;

    L = cur_seg->linedef;
    CHECKVAL(smov->line_len);
    
    x1 = L->v1->x + L->dx * start / smov->line_len;
    y1 = L->v1->y + L->dy * start / smov->line_len;

    x2 = L->v1->x + L->dx * end / smov->line_len;
    y2 = L->v1->y + L->dy * end / smov->line_len;

    switch (smov->info->type * 2 + (mid_masked & 1))
    {
      case (SLIDE_Left * 2 + 0):
        tex_x1 = seg_start + smov->opening;
        tex_x2 = seg_end   + smov->opening;
        break;

      case (SLIDE_Right * 2 + 0):
        tex_x1 = seg_start - smov->opening;
        tex_x2 = seg_end   - smov->opening;
        break;

      case (SLIDE_Center * 2 + 0):
        tex_x1 = seg_start + smov->opening/2;
        tex_x2 = seg_end   + smov->opening/2;
        break;

      case (SLIDE_Center * 2 + 1):
        tex_x1 = seg_start - smov->opening/2;
        tex_x2 = seg_end   - smov->opening/2;
        break;
    }

    if (side == 1)
    {
      float_t tex_tmp = tex_x1;
      tex_x1 = tex_x2;
      tex_x2 = tex_tmp;
    }
  }
  else if (mid_masked == 1 && cur_seg->linedef->special &&
      cur_seg->linedef->special->s.type != SLIDE_None)
  {
    // which side is seg on ?
    int side = 0;

    if ((cur_seg->v2->x - cur_seg->v1->x) * cur_seg->linedef->dx < 0 ||
        (cur_seg->v2->y - cur_seg->v1->y) * cur_seg->linedef->dy < 0)
    {
      side = 1;
    }

    if (side == 1)
    {
      tex_x1 = w - tex_x1;
      tex_x2 = w - tex_x2;
    }
  }
 
  tex_x1 /= (float_t) w * part->x_mat.x;
  tex_x2 /= (float_t) w * part->x_mat.x;

  data.tx  = tex_x1;
  data.tdx = tex_x2 - tex_x1;
  
  tex_y1 /= (float_t) h * part->y_mat.y;
  tex_y2 /= (float_t) h * part->y_mat.y;
  tex_dy /= (float_t) h * part->y_mat.y;

  tex_y1 = 1.0 - t_bottom * tex_y1;
  tex_y2 = 1.0 - t_bottom * tex_y2;

  data.ty_top = tex_top_h;

  data.ty = 1.0;
  data.ty_mul = -1.0 / (float_t)part->image->total_h / part->y_mat.y;
  data.ty_skew = 0;  //!!!! FIXME

#if (DEBUG >= 3) 
    L_WriteDebug( "WALL (%d,%d,%d) -> (%d,%d,%d)\n", 
      (int) x1, (int) y1, (int) top, (int) x2, (int) y2, (int) bottom);
#endif

  data.normal.x = y2 - y1;
  data.normal.y = x1 - x2;
  data.normal.z = 0;

  data.div.x  = cur_seg->v1->x;
  data.div.y  = cur_seg->v1->y;
  data.div.dx = cur_seg->v2->x - data.div.x;
  data.div.dy = cur_seg->v2->y - data.div.y;

  poly = RGL_NewPolyQuad(4, true);

  PQ_ADD_VERT(poly, x1, y1, bottom);
  PQ_ADD_VERT(poly, x1, y1, top);
  PQ_ADD_VERT(poly, x2, y2, bottom);
  PQ_ADD_VERT(poly, x2, y2, top);

  RGL_BoundPolyQuad(poly);
  
  if (use_dlights && data.dlights)
  {
    RGL_SplitPolyQuadLOD(poly, 1, 128 >> detail_level);
  }

  RGL_RenderPolyQuad(poly, &data, WallCoordFunc, tex_id, 
      mid_masked, blended);
   
  RGL_FreePolyQuad(poly);
}


//
// RGL_BuildWalls
//
// Analyses floor/ceiling heights, and add corresponding walls/floors
// to the drawfloor.  Returns true if the whole region was "solid".
//
static boolean_t RGL_BuildWalls(drawfloor_t *dfloor)
{
  side_t *sd = cur_seg->sidedef;

  float_t f1 = dfloor->f_h;
  float_t c1 = dfloor->c_h;

  float_t f, c, x_offset;
  float_t tex_top_h;

  int j;
  wall_tile_t *wt;
  boolean_t opaque;

#if (DEBUG >= 3)
    L_WriteDebug( "   BUILD WALLS %1.1f .. %1.1f\n", f1, c1);
#endif

  // handle TRANSLUCENT + THICK floors (a bit of a hack)
  if (dfloor->ef && dfloor->higher &&
      (dfloor->ef->ef_info->type & EXFL_Thick) &&
      (dfloor->ef->top->translucency <= 0.99))
  {
    c1 = dfloor->ef->top_h;
  }

  for (j=0; j < sd->tile_used; j++)
  {
    wt = sd->tiles + j;

    c = MIN(c1, wt->z2);
    f = MAX(f1, wt->z1);

    // not visible ?
    if (f >= c)
      continue;

    DEV_ASSERT2(wt->surface->image);
 
    tex_top_h = wt->tex_z + wt->surface->offset.y;

    if (wt->flags & WTILF_Extra)
      x_offset = cur_seg->sidedef->middle.offset.x;
    else
      x_offset = wt->surface->offset.x;
    
    opaque = (! cur_seg->backsector) ||
      (wt->surface->translucency > 0.99 && wt->surface->image->solid);
    
    // check for horizontal sliders
    if ((wt->flags & WTILF_MidMask) && cur_seg->linedef->special && 
        cur_seg->linedef->special->s.type != SLIDE_None &&
        cur_seg->linedef->slider_move)
    {
      RGL_DrawWall(dfloor, c, f, tex_top_h,
          wt->surface, 2, opaque, x_offset);

      if (cur_seg->linedef->special->s.type == SLIDE_Center)
      {
        RGL_DrawWall(dfloor, c, f, tex_top_h,
            wt->surface, 3, opaque, x_offset);
      }
      continue;
    }
     
    RGL_DrawWall(dfloor, c, f, tex_top_h,
        wt->surface, (wt->flags & WTILF_MidMask) ? 1 : 0, 
        opaque, x_offset);
  }

  if (cur_seg->sidedef->middle.image == NULL)
  {
    // -AJA- hack for transparent doors (this test would normally be
    // above this block, not inside it).
    //
    if (f1 >= c1)
      return true;

    return false;
  }

  // handle sliders that are totally solid and closed
  if (cur_seg->linedef->special &&
      cur_seg->linedef->special->s.type != SLIDE_None &&
      ! cur_seg->linedef->special->s.see_through &&
      ! cur_seg->linedef->slider_move)
  {
    return true;
  }

  return false;
}

//
// RGL_WalkWall
//
static void RGL_WalkWall(seg_t *seg)
{
  drawfloor_t *dfloor;

  cur_seg = seg;

#if (DEBUG >= 2)
    L_WriteDebug("   DRAW SEG %p\n", seg);
#endif

  DEV_ASSERT2(!seg->miniseg && seg->linedef);
  
  // mark the segment on the automap
  seg->linedef->flags |= ML_Mapped;

  // --- handle each floor ---
  
  frontsector = seg->front_sub->sector;
  backsector  = NULL;

  if (seg->back_sub)
    backsector = seg->back_sub->sector;

  cur_upper_sky = cur_lower_sky = false;
  cur_upper_invis = cur_lower_invis = false;

  // --- handle sky ---
  
  if (backsector && IS_SKY(frontsector->floor) && IS_SKY(backsector->floor))
    cur_lower_sky = true;

  if (backsector && IS_SKY(frontsector->ceil) && IS_SKY(backsector->ceil))
    cur_upper_sky = true;

  if (cur_lower_sky && solid_mode &&
      frontsector->f_h < backsector->f_h)
  {
    RGL_DrawSkyWall(cur_seg, frontsector->f_h, backsector->f_h);
  }

  if (IS_SKY(frontsector->ceil) && solid_mode)
  {
    if (frontsector->c_h < frontsector->sky_h &&
        (! backsector || ! IS_SKY(backsector->ceil) ||
         backsector->f_h >= frontsector->c_h))
    {
      RGL_DrawSkyWall(cur_seg, frontsector->c_h, frontsector->sky_h);
    }
    else if (backsector && IS_SKY(backsector->ceil))
    {
      float_t max_f = MAX(frontsector->f_h, backsector->f_h);

      if (backsector->c_h <= max_f && max_f < frontsector->sky_h)
      {
        RGL_DrawSkyWall(cur_seg, max_f, frontsector->sky_h);
      }
    }
  }

  // -AJA- hack to allow transparent doors
  if (backsector)
  {
    cur_lower_invis = (seg->sidedef->bottom.image == NULL) &&
        (backsector->f_h != frontsector->f_h);

    cur_upper_invis = (seg->sidedef->top.image == NULL) &&
        (backsector->c_h != frontsector->c_h);
  }

  for (dfloor=cur_sub->floors; dfloor; dfloor=dfloor->next)
  {
    RGL_BuildWalls(dfloor);
  }
}


//
// RGL_WalkSeg
//
// Visit a single seg of the subsector, and for one-sided lines update
// the 1D occlusion buffer.
//
static void RGL_WalkSeg(seg_t *seg)
{
  angle_t angle1, angle2;
  angle_t span, tspan1, tspan2;

  seg->visible = false;

  // compute distances
  {
    float_t tx1 = seg->v1->x - viewx;
    float_t ty1 = seg->v1->y - viewy;
    float_t tx2 = seg->v2->x - viewx;
    float_t ty2 = seg->v2->y - viewy;

    seg->tx1 = tx1 * viewsin - ty1 * viewcos;
    seg->tz1 = tx1 * viewcos + ty1 * viewsin;

    seg->tx2 = tx2 * viewsin - ty2 * viewcos;
    seg->tz2 = tx2 * viewcos + ty2 * viewsin;
  }

  angle1 = R_PointToAngle(viewx, viewy, seg->v1->x, seg->v1->y);
  angle2 = R_PointToAngle(viewx, viewy, seg->v2->x, seg->v2->y);

#if (DEBUG >= 3)
    L_WriteDebug( "INIT ANGLE1 = %1.2f  ANGLE2 = %1.2f\n", 
        ANG_2_FLOAT(angle1), ANG_2_FLOAT(angle2));
#endif

  // Clip to view edges.
  // -ES- 1999/03/20 Replaced clipangle with clipscope/leftclipangle/rightclipangle

  span = angle1 - angle2;

  // back side ?
  if (span >= ANG180)
    return;

  angle1 -= viewangle;
  angle2 -= viewangle;

#if (DEBUG >= 3)
    L_WriteDebug( "ANGLE1 = %1.2f  ANGLE2 = %1.2f\n", 
        ANG_2_FLOAT(angle1), ANG_2_FLOAT(angle2));
#endif

  tspan1 = angle1 - rightclipangle;
  tspan2 = leftclipangle - angle2;

  if (tspan1 > clipscope)
  {
    // Totally off the left edge?
    if (tspan2 >= ANG180)
      return;

    angle1 = leftclipangle;
  }

  if (tspan2 > clipscope)
  {
    // Totally off the left edge?
    if (tspan1 >= ANG180)
      return;

    angle2 = rightclipangle;
  }

#if (DEBUG >= 3)
    L_WriteDebug( "CLIPPED ANGLE1 = %1.2f  ANGLE2 = %1.2f\n", 
        ANG_2_FLOAT(angle1), ANG_2_FLOAT(angle2));
#endif

  // The seg is in the view range,
  // but not necessarily visible.

#if (DEBUG >= 2)
    L_WriteDebug( "  %sSEG %p (%1.1f, %1.1f) -> (%1.1f, %1.1f)\n",
        seg->miniseg ? "MINI" : "", seg, seg->v1->x, seg->v1->y, 
        seg->v2->x, seg->v2->y);
#endif
 
  // check if visible
#if 1
  if (span > (ANG1/4) && RGL_1DOcclusionTest(angle2, angle1))
  {
    return;
  }
#endif

  seg->visible = span > (ANG1 / 64);

  seg->angle1 = angle1;
  seg->angle2 = angle2;

  if (seg->miniseg)
    return;

  // only 1 sided walls affect the 1D occlusion buffer
  // FIXME: may need a hack for transparent doors
  if (seg->linedef->blocked)
    RGL_1DOcclusionSet(angle2, angle1);
}

//
// RGL_CheckBBox
//
// Checks BSP node/subtree bounding box.
// Returns true if some part of the bbox might be visible.
//
// Placed here to be close to RGL_WalkSeg(), which has similiar angle
// clipping stuff in it.
//
extern int checkcoord[12][4];

boolean_t RGL_CheckBBox(float_t *bspcoord)
{
  int boxx, boxy;
  int boxpos;

  float_t x1, y1, x2, y2;

  angle_t angle1, angle2;
  angle_t span, tspan1, tspan2;

  // Find the corners of the box
  // that define the edges from current viewpoint.
  if (viewx <= bspcoord[BOXLEFT])
    boxx = 0;
  else if (viewx < bspcoord[BOXRIGHT])
    boxx = 1;
  else
    boxx = 2;

  if (viewy >= bspcoord[BOXTOP])
    boxy = 0;
  else if (viewy > bspcoord[BOXBOTTOM])
    boxy = 1;
  else
    boxy = 2;

  boxpos = (boxy << 2) + boxx;

  if (boxpos == 5)
    return true;

  x1 = bspcoord[checkcoord[boxpos][0]];
  y1 = bspcoord[checkcoord[boxpos][1]];
  x2 = bspcoord[checkcoord[boxpos][2]];
  y2 = bspcoord[checkcoord[boxpos][3]];

  // check clip list for an open space
  angle1 = R_PointToAngle(viewx, viewy, x1, y1) - viewangle;
  angle2 = R_PointToAngle(viewx, viewy, x2, y2) - viewangle;

  span = angle1 - angle2;

  // Sitting on a line?
  if (span >= ANG180)
    return true;

  // -ES- 1999/03/20 Replaced clipangle with clipscope/leftclipangle/rightclipangle

  tspan1 = angle1 - rightclipangle;
  tspan2 = leftclipangle - angle2;

  if (tspan1 > clipscope)
  {
    // Totally off the left edge?
    if (tspan2 >= ANG180)
      return false;

    angle1 = leftclipangle;
  }

  if (tspan2 > clipscope)
  {
    // Totally off the right edge?
    if (tspan1 >= ANG180)
      return false;

    angle2 = rightclipangle;
  }

  return ! RGL_1DOcclusionTest(angle2, angle1);
}

//
// RGL_DrawPlane
//
static void RGL_DrawPlane(drawfloor_t *dfloor, float_t h,
    surface_t *info, int face_dir)
{
  seg_t *seg;

  boolean_t mid_masked = false;
  boolean_t blended;

  wall_plane_data_t data;
  raw_polyquad_t *poly;

  GLuint tex_id;
  const cached_image_t *cim;

  int num_vert, i;

  region_properties_t *props = info->override_p ?
      info->override_p : dfloor->props;

  int lit_Nom = RGL_Light(props->lightlevel);
  const colourmap_t *colmap = props->colourmap;
  float_t c_r, c_g, c_b;

  float_t trans = info->translucency;

  // ignore sky
  if (IS_SKY(*info))
    return;

  // ignore invisible planes
  if (trans < 0.01)
    return;
 
  // ignore non-facing planes
  if ((viewz > h) != (face_dir > 0))
    return;

  // ignore non-solid planes in solid_mode (& vice versa)
  blended = (trans <= 0.99) ? true : false;
  if (blended == solid_mode)
    return;

  // ignore dud regions (floor >= ceiling)
  if (dfloor->f_h > dfloor->c_h)
    return;

  // ignore empty subsectors
  if (cur_sub->segs == NULL)
    return;

  // count number of actual vertices
  for (seg=cur_sub->segs, num_vert=0; seg; seg=seg->sub_next, num_vert++)
  {
    /* nothing here */
  }

  if (num_vert > MAX_PLVERT)
    num_vert = MAX_PLVERT;
 
  V_GetColmapRGB(colmap, &c_r, &c_g, &c_b, false);

  data.col[0] = lit_Nom * c_r;
  data.col[1] = lit_Nom * c_g;
  data.col[2] = lit_Nom * c_b;

  data.trans = trans;

  data.dlights = dfloor->dlights;

  data.normal.x = 0;
  data.normal.y = 0;
  data.normal.z = (viewz > h) ? 1.0 : -1.0;
 
  DEV_ASSERT2(info->image);
  cim = W_ImageCache(info->image, IMG_OGL, 0, true);
  tex_id = W_ImageGetOGL(cim);
  
  // Note: normally this would be wrong, since we're using the GL
  // texture ID later on (after W_ImageDone).  The W_LockImagesOGL
  // call saves us though.
  //
  W_ImageDone(cim);

  data.tx = info->offset.x;
  data.ty = info->offset.y;

  data.x_mat = info->x_mat;
  data.y_mat = info->y_mat;

  poly = RGL_NewPolyQuad(num_vert, false);

  for (seg=cur_sub->segs, i=0; seg && (i < MAX_PLVERT); 
       seg=seg->sub_next, i++)
  {
    PQ_ADD_VERT(poly, seg->v1->x, seg->v1->y, h);
  }

  RGL_BoundPolyQuad(poly);
 
  if (use_dlights && data.dlights)
  {
    RGL_SplitPolyQuadLOD(poly, 1, 128 >> detail_level);
  }

  RGL_RenderPolyQuad(poly, &data, PlaneCoordFunc, tex_id, 
      mid_masked, blended);
 
  RGL_FreePolyQuad(poly);
}

//
// RGL_WalkThing
//
// Visit a single thing that exists in the current subsector.
//
static void RGL_WalkThing(mobj_t *mo)
{
  drawthing_t *dthing;
  
  float_t tr_x;
  float_t tr_y;

  float_t pos1, pos2;
  float_t tx, tx1, tx2;
  float_t tz;

  float_t gzb, gzt;

  float_t xscale;
  float_t yscale;
  float_t dist_scale;
  int clip_vert = 0;

  const image_t *image;
  int sprite_height;
  int top_offset;

  boolean_t spr_flip;

  // ignore invisible things
  if (mo->visibility == INVISIBLE)
      return;

  // transform the origin point
  tr_x = mo->x - viewx;
  tr_y = mo->y - viewy;

  tz = tr_x * viewcos + tr_y * viewsin;

  // thing is behind view plane?
  if (tz < MINZ)
    return;

  // -ES- 1999/03/14 Use distunit for y and x scale.
  CHECKVAL(tz);
  xscale = x_distunit / tz;
  yscale = y_distunit / tz;
  dist_scale = yscale;

  tx = tr_x * viewsin - tr_y * viewcos;

  // too far off the side?
  // -ES- 1999/03/13 Fixed clipping to work with large FOVs (up to 176 deg)
  // rejects all sprites where angle>176 deg (arctan 32), since those
  // sprites would result in overflow in future calculations
  if (fabs(tx) / 32 > tz)
    return;
  
  image = R2_GetThingSprite(mo, &spr_flip);

  if (!image)
    return;

  // calculate edges of the shape
  pos1 = - IM_OFFSETX(image) * mo->info->xscale;
  pos2 = pos1 + IM_WIDTH(image) * mo->info->xscale;
  
  tx1 = tx + pos1;
  tx2 = tx + pos2;

  xscale *= mo->info->xscale;
  yscale *= mo->info->yscale;

  sprite_height = IM_HEIGHT(image);
  top_offset = IM_OFFSETY(image);

  gzt = mo->z + top_offset * mo->info->yscale;
  gzb = gzt - sprite_height * mo->info->yscale;

  // fix for sprites that sit wrongly into the floor/ceiling

  if (mo->flags & MF_FUZZY)
  {
    clip_vert = -1;
  }
  else if (sprite_kludge==0 && gzb < mo->floorz)
  {
    // explosion ?
    if (mo->info->flags & MF_MISSILE)
    {
      clip_vert = +1;
    }
    else
    {
      gzt += mo->floorz - gzb;
      gzb = mo->floorz;
    }
  }
  else if (sprite_kludge==0 && gzt > mo->ceilingz)
  {
    // explosion ?
    if (mo->info->flags & MF_MISSILE)
    {
      clip_vert = +1;
    }
    else
    {
      gzb -= gzt - mo->ceilingz;
      gzt = mo->ceilingz;
    }
  }

  if (gzb >= gzt)
    return;

  // create new draw thing

  dthing = R2_GetDrawThing();
  R2_CommitDrawThing(1);

  dthing->mo = mo;
  dthing->props = cur_sub->floors->props;
  dthing->clip_vert = clip_vert;
  dthing->clipped_left = dthing->clipped_right = false;

  dthing->image  = image;
  dthing->flip   = spr_flip;
  dthing->bright = mo->bright ? true : false;
  dthing->is_shadow = false;
  dthing->is_halo   = false;
  
  dthing->xscale = xscale;
  dthing->yscale = yscale;
  dthing->iyscale = 1.0 / mo->info->yscale;
  dthing->dist_scale = dist_scale;

  dthing->tx = tx;
  dthing->tz = tz;
  dthing->tx1 = tx1;
  dthing->tx2 = tx2;

  dthing->top = gzt;
  dthing->bottom = gzb;
  dthing->area.y_offset = 0;

  dthing->left_dx  = pos1 *  viewsin;
  dthing->left_dy  = pos1 * -viewcos;
  dthing->right_dx = pos2 *  viewsin;
  dthing->right_dy = pos2 * -viewcos;
  
  // translation support
  if (mo->info->palremap)
    dthing->trans_table = V_GetTranslationTable(mo->info->palremap);
  else
    dthing->trans_table = NULL;

  // create shadow
  if (level_flags.shadows && mo->info->shadow_trans > 0 &&
      mo->floorz < viewz && ! IS_SKY(mo->subsector->sector->floor))
  {
    drawthing_t *dshadow = R2_GetDrawThing();
    R2_CommitDrawThing(1);

    dshadow[0] = dthing[0];

    dshadow->is_shadow = true;
    dshadow->clip_vert = -1;
    dshadow->trans_table = NULL;
    dshadow->tz += 1.5;

    // shadows are 1/4 the height
    dshadow->yscale *= 4.0;
    dshadow->iyscale *= 4.0;

    gzb = mo->floorz;
    gzt = gzb + sprite_height / 4.0 * mo->info->yscale;

    dshadow->top = gzt;
    dshadow->bottom = gzb;

    R2_ClipSpriteVertically(cur_sub, dshadow);
  }

#if 0  // DISABLED
  // create halo
  if (level_flags.halos && mo->info->halo.height > 0)
  {
    drawthing_t *dhalo = R2_GetDrawThing();
    R2_CommitDrawThing(1);

    dhalo[0] = dthing[0];

    dhalo->is_halo = true;
    dhalo->image = W_ImageFromHalo(mo->info->halo.graphic);
    dhalo->clip_vert = -1;
    dhalo->trans_table = NULL;
    dhalo->tz -= 7.5;

    gzb = mo->z + mo->height * 0.75 - mo->info->halo.height / 2;
    gzt = mo->z + mo->height * 0.75 + mo->info->halo.height / 2;

    dhalo->top = gzt;
    dhalo->bottom = gzb;

    pos1 = - mo->info->halo.height / 2;
    pos2 = + mo->info->halo.height / 2;

    dhalo->tx1 = tx + pos1;
    dhalo->tx2 = tx + pos2;

    dhalo->left_dx  = pos1 *  viewsin;  // FIXME: move forward
    dhalo->left_dy  = pos1 * -viewcos;
    dhalo->right_dx = pos2 *  viewsin;
    dhalo->right_dy = pos2 * -viewcos;

    dhalo->iyscale = mo->info->halo.height / IM_HEIGHT(dhalo->image);

    R2_ClipSpriteVertically(cur_sub, dhalo);
  }
#endif

  R2_ClipSpriteVertically(cur_sub, dthing);
}

//
// RGL_DrawThing
//
void RGL_DrawThing(drawfloor_t *dfloor, drawthing_t *dthing)
{
  int w, h;
  float_t right, bottom;

  float_t dx, dy;
  float_t tex_x1, tex_y1;
  float_t tex_x2, tex_y2;

  boolean_t blended;

  local_gl_vert_t *vert, *orig;

  float_t x1b, y1b, z1b, x1t, y1t, z1t;
  float_t x2b, y2b, z2b, x2t, y2t, z2t;

  int L_r, L_g, L_b;

  const image_t *image;
  const cached_image_t *cim;
  GLuint tex_id;

  int lit_Nom = dthing->bright ? 255 : 
      RGL_Light(dthing->props->lightlevel);
  const colourmap_t *colmap = dthing->props->colourmap;
  float_t c_r, c_g, c_b;

  int fuzzy = (dthing->mo->flags & MF_FUZZY);
  float_t trans = fuzzy ? FUZZY_TRANS : dthing->mo->visibility;

  V_GetColmapRGB(colmap, &c_r, &c_g, &c_b, false);

  dx = dy = 0;

  if (dthing->is_shadow)
  {
    L_r = L_g = L_b = 0;

    trans = dthing->mo->visibility * 
        PERCENT_2_FLOAT(dthing->mo->info->shadow_trans);

    dx = viewcos * 2;
    dy = viewsin * 2;
  }
#if 0  // HALO disabled
  else if (dthing->is_halo)
  {
    L_r = L_g = L_b = 255;  //!!

    trans = 0.5;  //!!

    dx = -viewcos * 5;
    dy = -viewsin * 5;
  }
#endif
  else
  {
    L_r = lit_Nom * c_r;
    L_g = lit_Nom * c_g;
    L_b = lit_Nom * c_b;
  }

  image = dthing->image;

  w = IM_WIDTH(image);
  h = IM_HEIGHT(image);
  right = IM_RIGHT(image);
  bottom = IM_BOTTOM(image);

  x1b = x1t = dthing->mo->x + dthing->left_dx;
  y1b = y1t = dthing->mo->y + dthing->left_dy;
  x2b = x2t = dthing->mo->x + dthing->right_dx;
  y2b = y2t = dthing->mo->y + dthing->right_dy;

  z1b = z2b = dthing->bottom;
  z1t = z2t = dthing->top;

  tex_x1 = 0.01;
  tex_x2 = right - 0.01;

  tex_y1 = dthing->area.y_offset * dthing->iyscale;
  tex_y2 = tex_y1 + (z1t - z1b) * dthing->iyscale;

  CHECKVAL(h);
  tex_y1 = 1.00 - tex_y1 / h * bottom;
  tex_y2 = 1.00 - tex_y2 / h * bottom;
 
  if (dthing->flip)
  {
    float_t temp = tex_x2;
    tex_x1 = right - tex_x1;
    tex_x2 = right - temp;
  }

  //
  //  Dynamic Lighting Stuff
  //

  if (use_dlights && !dthing->is_shadow && !dthing->is_halo && !fuzzy)
  {
    drawthing_t *dl;
    float_t wx[4], wy[4], wz[4];

    wx[0] = x1b;  wy[0] = y1b;  wz[0] = z1b;
    wx[1] = x1t;  wy[1] = y1t;  wz[1] = z1t;
    wx[2] = x2t;  wy[2] = y2t;  wz[2] = z2t;
    wx[3] = x2b;  wy[3] = y2b;  wz[3] = z2b;

    for (dl=dfloor->dlights; dl; dl=dl->next)
    {
      if (dl->mo == dthing->mo)
        continue;

      R2_AddColourDLights(1, &L_r, &L_g, &L_b, wx, wy, wz, dl->mo);
    }
  }
 
  //
  // Special FUZZY effect
  //
  if (fuzzy && !dthing->is_shadow && !dthing->is_halo)
  {
    float_t range_x = fabs(dthing->right_dx - dthing->left_dx) / 12.0;
    float_t range_y = fabs(dthing->right_dy - dthing->left_dy) / 12.0;
    float_t range_z = fabs(z1t - z1b) / 24.0 / 2;

    float_t bl = sin(rgl_currtimeval / 5.0);
    float_t tl = sin(rgl_currtimeval / 11.0);
    float_t tr = sin(rgl_currtimeval / 7.0);
    float_t br = sin(rgl_currtimeval / 13.0);

    float_t zbl = 1.0 + cos(rgl_currtimeval / 5.0);
    float_t ztl = 1.0 + cos(rgl_currtimeval / 11.0);
    float_t ztr = 1.0 + cos(rgl_currtimeval / 7.0);
    float_t zbr = 1.0 + cos(rgl_currtimeval / 13.0);
    
    x1b += bl * range_x; y1b += bl * range_y; z1b += zbl * range_z;
    x1t += tl * range_x; y1t += tl * range_y; z1t += ztl * range_z;
    x2t += tr * range_x; y2t += tr * range_y; z2t += ztr * range_z;
    x2b += br * range_x; y2b += br * range_y; z2b += zbr * range_z;

    L_r = L_g = L_b = 0;
  }

  cim = W_ImageCache(image, IMG_OGL, 0, true);

  tex_id = W_ImageGetOGL(cim);

  // Blended sprites, even if opaque (trans > 0.99), have nicer
  // edges.
  blended = true;

  vert = orig = RGL_BeginUnit(GL_QUADS, 4, tex_id, true, blended);

  SET_COLOR(LT_RED(L_r), LT_GRN(L_g), LT_BLU(L_b), trans);
  SET_TEXCOORD(tex_x1, tex_y2);
  SET_NORMAL(-viewcos, -viewsin, 0.0);
  SET_EDGE_FLAG(GL_TRUE);
  SET_VERTEX(x1b+dx, y1b+dy, z1b);
  vert++;
  
  SET_COLOR(LT_RED(L_r), LT_GRN(L_g), LT_BLU(L_b), trans);
  SET_TEXCOORD(tex_x1, tex_y1);
  SET_NORMAL(-viewcos, -viewsin, 0.0);
  SET_EDGE_FLAG(GL_TRUE);
  SET_VERTEX(x1t+dx, y1t+dy, z1t);
  vert++;
  
  SET_COLOR(LT_RED(L_r), LT_GRN(L_g), LT_BLU(L_b), trans);
  SET_TEXCOORD(tex_x2, tex_y1);
  SET_NORMAL(-viewcos, -viewsin, 0.0);
  SET_EDGE_FLAG(GL_TRUE);
  SET_VERTEX(x2t+dx, y2t+dy, z2t);
  vert++;
  
  SET_COLOR(LT_RED(L_r), LT_GRN(L_g), LT_BLU(L_b), trans);
  SET_TEXCOORD(tex_x2, tex_y2);
  SET_NORMAL(-viewcos, -viewsin, 0.0);
  SET_EDGE_FLAG(GL_TRUE);
  SET_VERTEX(x2b+dx, y2b+dy, z2b);
  vert++;

  RGL_EndUnit(vert - orig);

  // Note: normally this would be wrong, since we're using the GL
  // texture ID later on (after W_ImageDone).  The W_LockImagesOGL
  // call saves us though.
  //
  W_ImageDone(cim);
}

//
// RGL_DrawSortThings
//
static drawthing_t **mapdrawthings = NULL;
static int mapdrawthing_num = 0;

static void RGL_DrawSortThings(drawfloor_t *dfloor)
{
  drawthing_t *dthing;

  int i, count=0;

  for (dthing=dfloor->things; dthing; dthing=dthing->next)
    count++;

  if (count == 0)
    return;

  // maybe reallocate array
  if (count > mapdrawthing_num)
  {
    Z_Resize(mapdrawthings, drawthing_t *, count);
    mapdrawthing_num = count;
  }

  // fill array
  count=0;

  for (dthing=dfloor->things; dthing; dthing=dthing->next)
    mapdrawthings[count++] = dthing;

  // sort array into decreasing distance
  #define CMP(a,b)  ((a)->tz > (b)->tz)
  QSORT(drawthing_t *, mapdrawthings, count, CUTOFF);
  #undef CMP

  // draw array
  for (i=0; i < count; i++)
    RGL_DrawThing(dfloor, mapdrawthings[i]);
}

static void LightUpPlayerWeapon(player_t *p, drawfloor_t *dfloor)
{
  drawthing_t *dl;

  float_t x, y, z;

  x = p->mo->x;
  y = p->mo->y;
  z = p->mo->z + p->mo->height *
      PERCENT_2_FLOAT(p->mo->info->shotheight);

  for (dl=dfloor->dlights; dl; dl=dl->next)
  {
    R2_AddColourDLights(1, &rgl_weapon_r, &rgl_weapon_g, &rgl_weapon_b, 
        &x, &y, &z, dl->mo);
  }
}


static INLINE void AddNewDrawFloor(extrafloor_t *ef,
    float_t f_h, float_t c_h, float_t top_h,
    surface_t *floor, surface_t *ceil,
    region_properties_t *props)
{
  drawfloor_t *dfloor;
  drawfloor_t *tail;

  dfloor = R2_GetDrawFloor();
  R2_CommitDrawFloor(1);

  Z_Clear(dfloor, drawfloor_t, 1);

  dfloor->f_h   = f_h;
  dfloor->c_h   = c_h;
  dfloor->top_h = top_h;
  dfloor->floor = floor;
  dfloor->ceil  = ceil;
  dfloor->ef    = ef;
  dfloor->props = props;
  
  // link it in
  // (order is very important)

  if (cur_sub->floors == NULL || f_h > viewz)
  {
    // add to head
    dfloor->next = cur_sub->floors;
    dfloor->prev = NULL;

    if (cur_sub->floors)
      cur_sub->floors->prev = dfloor;

    cur_sub->floors = dfloor;
  }
  else
  {
    // add to tail
    for (tail=cur_sub->floors; tail->next; tail=tail->next)
    { /* nothing here */ }

    dfloor->next = NULL;
    dfloor->prev = tail;

    tail->next = dfloor;
  }

  // add to tail of height order list (for sprite clipping)
  for (tail=cur_sub->z_floors; tail && tail->higher; tail=tail->higher)
  { /* nothing here */ }

  dfloor->higher = NULL;
  dfloor->lower = tail;

  if (tail)
    tail->higher = dfloor;
  else
    cur_sub->z_floors = dfloor;

  if (use_dlights && blocklights)
  {
    R2_FindDLights(cur_sub, dfloor);

    if (cur_sub == viewsubsector && f_h <= viewz && viewz <= c_h)
      LightUpPlayerWeapon(consoleplayer, dfloor);
  }
}
 

//
// RGL_WalkSubsector
//
// Visit a subsector, and collect information, such as where the
// walls, planes (ceilings & floors) and things need to be drawn.
//
static void RGL_WalkSubsector(int num)
{
  subsector_t *sub = &subsectors[num];
  seg_t *seg;
  mobj_t *mo;
  sector_t *sector;
  surface_t *floor_s;
  float_t floor_h;

  extrafloor_t *S, *L, *C;

#if (DEBUG >= 1)
    L_WriteDebug( "\nVISITING SUBSEC %d\n\n", num);
#endif

  cur_sub = sub;
  sector = cur_sub->sector;

  sub->ranges = NULL;
  sub->floors = sub->z_floors = NULL;
  sub->raw_things = NULL;

  // add in each extrafloor, traversing strictly upwards

  floor_s = &sector->floor;
  floor_h = sector->f_h;

  S = sector->bottom_ef;
  L = sector->bottom_liq;

  while (S || L)
  {
    if (!L || (S && S->bottom_h < L->bottom_h))
    {
      C = S;  S = S->higher;
    }
    else
    {
      C = L;  L = L->higher;
    }

    DEV_ASSERT2(C);

    // ignore liquids in the middle of THICK solids, or below real
    // floor or above real ceiling
    //
    if (C->bottom_h < floor_h || C->bottom_h > sector->c_h)
      continue;
    
    AddNewDrawFloor(C, floor_h, C->bottom_h, C->top_h, floor_s, C->bottom, C->p);

    floor_s = C->top;
    floor_h = C->top_h;
  }

  AddNewDrawFloor(NULL, floor_h, sector->c_h, sector->c_h, 
      floor_s, &sector->ceil, sector->p);
 

  // handle each sprite in the subsector.  Must be done before walls,
  // since the wall code will update the 1D occlusion buffer.
  
  for (mo=cur_sub->thinglist; mo; mo=mo->snext)
  {
    RGL_WalkThing(mo);
  }

  // clip 1D occlusion buffer.
  for (seg=sub->segs; seg; seg=seg->sub_next)
  {
    RGL_WalkSeg(seg);
  }

  // add drawsub to list
  // (add to head, thus the eventual order is furthest -> closest)
  
  sub->rend_next = drawsubs_head;
  sub->rend_prev = NULL;
  
  if (drawsubs_head)
    drawsubs_head->rend_prev = sub;
  else
    drawsubs_tail = sub;

  drawsubs_head = sub;
}


//
// RGL_DrawSubsector
//
static void RGL_DrawSubsector(subsector_t *sub)
{
  drawfloor_t *dfloor;
  seg_t *seg;
  int vis_num = 0;

#if (DEBUG >= 1)
    L_WriteDebug("\nREVISITING SUBSEC %d\n\n", (int)(sub - subsectors));
#endif

  cur_sub = sub;

  // handle each seg in the subsector
  for (seg=sub->segs; seg; seg=seg->sub_next)
  {
    if (seg->visible)
    {
      vis_num++;

      if (! seg->miniseg)
        RGL_WalkWall(seg);
    }
  }

  // are any segs visible ?  If not, abort now
  if (vis_num == 0)
    return;

  // handle sky
  if (solid_mode)
  {
    if (IS_SKY(cur_sub->sector->floor) && 
        viewz > cur_sub->sector->f_h)
    {
      RGL_DrawSkyPlane(cur_sub, cur_sub->sector->f_h);
    }

    if (IS_SKY(cur_sub->sector->ceil) && 
        viewz < cur_sub->sector->sky_h)
    {
      RGL_DrawSkyPlane(cur_sub, cur_sub->sector->sky_h);
    }
  }

  // handle each floor, drawing planes and things
  for (dfloor=sub->floors; dfloor; dfloor=dfloor->next)
  {
    RGL_DrawPlane(dfloor, dfloor->c_h, dfloor->ceil,  -1);
    RGL_DrawPlane(dfloor, dfloor->f_h, dfloor->floor, +1);

    if (solid_mode)
      continue;

    RGL_DrawSortThings(dfloor);
  }
}

//
// RGL_WalkBSPNode
//
// Walks all subsectors below a given node, traversing subtree
// recursively, collecting information.  Just call with BSP root.
//
static void RGL_WalkBSPNode(int bspnum)
{
  node_t *node;
  int side;

  // Found a subsector?
  if (bspnum & NF_SUBSECTOR)
  {
    RGL_WalkSubsector(bspnum & (~NF_SUBSECTOR));
    return;
  }

  node = &nodes[bspnum];

  // Decide which side the view point is on.
  side = P_PointOnDivlineSide(viewx, viewy, &node->div);

#if (DEBUG >= 2)
    L_WriteDebug( "NODE %d (%1.1f, %1.1f) -> (%1.1f, %1.1f)  SIDE %d\n",
        bspnum, node->div.x, node->div.y, node->div.x +
        node->div.dx, node->div.y + node->div.dy, side);
#endif

  // Recursively divide front space.
  if (RGL_CheckBBox(node->bbox[side]))
    RGL_WalkBSPNode(node->children[side]);

  // Recursively divide back space.
  if (RGL_CheckBBox(node->bbox[side ^ 1]))
    RGL_WalkBSPNode(node->children[side ^ 1]);
}

//
// RGL_RenderTrueBSP
//
// OpenGL BSP rendering.  Initialises all structures, then walks the
// BSP tree collecting information, then renders each subsector:
// firstly front to back (drawing all solid walls & planes) and then
// from back to front (drawing everything else, sprites etc..).
//
void RGL_RenderTrueBSP(void)
{
  subsector_t *cur;

  rgl_currtimeval = I_GetTime();

  // compute the 1D projection of the view angle
  {
    float_t k, d;

    // k is just the mlook angle (in radians)
    k = fabs(atan(viewvertangle));

    if (k >= M_PI/4 || normalfov > ANG90)
      oned_side_angle = ANG90;
    else
    {
      // d is just the distance horizontally forward from the eye to
      // the top/bottom edge of the view rectangle.
      d = M_ROOT2 * sin(M_PI/2 - (k + M_PI/4));

      oned_side_angle = ANG90 - M_ATan(d);
    }
  }

  // setup clip angles
  {
    leftclipangle  = 0 + oned_side_angle;
    rightclipangle = 0 - oned_side_angle;
    clipscope = leftclipangle - rightclipangle;
  }

  // clear extra light on player's weapon
  rgl_weapon_r = rgl_weapon_g = rgl_weapon_b = 0;

  R2_ClearBSP();
  RGL_1DOcclusionClear(0 - oned_side_angle, 0 + oned_side_angle);

  drawsubs_head = drawsubs_tail = NULL;

  // walk the bsp tree
  RGL_WalkBSPNode(root_node);

  // handle powerup effects
  RGL_RainbowEffect(consoleplayer);

//---  // NOTE: this call will be unneeded later.
//---  RGL_SetupMatrices2D();
//---  RGL_DrawSky();

  if (hom_detect)
  {
    glClearColor(0.0, 1.0, 0.0, 0.0);
    glClear(GL_COLOR_BUFFER_BIT);
  }

  RGL_SetupMatrices3D();

  // enable depth buffer, and clear it
  glEnable(GL_DEPTH_TEST);
  glClear(GL_DEPTH_BUFFER_BIT);

  // make sure we don't delete any GL textures
  W_LockImagesOGL();

  // draw all solid walls and planes
  solid_mode = true;
  RGL_StartUnits(solid_mode);

  for (cur=drawsubs_tail; cur; cur=cur->rend_prev)
    RGL_DrawSubsector(cur);
    
  RGL_FinishUnits();
  
  // draw all sprites and masked/translucent walls/planes
  solid_mode = false;
  RGL_StartUnits(solid_mode);
  
  for (cur=drawsubs_head; cur; cur=cur->rend_next)
    RGL_DrawSubsector(cur);

  RGL_FinishUnits();
  glDisable(GL_DEPTH_TEST);

  W_UnlockImagesOGL();

  // now draw 2D stuff like psprites, and add effects
  RGL_SetupMatrices2D();

  RGL_DrawPlayerSprites(consoleplayer);

  RGL_ColourmapEffect(consoleplayer);

  // NOTE: this call will move for layer system (should be topmost).
  RGL_PaletteEffect(consoleplayer);
 
#if (DEBUG >= 3) 
    L_WriteDebug( "\n\n");
#endif
}


void RGL_RenderScene(int x, int y, int w, int h, vid_view_t *view)
{
  //!!!
  RGL_RenderTrueBSP();
}


#endif  // USE_GL
