//----------------------------------------------------------------------------
//  EDGE Screen system
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------

#include "i_defs.h"
#include "v_screen.h"

#include "dm_state.h"
#include "z_zone.h"

/*

   I have split up the creation/destruction routines, so that there is one
   routine that handles the content of the struct, and another one that also
   handles the memory managment of the struct itself.

   The Empty/Init routines should be used if you already have allocated memory
   for the screen_t struct (this is the case with viewbitmaps).
   The Create/Destroy routines should be used otherwise, they allocate the
   memory for the screen_ts before initialising.

 */

long screen_id_count = 1;

//
// Destroys the contents of a screen.
//
void V_EmptyScreen(screen_t * scr)
{
  if (scr->children)
    I_Error("V_EmptyScreen: Screen has %d subscreens!", scr->children);

  if (!scr->parent)
  {
    if (scr->data)
      Z_Free(scr->data);
  }
  else
    scr->parent->children--;
}

//
// Destroys a screen & frees it.
//
void V_DestroyScreen(screen_t * scr)
{
  V_EmptyScreen(scr);
  Z_Free(scr);
}

//
// Inits a screen for the given dimensions.
//
void V_InitScreen(screen_t * s, int width, int height, int bytepp)
{
  s->width = width;
  s->height = height;
  s->bytepp = bytepp;
  
  // align
  s->pitch = V_GetPitch(width, bytepp);
  
  // allocate data
  s->data = Z_New(byte, s->pitch * height);
  s->parent = NULL;
  s->children = 0;
  s->x = s->y = 0;
  s->id = screen_id_count++;

  //L_WriteDebug("V_InitScreen: %d %d %d %d\n", s->width, s->height, s->bytepp, s->id);
}

//
// Creates a new screen.
//
screen_t *V_CreateScreen(int width, int height, int bytepp)
{
  screen_t *s;

  s = Z_New(screen_t, 1);

  V_InitScreen(s, width, height, bytepp);

  return s;
}

//
// Allocates a screen. Must be initialised with V_InitScreen or V_InitSubScreen.
//
screen_t *V_CreateEmptyScreen(void)
{
  screen_t *s;
  s = Z_ClearNew(screen_t, 1);
  return s;
}

//
// Inits a sub-screen for the given dimensions.
//
void V_InitSubScreen(screen_t * s, screen_t * p, int x, int y, int width, int height)
{
  s->width = width;
  s->height = height;
  s->parent = p;
  s->children = 0;
  p->children++;
  s->pitch = p->pitch;
  s->bytepp = p->bytepp;
  s->x = x;
  s->y = y;

  if (x < 0 || y < 0)
    I_Error("V_InitSubScreen: Negative upper-left coordinates!");
  
  if (x + width > p->width)
    I_Error("V_InitSubScreen: Subscreen wider than parent!");

  if (y + height > p->height)
    I_Error("V_InitSubScreen: Subscreen higher than parent!");

  s->data = p->data + p->pitch * y + p->bytepp * x;
  s->id = screen_id_count++;

}

//
// Creates a new sub-screen.
//
screen_t *V_CreateSubScreen(screen_t * p, int x, int y, int width, int height)
{
  screen_t *s;
  s = Z_New(screen_t, 1);

  V_InitSubScreen(s, p, x, y, width, height);
  return s;
}

screen_t *V_ResizeScreen(screen_t * s, int newwidth, int newheight, int newbytepp)
{
  if (!s)
    return V_CreateScreen(newwidth, newheight, newbytepp);

  if (s->width == newwidth && s->height == newheight && s->bytepp == newbytepp)
    return s;

  if (s->parent)
  {
    if (newwidth + s->x > s->parent->width)
      I_Error("V_ResizeScreen: Subscreen wider than parent!");
    s->width = newwidth;
    if (newheight + s->y > s->parent->height)
      I_Error("V_ResizeScreen: Subscreen higher than parent!");
    s->height = newheight;
    if (newbytepp != s->parent->bytepp)
      I_Error("V_ResizeScreen: BPP differs from parent!");
  }
  else
  {
    if (s->children)
      // can't resize a screen which has subscreens.
      I_Error("V_ResizeScreen: Screen has %d subscreens!", s->children);

    s->pitch = V_GetPitch(newwidth, newbytepp);
    s->width = newwidth;
    s->height = newheight;
    s->bytepp = newbytepp;
    Z_Resize(s->data, byte, newheight * s->pitch);
  }
  s->id = screen_id_count++;

  return s;
}

//
// Moves the sub-screen to the given coordinates relative to the parent
//
void V_MoveSubScreen(screen_t * s, int x, int y)
{
  if (!s->parent)
    I_Error("V_MoveSubScreen: Screen is not a subscreen!");

  // can't move a screen which has subscreens.
  if (s->children)
    I_Error("V_MoveSubScreen: Screen has %d subscreens!", s->children);

  s->x = x;
  s->y = y;
  
  if (s->width + x > s->parent->width || x < 0 || s->height + y > s->parent->height || y < 0)
    I_Error("V_MoveSubScreen: Can't move screen outside parent's boundaries!");
  
  s->data = s->parent->data + y * s->pitch + x * s->bytepp;

  //L_WriteDebug("V_MoveSubScreen: old ID:%d\n", s->id);
  s->id = screen_id_count++;
  //L_WriteDebug("V_MoveSubScreen: new ID:%d\n", s->id);

}

//
// V_GetPitch
//
// Returns the pitch that should be used for the given width/bytepp combination
//
int V_GetPitch(int width, int bytepp)
{
  // -ES- 1999/07/18 Of cache reasons, pitch is set to the next
  // 32-multiple after width*bytepp that not is divisible by 64.
  return ((width * bytepp + 31) & ~63) + 32;
}

//
// V_ScreenIsSubScreen
//
// returns whether parts s1 and s2 share the same memory.
boolean_t V_ScreensIntersect(screen_t * s1, screen_t * s2)
{
  int x1, y1, x2, y2;
  screen_t *s, *p;

  x1 = y1 = x2 = y2 = 0;
  for (s = s1; s->parent; s = s->parent)
  {
    x1 += s->x;
    y1 += s->y;
  }
  p = s;
  for (s = s2; s->parent; s = s->parent)
  {
    x2 += s->x;
    y2 += s->y;
  }
  if (s == p)
  {
    // s1 and s2 have the same origin screen, so they can intersect.
    // Just do a simple bounding box collision detection.
    if ((x1 + s1->width < x2) || (x2 + s2->width < x1) ||
        (y1 + s1->height < y2) || (y2 + s2->height < y1))
      return false;

    return true;
  }

  return false;
}

//
// Checks whether scr has the same dimensions as the current resolution
// (ie SCREENWIDTH*SCREENHEIGHT), and thereby can use routines like
// DrawPatchInDirect.
int V_ScreenHasCurrentRes(screen_t *scr)
{
  if (scr->width == SCREENWIDTH &&
      scr->height == SCREENHEIGHT &&
      scr->pitch == SCREENPITCH &&
      scr->bytepp == BPP)
  {
    return true;
  }
  else
  {
    return false;
  }
}
