//----------------------------------------------------------------------------
//  EDGE Top-level Video System
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------

#include "i_defs.h"
#include "v_toplev.h"

#include "e_search.h"
#include "dm_state.h"
#include "dm_defs.h"
#include "m_argv.h"
#include "m_bbox.h"
#include "m_swap.h"
#include "p_local.h"
#include "r_layers.h"
#include "r_local.h"
#include "r_sky.h"
#include "rgl_defs.h"
#include "v_colour.h"
#include "v_ctx.h"
#include "w_image.h"
#include "w_wad.h"
#include "z_zone.h"


#ifdef USE_LAYERS


static boolean_t begun_draw = false;

static int clipper_x1 = 0;
static int clipper_y1 = 0;
static int clipper_x2 = 0;
static int clipper_y2 = 0;
static int cur_solid_index;

//
// VTOP_NewScreenSize
//
void VTOP_NewScreenSize(int width, int height, int bpp)
{
  DEV_ASSERT2(! begun_draw);

  vctx.NewScreenSize(width, height, bpp);
}

//
// VTOP_RenderScene
//
void VTOP_RenderScene(int x1, int y1, int x2, int y2, vid_view_t *view)
{
  DEV_ASSERT2(! begun_draw);

  vctx.RenderScene(x1, y1, x2, y2, view);
}

//
// VTOP_BeginDraw
//
// Coordinates are the clipping rectangle.  The extra parameter
// `solid_index' is the index for this layer into the solid rectangle
// list (needed for clipping).
//
void VTOP_BeginDraw(int x1, int y1, int x2, int y2, int solid_index)
{
  DEV_ASSERT2(! begun_draw);
  
  DEV_ASSERT2(x2 >= x1);
  DEV_ASSERT2(y2 >= y1);
  DEV_ASSERT2(solid_index >= 0);

  // keep a copy of clip rectangle for us to use...
  clipper_x1 = x1;
  clipper_y1 = y1;
  clipper_x2 = x2;
  clipper_y2 = y2;
  cur_solid_index = solid_index;

  vctx.BeginDraw(x1, y1, x2, y2);

  begun_draw = true;
}

//
// VTOP_EndDraw
//
void VTOP_EndDraw(void)
{
  DEV_ASSERT2(begun_draw);

  vctx.EndDraw();

  begun_draw = false;
}

//
// VTOP_DrawImage
//
// Draw an image.  If `is_vol' is true, then it is assumed that this
// image forms part of a volatile layer and thus doesn't need to be
// clipped to the dirty matrix.
//
void VTOP_DrawImage(int x1, int y1, int x2, int y2,
    float_t tx1, float_t ty1, float_t tx2, float_t ty2,
    const image_t *image, const colourmap_t *colmap, 
    int alpha, boolean_t is_vol)
{
  float_t tdx, tdy;
  
  DEV_ASSERT2(x2 >= x1);
  DEV_ASSERT2(y2 >= y1);

  if (alpha <= 1)
    return;

  tdx = (tx2 - tx1) / (x2 - x1 + 1);
  tdy = (ty2 - ty1) / (y2 - y1 + 1);

  if (x1 < clipper_x1)
  {
    tx1 += tdx * (clipper_x1 - x1);
    x1 = clipper_x1;
  }
  
  if (y1 < clipper_y1)
  {
    ty1 += tdy * (clipper_y1 - y1);
    y1 = clipper_y1;
  }

  if (x2 > clipper_x2)
  {
    tx2 += tdx * (clipper_x2 - x2);
    x2 = clipper_x2;
  }

  if (y2 > clipper_y2)
  {
    ty2 += tdy * (clipper_y2 - y2);
    y2 = clipper_y2;
  }

  // nothing left to draw ?
  if (x1 > x2 || y1 > y2)
    return;

  if (is_vol || dirty_region_whole)
  {
    vctx.DrawImage(x1,y1,x2,y2, image, tx1,ty1,tx2,ty2, colmap, alpha);
    return;
  }

  // FIXME: implement dirty matrix & solid rect clips
  I_Error("non-volatile VTOP_SolidBox not yet implemented !\n");
}
 
//
// VTOP_SolidBox
//
// Draw a solid box.  If `is_vol' is true, then it is assumed that
// this image forms part of a volatile layer and thus doesn't need to
// be clipped to the dirty matrix.
//
void VTOP_SolidBox(int x1, int y1, int x2, int y2,
    int colour, int alpha, boolean_t is_vol)
{
  DEV_ASSERT2(x2 >= x1);
  DEV_ASSERT2(y2 >= y1);

  if (alpha <= 1)
    return;

  if (x1 < clipper_x1)
    x1 = clipper_x1;

  if (y1 < clipper_y1)
    y1 = clipper_y1;

  if (x2 > clipper_x2)
    x2 = clipper_x2;

  if (y2 > clipper_y2)
    y2 = clipper_y2;

  // nothing left to draw ?
  if (x1 > x2 || y1 > y2)
    return;

  if (is_vol || dirty_region_whole)
  {
    vctx.SolidBox(x1, y1, x2, y2, colour, alpha);
    return;
  }

  // FIXME: implement dirty matrix & solid rect clips
  I_Error("non-volatile VTOP_SolidBox not yet implemented !\n");
}

//
// VTOP_SolidLine
//
// Note: assumed to be volatile.

void VTOP_SolidLine(int x1, int y1, int x2, int y2,
    int colour, int alpha)
{
  if (alpha <= 1)
    return;

  // we let the video context handle clipping...
  vctx.SolidLine(x1, y1, x2, y2, colour);
}

//
// VTOP_ReadScreen
//
// This one takes care to allocate the buffer.  It must be freed using
// Z_Free().
//
byte *VTOP_ReadScreen(int x1, int y1, int x2, int y2)
{
  byte *rgb_buffer;
  
  DEV_ASSERT2(x2 >= x1);
  DEV_ASSERT2(y2 >= y1);

  rgb_buffer = Z_Malloc((x2 - x1 + 1) * (y2 - y1 + 1) * 3);

  vctx.ReadScreen(x1, y1, x2, y2, rgb_buffer);

  return rgb_buffer;
}


//
//  ANALYSIS UTILITY ROUTINES
//

static byte analysis_region[DIRT_REG_H][DIRT_REG_W];
static boolean_t analysis_begun = false;

static int analysis_x1, analysis_y1;
static int analysis_x2, analysis_y2;
static int analysis_solid;

//
// VTOP_AnalyseBegin
//
// Begin analysis of the dirty matrix / solid rectangles for the layer
// with the given coordinates and solid index.  Coordinates are
// inclusive.  Note: Only one analysis can be in progress at any one
// time.
//
void VTOP_AnalyseBegin(int x1, int y1, int x2, int y2, int solid_index)
{
  int i, x, y;
  
  DEV_ASSERT2(! analysis_begun);

  DEV_ASSERT2(2048 >= x2 && x2 >= x1 && x1 >= 0);
  DEV_ASSERT2(1536 >= y2 && y2 >= y1 && y1 >= 0);
  
  analysis_x1 = x1;
  analysis_y1 = y1;
  analysis_x2 = x1;
  analysis_y2 = y1;

  // setup initial buffer, using solid rectangles

  x1 /= DIRT_X; y1 /= DIRT_Y;
  x2 /= DIRT_X; y2 /= DIRT_Y;

  for (y=y1; y <= y2; y++)
  for (x=x1; x <= x2; x++)
  {
    int px = x * DIRT_X;
    int py = y * DIRT_Y;

    // is this block clean ?
    if (! dirty_region_whole && ! dirty_region[y][x])
    {
      analysis_region[y][x] = 0;
      continue;
    }

    // is this block totally under a solid rect ?
    for (i=solid_index; i >= 0; i--)
    {
      int sx1 = solid_rects[i]->x1;
      int sy1 = solid_rects[i]->y1;
      int sx2 = solid_rects[i]->x2;
      int sy2 = solid_rects[i]->y2;

      if (sx1 <= px && px <= sx2 && sy1 <= py && py <= sy2)
        break;
    }

    analysis_region[y][x] = (i >= 0) ? 0 : 1;
  }

  analysis_begun = true;
}

//
// VTOP_AnalyseEnd
//
// Finish analysis, freeing any resources used.
//
void VTOP_AnalyseEnd(void)
{
  DEV_ASSERT2(analysis_begun);

  analysis_begun = false;
}

boolean_t VTOP_AnalyseNextRect(int *x1, int *y1, int *x2, int *y2)
{
  DEV_ASSERT2(analysis_begun);

  // FIXME: this could be optimised

  return false;
}

#endif  // USE_LAYERS
