//----------------------------------------------------------------------------
//  EDGE Generalised Image Handling
//----------------------------------------------------------------------------
// 
//  Copyright (c) 1999-2001  The EDGE Team.
// 
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version 2
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//----------------------------------------------------------------------------
//
//  Based on the DOOM source code, released by Id Software under the
//  following copyright:
//
//    Copyright (C) 1993-1996 by id Software, Inc.
//
//----------------------------------------------------------------------------
//
// -AJA- 2000/06/25: Began this image generalisation, based on Erik
//       Sandberg's w_textur.c/h code.
//
// TODO HERE:
//   +  faster search methods.
//   -  do some optimisation
//

#include "i_defs.h"
#include "w_image.h"

#include "e_search.h"
#include "dm_state.h"
#include "dm_defs.h"
#include "m_argv.h"
#include "m_misc.h"
#include "m_swap.h"
#include "p_local.h"
#include "r_local.h"
#include "r_sky.h"
#include "rgl_defs.h"
#include "v_colour.h"
#include "w_textur.h"
#include "w_wad.h"
#include "z_zone.h"


// range of mip values is 0 to 11, allowing images upto 2048x2048
#define MAX_MIP  11


#ifndef USE_GL
typedef unsigned int  GLuint;
#endif

struct real_image_s;


//
// This structure is for "cached" images (i.e. ready to be used for
// rendering), and is the non-opaque version of cached_image_t.  A
// single structure is used for all three image modes (Post, Block and
// OGL).
//
// Note: multiple modes and/or multiple mips of the same image_t can
// indeed be present in the cache list at any one time.
//
typedef struct real_cached_image_s
{
  // link in cache list
  struct real_cached_image_s *next, *prev;

  // number of current users
  int users;

  // true if image has been invalidated -- unload a.s.a.p
  boolean_t invalidated;
 
  // parent image
  struct real_image_s *parent;
  
  // mip value (>= 0)
  unsigned short mip;

  // current image mode
  image_mode_e mode;

  union
  {
    // case IMG_Post:
    struct
    {
      // array of columns.  Size is MIP_SIZE(total_w, mip).  Totally
      // empty columns can be NULL.
      w_post_t ** columns;
    }
    post;

    // case IMG_Block:
    struct
    {
      // pixel block.  MIP_SIZE(total_h,mip) * MIP_SIZE(total_w,mip)
      // is the size.  Pixel order is down columns first (i.e. not the
      // usual layout).  Transparent pixels are TRANS_PIXEL.
      byte *pixels;
    }
    block;

    // case IMG_OGL:
    struct
    {
      // texture identifier within GL
      GLuint tex_id;
    }
    ogl;
  }
  info;

  // total memory size taken up by this image.  Includes this
  // structure.
  int size;

  // NOTE: Post/Block data may follow this structure...
}
real_cached_image_t;


typedef enum
{
  // Source was a patch name
  IMSRC_Patch = 0,

  // Source was a font patch name
  IMSRC_Font = 1,

  // Source was a halo patch name
  IMSRC_Halo = 2,

  // Source was a sprite name
  IMSRC_Sprite = 3,

  // Source was a flat name
  IMSRC_Flat = 4,

  // Source was a raw block of 320x200 bytes (Heretic/Hexen)
  IMSRC_Raw320x200 = 5,

  // Source was a texture name
  IMSRC_Texture = 6,

  // Source is dummy image
  IMSRC_Dummy = 7,

  // Source is from IMAGE.DDF
  IMSRC_User = 8
}
image_source_e;

//
// This structure is the full version of image_t.  It contains all the
// information needed to create the actual cached images when needed.
//
typedef struct real_image_s
{
  // base is the publicly visible structure
  image_t pub;

  // --- information about where this image came from ---

  char name[10];

  image_source_e source_type;
 
  union
  {
    // case IMSRC_Patch:
    // case IMSRC_Font:
    // case IMSRC_Halo:
    // case IMSRC_Sprite:
    struct { int lump; } patch;

    // case IMSRC_Flat:
    // case IMSRC_Raw320x200:
    struct { int lump; } flat;

    // case IMSRC_Texture:
    struct { texturedef_t *tdef; } texture;

    // case IMSRC_Dummy:
    struct { byte fg, bg; } dummy;

    // case IMSRC_User:
    struct { void /* image_def_t */ * def; } user;
  }
  source;

  // palette lump, or -1 to use the "GLOBAL" palette
  int source_palette;

  // --- information about caching ---

  struct
  {
    int num_mips;
    real_cached_image_t ** mips;
  }
  block_cache;

  // no mipmapping for posters, there is no improvement
  real_cached_image_t * post_cache;

  // no mipmapping here, GL does this itself
  real_cached_image_t * ogl_cache;

  // --- animation info ---

  struct
  {
    // current version of this image in the animation.  Initially points
    // to self.  For non-animated images, doesn't change.  Otherwise
    // when the animation flips over, it becomes cur->next.
    struct real_image_s *cur;

    // next image in the animation, or NULL.
    struct real_image_s *next;

    // tics before next anim change, or 0 if non-animated.
    unsigned short count;

    // animation speed (in tics), or 0 if non-animated.
    unsigned short speed;
  }
  anim;
}
real_image_t;


// mipmapping enabled ?
// 0 off, 1 bilinear, 2 trilinear
int use_mipmapping = 1;

boolean_t use_smoothing = true;
boolean_t use_dithering = false;

static boolean_t w_locked_ogl = false;

// total set of images
static stack_array_t real_images_a;
static real_image_t ** real_images = NULL;
static int num_real_images = 0;

#define RIM_DUMMY_TEX    real_images[0]
#define RIM_DUMMY_FLAT   real_images[1]
#define RIM_SKY_FLAT     real_images[2]
#define RIM_DUMMY_PATCH  real_images[3]
#define RIM_DUMMY_SPRITE real_images[4]
#define RIM_DUMMY_FONT   real_images[5]
#define RIM_DUMMY_HALO   real_images[6]

const struct image_s *skyflatimage;


// image cache (actually a ring)
static real_cached_image_t imagecachehead;


// tiny ring helpers
static INLINE void InsertAtTail(real_cached_image_t *rc)
{
  DEV_ASSERT2(rc != &imagecachehead);

  rc->prev =  imagecachehead.prev;
  rc->next = &imagecachehead;

  rc->prev->next = rc;
  rc->next->prev = rc;
}
static INLINE void Unlink(real_cached_image_t *rc)
{
  DEV_ASSERT2(rc != &imagecachehead);

  rc->prev->next = rc->next;
  rc->next->prev = rc->prev;
}

// the number of bytes of the texture cache that currently can be
// freed.  Useful when we decide how much to flush.  This changes when
// the number of users of a block changes from 1 to 0 or back.
static int cache_size = 0;


// Dummy image, for when texture/flat/patch is unknown.  Row major
// order.  Could be packed, but why bother ?
#define DUMMY_X  16
#define DUMMY_Y  16
static byte dummy_graphic[DUMMY_X * DUMMY_Y] =
{
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,1,1,1,1,1,1,1,1,0,0,0,0,
  0,0,0,1,1,1,1,1,1,1,1,1,1,0,0,0,
  0,0,1,1,1,1,0,0,0,0,0,1,1,1,0,0,
  0,0,0,1,1,0,0,0,0,0,0,1,1,1,0,0,
  0,0,0,0,0,0,0,0,0,0,1,1,1,0,0,0,
  0,0,0,0,0,0,0,0,0,1,1,1,0,0,0,0,
  0,0,0,0,0,0,0,0,1,1,1,0,0,0,0,0,
  0,0,0,0,0,0,0,1,1,1,0,0,0,0,0,0,
  0,0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,
  0,0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,
  0,0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};


//----------------------------------------------------------------------------

//
//  IMAGE CREATION
//

int W_MakeValidSize(int value)
{
  DEV_ASSERT2(value > 0);

  if (value <=    1) return    1;
  if (value <=    2) return    2;
  if (value <=    4) return    4;
  if (value <=    8) return    8;
  if (value <=   16) return   16;
  if (value <=   32) return   32;
  if (value <=   64) return   64;
  if (value <=  128) return  128;
  if (value <=  256) return  256;
  if (value <=  512) return  512;
  if (value <= 1024) return 1024;
  if (value <= 2048) return 2048;

  I_Error("Texture size (%d) too large !\n", value);
  return -1;
}

static real_image_t *NewImage(int width, int height, boolean_t solid)
{
  real_image_t *rim;
  short scale = 0x0100;
  
  Z_SetArraySize(&real_images_a, ++num_real_images);
  rim = real_images[num_real_images-1];

  // clear newbie
  Z_Clear(rim, real_image_t, 1);

  rim->pub.actual_w = width;
  rim->pub.actual_h = height;
  rim->pub.total_w  = W_MakeValidSize(width);
  rim->pub.total_h  = W_MakeValidSize(height);
  rim->pub.offset_x = rim->pub.offset_y = 0;
  rim->pub.scale_x  = rim->pub.scale_y = scale;
  rim->pub.solid    = solid;

  // set initial animation info
  rim->anim.cur = rim;
  rim->anim.next = NULL;
  rim->anim.count = rim->anim.speed = 0;

  return rim;
}

static real_image_t *AddImageDummy(image_source_e realsrc)
{
  real_image_t *rim;
  
  rim = NewImage(DUMMY_X, DUMMY_Y, 
      (realsrc != IMSRC_Patch) && (realsrc != IMSRC_Font) &&
      (realsrc != IMSRC_Halo) && (realsrc != IMSRC_Sprite));
 
  DEV_ASSERT2(num_real_images <= 10);
  sprintf(rim->name, "DUMMY__%d", num_real_images-1);

  rim->source_type = IMSRC_Dummy;
  rim->source_palette = -1;

  switch (realsrc)
  {
    case IMSRC_Texture:
      rim->source.dummy.fg = pal_black;
      rim->source.dummy.bg = pal_brown1;
      break;

    case IMSRC_Flat:
    case IMSRC_Raw320x200:
      rim->source.dummy.fg = pal_black;
      rim->source.dummy.bg = pal_green1;
      break;

    case IMSRC_Patch:
    case IMSRC_Sprite:
      rim->source.dummy.fg = pal_yellow;
      rim->source.dummy.bg = TRANS_PIXEL;
      break;
    
    case IMSRC_Font:
    case IMSRC_Halo:
      rim->source.dummy.fg = pal_white;
      rim->source.dummy.bg = TRANS_PIXEL;
      break;
    
    default:
      I_Error("AddImageDummy: bad realsrc value %d !\n", realsrc);
  }

  return rim;
}

static real_image_t *AddImagePatch(const char *name, 
    image_source_e type, int lump)
{
  patch_t *pat;
  int width, height, offset_x, offset_y;
  
  real_image_t *rim;

  pat = (patch_t *) W_CacheLumpNum(lump);
  
  // FIXME: do a validity check on patch ??

  width  = SHORT(pat->width);
  height = SHORT(pat->height);
  offset_x = SHORT(pat->leftoffset);
  offset_y = SHORT(pat->topoffset);
  
  W_DoneWithLump(pat);

  // check for Heretic/Hexen images, which are raw 320x200 
  if (W_LumpLength(lump) == 64000 &&
      (width <= 0 || height <= 0 || (width * height) > 80000))
  {
    if (type != IMSRC_Patch)
      return NULL;

    rim = NewImage(320, 200, true);
    strcpy(rim->name, name);

    rim->source_type = IMSRC_Raw320x200;
    rim->source.flat.lump = lump;
    rim->source_palette = W_GetPaletteForLump(lump);
    return rim;
  }
 
  // create new image
  rim = NewImage(width, height, false);
 
  rim->pub.offset_x = offset_x;
  rim->pub.offset_y = offset_y;

  strcpy(rim->name, name);

  rim->source_type = type;
  rim->source.patch.lump = lump;
  rim->source_palette = W_GetPaletteForLump(lump);

  return rim;
}

static real_image_t *AddImageTexture(const char *name, 
    texturedef_t *tdef)
{
  real_image_t *rim;
 
  // assume it is non-solid, we'll update it when we know for sure
  rim = NewImage(tdef->width, tdef->height, false);
 
  strcpy(rim->name, name);

  rim->source_type = IMSRC_Texture;
  rim->source.texture.tdef = tdef;
  rim->source_palette = tdef->palette_lump;

  return rim;
}

static real_image_t *AddImageFlat(const char *name, int lump)
{
  real_image_t *rim;
  int len, size;
  
  len = W_LumpLength(lump);
  
  switch (len)
  {
    case 64 * 64: size = 64; break;

    // support for odd-size Hexen flats
    case 64 * 128: size = 64; break;
  
    // -- EDGE feature: bigger than normal flats --
  
    case 128 * 128: size = 128; break;
    case 256 * 256: size = 256; break;
    case 512 * 512: size = 512; break;
    case 1024 * 1024: size = 1024; break;
    
    default:
      return NULL;
  }
   
  rim = NewImage(size, size, true);
 
  strcpy(rim->name, name);

  rim->source_type = IMSRC_Flat;
  rim->source.flat.lump = lump;
  rim->source_palette = W_GetPaletteForLump(lump);

  return rim;
}

//
// W_ImageCreateFlats
//
// Used to fill in the image array with flats from the WAD.  The set
// of lumps is those that occurred between F_START and F_END in each
// existing wad file, with duplicates set to -1.
//
// NOTE: should only be called once, as it assumes none of the flats
// in the list have names colliding with existing flat images.
// 
void W_ImageCreateFlats(int *lumps, int number)
{
  int i;

  DEV_ASSERT2(lumps);

  for (i=0; i < number; i++)
  {
    if (lumps[i] < 0)
      continue;
    
    AddImageFlat(W_GetLumpName(lumps[i]), lumps[i]);
  }
}

//
// W_ImageCreateTextures
//
// Used to fill in the image array with textures from the WAD.  The
// list of texture definitions comes from each TEXTURE1/2 lump in each
// existing wad file, with duplicates set to NULL.
//
// NOTE: should only be called once, as it assumes none of the
// textures in the list have names colliding with existing texture
// images.
// 
void W_ImageCreateTextures(struct texturedef_s ** defs, int number)
{
  int i;

  DEV_ASSERT2(defs);

  for (i=0; i < number; i++)
  {
    if (defs[i] == NULL)
      continue;
    
    AddImageTexture(defs[i]->name, defs[i]);
  }
}

//
// W_ImageCreateSprite
// 
// Used to fill in the image array with sprites from the WAD.  The
// lumps come from those occurring between S_START and S_END markers
// in each existing wad.
//
// NOTE: it is assumed that each new sprite is unique i.e. the name
// does not collide with any existing sprite image.
// 
const image_t *W_ImageCreateSprite(int lump)
{
  const real_image_t *rim;

  DEV_ASSERT2(lump >= 0);

  rim = AddImagePatch(W_GetLumpName(lump), IMSRC_Sprite, lump);

  if (!rim)
    rim = RIM_DUMMY_SPRITE;

  return &rim->pub;
}


//----------------------------------------------------------------------------

//
//  UTILITY
//

#define PIXEL_RED(pix)  (what_palette[pix*3 + 0])
#define PIXEL_GRN(pix)  (what_palette[pix*3 + 1])
#define PIXEL_BLU(pix)  (what_palette[pix*3 + 2])

#define GAMMA_RED(pix)  (gammatable[usegamma][PIXEL_RED(pix)])
#define GAMMA_GRN(pix)  (gammatable[usegamma][PIXEL_GRN(pix)])
#define GAMMA_BLU(pix)  (gammatable[usegamma][PIXEL_BLU(pix)])

//
// ShrinkBlock
//
// Take a block of pixels, and compute a shrunk down version of it
// (for mipmapping), returning the new block of pixels.
//
static real_cached_image_t *ShrinkBlock(real_cached_image_t *src, int mip,
    const byte *what_palette)
{
  real_cached_image_t *dest;
  
  byte *pixels, *src_pixels;
  
  int x, y;
  int dx, dy;
  int step_x, step_y;

  int total_w = MIP_SIZE(src->parent->pub.total_w, src->mip);
  int total_h = MIP_SIZE(src->parent->pub.total_h, src->mip);

  int new_w = MIP_SIZE(src->parent->pub.total_w, mip);
  int new_h = MIP_SIZE(src->parent->pub.total_h, mip);

  int size = new_w * new_h;
  
  DEV_ASSERT2(mip > src->mip);
  DEV_ASSERT2(new_w > 0);
  DEV_ASSERT2(new_h > 0);
  DEV_ASSERT2(size > 0);
  DEV_ASSERT2(src->mode == IMG_Block);

  size += sizeof(real_cached_image_t);

  dest = (real_cached_image_t *) Z_Malloc(size);

  dest->next = dest->prev = NULL;
  dest->parent = src->parent;
  dest->mip = mip;
  dest->users = 0;
  dest->invalidated = false;
  dest->mode = IMG_Block;
  dest->info.block.pixels = (byte *)(dest + 1);
  dest->size = size;

  DEV_ASSERT2((total_w % new_w) == 0);
  DEV_ASSERT2((total_h % new_h) == 0);

  step_x = total_w / new_w;
  step_y = total_h / new_h;

  DEV_ASSERT2(step_x > 1 || step_y > 1);

  pixels = dest->info.block.pixels;
  src_pixels = src->info.block.pixels;
  
  // remember, blocks are in column-major order
  for (x=0; x < new_w; x++)
  for (y=0; y < new_h; y++)
  {
    int px = x * step_x;
    int py = y * step_y;

    int tot_r=0, tot_g=0, tot_b=0, a_count=0;
    int total = step_x * step_y;

    // compute average colour of block
    for (dx=0; dx < step_x; dx++)
    for (dy=0; dy < step_y; dy++)
    {
      byte pix = src_pixels[(px+dx) * total_h + (py+dy)];

      if (pix == TRANS_PIXEL)
        a_count++;
      else
      {
        tot_r += PIXEL_RED(pix);
        tot_g += PIXEL_GRN(pix);
        tot_b += PIXEL_BLU(pix);
      }
    }

    if (a_count > total/2) 
      pixels[x * new_h + y] = TRANS_PIXEL;
    else
    {
      total -= a_count;

      tot_r = tot_r / total * 32 / 256;
      tot_g = tot_g / total * 32 / 256;
      tot_b = tot_b / total * 32 / 256;
      
      pixels[x * new_h + y] = rgb_32k[tot_r][tot_b][tot_g];
    }
  }

  return dest;
}

//
// EnlargeBlock
//
// Take a block of pixels, and compute a larger version of it
// (anti-mipmapping), returning the new block of pixels.
//
static real_cached_image_t *EnlargeBlock(real_cached_image_t *src, int mip,
    const byte *what_palette)
{
  real_cached_image_t *dest;
  
  byte *pixels, *src_pixels;
  
  int x, y;
  int dx, dy;
  int step_x, step_y;

  int total_w = MIP_SIZE(src->parent->pub.total_w, src->mip);
  int total_h = MIP_SIZE(src->parent->pub.total_h, src->mip);

  int new_w = MIP_SIZE(src->parent->pub.total_w, mip);
  int new_h = MIP_SIZE(src->parent->pub.total_h, mip);

  int size = new_w * new_h;
 
  DEV_ASSERT2(mip < src->mip);
  DEV_ASSERT2(new_w > 0);
  DEV_ASSERT2(new_h > 0);
  DEV_ASSERT2(size > 0);
  DEV_ASSERT2(src->mode == IMG_Block);

  size += sizeof(real_cached_image_t);

  dest = (real_cached_image_t *) Z_Malloc(size);

  dest->next = dest->prev = NULL;
  dest->parent = src->parent;
  dest->mip = mip;
  dest->users = 0;
  dest->invalidated = false;
  dest->mode = IMG_Block;
  dest->info.block.pixels = (byte *)(dest + 1);
  dest->size = size;

  DEV_ASSERT2((new_w % total_w) == 0);
  DEV_ASSERT2((new_h % total_h) == 0);

  step_x = new_w / total_w;
  step_y = new_h / total_h;

  DEV_ASSERT2(step_x > 1 || step_y > 1);

  pixels = dest->info.block.pixels;
  src_pixels = src->info.block.pixels;
  
  // remember, blocks are in column-major order
  for (x=0; x < new_w; x++)
  for (y=0; y < new_h; y++)
  {
    int tot_r=0, tot_g=0, tot_b=0, a_count=0;
    int total = step_x * step_y;

    // compute average colour of sub-pixel block
    for (dx=0; dx < step_x; dx++)
    for (dy=0; dy < step_y; dy++)
    {
      int px = MIN(x+dx, new_w-1) / step_x;
      int py = MIN(y+dy, new_h-1) / step_y;

      byte pix = src_pixels[px * total_h + py];

      if (pix == TRANS_PIXEL)
        a_count++;
      else
      {
        tot_r += PIXEL_RED(pix);
        tot_g += PIXEL_GRN(pix);
        tot_b += PIXEL_BLU(pix);
      }
    }

    if (a_count > 0)
      pixels[x * new_h + y] = TRANS_PIXEL;
    else
    {
      total -= a_count;

      tot_r = tot_r / total * 32 / 256;
      tot_g = tot_g / total * 32 / 256;
      tot_b = tot_b / total * 32 / 256;
      
      pixels[x * new_h + y] = rgb_32k[tot_r][tot_b][tot_g];
    }
  }

  return dest;
}

//
// DrawColumnIntoBlock
//
// Clip and draw an old-style column from a patch into a blockified
// image.  Can also do a font conversion (make white).
//
static void DrawColumnIntoBlock(real_cached_image_t *rc, 
    const column_t *patchcol, int x, int y, int font)
{
  int w1, h1, w2, h2;
  
  DEV_ASSERT2(patchcol);
  DEV_ASSERT2(rc->mode == IMG_Block);
  DEV_ASSERT2(rc->mip  == 0);

  w1 = MIP_SIZE(rc->parent->pub.actual_w, rc->mip);
  h1 = MIP_SIZE(rc->parent->pub.actual_h, rc->mip);
  w2 = MIP_SIZE(rc->parent->pub.total_w,  rc->mip);
  h2 = MIP_SIZE(rc->parent->pub.total_h,  rc->mip);

  // clip horizontally
  if (x < 0 || x >= w1)
    return;

  while (patchcol->topdelta != P_SENTINEL)
  {
    int top = y + (int) patchcol->topdelta;
    int count = patchcol->length;

    byte *src = (byte *) patchcol + 3;
    byte *dest = rc->info.block.pixels + (x * h2);

    if (top < 0)
    {
      count += top;
      top = 0;
    }

    if (top + count > h1)
      count = h1 - top;

    // copy the pixels, remapping any TRANS_PIXEL values
    for (dest += top; count > 0; count--, src++, dest++)
    {
      if (*src == TRANS_PIXEL)
        *dest = pal_black;
#ifndef USE_GL
      else if (font == 2)
        *dest = halo_conv_table[*src];
#endif
      else if (font)
        *dest = font_whitener[*src];
      else
        *dest = *src;
    }

    patchcol = (const column_t *) ((const byte *) patchcol + 
        patchcol->length + 4);
  }
}

#if 0  // NOT USED.  NOTE: code has rotted a bit
//
// DrawWPostIntoBlock
//
// Clip and draw a new-style w_post from a postified image into a
// blockified image. 
//
static void DrawWPostIntoBlock(real_cached_image_t *rc, 
    const w_post_t *post, int x, int y)
{
  DEV_ASSERT2(post);
  DEV_ASSERT2(rc->mode == IMG_Block);
  DEV_ASSERT2(rc->mip  == 0);

  // clip horizontally
  if (x < 0 || x >= rc->parent->pub.actual_w)
    return;

  while (post->skip != P_SENTINEL)
  {
    int top = y + post->skip;
    int count = post->length;

    byte *src = (byte *) post + 3;
    byte *dest = rc->info.block.pixels + (x * rc->parent->pub.total_h);

    if (top < 0)
    {
      count += top;
      top = 0;
    }

    if (top + count > rc->parent->pub.actual_h)
      count = rc->parent->pub.actual_h - top;

    // copy the pixels (if any).  No remapping needed.
    if (count > 0)
      Z_MoveData(dest+top, src, byte, count);

    y += post->skip + post->length;

    post = (const w_post_t *) ((const byte *) post + post->length + 4);
  }
}
#endif

//
// CheckBlockSolid
//
// FIXME: Avoid future checks.
//
#define MAX_STRAY_PIXELS  2

static void CheckBlockSolid(real_cached_image_t *rc)
{
  int x, y;
  int total_num;
  int stray_count=0;

  int w1, h1, w2, h2;

  byte *src;

  DEV_ASSERT2(rc->mode == IMG_Block);
  DEV_ASSERT2(rc->mip  == 0);

  w1 = MIP_SIZE(rc->parent->pub.actual_w, rc->mip);
  h1 = MIP_SIZE(rc->parent->pub.actual_h, rc->mip);
  w2 = MIP_SIZE(rc->parent->pub.total_w,  rc->mip);
  h2 = MIP_SIZE(rc->parent->pub.total_h,  rc->mip);

  src = rc->info.block.pixels;

  total_num = w1 * h1;

  for (x=0; x < w1; x++)
  for (y=0; y < h1; y++)
  {
    byte src_pix = src[x * h2 + y];

    if (src_pix != TRANS_PIXEL)
      continue;

    stray_count++;

    // only ignore stray pixels on large images
    if (total_num < 256 || stray_count > MAX_STRAY_PIXELS)
      return;
  }

  // image is totally solid.  Blacken any transparent parts.
  rc->parent->pub.solid = true;

  for (x=0; x < w2; x++)
  for (y=0; y < h2; y++)
  {
    if (x >= w1 || y >= h1 || src[x * h2 + y] == TRANS_PIXEL)
    {
      src[x * h2 + y] = pal_black;
    }
  }
}

//
// ConvertBlockToPost
//
// Converts a blockified image to a postified one.  Returns a new
// cached image.  New image has same parent and mip as source, but
// users field will be 0 (bump up if used).
//
#define MAX_POST_H  240

#define ADD_BYTE(value)  \
    do {  \
      if (buf_pos == buf_size)  \
      {  \
        buf_size += 1024;  \
        Z_Resize(buf_start, byte, buf_size);  \
      }  \
      buf_start[buf_pos++] = (value);  \
    } while(0)
  
static real_cached_image_t *ConvertBlockToPost(real_cached_image_t *rc)
{
  int i, x, y;
  int len, skip, size;
  int *offsets; 

  byte *buf_start, *dest;
  int buf_size = 1024;
  int buf_pos = 0;
  
  int w1, h1, w2, h2;

  real_cached_image_t *result;
  
  DEV_ASSERT2(rc);
  DEV_ASSERT2(rc->mode == IMG_Block);

  w1 = MIP_SIZE(rc->parent->pub.actual_w, rc->mip);
  h1 = MIP_SIZE(rc->parent->pub.actual_h, rc->mip);
  w2 = MIP_SIZE(rc->parent->pub.total_w, rc->mip);
  h2 = MIP_SIZE(rc->parent->pub.total_h, rc->mip);

  // allocate initial post & offset buffers
  buf_start = Z_New(byte, buf_size);
  offsets = Z_New(int, w2);

  // handle columns outside of actual width: for transparent images,
  // mark the columns as empty, otherwise for solid images add a
  // custom post at offset 0 that contains all black.

  if (w1 < w2)
  {
    if (rc->parent->pub.solid)
    {
      for (y=0; y < h1; )
      {
        len = MIN(h1 - y, MAX_POST_H);

        DEV_ASSERT2(len > 0);

        ADD_BYTE(0);     // skip
        ADD_BYTE(len);   // length
        
        for (i=0; i < len+2; i++)
        {
          ADD_BYTE(pal_black);  // pixels and two pad bytes
        }
      }

      // add end-of-column sentinel
      ADD_BYTE(P_SENTINEL);

      for (x=w1; x < w2; x++)
        offsets[x] = 0;
    }
    else
    {
      for (x=w1; x < w2; x++)
        offsets[x] = -1;
    }
  }

  // convert the blockified columns into posts
  for (x=0; x < w1; x++)
  {
    byte *src = &rc->info.block.pixels[x * h2];
    
    // completely transparent ?
    for (len=0; len < h1 && src[len] == TRANS_PIXEL; len++)
    { /* nothing here */ }
    
    if (len == h1)
    {
      offsets[x] = -1;
      continue;
    }

    // postify the column
    offsets[x] = buf_pos;
    
    for (y=0; y < h1; )
    { 
      // determine how many transparent pixels to skip
      for (skip=0; (y + skip < h1) && 
           src[y + skip] == TRANS_PIXEL; skip++)
      { /* nothing here */ }
       
      y += skip;
      DEV_ASSERT2(y <= h1);

      // rest of column is totally transparent ?
      if (y == h1)
        break;

      // if necessary, add empty posts to get around limit
      for (; skip > MAX_POST_H; skip -= MAX_POST_H)
      {
        ADD_BYTE(MAX_POST_H);  // skip
        ADD_BYTE(0);           // length
        ADD_BYTE(pal_black);   // pad1
        ADD_BYTE(pal_black);   // pad2
      }

      // determine number of non-transparent pixels
      for (len=0; (y + len < h1) && (len < MAX_POST_H) &&
           src[y + len] != TRANS_PIXEL; len++)
      { /* nothing here */ }

      DEV_ASSERT2(len > 0);
      DEV_ASSERT2(len <= MAX_POST_H);

      ADD_BYTE(skip);     // skip
      ADD_BYTE(len);      // length
      ADD_BYTE(src[y]);   // pad1

      // copy pixels in post
      for (; len > 0; len--, y++)
      {
        ADD_BYTE(src[y]);
      }
      
      ADD_BYTE(src[y-1]);  // pad2
    }

    DEV_ASSERT2(buf_pos > offsets[x]);

    // add end-of-column sentinel
    ADD_BYTE(P_SENTINEL);
  }

  // Phew... Now create and fill out the result image

  size = sizeof(real_cached_image_t) + buf_pos + w2 * sizeof(w_post_t*);

  result = (real_cached_image_t *) Z_Malloc(size);

  result->next = result->prev = NULL;
  result->parent = rc->parent;
  result->mip = rc->mip;
  result->users = 0;
  result->invalidated = false;
  result->mode = IMG_Post;
  result->info.post.columns = (w_post_t **)(result + 1);
  result->size = size;

  dest = (byte *) (&result->info.post.columns[w2]);

  if (buf_pos > 0)
    Z_MoveData(dest, buf_start, byte, buf_pos);

  for (x=0; x < w2; x++)
    result->info.post.columns[x] =
        (offsets[x] < 0) ? NULL : (w_post_t *)(dest + offsets[x]);

  // free stuff
  Z_Free(buf_start);
  Z_Free(offsets);
 
  return result;
}

#undef ADD_BYTE


//----------------------------------------------------------------------------

//
//  GL UTILITIES
//

#ifdef USE_GL

//
// ShrinkBlockRGBA
//
// Just like ShrinkBlock() above, but the returned format is RGBA.
// Source format is column-major (i.e. normal block), whereas result
// is row-major (ano note that GL textures are _bottom up_ rather than
// the usual top-down ordering).  The new size should be scaled down
// to fit into glmax_tex_size.
//
static byte *ShrinkBlockRGBA(byte *src, int total_w, int total_h,
    int new_w, int new_h, const byte *what_palette)
{
  byte *dest;
 
  int x, y, dx, dy;
  int step_x, step_y;

  DEV_ASSERT2(new_w > 0);
  DEV_ASSERT2(new_h > 0);
  DEV_ASSERT2(new_w <= glmax_tex_size);
  DEV_ASSERT2(new_h <= glmax_tex_size);
  DEV_ASSERT2((total_w % new_w) == 0);
  DEV_ASSERT2((total_h % new_h) == 0);

  dest = (byte *) Z_New(byte, new_w * new_h * 4);

  step_x = total_w / new_w;
  step_y = total_h / new_h;

  // faster method for the usual case (no shrinkage)

  if (step_x == 1 && step_y == 1)
  {
    for (y=0; y < total_h; y++)
    for (x=0; x < total_w; x++)
    {
      byte src_pix = src[x * total_h + y];
      byte *dest_pix = dest + (((total_h-1-y) * total_w + x) * 4);

      if (src_pix == TRANS_PIXEL)
      {
        dest_pix[0] = dest_pix[1] = dest_pix[2] = dest_pix[3] = 0;
      }
      else
      {
        dest_pix[0] = GAMMA_RED(src_pix);
        dest_pix[1] = GAMMA_GRN(src_pix);
        dest_pix[2] = GAMMA_BLU(src_pix);
        dest_pix[3] = 255;
      }
    }
    return dest;
  }

  // slower method, as we must shrink the bugger...

  for (y=0; y < new_h; y++)
  for (x=0; x < new_w; x++)
  {
    byte *dest_pix = dest + (((new_h-1-y) * new_w + x) * 4);

    int px = x * step_x;
    int py = y * step_y;

    int tot_r=0, tot_g=0, tot_b=0, a_count=0, alpha;
    int total = step_x * step_y;

    // compute average colour of block
    for (dx=0; dx < step_x; dx++)
    for (dy=0; dy < step_y; dy++)
    {
      byte src_pix = src[(px+dx) * total_h + (py+dy)];

      if (src_pix == TRANS_PIXEL)
        a_count++;
      else
      {
        tot_r += GAMMA_RED(src_pix);
        tot_g += GAMMA_GRN(src_pix);
        tot_b += GAMMA_BLU(src_pix);
      }
    }

    if (a_count >= total)
    {
      // some pixels were translucent.  Keep r/g/b as zero.
      alpha = 0;
    }
    else
    {
      alpha = (total - a_count) * 255 / total;

      total -= a_count;

      tot_r /= total;
      tot_g /= total;
      tot_b /= total;
    }

    dest_pix[0] = tot_r;
    dest_pix[1] = tot_g;
    dest_pix[2] = tot_b;
    dest_pix[3] = alpha;
  }

  return dest;
}

//
// ConvertHaloGL
//
// Convert the RGBA image (which should have come from an IMSRC_Halo
// patch, which gets passed through font_whitener[]) into a halo image
// (with smooth alpha gradients).
//
static void ConvertHaloGL(byte *src, int width, int height)
{
  int x, y;

  DEV_ASSERT2(width  > 0);
  DEV_ASSERT2(height > 0);

  for (y=0; y < height; y++)
  for (x=0; x < width;  x++)
  {
    byte *pix = src + ((y * width + x) * 4);

#if 1
    pix[3] = pix[1] * pix[3] / 255;
#else
    int dx = x - width/2;
    int dy = y - height/2;

    float_t len = sqrt(width*width/4.0 + height*height/4.0);
    float_t dist = sqrt(dx*dx + dy*dy);

    pix[3] = MAX(0, (int)(255 * (1.0 - 2.0 * dist / len)));
#endif

    // halos are all white, only the alpha varies
    pix[0] = pix[1] = pix[2] = 255;
    pix[2] = 0;
  }
}

//
// W_SendGLTexture
//
// Send the texture data to the GL, and returns the texture ID
// assigned to it.  The format of the data must be a normal block if
// palette-indexed pixels.
//
static GLuint minif_modes[2*3] =
{
  GL_NEAREST,
  GL_NEAREST_MIPMAP_NEAREST,
  GL_NEAREST_MIPMAP_LINEAR,
  
  GL_LINEAR,
  GL_LINEAR_MIPMAP_NEAREST,
  GL_LINEAR_MIPMAP_LINEAR
};

GLuint W_SendGLTexture(byte *src, int total_w, int total_h,
    boolean_t clamp, boolean_t halo, boolean_t nomip, 
    const byte *what_palette)
{
  GLuint id;
  
  byte *rgba_src;
  int new_w, new_h;
  int mip;

  // scale down, if necessary, to fix the maximum size
  for (new_w = total_w; new_w > glmax_tex_size; new_w /= 2)
  { /* nothing here */ }

  for (new_h = total_h; new_h > glmax_tex_size; new_h /= 2)
  { /* nothing here */ }

  glEnable(GL_TEXTURE_2D);

  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

  glGenTextures(1, &id);
  glBindTexture(GL_TEXTURE_2D, id);

  if (clamp)
  {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
  }
  else
  {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  }

  // magnification mode
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
      use_smoothing ? GL_LINEAR : GL_NEAREST);

  // minification mode
  use_mipmapping = MIN(2, MAX(0, use_mipmapping));

  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, 
      minif_modes[(use_smoothing ? 3 : 0) +
      (nomip ? 0 : use_mipmapping)]);

  for (mip=0; ; mip++)
  {
    rgba_src = ShrinkBlockRGBA(src, total_w, total_h, new_w, new_h, 
        what_palette);
    
    if (halo)
      ConvertHaloGL(rgba_src, new_w, new_h);

    glTexImage2D(GL_TEXTURE_2D, mip, 4, new_w, new_h,
                 0, GL_RGBA, GL_UNSIGNED_BYTE, rgba_src);
    
    Z_Free(rgba_src);

    // stop if mipmapping disabled or we have reached the end
    if (nomip || !use_mipmapping || (new_w == 1 && new_h == 1))
      break;
    
    new_w = MAX(1, new_w / 2);
    new_h = MAX(1, new_h / 2);
  }

  glDisable(GL_TEXTURE_2D);

  return id;
}

#endif  // USE_GL


//----------------------------------------------------------------------------

//
//  BLOCK READING STUFF
//

//
// ReadFlatAsBlock
//
// Loads a flat from the wad and returns the image block for it.
// Doesn't do any mipmapping (this is too "raw" if you follow).
//
static real_cached_image_t *ReadFlatAsBlock(real_image_t *rim)
{
  real_cached_image_t *rc;

  const byte *src;
  byte *dest;

  int x, y, w, h, tw, th, size;

  DEV_ASSERT2(rim->source_type == IMSRC_Flat ||
              rim->source_type == IMSRC_Raw320x200);

  tw = MAX(rim->pub.total_w, 1);
  th = MAX(rim->pub.total_h, 1);

  w = rim->pub.actual_w;
  h = rim->pub.actual_h;

  size = tw * th;
  size += sizeof(real_cached_image_t);
  
  rc = (real_cached_image_t *) Z_Malloc(size);
  dest = (byte *)(rc + 1);

  // clear initial image to black
  memset(dest, pal_black, tw * th);
   
  rc->next = rc->prev = NULL;
  rc->parent = rim;
  rc->mip = 0;
  rc->users = 0;
  rc->invalidated = false;
  rc->mode = IMG_Block;
  rc->info.block.pixels = dest;
  rc->size = size;

  // read in pixels
  src = W_CacheLumpNum(rim->source.flat.lump);

  for (y=0; y < h; y++)
  for (x=0; x < w; x++)
  {
    byte src_pix = src[y * w + x];

    // remember, blocks are in column-major format
    byte *dest_pix = &dest[x * th + y];

    // make sure TRANS_PIXEL values (which do not occur naturally in
    // Doom images) are properly remapped.
    if (src_pix == TRANS_PIXEL)
      dest_pix[0] = pal_black;
    else
      dest_pix[0] = src_pix;
  }

  W_DoneWithLump(src);

  return rc;
}

//
// ReadTextureAsBlock
//
// Loads a texture from the wad and returns the image block for it.
// Doesn't do any mipmapping (this is too "raw" if you follow).  This
// routine will also update the `solid' flag if texture turns out to
// be solid.
//
static real_cached_image_t *ReadTextureAsBlock(real_image_t *rim)
{
  real_cached_image_t *rc;

  texturedef_t *tdef;
  texpatch_t *patch;
  const patch_t *realpatch;
  const column_t *patchcol;

  byte *dest;

  int i, x, x1, x2, y1;
  int pix_size, size;

  DEV_ASSERT2(rim->source_type == IMSRC_Texture);

  tdef = rim->source.texture.tdef;
  DEV_ASSERT2(tdef);

  pix_size = rim->pub.total_w * rim->pub.total_h;
  size = pix_size + sizeof(real_cached_image_t);
  
  rc = (real_cached_image_t *) Z_Malloc(size);
  dest = (byte *)(rc + 1);

  rc->next = rc->prev = NULL;
  rc->parent = rim;
  rc->mip = 0;
  rc->users = 0;
  rc->invalidated = false;
  rc->mode = IMG_Block;
  rc->info.block.pixels = dest;
  rc->size = size;

  // Clear initial pixels to either totally transparent, or totally
  // black (if we know the image should be solid).  If the image turns
  // out to be solid instead of transparent, the transparent pixels
  // will be blackened.
  
  if (rc->parent->pub.solid)
    memset(dest, pal_black, pix_size);
  else
    memset(dest, TRANS_PIXEL, pix_size);

  // Composite the columns into the block.
  for (i=0, patch=tdef->patches; i < tdef->patchcount; i++, patch++)
  {
    realpatch = W_CacheLumpNum(patch->patch);

    x1 = patch->originx;
    y1 = patch->originy;
    x2 = x1 + SHORT(realpatch->width);

    x  = MAX(0, x1);
    x2 = MIN(tdef->width, x2);

    for (; x < x2; x++)
    {
      patchcol = (const column_t *) ((const byte *) realpatch +
          LONG(realpatch->columnofs[x - x1]));

      DrawColumnIntoBlock(rc, patchcol, x, y1, false);
    }

    W_DoneWithLump(realpatch);
  }

  // update solid flag, if needed
  if (! rc->parent->pub.solid)
    CheckBlockSolid(rc);

  return rc;
}

//
// ReadPatchAsBlock
//
// Loads a patch from the wad and returns the image block for it.
// Very similiar to ReadTextureAsBlock() above.  Doesn't do any
// mipmapping (this is too "raw" if you follow).  This routine will
// also update the `solid' flag if it turns out to be 100% solid.
//
static real_cached_image_t *ReadPatchAsBlock(real_image_t *rim,
    int font)
{
  real_cached_image_t *rc;

  const patch_t *realpatch;
  const column_t *patchcol;

  byte *dest;

  int x;
  int pix_size, size;

  DEV_ASSERT2(rim->source_type == IMSRC_Patch ||
      rim->source_type == IMSRC_Font ||
      rim->source_type == IMSRC_Halo ||
      rim->source_type == IMSRC_Sprite);

  pix_size = rim->pub.total_w * rim->pub.total_h;
  size = pix_size + sizeof(real_cached_image_t);
  
  rc = (real_cached_image_t *) Z_Malloc(size);
  dest = (byte *)(rc + 1);

  rc->next = rc->prev = NULL;
  rc->parent = rim;
  rc->mip = 0;
  rc->users = 0;
  rc->invalidated = false;
  rc->mode = IMG_Block;
  rc->info.block.pixels = dest;
  rc->size = size;

  // Clear initial pixels to either totally transparent, or totally
  // black (if we know the image should be solid).  If the image turns
  // out to be solid instead of transparent, the transparent pixels
  // will be blackened.
  
  if (rc->parent->pub.solid)
    memset(dest, pal_black, pix_size);
  else
    memset(dest, TRANS_PIXEL, pix_size);

  // Composite the columns into the block.
  realpatch = W_CacheLumpNum(rim->source.patch.lump);

  DEV_ASSERT2(rim->pub.actual_w == SHORT(realpatch->width));
  DEV_ASSERT2(rim->pub.actual_h == SHORT(realpatch->height));
  
  for (x=0; x < rim->pub.actual_w; x++)
  {
    patchcol = (const column_t *) ((const byte *) realpatch +
        LONG(realpatch->columnofs[x]));

    DrawColumnIntoBlock(rc, patchcol, x, 0, font);
  }

  W_DoneWithLump(realpatch);

  // update solid flag, if needed
  if (! rc->parent->pub.solid)
    CheckBlockSolid(rc);

  return rc;
}

//
// ReadDummyAsBlock
//
// Creates a dummy image.
//
static real_cached_image_t *ReadDummyAsBlock(real_image_t *rim)
{
  real_cached_image_t *rc;
  byte *dest;

  int x, y;
  int size;

  DEV_ASSERT2(rim->source_type == IMSRC_Dummy);
  DEV_ASSERT2(rim->pub.actual_w == rim->pub.total_w);
  DEV_ASSERT2(rim->pub.actual_h == rim->pub.total_h);
  DEV_ASSERT2(rim->pub.total_w == DUMMY_X);
  DEV_ASSERT2(rim->pub.total_h == DUMMY_Y);

  size = rim->pub.total_w * rim->pub.total_h;
  size += sizeof(real_cached_image_t);
  
  rc = (real_cached_image_t *) Z_Malloc(size);
  dest = (byte *)(rc + 1);

  rc->next = rc->prev = NULL;
  rc->parent = rim;
  rc->mip = 0;
  rc->users = 0;
  rc->invalidated = false;
  rc->mode = IMG_Block;
  rc->info.block.pixels = dest;
  rc->size = size;

  // copy pixels
  for (y=0; y < DUMMY_Y; y++)
  for (x=0; x < DUMMY_X; x++)
  {
    byte src_pix = dummy_graphic[y * DUMMY_X + x];

    // remember, blocks are in column-major format
    byte *dest_pix = dest + (x * rim->pub.total_h + y);

    *dest_pix = src_pix ? rim->source.dummy.fg : rim->source.dummy.bg;
  }

  return rc;
}

//
// ReadAsBlock
//
// Read the image from the wad, applying the mip value to shrink it
// appropriately, and returns the cached image (blockified).  Never
// returns NULL.  Mainly just a switch to more specialised image
// readers.
//
static real_cached_image_t *ReadAsBlock(real_image_t *rim, int mip)
{
  real_cached_image_t *rc, *rc2;
  const byte *what_palette;

  switch (rim->source_type)
  {
    case IMSRC_Flat:
    case IMSRC_Raw320x200:
      rc = ReadFlatAsBlock(rim);
      break;

    case IMSRC_Texture:
      rc = ReadTextureAsBlock(rim);
      break;

    case IMSRC_Patch:
    case IMSRC_Sprite:
      rc = ReadPatchAsBlock(rim, 0);
      break;

    case IMSRC_Font:
      rc = ReadPatchAsBlock(rim, 1);
      break;

    case IMSRC_Halo:
      rc = ReadPatchAsBlock(rim, 2);
      break;

    case IMSRC_Dummy:
      rc = ReadDummyAsBlock(rim);
      break;
    
    case IMSRC_User:
      I_Error("IMSRC_User: not yet implemented.\n");
      rc = NULL;  // keep GCC happy
      break;
      
    default:
      I_Error("ReadAsBlock: unknown source_type %d !\n", rim->source_type);
      return NULL;
  }

  if (mip == rc->mip)
    return rc;

  what_palette = (const byte *) &playpal_data[0];

  if (mip < rc->mip)
    rc2 = EnlargeBlock(rc, mip, what_palette);
  else
    rc2 = ShrinkBlock(rc, mip, what_palette);
 
  // free the old one.  Note: there's nothing in it that needs special
  // handling.
  Z_Free(rc);

  return rc2;
}


//----------------------------------------------------------------------------

//
//  POST READING STUFF
//

#define O_TOPDELTA  W_SKIP
#define O_LENGTH    W_LEN

static void TranslateColumn(real_cached_image_t *rc,
    w_post_t *pos, byte *processed, int remaining, int font)
{
  int i, y;

  DEV_ASSERT2(remaining > 0);

  for (y=0; pos[O_TOPDELTA] != P_SENTINEL; )
  {
    int top   = pos[O_TOPDELTA];
    int count = pos[O_LENGTH];

    if (processed[0])
      return;
 
    // convert topdelta to w_post_t skip value
    pos[W_SKIP] = MAX(0, top - y);
    pos[W_LEN]  = count;
    pos[W_PAD1] = pos[W_DATA];
    pos[W_DATA+count] = pos[W_DATA+count-1];

    *processed++ = 1;
    *processed++ = 1;
 
    // translate post pixels
    for (i=0; i < count+2; i++)
    {
      // make sure TRANS_PIXEL values are properly remapped, since
      // they don't have any special meaning in DOOM images.
      if (pos[W_PAD1+i] == TRANS_PIXEL)
        pos[W_PAD1+i] = pal_black;
#ifndef USE_GL
      else if (font == 2)
        pos[W_PAD1+i] = halo_conv_table[pos[W_PAD1+i]];
#endif
      else if (font)
        pos[W_PAD1+i] = font_whitener[pos[W_PAD1+i]];
      else
        pos[W_PAD1+i] = pos[W_PAD1+i];

      *processed++ = 1;
    }

    y = top + count;

    pos += count + W_TOTAL;
    remaining -= count + W_TOTAL;

    // size check
    DEV_ASSERT2(remaining > 0);
  }

  *processed = 1;
}

static real_cached_image_t *ReadPatchAsPost(real_image_t *rim, 
    int font)
{
  real_cached_image_t *rc;
  const patch_t *realpatch;
  
  byte *start;
  byte *processed;

  int x, size, post_size, head_size;

  DEV_ASSERT2(rim->source_type == IMSRC_Patch ||
      rim->source_type == IMSRC_Font ||
      rim->source_type == IMSRC_Halo ||
      rim->source_type == IMSRC_Sprite);

  // compute size of the post data itself (nothing else)
  head_size = 8 + rim->pub.actual_w * 4;
  post_size = W_LumpLength(rim->source.patch.lump) - head_size;

  // minimum size is 1 byte
  DEV_ASSERT2(post_size >= 1);

  size = post_size + rim->pub.total_w * sizeof(w_post_t *);
  size += sizeof(real_cached_image_t);
  
  rc = (real_cached_image_t *) Z_Malloc(size);

  rc->next = rc->prev = NULL;
  rc->parent = rim;
  rc->mip = 0;
  rc->users = 0;
  rc->invalidated = false;
  rc->mode = IMG_Post;
  rc->info.post.columns = (w_post_t **)(rc + 1);
  rc->size = size;

  start  = (byte *) (&rc->info.post.columns[rim->pub.total_w]);

  realpatch = W_CacheLumpNum(rim->source.patch.lump);

  // copy the raw columns
  memcpy(start, (byte *)realpatch + head_size, post_size);

  // allocate a map of processed bytes within the patch.  Since some
  // patches contain multiple references to the same column (a kind of
  // compression), the map prevents translating the same column twice.
  processed = Z_ClearNew(byte, post_size);

  for (x=0; x < rim->pub.actual_w; x++)
  {
    int offset = LONG(realpatch->columnofs[x]) - head_size;

    DEV_ASSERT2(0 <= offset && offset < post_size);

    rc->info.post.columns[x] = (w_post_t *)(start + offset);
    
    TranslateColumn(rc, rc->info.post.columns[x], processed + offset,
        post_size - offset, font);
  }
 
  W_DoneWithLump(realpatch);

  Z_Free(processed);

  // clear remaining columns (if any)
  for (x=rim->pub.actual_w+1; x < rim->pub.total_w; x++)
    rc->info.post.columns[x] = NULL;

  return rc;
}

//
// ReadAsPost
//
// Read the image from the wad, applying the mip value to shrink it
// appropriately, and returns the cached image (postified).  Never
// returns NULL.
//
static real_cached_image_t *ReadAsPost(real_image_t *rim, int mip)
{
  real_cached_image_t *rc, *rc2;

  // for patches at mip #0, read the posts in the lump directly
  if (mip == 0 &&
      (rim->source_type == IMSRC_Patch ||
       rim->source_type == IMSRC_Font ||
       rim->source_type == IMSRC_Halo ||
       rim->source_type == IMSRC_Sprite))
  {
    return ReadPatchAsPost(rim, (rim->source_type == IMSRC_Font) ? 1 :
        (rim->source_type == IMSRC_Halo) ? 2 : 0);
  }

  // Handle everything else by reading as a block and converting. 
  // OPTIMISE: For single-patch textures, read posts directly.

  rc = ReadAsBlock(rim, mip);

  rc2 = ConvertBlockToPost(rc);

  // free the old one.  Note: there's nothing in it that needs special
  // handling.
  Z_Free(rc);

  return rc2;
}


//----------------------------------------------------------------------------

//
//  IMAGE LOADING / UNLOADING
//

static INLINE
real_cached_image_t *LoadImagePost(real_image_t *rim)
{
  // OPTIMISE: check if a blockified version at mip #0 already exists,
  // saving us the trouble to read from the WAD.
 
  real_cached_image_t *rc;
  
  rc = ReadAsPost(rim, 0);

  DEV_ASSERT2(rc->mode == IMG_Post);

  rc->users++;
  InsertAtTail(rc);

  return rc;
}

static INLINE
real_cached_image_t *LoadImageBlock(real_image_t *rim, int mip)
{
  // OPTIMISE: check if a blockified version at a lower mip already
  // exists, saving us the trouble to read the stuff from the WAD.
 
  real_cached_image_t *rc;
  
  rc = ReadAsBlock(rim, mip);

  DEV_ASSERT2(rc->mode == IMG_Block);

  rc->users++;
  InsertAtTail(rc);

  return rc;
}

#ifdef USE_GL

static INLINE
real_cached_image_t *LoadImageOGL(real_image_t *rim)
{
  real_cached_image_t *rc, *tmp_rc;
  int size;

  boolean_t clamp = false;
  boolean_t halo  = false;
  
  const byte *what_palette;
  boolean_t what_pal_cached = false;
  
  tmp_rc = ReadAsBlock(rim, 0);

  DEV_ASSERT2(tmp_rc->mode == IMG_Block);

  rc = (real_cached_image_t *) Z_New(real_cached_image_t,1);

  if (rim->source_type <= IMSRC_Sprite)
    clamp = true;
   
  if (rim->source_type == IMSRC_Halo)
    halo = true;
   
  if (rim->source_palette < 0)
    what_palette = (const byte *) &playpal_data[0];
  else
  {
    what_palette = W_CacheLumpNum(rim->source_palette);
    what_pal_cached = true;
  }

  // compute approximate size (including mipmaps)
  size = sizeof(real_cached_image_t) +
         rim->pub.total_w * rim->pub.total_h * 16 / 3;

  rc->next = rc->prev = NULL;
  rc->parent = rim;
  rc->mip = 0;
  rc->users = 0;
  rc->invalidated = false;
  rc->mode = IMG_OGL;
  rc->size = size;

  rc->info.ogl.tex_id = W_SendGLTexture(tmp_rc->info.block.pixels,
      rim->pub.total_w, rim->pub.total_h, clamp, halo, false, 
      what_palette);

  // free the blockified one.  Note: there's nothing in it that needs
  // special handling.
  Z_Free(tmp_rc);

  if (what_pal_cached)
    W_DoneWithLump(what_palette);

  rc->users++;
  InsertAtTail(rc);

  return rc;
}
#endif  // USE_GL


static INLINE 
void UnloadImagePost(real_cached_image_t *rc, real_image_t *rim)
{
  // nothing to do, column & post data were allocated with the
  // real_cached_image_t structure.
}

static INLINE 
void UnloadImageBlock(real_cached_image_t *rc, real_image_t *rim)
{
  // nothing to do, pixel data was allocated along with the
  // real_cached_image_t structure.
}

#ifdef USE_GL

static INLINE 
void UnloadImageOGL(real_cached_image_t *rc, real_image_t *rim)
{
  glDeleteTextures(1, &rc->info.ogl.tex_id);
}
#endif  // USE_GL


//
// UnloadImage
//
// Unloads a cached image from the cache list and frees all resources.
// Mainly just a switch to more specialised image unloaders.
//
static void UnloadImage(real_cached_image_t *rc)
{
  real_image_t *rim = rc->parent;

  DEV_ASSERT2(rc);
  DEV_ASSERT2(rc != &imagecachehead);
  DEV_ASSERT2(rim);
  DEV_ASSERT2(rc->users == 0);

  // unlink from the cache list
  Unlink(rc);

  cache_size -= rc->size;

  switch (rc->mode)
  {
    case IMG_Post:
    {
      UnloadImagePost(rc, rim);
      rim->post_cache = NULL;
      break;
    }
      
    case IMG_Block:
    {
      UnloadImageBlock(rc, rim);
      rim->block_cache.mips[rc->mip] = NULL;
      break;
    }

    case IMG_OGL:
#ifdef USE_GL
    {
      UnloadImageOGL(rc, rim);
      rim->ogl_cache = NULL;
    }
#endif
      break;

    default:
      I_Error("UnloadImage: bad mode %d !\n", rc->mode);
  }

  // finally, free the rest of the mem
  Z_Free(rc);
}


//----------------------------------------------------------------------------

//
//  IMAGE LOOKUP
//

//
// W_ImageFromTexture
//
// Returns NULL for the no texture marker '-', otherwise the result is
// a valid image.  Note: search must be case insensitive.
//
const image_t *W_ImageFromTexture(const char *tex_name)
{
  int i;
  const real_image_t *rim;

  // "NoTexture" marker.
  if (!tex_name || !tex_name[0] || tex_name[0] == '-')
    return NULL;
 
  // look for name in current list
  for (i=0; i < num_real_images; i++)
  {
    rim = real_images[i];
    
    if (rim->source_type != IMSRC_Texture)
      continue;

    if (stricmp(tex_name, rim->name) == 0)
      return &rim->pub;
  }
  
  // backup plan: try a flat with the same name

  for (i=0; i < num_real_images; i++)
  {
    rim = real_images[i];
    
    if (rim->source_type != IMSRC_Flat)
      continue;

    if (stricmp(tex_name, rim->name) == 0)
      return &rim->pub;
  }

  M_WarnError("Unknown texture found in level: '%s'\n", tex_name);

  // return the texture dummy image
  rim = RIM_DUMMY_TEX;
  return &rim->pub;
}

//
// W_ImageFromFlat
//
// Note: search must be case insensitive.
//
const image_t *W_ImageFromFlat(const char *flat_name)
{
  int i;
  const real_image_t *rim;

  // "NoTexture" marker.
  if (!flat_name || !flat_name[0] || flat_name[0] == '-')
    return NULL;

  // "Sky" marker.
  if (stricmp(flat_name, SKYFLATNAME) == 0)
    return skyflatimage;
  
  for (i=0; i < num_real_images; i++)
  {
    rim = real_images[i];
    
    if (rim->source_type != IMSRC_Flat)
      continue;

    if (stricmp(flat_name, rim->name) == 0)
      return &rim->pub;
  }

  // backup plan 1: if lump exists and is right size, add it.

  i = W_CheckNumForName(flat_name);

  if (i >= 0)
  {
    rim = AddImageFlat(flat_name, i);
    if (rim)
      return &rim->pub;
  }

  // backup plan 2: Texture with the same name ?

  for (i=0; i < num_real_images; i++)
  {
    rim = real_images[i];
    
    if (rim->source_type != IMSRC_Texture)
      continue;

    if (stricmp(flat_name, rim->name) == 0)
      return &rim->pub;
  }

  M_WarnError("Unknown flat found in level: '%s'\n", flat_name);

  // return the flat dummy image
  rim = RIM_DUMMY_FLAT;
  return &rim->pub;
}

//
// W_ImageFromPatch
//
const image_t *W_ImageFromPatch(const char *patch_name)
{
  int i;
  const real_image_t *rim;

  for (i=0; i < num_real_images; i++)
  {
    rim = real_images[i];
    
    if (rim->source_type != IMSRC_Patch &&
        rim->source_type != IMSRC_Raw320x200)
      continue;

    if (stricmp(patch_name, rim->name) == 0)
      return &rim->pub;
  }

  // backup plan 1: look for sprites

  for (i=0; i < num_real_images; i++)
  {
    rim = real_images[i];
    
    if (rim->source_type != IMSRC_Sprite)
      continue;

    if (stricmp(patch_name, rim->name) == 0)
      return &rim->pub;
  }

  // not already loaded ?  Check if lump exists in wad, if so add it.

  i = W_CheckNumForName(patch_name);

  if (i >= 0)
  {
    rim = AddImagePatch(patch_name, IMSRC_Patch, i);
    if (rim)
      return &rim->pub;
  }

  M_WarnError("Unknown patch: '%s'\n", patch_name);

  // return the patch dummy image
  rim = RIM_DUMMY_PATCH;
  return &rim->pub;
}

//
// W_ImageFromFont
//
// This is like W_ImageFromPatch above, but for getting font
// characters.
//
const image_t *W_ImageFromFont(const char *patch_name)
{
  int i;
  const real_image_t *rim;

  for (i=0; i < num_real_images; i++)
  {
    rim = real_images[i];
    
    if (rim->source_type != IMSRC_Font)
      continue;

    if (stricmp(patch_name, rim->name) == 0)
      return &rim->pub;
  }

  // not already loaded ?  Check if lump exists in wad, if so add it.

  i = W_CheckNumForName(patch_name);

  if (i >= 0)
  {
    rim = AddImagePatch(patch_name, IMSRC_Font, i);
    if (rim)
      return &rim->pub;
  }

  M_WarnError("Unknown font patch: '%s'\n", patch_name);

  // return the font dummy image
  rim = RIM_DUMMY_FONT;
  return &rim->pub;
}

//
// W_ImageFromHalo
//
// Also like W_ImageFromPatch above, but for halos.
//
const image_t *W_ImageFromHalo(const char *patch_name)
{
  int i;
  const real_image_t *rim;

  for (i=0; i < num_real_images; i++)
  {
    rim = real_images[i];
    
    if (rim->source_type != IMSRC_Halo)
      continue;

    if (stricmp(patch_name, rim->name) == 0)
      return &rim->pub;
  }

  // not already loaded ?  Check if lump exists in wad, if so add it.

  i = W_CheckNumForName(patch_name);

  if (i >= 0)
  {
    rim = AddImagePatch(patch_name, IMSRC_Halo, i);
    if (rim)
      return &rim->pub;
  }

  M_WarnError("Unknown halo graphic: '%s'\n", patch_name);

  // return the dummy halo image
  rim = RIM_DUMMY_HALO;
  return &rim->pub;
}

//
// W_ImageForDummySprite
// 
const image_t *W_ImageForDummySprite(void)
{
  const real_image_t *rim = RIM_DUMMY_SPRITE;
  return &rim->pub;
}

//
// W_ImageFromString
//
// Used by the savegame code.  One special value for type is `*',
// which means that we don't care too much about the actual type (not
// actually used yet -- but reserved for future use).
// 
const image_t *W_ImageFromString(char type, const char *name)
{
  int i;
  const real_image_t *rim;

  switch (type)
  {
    case 'T': return W_ImageFromTexture(name);
    case 'F': return W_ImageFromFlat(name);

    case 'P':
    case 'S':
    case 'r': return W_ImageFromPatch(name);

    case 'o': return W_ImageFromFont(name);
    case 'h': return W_ImageFromHalo(name);

    case 'U': /* user */
    case 'd': /* dummy */
    case '*': /* don't care, equivalent to 'U' */
      break;
     
    default:
      I_Warning("W_ImageFromString: unknown type `%c'\n", type);
      break;
  }

  for (i=0; i < num_real_images; i++)
  {
    rim = real_images[i];
    
    if (stricmp(name, rim->name) == 0)
      return &rim->pub;
  }

  I_Warning("W_ImageFromString: image [%c:%s] not found.\n", type, name);

  // return the texture dummy image
  rim = RIM_DUMMY_TEX;
  return &rim->pub;
}

//
// W_ImageToString
//
// Used by the savegame code.
// 
void W_ImageToString(const image_t *image, char *type, char *namebuf)
{
  const real_image_t *rim;

  rim = (const real_image_t *) image;

  strcpy(namebuf, rim->name);

  switch (rim->source_type)
  {
    case IMSRC_Flat:    (*type) = 'F'; break;
    case IMSRC_Texture: (*type) = 'T'; break;
    case IMSRC_Patch:   (*type) = 'P'; break;
    case IMSRC_Sprite:  (*type) = 'S'; break;
    case IMSRC_User:    (*type) = 'U'; break;

    case IMSRC_Font:    (*type) = 'o'; break;
    case IMSRC_Halo:    (*type) = 'h'; break;
    case IMSRC_Dummy:   (*type) = 'd'; break;
    case IMSRC_Raw320x200: (*type) = 'r'; break;

    default:            (*type) = '*'; break;
  }
}

//
// W_ImageDebugName
//
// Used for debugging (and ONLY debugging !).
//
#ifdef DEVELOPERS
const char *W_ImageDebugName(const image_t *image)
{
  const real_image_t *rim;

  rim = (const real_image_t *) image;

  return rim->name;
}
#endif


//----------------------------------------------------------------------------

//
//  IMAGE USAGE
//

static INLINE
const cached_image_t *ImageCachePost(real_image_t *rim, int mip)
{
  real_cached_image_t *rc;

  DEV_ASSERT2(mip == 0);

  rc = rim->post_cache;

  if (rc && rc->users == 0 && rc->invalidated)
  {
    UnloadImagePost(rc, rim);
    rc = rim->post_cache = NULL;
  }

  // already cached ?
  if (rc)
  {
    if (rc->users == 0)
    {
      // no longer freeable
      cache_size -= rc->size;
    }
    rc->users++;
  }
  else
  {
    // load into cache
    rc = rim->post_cache = LoadImagePost(rim);
  }

  DEV_ASSERT2(rc);
  DEV_ASSERT2(rc->mode == IMG_Post);

  return (const cached_image_t *)(rc + 1);
}

static INLINE
const cached_image_t *ImageCacheBlock(real_image_t *rim, int mip)
{
  real_cached_image_t *rc;
  
  if (mip >= rim->block_cache.num_mips)
  {
    // reallocate mip array
    
    Z_Resize(rim->block_cache.mips, real_cached_image_t *, mip+1);

    while (rim->block_cache.num_mips <= mip)
      rim->block_cache.mips[rim->block_cache.num_mips++] = NULL;
  }

  rc = rim->block_cache.mips[mip];

  if (rc && rc->users == 0 && rc->invalidated)
  {
    UnloadImageBlock(rc, rim);
    rc = rim->block_cache.mips[rc->mip] = NULL;
  }

  // already cached ?
  if (rc)
  {
    if (rc->users == 0)
    {
      // no longer freeable
      cache_size -= rc->size;
    }
    rc->users++;
  }
  else
  {
    // load into cache
    rc = rim->block_cache.mips[mip] = LoadImageBlock(rim, mip);
  }

  DEV_ASSERT2(rc);
  DEV_ASSERT2(rc->mode == IMG_Block);

  return (const cached_image_t *)(rc + 1);
}

#ifdef USE_GL

static INLINE
const cached_image_t *ImageCacheOGL(real_image_t *rim, int mip)
{
  real_cached_image_t *rc;

  DEV_ASSERT2(mip == 0);

  rc = rim->ogl_cache;

  if (rc && rc->users == 0 && rc->invalidated)
  {
    UnloadImageOGL(rc, rim);
    rc = rim->ogl_cache = NULL;
  }

  // already cached ?
  if (rc)
  {
    if (rc->users == 0)
    {
      // no longer freeable
      cache_size -= rc->size;
    }
    rc->users++;
  }
  else
  {
    // load into cache
    rc = rim->ogl_cache = LoadImageOGL(rim);
  }

  DEV_ASSERT2(rc);
  DEV_ASSERT2(rc->mode == IMG_OGL);

  return (const cached_image_t *)(rc + 1);
}
#endif  // USE_GL


//
// W_ImageCache
//
// The top-level routine for caching in an image.  Mainly just a
// switch to more specialised routines.  Never returns NULL.
//
const cached_image_t *W_ImageCache(const image_t *image, 
    image_mode_e mode, int mip, boolean_t anim)
{
  // Intentional Const Override
  real_image_t *rim = (real_image_t *) image;
 
  DEV_ASSERT2(mip >= 0);

  if (mip > MAX_MIP)
    mip = MAX_MIP;

  // handle animations
  if (anim)
    rim = rim->anim.cur;

  switch (mode)
  {
    case IMG_Post:
      return ImageCachePost(rim, mip);
      
    case IMG_Block:
      return ImageCacheBlock(rim, mip);

    case IMG_OGL:
#ifdef USE_GL
      return ImageCacheOGL(rim, mip);
#else
      break; // -ACB- 2000/07/10 Required to compile file
#endif
  }

  I_Error("W_ImageCache: bad mode %d !\n", mode);
  return NULL;
}


//
// W_ImageDone
//
void W_ImageDone(const cached_image_t *c)
{
  real_cached_image_t *rc;

  DEV_ASSERT2(c);

  // Intentional Const Override
  rc = ((real_cached_image_t *) c) - 1;

  DEV_ASSERT(rc->users > 0, ("W_ImageDone: No users"));

  rc->users--;

  if (rc->users == 0)
  {
    cache_size += rc->size;

    // move cached image to the end of the cache list.  This way,
    // the Most Recently Used (MRU) images are at the tail of the
    // list, and thus the Least Recently Used (LRU) images are at the
    // head of the cache list.

    Unlink(rc);
    InsertAtTail(rc);
  }
}


//
// W_ImageGetPost
//
// Return the posts for the cached image at the given column.  Returns
// NULL if the column is empty.  `column' will be suitably converted
// if it lies outside of the horizontal range of the image.
//
const w_post_t *W_ImageGetPost(const cached_image_t *c, int column)
{
  real_cached_image_t *rc;
  int mip_w;
 
  DEV_ASSERT2(c);

  // Intentional Const Override
  rc = ((real_cached_image_t *) c) - 1;

  DEV_ASSERT2(rc->parent);
  DEV_ASSERT2(rc->mode == IMG_Post);
  DEV_ASSERT2(rc->info.post.columns);

  mip_w = MIP_SIZE(rc->parent->pub.total_w, rc->mip);

  DEV_ASSERT2(mip_w >= 0);

  // power of two check
  DEV_ASSERT2((mip_w & (mip_w - 1)) == 0);
 
  return rc->info.post.columns[column & (mip_w - 1)];
}


//
// W_ImageGetBlock
//
// Return the pixel block for the cached image.  Computing the exact
// start for a particular column is left as an exercise to the caller.
// Never returns NULL.
//
const byte *W_ImageGetBlock(const cached_image_t *c)
{
  real_cached_image_t *rc;
 
  DEV_ASSERT2(c);

  // Intentional Const Override
  rc = ((real_cached_image_t *) c) - 1;

  DEV_ASSERT2(rc->parent);
  DEV_ASSERT2(rc->mode == IMG_Block);

  DEV_ASSERT2(rc->info.block.pixels);

  return rc->info.block.pixels;
}


#ifdef USE_GL

//
// W_ImageGetOGL
//
GLuint W_ImageGetOGL(const cached_image_t *c)
{
  real_cached_image_t *rc;

  DEV_ASSERT2(c);

  // Intentional Const Override
  rc = ((real_cached_image_t *) c) - 1;

  DEV_ASSERT2(rc->parent);
  DEV_ASSERT2(rc->mode == IMG_OGL);

  return rc->info.ogl.tex_id;
}

#endif  // USE_GL


static void FlushImageCaches(z_urgency_e urge)
{
  int bytes_to_free = 0;
  real_cached_image_t *rc, *next;

  if (w_locked_ogl)
    return;

  //!!! FIXME: make triple sure that if this is called from _within_
  //!!! one of routines above, nothing bad will happen.

  switch (urge)
  {
    case Z_UrgencyLow: bytes_to_free = cache_size / 16; break;
    case Z_UrgencyMedium: bytes_to_free = cache_size / 8; break;
    case Z_UrgencyHigh: bytes_to_free = cache_size / 2; break;
    case Z_UrgencyExtreme: bytes_to_free = INT_MAX; break;

    default:
      I_Error("FlushImageCaches: Invalid urgency level %d\n", urge);
  }

  // the Least Recently Used (LRU) images are at the head of the image
  // cache list, so we unload those ones first.
 
  for (rc = imagecachehead.next; 
       rc != &imagecachehead && bytes_to_free > 0; rc = next)
  {
    next = rc->next;

    if (rc->users == 0)
    {
      bytes_to_free -= rc->size;
      UnloadImage(rc);
    }
  }
}

//
// W_ImagePreCache
// 
void W_ImagePreCache(const image_t *image)
{
#ifdef USE_GL
  W_ImageDone(W_ImageCache(image, IMG_OGL, 0, false));
#else
  W_ImageDone(W_ImageCache(image, IMG_Block, 0, false));
#endif
}


//----------------------------------------------------------------------------

//
// W_InitImages
//
// Initialises the image system.
//
boolean_t W_InitImages(void)
{
  // the only initialisation the cache list needs
  imagecachehead.next = imagecachehead.prev = &imagecachehead;

  Z_RegisterCacheFlusher(FlushImageCaches);

  Z_InitStackArray(&real_images_a, (void ***)&real_images, 
      sizeof(real_image_t), 0);

  // setup dummy images
  AddImageDummy(IMSRC_Texture);
  AddImageDummy(IMSRC_Flat);
  AddImageDummy(IMSRC_Flat);  // this one is for sky
  AddImageDummy(IMSRC_Patch);
  AddImageDummy(IMSRC_Sprite);
  AddImageDummy(IMSRC_Font);
  AddImageDummy(IMSRC_Halo);

  skyflatimage = &RIM_SKY_FLAT->pub;

  // check options
  M_CheckBooleanParm("smoothing", &use_smoothing, false);
  M_CheckBooleanParm("dither", &use_dithering, false);

  if (M_CheckParm("-nomipmap"))
    use_mipmapping = 0;
  else if (M_CheckParm("-mipmap"))
    use_mipmapping = 1;
  else if (M_CheckParm("-trilinear"))
    use_mipmapping = 2;

  return true;
}

//
// W_UpdateImageAnims
//
// Animate all the images.
//
void W_UpdateImageAnims(void)
{
  int i;

  for (i=0; i < num_real_images; i++)
  {
    real_image_t *rim = real_images[i];
    
    // not animated ?
    if (rim->anim.speed == 0)
      continue;

    DEV_ASSERT2(rim->anim.count > 0);

    rim->anim.count--;

    if (rim->anim.count == 0 && rim->anim.cur->anim.next)
    {
      rim->anim.cur = rim->anim.cur->anim.next;
      rim->anim.count = rim->anim.speed;
    }
  }
}

//
// W_ResetImages
//
// Resets all images, causing all cached images to be invalidated.
// Needs to be called when gamma changes (GL renderer only), or when
// certain other parameters change (e.g. GL mipmapping modes).
//
void W_ResetImages(void)
{
  real_cached_image_t *rc, *next;

  for (rc=imagecachehead.next; rc != &imagecachehead; rc=next)
  {
    next = rc->next;
    
    if (rc->users == 0)
      UnloadImage(rc);
    else
      rc->invalidated = true;
  }
}

//
// W_LockImagesOGL
//
// Prevents OGL texture ids from being deleted.  Essentially this
// routine is like giving all cached images an extra user.  It is
// needed due to the curreny way the RGL_UNIT code works.
//
void W_LockImagesOGL(void)
{
  DEV_ASSERT(!w_locked_ogl, ("W_LockImagesOGL: Already locked."));

  w_locked_ogl = true;
}

//
// W_UnlockImagesOGL
//
void W_UnlockImagesOGL(void)
{
  DEV_ASSERT(w_locked_ogl, ("W_UnlockImagesOGL: NOT locked."));

  w_locked_ogl = false;
}

//
// W_AnimateImageSet
//
// Sets up the images so they will animate properly.
//
// NOTE: modifies the input array of images.
// 
void W_AnimateImageSet(const image_t ** images, int number, int speed)
{
  int i, total;
  real_image_t *rim, *other;

  DEV_ASSERT2(images);
  DEV_ASSERT2(speed > 0);

  // ignore images that are already animating
  for (i=0, total=0; i < number; i++)
  {
    // Intentional Const Override
    rim = (real_image_t *) images[i];

    if (rim->anim.speed > 0)
      continue;

    images[total++] = images[i];
  }

  // anything left to animate ?
  if (total < 2)
    return;

  for (i=0; i < total; i++)
  {
    // Intentional Const Override
    rim   = (real_image_t *) images[i];
    other = (real_image_t *) images[(i+1) % total];

    rim->anim.next = other;
    rim->anim.speed = rim->anim.count = speed;
  }
}

